/************************************************************
 *
 minires 0.96         stub synchronous resolver for Cygwin
 Pierre A. Humblet
 # June 2003

 Copyright (c) 2001, 2002, 2003 Pierre A. Humblet
 This program is free software; you can redistribute it and/or modify it
 under the terms of the GNU General Public License as published by the Free
 Software Foundation; either version 2 of the License, or (at your option)
 any later version

************************************************************/
#include "resolver.h"

/***********************************************************************

Utilities

***********************************************************************/
/***********************************************************************

dprintf
***********************************************************************/
void dprintf(char * format, ...)
{
  va_list args;
  
  va_start(args, format);
  fprintf(stderr, "Minires: ");
  vfprintf(stderr, format, args);
  va_end(args);
}

/***********************************************************************

scanline
Put pointers in list[] to the beginning of each space delimited word in "in",
and put the lengths in sizes[]
Return the number of words found  
***********************************************************************/
static int scanline(char * in, char **list, int * sizes, int maxnum)
{
  int i;
  char * startp;
  for (i = 0; i < maxnum; i++) {
    while((*in) && isspace(*in)) in++;
    if (*in == 0) 
      break;
    startp = in++;
    while((*in) && !isspace(*in)) in++;
    list[i] = startp;
    sizes[i] = in - startp; 
  }
  return i;
}

/***********************************************************************

Make string

***********************************************************************/
char * make_string( char * from, int len, int debug)
{
  char * ptr;
  if ( len < 0 ) 
    len = strlen(from); 
  if ((ptr = (char *) malloc(len+1)) == 0) {
    DPRINTF(debug, "malloc: %s\n", strerror(errno));
  }
  else {
    memcpy(ptr, from, len);
    ptr[len] = 0;
  }
  return ptr;
}   

/***********************************************************************

Read the LOCALDOMAIN string (search list) 

***********************************************************************/
static void get_env_domain( char * envstring, char ** dnsrch, int debug)
{
  char * words[MAXDNSRCH];
  int sizes[MAXDNSRCH];
  int i, j;
  i = scanline(envstring, words, sizes, MAXDNSRCH);
  for (j = 0; j < MAXDNSRCH; j++) {
    if (dnsrch[j] != NULL ) {
      DPRINTF(debug, "removing \"%s\"\n", dnsrch[j]);
      free(dnsrch[j]);  /* was already allocated */
    }
    if (j < i) {
      dnsrch[j] = make_string(words[j], sizes[j], debug);
      DPRINTF(debug, "LOCALDOMAIN \"%s\"\n", dnsrch[j]);
    }
    else dnsrch[j] = 0;
  }
}

/***********************************************************************

Read the resolv.conf file.
We only look for nameserver, domain and search

***********************************************************************/
#if MAXNS > MAXDNSRCH
#define MAXSIZE MAXNS
#else 
#define MAXSIZE MAXDNSRCH
#endif
static int get_resolv(struct sockaddr_in * SockAddrPtr, char ** dnsrch, int debug)
{
  FILE * fd;
  char *words[MAXSIZE + 1], line[4096];
  int sizes[MAXSIZE + 1];
  int i, j, ns = 0;
  if ((fd = fopen(_PATH_RESCONF, "r")) == NULL) 
    return 0;
  while ( fgets(line, sizeof(line), fd) != 0) {
    DPRINTF(debug, "resolv.conf %s", line);
    if ((i = scanline(line, words, sizes, MAXSIZE + 1)) > 0) {
      if (strncasecmp("nameserver", words[0], sizes[0]) == 0) {
        for ( j = 1; j < i ; j++) {
	  unsigned int address;
          *(words[j] + sizes[j]) = 0;
	  address = inet_addr(words[j]);
          if (address == -1) {
	    DPRINTF(debug, "invalid server \"%s\"\n", words[j]);
	  }
	  else if (ns >= MAXNS) {
	    DPRINTF(debug, "no space for server \"%s\"\n", words[j]);
	  }
	  else {
	    SockAddrPtr[ns++].sin_addr.s_addr = address;
	    DPRINTF(debug, "server \"%s\"\n", words[j]);
	  }
        }
      }
      else if (strncasecmp("search", words[0], sizes[0]) == 0) {
        for (j = 0; j < MAXDNSRCH; j++) {
          if (dnsrch[j] != NULL ) {
	    DPRINTF(debug, "removing \"%s\"\n", dnsrch[j]);
	    free(dnsrch[j]);  /* was already allocated */
	  }
          if (j < i -1) { 
	    dnsrch[j] = make_string(words[j+1], sizes[j+1], debug);
	    DPRINTF(debug, "domain \"%s\"\n", dnsrch[j]);
	  }
          else 
	    dnsrch[j] = 0;
        }
      }
      /* Domain goes into 0th element of dnssrch */
      else if (strncasecmp("domain", words[0], sizes[0]) == 0) {
        for (j = 0; j < MAXDNSRCH; j++) {
          if (dnsrch[j] != NULL ) {
	    DPRINTF(debug, "removing \"%s\"\n", dnsrch[j]);
	    free(dnsrch[j]);  /* was already allocated */
	  }
          if (j < 1) {
	    dnsrch[j] = make_string(words[1], sizes[1], debug);
	    DPRINTF(debug, "domain \"%s\"\n", dnsrch[j]);
	  }
          else dnsrch[j] = 0;
        }
      }
    }
  }
  fclose(fd);
  return ns;
}

/****************************************************************************/
/* 
   open_sock()
   Create a datagram socket and call bind.

****************************************************************************/

static int open_sock(struct sockaddr_in *CliAddr, int debug)
{
  int fd;
  
  /* Create a datagram socket */
  if ((fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
    DPRINTF(debug, "socket: %s\n", strerror(errno));
    return -1;
  }
  CliAddr->sin_family = AF_INET;
  CliAddr->sin_addr.s_addr = htonl(INADDR_ANY);
  CliAddr->sin_port = htons(0);
  bzero(CliAddr->sin_zero, sizeof(CliAddr->sin_zero));
  /* Get a port */
  if (bind(fd, (struct sockaddr *) CliAddr, sizeof(*CliAddr)) < 0) {
    DPRINTF(debug, "bind: %s\n", strerror(errno));
    return -1;
  }
  return fd;
}

/*****************************************************************
 *
 __res_state()
 Undocumented but public. Accessed through _res

 *****************************************************************/
static struct __res_state res = {};
struct __res_state *__res_state(void)
{
  return & res;
}

/*****************************************************************
 *
 res_init()

 *****************************************************************/
int res_ninit(res_state statp)
{
  int i, num, num2;
  char * ptr;
  
  statp->res_h_errno = NETDB_SUCCESS;
  /* Get the domainname (dnsrch[0]) 
     the search list
     the number of nameservers and their IP addresses (Network Byte Order) */
  for (i = 0; i < MAXDNSRCH + 1; i++)  statp->dnsrch[i] = 0;
  /* Get domain and dns server info from an os-specific routine */
  num = get_dns_info(statp->dnsrch, statp->nsaddr_list, MAXNS, statp->options & RES_DEBUG );
  /* resolv.conf (dns, domain & srch)*/
  num2 = get_resolv(statp->nsaddr_list, statp->dnsrch, statp->options & RES_DEBUG); 
  if (num2 > 0) 
    num = num2; /* Give preference to resolv.conf */
  if ((ptr = getenv("LOCALDOMAIN")) != 0 ) 
    get_env_domain( ptr, statp->dnsrch, statp->options & RES_DEBUG); /* domain & srch */
  if (num <= 0) {
    statp->res_h_errno = NETDB_INTERNAL;
    DPRINTF(statp->options & RES_DEBUG, "no dns server found\n"); 
    return -1;
  }
  for (i = 0; i < num; i++) {
    statp->nsaddr_list[i].sin_family = AF_INET;
    statp->nsaddr_list[i].sin_port = htons(NAMESERVER_PORT);
    bzero(statp->nsaddr_list[i].sin_zero, sizeof(statp->nsaddr_list[i].sin_zero));
  }
  statp->sockfd = -1;
  statp->nscount = num;         /* number of servers */
  statp->retrans = RES_TIMEOUT; /* timeout in seconds */
  statp->retry = RES_MAXRETRY;  /* max number of retries */
  /* only debug may be set before calling init */
  statp->options &= RES_DEBUG;
  statp->options |= RES_INIT | RES_DEFAULT;
  return 0;
}

int res_init()
{
  int r = res_ninit(& res);
  h_errno = res.res_h_errno;
  return r;
}

/*****************************************************************
 *
 res_close()

 *****************************************************************/
void res_nclose(res_state statp)
{
  int res;
  if (statp->sockfd != -1) {
    res = close(statp->sockfd);
    DPRINTF(statp->options & RES_DEBUG, "close sockfd %d: %s\n", 
	    statp->sockfd, (res == 0)?"OK":strerror(errno));
    statp->sockfd = -1;
  }
}

void res_close()
{
  res_nclose(& res);
}  

/*****************************************************************
 **
 res_send
 Assumes that the message is a query starting with a short id.
 Handles retransmissions until that id is received.
 
*****************************************************************/
int res_nsend( res_state statp, const unsigned char * MsgPtr, 
	       int MsgLength, unsigned char * AnsPtr, int AnsLength)
{
  /* Current server, shared by all tasks */
  volatile static unsigned int SServ = 0XFFFFFFFF;

  fd_set fdset_read; 
  int rslt, addrLen, transNum, wServ;
  struct sockaddr mySockAddr, dnsSockAddr;
  struct timeval timeOut;
  
  statp->res_h_errno = NETDB_SUCCESS;
  if (((statp->options & RES_INIT) == 0) && (res_ninit(statp) != 0)) 
    return -1;

  /* Close the socket if it had been opened before a fork.
     Reuse of pid's doesn't hurt */
  if ((statp->sockfd != -1) && (statp->mypid != getpid())) {
    res_nclose(statp);
  }

  /* Open a socket for this process */
  if (statp->sockfd == -1) {
    /* Create a socket and bind it (to any port) */
    statp->sockfd = open_sock((struct sockaddr_in *) & mySockAddr, 
			      statp->options & RES_INIT);
    DPRINTF(statp->options & RES_DEBUG, "open sockfd %d: %s\n", 
	    statp->sockfd, (statp->sockfd > 0)?"OK":strerror(errno));
    if (statp->sockfd  < 0 ) {
      statp->res_h_errno = NETDB_INTERNAL;
      return -1;
    }
    /* Set close on exec bit */
    if (fcntl(statp->sockfd, F_SETFD, 1) == -1) {
      DPRINTF(statp->options & RES_DEBUG, "fcntl: %s\n", 
                strerror(errno));
      statp->res_h_errno = NETDB_INTERNAL;
      return -1;
    }
    statp->mypid = getpid();
    if (SServ == 0XFFFFFFFF) /* Pseudo random */
      SServ =  statp->mypid % statp->retry;
  }

  transNum = 0;  
  while ( transNum++ < statp->retry) {
    if ((wServ = SServ + 1) >= statp->nscount) 
      wServ = 0;
    SServ = wServ;
    /* Send the message */
    rslt = sendto(statp->sockfd, MsgPtr, MsgLength, 0, 
		  (struct sockaddr *) &statp->nsaddr_list[wServ],
		  sizeof(struct sockaddr));
    DPRINTF(statp->options & RES_DEBUG, "sendto: server %08x sockfd %d %s\n",
	    statp->nsaddr_list[wServ].sin_addr.s_addr, 
	    statp->sockfd, (rslt == MsgLength)?"OK":strerror(errno));
    if (rslt != MsgLength) {
      statp->res_h_errno = NETDB_INTERNAL;
      return -1;
    };
    /*
      Wait for a reply with select()
    */
    FD_ZERO(&fdset_read);
    FD_SET (statp->sockfd, &fdset_read );
    timeOut.tv_sec = statp->retrans;
    timeOut.tv_usec = 0;
    rslt = select( statp->sockfd + 1, &fdset_read, NULL, NULL, &timeOut );
    if ( rslt == 0 ) { /* Timeout */
      DPRINTF(statp->options & RES_DEBUG, "timeout for server %08x\n",
	      statp->nsaddr_list[wServ].sin_addr.s_addr);
    }
    else if ((rslt != 1) || (FD_ISSET(statp->sockfd, &fdset_read) == 0)) {
      DPRINTF(statp->options & RES_DEBUG, "select: %s\n", strerror(errno));
      statp->res_h_errno = NETDB_INTERNAL;
      return -1;
    }
    else {
      addrLen = sizeof(struct sockaddr);
      rslt = recvfrom( statp->sockfd, AnsPtr, AnsLength, 0, 
		       & dnsSockAddr, & addrLen);
      if (rslt <= 0) {
        DPRINTF(statp->options & RES_DEBUG, "recvfrom: %s\n", 
		strerror(errno));
        statp->res_h_errno = NETDB_INTERNAL;
        return -1;
      }
      /* Check if this is the message we expected */
// #define NETDB_INTERNAL -1 /* see errno */
// #define NETDB_SUCCESS   0 /* no problem */
// #define HOST_NOT_FOUND  1 /* Authoritative Answer Host not found */
// #define TRY_AGAIN       2 /* Non-Authoritive Host not found, or SERVERFAIL */
// #define NO_RECOVERY     3 /* Non recoverable errors, FORMERR, REFUSED, NOTIMP */
// #define NO_DATA         4 /* Valid name, no data record of requested type */
      if ((*MsgPtr == *AnsPtr)     /* Ids match */
	  && (*(MsgPtr + 1) == *(AnsPtr + 1))
	  && (rslt >= MsgLength )     /* At least as long */
	  && (memcmp(MsgPtr + 12, AnsPtr + 12, MsgLength - 12) == 0)
	  && ((AnsPtr[2] & QR) != 0)) {
        DPRINTF(statp->options & RES_DEBUG, 
		"answer from %08x to query to %08x. Error %d. Count %d\n",
		((struct sockaddr_in *)&dnsSockAddr)->sin_addr.s_addr, 
		statp->nsaddr_list[wServ].sin_addr.s_addr,
		AnsPtr[3] & ERR_MASK, AnsPtr[6]*256 + AnsPtr[7]);
        if ((AnsPtr[3] & ERR_MASK) == NOERROR) {
          if ((AnsPtr[6]|AnsPtr[7])!= 0) 
	    return rslt;
          else 
	    statp->res_h_errno = NO_DATA;
        }
        else {
          /* return HOST_NOT_FOUND even for non-authoritative answers */
          if ((AnsPtr[3] & ERR_MASK) == NXDOMAIN) 
	    statp->res_h_errno = HOST_NOT_FOUND;
          else if ((AnsPtr[3] & ERR_MASK) == SERVFAIL) 
	    statp->res_h_errno = TRY_AGAIN;
          else statp->res_h_errno = NO_RECOVERY;
        }
        return -1;
      }
      else 
	DPRINTF(statp->options & RES_DEBUG, "unexpected answer from %x to query to %x\n",
		((struct sockaddr_in *)&dnsSockAddr)->sin_addr.s_addr, 
		statp->nsaddr_list[wServ].sin_addr.s_addr);
    }
  }
  DPRINTF(statp->options & RES_DEBUG, "too many timeouts\n");
  statp->res_h_errno = TRY_AGAIN; /* Too many timeouts */
  return -1;
}

int res_send( const unsigned char * MsgPtr, int MsgLength, 
	      unsigned char * AnsPtr, int AnsLength)
{
  int r = res_nsend(& res, MsgPtr, MsgLength, AnsPtr, AnsLength);
  h_errno = res.res_h_errno;
  return r;
}

/*****************************************************************
 *
 res_mkquery

 Return: packet size
        -1 name format is incorrect 
        (enforces rule on max label and domain lengths) 
*****************************************************************/
int res_nmkquery (res_state statp,
		  int op, const char * dnameptr, int qclass, int qtype, 
		  const unsigned char * dataptr, int datalen,
		  const unsigned char * newrr, unsigned char * buf, int buflen)
{
  unsigned char * cptr, * lptr;
  int len;
  short id;

  if (op == QUERY) {
    /* Check if name is too long
       nameserver.h defines MAXDNAME and MAXCDNAME, with different systems
       having different MAXDNAME. MAXCDNAME is 255, RFC 1035, p. 9
       A '.' is added at the end of dnameptr if none is present
       The size of the encoded name is 2 char longer than the original,
       if it doesn't have a final .  */
    len = strlen(dnameptr);
    if ((len >= (MAXCDNAME-1))  /* <= MAXDCNAME - 2 is OK */ 
        && ((len > (MAXCDNAME-1)) || dnameptr[MAXCDNAME] == 0  )) {
      DPRINTF(statp->options & RES_DEBUG, "total length too large\n");
      statp->res_h_errno = NO_RECOVERY;
      return -1;
    }
    cptr = buf;
    /* Fill the header */
    id = statp->id++;
    PUTSHORT(id, cptr);
    PUTSHORT(RD, cptr); 
    PUTSHORT(1, cptr); /* Number of */
    PUTSHORT(0, cptr); /* Number of */
    PUTSHORT(0, cptr); /* Number of */
    PUTSHORT(0, cptr); /* Number of */
    /* Put the name */
    do {
      /* New label */
      lptr = cptr++;   /* Save the length ptr */
      len = MAXLABEL;
      while ((*dnameptr != '.') && (*dnameptr != 0)) {
        if (len-- == 0) {
          DPRINTF(statp->options & RES_DEBUG, "label too long\n");
          statp->res_h_errno = NO_RECOVERY;
          return -1;
        }
        *cptr++ = *dnameptr++;
      }
      *lptr = MAXLABEL - len; /* Write label length */
      if (*lptr == 0) 
	goto fill_done; /* Have 0 */
    } while(*dnameptr++ != 0); /* Skip over . */
    *cptr++ = 0;               /* Final 0 */
  fill_done:
  
    /* Write qtype and qclass */
    PUTSHORT(qtype, cptr);
    PUTSHORT(qclass, cptr);
    return cptr - buf; /* packet size */
  }
  else { /* Not implemented */
    errno = ENOSYS;
    statp->res_h_errno = NETDB_INTERNAL;
    return -1;
  }
}

int res_mkquery (int op, const char * dnameptr, int qclass, int qtype, 
		 const unsigned char * dataptr, int datalen,
		 const unsigned char * newrr, unsigned char * buf, int buflen)
{
  int r = res_nmkquery (& res, op, dnameptr, qclass, qtype, 
		       dataptr, datalen, newrr, buf, buflen);
  h_errno = res.res_h_errno;
  return r;

}

/*****************************************************************
 *
 res_query()
                                                                  
 *****************************************************************/

int res_nquery( res_state statp, const char * DomName, int Class, int Type, 
		unsigned char * AnsPtr, int AnsLength)
{
  char packet[PACKETSZ];
  int len;
  
  DPRINTF(statp->options & RES_DEBUG, "query %s type %d\n", DomName, Type);
  statp->res_h_errno = NETDB_SUCCESS;  
  if ((len = res_nmkquery (statp, QUERY, DomName, Class, Type, 
			   0, 0, 0, packet, PACKETSZ)) < 0)
    return -1;
  return res_nsend( statp, packet, len, AnsPtr, AnsLength);
}

int res_query( const char * DomName, int Class, int Type, unsigned char * AnsPtr, int AnsLength)
{
  int r = res_nquery(& res, DomName, Class, Type, AnsPtr, AnsLength);
  h_errno = res.res_h_errno;
  return r;
}

/*****************************************************************
 *
 res_querydomain()
                                                                  
 *****************************************************************/
int res_nquerydomain( res_state statp, const char * Name, const char * DomName,
		      int Class, int Type, unsigned char * AnsPtr, int AnsLength)
{
  char fqdn[MAXCDNAME+1], *ptr;
  int nlen;

  if (!DomName)
    ptr = (char *) Name;
  else if ((nlen = strlen(Name)) >= sizeof(fqdn) - 1) {
    DPRINTF(statp->options & RES_DEBUG, "querydomain: name too long\n");
    statp->res_h_errno = NO_RECOVERY;
    return -1;
  }
  else {
    strcpy(fqdn, Name);
    ptr = &fqdn[nlen];
    if (nlen && *(ptr - 1) != '.') 
      *(ptr++ - 1) = '.';
    /* Length checking and res_h_errno setting are done in res_mkquery */
    strlcpy(ptr, DomName, sizeof(fqdn) - (ptr - fqdn));
    ptr = fqdn;
  }
  return res_nquery(statp, ptr, Class, Type, AnsPtr, AnsLength);
}

int res_querydomain( const char * Name, const char * DomName, int Class,
		     int Type, unsigned char * AnsPtr, int AnsLength)
{
  int r = res_nquerydomain(& res, Name, DomName, Class, Type, AnsPtr,
			   AnsLength);
  h_errno = res.res_h_errno;
  return r;
}

/*****************************************************************
 *
 res_search()

 *****************************************************************/

int res_nsearch( res_state statp, const char * DomName, int Class, int Type, 
		 unsigned char * AnsPtr, int AnsLength)
{
  int len, stat, i;
  char fullDomName[MAXCDNAME + 2], *ptr; /* Space for 0 + space for overflow detection */

  DPRINTF(statp->options & RES_DEBUG, "search %s type %d\n", DomName, Type);
  if (((statp->options & RES_INIT) == 0) && (res_ninit(statp) != 0)) 
    return -1;
  /* Straight query. Will set res_h_errno */
  stat = res_nquery( statp, DomName, Class, Type, AnsPtr, AnsLength);
  /* Check if will skip search */
  if ((statp->res_h_errno != HOST_NOT_FOUND)         /* Success, or bad error */
      || (((statp->options & RES_DNSRCH) == 0)       /* No search */
	  && (((statp->options & RES_DEFNAMES) == 0) /* and no def domain */
	      || (strchr(DomName, '.') != NULL)))    /*   with no dot */
      || (statp->dnsrch[0] == NULL))         /* No search list */
    return stat;  

  /* Must search */
  len = strlen(DomName); /* HOST_NOT_FOUND implies len <= MAXLABEL */
  strcpy(fullDomName, DomName); 
  fullDomName[len++] = '.';
  fullDomName[MAXCDNAME+1] = 0; /* Overflow caught by res_mkquery() */
  i = 0;
  do { /* Know  statp->dnsrch[i] != NULL */
    ptr = statp->dnsrch[i];
    strncpy(fullDomName + len, ptr, (sizeof(fullDomName)-1) - len);
    stat = res_nquery( statp, fullDomName, Class, Type, AnsPtr, AnsLength);
  } while ((statp->dnsrch[++i] != NULL)
           && (statp->res_h_errno == HOST_NOT_FOUND)    /* Severe failure or host found */
           && ((statp->options & RES_DNSRCH) != 0));
  /* Return last stat */
  return stat;
}

int res_search( const char * DomName, int Class, int Type, 
		unsigned char * AnsPtr, int AnsLength)
{
  int r = res_nsearch(& res, DomName, Class, Type, AnsPtr, AnsLength);
  h_errno = res.res_h_errno;
  return r;
}

/*****************************************************************
 *
 dn_expand
                                                                  
 *****************************************************************/

int dn_expand(const unsigned char *msg, const unsigned char *eomorig,
              const unsigned char *comp_dn, char *exp_dn, int length)
{
  int len, complen = 0;
  const unsigned char *comp_dn_orig = comp_dn;
//  char * exp_start = exp_dn;

  if ((len = *comp_dn++) == 0)       /* Weird case */
    exp_dn++;
  else do {
    if (len <= MAXLABEL) {
      if ((length -= (len + 1)) > 0 /* Need space for final . */
	  && comp_dn + len <= eomorig) {
        do { *exp_dn++ = *comp_dn++; } while (--len != 0);
        *exp_dn++ = '.';
      }
      else 
	goto expand_fail;
    }
    else if (len >= (128+64)) {
      if (!complen)   /* Still in the original field? */
	complen = (comp_dn - comp_dn_orig) + 1; 
      comp_dn = msg + (((len & ~(128+64)) << 8) + *comp_dn);
      if (comp_dn >= eomorig) 
	goto expand_fail;
    }
    else 
      goto expand_fail;
  } while ((len = *comp_dn++) != 0);
  /* Replace last . with a 0 */
  *(--exp_dn) = 0;
  if (!complen) 
    complen = comp_dn - comp_dn_orig;
//  fprintf(stderr, "dn_expand %s\n", exp_start);
  return complen;
  
expand_fail:
//  fprintf(stderr, "dn_expand fails\n");  
  return -1;
}


/*****************************************************************
 *
 dn_comp

 NOT IMPLEMENTED
 *****************************************************************/
int dn_comp(const char * a, u_char * b, int c, u_char ** d, u_char ** e)
{
  errno = ENOSYS;
  return -1;
}

/*****************************************************************
 *
 dn_skipname

 Measures the compressed domain name length and returns it.
 *****************************************************************/
int dn_skipname(const unsigned char *comp_dn, const unsigned char *eom)
{
  int len;
  const unsigned char *comp_dn_orig = comp_dn;

  do {
    len = *comp_dn++;
    if (len >= (128 + 64)) {
      comp_dn++;
      break;
    }
    if (len > MAXLABEL ||
	(comp_dn += len) > eom)
      return -1;	
  } while (len != 0);

  return comp_dn - comp_dn_orig;
}
