// Camera.m

// Copyright (C) 1996 The Santa Fe Institute.
// No warranty implied, see LICENSE for terms.


#import "CameraDefaults.h"
#import "Camera.h"
#import "TwoD.h"
#import "ThreeD.h"
#import <collections.h>

#include <math.h>

@implementation Camera

// initialize camera parameters: intrinsic and extrinsic
- initCamera: (char *)identifier
{

  Intr = [Intrinsic createBegin: [self getZone]];
  Extr = [Extrinsic createBegin: [self getZone]];
  Intr = [Intr createEnd];
  Extr = [Extr createEnd];
  [Intr initialize: F: KU: KV: U0: V0]; // with default parameters
  [Extr initialize: XInit: YInit: ZInit: RotationInit];
  strcpy(name, identifier);
  return self;
}

//send intrinsic parameters: returns the address of intrinsic
- sendIntr
{
  return Intr;
}

- sendExtr
{
  return Extr;
}

- (char *)sendName
{
  return name;
}

// print name
- print
{
  printf("camera %s\n", name);
  return self;
}

// compute a list of imaged objects from a list of 3D objects
- (int)ComputeImage: (id)point_3D
                   : (id)pixel_list
                   : (int)numOfObj
                   : (int)YSizeOfSwarmGrid
{
  
  double xc, yc, zc;
  double xoc, yoc, zoc;
  double up, vp;
  int u, v, size;
  int i, nbi=0;
  Point2D *pt;
 
  // numOfObj is the number of 3 D objects in the swarm environment

  //printf("numOfObj is %d\n", numOfObj);
  for (i=0; i<numOfObj; i++) {

    //printf("Object %d: ", [[point_3D atOffset: i] sendlabel]);
    // to express in a new coord system, where:
    //						new x is equal to old x
    //						new y is equal to old y
    //						new z is equal to - old z

    // (top view), old coordinate system
    //  X .-----------------> Y
    //    |
    //    | 
    //    |
    //  . |
    //    |
    //    Z
    // 
    // (top view), new coordinate system
    //    Z
    //    |
    //    | 
    //    |
    //  . |
    //    |
    //  X .-----------------> Y
    // 

    xoc = (double) [[point_3D atOffset: i] sendx];
    yoc = (double) [[point_3D atOffset: i] sendy];
    zoc = (double) (YSizeOfSwarmGrid-1 - [[point_3D atOffset: i] sendz]);

    // now express all objects in the camera system, without taking into account the rotation
    xoc = xoc - (double)[Extr sendx];
    yoc = yoc  - (double)[Extr sendy];
    zoc = zoc - (double)(YSizeOfSwarmGrid-1 - [Extr sendz]);
    //printf(" in the camera system, without rotation, is at coords: xoc = %f, yoc = %f, zoc = %f\n", xoc, yoc, zoc);

   // to express coords in the camera system with the rotation: remember, just rotation around x !	
    xc = xoc; 		     
    yc = (yoc * cos([Extr sendtheta])) - (zoc * sin([Extr sendtheta]));
    zc = (yoc * sin([Extr sendtheta])) + (zoc * cos([Extr sendtheta]));

    // in case of some kind of scaling
    xc = xc * CellSideHeight;
    yc = yc * CellSideWidth;
    zc = zc * CellSideLength;

    if (zc>0) {
      // in pixels (double to get more accuracy
      // pin-hole model is used
      up = ((-(double)[Intr sendku]*xc)/zc)*(double)[Intr sendf] + (double)[Intr sendu0];
      vp = (((double)[Intr sendkv]*yc)/zc)*(double)[Intr sendf] + (double)[Intr sendv0];
      //printf("up=%f, vp=%f\n", up, vp);
      u = (int)(up+0.5); // round up values
      v = (int)(vp+0.5);
      if ((v>=FirstPixelV)&&(v<=LastPixelV)) {
        if (u<=LastPixelU){
          if (u<FirstPixelU){
            u=0; //very high stack: thresolded
          }
          pt = [Point2D createBegin: [self getZone]];
          pt = [pt createEnd];    
        
          size = [Intr sendu0] - u;
          [pt fill: u: v: size: nbi];
          [pixel_list addLast: pt];
          //printf(" \nis captured\n");
          nbi+=1;
        }
      } else {
        //printf(" is not imaged\n");
      }
    } else {
      //printf(" is not imaged\n");
    }
  }
  return nbi;
}

@end
