// AgentModelSwarm.m
// swarm room 1996

#import "AgentModelSwarm.h"
#import <random.h>
#import <objectbase.h> // arguments

@implementation AgentModelSwarm

// creation stuff
+ createBegin: aZone
{
  AgentModelSwarm * obj;

  // First, call our superclass createBegin - the return value is the
  // allocated HeatbugModelSwarm object.
  obj = [super createBegin: aZone];

  // Now fill in various simulation parameters with default values.
  obj->allAgents = 154;
  obj->worldXSize = 220;
  obj->worldYSize = 200;
  obj->which_yield = 0;
  obj->random_yield = 442;

  return obj;
}

// createEnd: we could create some objects here if we knew we needed
// them. But this method is called before the user is allowed to fill
// in any customization of the model, so we defer most object creation
// to later. (In this example, this method does nothing at all and could
// just be inherited. But it's here to show you a place to customize.)
-createEnd {
  return [super createEnd];
}

// Now it's time to build the model objects. We use various parameters
// inside ourselves to choose how to create things.
-buildObjects {
  // allow our parent class to build anything.
  [super buildObjects];

  // create the objects in our world
  [self initWorld];
  [self initDataBase];
  return self;
}

-initWorld {
  int i,j;
  // First, set up objects used to represent the environment.
  world = [Grid2d createBegin: [self getZone]];
  [world setSizeX: worldXSize Y: worldYSize];
  world = [world createEnd];
  
  // create sets to keep track of agents and cells
  agentList = [List create: [self getZone]];
  cellList = [List create: [self getZone]];

  // add things to the world without warnings
  [world setOverwriteWarnings: 0];
  
  // create the cell world
  for (i=0;i<worldYSize;i++) {
    for (j=0;j<worldXSize;j++) {
      Cell * cell;
      id settlerSet;

      // initialize new cell
      settlerSet = [Set create: [self getZone]];  // settler set in cell
      cell = [Cell createBegin: [self getZone]];
      [cell setWorld: world];
      [cell setX: j Y: i];
      [cell setSettlerSet: (id) settlerSet];
      cell = [cell createEnd];
      [cellList addLast: cell];
    }
  }
  
  // create some agents
  for (i = 0; i < allAgents; i++) {
    Agent * agent;
    int a,b;

    // get ages for the parents and number kids

    // CGL-NOTE
    // The use of uniformRandom with rMax: (a/5) seems incorrect
    // as this form should be used only with an unsigned int
    // I've cast (a/5) to unsigned int to use uniformUnsRand 

//  a = [uniformRandom rMin: 16 Max: 40];
//  b = [uniformRandom rMax: (a/5)];		
    a = [uniformIntRand getIntegerWithMin: 16 withMax:40];
    b = [uniformUnsRand getUnsignedWithMin: 0 withMax: (unsigned int)(a/5)];

    agent = [Agent createBegin: [self getZone]];
    [agent setTag: i];				  // unique ID
    [agent setFormationDate: time];
    [agent setMySwarm: self];
    [agent setWorld: world];
//  [agent setX: [uniformRandom rMax: worldXSize]
// 	      Y: [uniformRandom rMax: worldYSize]];
    [agent setX: [uniformUnsRand getUnsignedWithMin: 0 withMax: worldXSize]
              Y: [uniformUnsRand getUnsignedWithMin: 0 withMax: worldYSize]];
    [agent setParentAges: a and: a];
    [agent setRandNumKids: b];
    agent = [agent createEnd];
    [agentList addLast: agent];
  }
  [world setOverwriteWarnings: 1];
  return self;
}

-initDataBase
{
  // init database
  // which_yield values: 0 use data file
  //                     1 random normal around avg
  //                     2 user supplied
  //                     3 peaked

  database = [DataBase createBegin: [self getZone]];
  [database setWorld: world];
  [arguments setDefaultAppDataPath: "./data"];
  [database setDataDirectory: [arguments getAppDataPath]];
  database = [database createEnd];

  // init water resources
  [database setDataFile: "hydro.data"];
  [database setDataFileLength: 1];
  [database setSelMethod: M(setWaterType:)];
  [database setWhichYield: -1];
  [database updateCellWorld];
  [database closeDataFile];
  
  // init elevation data
  [database setDataFile: "dem.data"];
  [database setDataFileLength: 1];
  [database setSelMethod: M(setElevation:)];
  [database setWhichYield: -1];
  [database updateCellWorld];
  [database closeDataFile];
	  
  // init soil type
  [database setDataFile: "soil.data"];
  [database setDataFileLength: 1];
  [database setSelMethod: M(setSoilType:)];
  [database setWhichYield: -1];
  [database updateCellWorld];  
  [database closeDataFile];
	  
  // setup for reading in yields
  [database setDataFile: DATAFILE];
  [database setDataFileLength: 25];
  [database setSelMethod: M(setMaizePotential:)];
  [database setWhichYield: which_yield];
  [database setRandomYield: random_yield];
  [database initYields];

  return self;
}

// Here is where the model schedule is built, the data structures
// that define the simulation of time in the mode. The core is an
// actionGroup that has a list of actions. That's then put in a Schedule.
-buildActions {
  [super buildActions];

  // Create the list of simulation actions. We put these in an action
  // group, because we want these actions to be executed in a specific
  // order, but these steps should take no (simulated) time. The
  // M(foo) means "The message called <foo>". You can send a message
  // To a particular object, or ForEach object in a collection.
  modelActions = [ActionGroup create: [self getZone]];
  [modelActions createActionTo: database message: M(updateCellWorld)];
  [modelActions createActionForEach: agentList message: M(step)];
  [modelActions createActionTo: self message: M(removeAgents)];

  // debugging stub //
  [modelActions createActionTo: self message: M(debug)];

  // Then we create a schedule that executes the modelActions. modelActions
  // is an ActionGroup, by itself it has no notion of time. In order to
  // have it executed in time, we create a Schedule that says to use
  // the modelActions ActionGroup at particular times.
  // This schedule has a repeat interval of 1, it will loop every time step.
  // The action is executed at time 0 relative to the beginning of the loop.

  // This is a simple schedule, with only one action that is just
  // repeated every time. See mousetraps for more complicated schedules.

  modelSchedule = [Schedule createBegin: [self getZone]];
  [modelSchedule setRepeatInterval: 1];
  modelSchedule = [modelSchedule createEnd];
  [modelSchedule at: 0 createAction: modelActions];

  return self;
}

// Now set up the model's activation. swarmContext indicates where
// we're being started in - typically, this model is run as a subswarm
// of an observer swarm.
-activateIn: (id) swarmContext {
  // First, activate ourselves via the superclass activateIn: method.
  // Just pass along the context: the activity library does the right thing.
  [super activateIn: swarmContext];

  // Now activate our own schedule.
  [modelSchedule activateIn: self];

  // Finally, return our activity.
  return [self getActivity];
}


// These methods provide access to the objects inside the ModelSwarm.
// These objects are the ones visible to other classes via message call.
// In theory we could just let other objects use Probes to read our state,
// but message access is frequently more convenient.
-getAgentList {
  return agentList;
}

-getCellList {
  return cellList;
}

-getWorld {
  return world;
}

-(int) getNumAgents {
  return [agentList count];
}

-(int) getEstAgents {
  static int est[11] = {154,157,983,964,964,1124,1334,2268,2828,2828,14};
  static int i = 0;
  i++;
  if (i<400){
    return est[i/40];
  } else {
    return est[10];
  }
}

// object addition methods
-addRandomAgents: (int) num {
  int i;

  // make some agents at random places on the grid
  for (i = 0; i < num; i++) {
    Agent * agent;

    agent = [Agent createBegin: [self getZone]];
    [agent setTag: allAgents];
    [agent setFormationDate: time];
    [agent setMySwarm: self];
    [agent setWorld: world];
//  [agent setX: [uniformRandom rMax: worldXSize]
// 	      Y: [uniformRandom rMax: worldYSize]];
    [agent setX: [uniformUnsRand getUnsignedWithMin: 0 withMax: worldXSize]
              Y: [uniformUnsRand getUnsignedWithMin: 0 withMax: worldYSize]];
    [agent setParentAges: 16 and: 16];
    [agent setColor: 200];
    agent = [agent createEnd];
    [agentList addLast: agent];
    allAgents++;
  }
  return self;
}

-addAgentX: (int) atx Y: (int) aty {
  Agent * agent;

  agent = [Agent createBegin: [self getZone]];
  [agent setTag: allAgents];
  [agent setFormationDate: time];
  [agent setMySwarm: self];
  [agent setWorld: world];
  [agent setX: atx Y: aty];
  [agent setParentAges: 16 and: 16];
  [agent setColor: 200];
  agent = [agent createEnd];
  [agentList addLast: agent];
  allAgents++;
  return self;
}

-removeAgents {
  id index,t;

  index = [agentList begin: [self getZone]];
  while ( (t = [index next]) ){
    if ([t getFamilySize] <= 0)
      [[index remove] drop];
  }
  [index drop];

  printf("REMOVE: num of agents %d total %d\n",[agentList count],allAgents);

  return self;
}

-debug {
  static int i = 0;
  id index;
  Agent * a;
  index = [agentList begin: [self getZone]];
  if ((a = [index next]) && [a getTag] < i)
    i = allAgents-1;
  [a debug: -1];
  //   [agentList forEach: @selector(debug:) : (id) i];
  return self;
}

@end








