// file: Agent.m
// defines household behavior in world

#import "Agent.h"

// birth and death global
#define STATE_GOOD 0.05
#define STATE_BAD -0.10

static float birth_per[7]  = {.098,.264,.264,.214,.148,.062,.012};
static float death_per[15] = {.2330,.1400,.1000,.0735,.1305,.1336,.1367,.1400,
                              .1432,.1466,.1500,.1536,.2513,.3270,1.0};

@implementation Agent

// creation stuff
+createBegin: (id) aZone {
  Agent * obj;

  // First, call our superclass createBegin - the return value is the
  // allocated HeatbugModelSwarm object.
  obj = [super createBegin: aZone];
  return obj;
}


// **** initialization stuff ****

-setTag: (int) num {
  tag = num;
  return self;
}

-setFormationDate: (int) date {
  form_date = date;
  return self;
}

-setMySwarm: (id) s {
  mySwarm = s;
  return self;
}

-setWorld: (id) w {
  world = w;
  // cache locally the world size for later use
  worldX = [world getSizeX];
  worldY = [world getSizeY];
  return self;
}


// set our own internal coordinates, also set cell neighborhood
-setX: (int) inX Y: (int) inY {
  x = inX;
  y = inY;
  // place agent into the world by messaging cell to keep track of
  [self setCells];                                // init cell cache
  [cell[4] addSettler: self];
  return self;
}

-setParentAges: (int) i and: (int) j {
  //first value in ages array is for wife, second is for husband
  ages[0] = i;
  ages[1] = j;
  num_family = 2;
  return self;
}

-setRandNumKids: (int) num {
  int i;

  for (i=0;i<num;i++)
//  ages[2+i] = [uniformRandom rMax: 16];
    ages[2+i] = [uniformUnsRand getUnsignedWithMin: 0 withMax: 16];
  num_family += num;
  num_kids = num;
  return self;
}
  
-createEnd {
  int i;

  // add the final touches and check for agent creation
  agent_time = 0;
  
  // initial planting habits and cal needs
  state = 2;
  maize_storage = 450 * num_family;		  // initial endowment in kg
  max_store = 450 * num_family;
  EXP_cal_need = [self calcExpCalUse];
  EXP_yield = 442;  //based on Van West 1994 Appendix E, where
                    //average production across total study area
                    //is 44156 kg per sq km (discounting land with
                    //very low production) or 442 kg per ha;
  AVG_cal_prod = MAIZE_PER*EXP_cal_need;
  plot_need = 2 + num_kids / 2;;
  tot_plots = 0;
  for (i=0;i<9;i++)
    farm_pl[i] = 0;
  traders = [Exchange create: [self getZone]];
  
  // set past expectations of yield to overall avg expectation from data
  for (i=0;i<9;i++)
    past_yield[i][0] = past_yield[i][1] = past_yield[i][2] = 442;

  for (i=num_family;i<10;i++)
    ages[i] = -1;

  // check world and the ages array, parents should be set
  if (ages[0]<16 || ages[1]< 16 || ages[0] > 40 || ages[1] > 40){
    fprintf(stderr,"ERROR: agent %d parent ages (%d,%d) out of range\n",
            tag,ages[0],ages[1]); exit(-1);
  }
  if (world == nil){
    fprintf(stderr,"ERROR: agent %d no world initialized\n",tag); exit(-1);
  }

  // try to get the agents to move immediately, to better places hopefully
  [self moveHouse];

  return self;
}


// **** agent internal state functions ****

-setCells {					  // sets cell neighborhood
  int i,j,k;
  k = 0;
  for (i=-1;i<2;i++)
    for (j=-1;j<2;j++,k++)
      cell[k] = [world getObjectAtX: wrapX(j) Y: wrapY(i)];
  return self;
}

-setFamilyTag: (int) t {
  family_tag = t;
  return self;
}

-setMaizeStorage: (int) food {
  maize_storage = food;
  return self;
}

-setFarmPl: (int) plot At: (int) i {
  farm_pl[i] = plot;
  return self;
}


// **** agent actions ****

-step {
  // begin the year
  agent_time++;					  // another agent year passed
  total_cal = 0;				  // cals spent during year
  
  // [self debug: 0];

  // act the season, a whole year each step
  [self spring];
  if (state > -1) [self summer];
  if (state > -1) [self fall];
  if (state > -1) [self winter];
  maize_storage -= (int) (STORAGE_DECR*maize_storage);
  
  // calculate annual birth
  if (state > -1) [self birth: ages[0]];          // mother is ages[0]
  if (state > -1) [self updateFamily];            // execute family happenings
  if (state > -1) [self evalState];               // how am I doing
  
  if (state == -1 && num_family != 0){
    fprintf(stderr,"ERROR: agent %d ending with bad fam size %d\n",
            tag,num_family); exit(-1);
  }         
  
  // end the year
  return self;
}

// **** the seasons ****

-spring {
  int dx,dy,k=0;
  int i;

  // expenditure of cals
  actual_cal = num_kids * BASE_CAL_KID;
  if (ages[0]>0) actual_cal += BASE_CAL_WOM;
  if (ages[1]>0) actual_cal += BASE_CAL_MAN;

  // plot_need should be equal to what needs to be planted/shedded this period
  // but must check to see if there is enough labor
  // only >7 yr olds contribute meaningful labor
  for (i=0;i<num_family;i++)
    if (ages[i]>7) k++;
  if (tot_plots > k+1){
    plot_need = -1;
  } else if (tot_plots == k+1 && plot_need > 0){
    plot_need = 0;
  } else if (tot_plots+plot_need > k+1){
    i = k - tot_plots;
    plot_need = (i>plot_need) ? plot_need : i;
  }
  
  if (DEBUG == 1 && tag == TAG)
    printf("DEBUG: agent %d wants to plant %d, has %d k %d\n",
           tag,plot_need,tot_plots,k);
  
  // plant until done, start with homestead 4 has
  // passing instance var, might want function for later uses not with plot_need
  if (plot_need>0){
    [self plantPl: &plot_need At: 4];
  } else if (plot_need<0 && tot_plots == farm_pl[4]){
    [self plantPl: &plot_need At: 4];
  }
  if (DEBUG == 1 && tag == TAG)
    printf("DEBUG: agent %d planting need %d, total %d\n",
           tag,plot_need,tot_plots);

  k = 0;
  while (plot_need!=0 && k<7){                    // no room in home plots
    dx = dy = 0;
    if (plot_need>0){
      [self searchNeighborhoodDX: &dx DY: &dy Rad: 1];
    } else {
      [self unplotDX: &dx DY: &dy];
    }
    dx++; dy++;
    if ((dy*3+dx)>8 || (dy*3+dx)<0){
      fprintf(stderr,"ERROR: agent %d plot ranged dx %d dy %d\n",tag,dx,dy);
      exit(-1);
    }

    [self plantPl: &plot_need At: (dy*3+dx)];
    k++;                                          // safety
  }
  if (DEBUG == 1 && tag == TAG)
    printf("DEBUG: agent %d after need %d, total %d\n",
           tag,plot_need, tot_plots);
  
  // have has determined, 3 days to plant
  for (i=0;i<9;i++)
    actual_cal += 3 * farm_pl[i] * 21;
  actual_cal -= 3 * farm_pl[4] * 11;              // cheaper planting at home
  // need to take into account that new
  // fields have 11 days of clearing/hoeing costs;
  // old fields have only 3.
  // food usage during season
  [self eatMaize: actual_cal];
  total_cal += actual_cal;
  return self;
}

-summer {
  int i;
  actual_cal = num_kids * BASE_CAL_KID;
  if (ages[0]>0) actual_cal += BASE_CAL_WOM;
  if (ages[1]>0) actual_cal += BASE_CAL_MAN;

  // weeding activity for 45 days, travel cost, etc (everyone can have fun with this
  for (i=0;i<9;i++)
    actual_cal += 45 * 17 * (i==4?1:2);

// food usage during season
  total_cal += actual_cal;
  [self eatMaize: actual_cal];
  return self;
}

-fall
{
  int i,harvest=0;
  actual_cal = num_kids * BASE_CAL_KID;
  if (ages[0]>0) actual_cal += BASE_CAL_WOM;
  if (ages[1]>0) actual_cal += BASE_CAL_MAN;

  // weeding activity for 15 days
  for (i=0;i<9;i++)
    actual_cal += 15 * 8 * WORK_CAL_MAN * (i==4?1:2);

  // get crop
  for (i=0;i<9;i++)
    harvest += (farm_pl[i] * [cell[i] getMaizePotential] * 4)/PLOTS;
  //VegPotential is in kg per ha, so multiplying by 4 yields kg/cell;
  harvest /= FALLOW_FACTOR; // * ADJUST_FACTOR;

  // harvest costs
  actual_cal += harvest/25 * 50 - farm_pl[4] * 10;

  AVG_cal_prod = (harvest*MAIZE_KG_CAL+AVG_cal_prod)/2;
  maize_storage += harvest;
  if (DEBUG == 1 && tag == TAG)
    printf("DEBUG: agent %d harvested %d\n",tag,harvest);
  
  // update yield history record
  [self updateYields];                            
  
  // food usage during season
  [self eatMaize: actual_cal];
  total_cal += actual_cal;
  return self;
}


-winter
{
  actual_cal = num_kids * BASE_CAL_KID;
  if (ages[0]>0) actual_cal += BASE_CAL_WOM;
  if (ages[1]>0) actual_cal += BASE_CAL_MAN;

  // food usage during season
  [self eatMaize: actual_cal];
  total_cal += actual_cal;
  return self;
}


// **** family yearly actions ****

-updateFamily
{
  int i;

  // age parents
  for (i=0;i<2;i++) {
    if (ages[i]>0){
      ages[i]++;                                  // age one more year
      if ( [self mortality: ages[i]] ) {          // check mortality on new age
        // printf("DEBUG: par death %d\n",ages[i]);
        [self removePerson: i]; break;
      }
    } else {
      // remarriage in a year if other parent alive
      ages[i] = (i==0 ? ages[1] : ages[0]);
      if (ages[i]>0) num_family++;
    }
  }
  
  // update kids
  for (i=2;i<2+num_kids;i++){
    ages[i]++;
    if ( [self mortality: ages[i]] ) {            // mortality rate check
      ages[i] = -3;
//  } else if ( ages[i] > 14 && [uniformRandom rFloat] < .8){
    } else if ( ages[i] > 14 && 
		[uniformDblRand getDoubleWithMin: 0.0 withMax: 1.0]  < .8){
      // creation of new household thru marriage
      ages[i] = -3;
//    if ([uniformRandom rFloat] < .5){           // .4 chance moving away
      if ([uniformDblRand getDoubleWithMin: 0.0 
                                   withMax: 1.0] < .5){   // .4 chance moving away
        [self addHousehold];
      }
    }
  }
  for (i=2;i<2+num_kids;i++){
    if (ages[i] == -3){
      [self removePerson: i];
      i--;
    }
  }
  
  // check to see if household is dead
  // for now that includes death of both parents
  // hopefully later this evening a transfer of kids to related kin will occur
  if (num_family < 1 || (num_family-num_kids<1))
    [self death];

  // some state checking
  if (num_family < 0){
    fprintf(stderr,"ERROR: agent %d num_family is negative\n",tag);
    [self debug: -1]; exit(-1);
  }
  return self;
}

-birth: (int) m_age
{
  float tmp=1.0;
  int index;
  // checks state, alters percentage from that
  if (state<1)
    tmp += STATE_BAD;
  if (state>2)
    tmp += STATE_GOOD;

  if (14 < m_age && m_age < 50 && num_kids < 8){
    index = m_age/5 - 3;
    // increase/decrease birth percentage based on state
//  if ( [uniformRandom rFloat] < birth_per[index]*tmp )
    if ( [uniformDblRand getDoubleWithMin: 0.0 withMax: 1.0] < birth_per[index]*tmp )
      // fprintf(stderr,"DEBUG: *** a birth ***\n");
      [self addPerson: 0];
  }    
  return self;
}


-(int) mortality: (int) age
{
  // check to see if person dies randomly (within age group)
  // this is the modifed version which depends on the state of the household
  float tmp=1.0;
  if (state<1)
    tmp -= STATE_BAD;
  if (state>2)
    tmp -= STATE_GOOD;

  if (age<1)
//  return ( [uniformRandom rFloat]<0.2330*tmp ? 1 : 0);
    return ( [uniformDblRand getDoubleWithMin: 0.0 withMax: 1.0] < 0.2330*tmp ? 1 : 0);
  if (age>64)
    return (1);

  // evaluate mortality at 3,8,13,...
  if (age%5 == 3){
    int index = age/5;
//  return ( [uniformRandom rFloat] < death_per[index]*tmp ? 1 : 0 );
    return ( [uniformDblRand getDoubleWithMin: 0.0 
                                      withMax: 1.0] < death_per[index]*tmp ? 1 : 0 );
  }
  return (0);
}

-death
{
  if (state != -1){
    state = -1;
    [self unPlotAll];
    [cell[4] removeSettler: self];
    //    [self clearSettler];
    num_family = num_kids = 0;
  } else {
    fprintf(stderr,"ERROR: agent %d died twice\n",tag);
    exit(-1);
  }

  return self;
}

-addPerson: (int) age
{
  ages[2+num_kids] = age;
  num_kids++;
  num_family++;
  return self;
}

-removePerson: (int) index
{
  int i;
  if (index < 2){ // parent remove
    ages[index] = -2;
    num_family--;
  } else {        // kid remove
    for (i=index;i<=num_kids;i++)
      ages[i] = ages[i+1]; // shift kids forward in array
    ages[num_kids+1] = -1;
    num_family--; num_kids--;
  }
  return self;
}

-evalState
{
  // calc use of stor for the year and use weighed system
  max_store = (int)2*(MAIZE_PER*(float)total_cal/MAIZE_KG_CAL)+max_store/4;
  if (DEBUG == 1 && tag == TAG)
    printf("DEBUG: agent %d MAXSTORE is %d\n",tag,max_store);
  // this calculation is the last thing an agent does in a year
  // internal state variables should represent what happened in year

  // update expectations of cal need, est. by avg last 2 years usage
  EXP_cal_need = total_cal+EXP_cal_need/2;        

  // calculate internal state
  state = 0;
  state += ([traders getBalance] > 0) ? 1:0;
// what is the /2 for ???
  state += (maize_storage*MAIZE_KG_CAL > (int)(MAIZE_PER*EXP_cal_need)/2)? 1:0;
  state += (AVG_cal_prod  > MAIZE_PER*EXP_cal_need)?1:0;
  
  if (state == 3){
    if (plot_need > 0 || tot_plots == 0)
      plot_need = 0;
    if (tot_plots > 0)
      plot_need--;
  } else if (state == 2){ // they are happy ! just right
    plot_need = 0;
  } else if (state == 1){
    if (maize_storage < max_store) plot_need++;
  } else if (state == 0){
    [self moveHouse];
  }

  // this is to be removed when exchange is added and the 3rd state is possible
  if (maize_storage > max_store) plot_need--; 

  // safety on desired plots
  if (tot_plots+plot_need >= PLOTS)
    plot_need = PLOTS - tot_plots;
  return self;
}


// **** household utilities ****

-eatMaize: (int) cals_spent
{
  // eat food, include percent of Cals from maize
  maize_storage -= (int) (MAIZE_PER*cals_spent/(float)MAIZE_KG_CAL);
  if (maize_storage < 0) {                        // family starves?
    maize_storage = -1;
    [self death];    
    if (DEBUG == 1) printf("DEBUG: agent %d starved\n",tag);
  }
  return self;
}


-(int) calcExpCalUse
{
  // expected family Cal need for 2 years
  int cals = 0;
  
  // seasonal estimates, so 2 years is 8 seasons + other stuff
  cals += 9 * (ages[0]>1?BASE_CAL_WOM:0);
  cals += 9 * (ages[1]>1?BASE_CAL_MAN:0);
  cals += 9 * (num_kids*BASE_CAL_KID);
  return (cals);
}


-(int) calcFarmPl
{
  // hardcoded in assumption about MAIZE_PER is percent of Cals from maize
  int exp_needed;                    // Cals needed in fall for 2 years
  int exp_cal_yieldPl;               // exp yield of land per plot
  int farming_need;

  // Cal needed (in storage)
  exp_needed = (int)(MAIZE_PER*EXP_cal_need) - maize_storage * MAIZE_KG_CAL;

  // exp yield from each plot (4 plots in ha)
  if (EXP_yield<1) EXP_yield = 1;
  exp_cal_yieldPl = EXP_yield * MAIZE_KG_CAL / 4;

  // how many plots needed to be planted to fulfill Cal needs
  farming_need = (int)(exp_needed/exp_cal_yieldPl) + 1;
  if (farming_need < 0) exp_needed = 0;   // if have no need
  return (farming_need);
}


-plantPl: (int *) plots At: (int) num
{
  // planting should be in [0,PLOTS] plots
  int z,t;
  t = [cell[num] getFarmPl] + *plots;
  
  if (farm_pl[num]+*plots>=0){
    if (t>PLOTS){    
      z = PLOTS - t + *plots;                     // how much land left
    } else if (t<0){
      z = *plots - t;
    } else {
      z = *plots;
    }
    if (z>0){
      actual_cal += z*15000;            // cal cost of planting new land
      if (num!=4) actual_cal += z*287;  // movement cost
    }
    //why 15000? why 287? revisit this
    [cell[num] changeFarmPl: z];                          // sow fields
    farm_pl[num] += z;
    tot_plots += z;
    *plots -= z;                                          // return plots left
  }
  z = [cell[num] getFarmPl];
  if (z<0 || z>PLOTS){
    fprintf(stderr,"ERROR: agent %d plotted cell out of bounds\n",tag);
    exit(-1);
  }
  return self;
}

-unplotDX: (int *) dx DY: (int *) dy
{
  int i;
  for (i=-1;i<2;i++)
    if (farm_pl[i+1]>0){
      *dx = i; *dy = -1;
      return self;
    }
  for (i=-1;i<2;i++)
    if (farm_pl[i+7]>0){
      *dx = i; *dy = 1;
      return self;
    }
  if (farm_pl[3]>0){
    *dx = -1; *dy = 0;
    return self;
  } else if (farm_pl[5]>0){
    *dx = 1; *dy = 0;
    return self;
  }
  *dx = *dy = 0;
  return self;
}

-unPlotAll {
  int i;
  // deplant
  for (i=0;i<9;i++){    
    [cell[i] changeFarmPl: -1 * farm_pl[i]];
    tot_plots -= farm_pl[i];
    farm_pl[i] = 0;
  }
  if (tot_plots != 0){
    fprintf(stderr,"ERROR: agent %d unplanted wrong %d\n",tag,tot_plots);
    exit(-1);
  }
  return self;
} 


-updateYields
{
  int i,j,k;

  // update local record of yields for the spring
  for (i=0;i<2;i++)
    for (j=0;j<9;j++)
      past_yield[j][i+1] = past_yield[j][i];
  for (j=0;j<9;j++)
    past_yield[j][0] = [cell[j] getMaizePotential]; // add current yield to history

  // calc yield from what happened this season
  k = 0;
  for (i=0;i<9;i++)
    k+= farm_pl[i] * past_yield[i][0];
  if (tot_plots>0){
    EXP_yield = (k/tot_plots+EXP_yield)/2;
  } else {
    EXP_yield = (EXP_yield+past_yield[4][0])/2;
  }
  return self;
}


// **** movement and search ****

-moveHouse
{
  int dx,dy;
  // deplant and remove self from cell
  [self unPlotAll];
  [cell[4] removeSettler: self];
  dx = dy = 0;
  // find best site in the radius
  [self searchNeighborhoodDX: &dx DY: &dy Rad: 20];
  [self setX: wrapX(dx) Y: wrapY(dy)];
  if (DEBUG == 1 && tag == TAG)
    printf("DEBUG: agent %d moving to (%d,%d)\n",tag,x,y);
  state = 1;
  plot_need = 2 + num_kids / 2;
  return self;
}

-addHousehold
{
  //  id kid;
  int dx,dy;
  dx = dy = 0;
  // want agents to locate same cell as parents for now
  // limit to HOUSE_LIMIT on cell
  if ([cell[4] getNumHouses]>HOUSE_LIMIT)
    [self searchNeighborhoodDX: &dx DY: &dy Rad: 1];
  // adding to the world is like adding to the family tree!
  //  kid = [mySwarm addAgentX: wrapX(dx) Y: wrapY(dy)];
  [mySwarm addAgentX: wrapX(dx) Y: wrapY(dy)];
  // fprintf(stderr,"HI: %d before adding household to tree\n",tag);
  //  [self addKid: kid];
  //  [kid setParent: (id) self];
  // fprintf(stderr,"HI: %d after adding household\n",tag);
  return self;
}


-searchNeighborhoodDX: (int *) a DY: (int *) b Rad: (int) radius
{
  int i,j,k,n,n1,n2,t,max;
  int dx,dy;
  static order[4] = {0,1,2,3};
  k = max = 0;
  for (i=1;i<=radius;i++){
    // make sure search directions are randomized
    for (n=0;n<4;n++){
//    n1 = [uniformRandom rMax: 4];
//    n2 = [uniformRandom rMax: 4];
      n1 = [uniformUnsRand getUnsignedWithMin: 0 withMax: 4];
      n2 = [uniformUnsRand getUnsignedWithMin: 0 withMax: 4];
      t = order[n1];
      order[n1] = order[n2];
      order[n2] = t;
    }
    dx = dy = -1 * radius;
    // north
    for (j=0;j<2*radius;j++,dx++){
      // (dx+j,dy) 
      k = [self evalCellX: wrapX(dx) Y: wrapY(dy) Max: max];
      if (k){
        max = k;
        *a = dx; *b = dy;
      }}
    // east
    for (j=0;j<2*radius;j++,dy++){
      // (dx,dy+j)
      k = [self evalCellX: wrapX(dx) Y: wrapY(dy) Max: max];
      if (k){
        max = k;
        *a = dx; *b = dy;
      }}
    // south
    for (j=0;j<2*radius;j++,dx--){
      // (dx-j,dy)
      k = [self evalCellX: wrapX(dx) Y: wrapY(dy) Max: max];
      if (k){
        max = k;
        *a = dx; *b = dy;
      }}
    // west
    for (j=0;j<2*radius;j++,dy--){
      // (dx,dy-j)
      k = [self evalCellX: wrapX(dx) Y: wrapY(dy) Max: max];
      if (k){
        max = k;
        *a = dx; *b = dy;
      }}
  }
  return self;
}

-(int) evalCellX: (int) inx Y: (int) iny Max: (int) max
{
  // need some notion of weighting crowded land
  int pot,full;
  Cell * c;
//  printf("\tevalCell (%d,%d)\n",inx,iny);
  c = [world getObjectAtX: inx Y: iny];
  pot = [c getMaizePotential];
  full = [c getFarmPl];

  if (pot > max && full < PLOTS)
    return (pot);
  return (0);
}


// **** get agent states ****

-(int) getFamilySize {
  return (num_family);
}

-(int) getTag {
  return (tag);
}

-(int) getFamilyTag {
  return (family_tag);
}

-(int) getFormationDate {
  return (form_date);
}

-(int) getNumPlots {
  return (tot_plots);
}

-(int) getMaizeStorage {
  return (maize_storage);
}

-(int) getFarmPlAt: (int) i {
  return (farm_pl[i]);
}

-(int) getYield {
  return [cell[4] getMaizePotential];
}


-(int) getState {
  return (state);
}

-(int) getMaxStore{
  return (max_store);
}

// Extra bits of display code: setting our colour, drawing on a window.
// This code works, but it'd be better if there were a generic object
// that knew how to draw agents on grids.
-setColor: (Color) c {
  myColor = c;
  return self;
}

-drawSelfOn: (Raster *) r {
  [r drawPointX: x Y: y Color: myColor];
  return self;
}


// debugging stuff

// agent debugging, print out internal state
-debug: (int) key {
  // if key is -1, automatically print out own internal state
  if ( tag==key || key ==-1 ) {
    int i,j;

    // print general info about the cell occupied and household state
    printf("DEBUG:  AGENT %d\n",tag);
    printf("\tinfo:\tlocation (%d,%d) state %d plot_need %d\n",
           x,y,state,plot_need);
    printf("\thist:\tformation %d agentTime %d\n",form_date,agent_time);

    // print out the ages
    printf("\tnums:\ttotal %d kids %d ages: ",num_family,num_kids);
    printf("(%d %d) ",ages[0],ages[1]);
    for (i=2;i<2+num_kids;i++)
      printf("%d ",ages[i]);

    for (i=0,j=0;i<9;i++)
      j += farm_pl[i];
    printf("\n\tfarm:\tplots %d store %d max_store %d ",
           j,maize_storage,max_store);
    printf("E(yield) %d E(use) %d\n",
           EXP_yield,(int)(MAIZE_PER*(float)total_cal/MAIZE_KG_CAL));

    // print cell info
    printf("\tcell:\tplots %d yield %d num_settlers %d\n\n",
           [cell[4] getFarmPl],[cell[4] getMaizePotential],
           [cell[4] getNumHouses]);
  }
  return self;
}

-test {
  int i,j;
  printf("BEFORE: agent %d pos (%d,%d)\n",tag,x,y);
  [self searchNeighborhoodDX: &j DY: &i Rad: 2];
  printf("AFTER:  agent %d dxL %d dy %d\n",tag,j,i);
  return self;
}

@end

