// Turmites application. Copyright (C) 1996-1999 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

#import "TurmiteModelSwarm.h"
#import <simtools.h>

@implementation TurmiteModelSwarm

// These methods provide access to the objects inside the ModelSwarm.
// These objects are the ones visible to other classes via message call.
// In theory we could just let other objects use Probes to read our state,
// but message access is frequently more convenient.

-getTurmiteList {
  return turmiteList;
}

-getTape {
  return tape;
}


// This method isn't normally used, but is convenient when running probes:
// it lets you easily clone a turmite and drag it into the model.

-addTurmite: (Turmite *) tmite {
  [turmiteList addLast: tmite];
  return self;
}

// createBegin: here we set up the default simulation parameters.

+createBegin: (id) aZone {
  TurmiteModelSwarm * obj;
  id <ProbeMap> probeMap;

  // First, call our superclass createBegin - the return value is the
  // allocated TurmiteModelSwarm object.

  obj = [super createBegin: aZone];

  // Now fill in various simulation parameters with default values.

  obj->numMites = 40;
  obj->numStates = 16;
  obj->numTapeStates = 4;
  obj->numMoves = 4;
  obj->worldXSize = 100;
  obj->worldYSize = 100;

  // And build a customized probe map. Without a probe map, the default
  // is to show all variables and messages. Here we choose to
  // customize the appearance of the probe, give a nicer interface.

  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [self class]];
  probeMap = [probeMap createEnd];

  // Add in a bunch of variables, one per simulation parameter

  [probeMap addProbe: [probeLibrary getProbeForVariable: "numMites"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "numStates"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "numTapeStates"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "numMoves"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "worldXSize"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "worldYSize"
				    inClass: [self class]]];

  // And one method, the "addTurmite" method for a probe demo.
  // we also hide the return value for this message probe, just for nicety.

  [probeMap addProbe: [[probeLibrary getProbeForMessage: "addTurmite:"
			     inClass: [self class]]
			setHideResult: 1]];

  // Now install our custom probeMap into the probeLibrary.

  [probeLibrary setProbeMap: probeMap For: [self class]];
  
  return obj;
}

// createEnd: we could create some objects here if we knew we needed
// them. But this method is called before the user is allowed to fill
// in any customization of the model, so we defer most object creation
// to later. (In this example, this method does nothing at all and could
// just be inherited. But it's here to show you a place to customize.)

-createEnd {
  return [super createEnd];
}

// Now it's time to build the model objects. We use various parameters
// inside ourselves to choose how to create things.

-buildObjects {
  int i;

  // allow our parent class to build anything.

  [super buildObjects];
  
  // First, set up objects used to represent the environment.
  // The TapeSpace agent represents the spatial property of tape.
  // It is initialized via various model parameters.

  tape = [TapeSpace createBegin: [self getZone]];
  [tape setSizeX: worldXSize Y: worldYSize];
  [tape setNumStates: numTapeStates];
  tape = [tape createEnd];

  [tape initializeLattice];

  // Create a list to keep track of the turmites in the model.

  turmiteList = [List create: [self getZone]];
  
  // Create turmites themselves. This is a fairly complex step, as is
  // appropriate: the turmites are essential aspects of the simulation.

  // Now a loop to create a bunch of turmites.

  for (i = 0; i < numMites; i++) {
    Turmite * tmite;

    // Create the turmites, set the creation time variables

    tmite = [Turmite createBegin: [self getZone]];
    [tmite setTape: tape];
    [tmite setNumStates: numStates];
    [tmite setNumTapeStates: numTapeStates];
    [tmite setNumMoves: numMoves];
    tmite = [tmite createEnd];

    // Add the mite to the end of the list.

    [turmiteList addLast: tmite];

    // Now initialize the tmite's state, position, and trans-table randomly.

    [tmite setRandom];
  }

  return self;
}

// Here is where the model schedule is built, the data structures
// that define the simulation of time in the mode. The core is an
// actionGroup that has a list of actions. That's then put in a Schedule.

-buildActions {

  [super buildActions];
  
  // Create the list of simulation actions. We put these in an action
  // group, because we want these actions to be executed in a specific
  // order, but these steps should take no (simulated) time. The
  // M(foo) means "The message called <foo>". You can send a message
  // To a particular object, or ForEach object in a collection.

  modelActions = [ActionGroup create: [self getZone]];

  // If we wanted to have the tape perform any actions on
  // its own, we'd call stepRule on the tape here like this:
  // [modelActions createActionTo:      tape message: M(stepRule)];

  [modelActions createActionForEach: turmiteList message: M(step)];  
  [modelActions createActionTo:      tape 	 message: M(updateLattice)];

  // Then we create a schedule that executes the modelActions. modelActions
  // is an ActionGroup, by itself it has no notion of time. In order to
  // have it executed in time, we create a Schedule that says to use
  // the modelActions ActionGroup at particular times.
  // This schedule has a repeat interval of 1, it will loop every time step.
  // The action is executed at time 0 relative to the beginning of the loop.

  // This is a simple schedule, with only one action that is just
  // repeated every time. See mousetraps for more complicated schedules.
  
  modelSchedule = [Schedule createBegin: [self getZone]];
  [modelSchedule setRepeatInterval: 1];
  modelSchedule = [modelSchedule createEnd];
  [modelSchedule at: 0 createAction: modelActions];

  return self;
}

// Now set up the model's activation. swarmContext indicates where
// we're being started in - typically, this model is run as a subswarm
// of an observer swarm.

-activateIn: (id) swarmContext {

  // First, activate ourselves via the superclass activateIn: method.
  // Just pass along the context: the activity library does the right thing.

  [super activateIn: swarmContext];

  // Now activate our own schedule.

  [modelSchedule activateIn: self];

  // Finally, return our activity.

  return [self getActivity];
}

@end
