/* 
  Link.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  30-SEP-1996

*/

#import <simtools.h>

#import "SCLGlobals.h"
#import "SCLModelSwarm.h"

#import "AgentManager.h"
#import "PRNG.h"

#import "Hole.h"
#import "Substrate.h"
#import "Bond.h"
#import "Catalyst.h"
#import "Link.h"

@implementation Link


static double linkMobilityFactor;
static Color linkColor;
static Color absorbedSubstrateColor;
static Color disintegratingColor;

/*
  See docs in Particle.m
*/

+(void) setupProbeMap: (id) aZone {
  id <ProbeMap> probeMap;

  probeMap = [probeLibrary getProbeMapFor: [self class]];

  [probeMap dropProbeForMessage: "getBondCount"];
  [probeMap dropProbeForMessage: "allowBondAtNeighbor:"];
  [probeMap dropProbeForMessage: "setBond:atNeighbor:"];
  [probeMap dropProbeForMessage: "getBondAtNeighbor:"];
  [probeMap dropProbeForMessage: "findOrientation:"];
  [probeMap dropProbeForMessage: "createBondToNeighbor:"];
  [probeMap dropProbeForMessage: "attemptBondAtNeighbor:"];
  [probeMap dropProbeForMessage: "clearBondAtNeighbor:"];
  [probeMap dropProbeForMessage: "clearBond:"];
  [probeMap dropProbeForMessage: "doBonding"];
  [probeMap dropProbeForMessage: "doAbsorption"];
  [probeMap dropProbeForMessage: "attemptEmission"];
  [probeMap dropProbeForMessage: "doEmission"];
  [probeMap dropProbeForMessage: "attemptDisintegration"];
  [probeMap dropProbeForMessage: "doDisintegration"];
  [probeMap dropProbeForMessage: "warp"];
  [probeMap dropProbeForMessage: "doMotion"];
  [probeMap dropProbeForMessage: "canMove"];
  [probeMap dropProbeForMessage: "step"];
  [probeMap dropProbeForMessage: "drawSelfOn:"];
  [probeMap dropProbeForMessage: "saveTo:"];
  [probeMap dropProbeForMessage: "loadFrom:"];
  [probeMap dropProbeForVariable: "absorbedSubstrate"];
  [probeMap dropProbeForVariable: "disintegrating"];
  [probeMap dropProbeForVariable: "bondCount"];
}


+(void) setClassMobilityFactor: (double) inMobilityFactor {
  linkMobilityFactor = inMobilityFactor;
}

+(double) getClassMobilityFactor {
  return(linkMobilityFactor);
}

+(void) setClassColor: (Color) color {
  linkColor = color;
}

+(Color) getClassColor {
  return(linkColor);
}

+(void) setAbsorbedSubstrateColor: (Color) color {
  absorbedSubstrateColor = color;
}

+(Color) getAbsorbedSubstrateColor {
  return(absorbedSubstrateColor);
}

+(void) setDisintegratingColor: (Color) color {
  disintegratingColor = color;
}

+(Color) getDisintegratingColor {
  return(disintegratingColor);
}

static BOOL chainInhibitBondFlag;

+(void) setChainInhibitBondFlag: (BOOL) value {
  chainInhibitBondFlag = value;
}

+(BOOL) getChainInhibitBondFlag {
  return(chainInhibitBondFlag);
}

static BOOL catInhibitBondFlag;

+(void) setCatInhibitBondFlag: (BOOL) value {
  catInhibitBondFlag = value;
}

+(BOOL) getCatInhibitBondFlag {
  return(catInhibitBondFlag);
}

static double disintegrationProbability;

+(void) setDisintegrationProbability: (double) inDisintegrationProbability {
  disintegrationProbability = inDisintegrationProbability;
}

+(double) getDisintegrationProbability {
  return(disintegrationProbability);
}

static double chainInitiateProbability;

+(void) setChainInitiateProbability: (double) probability {
  chainInitiateProbability = probability;
}

+(double) getChainInitiateProbability {
  return(chainInitiateProbability);
}

static double chainExtendProbability;

+(void) setChainExtendProbability: (double) probability {
  chainExtendProbability = probability;
}

+(double) getChainExtendProbability {
  return(chainExtendProbability);
}

static double chainSpliceProbability;

+(void) setChainSpliceProbability: (double) probability {
  chainSpliceProbability = probability;
}

+(double) getChainSpliceProbability {
  return(chainSpliceProbability);
}

static double absorptionProbability;

+(void) setAbsorptionProbability: (double) inAbsorptionProbability {
  absorptionProbability = inAbsorptionProbability;
}

+(double) getAbsorptionProbability {
  return(absorptionProbability);
}

static double emissionProbability;

+(void) setEmissionProbability: (double) inEmissionProbability {
  emissionProbability = inEmissionProbability;
}

+(double) getEmissionProbability {
  return(emissionProbability);
}



-(int) getBondCount {
  return bondCount;
}

-(BOOL) neighboringChain {
  /* 
    This is loosely inspired by the PROTOBIO/exp27.for 
    code. Returns YES if there is at least 1 doubly bonded
    link in the Moore neighborhood of self...
  */
  BOOL chainFound;
  BOOL done;
  neighbor_t neighbor;
  Agent2d *neighborAgent;
  Link *neighborLink;

  done = NO;
  chainFound = NO;
  neighbor = North;
  while (!done) {
    neighborAgent = [self getAgentAtNeighbor: neighbor];
    if ([neighborAgent isMemberOf: [Link class]]) {
      neighborLink = (Link *) neighborAgent;
      if ([neighborLink getBondCount] == 2) 
        chainFound = YES;
    }
    neighbor = [mooreNeighbor getNext: neighbor];
    done = (chainFound || (neighbor == North));
  }

  return (chainFound);
}

-(BOOL) neighboringCat {
  BOOL catFound;
  BOOL done;
  neighbor_t neighbor;
  Agent2d *neighborAgent;

  done = NO;
  catFound = NO;
  neighbor = North;
  while (!done) {
    neighborAgent = [self getAgentAtNeighbor: neighbor];
    catFound = [neighborAgent isMemberOf: [Catalyst class]];
    neighbor = [mooreNeighbor getNext: neighbor];
    done = (catFound || (neighbor == North));
  }

  return (catFound);
}

-(BOOL) mayBond {
  BOOL inhibit;

  inhibit = (bondCount >= 2);

  if (!inhibit && (bondCount == 0)) 
    inhibit = 
      ([Link getChainInhibitBondFlag] && [self neighboringChain]) ||
      ([Link getCatInhibitBondFlag] && [self neighboringCat]);

  return (!inhibit);
}

-(BOOL) allowBondAtNeighbor: (neighbor_t) neighbor {
  /* You can't form a new bond with an angle < 90 deg relative
     to an existing bond; so you can't bond where there is an
     existing bond, or at the next or previous Moore neighbors. */

  return ( 
    ([self getBondAtNeighbor: neighbor] == nil) &&
    ([self getBondAtNeighbor: [mooreNeighbor getNext: neighbor]] == nil) &&
    ([self getBondAtNeighbor: [mooreNeighbor getPrevious: neighbor]] == nil));
}


-setBond: (Bond *) bond atNeighbor: (neighbor_t) neighbor {
  /* This method *should* only be invoked if it is
     already known that it is "allowed" to set the
     requested bond; there are *no* checks here!!! */

  switch (neighbor) {
    case North:     bondNorth     = bond;    break;
    case NorthEast: bondNorthEast = bond;    break;
    case East:      bondEast      = bond;    break;
    case SouthEast: bondSouthEast = bond;    break;
    case South:     bondSouth     = bond;    break;
    case SouthWest: bondSouthWest = bond;    break;
    case West:      bondWest      = bond;    break;
    case NorthWest: bondNorthWest = bond;    break;
    default:
      [InternalError raiseEvent:
       "Out-of-range neighbor (%d) encountered.\n", neighbor];

  }

  bondCount++;

  return self;
}

-clearBondAtNeighbor: (neighbor_t) neighbor {
  if ([self getBondAtNeighbor: neighbor] == nil) 
    [InternalError raiseEvent:
      "Attempt to clear nil bond on link %d at neighbor %d.\n",
      self, neighbor];
  else {
    switch (neighbor) {
      case North:     bondNorth     = nil;    break;
      case NorthEast: bondNorthEast = nil;    break;
      case East:      bondEast      = nil;    break;
      case SouthEast: bondSouthEast = nil;    break;
      case South:     bondSouth     = nil;    break;
      case SouthWest: bondSouthWest = nil;    break;
      case West:      bondWest      = nil;    break;
      case NorthWest: bondNorthWest = nil;    break;
      default:
	[InternalError raiseEvent:
	 "Out-of-range neighbor (%d) encountered.\n", neighbor];
    }
    bondCount--;
  }

  return self;
}

-(Bond *) getBondAtNeighbor: (neighbor_t) neighbor {
  Bond *bond;

  switch (neighbor) {
  case North:     bond = bondNorth;        break;
  case NorthEast: bond = bondNorthEast;    break;
  case East:      bond = bondEast ;        break;
  case SouthEast: bond = bondSouthEast ;   break;
  case South:     bond = bondSouth;        break;
  case SouthWest: bond = bondSouthWest;    break;
  case West:      bond = bondWest ;        break;
  case NorthWest: bond = bondNorthWest ;   break;
  default:
    [InternalError raiseEvent:
     "Out-of-range neighbor (%d) encountered.\n", neighbor];
    bond = (Bond *) nil; // Defensive...
  }

  return bond;
}

-(orientation_t) findOrientation: (neighbor_t) neighbor {
  orientation_t orientation;

  switch (neighbor) {
  case North:     
  case South:     orientation = NorthSouth;         break;
  case East:     
  case West:      orientation = EastWest;           break;
  case NorthEast:     
  case SouthWest: orientation = NorthEastSouthWest; break;
  case NorthWest:     
  case SouthEast: orientation = NorthWestSouthEast; break;
  default:
    [InternalError raiseEvent:
     "Out-of-range neighbor (%d) encountered.\n", neighbor];
    orientation = NorthSouth; // Defensive...
  }

  return orientation;
}


-(Bond *) createBondToNeighbor: (neighbor_t) neighbor {
  Bond *bond;

  if ([self mayBond] && [self allowBondAtNeighbor: neighbor]) 
  {
    bond = [bondMgr createAgent];
    [bond setLink0: self];
    [bond setNeighbor0: neighbor];
    [bond setOrientation: [self findOrientation: neighbor]];
    [self setBond: bond atNeighbor: neighbor];
  }
  else
  {
    [InternalError raiseEvent: 
      "Attempt to create illegal bond at neighbor %d of link %d.\n",
      neighbor, self];
    bond = (Bond *) nil; // Defensive...
  }

  return bond;
}


-attemptBondAtNeighbor: (neighbor_t) neighbor {
  Particle *neighborParticle;
  Link *neighborLink;
  Bond *bond;
  neighbor_t oppositeNeighbor;

  if ([self allowBondAtNeighbor: neighbor]) {
    neighborParticle = [self getAgentAtNeighbor: neighbor];
    if (neighborParticle == nil) 
      [InternalError raiseEvent:
	 "nil encountered at neighbor %d of particle %d.\n",
	 neighbor, self];
    else {
      if ([neighborParticle isMemberOf: [Link class]]) {
	neighborLink = (Link *) neighborParticle;
        oppositeNeighbor = [mooreNeighbor getOpposite: neighbor];
        if ([neighborLink mayBond] && 
            [neighborLink allowBondAtNeighbor: oppositeNeighbor]) {
	  bond = [neighborLink createBondToNeighbor: oppositeNeighbor];
	  [bond setLink1: self];
	  [bond setNeighbor1: neighbor];
	  [self setBond: bond atNeighbor: neighbor];
	}
      }
    }
  }

  return self;
}


-attemptBondNorth {
  [self attemptBondAtNeighbor: North];
  [worldManager update];

  return self;
}

-attemptBondNorthEast {
  [self attemptBondAtNeighbor: NorthEast];
  [worldManager update];

  return self;
}

-attemptBondEast {
  [self attemptBondAtNeighbor: East];
  [worldManager update];

  return self;
}

-attemptBondSouthEast {
  [self attemptBondAtNeighbor: SouthEast];
  [worldManager update];

  return self;
}

-attemptBondSouth {
  [self attemptBondAtNeighbor: South];
  [worldManager update];

  return self;
}

-attemptBondSouthWest {
  [self attemptBondAtNeighbor: SouthWest];
  [worldManager update];

  return self;
}

-attemptBondWest {
  [self attemptBondAtNeighbor: West];
  [worldManager update];

  return self;
}

-attemptBondNorthWest {
  [self attemptBondAtNeighbor: NorthWest];
  [worldManager update];

  return self;
}

-doBonding {
  neighbor_t neighbor, oppositeNeighbor;
  Particle *neighborParticle;
  Link *neighborLink;
  Bond *bond;
  int combinedBondCount;
  double bondingProbability;

  if ([self mayBond]) {
    neighbor = [defaultNeighbor getRandom];
    if ([self allowBondAtNeighbor: neighbor]) {
      neighborParticle = [self getAgentAtNeighbor: neighbor];
      if (neighborParticle == nil) 
        [InternalError raiseEvent:
	  "nil encountered at neighbor %d of particle %d.\n",
	  neighbor, self];
      else {
        if ([neighborParticle isMemberOf: [Link class]]) {
	  neighborLink = (Link *) neighborParticle;
          if ([neighborLink mayBond]) {
	    oppositeNeighbor = [mooreNeighbor getOpposite: neighbor];
	    if ([neighborLink allowBondAtNeighbor: oppositeNeighbor]) {
	      combinedBondCount = bondCount + [neighborLink getBondCount];
	      switch (combinedBondCount) {
		case (0) : bondingProbability = chainInitiateProbability; break;
		case (1) : bondingProbability = chainExtendProbability; break;
		case (2) : bondingProbability = chainSpliceProbability; break;
		default:
		  [InternalError raiseEvent:
		    "Invalid combinedBondCount: %d \n", combinedBondCount];
		  bondingProbability = 0.0; // Defensive...
	      }
	      if ([prng getTossWithProb: bondingProbability]) {
		bond = [neighborLink createBondToNeighbor: oppositeNeighbor];
		[bond setLink1: self];
		[bond setNeighbor1: neighbor];
		[self setBond: bond atNeighbor: neighbor];
	      }
	    }
	  }
	}
      }
    }
  }

  return self;
}

-clearBond: (Bond *) bond {
  neighbor_t neighbor;
  BOOL done;

  neighbor = North;
  done = NO;
  while (!done) {
    if ([self getBondAtNeighbor: neighbor] == bond) {
      [self clearBondAtNeighbor: neighbor];
      done = YES;
    }
    else {
      neighbor = [mooreNeighbor getNext: neighbor];
      if (neighbor == North)
        [InternalError raiseEvent:
           "Attempt to clear non-existent bond %d on link %d.\n",
           bond, self];
    }
  }

  return self;
}

-dropBonds {
  neighbor_t neighbor;
  Bond *bond;

  neighbor = North;
  while (bondCount > 0) {
    bond = [self getBondAtNeighbor: neighbor];
    if (bond != nil) {
      [bond clearLinks];
      [bondMgr dropAgent: bond];
    }
    if (bondCount > 0) {
      neighbor = [mooreNeighbor getNext: neighbor];
      if (neighbor == North) 
        [InternalError raiseEvent:
           "Inconsistent bond count %d on link %d.\n",
           bondCount, self];
    }
  }

  return self;
}

-doAbsorption {
  neighbor_t neighbor;
  Particle *neighborParticle;
  int neighborX, neighborY;
  Hole *hole;

  if (!absorbedSubstrate) {
    if ([prng getTossWithProb: absorptionProbability]) {
      neighbor = [defaultNeighbor getRandom];
      neighborParticle = [self getAgentAtNeighbor: neighbor];
      if (neighborParticle == nil)
	[InternalError raiseEvent:
	   "nil encountered at neighbor %d of particle %d.\n",
	   neighbor, self];
      else {
	if ([neighborParticle isMemberOf: [Substrate class]]) {
	  neighborX = [neighborParticle getX];
	  neighborY = [neighborParticle getY];
	  [substrateMgr dropAgent: (Substrate *) neighborParticle];
	  hole = [holeMgr createAgent];
	  [hole unwarpToX: neighborX Y: neighborY];
          absorbedSubstrate = YES;
	}
      }
    }
  }

  return self;
}

-attemptEmission {
  neighbor_t neighbor;
  Particle *neighborParticle;
  int neighborX, neighborY;
  Substrate *substrate;

  neighbor = [defaultNeighbor getRandom];
  neighborParticle = [self getAgentAtNeighbor: neighbor];
  if (neighborParticle == nil)
    [InternalError raiseEvent:
       "nil encountered at neighbor %d of particle %d.\n",
       neighbor, self];
  else {
    if ([neighborParticle isMemberOf: [Hole class]]) {
      neighborX = [neighborParticle getX];
      neighborY = [neighborParticle getY];
      [holeMgr dropAgent: (Hole *) neighborParticle];
      substrate = [substrateMgr createAgent];
      [substrate unwarpToX: neighborX Y: neighborY];
      absorbedSubstrate = NO;
    }
  }

  return self;
}

-doEmission {
  if (absorbedSubstrate) 
    if ([prng getTossWithProb: emissionProbability]) 
      [self attemptEmission];

  return self;
}


-attemptDisintegration {
  neighbor_t neighbor;
  Particle *neighborParticle;
  int myX, myY, neighborX, neighborY;
  Substrate *substrate0, *substrate1;

  if (absorbedSubstrate) 
    [self attemptEmission];
  if (!absorbedSubstrate) {
    neighbor = [defaultNeighbor getRandom];
    neighborParticle = [self getAgentAtNeighbor: neighbor];
    if (neighborParticle == nil)
      [InternalError raiseEvent:
	 "nil encountered at neighbor %d of particle %d.\n",
	 neighbor, self];
    else {
      if ([neighborParticle isMemberOf: [Hole class]]) {
	/* We can only *actually* disintegrate if we have a hole to drop
	   the extra substrate into... */
	neighborX = [neighborParticle getX];
	neighborY = [neighborParticle getY];
	[holeMgr dropAgent: (Hole *) neighborParticle];
	substrate0 = [substrateMgr createAgent];
	[substrate0 unwarpToX: neighborX Y: neighborY];

	myX = x;
	myY = y;
        [self dropBonds];
	[linkMgr dropAgent: self];
	substrate1 = [substrateMgr createAgent];
	[substrate1 unwarpToX: myX Y: myY];
      }
    }
  }

  return self;
}

-doDisintegration {
  if (!disintegrating)
    disintegrating = 
      [prng getTossWithProb: disintegrationProbability];

  return self;
}



-warp {
  if (bondCount > 0) 
    [self dropBonds];
  [super warp];

  return self;
}

-(BOOL) canMove {
  if (bondCount > 0)
    return NO;
  else
    return [super canMove];
}

-step {
  [self doMotion];
  [self doBonding];
  [self doDisintegration];

  if (disintegrating)
    [self attemptDisintegration];
  else {
    [self doEmission];
    [self doAbsorption];
  }

  return self;
}

-drawSelfOn: (id <Raster>) r {
  int rasterX, rasterY;

  if ((particleColor == linkColor) || 
      (particleColor == disintegratingColor)) 
  {
    if (disintegrating) particleColor = disintegratingColor;
      else particleColor = linkColor;
  }

  rasterX = (8 * x);
  rasterY = (8 * y);

  [r drawPointX: rasterX   Y: rasterY++ Color: particleColor];
  [r drawPointX: rasterX   Y: rasterY++ Color: particleColor];
  [r drawPointX: rasterX   Y: rasterY++ Color: particleColor];
  [r drawPointX: rasterX   Y: rasterY++ Color: particleColor];
  [r drawPointX: rasterX++ Y: rasterY   Color: particleColor];
  [r drawPointX: rasterX++ Y: rasterY   Color: particleColor];
  [r drawPointX: rasterX++ Y: rasterY   Color: particleColor];
  [r drawPointX: rasterX++ Y: rasterY   Color: particleColor];
  [r drawPointX: rasterX   Y: rasterY-- Color: particleColor];
  [r drawPointX: rasterX   Y: rasterY-- Color: particleColor];
  [r drawPointX: rasterX   Y: rasterY-- Color: particleColor];
  [r drawPointX: rasterX   Y: rasterY-- Color: particleColor];
  [r drawPointX: rasterX-- Y: rasterY   Color: particleColor];
  [r drawPointX: rasterX-- Y: rasterY   Color: particleColor];
  [r drawPointX: rasterX-- Y: rasterY   Color: particleColor];
  [r drawPointX: rasterX   Y: rasterY   Color: particleColor];


  if (absorbedSubstrate) {
    rasterX = (8 * x) + 1;
    rasterY = (8 * y) + 1;

    [r drawPointX: rasterX   Y: rasterY++ Color: absorbedSubstrateColor];
    [r drawPointX: rasterX   Y: rasterY++ Color: absorbedSubstrateColor];
    [r drawPointX: rasterX++ Y: rasterY   Color: absorbedSubstrateColor];
    [r drawPointX: rasterX++ Y: rasterY   Color: absorbedSubstrateColor];
    [r drawPointX: rasterX   Y: rasterY-- Color: absorbedSubstrateColor];
    [r drawPointX: rasterX   Y: rasterY-- Color: absorbedSubstrateColor];
    [r drawPointX: rasterX-- Y: rasterY   Color: absorbedSubstrateColor];
    [r drawPointX: rasterX   Y: rasterY   Color: absorbedSubstrateColor];
  }

  return self;
}

-saveTo: (id <OutFile>) file {
  /* Note that we do *not* save bond information here; that info
     is already implicit in the bond agent list; of course, it
     would be more conservative to save the redundancies and then
     crosscheck on loading - but, for the time being at least,
     I judge that to be an unwarrananted complication... */

  [super saveTo: file];

  [file putInt: absorbedSubstrate];
  [file putString: " "];
  [file putInt: disintegrating];  
  [file putString: " "];

  return self;
}

-loadFrom: (id <InFile>) file {
  int intFlag;

  [super loadFrom: file];

  [file getInt: &intFlag];
  if ((intFlag != YES) && (intFlag != NO))
    [InternalError raiseEvent:
      "Loaded BOOL value (%d) not YES or NO.", intFlag];
  absorbedSubstrate = (BOOL) intFlag; 


  [file getInt: &intFlag];
  if ((intFlag != YES) && (intFlag != NO))
    [InternalError raiseEvent:
      "Loaded BOOL value (%d) not YES or NO.", intFlag];
   disintegrating = (BOOL) intFlag;  

  return self;
}


@end
