/*
  PRNG.m

   Barry McMullin <mcmullin@eeng.dcu.ie>
   09-OCT-1996
*/

#import <stdlib.h>
#import <math.h>

#import "PRNG.h"


@implementation PRNG

#define unsignedRange (2147483646) /* 2^31 - 2 */

#define A (16807L)
#define MODULUS (unsignedRange + 1)
#define Q (127773L)
#define R (2836L)


-createEnd {
  PRNG *finalSelf;

  finalSelf = [super createEnd];

  finalSelf->state = 1;

  finalSelf->doubleRange = (double)(unsignedRange);
  /* Cache now to avoid recalculating on
     every call to -getTossWithProb:  */

  finalSelf->count = 0U;

  return finalSelf;
}

-setPRNGstate: (unsigned) inState {
  state = inState;

  return self;
}

-(unsigned) getPRNGstate {
  return state;
}

-setPRNGcount: (unsigned) inCount {
  count = inCount;

  return self;
}

-(unsigned) getPRNGcount {
  return count;
}

-saveTo: (OutFile *) file {
  [file putString: "# PRNG state...\n"];
  [file putInt: state]; 
  [file putNewLine];
  [file putString: "# PRNG count...\n"];
  [file putInt: count]; 
  [file putNewLine];
    // Using putInt is horrible; but putUnsigned
    // is not (yet) implemented...

  return self;
}

-loadFrom: (InFile *) file {
  [file skipLine]; // "# PRNG state...\n"
  [file getInt: &(state)]; 
  [file skipLine];
  [file skipLine]; // "# PRNG count...\n"
  [file getInt: &count]; 
  [file skipLine];
    // Using getInt is horrible; but getUnsigned
    // is not (yet) implemented...

  return self;
}

-(unsigned) getPRNGsample {
  unsigned sample;
  int test;
  ldiv_t lohi;

  lohi = ldiv(state, Q);
  test = A * lohi.rem - R * lohi.quot;
  if (test > 0) state = test;
  else state = test + MODULUS;

  sample = state - 1;

  count++;
  if (count >= unsignedRange)
    [WarningMessage raiseEvent: 
    "PRNG: count (%u) >= range (%u)\n", 
    count, unsignedRange];
    
  return sample;
}


-(BOOL) getTossWithProb: (double) p {
  unsigned cutoff;
  unsigned sample;
  BOOL result;

  if ((p < 0.0) || (p > 1.0))
    [InternalError raiseEvent:
    "Invalid probability value: %e\n", p];

  cutoff = (unsigned) floor((p * doubleRange) + 0.5);
    /* This aritmetic is a little tricky; we add 0.5 to
       round rather than truncate.  It should work OK in the
       "normal" case where we know that
       doubleRange is certainly < 2^32 and type double is
       IEEE (mantissa is 53 bits).  Specifically, if p == 1.0
       this should correctly yield unsignedRange... */

  sample = [self getPRNGsample];
  result = (sample < cutoff);

  return(result);
}

-(unsigned) getUniformWithRange: (unsigned) n {
    unsigned cutoff;
    unsigned sample;
    unsigned result;

  if ((n < 1) || (n > unsignedRange))
    [InternalError raiseEvent:
    "Invalid range parameter: %u\n", n];

  cutoff = (unsignedRange/n) * n; 

  do (sample = [self getPRNGsample]);
  while (sample >= cutoff);

  /* We discard values of sample >= cutoff because, in general, the
     range of PMMLCG1 will not be an integral multiple of the
     range required; in that case, -getUniformWithRange can only 
     properly deliver a uniform (pseudo) probability function by
     discarding the "extra" values covered by PMMLCG1. The number
     of such "extra" values will always be less than n.  So, as
     long as n is much less than unsignedRange, the probability of
     having to discard will be small and we will "almost
     always" exit the loop immediately (i.e. after one
     iteration).  But even in the worst case (n marginally
     bigger than unsignedRange/2) the probability of having to
     discard has a peak value of 0.5 and the expected number
     of iterations is still only 2.
   */

  result = (sample % n);

  return(result);
}



@end
