/* 
  Bond.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  01-OCT-1996


*/


#import "SCLGlobals.h"
#import "Link.h"
#import "Bond.h"


@implementation Bond

static Color bondColor = 5;
static double bondDecayProbability = 0.1;
// Pseudo Class variables...

+(void) setClassColor: (Color) color {
  bondColor = color;
}

+(Color) getClassColor {
  return bondColor;
}

+(void) setBondDecayProbability: (double) probability {
  bondDecayProbability = probability;
}

+(double) getBondDecayProbability {
  return bondDecayProbability;
}

-setLink0: (Link *) link {
  link0 = link;

  return self;
}

-(Link *) getLink0 {
  return link0;
}

-setNeighbor0: (neighbor_t) neighbor {
  neighbor0 = neighbor;

  return self;
}

-(neighbor_t) getNeighbor0 {
  return neighbor0;
}

-setLink1: (Link *) link {
  link1 = link;

  return self;
}

-(Link *) getLink1 {
  return link1;
}

-setNeighbor1: (neighbor_t) neighbor {
  neighbor1 = neighbor;

  return self;
}

-(neighbor_t) getNeighbor1 {
  return neighbor1;
}


-setOrientation: (orientation_t) inOrientation {
  orientation = inOrientation;

  return self;
}

-(orientation_t) getOrientation {
  return orientation;
}

-clearLinks {
  if ((link0 == nil) || (link1 == nil))
    [InternalError raiseEvent:
       "Nil link encountered on bond %d.\n"
       "link0: %d  link1: %d\n",
       self, link0, link1];

  [link0 clearBond: self];
  link0 = nil;
  neighbor0 = Center;
  [link1 clearBond: self];
  link1 = nil;
  neighbor0 = Center;

  return self;
}


-drawSelfOn: (Raster *) r {
  int x0, x1, y0, y1;
  int rasterX, rasterY;
  int i;

  if ((link0 == nil) || (link1 == nil))
    [InternalError raiseEvent:
       "Nil link encountered on bond %d.\n"
       "link0: %d  link1: %d\n",
       self, link0, link1];

  x0 = [link0 getX];
  x1 = [link1 getX];

  rasterX = (4 * (x0 + x1)) + 1; // Simple case...

  // Now allow for bond at edge of raster...
  if (x0 != x1) {
    if (x0 == 0) {
      if (x1 != 1)
	rasterX = (8 * x1) + 5;
    }
    else {
      if (x1 == 0) {
	if (x0 != 1)
	  rasterX = (8 * x0) + 5;
      }
    }
  }

  y0 = [link0 getY];
  y1 = [link1 getY];

  rasterY = (4 * (y0 + y1)) + 1; // Simple case...

  // Now allow for bond at edge of raster...
  if (y0 != y1) {
    if (y0 == 0) {
      if (y1 != 1)
	rasterY = (8 * y1) + 5;
    }
    else {
      if (y1 == 0) {
	if (y0 != 1)
	  rasterY = (8 * y0) + 5;
      }
    }
  }

  switch (orientation) {
  case NorthSouth :
    rasterX++;
    for (i = 0; i < 3; i++) 
      [r drawPointX: rasterX Y: rasterY++ Color: bondColor];
    break;

  case EastWest:
    rasterY++;
    for (i = 0; i < 3; i++) 
      [r drawPointX: rasterX++ Y: rasterY Color: bondColor];
    break;

  case NorthEastSouthWest:
    rasterX += 2;
    for (i = 0; i < 3; i++) 
      [r drawPointX: rasterX-- Y: rasterY++ Color: bondColor];
    break;

  case NorthWestSouthEast:
    for (i = 0; i < 3; i++) 
      [r drawPointX: rasterX++ Y: rasterY++ Color: bondColor];
    break;

  default:
    [InternalError raiseEvent:
     "Out-of-range orientation (%d) encountered.\n", orientation];
  }

  return self;
}

-(void) drop {
  if ((link0 != nil) || (link1 != nil))
    [self clearLinks];

  [super drop];
}

-step {
  if ([prng getTossWithProb: bondDecayProbability]) {
    [self clearLinks];
    [bondMgr dropAgent: self];
  }

  return self;
}

-saveTo: (OutFile *) file {
  if ((link0 == nil) || (link1 == nil))
    [InternalError raiseEvent:
       "Attempt to save bond %d with nil link(s).\n"
       "link0: %d  link1: %d\n",
       self, link0, link1];

  [file putInt: [link0 getX]];
  [file putString: " "];
  [file putInt: [link0 getY]];
  [file putString: " "];
  [file putInt: neighbor0];
  [file putString: " "];

  [file putInt: [link1 getX]];
  [file putString: " "];
  [file putInt: [link1 getY]];
  [file putString: " "];
  [file putInt: neighbor1];
  [file putString: " "];

  [file putInt: orientation];
  [file putString: " "];

  return self;
}

-loadFrom: (InFile *) file {
  int x, y;
  int neighborInt;
  int orientationInt;

  if ((link0 != nil) || (link1 != nil))
    [InternalError raiseEvent:
       "Attempt to load bond %d with non-nil link(s).\n"
       "link0: %d  link1: %d\n",
       self, link0, link1];

  [file getInt: &x];
  [file getInt: &y];
  [file getInt: &neighborInt];
  neighbor0 = (neighbor_t) neighborInt;
  if (![mooreNeighbor isValidNeighborCode: neighbor0])
    [InternalError raiseEvent:
      "Invalid neighbor code %d loaded.\n", neighbor0];

  link0 = [world getObjectAtX: x Y: y];
  if (![link0 isMemberOf: [Link class]])
    [InternalError raiseEvent:
       "Attempt to bond non-link particle %d at (%d,%d)\n",
       link0, x, y];
  [link0 setBond: self atNeighbor: neighbor0];

  [file getInt: &x];
  [file getInt: &y];
  [file getInt: &neighborInt];
  neighbor1 = (neighbor_t) neighborInt;
  if (![mooreNeighbor isValidNeighborCode: neighbor1])
    [InternalError raiseEvent:
      "Invalid neighbor code %d loaded.\n", neighbor1];

  link1 = [world getObjectAtX: x Y: y];
  if (![link1 isMemberOf: [Link class]])
    [InternalError raiseEvent:
       "Attempt to bond non-link particle %d at (%d,%d)\n",
       link1, x, y];
  [link1 setBond: self atNeighbor: neighbor1];

  [file getInt: &orientationInt];
  orientation = (orientation_t) orientationInt;
  switch (orientation) {
    case NorthSouth:
    case EastWest:
    case NorthEastSouthWest:
    case NorthWestSouthEast:
      break;
    default:
      [InternalError raiseEvent: 
	"Invalid orientation code %d loaded.\n", orientation];
  }

  return self;
}


@end
