// main.m
// main for the game-theory world for 2x2 games with agent strategies

// Swarm header files
#import <swarmtcl/global.h>
#import <swarmtcl/TkExtra.h>
#import <swarmtcl/BLTGraph.h>
#import <swarmtcl/Raster.h>
#import <simtools/SimControl.h> // For simulation parameters
#import <simtools/SimParameters.h>
#import <simtools/global.h>
#import <collections/collections.h>

#import "gameworld.h"
#import "World.h"
#import "Interact.h"
#import "Agent.h"
#import "PopStep.h"
#import "Sort.h"

// main characteristics
#define STEPS 10000
#define PIX 15
#define COLORS 16

// Global variables: simulation parameters
int world_sizeX = 20;
int world_sizeY = 20;
int number_agents;
int memory_length = 1;
int num_iterations = 128;
double action_error = .05;
double crossover_percent = .2;
double point_mutate = .02;
double duplicate_mutate = .01;
double split_mutate = .01;
int pay0 = 1;
int pay1 = 5;
int pay2 = 0;
int pay3 = 3;

// Prototype for function that handles setting simulation parameters.
void doForm();

int main(int argc, char **argv)
{
  id * playerList;				  // the players
  
  World * worldSpace;				  // agent world
  Interact * compete;
  PopStep * pop_update;
  Sort * sort;
  
  int i,j,k;
  int x,y;
  
  XColormap * colormap;                           // display objects
  Raster * z;
  BLTGraph * averageGraph;
  GraphElement * averageData;
  
  // initialize swarm
  initSwarm(argc, argv);

  // utils			   
  r250_init( DEFAULTSEED );
  sort = [[Sort alloc] init];
  if (sort == nil) {
    fprintf(stderr,"Error: Creation of Sorting object.\n");
    return (-1);
  }

  // Handle pre-simulation setup, mostly simulation parameters.
  doForm();
  
  // set up compete and pop_update
  compete = [[Interact alloc] init];
  if (compete == nil) {
    fprintf(stderr,"Error: Creation of Interact object.\n");
    return (-1);
  }
  [compete setIterations: num_iterations];
  [compete setError: action_error];
  
  pop_update = [[PopStep alloc] init];
  if (pop_update == nil) {
    fprintf(stderr,"Error: Creation of PopStep object.\n");
    return (-1);
  }
  [pop_update setCrossover: crossover_percent];
  [pop_update setPP: point_mutate];
  [pop_update setPD: duplicate_mutate];
  [pop_update setPS: split_mutate];
  
  // set up grid world
  worldSpace = [[ World alloc] init];
  if (worldSpace == nil) {
    fprintf(stderr,"Error: Creation of World object.\n");
    return (-1);
  }
  [worldSpace setSizeX: world_sizeX Y: world_sizeY];
  [worldSpace setNumberAgents: (world_sizeX*world_sizeY)];
  [worldSpace setInteract: compete];
  [worldSpace setPopStep: pop_update];
  [worldSpace setSort: sort];
  
  // set up agents
  number_agents = world_sizeX * world_sizeY;
  playerList = (id *) malloc(number_agents * sizeof(id));
  if (playerList == NULL){
    fprintf(stderr,"Error: Memory allocation of Agents.\n");
    exit(-1);
  }
  
  for (i=0;i<world_sizeY;i++) {
    for (j=0;j<world_sizeX;j++) {
      k = i*world_sizeY+j;
      
      playerList[k] = [[Agent alloc] init];
      [playerList[k] setWorld: worldSpace X: world_sizeX Y: world_sizeY];
      [playerList[k] setX: j Y: i];		  // also sets tag
      [playerList[k] setMemory: memory_length];
      [playerList[k] setStrategy: r250() & ((1 << (1 << memory_length)) - 1)];
    }
  }
  [worldSpace setPlayers: playerList];		  // init world with agent list
  [worldSpace setTmpArray];
  
  // Create a colormap for display (will go in higher level display objects)
  colormap = [[XColormap alloc] init];
  
  /*
    for (i = 0; i < COLORS; i++)
    [colormap setColor: i ToRed: (double) (r250() % 256) / 256.0
    Green: (double) (r250() % 256) / 256.0
    Blue: (double) (r250() % 256) / 256.0];
  */ 
  [colormap setColor: 0 ToName: "black"];     
  [colormap setColor: 1 ToName: "blue"];     
  [colormap setColor: 2 ToName: "red"];      
  [colormap setColor: 3 ToName: "white"];     
  [colormap setColor: 4 ToName: "purple"];     
  [colormap setColor: 5 ToName: "yellow"];
  [colormap setColor: 6 ToName: "orange"];     
  [colormap setColor: 7 ToName: "green"];
  
  [colormap setColor: 8 ToName: "firebrick"];     
  [colormap setColor: 9 ToName: "forestgreen"];     
  [colormap setColor: 10 ToName: "springgreen"];      
  [colormap setColor: 11 ToGrey: 0.3];     
  [colormap setColor: 12 ToGrey: 0.4];     
  [colormap setColor: 13 ToGrey: 0.5];
  [colormap setColor: 14 ToGrey: 0.6];     
  [colormap setColor: 15 ToGrey: 0.7];
  
  // create a window for display 
  z = [[Raster alloc] init];
  [z setColormap: colormap];
  [z setWidth: world_sizeX*PIX Height: world_sizeY*PIX];
  [z pack];
  
  // create avg graph
  averageGraph = [[BLTGraph alloc] init];
  [averageGraph title: "Average score of population vs. time"];
  [averageGraph axisLabelsX: "time" Y: "average score"];
  [averageGraph packWith: "-fill both"];
  averageData = [averageGraph createElement];
  [averageData setLabel: "avg"];
  [[averageData setWidth: 2] setDashes: 2];
  
  k=0;					  // counts steps in world

// print initial configuration
  for (y=0;y<world_sizeY;y++) {
    for (x=0;x<world_sizeX;x++) {
      [z fillRectangleX0: x*PIX Y0: y*PIX X1: x*PIX+PIX Y1: y*PIX+PIX
	 Color: [[worldSpace getObjectAtX: x Y: y] getStrategy] % COLORS];
    }
  }
  
  [z drawSelf];
  [averageGraph addTo: averageData X: k Y: [worldSpace getAverage]];
    
  // printf("\nINITIAL STATE\n");
  [worldSpace printSpatialStrategies];
  // [worldSpace printSpatialWorld];

  while ( tk_NumMainWindows > 0 ) {
    while( ! simControl->running )
      Tk_DoOneEvent(0);
    while( Tk_DoOneEvent(TK_ALL_EVENTS|TK_DONT_WAIT))
      ;
    
    [compete setIterations: num_iterations];
    [compete setError: action_error];
    [pop_update setCrossover: crossover_percent];
    [pop_update setPP: point_mutate];
    [pop_update setPD: duplicate_mutate];
    [pop_update setPS: split_mutate];

    for (i=0;i<number_agents;i++) {
      [playerList[i] setPayoff: 0 Val: pay0];
      [playerList[i] setPayoff: 1 Val: pay1];
      [playerList[i] setPayoff: 2 Val: pay2];
      [playerList[i] setPayoff: 3 Val: pay3];
    }


    [worldSpace stepSpatial];			  // iterate world one step
    k++;
      
    // DEBUG text stuff
    printf("STEP %d\n",k);
    [worldSpace printSpatialStrategies];
    // [worldSpace printSpatialWorld];
    printf("Avg: %g\n\n",[worldSpace getAverage]);
          
    for (y=0;y<world_sizeY;y++) {
      for (x=0;x<world_sizeX;x++) {
	[z fillRectangleX0: x*PIX Y0: y*PIX X1: x*PIX+PIX Y1: y*PIX+PIX
	   Color: [[worldSpace getObjectAtX: x Y: y] getStrategy] % COLORS];
      }
    }
    
    [z drawSelf];
    [averageGraph addTo: averageData X: k Y: [worldSpace getAverage]];
  }
     
  return(0);
}

// this is the current mechanism for setting simulation parameters.
// it will be changed when experiment objects are written.
void
doForm() {
#define NUMPARAMETERS 13
  SimParameters * simParameters;
  Parameter p[NUMPARAMETERS] = {
    {"Side Length X", &world_sizeX, TCL_LINK_INT},
    {"Side Length Y", &world_sizeY, TCL_LINK_INT},
    {"Memory Length", &memory_length, TCL_LINK_INT},
    {"Number of Iterations", &num_iterations, TCL_LINK_INT},
    {"Error in Action %", &action_error, TCL_LINK_DOUBLE},
    {"Crossover Replace %", &crossover_percent, TCL_LINK_DOUBLE},
    {"Point Mutate %", &point_mutate, TCL_LINK_DOUBLE},
    {"Duplicate Mutate %", &duplicate_mutate, TCL_LINK_DOUBLE},
    {"Split Mutate %", &split_mutate, TCL_LINK_DOUBLE},
    {"Payoff 0,0", &pay0, TCL_LINK_INT},
    {"Payoff 0,1", &pay1, TCL_LINK_INT},
    {"Payoff 1,0", &pay2, TCL_LINK_INT},
    {"Payoff 1,1", &pay3, TCL_LINK_INT}};
  
  simParameters = [[SimParameters alloc] init];
  [simParameters setParameterList: p Num: NUMPARAMETERS];
  [simParameters readValues: simControl];
}



// this code is ignorable (just some debugging stubs).
/*
void *_deftype_alloc( int size )
{
  return malloc( size );
}

void *_deftype_stack_alloc( int size )
{
  return malloc( size );
}

void _deftype_stack_free( void *block )
{
  return;
}

*/
