// ObserverSwarm.m

#import "ObserverSwarm.h"
#import <activity.h>
#import <simtoolsgui.h>

@implementation ObserverSwarm

+ createBegin: aZone
{
  ObserverSwarm *obj;
  id <ProbeMap> probeMap;

  // createBegin: here we set up the default simulation parameters.

  // Superclass createBegin to allocate ourselves.

  obj = [super createBegin: aZone];

  // Fill in the relevant parameters (only one, in this case).

  obj->displayFrequency = 100;
  obj->zoomFactor=2;

  // Also, build a customized probe map. Without a probe map, the default
  // is to show all variables and messages. Here we choose to
  // customize the appearance of the probe, give a nicer interface.

  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [self class]];
  probeMap = [probeMap createEnd];

  // Add in a bunch of variables, one per simulation parameters

  [probeMap addProbe: [probeLibrary getProbeForVariable: "displayFrequency"
                                    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "zoomFactor"
                                    inClass: [self class]]];

  // Now install our custom probeMap into the probeLibrary.

  [probeLibrary setProbeMap: probeMap For: [self class]];

  return obj;
}

- createEnd
{
  return [super createEnd];
}

- buildObjects
{
  [super buildObjects];

  // First, we create the model that we're actually observing. The
  // model is a subswarm of the observer. 

  // We create in the modelSwarm in "self" which creates a modelSwarm
  // in it's *own* Zone within the observerSwarm.

  modelSwarm = [ModelSwarm create: self];

  // Now create probe objects on the model and ourselves. This gives a
  // simple user interface to let the user change parameters.

  CREATE_PROBE_DISPLAY (modelSwarm);
  CREATE_PROBE_DISPLAY (self);

  // Instruct the control panel to wait for a button event.
  // We halt here until someone hits a control panel button.

  // Now that we're using Probes, the user can set the parameters
  // in the ModelSwarm probe window - we halt here to allow
  // the user to change parameters.

  [controlPanel setStateStopped];

  // When the user hits "go" on the control panel, we resume here

  // OK - the user said "go" so we're ready to start

  [modelSwarm buildObjects];

  // Now get down to building our own display objects.

  // First, create a colormap: this is a global resource, the information
  // here is used by lots of different objects.

  colorMap = [Colormap create: self];

  [colorMap setColor: 0 ToName: "black"];
  [colorMap setColor: 1 ToName: "green"];
  [colorMap setColor: 2 ToName: "red"];

  // Next, create a 2d window for display, set its size, zoom factor, title.

  worldRaster = [ZoomRaster create: self];
  [worldRaster setColormap: colorMap];
  [worldRaster setZoomFactor: zoomFactor];
  [worldRaster setWidth: [[modelSwarm getWorld] getSizeX]
                 Height: [[modelSwarm getWorld] getSizeY]];
  [worldRaster setWindowTitle: "Langton Ant Space"];
  [worldRaster pack];                             // draw the window.

  // Now create a Value2dDisplay: this is a special object that will
  // display arbitrary 2d value arrays on a given Raster widget.

  traceDisplay = [Value2dDisplay createBegin: self];
  [traceDisplay setDisplayWidget: worldRaster colormap: colorMap];
  [traceDisplay setDiscrete2dToDisplay: [modelSwarm getTraces]];
  traceDisplay = [traceDisplay createEnd];

  // And also create an Object2dDisplay: this object draws agents on
  // the worldRaster widget for us.

  agentDisplay = [Object2dDisplay createBegin: self];
  [agentDisplay setDisplayWidget: worldRaster];
  [agentDisplay setDiscrete2dToDisplay: [modelSwarm getWorld]];
  [agentDisplay setObjectCollection: [modelSwarm getAgentList]];
  [agentDisplay setDisplayMessage: M(drawSelfOn:)];   // draw method
  agentDisplay = [agentDisplay createEnd];

  // Also, tell the world raster to send mouse clicks to the agentDisplay
  // this allows the user to right-click on the display to probe the agents.

  [worldRaster setButton: ButtonRight
               Client: agentDisplay
               Message: M(makeProbeAtX:Y:)];

  return self;
}

- buildActions
{

// Create the actions necessary for the simulation. 

  [super buildActions];

  // First, let our model swarm build its own schedule.

  [modelSwarm buildActions];

  // Create an ActionGroup for display. 

  displayActions = [ActionGroup create: self];

  // Schedule up the methods to draw the display of the world

  [displayActions createActionTo: traceDisplay        message: M(display)];
  [displayActions createActionTo: agentDisplay        message: M(display)];
  [displayActions createActionTo: worldRaster         message: M(drawSelf)];

  [displayActions createActionTo: actionCache         message: M(doTkEvents)];

  // And the display schedule. Note the repeat interval is set from our
  // own Swarm data structure. Display is frequently the slowest part of a
  // simulation, so redrawing less frequently can be a help.

  displaySchedule = [Schedule createBegin: self];
  [displaySchedule setRepeatInterval: displayFrequency]; // note frequency!
  displaySchedule = [displaySchedule createEnd];
  [displaySchedule at: 0 createAction: displayActions];
 
  return self;
}

- activateIn: swarmContext
{
// activateIn: - activate the schedules so they're ready to run.

  [super activateIn: swarmContext];

  // Activate the model swarm in ourselves. The model swarm is a
  // subswarm of the observer swarm.

  [modelSwarm activateIn: self];

  // Now activate our schedule in ourselves. This arranges for the
  // execution of the schedule we built.

  [displaySchedule activateIn: self];

  // Activate returns the swarm activity - the thing that's ready to run.

  return [self getSwarmActivity];
}

@end








