// Swarm library. Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

// Class "MoGrid2d" by Sven Thommesen
// with some changements introduced by Pietro Terna, see //pt below


// This replacement for Grid2d allows for Multiple Occupancy!
// Each occupied cell of the grid holds a List, which may contain any
// number of objects (agents). Objects/agents are added to or 
// removed from such lists, rather than from the underlying grid.
// Empty list objects are added and discarded as needed, dynamically.
// (The grid is created initially with no List objects.)

// If you want the lists to contain arbitrary data values rather than
// pointers to objects, just cast them to (id) before handing them to
// the methods. The lists don't care! But if you do this, avoid method 
// -dropAllAtX:Y:, which attempts to drop the 'objects' contained in 
// the given list!

// (c) Sven Thommesen 1997 <sthomme@humsci.auburn.edu>.

// #import <swarmobject/SwarmObject.h>
//pt #import <objectbase/SwarmObject.h> // 1.2
//pt the followng is introduced to have MoGrid2d reponding to message
// makeProbeAtX:Y: if we click the second button on worldRaster
#import <space/Grid2d.h>

@interface MoGrid2d: Grid2d {

//pt the following are defined in Grid2d
//pt  int xsize, ysize;             // size of grid
//pt  id  * lattice;                // lattice[] is the grid
//pt  int * offsets;                // offsets[] point to the start of lines
  id  shuffler;			// Shuffler to randomize lists

}

// Private methods:

-makeOffsets;
-(id *) allocLattice;
-setSizeX: (int) x Y: (int) y;
-createEnd;

// Creation:

+create: (id) aZone setSizeX: (int) x Y: (int) y;

// Return lattice data:

-(int)   getSizeX;		// x size
-(int)   getSizeY;		// y size
-(id *)  getLattice;		// pointer to the lattice itself


//pt -(int *) getOffsets;            // pointer to the offsets vector
  -(long *) getOffsets;            //pt long with Grid2d  inheritance

// Methods to add objects to cells on the lattice:

// Add the given object to the head of the list at (x,y):
-pushObject:  (id) anObject atX: (int) x Y: (int) y;	// stack
-addFirst:    (id) anObject atX: (int) x Y: (int) y;	// (synonym)

// Add the given object to the tail of the list at (x,y):
-enQueObject: (id) anObject atX: (int) x Y: (int) y;	// queue
-addLast:     (id) anObject atX: (int) x Y: (int) y;	// (synonym)

// Return the number of objects in the list at (x,y):
-(int) numberOfObjectsAtX: (int) x Y: (int) y;


// Methods to remove objects from cells on the lattice:

// Remove and return the object (if any) at the head of the list at (x,y),
// or nil if no such object:
-(id) popObjectAtX:   (int) x Y: (int) y;		// stack
-(id) deQueObjectAtX: (int) x Y: (int) y;		// queue
-(id) getFirstAtX:    (int) x Y: (int) y;		// (synonym)

// Remove and return the object (if any) at the tail of the list at (x,y),
// or nil if no such object:
-(id) getLastAtX:     (int) x Y: (int) y;

// Remove and return the given object from the list at (x,y),
// or (nil) if not found:
-(id) deleteObject: (id) anObject atX: (int) x Y: (int) y;


// Operations on lists:

// Remove all object pointers from the list at (x,y)
// (but the objects themselves still persist):
-(void) dropListAtX:  (int) x Y: (int) y;

// Remove all object pointers from the list at (x,y) 
// and also drop those objects (so don't do this if the list
// contains data values rather than object pointers!):
-(void) dropAllAtX:  (int) x Y: (int) y;

// Plug the given list into the cell at (x,y):
// WARNING: we can only have one list per cell!
// setList will abort if there's already a list there.
-setList: list atX: (int) x Y: (int) y;

// Shuffle (randomize) the list at cell (x,y) in place:
-(void) shuffleListAtX: (int) x Y: (int) y;

// Shuffle all lists found:
-(void) shuffleAll;

// Methods to retrieve list objects from cells on the lattice:

// Return a pointer to the list at (x,y), or nil if no list there:
-getListAtX: (int) x Y: (int) y;

// Return a COPY OF the list at (x,y) [a new object], 
// or nil if no list there:
-getCopyAtX: (int) x Y: (int) y;


// Methods to retrieve lists in shuffled (randomized) order:

// Shuffle the list at (x,y) and return a pointer to the list, 
//   or nil if there's no list there:
-(id) getShuffledListAtX: (int) x Y: (int) y;

// Return a SHUFFLED COPY OF the list at (x,y) [a new object],
//   or nil if there's no list there:
-(id) getShuffledCopyAtX: (int) x Y: (int) y;

@end

// Fast macro to access the lattice array. Use with caution.
// Identical to 'discrete2dSiteAt' in Discrete2d.
#define mo2dSiteAt(l, offsets, x, y) ((l) + (offsets) [(y)] + (x))

