// Programme dispersion.
// Bruno Cuvelier - ORSTOM.
// Date : 21/01/97.

// Le DispersionModelSwarm est le Swarm racine du modele que l'on concoit.

#import "DispersionModelSwarm.h"
#import <simtools.h>
#import <math.h>

@implementation DispersionModelSwarm

// Obtention de la table des SpeedSwarm gere par le ModelSwarm. 
- getSpeedSwarmArray {
  return speedSwarmArray;
}

// Obtention de la grille 2D dans laquelle se meuvent les agents.
-(Grid2d *) getWorld {
  return world;
}

// Obtention de la matrice qui gere la densite des agents en tout
// point de l'espace.
-(DensitySpace *) getDensityMap {
  return densityMap;
}

// Les agents reference dans la file "agentListInMigration" sont deplaces.
// Les agents dans la file d'attente sont traites sequentiellement.
// Connaissant l'ancienne vitesse et la nouvelle vitesse de l'agent,
// il est possible de le supprimer du speedSwarm ou il se trouve pour
// l'ajouter dans le speedSwarm correspondant a sa vitesse.
-doMigration {
  int oldSpeed, speed;
  Agent *currentAgent; // Agent en cours de traitement
  
  //fprintf(stderr, "Lancement de la migration\n");
  while ([agentListInMigration getCount] != 0) {
    fprintf(stderr, "Migration d'un agent\n");
    // Agent en cours de traitement
    currentAgent= [agentListInMigration getFirst];

    // Obtention des vitesse
    oldSpeed= [currentAgent getOldSpeed];
    speed= [currentAgent getSpeed];

    if (speed <= [speedSwarmArray getCount]) {
      // Retrait de l'ancien speedSwarm
      [[speedSwarmArray atOffset: oldSpeed-1] removeAgent: currentAgent];

      // Ajout dans le nouveau speedSwarm
      [[speedSwarmArray atOffset: speed-1] addAgent: currentAgent];
    }
    
    // Suppresion dans la file d'attente
    [agentListInMigration removeFirst];
  }
  return self;
}

// Methode recalculant la densite ideale des agents
-computeDensity {
  int i;
  int newDensity;

  if (isMinDensity)
    newDensity= maxIdealDensity;
  else
    newDensity= minIdealDensity;
  
  if (getCurrentTime() && (getCurrentTime()%timeRange) == 0) {
    isMinDensity= !isMinDensity;
    //[speedSwarmArray forEach: M(sendNewDensity) : (id) &newDensity];
    for (i=0; i<[speedSwarmArray getCount]; i++)
      [[speedSwarmArray atOffset: i] sendNewDensity: (id) &newDensity];
  }
    
  return self;
}

// Creation du ModelSwarm : initialisation
+createBegin: (id) aZone {
  DispersionModelSwarm *obj;
ProbeMap *probeMap;

  // Initialisation des variables
  obj = [super createBegin: aZone];
  obj->nbOfAgents= 100;
  obj->worldXSize = 80;
  obj->worldYSize = 80;
  obj->neighborhood = 5;
  obj->convergenceFactor = 10;
  obj->minIdealDensity = minDensity;
  obj->maxIdealDensity = maxDensity;
  obj->isMinDensity= True;
  obj->minSpeed = 1;
  obj->maxSpeed = 5;
  obj->randomMoveProbability = 0.0;
  obj->timeRange= 50;

  // Creation du visualisateur de l'objet.
  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [self class]];
  probeMap = [probeMap createEnd];

  // Insertion des parametres de simulations accessibles a l'utilisateur.
  // Ces parametres sont fixes a partir du moment ou la simulation a
  // debutee.
  [probeMap addProbe: [probeLibrary getProbeForVariable: "minIdealDensity"
 				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "maxIdealDensity"
 				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "timeRange"
 				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "convergenceFactor"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "nbOfAgents"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "worldXSize"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "worldYSize"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "neighborhood"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "minSpeed"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "maxSpeed"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "randomMoveProbability"
				    inClass: [self class]]];

  [probeLibrary setProbeMap: probeMap For: [self class]];

  return obj;
}

// Creation du ModelSwarm : validation.
-createEnd {
  return [super createEnd];
}

// Constructions des objets constituant le modele.
// La creation des cartes et des speedSwarms trouve naturellement
// sa place dans cette methode. Par contre, la creation des agents
// peut etre discutee. Ici le modelSwarm en prend la responsabilite
// et envoie l'ordre d'ajout au speedSwarm concerne. Cette creation
// pourrait etre deleguee au speedSwarm. Pour cela, il suffit que le
// modelSwarm calcul la vitesse du futur agent pour deleguer l'ordre
// de creation au speedSwarm voulu.
-buildObjects {
  int i;
  int x, y;
  int speedRangeSize= maxSpeed-minSpeed+1;
  int Xinit, Yinit;
  int Xpos, Ypos;
  int dx, dy;

  [super buildObjects];

  // Creation de la carte de densite des populations.
  // La densityMap est initialisee en precisant les dimensions du monde
  // et le "rayon" de voisinage "visible" autour d'une cellule.
  // Avant la creation des agents toutes les cellules ont une densite
  // nulle.
  densityMap = [DensitySpace createBegin: [self getZone]];
  [densityMap setSizeX: worldXSize Y: worldYSize];
  [densityMap setNeighborhood: (int) neighborhood];
  densityMap = [densityMap createEnd];
  [densityMap fastFillWithValue: 0];

  // Creation de la carte des positions des agents.
  // La variable "world" correspond a l'environnement dans lequel se meuvent
  // les agents. Chaque element correspond a une position dans l'espace. Si
  // un agent s'y trouve l'element reference l'agent, sinon la reference est
  // nulle.
  world = [Grid2d createBegin: [self getZone]];
  [world setSizeX: worldXSize Y: worldYSize];
  world = [world createEnd];
  [world fastFillWithObject: nil];

  // Creation de la file d'attente pour les migrations d'agents.
  agentListInMigration= [List create: [self getZone]];

  // Creation des SpeedSwarm.
  // Les speedSwarms sont regroupes dans un tableau de la taille du nombre
  // de vitesses possibles dans la simulation. Les speedSwarms sont ordonnes
  // par ordre croissant de vitesse. Leur indice correspond a la vitesse des
  // agents qu'ils gerent (-1).
  {
    id newSpeedSwarm;

    speedSwarmArray = [Array create: [self getZone] setCount: speedRangeSize];
    for (i=0; i<speedRangeSize; i++) {
      // Creation d'un nouveau SpeedSwarm
      newSpeedSwarm= [SpeedSwarm createBegin: [self getZone] withSpeed: i+1];
      newSpeedSwarm= [newSpeedSwarm createEnd];

      // Ajout du SpeedSwarm dans la liste
      [speedSwarmArray atOffset: i put: newSpeedSwarm];
    }
  }
    
  [world setOverwriteWarnings: 0];

  // Les speedSwarms etant crees, ils peuvent a leur tour creer leur
  // objets.
  [speedSwarmArray forEach: M(buildObjects)];
  //[speedSwarmArray forEach: M(getAgentList)];

  // Calcul de la position initiale de la repartition. Les agents sont
  // positionne en carre au centre de l'ecran. Ce choix est arbitraire.
  // Il correspond a l'existance d'un groupe d'agents compacts (banc de
  // poisson par exemple).
  // ATTENTION : il peut y avoir des effet de bord si dx > worldXsize
  // ou dy > worldYsize.

  // Dimensions du groupe
  dx= dy= sqrt(nbOfAgents);

  // Position initiale du groupe ("coin superieur gauche")
  Xinit= (worldXSize-dx)/2;
  Yinit= (worldYSize-dy)/2;

  // Position de l'agent dans le groupe.
  Xpos= Ypos= 0;

  // Redefinition de la dimension "dy" si le nombre d'agents n'est pas un
  // carre.
  if (nbOfAgents != (dx*dy))
    dy= dx+(nbOfAgents-(dx*dy));

  // Boucle de creation des agents
  for (i = 0; i < nbOfAgents; i++) {
    Agent *agent;
    int agentIdealDensity;
    int agentSpeed;

    // Choix aleatoire des densite ideale et vitesse de deplacement des agents.
    // Les choix aleatoires sont purement arbitraires et ne correspondent
    // surement pas a la realite comportementale d'un groupe.
    agentSpeed= [uniformIntRand getIntegerWithMin: minSpeed 
				withMax: maxSpeed];   

    // Choix aleatoire de la densite
    //agentIdealDensity= [uniformIntRand getIntegerWithMin: minIdealDensity 
    //			       withMax: maxIdealDensity];

    // Densite ideale alternee
    agentIdealDensity= minIdealDensity;

    // Calcul de la position de l'agent dans le groupe.
    x= Xinit+Xpos;
    y= Yinit+Ypos;

    // Creation de l'agent.
    // L'agent est initialise en lui donnant la reference du monde dans
    // lequel il se deplace ("world"). La reference de la carte des densites lui
    // permettant de connaitre la densite a la position ou il se trouve et
    // de mettre a jour la carte des densites ("densityMap") lorsqu'il de deplace.
    
    // ATTENTION !
    // Cette connaissance des agents du monde et de la carte des densites n'est
    // pas satisfaisante. Un agent pourrait ainsi avoir des comportement
    // "anormaux" comme la teletransportation ou rendre "aveugle" d'autres agents
    // en modifiant de facon non protocolaire la "densityMap".
    // Toutefois, cette solution est facile a implementer. Ceci pourrat etre
    // revue dans une version ulterieure.
    agent = [Agent createBegin: [self getZone]];
    [agent setWorld: world Density: densityMap];
    [agent setMigrationRef: &agentListInMigration]; 
    agent = [agent createEnd];

    // L'agent se positionne dans l'espace par l'intermdiaire de sa reference
    // sur le monde ("world").
    // ATTENTION ! (cf. remarque ci-dessus)
    [agent setX: x Y: y];

    // Initialisation de l'etat de l'agent.
    [agent setSpeed: agentSpeed ForInit: 1];
    [agent setIdealDensity: (id) &agentIdealDensity];
    [agent setConvergenceFactor: convergenceFactor];

    // Ajout de l'agent dans le SpeedSwarm correspondant a sa vitesse.
    [[speedSwarmArray atOffset: agentSpeed-1] addAgent: agent];

    // Mise a jour de la densite autour de la position de l'agent
    [densityMap incDensityAroundX: x Y: y];
    
    // Calcul de la position dans le groupe de l'agent suivant.
    Xpos= (Xpos+1)%dx;
    if (Xpos == 0) {
      Ypos= (Ypos+1)%dy;
    }
  }

  [world setOverwriteWarnings: 1];

  return self;
}

// Creation du plan d'actions et du scheduler. Le modele active l'operation
// de migration des agents d'un Swarm a un autre.
-buildActions {
  [super buildActions];
  
  // Activation des speedSwarms
  [speedSwarmArray forEach: M(buildActions)];
  
  // Creation de groupes d'actions du modele.
  // Le modele se charge d'effectuer la migration de agents.
  modelActions = [ActionGroup create: [self getZone]];
  [modelActions createActionTo: self message: M(doMigration)];
  [modelActions createActionTo: self message: M(computeDensity)];
  //[modelActions createActionTo: migration message: M(getNewSpeedSwarmArray)];

  // Creation du scheduler.
  // Le role du scheduler ("modelSchedule") est d'ordonnacer l'activite
  // du modele definie dans "modelActions".
  modelSchedule = [Schedule createBegin: [self getZone]];
  [modelSchedule setRepeatInterval: 1];
  modelSchedule = [modelSchedule createEnd];
  [modelSchedule at: 0 createAction: modelActions];
  
  return self;
}

// Activation du scheduling du modele.
-activateIn: (id) swarmContext {

  // Activation du modele lui-meme par l'intermediaire de la superclasse
  [super activateIn: swarmContext];

  // Activation du scheduler du model.
  [modelSchedule activateIn: self];

  // Activation des scheduler des speedSwarms.
  [speedSwarmArray forEach: M(activateIn:) : self];

  // renvoie sa propre activite
  return [self getSwarmActivity];
}

@end
