// Programme dispersion.
// Bruno Cuvelier - ORSTOM.
// Date : 20/01/97
// Implementation des agents.

#import "Agent.h"
#import <simtools.h>
#import <random.h>

@implementation Agent

// Prise en compte des references sur le monde et la carte des densite.
-setWorld: (Grid2d *) w Density: (DensitySpace *) d {
  if (world != nil) {
    [InvalidArgument raiseEvent: "L'agent ne peut pas se teletranporter !\n"];
  }
  world= w;
  densityMap= d;
  return self;
}

// Validation de la creation de l'agent.
-createEnd{
  if (world == nil) {
    [InvalidCombination raiseEvent: "L'agent n'appartient a aucun monde !\n"];
  }
  // Prise en compte des dimensions du monde.
  worldXSize = [world getSizeX];
  worldYSize = [world getSizeY];

  return self;
}

// Modification de la densite maximale acceptee par l'agent.
-setIdealDensity: d {
  idealDensity= *((Density*) d);
  return self;
}

// Modification du facteur de convergence de la densite ideale
// vers la densite environnementale. Plus le facteur de convergeance
// est eleve, plus la convergence est faible.
-setConvergenceFactor: (int) c {
  convergenceFactor= c;
  return self;
}

// Modification de la vitesse de l'agent.
// A l'initialisation il ne se signale pas au migration Swarm.
-setSpeed: (int) s ForInit: (int) b {
  oldSpeed= speed;
  speed= s;

  // L'agent signale au swarm migration qu'il a modifie sa vitesse
  // sauf a l'initialisation.
  if (b == 0) {
    [(*migration) addLast: self];
  }

  return self;
}

// Positionnement de la variable de mouvement aleatoire
-setRandomMoveProbability: (float) p {
  randomMoveProbability= p;
  return self;
}

// Positionnement de l'agent dans le monde.
-setX: (int) inX Y: (int) inY{
  x = inX;
  y = inY;
  [world putObject: self atX: x Y: y];
  return self;
}

// Reference sur l'objet assurant la migration des agents entre les
// speedSwarms.
-setMigrationRef: (id *) m {
  migration= m;
  return self;
}

// Changement de la couleur visualisant l'agent.
-setAgentColor: (Color) c{
  agentColor = c;
  return self;
}

-(int) getOldSpeed {
  return oldSpeed;
}

-(int) getSpeed {
  return speed;
}

// Methode statistiques
-(double) getUnhappiness {
  return unhappiness;
}

// Comportement de l'agent.
// L'agent se deplace vers une cellule voisine en fonction de sa rapidite.
// Parmi les cellules voisines il peut choisir n'importe quelle cellule
// dont la densite de population est la plus proche de celle qu'il recherche.

// La frequence d'activation de cette methode depend de la vitesse de deplacement
// de l'agent. Par consequent, entre deux deplacements la valeur qu'il possede
// sur la densite locale est erronee.
const int SIZE= 9;
-step {
  Density densityArray[SIZE]; // Tableau des cellules voisines
  int indice;              // Position courante dans le tableau densityArray
  Density bestDensity;   // Meilleure densite dans le voisinage immediat
  Density localDensity;    // Densite a la position de l'agent
  int nbOfBetter;          // Nombre de meilleure a egalite
  int i, j;      

  // Obtention de la densite locale.
  localDensity= [densityMap getValueAtX: x Y: y];
  actualDensity= localDensity;

  // L'agent adapte sa densite ideale a son environnement.
  if (localDensity == idealDensity) {
    unhappiness = 0.0;
    return self;
  }

  if (localDensity > idealDensity) {
    unhappiness = (double)(localDensity-idealDensity)/idealDensity;
    idealDensity += (localDensity-idealDensity)/convergenceFactor;
  }
  else {
    unhappiness = (double)(idealDensity-localDensity)/idealDensity;
    idealDensity -= (idealDensity-localDensity)/convergenceFactor;
  }
  
  // Creation du tableau de voisinage
  //densityArray= [Array create: [self getZone] setCount: 9];

  // Remplissage du tableau densityArray
  
  indice= 0;
  for (i=y-1; i<=y+1; i++) {
    for (j=x-1; j<=x+1; j++) {
      densityArray[indice]= [densityMap getDensityAtX: j
					Y: i];
      indice++;
    }  
  }

  // Recherche du nombre de voisin ayant la meilleure densite
  {
    Density densityHere;
    BOOL low, hereIsBetter;

    bestDensity= localDensity;
    nbOfBetter= 1;
    low= (localDensity < idealDensity)? True : False;
    for (i=0; i<SIZE; i++) {
      densityHere= densityArray[i];
      if (densityHere == bestDensity) {
	nbOfBetter++;      
      }
      else {
	hereIsBetter= 
	(low == True) ? 
	(densityHere > bestDensity) : 
	(densityHere < bestDensity);

	if (hereIsBetter) {
	  bestDensity= densityHere;
	  nbOfBetter= 1;
	}
      }
    }
  }
  
  // Choix d'une position
  {
    int choice= 1;

    if (nbOfBetter > 1)
      choice= [uniformIntRand getIntegerWithMin: 1 withMax: nbOfBetter];

    indice=0;
    while ((indice<SIZE) && (choice != 0)) {
      if (bestDensity == densityArray[indice])
	choice--;
      indice++; 
      //fprintf(stderr, "indice000= %d\n", indice);
    }
    if (indice == SIZE) { // Il n'y a pas de meilleure solution
      //fprintf(stderr, "I don't want to move\n");
      return self;
    }
    indice--;
    //fprintf(stderr, "indice00= %d\n", indice);
  }

  // Verification que cette position est accessible sinon prendre la
  // suivante.
  {
    int newX, newY;
    int dx, dy;
    int tries = 0;

    do{
      // Position dans le voisinage
      //fprintf(stderr, "indice0= %d\n", indice);
      dy= (int)indice/3;
      dx= indice-dy*3;
      //fprintf(stderr,"dx=%d; dy=%d\n", dx, dy);
      if ((dx+dy)>4) exit(1);

      // Position dans l'espace
      newX= (x+dx-1+worldXSize)%worldXSize;
      newY= (y+dy-1+worldYSize)%worldYSize;
      //fprintf(stderr, "0newX: %d; newY: %d\n", newX, newY);

      // Verifier que la position est libre
      if ([world getObjectAtX: newX Y: newY] == nil) {// c'est termine
	// Il faut mettre a jour la carte des densites.
	[densityMap decDensityAroundX: x Y: y];
	[densityMap incDensityAroundX: newX Y: newY];
	
	// Effectuer le deplacement de l'agent.
	[world putObject: nil atX: x Y: y];
	[world putObject: self atX: newX Y: newY];
	
	actualDensity= [densityMap getValueAtX: newX Y: newY];
	//fprintf(stderr, "Density: %d  atX: %d  Y: %d\n", actualDensity, newX, newY);
	
	// Prise en compte des nouvelles coordonnees.
	x = newX;
	y = newY;
	
	///fprintf(stderr, "I'm an happy agent :-)\n");
	return self;
      }
      else { // chercher la suivante
	///fprintf(stderr, "I search after a new position :-(\n");	
	nbOfBetter--;
	do {
	  //fprintf(stderr, "indice= %d\n", indice);
	  indice= (indice+1)%SIZE;
	} 
	while (bestDensity != densityArray[indice]);
      }
    }
    while (nbOfBetter > 0);

    // L'agent n'a pas trouve de meilleure place. Exeder par cette injustice
    // il va choisir aleatoirement la premiere place libre.
    //fprintf(stderr, "I'm very angry 8-0\n");	
    while ([world getObjectAtX: newX Y: newY] != nil && tries < 10) {
      newX = (x + [uniformIntRand getIntegerWithMin: -1 
				  withMax: 1] + worldXSize) % worldXSize;
      newY = (y + [uniformIntRand getIntegerWithMin: -1 
				  withMax: 1] + worldYSize) % worldYSize;
      //fprintf(stderr, "1newX: %d; newY: %d\n", newX, newY);
      tries++;
    }
    
    // Epuise devant tant d'energie depensee pour rien il s'endort en revant
    // a de meilleurs lendemains.
    if (tries == 10) {
      //fprintf(stderr, "I'm so tired :-|\n");
      return self;
    }
    
    // Il faut mettre a jour la carte des densites.
    [densityMap decDensityAroundX: x Y: y];
    [densityMap incDensityAroundX: newX Y: newY];
    
    // Effectuer le deplacement de l'agent.
    [world putObject: nil atX: x Y: y];
    [world putObject: self atX: newX Y: newY];
    
    actualDensity= [densityMap getValueAtX: newX Y: newY];
    
    x= newX;
    y= newY;
  }

  return self;
}

-step1{
  Density localDensity;     // Nombre de voisins
  int newX, newY;           // Future position
  int tries;                // Nombre d'essais de deplacements
  
  // Obtention de la densite locale.
  localDensity= [densityMap getValueAtX: x Y: y];
  actualDensity= localDensity;

  // L'agent adapte sa densite ideale a son environnement.
  if (localDensity == idealDensity) {
    unhappiness = 0.0;
    return self;
  }

  if (localDensity > idealDensity) {
    unhappiness = (double)(localDensity-idealDensity)/idealDensity;
    idealDensity += (localDensity-idealDensity)/convergenceFactor;
  }
  else {
    unhappiness = (double)(idealDensity-localDensity)/idealDensity;
    idealDensity -= (idealDensity-localDensity)/convergenceFactor;
  }
  
  // Recherche du voisinage le mieux adapte
  if (localDensity == maxDensity) // Mouvement impossible
    return self;

  newX = x;
  newY = y;
  // Recherche de la densite la mieux adaptee dans le voisinage immediat.
  [densityMap findBetterDensity: (localDensity < idealDensity) ? low : high
	      X: &newX Y: &newY];
  
  // Apres avoir trouve le site le mieux adapte, l'agent peut se comporter
  // de facon imprevisible.
  if (((float)[uniformDblRand
        getDoubleWithMin: 0.0 withMax: 1.0]) < randomMoveProbability) {

    newX = (x +
            [uniformIntRand
              getIntegerWithMin: -1L withMax: 1L]); // pick a random spot

    newY = (y + [uniformIntRand getIntegerWithMin: -1L withMax: 1L]);

    newX = (newX + worldXSize) % worldXSize;
    newY = (newY + worldYSize) % worldYSize;
  }

  // Il est possible que le site choisi par l'agent soit deja occupe par
  // un autre agent. Dans ce cas, il doit choisir un autre site au hasard.
  tries = 0;
  while ([world getObjectAtX: newX Y: newY] != nil && tries < 10) {
    newX = (x + [uniformIntRand getIntegerWithMin: -1 withMax: 1] + worldXSize) % worldXSize;
    newY = (y + [uniformIntRand getIntegerWithMin: -1 withMax: 1] + worldYSize) % worldYSize;
    tries++;
  }
  if (tries == 10) {
    newX = x;
    newY = y;
    return self;
  }
  
  // Il faut mettre a jour la carte des densites.
  [densityMap decDensityAroundX: x Y: y];
  [densityMap incDensityAroundX: newX Y: newY];

  // Effectuer le deplacement de l'agent.
  [world putObject: nil atX: x Y: y];
  [world putObject: self atX: newX Y: newY];

  actualDensity= [densityMap getValueAtX: newX Y: newY];
  //fprintf(stderr, "Density: %d  atX: %d  Y: %d\n", actualDensity, newX, newY);

  // Prise en compte des nouvelles coordonnees.
  x = newX;
  y = newY;

  return self;
}

-drawSelfOn: (Raster *) r {
  [r drawPointX: x Y: y Color: agentColor];
  return self;
}

@end
