// Swarm library. Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

// Class "MoXGrid2d" by Sven Thommesen <sthomme@humsci.auburn.edu>

// Like its superclass MoGrid2d, this class allows more than one object (agent)
// to occupy a site at one time. Each non-empty cell (site) has a List object
// to hold all the occupants.

#import <math.h>

#import "MoXGrid2d.h"

#import "ListShuffler.h"

@implementation MoXGrid2d

-createEnd {

   shuffleObject = [ListShuffler create: [self getZone]];

   return [super createEnd];
}


// Set default parameters:


-setNeighborhoodType: (NeighborhoodType) type Size: (int) size {

// We keep the tests in this method, on the assumption
// that it won't be called all that often.

// #ifdef DOALLTESTS

   if ( (size > xsize/2) || (size > ysize/2))
   [InvalidCombination raiseEvent:
   "MoXGrid2d: (%d) default neighborhood is too large\n", size];

   if (size < 1) 
   [InvalidCombination raiseEvent:
   "MoXGrid2d: (%d) default neighborhood is too small\n", size];

   if ( (type < 1) || (type >= InvalidType) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d: (%d) default neighborhood type is invalid\n", type];

// #endif

   defaultType = type;
   defaultSize = size;

   return self;
}

-(int) getNeighborhoodType {
   return defaultType;
}

-(int) getNeighborhoodSize {
   return defaultSize;
}

-setShuffled: (BOOL) s {
   defaultShuffle = s;
   return self;
}

-(BOOL) getShuffled {
   return defaultShuffle;
}


// Create instances of self with given parameters:


+createWithDefaults: (id) aZone 
	setSizeX: (int) x Y: (int) y {
   MoXGrid2d * thisGrid;

   thisGrid = [MoXGrid2d createBegin: aZone];
   [thisGrid setSizeX: x Y: y];
   thisGrid = [thisGrid createEnd];

   // These defaults are the same as used in Grid2d:

   [thisGrid setNeighborhoodType: AxesOnly Size: 1];	// von Neumann
   [thisGrid setShuffled: NO];

   return thisGrid;
}

+create: (id) aZone setSizeX: (int) x Y: (int) y
		withType: (NeighborhoodType) type 
		Size: (int) size 
		Shuffled: (BOOL) s {

   MoXGrid2d * thisGrid;

   thisGrid = [MoXGrid2d createBegin: aZone];
   [thisGrid setSizeX: x Y: y];
   thisGrid = [thisGrid createEnd];

   [thisGrid setNeighborhoodType: type Size: size];
   [thisGrid setShuffled: s];

   return thisGrid;
}


// Add data for all points in a given neighborhood to data arrays:


-addAxesAtX: (int) x Y: (int) y Size: (int) s {
   int ii,jj;
   int xxTemp, yyTemp;

   // X axis:

      yyTemp = y;
      for (ii = -s; ii < s+1; ii++) { 
        if (ii != 0) {
          xxTemp = (x + ii + xsize) % xsize;

        index++ ;

        objArray[index] = *mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
        // objArray[index] = [self getObjectAtX: xxTemp Y: yyTemp];

        xyArray[index] = (xxTemp << 16) + yyTemp;
        // xyArray[index] = xxTemp*65536 + yyTemp;

        pArray[index] = (id) mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
        xArray[index] = xxTemp;
        yArray[index] = yyTemp;

        };
      };

   // Y axis:

      xxTemp = x;
      for (jj = -s; jj < s+1; jj++) {
        if (jj != 0) {
          yyTemp = (y + jj + ysize) % ysize;

        index++ ;

        objArray[index] = *mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
        // objArray[index] = [self getObjectAtX: xxTemp Y: yyTemp];

        xyArray[index] = (xxTemp << 16) + yyTemp;
        // xyArray[index] = xxTemp*65536 + yyTemp;

        pArray[index] = (id) mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
        xArray[index] = xxTemp;
        yArray[index] = yyTemp;

        };
      };

   return self;
}

-addDiagsAtX: (int) x Y: (int) y Size: (int) s {
   int ii;
   int xxTemp, yyTemp;

   // Positive diagonal:

      for (ii = -s; ii < s+1; ii++) {
        if (ii != 0) {
          xxTemp = (x + ii + xsize) % xsize;
          yyTemp = (y + ii + ysize) % ysize;

        index++ ;

        objArray[index] = *mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
        // objArray[index] = [self getObjectAtX: xxTemp Y: yyTemp];

        xyArray[index] = (xxTemp << 16) + yyTemp;
        // xyArray[index] = xxTemp*65536 + yyTemp;

        pArray[index] = (id) mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
        xArray[index] = xxTemp;
        yArray[index] = yyTemp;

        };
      };

   // Negative diagonal:

      for (ii = -s; ii < s+1; ii++) {
        if (ii != 0) {
          xxTemp = (x + ii + xsize) % xsize;
          yyTemp = (y - ii + ysize) % ysize;

        index++ ;

        objArray[index] = *mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
        // objArray[index] = [self getObjectAtX: xxTemp Y: yyTemp];

        xyArray[index] = (xxTemp << 16) + yyTemp;
        // xyArray[index] = xxTemp*65536 + yyTemp;

        pArray[index] = (id) mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
        xArray[index] = xxTemp;
        yArray[index] = yyTemp;

        };
      };


   return self;
}


-addBoxAtX: (int) x Y: (int) y Size: (int) s {
   int ii,jj;
   int xxTemp, yyTemp;

   // Square Box:

      for (ii = -s; ii < s+1; ii++) {
      for (jj = -s; jj < s+1; jj++) {
        if ( (ii != 0) | (jj != 0) ) {
          xxTemp = (x + ii + xsize) % xsize;
          yyTemp = (y + jj + ysize) % ysize;

          index++ ;

          objArray[index] = *mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
          // objArray[index] = [self getObjectAtX: xxTemp Y: yyTemp];

          xyArray[index] = (xxTemp << 16) + yyTemp;
          // xyArray[index] = xxTemp*65536 + yyTemp;

          pArray[index] = (id) mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
          xArray[index] = xxTemp;
          yArray[index] = yyTemp;

        };
      }; };


   return self;
}


-addCircleAtX: (int) x Y: (int) y Size: (int) s {
   int ii,jj;
   int xxTemp, yyTemp;
   double distTemp;

   // Circle (approximated):

      for (ii = -s; ii < s+1; ii++) {
      for (jj = -s; jj < s+1; jj++) {
        if ( (ii != 0) | (jj != 0) ) {
        distTemp = sqrt( (double) (ii*ii + jj*jj) );
        if (distTemp <= (double) s) {
          xxTemp = (x + ii + xsize) % xsize;
          yyTemp = (y + jj + ysize) % ysize;

          index++ ;

          objArray[index] = *mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
          // objArray[index] = [self getObjectAtX: xxTemp Y: yyTemp];

          xyArray[index] = (xxTemp << 16) + yyTemp;
          // xyArray[index] = xxTemp*65536 + yyTemp;

          pArray[index] = (id) mo2dSiteAt(lattice,offsets,xxTemp,yyTemp);
          xArray[index] = xxTemp;
          yArray[index] = yyTemp;

        };
        };
      }; };

   return self;
}


// Select neighborhood cells based on type and size:


-addCells {
   // int i;

   index = -1;
 
   switch (thisType) {
     case AxesOnly: 
	[self addAxesAtX:   thisX Y: thisY Size: thisSize];
	break;
     case DiagsOnly: 
	[self addDiagsAtX:  thisX Y: thisY Size: thisSize];
	break;
     case AxesNDiags: 
	[self addAxesAtX:   thisX Y: thisY Size: thisSize];
	[self addDiagsAtX:  thisX Y: thisY Size: thisSize];
	break;
     case aStar: 
	[self addAxesAtX:   thisX Y: thisY Size: thisSize];
	[self addDiagsAtX:  thisX Y: thisY Size: thisSize/2];
	break;
     case aBox: 
	[self addBoxAtX:    thisX Y: thisY Size: thisSize];
	break;
     case aCircle: 
	[self addCircleAtX: thisX Y: thisY Size: thisSize];
	break;
   default: 
	[InvalidCombination raiseEvent:
	 "MoXGrid2d addCells: (%d) invalid neighborhood type\n", thisType];
	break;
   }

   if (index < 0) 
   [InvalidCombination raiseEvent:
   "MoXGrid2d addCells: major error -- no cells found\n"];

/*
   // Printing for debug purposes:
   printf("\naddCells found %d objects:\n", index+1);
   for (i=0; i<=index; i++) {
     printf("x=%3d y=%3d p=%10p obj=%10p xy=%d -> x=%3d y=%3d or x=%3d y=%3d\n",
            xArray[i], yArray[i],
	    pArray[i], objArray[i], xyArray[i],
            xyArray[i] / 65536, xyArray[i] % 65536,
	    (xyArray[i] >> 16), (xyArray[i] & 0xffff) );
   }
   printf("\n");
*/

   return self;
}

// Return pointers to the whole neighborhood,
// using one-time parameters:

-getNeighborhoodAtX: (int) x Y: (int) y 
               Type: (int) type Size: (int) size Shuffled: (BOOL) s {

   id list;
   int i;

#ifdef DOALLTESTS

   if ( (size > xsize/2) || (size > ysize/2))
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getNeighborhood: (%d) size too large\n", size];

   if ( (type < 1) || (type >= InvalidType) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getNeighborhood: (%d) type not valid\n", type];

   if ( (x < 0) || (x >= xsize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getNeighborhood: (%d) x not valid\n", x];

   if ( (y < 0) || (y >= ysize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getNeighborhood: (%d) y not valid\n", y];

#endif

   if (size < 1) return nil;

   thisType = type;
   thisX    = x;
   thisY    = y;
   thisSize = size;

   [self addCells];

   // Create a new empty list:
   list = [List create: [self getZone]];

   // Fill the list with data for all cells.
   for (i=0; i<=index; i++)
     [list addLast: (id) xyArray[i] ];		// compressed x+y values

   if ([list getCount] == 0) {			// Should not happen
     [list drop];
     return nil;
   } else {
     if (s) [shuffleObject shuffleWholeList: list];
     return list;
   }

}


// Return pointers to vacant cells in the neighborhood,
// using one-time parameters:

-getVacanciesAtX: (int) x Y: (int) y 
            Type: (int) type Size: (int) size Shuffled: (BOOL) s {

   id list;
   int i;

#ifdef DOALLTESTS

   if ( (size > xsize/2) || (size > ysize/2))
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getVacancies: (%d) size too large\n", size];

   if ( (type < 1) || (type >= InvalidType) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getVacancies: (%d) type not valid\n", type];

   if ( (x < 0) || (x >= xsize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getVacancies: (%d) x not valid\n", x];

   if ( (y < 0) || (y >= ysize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getVacancies: (%d) y not valid\n", y];

#endif

   if (size < 1) return nil;

   thisType = type;
   thisX    = x;
   thisY    = y;
   thisSize = size;

   [self addCells];

   // Create a new empty list:
   list = [List create: [self getZone]];

   // Fill the list with data for unoccupied (vacant) cells only.
   for (i=0; i<=index; i++)
     if (objArray[i] == nil) 
	[list addLast: (id) xyArray[i]];	// compressed x+y values

   if ([list getCount] == 0) {
     [list drop];
     return nil;
   } else {
     if (s) [shuffleObject shuffleWholeList: list];
     return list;
   }
}


// Return pointers to occupied cells in the neighborhood,
// in the form of a LIST of LISTS,
// using one-time parameters:
-getNeighborListsAtX: (int) x Y: (int) y 
	    Type: (int) type Size: (int) size 
	    Shuffled: (BOOL) s {

   id list;
   int i;

#ifdef DOALLTESTS

   if ( (size > xsize/2) || (size > ysize/2))
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getNeighbors: (%d) size too large\n", size];

   if ( (type < 1) || (type >= InvalidType) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getNeighbors: (%d) type not valid\n", type];

   if ( (x < 0) || (x >= xsize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getNeighbors: (%d) x not valid\n", x];

   if ( (y < 0) || (y >= ysize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d: getNeighbors: (%d) y not valid\n", y];

#endif

   if (size < 1) return nil;

   thisType = type;
   thisX    = x;
   thisY    = y;
   thisSize = size;

   [self addCells];

   // Create a new empty list:
   list = [List create: [self getZone]];

   // Fill the list with data for occupied cells only.
   for (i=0; i<=index; i++) 
     if (objArray[i] != nil) 
	[list addLast: objArray[i]];		// object pointers

   if ([list getCount] == 0) {
     [list drop];
     return nil;
   } else {
     if (s) [shuffleObject shuffleWholeList: list];
     return list;
   }

}

// Return pointers to the whole neighborhood,
// using previously set values for type, size, shuffle:

-getNeighborhoodAtX: (int) x Y: (int) y {
   id list;
   int i;

#ifdef DOALLTESTS

   // No need to test type or size.

   if ( (x < 0) || (x >= xsize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d getNeighborhood: (%d) invalid x-value\n", x];

   if ( (y < 0) || (y >= ysize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d getNeighborhood: (%d) invalid y-value\n", y];

#endif

   thisType = defaultType;
   thisX    = x;
   thisY    = y;
   thisSize = defaultSize;

   [self addCells];

   // Create a new empty list:
   list = [List create: [self getZone]];

   // Fill the list with data for all cells.
   for (i=0; i<=index; i++)
     [list addLast: (id) xyArray[i]];		// compressed x+y values

   if ([list getCount] == 0) {			// Should not happen
     [list drop];
     return nil;
   } else {
     if (defaultShuffle) [shuffleObject shuffleWholeList: list];
     return list;
   }

}

// Return pointers to vacant cells in the neighborhood,
// using previously set values for type, size, shuffle:

-getVacanciesAtX: (int) x Y: (int) y {
   id list;
   int i;

#ifdef DOALLTESTS

   // No need to test type or size.

   if ( (x < 0) || (x >= xsize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d getVacancies: (%d) invalid x-value\n", x];

   if ( (y < 0) || (y >= ysize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d getVacancies: (%d) invalid y-value\n", y];

#endif

   thisType = defaultType;
   thisX    = x;
   thisY    = y;
   thisSize = defaultSize;

   [self addCells];

   // Create a new empty list:
   list = [List create: [self getZone]];

   // Fill the list with data for unoccupied (vacant) cells only.
   for (i=0; i<=index; i++)
     if (objArray[i] == nil) 
	[list addLast: (id) xyArray[i]];	// compressed x+y values

   if ([list getCount] == 0) {
     [list drop];
     return nil;
   } else {
     if (defaultShuffle) [shuffleObject shuffleWholeList: list];
     return list;
   }
}

// Return pointers to occupied cells in the neighborhood,
// in the form of a LIST of LISTS,
// using previously set values for type, size, shuffle:

-getNeighborListsAtX: (int) x Y: (int) y {
   id list;
   int i;

#ifdef DOALLTESTS

   // No need to test type or size.

   if ( (x < 0) || (x >= xsize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d getNeighbors: (%d) invalid x-value\n", x];

   if ( (y < 0) || (y >= ysize) )
   [InvalidCombination raiseEvent:
   "MoXGrid2d getNeighbors: (%d) invalid y-value\n", y];

#endif

   thisType = defaultType;
   thisX    = x;
   thisY    = y;
   thisSize = defaultSize;

   [self addCells];

   // Create a new empty list:
   list = [List create: [self getZone]];

   // Fill the list with data for occupied cells only.
   for (i=0; i<=index; i++) 
     if (objArray[i] != nil) 
	[list addLast: objArray[i]];		// object pointers

   if ([list getCount] == 0) {
     [list drop];
     return nil;
   } else {
     if (defaultShuffle) [shuffleObject shuffleWholeList: list];
     return list;
   }
}

@end

