//  2d Feeding Optimization
//  Copyright 8/18/00 Andrew Lovett, under the supervision of Louis Gross
//  Check the file Info/CopyrightInfo.txt for copyright information

// This class combines the 2d world of the agents with the regions of Foodspace

import swarm.objectbase.Swarm;
import swarm.objectbase.SwarmImpl;
import swarm.Globals;
import swarm.space.Grid2d;
import swarm.space.Grid2dImpl;
import swarm.objectbase.VarProbe;
import swarm.objectbase.MessageProbe;
import swarm.objectbase.EmptyProbeMapImpl;
import swarm.defobj.Zone;
import swarm.activity.Schedule;
import swarm.activity.ScheduleImpl;
import swarm.activity.Activity;
import swarm.Selector;

import java.util.LinkedList;
import java.util.List;
 	
public class Space2d extends Grid2dImpl {
	public static SimulationState state; //for whenever we need to update or view 							//the simulation state

	public Schedule seasonSchedule;  //we use this to schedule all the seasons
	public Schedule foodSchedule; //we use this to dump food into the regions at 							//regular intervals
	public Selector Winter, Spring, Summer, Fall;  //these selectors refer to the 				//functions which will be called each time the season changes

	private Foodspace[][] region; //the array of foodspaces; each Foodspace is a 2d 		//array of ints giving the location of all food found within that region

	private int regionX, regionY; //the dimenions of the regions
	public static int numRegions; //the total number of Foodspace regions
	private int fsRows; //number of foodspace rows and columns 
	private int fsCols; //used to fill the entire grid
	private double seasonMod = 1; //a modifier for the amount of food that is 							//dumped changes with the seasons
	private int foodDumpType = 3; //determines which function is used to dump food

	//Information on the total food found in each region at a given time and the 	//total number of death that have occured in that region up to that time; 	//allows for up to eight regions; I really wish this information could be held 	//in arrays, but the Probe Window can only display values for basic variable 	//types; so, if you want to create more than 8 regions and you're running in 	//graphics mode, you're going to have to add more variables in here
	public int Food_In_Region_1;
	public int Food_In_Region_2;
	public int Food_In_Region_3;
	public int Food_In_Region_4;
	public int Food_In_Region_5;
	public int Food_In_Region_6;
	public int Food_In_Region_7;
	public int Food_In_Region_8;
	public int Deaths_In_Region_1;
	public int Deaths_In_Region_2;
	public int Deaths_In_Region_3;
	public int Deaths_In_Region_4;
	public int Deaths_In_Region_5;
	public int Deaths_In_Region_6;
	public int Deaths_In_Region_7;
	public int Deaths_In_Region_8;

	//amount of food to be dropped in each region each season
	private static int[] foodForRegion;

	//these arrays mirror the Food_In_Region_x and Deaths_In_Region_x variables
	public static int[] foodInRegion;
	public static int[] deathsInRegion;

	//the number of living animals currently in a region
	public static int[] animalsInRegion;

	//we'll keep the next three variables in all-caps and treat them as constants, 	//as they are given values only once, at the program's beginning, when their 	//values are read from the file EnvironmentalData.txt
	public static int TIME_BETWEEN_FOOD; //time between food dumps
	public static int SEASON_LENGTH; //number of timesteps between 										//season changes

	//and now let's create mirrors of these two values that will not be static, 	//meaning they will be able to appear in the probe display window
	public int time_between_food;
	public int season_length;

	//the maximum amount of food that can be set to be dumped in all the regions 	//each time food is dumped; is reset each time 
	public static int MAX_FOOD_DUMP;

	//the number of functions that can be used to dump food
	public final static int NUM_DUMP_METHODS = 8;

	//the minimum ratio which can be assigned to a region in dump methods 2 & 5 or 	//the portion of method 6 based on method 2
	private final double MIN_RATIO = .05;

//Instructions for adding a food dump method:
//1: add 1 to NUM_DUMP_METHODS, the new method's number will be the same as your sum
//ie, if you are changine NUM_DUMP_METHODS from 10 to 11, 11 is the number of your //method
//2: write your function, naming it "dumpAllFood" + i; i is the number of the method

	//constructor
	public Space2d (Zone thZone, int x, int y, int fsCols, 
												int fsRows) {
		//the x and y here are actually the dimensions 
		//of the Foodspaces within Space2d
		//fsRows and fsCols tell how many rows and columns of Foodspaces
		//should fill the Space2d

		super (thZone, fsCols * x, fsRows * y);
		
		this.fsRows = fsRows;
		this.fsCols = fsCols;
		numRegions = fsRows * fsCols;
		regionX = x;
		regionY = y;

		region = new Foodspace [fsCols][fsRows];

		foodForRegion = new int[numRegions];
		deathsInRegion = new int[numRegions];
		foodInRegion = new int[numRegions];
		animalsInRegion = new int[numRegions];		

		//empty the array which tells how much food to drop during each season
		for (int i = 0; i <= numRegions - 1; i++) {
			foodForRegion[i] = 0;
			deathsInRegion[i] = 0;
			foodInRegion[i] = 0;
			animalsInRegion[i] = 0;
		}
	
		//fill our 2d array with empty Foodspaces
		for (int i = 0; i <= fsCols - 1; i++)
			for (int j = 0; j <= fsRows - 1; j++)
				region [i][j] = new Foodspace (x, y, i + 1 + j * fsCols, 0);
		
		//set our variables for the display window
		time_between_food = TIME_BETWEEN_FOOD;
		season_length = SEASON_LENGTH;

		//set up the food-dumping schedule
		foodSchedule = new ScheduleImpl (getZone (), TIME_BETWEEN_FOOD);

		try {
			foodSchedule.at$createActionTo$message (0, this, 
				new Selector (getClass (), "dumpAllFood", false));
		} catch (Exception e) {
			e.printStackTrace (System.err);
			System.exit (1);
		}

		//And now we set up and begin scheduling the seasons
		seasonSchedule = new ScheduleImpl (getZone (), true);

		//assign a function to each of the selectors
		try {
			Winter = new Selector (getClass (), "doWinter", false);
			Spring = new Selector (getClass (), "doSpring", false);
			Fall = new Selector (getClass (), "doFall", false);
			Summer = new Selector (getClass (), "doSummer", false);
		
			//the first season, Summer, is scheduled to occur immediately after 			//the program begins running
			seasonSchedule.at$createActionTo$message (0, 
										this, Summer);
		} catch (Exception e) {
			e.printStackTrace (System.err);
			System.exit (1);
		}
	
		//now we're done, unless we're using graphics
		if (!Landscape.isGraphical)
			return;	

		//Here's Space2d's own Probe Map, which displays info about individual
		//regions.  This could all go in Landscape's Probe Map, but it's easier 
		//to follow everything if info is split up into multiple ProbeMaps
		class Space2dProbeMap extends EmptyProbeMapImpl {
			private VarProbe probeVariable (String name) {
				return Globals.env.probeLibrary.getProbeForVariable$inClass
					(name, Space2d.this.getClass ());
			}

			private MessageProbe probeMessage (String name) {
       			return 
					Globals.env.probeLibrary.getProbeForMessage$inClass
          				(name, Space2d.this.getClass ());
      		}

			//our probe map constructor; this gets a little complicated because 			//a for loop is used to create a food probe, a death probe, and a 			//message to set the amount that will be dropped in a region for 				//each region
			public Space2dProbeMap (Zone aZone, Class thClass) {
				super (aZone, thClass);

				addProbe (probeVariable ("season_length"));
				addProbe (probeVariable ("time_between_food"));

				for (int i = 1; i <= numRegions; i++) {
					addProbe (probeVariable ("Food_In_Region_" + i));
					addProbe (probeVariable ("Deaths_In_Region_" + i));
				}

				addProbe (probeMessage ("setFoodForRegion::"));
				addProbe (probeMessage ("setFoodDumpType:"));
				addProbe (probeMessage ("getInfoOnFoodDumpType:"));
			}
		}

		Globals.env.probeLibrary.setProbeMap$For 
			(new Space2dProbeMap (thZone, getClass ()), getClass ()); 
	}
	
	public Foodspace getRegion (int x, int y) {
		return region[x][y];
	}

	//each season function adds the appropriate amount of food to each region and 	//then schedules the next season to occur SEASON_LENGTH time steps later
	public void doWinter () {
		state.Season = 'W';
		seasonMod = .6;

		seasonSchedule.at$createActionTo$message 
			(Globals.env.getCurrentTime () + SEASON_LENGTH, this, Spring);
	}

	public void doSpring () {
		state.Season = 'S';
		seasonMod = 1.2;

		seasonSchedule.at$createActionTo$message 
			(Globals.env.getCurrentTime () + SEASON_LENGTH, this, Summer);
	}
	
	public void doSummer () {
		state.Season = 'U';
		seasonMod = 1;
			
		seasonSchedule.at$createActionTo$message 
			(Globals.env.getCurrentTime () + SEASON_LENGTH, this, Fall);
	}

	public void doFall () {
		state.Season = 'F';
		seasonMod = .8;

		seasonSchedule.at$createActionTo$message 
			(Globals.env.getCurrentTime () + SEASON_LENGTH, this, Winter);
	}

	//calls whichever dumpAllFood function the user has selected, 1 is the default
	public void dumpAllFood () {
		Class c = this.getClass ();
		try {
			c.getMethod ("dumpAllFood" + foodDumpType, null).invoke 
										(this, null);
		}
		catch (Exception e) {
			System.err.println ("There's a problem with your food dump type.");
			System.exit (1);
		}
	}

	//dumps the appropriate amount of food in each region, the appropriate amount 	//being the amount chosen by the user
	public void dumpAllFood1 () {
		int regNum; //the number of the region
		int foodToAdd; //amount of food that will be added to a region

		for (int i = 0; i <= fsCols - 1; i++)
			for (int j = 0; j <= fsRows - 1; j++) {
				regNum = regionNum (i * regionX, j * regionY);
				foodToAdd = (int) (foodForRegion [regNum - 1] * seasonMod);

				(region [i][j]).dumpFood (foodToAdd);
			}	
	}

	//an alternate food dump method
	//this one dumps a greater amount of food in those regions that currently hold 	//a lesser amount of food; each region must get a percentage of the food no 	//less than MIN_RATIO
	public void dumpAllFood2 () {
		int regNum; //number of the given region
		int foodToAdd; //the amount of food that will be added to a region
		double totalFood = state.Total_Food; //used to find ratios
		double ratio; //the ratio that is found
		double maxratio = (1 - MIN_RATIO) / (numRegions - 1); //the highest 					//percentage of the total food which one region can get
	
		for (int i = 0; i <= fsCols - 1; i++)
			for (int j = 0; j <= fsRows - 1; j++) {
				regNum = regionNum (i * regionX, j * regionY);
				if (totalFood != 0) {
					ratio = (region [i][j]).getTotalFood () / totalFood;

					//we want the region with a greater ratio of the food 					//to get a lesser ration of the amount of food 							//available, so...
					ratio = 1 - ratio;
					//now fit the ratio within the bounds of min/max ratios
					ratio = ratio * (maxratio - MIN_RATIO) + MIN_RATIO;

					foodToAdd = (int) (ratio * MAX_FOOD_DUMP * seasonMod);
				}
				else  //if the total food is 0, just split everything evenly
					foodToAdd = (int) (MAX_FOOD_DUMP * seasonMod / 														numRegions);

				(region [i][j]).dumpFood (foodToAdd);
			} //for loop
	}

	//this method finds the region with the most food and attempts to dump enough 	//food in all regions to give them equal amounts, splitting any remaining food 	//evenly among the regions; it begins by adding food to those regions with the 	//least food
	public void dumpAllFood3 () {
		int regNum; //number of a given region
		int foodHere; //amount of food in a given region
		int foodToAdd; //the amount of food that will be added to a region
		int maxFood = -1; //info on the food in theregion with the most food
		int minFood; //info on the food in the region with the least food
		int numMin; //number of regions ties with the least food
		int foodLeft = (int) (MAX_FOOD_DUMP * seasonMod); //each time food is 										//dumped, it is taken from here
		LinkedList minList = new LinkedList (); //if there are ties, we randomly 										//pick one from the list
		for (int i = 0; i <= fsCols - 1; i++)
			for (int j = 0; j <= fsRows - 1; j++) {
				foodHere = (region[i][j]).getTotalFood ();
				if (foodHere > maxFood)
					maxFood = foodHere;
			} 

		do {	
			//initialize min stuff
			minFood = 99999;
			numMin = 0;	
			for (int i = 0; i <= fsCols - 1; i++)
				for (int j = 0; j <= fsRows - 1; j++) {
					foodHere = (region[i][j]).getTotalFood ();
					if (foodHere < minFood) {
						minFood = foodHere;
						numMin = 1;
						minList.clear ();
						minList.add (region[i][j]);
					}
					else if (foodHere == minFood) {
						numMin++;
						minList.add (region[i][j]);
					}
				}

			int numMin2 = numMin; //hafta copy numMin so we can modify it 							//without affecting the for loop
			for (int i = 0; i <= numMin - 1; i ++) {
				foodToAdd = (foodLeft > 9) ? 10 : foodLeft;
				regNum = Globals.env.uniformIntRand.getIntegerWithMin$withMax 
											(0, numMin2 - 1);

				((Foodspace ) (minList.remove (regNum))).dumpFood 												(foodToAdd);
				numMin2--;
				foodLeft -= foodToAdd;
				if (foodLeft == 0)
					return;
			}
		} while (minFood < maxFood);

		if (foodLeft >= numRegions) {
			foodToAdd = foodLeft / numRegions;
			for (int i = 0; i <= fsCols - 1; i++)
				for (int j = 0; j <= fsRows - 1; j++) 
					(region[i][j]).dumpFood (foodToAdd);
		}
	}

	//dumps food only into regions in which animals are currently found, giving a 	//higher percentage of the food to those regions with more animals
	public void dumpAllFood4 () {
		double foodRatio; //percentage of the total food that will be added
		int reg; //a region's number
		for (int i = 0; i <= fsCols - 1; i++) 
			for (int j = 0; j <= fsRows - 1; j++) {
				reg = regionNum (i * regionX, j * regionY);
				foodRatio = animalsInRegion[reg - 1] / 
								((double) state.Total_Animals);
				
				region[i][j].dumpFood 
						((int) (foodRatio * MAX_FOOD_DUMP * seasonMod));
			}
	}

	//this one is almost like a combination of methods 4 and 2; like 4, it dumps 	//food depending on how many animals are in each region, but like 2, it never 	//dumps less than MIN_RATIO in a given region
	public void dumpAllFood5 () {
		int regNum; //number of the given region
		int foodToAdd; //the amount of food that will be added to a region
		double totalAnimals = (double) state.Total_Animals; //used to find 												//ratios
		double ratio; //the ratio that is found
		double maxratio = 1 - ((numRegions - 1) * MIN_RATIO); //the highest 					//percentage of the total food which one region can get

		for (int i = 0; i <= fsCols - 1; i++)
			for (int j = 0; j <= fsRows - 1; j++) {
				regNum = regionNum (i * regionX, j * regionY);
				ratio = animalsInRegion[regNum - 1] / totalAnimals;

				//now fit the ratio within the bounds of .1 -> maxratio
				ratio = ratio * (maxratio - MIN_RATIO) + MIN_RATIO;
				foodToAdd = (int) (ratio * MAX_FOOD_DUMP * seasonMod);
				
				(region [i][j]).dumpFood (foodToAdd);
			}
	}

	//this food dump method uses an average of the ratios obtained in methods 2 & 4
	public void dumpAllFood6 () {
		int regNum; //number of the given region
		int foodToAdd; //the amount of food that will be added to a region
		double totalFood = state.Total_Food; //used to find ratios
		double ratio1; //the ratio that is found using method 2
		double ratio2; //the ratio that is found using method 4
		double ratio; //the average of the two ratios
		double maxratio = (1 - MIN_RATIO) / (numRegions - 1); //the highest 					//ratio1 which one region can get

		for (int i = 0; i <= fsCols - 1; i++)
			for (int j = 0; j <= fsRows - 1; j++) {
				regNum = regionNum (i * regionX, j * regionY);
				if (totalFood != 0) {
					ratio1 = (region [i][j]).getTotalFood () / totalFood;

					//we want the region with a greater ratio of the food 					//to get a lesser ration of the amount of food 							//available, so...
					ratio1 = 1 - ratio1;
					//now fit the ratio within the bounds of min/max ratios
					ratio1 = ratio1 * (maxratio - MIN_RATIO) + MIN_RATIO;
				}
				else  //if the total food is 0, just split everything evenly
					ratio1 = 1 / ((double) numRegions);

				//now let's do ratio2
				ratio2 = animalsInRegion[regNum - 1] / 
							((double) state.Total_Animals);						
				//now average out the two ratios to create a final ratio
				ratio = (ratio1 * 2 + ratio2) / 3;

				foodToAdd = (int) (ratio * MAX_FOOD_DUMP * seasonMod);
				(region [i][j]).dumpFood (foodToAdd);
			} //for loop
	}

	//this dump method, which I don't plan to take very seriously, dumps food 	//randomly into the various regions until it has run out of food to dump; it 	//dumps 10 food at a time
	public void dumpAllFood7 () {
		final int foodToAdd = 10; //the amount of food that will be dumped
		int foodRemaining = (int) (MAX_FOOD_DUMP * seasonMod); //keeps track of 								//how much food we have left to dump

		while (foodToAdd <= foodRemaining) {
			int reg = Globals.env.uniformIntRand.getIntegerWithMin$withMax 												(1, numRegions);
			region[regXVal (reg)][regYVal (reg)].dumpFood (foodToAdd);
		
			foodRemaining -= foodToAdd;
		}
	}

	//just for fun, here is a slightly safer variation of the random dump method		//(method 7); this one splits a third of the food among the regions and then 	//dumps the rest randomly
	public void dumpAllFood8 () {
		int foodToAdd; //the amount of food that will be dumped
		int thirdOfFood; //a third of the total food;
		int foodRemaining = (int) (MAX_FOOD_DUMP * seasonMod); //keeps track of 								//how much food we have left to dump
		thirdOfFood = foodRemaining / 3;
		foodRemaining -= thirdOfFood;
		foodToAdd = thirdOfFood / numRegions;

		for (int i = 0; i <= fsCols - 1; i++)
			for (int j = 0; j <= fsRows - 1; j++)
				region[i][j].dumpFood (foodToAdd);

		foodToAdd = 10;
		while (foodToAdd <= foodRemaining) {
			int reg = Globals.env.uniformIntRand.getIntegerWithMin$withMax 												(1, numRegions);
			region[regXVal (reg)][regYVal (reg)].dumpFood (foodToAdd);
		
			foodRemaining -= foodToAdd;
		}
	}
	
	//outputs info on a food dump type for the user
	public void getInfoOnFoodDumpType (int type) {
		switch (type) {
			case 1: System.out.println 
("Dumps an amount of food determined by the user into each region.  Make sure to use setFoodDumpType to set an amount of food for each region, as the defaults are 0.");	
				break;
			case 2: System.out.println 
("Dumps a greater amount of food in those regions that currently contain a lesser amount of food.  Each region will get at least " + (MIN_RATIO * 100) + "% of the total food to be dumped.");
				break;
			case 3: System.out.println
("Finds the region with the most food and attempts to dump enough food in all other regions to give them the same total amount, splitting any remaining food evenly among the regions; begins by adding food to those regions with the least food.");
				break;
			case 4: System.out.println
("Dumps food only into regions in which animals are currently found, giving a higher percentage of the food to those regions with more animals.");
				break;
			case 5: System.out.println
("Dumps a greater amount of food in those regions that contain a greater number of animals.  Each region will get at least " + (MIN_RATIO * 100) + "% of the total food to be dumped.");
				break;
			case 6: System.out.println
("Dumps food using a weighted average of the ratios found using methods 2 and 4.");
				break;
			case 7: System.out.println
("Dumps food randomly, dumping 10 food into a region at a time, until all the food has been used up.");
				break;
			case 8: System.out.println
("Splits a third of the total food among the regions and then dumps the rest of it randomly, dumping 10 food into a region at a time.");
				break;
			default: System.out.println 
("There are only " + NUM_DUMP_METHODS + " dumps methods, so you must pick a number between 1 and " + NUM_DUMP_METHODS + ".");
		}
	}



	//returns a number for the region into which the point falls; starts with 1
	//regions are numbered left to right, starting with the top row
	public int regionNum (int x, int y) { 
		int xindex, yindex;
	
		xindex = x / regionX + 1;
		yindex = y / regionY;
	
		return (xindex + fsCols * yindex);	
	}

	//and now we return values for the location in region[][] of a region number
	//XVal is the index in the first dimension in the array, YVal is for the second
	public int regXVal (int reg) {
		return ((reg - 1) % fsCols);
	}

	public int regYVal (int reg) {
		return ((reg - 1) / fsCols);
	}

	//now  return values for the location in region[][] given the x and y values on 	//the grid
	public int regXVal2 (int xcoord) {
		return (xcoord / regionX);
	}

	public int regYVal2 (int ycoord) {
		return (ycoord / regionY);
	}

	//returns true only if one could move a distance of range in any direction from 	//the point (x, y) and still be in the same region on the grid as (x, y)
	public boolean rangeIsWithinRegion (int x, int y, int range) {
		if ((regXVal2 (x) == regXVal2 (x + range))
		&& (regXVal2 (x) == regXVal2 (x - range))
		&& (regYVal2 (y) == regYVal2 (y + range))
		&& (regYVal2 (y) == regYVal2 (y - range))) {
			return true;
		}
		
		return false;
	}

	//returns true only if one could move a distance of range in any direction from 	//the point (x, y) without enter a region that contains any food
	public boolean regionWithFoodWithinRange (int x, int y, int range) {
		for (int i = regXVal2 (x - range); i <= regXVal2 (x + range); i++)
			for (int j = regYVal2 (y - range); j <= regYVal2 (y + range); j++)
				if ((i >= 0) && (j >= 0) && (i < fsCols) && (j < fsRows)) 
					if (region[i][j].getTotalFood () > 0)
						return true;

		return false;
	}

	//returns the amount of food at a given point in the 2d grid by finding the 	//appropriate coordinates on the appropriate Foodspace
	public int getFood (int x, int y) {
		int xindex, yindex; //used to locate the correct foodspace
		int xloc, yloc; //the position within the correct foodspace;
		
		xindex = x / regionX;
		yindex = y / regionY;
		xloc = x % regionX;
		yloc = y % regionY;

		return (region [xindex][yindex]).getFood (xloc, yloc);
	}

	//uses a similar algorithm to add an amount of food to a point on the grid, 	//also uses modFoodDeathForRegion to update the correct field regarding how 	//much food remains in each region
	public void addFood (int x, int y, int amount) {
		int xindex, yindex; //used to locate the correct foodspace
		int xloc, yloc; //the position within the correct foodspace;
		
		xindex = x / regionX;
		yindex = y / regionY;
		xloc = x % regionX;
		yloc = y % regionY;

		(region [xindex][yindex]).addFood (xloc, yloc, amount);
	}

	//update number of deaths for a region, takes the coordinates on the grid
	public void dieHere (int x, int y) {
		int reg = regionNum (x, y);
		modFoodDeathForRegion (0, 1, reg);
	}
	
	//allows the user to choose which function will be used to dump food
	public int setFoodDumpType (int choice) {
		if ((choice > 0) && (choice <= NUM_DUMP_METHODS)) {
			foodDumpType = choice;
			return choice;
		}

		System.out.println ("That is not a valid choice.");
		return 0;
	}

	//function to set the amount of food to be dumped into each region
	public int setFoodForRegion (int amount, int region) {
		int oldFood = foodForRegion[region - 1];
		foodForRegion[region - 1] = amount;
		if (totalFoodForRegions () > MAX_FOOD_DUMP) {
			System.out.println 												("The total food cannot be greater than " + MAX_FOOD_DUMP);
			foodForRegion[region - 1] = oldFood;
			return oldFood;
		}
		return amount;
	}

	//function used to modify arrays that mirror those annoying ints that can't be 	//put into an array
	public static void modFoodDeathForRegion (int food, int death, int region) {
		foodInRegion[region - 1] += food;
		deathsInRegion[region - 1] += death;
	}

	//function used to update those annoying ints that can't be put into an array
	//using the info in the arrays that mirror them
	public void updateFoodAndDeath () {
		int amount; //the amount of food or the number of deaths in a region
		Class c;
		
		try {
			c = Class.forName ("Space2d");
			for (int reg = 1; reg <= numRegions; reg++) {
				c.getField ("Food_In_Region_" + reg).setInt 
								(this, foodInRegion[reg - 1]);

				c.getField ("Deaths_In_Region_" + reg).setInt 
								(this, deathsInRegion[reg - 1]);
			}
		}
		catch (Exception e) {
			System.err.println ("There was an error in updating the Food_In_Region_# and Deaths_In_Region_# fields.");

			System.exit (1);
		}
	}

	//finds the total amount of food to be dumped in all regions; used to ensure 	//that the user never tries to dump more the maximum amount of food
	public int totalFoodForRegions () {
		int sum = 0;
		for (int i = 0; i <= numRegions - 1; i++)
			sum += foodForRegion[i];

		return sum;
	}

	//activates our season schedule; this function uses the name "activateIn" but 	//is not a typical activateIn function because Space2d is not a subclass of 	//SwarmImpl; the different return type (void instead of Activity) gets us a 	//warning message, but it shouldn't matter
	public void activateIn (Swarm papaswarm) {
		seasonSchedule.activateIn (papaswarm);
		foodSchedule.activateIn (papaswarm);
	}	

	//splits the amount of food available for dumping up evenly among the regions
	public void evenRegions () {	
		int foodToDump = MAX_FOOD_DUMP / numRegions;
		for (int i = 0; i <= numRegions - 1; i++)
			foodForRegion[i] = foodToDump;
	}

	//overrides the Grid2d function so that we can use this function to update info 	//on the region in which each animal is found
	public Object putObject$atX$Y (Object anObject, int x, int y) {
		super.putObject$atX$Y (anObject, x, y);

		int reg = regionNum (x, y);
		if (anObject instanceof Animal)
			animalsInRegion[reg - 1]++;
		else //if it's not an animal, we must be using null 
					//to take away an animal
			animalsInRegion[reg - 1]--;

		return anObject;
	}
}			