// Copyright (C) 1999 University of Washington, Columbia Basin Research.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.


#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#import <math.h>
#import <random.h>
#import "SearchableTreeNode.h"
#import "SearchableGridSpace.h"
#import "SearchableSpace.h"

@implementation SearchableSpace
+ createBegin: aZone {
  return [super createBegin: aZone];
}

- setUseTree: (BOOL) bUse {
  useTree = bUse;
  return self;
}

- setAllowDups: (BOOL) bAllow {
  allowDups = bAllow;
  return self;
}  

- setIntCoords: (BOOL) bInt {
  intCoords = bInt;
  return self;
}

- setMinX: (double) minx maxX: (double) maxx
     minY: (double) miny maxY: (double) maxy {
  minX = minx;
  maxX = maxx;
  minY = miny;
  maxY = maxy;
  return self;
}

- setGridSpacing: (double) spacing {
  gridSpacing = spacing;
  return self;
}

- createEnd {

  if (useTree) {
    treeBase = nil;
    freeList = nil;
  }
  else {
    gridSpace = [SearchableGridSpace createBegin: [self getZone]];
    [gridSpace setAllowDups: allowDups];
    [gridSpace setIntCoords: intCoords];
    [gridSpace setMinX: minX maxX: maxX minY: minY maxY: maxY
               gridSpacing: gridSpacing];
    [gridSpace createEnd];
    [gridSpace clear];
  }

  [super createEnd];
  return self;
}


- (void)drop {
  if (useTree) {
    [self dropTreeNode: treeBase];
    [self dropTreeNode: freeList];
  }
  else {
    [gridSpace drop];
  }
  [super drop];
}

// This routine answers the question "which object is closest
//  to (x,y)?". This may involve recursively searching the binary
//  space partitioning tree, or it may involve iteratively searching the
//  SearchableGridSpace, with culling where possible.
// This routine accepts all objects.
- getNearestObjectToX: (double) x Y: (double) y {
  // Set up for recursive search.
  searchX = x;
  searchY = y;
  minDistSq = 1.0e20;
  minDistObj = nil;
  valueSwitch = ALL;

  // Perform search.
  if (!useTree)
    [self getClosest];
  else
    [self getClosestIn: treeBase];

  // Return result.
  if (minDistObj == nil)  // Should never happen
    printf( "getNearestObjectToX: %f Y: %f  failed!\n", searchX, searchY );
  return minDistObj;
}

// This routine answers the question "which object is closest
//  to (x,y)?". This may involve recursively searching the binary
//  space partitioning tree, or it may involve iteratively searching the
//  SearchableGridSpace, with culling where possible.
// This routine calls getVal on all objects and performs screening based
//  on a comparison type and a comparison value.
// For instance,
//  obj = [ss getNearestObjectToX: x Y: y WithValue: GT To: 0.0];
//  returns the object closest to x,y which satisfies ([obj getVal] > 0.0)
- getNearestObjectToX: (double) x Y: (double) y WithValue: (int) comparison
                                                To: (double) compareValue {

  // Set up for recursive search.
  searchX = x;
  searchY = y;
  minDistSq = 1.0e20;
  minDistObj = nil;
  valueSwitch = comparison;
  compareTo = compareValue;

  // Perform search.
  if (!useTree)
    [self getClosest];
  else
    [self getClosestIn: treeBase];

  // Return result. Result may be nil if no object satisfied the selection
  //  criteria.
  return minDistObj;
}

// Search the SearchableGridSpace of objects to find the closest one to
//  the given point (above).
- getClosest {
  int shell = 0;
  int i, j, centX, centY, cellX, cellY;
  id obj;
  double distSq;
  double delta;
  double value, x, y;
  BOOL accept;

  cellX = cellY = 0;  // keeps compiler happy

  centX = (int)((searchX - minX) / gridSpacing);
  centY = (int)((searchY - minY) / gridSpacing);

  // The outermost loop searches outwards in shells around the central
  //  grid cell. 
  while (1) {
    // Determine if any cell in the next ring out could possibly have the
    //  closest point. If not, we are done.
    if (shell > 0) {
      delta = searchX - ((1 + centX - shell) * gridSpacing + minX);
      distSq = delta * delta;
      delta = searchX - ((centX + shell) * gridSpacing + minX);
      if (distSq > delta * delta)
        distSq = delta * delta;
      delta = searchY - ((1 + centY - shell) * gridSpacing + minY);
      if (distSq > delta * delta)
        distSq = delta * delta;
      delta = searchY - ((centY + shell) * gridSpacing + minY);
      if (distSq > delta * delta)
        distSq = delta * delta;
      if (distSq > minDistSq)
        break;
    }

    // Loop through the grid cells in the current shell surrounding the
    //  grid cell which contains the search point.
    //  NOTE: some subtleties when shell == 0.
    for (i = -shell; i <= shell; i++) 
      for (j = 0; j < (shell == 0 ? 1 : 4); j++) {
        if (shell != 0 && i == shell)
          continue;
        switch (j) {
          case 0:
            cellX = centX + i;
            cellY = centY - shell;
            break;
          case 1:
            cellX = centX - shell;
            cellY = centY + i + 1;
            break;
          case 2:
            cellX = centX + shell;
            cellY = centY + i;
            break;
          case 3:
            cellX = centX + i + 1;
            cellY = centY + shell;
            break;
        }

        // cellX, cellY is the next grid cell to examine. If it really
        // is a grid cell and not off the edge somehow.
        if (cellX < 0 || cellY < 0)
          continue;
        if (cellX >= (int)[gridSpace getSizeX])
          continue;
        if (cellY >= (int)[gridSpace getSizeY])
          continue;

        // get the object (or list of objects) in that cell, if any.
        obj = [gridSpace getObjectAtX: (unsigned)cellX 
                                            Y: (unsigned)cellY];

        // Loop through the list of objects in a given grid cell.
        while (obj != nil) {
          // Apply screening by the object's getVal value, if requested.
          if (valueSwitch == ALL)
            accept = TRUE;
          else {
            value = [obj getVal];
            switch (valueSwitch) {
              case EQ:
                accept = value == compareTo;
                break;
              case NEQ:
                accept = value != compareTo;
                break;
              case GT:
                accept = value > compareTo;
                break;
              case GE:
                accept = value >= compareTo;
                break;
              case LT:
                accept = value < compareTo;
                break;
              case LE:
                accept = value <= compareTo;
                break;
              default:
                accept = TRUE;
                break;
            }
          }
          if (accept) {
            // We will actually test this object: get it's location.
            if (intCoords) {
              x = [obj getIntX];
              y = [obj getIntY];
            }
            else {
              x = [obj getDoubleX];
              y = [obj getDoubleY];
            }

            // do the test.
            x -= searchX;
            y -= searchY;
            distSq = x * x + y * y;
            if (distSq < minDistSq) {
              minDistSq = distSq;
              minDistObj = obj;
            }
          }

          // Loop around for next object in grid cell
          obj = [obj getNext];
        }
      }

    // Around we go to try the next shell outwards.
    shell++;
  }
  return self;
}

// Recursive traversal of binary space partitioning tree.
- getClosestIn: aTreeNode {
  double delX, delY;
  double x, y;
  double value, distSq;
  id obj;
  BOOL accept;

  // If this node is empty, just return.
  if (aTreeNode == nil)
    return self;

  // Do a simple bounding box cull: if the search starting point is outside
  //  the bounding box for this node, and it is far enough outside that the
  //  currently known minimum distance cannot be improved, return immediately.
  delX = 0.0;
  x = [aTreeNode getMinX];
  if (searchX < x) {
    // Clip to west side
    delX = searchX - x;
  }
  else {
    x = [aTreeNode getMaxX];
    if (searchX > x) {
      // Clip to east side
      delX = searchX - x;
    }
  }
  if (delX * delX > minDistSq)
    return self;

  delY = 0.0;
  y = [aTreeNode getMinY];
  if (searchY < y) {
    // Clip to north side
    delY = searchY - y;
  }
  else {
    y = [aTreeNode getMaxY];
    if (searchY > y) {
      // Clip to south side
      delY = searchY - y;
    }
  }
  if (delY * delY > minDistSq)
    return self;

  // If this is a leaf node, calculate distance and perhaps update the closest
  //  found so far. Screen objects by their getVal values if this has
  //  been requested.
  obj = [aTreeNode getObject];
  if (obj != nil) {
    // Loop through objects which may have the same location.
    while (obj != nil) {
      // Apply screening by the object's getVal value, if requested.
      if (valueSwitch == ALL)
        accept = TRUE;
      else {
        value = [obj getVal];
        switch (valueSwitch) {
          case EQ:
            accept = value == compareTo;
            break;
          case NEQ:
            accept = value != compareTo;
            break;
          case GT:
            accept = value > compareTo;
            break;
          case GE:
            accept = value >= compareTo;
            break;
          case LT:
            accept = value < compareTo;
            break;
          case LE:
            accept = value <= compareTo;
            break;
          default:
            accept = TRUE;
            break;
        }
      }
      if (accept) {
        delX = searchX - [aTreeNode getCentX];
        delY = searchY - [aTreeNode getCentY];
        distSq = delX * delX + delY * delY;
        if (distSq < minDistSq) {
          minDistSq = distSq;
          minDistObj = obj;
        }
      }

      // If duplicate positions are allowed, loop around for the next one.
      if (allowDups)
        obj = [obj getNext];
      else
        obj = nil;
    }
    return self;
  }

  // Otherwise, recurse on all existing subtrees.
  // We cleverly pick the order of traversal of the four subtrees. The
  // subtree which is closest to the search point is examined first. This
  // hopefully results in minDistSq (the minimum distance found so far,
  // squared) becoming small early in the search. This will allow the culling
  // code above to operate more efficiently.
  x = [aTreeNode getCentX];
  y = [aTreeNode getCentY];
  if (searchX < x) {
    if (searchY < y) {
      // Search point is NW of center of tree node's bounding box.
      [self getClosestIn: [aTreeNode getChildAt: NW]];
      if (x - searchX > y - searchY) {
        [self getClosestIn: [aTreeNode getChildAt: SW]];
        [self getClosestIn: [aTreeNode getChildAt: NE]];
      }
      else {
        [self getClosestIn: [aTreeNode getChildAt: NE]];
        [self getClosestIn: [aTreeNode getChildAt: SW]];
      }
      [self getClosestIn: [aTreeNode getChildAt: SE]];
    }
    else /* if (searchX < x && searchY >= y) */ {
      // Search point is SW of center of tree node's bounding box.
      [self getClosestIn: [aTreeNode getChildAt: SW]];
      if (x - searchX > searchY - y) {
        [self getClosestIn: [aTreeNode getChildAt: NW]];
        [self getClosestIn: [aTreeNode getChildAt: SE]];
      }
      else {
        [self getClosestIn: [aTreeNode getChildAt: SE]];
        [self getClosestIn: [aTreeNode getChildAt: NW]];
      }
      [self getClosestIn: [aTreeNode getChildAt: NE]];
    }
  }
  else /* if (searchX >= x */ {
    if (searchY < y) {
      // Search point is NE of center of tree node's bounding box.
      [self getClosestIn: [aTreeNode getChildAt: NE]];
      if (searchX - x > y - searchY) {
        [self getClosestIn: [aTreeNode getChildAt: SE]];
        [self getClosestIn: [aTreeNode getChildAt: NW]];
      }
      else {
        [self getClosestIn: [aTreeNode getChildAt: NW]];
        [self getClosestIn: [aTreeNode getChildAt: SE]];
      }
      [self getClosestIn: [aTreeNode getChildAt: SW]];
    }
    else /* if (searchX >= x && searchY >= y) */ {
      // Search point is SE of center of tree node's bounding box.
      [self getClosestIn: [aTreeNode getChildAt: SE]];
      if (searchX - x > searchY - y) {
        [self getClosestIn: [aTreeNode getChildAt: NE]];
        [self getClosestIn: [aTreeNode getChildAt: SW]];
      }
      else {
        [self getClosestIn: [aTreeNode getChildAt: SW]];
        [self getClosestIn: [aTreeNode getChildAt: NE]];
      }
      [self getClosestIn: [aTreeNode getChildAt: NW]];
    }
  }
  return self;
}

// This routine adds a list of objects to the tree or gridSpace.
- putObjects: (id <List>) aList {
  if (!useTree) {
    [gridSpace putObjects: aList];
  }
  else {
    // Add list of objects to a BSP tree which may or may not already have
    //  members.
    //  Internal node are regular binary divisions.
    //  Leaf nodes are actual objects
    id object;
    id indx;

    // If necessary, create a leaf node for the first object in the tree.
    indx = [aList begin: [self getZone]];
    if (treeBase == nil) {
      object = [indx next];
      if (object != nil)
        treeBase = [self createLeafNodeFor: object atMinX: minX  maxX: maxX
                                                     minY: minY  maxY: maxY];
    }

    // Then add each of the remaining objects to the tree.
    while ([indx getLoc] != End) {
      object = [indx next];
      [self addObject: object toTree: treeBase];
    }

    [indx drop];
  }
  return self;
}

- clear {
  if (useTree) {
    [self freeTreeNode: treeBase];
    treeBase = nil;
  }
  else {
    [gridSpace clear];
  }
  return self;
}

- clearAndPutObjects: (id <List>) aList {
  [self clear];
  [self putObjects: aList];
  return self;
}

- clearAndFree {
  [self clear];
  if (useTree) {
    [self dropTreeNode: freeList];
    freeList = nil;
  }
  return self;
}

- clearFreeAndPutObjects: (id <List>) aList {
  [self clearAndFree];
  [self putObjects: aList];
  return self;
}

// Recursively add an object to a binary space partitioning tree node.
- addObject: anObject toTree: aTreeNode {
  double x, y, nx, ny, cntX, cntY, mnX, mxX, mnY, mxY;
  id otherObj, subTree;

  if (intCoords) {
    x = [anObject getIntX];
    y = [anObject getIntY];
  }
  else {
    x = [anObject getDoubleX];
    y = [anObject getDoubleY];
  }
  mnX = [aTreeNode getMinX];
  mxX = [aTreeNode getMaxX];
  mnY = [aTreeNode getMinY];
  mxY = [aTreeNode getMaxY];

  otherObj = [aTreeNode getObject];
  if (otherObj == nil) {
    // The tree node is an interior node. Which subtree should the new
    //  object go into? If that subtree is empty, create a leaf node
    //  for it. Otherwise, recursively add the object to that subtree.
    cntX = [aTreeNode getCentX];
    cntY = [aTreeNode getCentY];

    if (x < cntX && y < cntY) {
      // new object goes to NW quadrant.
      subTree = [aTreeNode getChildAt: NW];
      if (subTree != nil)
        [self addObject: anObject toTree: subTree];
      else {
        subTree = [self createLeafNodeFor: anObject 
                        atMinX: mnX maxX: cntX 
                        minY: mnY maxY: cntY];
        [aTreeNode setChildAt: NW Id: subTree];
      }
    }
    else if (x >= cntX && y < cntY) {
      // new object goes to NE quadrant.
      subTree = [aTreeNode getChildAt: NE];
      if (subTree != nil)
        [self addObject: anObject toTree: subTree];
      else {
        subTree = [self createLeafNodeFor: anObject 
                        atMinX: cntX maxX: mxX 
                        minY: mnY maxY: cntY];
        [aTreeNode setChildAt: NE Id: subTree];
      }
    }
    else if (x < cntX && y >= cntY) {
      // new object goes to SW quadrant.
      subTree = [aTreeNode getChildAt: SW];
      if (subTree != nil)
        [self addObject: anObject toTree: subTree];
      else {
        subTree = [self createLeafNodeFor: anObject 
                        atMinX: mnX maxX: cntX 
                        minY: cntY maxY: mxY];
        [aTreeNode setChildAt: SW Id: subTree];
      }
    }
    else /* if (x >= cntX && y >= cntY) */ {
      // new object goes to SE quadrant.
      subTree = [aTreeNode getChildAt: SE];
      if (subTree != nil)
        [self addObject: anObject toTree: subTree];
      else {
        subTree = [self createLeafNodeFor: anObject 
                        atMinX: cntX maxX: mxX 
                        minY: cntY maxY: mxY];
        [aTreeNode setChildAt: SE Id: subTree];
      }
    }
  }
  else {
    // The tree node is a leaf node. If the two objects have the same location,
    //  either discard the new object or add it to the list of objects at this
    //  leaf. Otherwise, transform the tree node into an
    //  interior node with one subtree (which is a leaf node). Then
    //  recursively add the object to the same (transformed) tree node.
    nx = [aTreeNode getCentX];
    ny = [aTreeNode getCentY];
    if (x == nx && y == ny) {
      if (allowDups) {
        [anObject setNext: otherObj];
        [aTreeNode setObject: anObject];
      }
      else
        printf( "Error, two nodes at same location: %f %f. Discarding\n", 
                   x, y );
      return self;
    }
    cntX = (mnX + mxX) / 2;
    cntY = (mnY + mxY) / 2;
    [aTreeNode setCenterX: cntX Y: cntY Obj: nil];  // transform.
    if (nx < cntX && ny < cntY) {
      // new leaf node goes into NW quadrant.
      subTree = [self createLeafNodeFor: otherObj
                      atMinX: mnX maxX: cntX
                      minY: mnY maxY: cntY];
      [aTreeNode setChildAt: NW Id: subTree];
    }
    else if (nx >= cntX && ny < cntY) {
      // new leaf node goes into NE quadrant.
      subTree = [self createLeafNodeFor: otherObj
                      atMinX: cntX maxX: mxX
                      minY: mnY maxY: cntY];
      [aTreeNode setChildAt: NE Id: subTree];
    }
    else if (nx < cntX && ny >= cntY) {
      // new leaf node goes into SW quadrant.
      subTree = [self createLeafNodeFor: otherObj
                      atMinX: mnX maxX: cntX
                      minY: cntY maxY: mxY];
      [aTreeNode setChildAt: SW Id: subTree];
    }
    else /* if (nx >= cntX && ny >= cntY) */ {
      // new leaf node goes into SE quadrant.
      subTree = [self createLeafNodeFor: otherObj
                      atMinX: cntX maxX: mxX
                      minY: cntY maxY: mxY];
      [aTreeNode setChildAt: SE Id: subTree];
    }
    [self addObject: anObject toTree: aTreeNode];
  }
  return self;
}

// This subroutine is used to create a leaf node
//  to hold a single object. The new tree node object is returned.
- createLeafNodeFor: object atMinX: (double) mnX  maxX: (double) mxX
                              minY: (double) mnY  maxY: (double) mxY {
  id newNode;
  int i;
  double x, y;

  // Create node and set bounding box.
  if (freeList != nil) {
    newNode = freeList;
    freeList = [newNode getChildAt: FIRSTCHILD];
  }
  else
    newNode =  [SearchableTreeNode create: [self getZone]];    
  if (newNode == nil)
    printf( "Tree node create failed in createLeafNodeFor:\n" );
  [newNode setBoxMinX: mnX maxX: mxX minY: mnY maxY: mxY];

  // zero out pointers to subtrees and parent.
  for (i = FIRSTCHILD; i <= LASTCHILD; i++)
    [newNode setChildAt: i Id: nil];
  [newNode setParent: nil];

  // Set location and pointer to single object contained.
  if (intCoords) {
    x = [object getIntX];
    y = [object getIntY];
  }
  else {
    x = [object getDoubleX];
    y = [object getDoubleY];
  }
  [newNode setCenterX: x Y: y Obj: object];

  if (allowDups)
    [object setNext: nil];

  // Return pointer to leaf node just created.
  return newNode;
}

// Deallocate tree nodes.
- dropTreeNode: aTreeNode {
  int i;
  if (aTreeNode != nil) {
    for (i = FIRSTCHILD; i <= LASTCHILD; i++)
      [self dropTreeNode: [aTreeNode getChildAt: i]];
    [aTreeNode drop];
  }
  return self;
}

// Put tree nodes on free list.
- freeTreeNode: aTreeNode {
  int i;
  if (aTreeNode != nil) {
    for (i = FIRSTCHILD; i <= LASTCHILD; i++)
      [self freeTreeNode: [aTreeNode getChildAt: i]];
    for (i = FIRSTCHILD + 1; i <= LASTCHILD; i++)
      [aTreeNode setChildAt: i Id: nil];
    [aTreeNode setChildAt: FIRSTCHILD Id: freeList];
    freeList = aTreeNode;
  }
  return self;
}

// Messages to iterate the contents of a bounding box.
- iterateMinX: (double) mnX maxX: (double) mxX
         minY: (double) mnY maxY: (double) mxY {
  bbMinX = mnX;
  bbMaxX = mxX;
  bbMinY = mnY;
  bbMaxY = mxY;
  if (useTree) {
    currTreeNode = treeBase;
    if (currTreeNode != nil)
      [currTreeNode setSearchState: 0];
  }
  else {
    cellMinX = (int)((bbMinX - minX) / gridSpacing);
    cellMaxX = (int)((bbMaxX - minX) / gridSpacing);
    cellMinY = (int)((bbMinY - minY) / gridSpacing);
    cellMaxY = (int)((bbMaxY - minY) / gridSpacing);
    iterX = cellMinX - 1;
    iterY = cellMinY;
    currObject = nil;
  }
  return self;
}

// Iteration within a bounding box. Note that state of iteration is kept
//  between calls. In particular, note that recursive tree iteration is
//  simulated by storing state in the tree nodes themselves.
- next {
  id obj, retObj, node;
  double x, y;
  int state;

  if (useTree) {
    // Run through this loop visiting tree nodes. Do bounding-box culls
    //  to speed up the process. Each tree node maintains a notation of
    //  the current state of the search at that node. When the current
    //  node being searched becomes nil (the parent of the tree base),
    //  we are done.
    while (1) {
      if (currTreeNode == nil)
        return nil;
      state = [currTreeNode getSearchState];
      switch (state) {
        case 0:
          // Just starting to search this node; know nothing about it.
          obj = [currTreeNode getObject];
          if (obj == nil) {
            // interior node. First search the NW subtree, if necessary.
            [currTreeNode setSearchState: 2];
            node = [currTreeNode getChildAt: NW];
            if (node != nil) {
              if (bbMinX < [currTreeNode getCentX]
                            && bbMinY < [currTreeNode getCentY]) {
                currTreeNode = node;
                [currTreeNode setSearchState: 0];
              }
            }
          }
          else {
            // leaf node. Set up to iterate further.
            [currTreeNode setSearchState: 1];
            currObject = obj;
          }
          break;
        case 1:
          // Return current object of leaf node and decide what to do next.
          obj = currObject;
          x = [currTreeNode getCentX];
          y = [currTreeNode getCentY];
          if (allowDups)
            currObject = [currObject getNext];
          else
            currObject = nil;
          if (currObject == nil)
            currTreeNode = [currTreeNode getParent];
          if (bbMinX <= x && x <= bbMaxX && bbMinY <= y && y <= bbMaxY)
             return obj;
          break;
        case 2:
          // currTreeNode is interior, we are done with the NW subtree.
          //  Search the NE subtree, if necessary.
          [currTreeNode setSearchState: 3];
          node = [currTreeNode getChildAt: NE];
          if (node != nil) {
            if (bbMaxX >= [currTreeNode getCentX]
                          && bbMinY < [currTreeNode getCentY]) {
              currTreeNode = node;
              [currTreeNode setSearchState: 0];
            }
          }
          break;
        case 3:
          // currTreeNode is interior, we are done with the NW & NE subtrees.
          //  Search the SW subtree, if necessary.
          [currTreeNode setSearchState: 4];
          node = [currTreeNode getChildAt: SW];
          if (node != nil) {
            if (bbMinX < [currTreeNode getCentX]
                          && bbMaxY >= [currTreeNode getCentY]) {
              currTreeNode = node;
              [currTreeNode setSearchState: 0];
            }
          }
          break;
        case 4:
          // currTreeNode is interior, we are done with all but the SE subtree.
          //  Search the SE subtree, if necessary.
          [currTreeNode setSearchState: 5];
          node = [currTreeNode getChildAt: SE];
          if (node != nil) {
            if (bbMaxX >= [currTreeNode getCentX]
                          && bbMaxY >= [currTreeNode getCentY]) {
              currTreeNode = node;
              [currTreeNode setSearchState: 0];
            }
          }
          break;
        case 5:
          // currTreeNode is interior, and we have searched all subtrees.
          //  Climb up the tree (recursive "return").
          currTreeNode = [currTreeNode getParent];
          break;
        default:
          // Should never get here.
          printf( "SearchableSpace::next, internal error\n" );
          return nil;
      }
    }
  }
  else {
    // use grid. Run through this loop until a suitable object is found.
    while (1) {
      if (currObject == nil) {
        // No more objects to iterate in current cell: move on to next cell
        if (iterX == cellMaxX && iterY == cellMaxY)
          return nil;
        iterX++;
        if (iterX > cellMaxX) {
          iterX = cellMinX;
          iterY++;
        }
        currObject = [gridSpace getObjectAtX: (unsigned) iterX 
                                           Y: (unsigned) iterY];
      }
      else {
        // Check next object in current cell to see if it is inside box.
        //  Also, move current object pointer to next object in current
        //  cell (if any).
        retObj = currObject;
        currObject = [currObject getNext];
        if (intCoords) {
          x = [retObj getIntX];
          y = [retObj getIntY];
        }
        else {
          x = [retObj getDoubleX];
          y = [retObj getDoubleY];
        }
        if (bbMinX <= x && x <= bbMaxX && bbMinY <= y && y <= bbMaxY)
          return retObj;
      }
    }
  }
  return self;  // should never get here; keeps compiler happy.
}

@end

