//Copyright Simon Tokumine 2003. Freely distributable under the GNU General Public Licence

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.image.MemoryImageSource;
import java.util.ArrayList;

import javax.swing.BorderFactory;
import javax.swing.JPanel;

/**
 * @author 	S. Tokumine
 * @date	12-Aug-2003
 * @time	15:42:06
 * @file	GradientDisp.java
 * 
 * @desc GRADIENT DISPLAY CLASS creates and displays colour gradients
 *
 */
public class GradientDisp extends JPanel {

	Image image; //gradient image holder
	int gradH = 100; //height of gradient image
	int gradW = 512; //width of gradient image
	int colorBand = 256; //number of colours in the pallete
	int tempImage[][] = new int[gradH][gradW];
	//holder for the integer colours befor converted to gradient image
	int array1D[] = new int[gradH * gradW];
	//1D array which holds integers before image conversion
	int Alpha = 255; //reference value for Alpha
	int numberOfColors; //number of colours int he gradient
	ArrayList colorList; //shared list of colours
	ColorMapFactory colorMapFactory;
	//link to set the pallettes up when gradient is altered

	/**
	 * constructors sets up links to the pallete creators and creates border
	 */
	public GradientDisp(ColorMapFactory colorMapFactory_in) {
		super();
		colorMapFactory = colorMapFactory_in;
		setBorder(BorderFactory.createLineBorder(Color.GRAY, 1));
		// TODO Auto-generated constructor stub
	}

	/**
	 * accessor for the colorlist
	 * @return
	 */
	public ArrayList getColorList() {
		return colorList;
	}

	/**
	 * initial settings for the colorlist - sets it to be fire
	 * @param colorList_in
	 */
	public void setColorList(ArrayList colorList_in) {
		colorList = colorList_in;
		//		set colorList to be a firetype
		colorList.clear();
		colorList.add(Color.BLACK);
		colorList.add(Color.RED);
		colorList.add(Color.ORANGE);
		colorList.add(Color.YELLOW);
		colorList.add(Color.WHITE);
		colorSet();
		palleteSet();
	}

	/**
	 * creates gradient from the colorlist and draws it to screen
	 *
	 */
	public void colorSet() {
		drawGradient();
		repaint();
	}

	/**
	 * creates pallete from gradient in use and sets the colormapfactory to use it
	 *
	 */
	public void palleteSet() {
		colorMapFactory.setColorMap(createPallete());
	}

	/**
	 * creates a color gradient over a 1xlength array between two colors
	 */
	public int[] createGradient(Color color1, Color color2) {

		int[] tempGradient = new int[colorBand];
		boolean rNeg = false;
		boolean gNeg = false;
		boolean bNeg = false;

		int r1 = color1.getRed();
		int g1 = color1.getGreen();
		int b1 = color1.getBlue();

		int r2 = color2.getRed();
		int g2 = color2.getGreen();
		int b2 = color2.getBlue();

		//get the color steps per color. (on each step, decrease or increase)
		int[] rSteps = getColorArray(r1, r2);
		int[] gSteps = getColorArray(g1, g2);
		int[] bSteps = getColorArray(b1, b2);

		//check if they are going negative or positive
		if (r1 - r2 < 0) {
			rNeg = true;
		}
		if (b1 - b2 < 0) {
			bNeg = true;
		}
		if (g1 - g2 < 0) {
			gNeg = true;
		}

		//step through the colors

		int rI = 0; //index variables for step arrays
		int gI = 0;
		int bI = 0;

		for (int i = 0; i < colorBand; i++) {

			//create the graduated pallete
			tempGradient[i] = (Alpha << 24) | (r1 << 16) | (g1 << 8) | b1;

			//test and increment colors by 1 if the index is right.
			if (rSteps.length > 0) {

				if (rSteps[rI] == i) {
					if (rNeg == true) {
						r1++;
						rI++;
					} else {
						r1--;
						rI++;
					}
				}
			}
			if (gSteps.length > 0) {

				if (gSteps[gI] == i) {
					if (gNeg == true) {
						g1++;
						gI++;
					} else {
						g1--;
						gI++;
					}
				}
			}
			if (bSteps.length > 0) {
				if (bSteps[bI] == i) {
					if (bNeg == true) {
						b1++;
						bI++;
					} else {
						b1--;
						bI++;
					}
				}
			}
		}

		return tempGradient;
	}

	/**
	 * gets a list of index numbers at which the colour ought to be incremented if colorfrom is to change to colorTo
	 * over 256 steps smoothly
	 * @param colorFrom color changing from (one of the R. G. B.'s)
	 * @param colorTo colour changing to
	 * @return a list of index numbers along a 256 length array at whcih this colour ought to be incremented
	 * to result in a smooth gradient
	 */
	public int[] getColorArray(int colorFrom, int colorTo) {

		double inc = 0; //records the incrementing factor
		int[] steps; //records the increment numbers
		boolean sameColor = false;
		int difference;

		//calculate the difference between from and to (no sign)
		difference = (int) Math.sqrt(Math.pow((colorFrom - colorTo), 2));

		//check if they are the same number - if they are, make
		//and array with nothing in it and do nothing.
		if (colorFrom == colorTo) {
			sameColor = true;
		}

		//create new array at the right size for all the steps
		steps = new int[difference];

		//calculate incrementing factor, removing the sign
		if (sameColor == false) {
			inc = colorBand / (double) difference;
		}

		//create list based on difference
		if (sameColor == false) {
			int index = 0;
			for (double i = inc; i <= colorBand; i += inc) {
				steps[index] = (int) i;
				index++;
			}
		}

		return steps;
	}

	/**
	 * based on the number of colours in the colourarray, 
	 * drawgradient builds up a mastergradient from lots of smaller 2 colour interpolations
	 * then sets it as the main image.
	 *
	 */
	public void drawGradient() {

		int numberOfColors = colorList.size();
		int numberOfGradients = numberOfColors - 1;
		ArrayList gradientArray = new ArrayList();
		ArrayList smallGradientArray = new ArrayList();
		double stepper;
		int colorsPerGrad;

		//populate arraylist with numberofcolors-1 gradient arrays
		for (int i = 0; i < numberOfGradients; i++) {
			gradientArray.add(
				i,
				createGradient(
					(Color) colorList.get(i),
					(Color) colorList.get(i + 1)));
		}

		//calculate stepper amount using integer division (chop ends off)
		colorsPerGrad = gradW / numberOfGradients;

		stepper = (double) 255 / colorsPerGrad;

		//create array of index numbers to take colors from each gradient
		double[] indexPerGrad = new double[colorsPerGrad];

		int index = 0;
		for (double i = 0; i < 254.999; i += stepper) {
			indexPerGrad[index] = i;
			index++;
		}

		//create new arrays of smaller pallettes
		int[] temp;
		int[] smallTemp;

		for (int i = 0; i < numberOfGradients; i++) {

			temp = (int[]) gradientArray.get(i);
			smallTemp = new int[colorsPerGrad];

			for (int j = 0; j < colorsPerGrad; j++) {
				smallTemp[j] = temp[(int) indexPerGrad[j]];
			}

			//could be a problem with references here;
			smallGradientArray.add(smallTemp);

		}

		//get each pallet incrementally, and draw colors to the tempImage according to the 
		//steps dictated in the indexPerGrad array

		//rotate through 2d array layering the pallette and
		//increasing the color every index point per pallete		
		int gradIndex = 0;
		int x1 = 0;
		int[] loopArray;
		int cellCount = 0;

		for (int y = 0; y < gradH; y++) {
			for (int j = 0; j < numberOfGradients; j++) {
				loopArray = (int[]) smallGradientArray.get(j);

				for (int k = 0; k < loopArray.length; k++) {
					tempImage[y][x1] = loopArray[k];
					x1++;
					cellCount++;
				}
			}

			if (cellCount < 512) {
				for (int i = cellCount; i < 512; i++) {
					tempImage[y][i] = tempImage[y][cellCount - 1];
				}
			}

			cellCount = 0;
			x1 = 0;
		}

		//then convert the 2d array to 1d
		int j = 0;
		for (int y = 0; y < gradH; y++) {
			for (int x = 0; x < gradW; x++) {
				array1D[j++] = tempImage[y][x];
			}
		}

		//then create an image out of the 1d arry and set it.
		image =
			createImage(new MemoryImageSource(gradW, gradH, array1D, 0, gradW));
	}

	/**
	 * accessor to create basic black to white gradient key
	 * @return black and white gradient image in the key dimensions
	 */
	public Image createFKey() {
		return createKey(createGradient(Color.BLACK, Color.WHITE));
	}

	/**
	 * accessor to create colour key based on the current gradient/pallette
	 * @return colour gradient image in the key dimensions
	 */
	public Image createPKey() {
		return createKey(createPallete());
	}

	/**
	 * creates pallette of colour integers from the current gradient displayed
	 * @return integer colour pallete
	 */
	public int[] createPallete() {
		int[] pallete = new int[colorBand];

		pallete[0] = tempImage[1][0];

		int index = 1;
		for (int x = 1; x < gradW; x++) {
			if (x % 2 == 0) {
				pallete[index] = tempImage[1][x];
				index++;
			}
		}
		return pallete;
	}

	/**
	 * creates a key based on a pallette of 256 integer aRGB colours passed in 
	 * @param pallete (256 integer aRGB colours)
	 * @return image in key dimensions
	 */
	private Image createKey(int[] pallete) {
		int xSize = 84; //offset for imagepanel
		int ySize = 256;
		int[][] key = new int[ySize][xSize];
		int[] key1D = new int[ySize * xSize];

		//create 2d array representation of key image
		int colorAt = ySize - 1;
		for (int y = 0; y < ySize; y++) {
			for (int x = 0; x < xSize; x++) {
				key[y][x] = pallete[colorAt];
			}
			colorAt--;
		}

		//pipe it out to a 1d array
		int j = 0;
		for (int y = 0; y < ySize; y++) {
			for (int x = 0; x < xSize; x++) {
				key1D[j++] = key[y][x];
			}
		}
		
		//return key image
		return createImage(
			new MemoryImageSource(xSize, ySize, key1D, 0, xSize));
	}

	public void paintComponent(Graphics g) {
		//Make background white
		g.setColor(Color.LIGHT_GRAY);
		g.fillRect(0, 0, getSize().width - 1, getSize().height - 1);
		g.setColor(Color.black);
		if (image != null) {
			g.drawImage(image, 0, 0, this);
		}
	}

}
