// JSpatialEpidemic - A simulation framework to study the spatial aspect of an epidemic.
// JSpatialEpidemic implements a model from "Bailey N.T.J., The Mathematical Theory of 
//     Infectious Diseases and its application - II ed., Charles Griffin & Co. LTD, 1975,
//     pp. 182-188"
// Copyright  2003 Pasquale Cariello, Raul Bagni & Roberto Berchi. 
// See Readme for further details.


import swarm.Globals;
import swarm.Selector;
import swarm.defobj.Zone;

import swarm.activity.Activity;
import swarm.activity.ActionGroup;
import swarm.activity.ActionGroupImpl;
import swarm.activity.Schedule;
import swarm.activity.ScheduleImpl;

import swarm.objectbase.Swarm;
import swarm.objectbase.VarProbe;
import swarm.objectbase.MessageProbe;
import swarm.objectbase.EmptyProbeMapImpl;

import swarm.gui.Colormap;
import swarm.gui.ColormapImpl;
import swarm.gui.ZoomRaster;
import swarm.gui.ZoomRasterImpl;

import swarm.analysis.EZGraph;
import swarm.analysis.EZGraphImpl;
import swarm.analysis.EZBinC;
import swarm.analysis.EZBin;
import swarm.analysis.EZBinCImpl;
import swarm.analysis.EZBinImpl;

import swarm.simtoolsgui.GUISwarm;
import swarm.simtoolsgui.GUISwarmImpl;

import swarm.space.Value2dDisplay;
import swarm.space.Value2dDisplayImpl;
import swarm.space.Object2dDisplay;
import swarm.space.Object2dDisplayImpl;


/**
 The ObserverSwarm is a swarm of objects set up to observe a
 model when the graphical interface is running. The most
 important object is the ModelSwarm, but we also have graphical
 windows and data analysis */
 
public class ObserverSwarm extends GUISwarmImpl {
  /** parameters: update frequency and stop tick */
  public int displayFrequency;	
  public int checkPoint;
  
  /** ActionGroup for sequence of GUI events */
  public ActionGroup displayActions;				
  /** the single Schedule instance */
  public Schedule displaySchedule;

  /** the Swarm we're observing */
  public ModelSwarm modelSwarm;	  	

  /* Lots of display objects. First, widgets */

  /**  allocate colours */
  public Colormap colormap;		
  /**  2d display widget */
  public ZoomRaster worldRaster; 
  /** graphing widget */
  public EZGraph agentGraph, infectedGraph;			
  
  
  /* Now, higher order display and data objects */

  /** Draw the array on worldraster. 
      Note that you still have to tell the widget to draw itself afterwards. */
  public Value2dDisplay colorDisplay;
  /** Draw all objects in the collection */
  public Object2dDisplay spatialDisplay;	
  
  /** Constructor for class */
  public ObserverSwarm (Zone aZone) {
    super(aZone);

    // Fill in the relevant parameters (only one, in this case).
    displayFrequency = 1;
    checkPoint = 10;

    // Now, build a customized probe map using a `local' subclass
    // (a special kind of Java `inner class') of the
    // EmptyProbeMapImpl class.  Without a probe map, the default
    // is to show all variables and messages. Here we choose to
    // customize the appearance of the probe, give a nicer
    // interface.

    class SpatialObserverProbeMap extends EmptyProbeMapImpl {
      private VarProbe probeVariable (String name) {
        return
          Globals.env.probeLibrary.getProbeForVariable$inClass
          (name, ObserverSwarm.this.getClass ());
      }
      private MessageProbe probeMessage (String name) {
        return
          Globals.env.probeLibrary.getProbeForMessage$inClass
          (name, ObserverSwarm.this.getClass ());
      }
      private void addVar (String name) {
        addProbe (probeVariable (name));
      }
      private void addMessage (String name) {
        addProbe (probeMessage (name));
      }
      public SpatialObserverProbeMap (Zone _aZone, Class aClass) {
        super (_aZone, aClass);
        addVar ("displayFrequency");
        addMessage ("checkPoint");
      }
    } 
        
    // Install our custom probeMap class directly into the
    // probeLibrary
    Globals.env.probeLibrary.setProbeMap$For
      (new SpatialObserverProbeMap (aZone, getClass ()), getClass ());
  }
    
  public Object _worldRasterDeath_ (Object caller) {
    worldRaster.drop ();
    worldRaster = null;
    return this;
  }
    
  public Object _agentGraphDeath_ (Object caller) {
    agentGraph.drop ();
    agentGraph = null;
    return this;
  } 
  
  public Object _infectedGraphDeath_ (Object caller) {
    infectedGraph.drop ();
    infectedGraph = null;
    return this;
  } 
  
  
  
  /**
     Create the objects used in the display of the model. This code
     is fairly complicated because we build a fair number of
     widgets. It's also a good example of how to use the display
     code. */
  public Object buildObjects () {
    int i,j;
    int k; 
      
    super.buildObjects ();
        
    // First, we create the model that we're actually observing. The
    // model is a subswarm of the observer. 
        
    modelSwarm = new ModelSwarm (getZone ());
        
    // Now create probe objects on the model and ourselves. This gives a
    // simple user interface to let the user change parameters.
        
    Globals.env.createArchivedProbeDisplay (modelSwarm,
                                            "modelSwarm");
    Globals.env.createArchivedProbeDisplay (this, "observerSwarm");
    
    // Instruct the control panel to wait for a button event: we
    // halt here until someone hits a control panel button so the
    // user can get a chance to fill in parameters before the
    // simulation runs
    getControlPanel ().setStateStopped ();
        
    // OK - the user has specified all the parameters for the
    // simulation.  Now we're ready to start.
        
    // First, let the model swarm build its objects.
    modelSwarm.buildObjects ();
        
    // Now get down to building our own display objects.
        
    
    // First, create a colormap: this is a global resource, the information
    // here is used by lots of different objects.
    colormap = new ColormapImpl (getZone ());
    
    // Colours [0,k) are assigned to the range Blue [0, 1), for
    // heat display.
    k=modelSwarm.getK(); //half the side of the square
    
    /* colors for the k squares */
    for (int c = 0; c < k; c++)
      colormap.setColor$ToRed$Green$Blue 
        ((byte) c, (double)c/k, (double)c/k, (double)c/k);
    
    /* special colors */    
    colormap.setColor$ToName ((byte)  k   , "white");  //origin
    colormap.setColor$ToName ((byte) (k+1), "orange"); //infected
    colormap.setColor$ToName ((byte) (k+2), "red");    //infectious
    colormap.setColor$ToName ((byte) (k+3), "blue");   //immune
    
    
        
    // Next, create a 2d window for display, set its size, zoom
    // factor, title.
    worldRaster = new ZoomRasterImpl (getZone (), "worldRaster");
    try {
      worldRaster.enableDestroyNotification$notificationMethod 
        (this,
         new Selector (getClass (), "_worldRasterDeath_", false));
    } catch (Exception e) {
      System.err.println ("Exception _worldRasterDeath_: " 
                          + e.getMessage ());
    }
        
    worldRaster.setColormap (colormap);
    worldRaster.setZoomFactor ((int)(100/k+2));
    worldRaster.setWidth$Height 
      ((modelSwarm.getWorld ()).getSizeX (),
       (modelSwarm.getWorld ()).getSizeY ());
    worldRaster.setWindowTitle ("JSpatialEpidemics");    
    worldRaster.pack();				  // draw the window.
        
    // Now create a Value2dDisplay: this is a special object that
    // will display arbitrary 2d value arrays on a given Raster
    // widget.
     
    colorDisplay = new Value2dDisplayImpl 
      (getZone (), worldRaster, colormap, modelSwarm.blackGrid);
    //colorDisplay.setDisplayMappingM$C (105, 0); // turn [0,105) -> [0,1)
    
    
    // And also create an Object2dDisplay: this object draws
    // demos on the worldRaster widget for us, and also
    // receives probes.
    try {
      spatialDisplay = new Object2dDisplayImpl
        (getZone (), worldRaster, modelSwarm.getWorld (),
         new Selector (Class.forName ("Agent"), "drawSelfOn", false));
    } catch (Exception e) {
      System.err.println ("Exception drawSelfOn: " + e.getMessage ());
    }
        
    spatialDisplay.setObjectCollection 
      (modelSwarm.getAgentList ()); 
        
    // Also, tell the world raster to send mouse clicks to the
    // demoDisplay this allows the user to right-click on the
    // display to probe the agents.
    try {
      worldRaster.setButton$Client$Message 
        (3, spatialDisplay, new Selector (spatialDisplay.getClass (), 
                                          "makeProbeAtX$Y", true));
    } catch (Exception e) {
      System.err.println ("Exception makeProbeAtX$Y: " 
                          + e.getMessage ());
    }
        
    // Create the graph widget to display the number of infected agents.
    // Usage: EZGraphImpl(Zone aZone, String aTitle, String xl,
    //                    String yl, String windowGeometryRecordName)
      agentGraph = new EZGraphImpl
      (getZone (),
       "Population",
       "generation", "infected",
       "agentGraph"); 
       
      /* il nome del file su cui salvare la posizione del grafico */  
      Globals.env.setWindowGeometryRecordName (agentGraph, "agentGraph"); 
        
    // instruct this _agentGraphDeath_ method to be called when
    // the widget is destroyed
    try {
      agentGraph.enableDestroyNotification$notificationMethod 
        (this, new Selector (getClass (),
                             "_agentGraphDeath_",
                             false));
    } catch (Exception e) {
      System.err.println ("Exception _agentGraphDeath_: " 
                          + e.getMessage ());
    }
        
    // create the data for the the agents
    // Usage: EZAverageSequence createTotalSequence$withFeedFrom$andSelector
    //                            (String aName, Object aCollection, Selector aSel)
    
    try {
      agentGraph.createTotalSequence$withFeedFrom$andSelector 
        ("total cases", modelSwarm.getAgentList (),
         new Selector (Class.forName ("Agent"), "isInf", false));
    } catch (Exception e) {
      System.err.println ("Exception isInf: " 
                          + e.getMessage ());
    }
    
   try {
      agentGraph.createTotalSequence$withFeedFrom$andSelector 
        ("new cases", modelSwarm.getAgentList (),
         new Selector (Class.forName ("Agent"), "isNewInf", false));
    } catch (Exception e) {
      System.err.println ("Exception isInf: " 
                          + e.getMessage ());
    }
    
  
  
   if (k==5)  { //only for the example reported in the book
   
    // Create the graph widget to display the number of Infected agents by squares.
    // Usage: EZGraphImpl(Zone aZone, String aTitle, String xl,
    //                    String yl, String windowGeometryRecordName)
      infectedGraph = new EZGraphImpl
      (getZone (),
       "Infected agents by squares",
       "generation", "new cases per square",
       "infectedGraph"); 
       
      /* il nome del file su cui salvare la posizione del grafico */  
      Globals.env.setWindowGeometryRecordName (infectedGraph, "infectedGraph"); 
        
    // instruct this _infectedDeath_ method to be called when
    // the widget is destroyed
    try {
      infectedGraph.enableDestroyNotification$notificationMethod 
        (this, new Selector (getClass (),
                             "_infectedGraphDeath_",
                             false));
    } catch (Exception e) {
      System.err.println ("Exception _infectedGraphDeath_: " 
                          + e.getMessage ());
    }
        
    // create the data for the the total of agent
    // Usage: EZAverageSequence createTotalSequence$withFeedFrom$andSelector
    //                            (String aName, Object aCollection, Selector aSel)
    
    try {
      infectedGraph.createTotalSequence$withFeedFrom$andSelector 
        ("Square 1", modelSwarm.getAgentList (),
         new Selector (Class.forName ("Agent"), "newInf1", false));
    } catch (Exception e) {
      System.err.println ("Exception newInf1: " 
                          + e.getMessage ());
    }
    
    try {
      infectedGraph.createTotalSequence$withFeedFrom$andSelector 
        ("Square 2", modelSwarm.getAgentList (),
         new Selector (Class.forName ("Agent"), "newInf2", false));
    } catch (Exception e) {
      System.err.println ("Exception newInf2: " 
                          + e.getMessage ());
    }
    
    try {
      infectedGraph.createTotalSequence$withFeedFrom$andSelector 
        ("Square 3", modelSwarm.getAgentList (),
         new Selector (Class.forName ("Agent"), "newInf3", false));
    } catch (Exception e) {
      System.err.println ("Exception newInf3: " 
                          + e.getMessage ());
    }    
    
    try {
      infectedGraph.createTotalSequence$withFeedFrom$andSelector 
        ("Square 4", modelSwarm.getAgentList (),
         new Selector (Class.forName ("Agent"), "newInf4", false));
    } catch (Exception e) {
      System.err.println ("Exception newInf4: " 
                          + e.getMessage ());
    }    
    
    try {
      infectedGraph.createTotalSequence$withFeedFrom$andSelector 
        ("Square 5", modelSwarm.getAgentList (),
         new Selector (Class.forName ("Agent"), "newInf5", false));
    } catch (Exception e) {
      System.err.println ("Exception newInf5: " 
                          + e.getMessage ());
    }    
   } //end if (k==5)
    
    
 
    return this;
  }  

    
    
  public Object _update_ ()  {
    if (Globals.env.getCurrentTime() == checkPoint)  {
      getControlPanel().setStateStopped();
      //getControlPanel().setStateQuit();
    }
    if (worldRaster != null) {
      colorDisplay.display ();
      spatialDisplay.display ();
      worldRaster.drawSelf ();
      }
    if (agentGraph != null)
      agentGraph.step ();
    if (infectedGraph != null)
      infectedGraph.step ();  
    return this; 
  }  
  
  /**
     Create the actions necessary for the simulation. This is where
     the schedule is built (but not run!)  Here we create a display
     schedule - this is used to display the state of the world and
     check for user input. This schedule should be thought of as
     independent from the model - in particular, you will also want
     to run the model without any display.  */
  public Object buildActions () {
    super.buildActions();
        
    // First, let our model swarm build its own schedule.
    modelSwarm.buildActions();
  
    // Create an ActionGroup for display: a bunch of things that
    // occur in a specific order, but at one step of simulation
    // time. Some of these actions could be executed in parallel,
    // but we don't explicitly notate that here.
    displayActions = new ActionGroupImpl (getZone());

    // Add the methods to the ActionGroup to draw the display of
    // the world (compreso EZgraph)
    try {
      displayActions.createActionTo$message 
        (this, new Selector (getClass (), "_update_", false));
        
      // Schedule the update of the probe displays
      displayActions.createActionTo$message
        (Globals.env.probeDisplayManager, 
         new Selector (Globals.env.probeDisplayManager.getClass (),
                       "update", true));
            
      // Finally, schedule an update for the whole user
      // interface code.  This is crucial: without this, no
      // graphics update and the control panel will be
      // dead. It's best to put it at the end of the display
      // schedule
      displayActions.createActionTo$message
        (getActionCache (), new Selector 
          (getActionCache ().getClass (), "doTkEvents", true));
          
      //displayActions.createActionTo$message
        //(this, new Selector (getClass (), "move", false));    
          
    } catch (Exception e) {
      System.err.println ("Exception in setting up displayActions : " 
                          + e.getMessage ());
    }


    // And the display schedule. Note the repeat interval is set
    // from our own Swarm data structure. Display is frequently
    // the slowest part of a simulation, so redrawing less
    // frequently can be a help.
  
    // note frequency!
    displaySchedule = new ScheduleImpl (getZone (), displayFrequency);
        
    // insert ActionGroup instance on the repeating Schedule
    // instance
    displaySchedule.at$createAction (0, displayActions);

    return this;
  
  }  

  /**
     activateIn: - activate the schedules so they're ready to run.
     The swarmContext argument has to do with what we were activated
     *in*.  Typically the ObserverSwarm is the top-level Swarm, so
     it's activated in "null". But other Swarms and Schedules and
     such will be activated inside of us.  */
  public Activity activateIn (Swarm swarmContext) {
    // First, activate ourselves (just pass along the context).
    super.activateIn (swarmContext);

    // Activate the model swarm in ourselves. The model swarm is a
    // subswarm of the observer swarm.
    modelSwarm.activateIn (this);

    // Now activate our schedule in ourselves. This arranges for
    // the execution of the schedule we built.
    displaySchedule.activateIn (this);
  
    // Activate returns the swarm activity - the thing that's ready to run.
    return getActivity();
  }
    


  public void drop () {
    //if (agentGraph != null)
      //agentGraph.disableDestroyNotification (); 
    if (worldRaster != null)
      worldRaster.disableDestroyNotification ();
    super.drop ();
  }

}
