// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef __DB_H
#define __DB_H

//=== Database (abstract base class) =======================================
//
// This is the common interface of database classes used in pdr.
//
class Database {

	public:

	// a set of collection names
	typedef set<string> Collections;

	// a single data item
	typedef pair<ptime, any> CollectionItem;

	// data items of a single collection
	typedef map<ptime, any> CollectionItems;

	// data items of multiple collections
	typedef multimap<string, CollectionItem> CollectionsItems;

	// expressions
	typedef map<ptime, string> Expressions;

	virtual ~Database ();

	virtual void Connect () throw (Xception) =0;

	virtual	void ListCollections () throw (Xception) =0;
	virtual	void AddCollection (const string& name) throw (Xception) =0;
	virtual	void DeleteCollection (const string& name) throw (Xception) =0;
	virtual	void DeleteAllCollections () throw (Xception) =0;
	virtual	void ListRejections () throw (Xception) =0;
	virtual	void AddRejected (const ptime& timestamp, const string& expr) throw (Xception) =0;
	virtual	void DeleteAllRejections () throw (Xception) =0;

	virtual	char GetCollectionType (const string& name) const throw (Xception) =0;
	virtual void GetCollections (Collections& collections) const throw (Xception) =0;
	virtual void GetCollectionItems (const string& name, CollectionItems& items) const throw (Xception) =0;
	virtual	void AddCollectionsItems (const CollectionsItems& items) throw (Xception) =0;
	virtual void DeleteCollectionsItems (const ptime& timestamp, const Collections& collections) =0;

	virtual void GenerateExpressions (Expressions& expressions, const ptime& timestamp) const throw (Xception) =0;
};

//=== DBFactory ============================================================
//
// This factory creates one concrete database object according to the
// database specifications made in the configuration file. If Create()
// doesn't throw an exception the database object will immediately be
// connected and ready for use. The user should delete the object.
//
class DBFactory {

	public:

	static Database* Create (const Config& config) throw (Xception);
};

#endif
