// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "../libpdrx/encoding.h"
#include "db.h"
#include "in_impl.h"

#ifdef USE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#include <boost/tokenizer.hpp> 

//=== CSVFile ==============================================================
InputInteractive::InputInteractive (const string& option_key)
	: InputImpl(option_key)
{
}

#ifdef USE_READLINE
	static string s_expr;

	// completion means: we put the current expression onto the input
	// line, this is useful to correct longer misspelled expressions
	static char* command_generator (const char* text, int state)
	{
		string s(rl_line_buffer);
		trim(s);
		if (s.empty() && state == 0)
		{
			char* p = (char*)malloc(s_expr.length() + 1);
			strcpy(p, s_expr.c_str());
			return p;
		}
		else
			return NULL;
	}

	static char** completion_func (const char* text, int, int)
	{
		rl_attempted_completion_over = 1;
		return rl_completion_matches(text, command_generator);
	}
#endif

	static string fix_width (const string& s, int width)
	{
		// to format the output for fixed length we need to convert
		// it from UTF-8 into a character set with a fix character
		// size, otherwise we get wrong lengths for strings
		// containing UTF-8-multibyte-characters

		int spaces = width - (ConvertTo(s, GetEncoding("UTF-16")).length() / 2);
		return (spaces <= 0) ? s : s + string(spaces, ' ');
	}

void InputInteractive::Do (const Config& config, Database& database) const throw (Xception)
{
	int data_width = (int)config.GetDoubleOption(m_option_key + ".data_width");
	if (data_width == 0)
		data_width = 70;	// 10 characters for input on a line
					// with 80 characters

	cout << "pdr " << VERSION << ", interactive mode (press ? for help)" << endl;

#ifdef USE_READLINE
	// disable auto-complete
	rl_bind_key('\t', rl_abort);

	// allow the TAB key to be used for completion
	rl_attempted_completion_function = completion_func;
	rl_bind_key('\t', rl_complete);
#endif

	// do a database over all query
	Database::Expressions expressions;
	database.GenerateExpressions(expressions, not_a_date_time);

	Database::Expressions::iterator I = expressions.end();
	if (!expressions.empty())
		I--;

	// we set '-' as default command because the user starts at the end
	// of the data list
	char last_op = '-';

	// run the input loop
	string line;
	while (true)
	{
		string prompt;
		if (I != expressions.end())
			prompt = lexical_cast<string>((*I).first) + ' ' + fix_width((*I).second, data_width - 23);
		else
			prompt = fix_width("---database is empty---", data_width - 3);
		prompt += " > ";

#ifdef USE_READLINE
		if (I != expressions.end())
			s_expr = (*I).second;
		char* buf = readline(prompt.c_str());
		if (!buf)
		{
			cout << endl;
			break; // Ctrl-D
		}
		line = buf;
		free(buf);
#else
		cout << prompt;
		if (!getline(cin, line))
			break;
#endif
		trim(line);

		if (line == "q" || line == ".q" || line == "quit" || line == "exit" || line == "bye")
			break;

		if (line == "?" || line == "help")
		{
			cout	<< endl
				<< "type a command to let pdr do some actions, the following commands are available:" << endl
				<< endl
				<< "---line navigation---" << endl
				<< "-               make the previous line the current line (default)" << endl
				<< "+               make the next line the current line" << endl
				<< "^               make the first line the current line" << endl
				<< "$               make the last line the current line" << endl
				<< "RET             repeat the last line navigation command" << endl
				<< endl
				<< "---data manipulation---" << endl
				<< "D[collection+]  delete some specific or all data on the current line" << endl
				<< "expression      add or overwrite data on the current line" << endl
				<< endl
				<< "---other---" << endl
#ifdef USE_READLINE
				<< "TAB             use the current expression for input (you can modify it)" << endl
#endif
				<< "?               show this help screen" << endl
				<< "q               quit pdr" << endl
				<< endl
				;
			break;
		}

		// all other commands require a current data line, if the
		// database is empty we just do the loop but we don't allow
		// any further command execution
		if (I == expressions.end())
			break;

		try
		{
			char op = (line.empty()) ? last_op : line[0];
			switch (op)
			{
				// line navigation
				case '-':
				{
					if (I != expressions.begin())
						I--;
					last_op = op;
					break;
				}
				case '+':
				{
					if (++I == expressions.end())
						I--;
					last_op = op;
					break;
				}
				case '^':
				{
					I = expressions.begin();
					last_op = '+';
					break;
				}
				case '$':
				{
					I = expressions.end();
					if (I != expressions.begin())
						I--;
					last_op = '-';
					break;
				}

				// commands
				case 'D':
				{
					line.erase(0, 1);
					trim(line);
					try
					{
						const ptime& timestamp = (*I).first;
						bool delete_current_line = true;
						if (line.empty())
							database.DeleteCollectionsItems(timestamp, Database::Collections());
						else
						{
							Database::Collections collections;
							typedef tokenizer<boost::char_separator<char> > tokenizer;
							tokenizer tok(line);
							for (tokenizer::iterator J = tok.begin(); J != tok.end(); J++)
							{
								string collection(*J);
								if (collection == ";")
									collection = '#';
								collections.insert(collection);
							}
							database.DeleteCollectionsItems(timestamp, collections);
							Database::Expressions temp;
							database.GenerateExpressions(temp, timestamp);
							if (temp.size() == 1)
							{
								(*I).second = (*temp.begin()).second;
								delete_current_line = false;
							}
						}

						if (delete_current_line)
						{
							Database::Expressions::iterator J = I;
							ptime t(not_a_date_time);
							if (J != expressions.begin())
								t = (*--J).first;
							else
							{
								if (J != expressions.end())
									t = (*++J).first;
							}
							expressions.erase(I);
							I = (t == not_a_date_time) ? expressions.end() : expressions.find(t);
						}
					}
					catch (Xception& e)
					{
						cerr << e.Message() << endl;
					}
					break;
				}

				// input (overwrite a line)
				default:
				{
					trim(line);
					if (!line.empty())
					{
						try
						{
							const ptime& timestamp = (*I).first;
							Database::CollectionsItems items;
							Parse(line, timestamp, false, items);
							database.AddCollectionsItems(items);
							Database::Expressions temp;
							database.GenerateExpressions(temp, timestamp);
							if (temp.size() == 1)
								(*I).second = (*temp.begin()).second;
						}
						catch (Xception& e)
						{
							cerr << e.Message() << endl;
						}
					}
					break;
				}
			}
		}
		catch (Xception e)
		{
			cerr << e.Message() << endl;
		}
	}
}
