// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "common.h"

#include <Poco/Data/Common.h>
#include <Poco/Data/RecordSet.h>

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace Poco::Data;

#include "datatypes.h"
#include "config.h"
#include "poco_db.h"

//=== PocoDatabase (abstract base class) ===================================
PocoDatabase::PocoDatabase (const string& connect, bool verbose, const string& KEY, Connector* pConnector)
	: m_connect(connect)
	, m_verbose(verbose)
	, m_KEY(KEY)
	, m_pSession(NULL)
{
	SessionFactory::instance().add(m_KEY, pConnector);
}

PocoDatabase::~PocoDatabase ()
{
	if (m_verbose)
		cout << "disconnecting from database" << endl;

	if (m_pSession && m_pSession->isConnected())
		m_pSession->close();
	delete m_pSession;

	SessionFactory::instance().remove(m_KEY);
}

void PocoDatabase::Connect () throw (Xception)
{
	// open session
	try
	{
		if (m_verbose)
			cout << "connecting to database" << endl;

		m_pSession = new Session(m_KEY, m_connect);
	}
	catch (const DataException& x)
	{
		if (m_pSession)
		{
			delete m_pSession;
			m_pSession = NULL;
		}
		throw Xception(format("cannot open database: %s, %s") % m_connect % x.displayText());
	}

	// check schema
	try
	{
		if (m_verbose)
			cout << "checking schema" << endl;

		(*m_pSession) << "select * from TCollections;", now;
	}
	catch (...)
	{
		// no schema, react
		NoSchemaCallback();
	}

	// if the callback doesn't throw an exception we assume that we
	// have a schema now, so we go one here

	// get collection meta informations
	try
	{
		Statement select(*m_pSession);
		select << "select name,type,tbl_name from TCollections;";
		select.execute();

		RecordSet rs(select);
		if (rs.moveFirst())
		{
			do {
				CollectionMetaInfo cmi;
				cmi.m_collection = rs.value(0).convert<string>();
				cmi.m_type = rs.value(1).convert<string>()[0];
				cmi.m_tblname = rs.value(2).convert<string>();
				m_collectionMetaInfos.insert(cmi);
			} while (rs.moveNext());
		}
	}
	catch (...)
	{
		throw Xception("error getting schema meta data informations");
	}
}

const PocoDatabase::CollectionMetaInfo& PocoDatabase::GetCollectionMetaInfo (const string& name) const throw (Xception)
{
	CollectionMetaInfo cmi = {name, '\0', ""};
	CollectionMetaInfos::const_iterator I = m_collectionMetaInfos.find(cmi);
	if (I == m_collectionMetaInfos.end())
		throw Xception("unknown collection name");
	return *I;
}

char PocoDatabase::GetCollectionType (const string& name) const throw (Xception)
{
	try
	{
		return GetCollectionMetaInfo(name).m_type;
	}
	catch (...)
	{
		throw Xception("could not determine collection type");
	}
}

//=== DBTransactor =========================================================
DBTransactor::DBTransactor (Poco::Data::Session* pSession)
	: m_pSession(pSession)
{
	m_pSession->begin();
}

DBTransactor::~DBTransactor ()
{
	if (m_pSession)
		m_pSession->rollback();
}

void DBTransactor::Commit ()
{
	m_pSession->commit();
	m_pSession = NULL;
}
