/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: fix_missing_path_ref.c,v $
-- Revision 1.3  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.2  2006/11/11 12:29:19  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.1  2006/11/08 02:49:02  jsedwards
-- Program to fix missing reference to file_path object from file objects.
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "objectify_private.h"


bool reference_list_has_reference(ObjRef* ref_list_ref, ObjRef* ref)
{
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int i;

    ref_list_size = nwos_reference_list_size(ref_list_ref);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(ref_list_ref, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    for (i = 0; i < num_refs; i++)
    {
	if (is_same_object(&ref_list->references[i], ref)) break;
    }

    free(ref_list);
    ref_list = NULL;

    return i < num_refs;    /* return true if found it */
}

void dump_reference_list(ObjRef* ref_list_ref)
{
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int i;

    ref_list_size = nwos_reference_list_size(ref_list_ref);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(ref_list_ref, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    for (i = 0; i < num_refs; i++)
    {
	printf("   %08x\n", nwos_ref_to_word(&ref_list->references[i]));
    }
}


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}

void check_file_path(ObjRef* file_path_ref)
{
    int i;
    char buffer[64];
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* file_path_obj_ptr = (C_struct_File_Path*)kludge;
    C_struct_File file_obj;
#if 0
    size_t size;
    ObjRef ref;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
#endif

    printf("File Path: %08x ", nwos_ref_to_word(file_path_ref));
    fflush(stdout);

    nwos_read_variable_sized_object_from_disk(file_path_ref, kludge, &get_path_object_size);

    for (i = 0; i < file_path_obj_ptr->count; i++) buffer[i] = file_path_obj_ptr->storage[i];
    buffer[i] = '\0';

    printf(": %s\n", buffer);

    nwos_read_object_from_disk(&file_path_obj_ptr->file, &file_obj, sizeof(file_obj));

    if (!reference_list_has_reference(&file_obj.header.object.references, file_path_ref))
    {
	printf("ERROR: file path object not in file reference list: %08x - ",
	       nwos_ref_to_word(&file_obj.header.object.references));

	nwos_add_to_reference_list(file_path_ref, &file_obj.header.object.references);

	if (reference_list_has_reference(&file_obj.header.object.references, file_path_ref))
	{
	    printf("FIXED!");
	}
	printf("\n");
    }
}


void check_class(ObjRef* class_ref)
{
    int i;
#if 0
    size_t size;
    ObjRef ref;
#endif
    char buffer[64];
    C_struct_Class_Definition class_def_obj;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;

    printf("Reading class definition: %08x\n", nwos_ref_to_word(class_ref));

    nwos_read_object_from_disk(class_ref, &class_def_obj, sizeof(class_def_obj));

    nwos_name_to_string(&class_def_obj.name, buffer, sizeof(buffer));

    printf("Class name: %s\n", buffer);

    if (!is_same_object(&class_def_obj.header.common.class_definition, &nwos_class_definition_class_ref))
    {
	printf("Class is wrong: %08x  should be: %08x\n",
	       nwos_ref_to_word(&class_def_obj.header.common.class_definition),
	       nwos_ref_to_word(&nwos_class_definition_class_ref));
    }
    else if (strcasecmp(buffer, "REFERENCE LIST") != 0)    /* reference lists don't have reference lists */
    {
	printf("Reading reference list: %08x\n", nwos_ref_to_word(&class_def_obj.header.object.references));
	fflush(stdout);

	ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

	ref_list = malloc(ref_list_size);

	if (ref_list == NULL) 
	{
	    perror("reading reference list");
	    exit(1);
	}

	nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

	num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	printf("Number of refrences: %d\n", num_refs);

	for (i = 0; i < num_refs; i++)
	{
	    if (strcasecmp(buffer, "FILE PATH") == 0)
	    {
		check_file_path(&ref_list->references[i]);
	    }
	}

	free(ref_list);
    }
}


int main(int argc, char* argv[])
{
    int i;
#if 0
    size_t size;
    ObjRef ref;
#endif
    char buffer[64];
    C_struct_Class_Definition class_def_obj;
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, Drive_Or_Partition_RW, DEFAULT_FILE);

    printf("Setting root object: %08x\n", nwos_ref_to_word(&root_object_ref));

    nwos_set_root_object(&root_object_ref);

    printf("Reading class definition class: %08x\n", nwos_ref_to_word(&nwos_class_definition_class_ref));

    nwos_read_object_from_disk(&nwos_class_definition_class_ref, &class_def_obj, sizeof(class_def_obj));

    nwos_name_to_string(&class_def_obj.name, buffer, sizeof(buffer));

    printf("Class name: %s\n", buffer);

    printf("Reading class definition class reference list: %08x\n", nwos_ref_to_word(&class_def_obj.header.object.references));

    ref_list_size = nwos_reference_list_size(&class_def_obj.header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&class_def_obj.header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    printf("Number of refrences: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	check_class(&ref_list->references[i]);
    }

    free(ref_list);

    nwos_terminate_objectify();

    return 0;
}

