/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: fix_img_files.c,v $
-- Revision 1.2  2008/05/19 03:50:25  jsedwards
-- Fixed to open READ_WRITE instead of READ_ONLY.
--
-- Revision 1.1  2008/05/19 03:24:53  jsedwards
-- Initial version to clean up the mess left by the link_img_files program.
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../crc32.h"
#include "../objectify_private.h"   /* do private so that nwos_get_public_objects_path is defined */
#include "../time_stamp.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


void print_time(TimeStamp ts)
{
    if (nwos_time_stamp_is_zero(ts))
    {
	printf("  unknown");
    }
    else
    {
	printf("  %u-%02u-%02u %02u:%02u:%02u",
	       nwos_extract_year_from_time_stamp(ts),
	       nwos_extract_month_from_time_stamp(ts),
	       nwos_extract_day_of_month_from_time_stamp(ts),
	       nwos_extract_hour_from_time_stamp(ts),
	       nwos_extract_minute_from_time_stamp(ts),
	       nwos_extract_second_from_time_stamp(ts));
    }
}


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association bad_assoc_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    C_struct_File file_obj;
    ObjRef object_class;
    ObjRef file_class_ref;
    ObjRef assoc_class_ref;
    ReferenceList* ref_list;
    int num_refs;
    int num_files = 0;
    int i;
    int j;
    char name[256];
    ObjRef file_001_class_ref;
    ObjRef file_002_class_ref;
    ObjRef prev_class;
    ObjRef next_class;

    char* path = DEFAULT_FILE;

    if (argc != 1)
    {
	fprintf(stderr, "usage: %s\n", argv[0]);
	fprintf(stderr, "  This repairs the links to the association objects that were created by link_img_files.\n");
	exit(1);
    }

    //    nwos_log_arguments(argc, argv);
    nwos_log_arguments(0, NULL);

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, READ_WRITE, path);

    nwos_set_root_object(&root_object_ref);

    if (!nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref))
    {
	printf("No files are stored in the system!\n");
	exit(1);
    }

    assert(nwos_find_private_class_definition("FILE", &file_class_ref));

    nwos_read_object_headers_from_disk(&file_class_ref, &class_def_obj.header);

    assert(!is_void_reference(&class_def_obj.header.object.prev_version));

    /* This could be bad: newly created objects have old public file in previous version field */

    if (nwos_reference_type(&class_def_obj.header.object.prev_version) == Private_Reference)
    {
	copy_reference(&file_002_class_ref, &class_def_obj.header.object.prev_version);

	nwos_read_object_headers_from_disk(&file_002_class_ref, &class_def_obj.header);

	assert(is_same_object(&class_def_obj.header.object.next_version, &file_class_ref));
	assert(!is_void_reference(&class_def_obj.header.object.prev_version));

	if (nwos_reference_type(&class_def_obj.header.object.prev_version) == Private_Reference)
	{
	    copy_reference(&file_001_class_ref, &class_def_obj.header.object.prev_version);

	    nwos_read_object_headers_from_disk(&file_001_class_ref, &class_def_obj.header);

	    assert(is_same_object(&class_def_obj.header.object.next_version, &file_002_class_ref));
	    assert(is_void_reference(&class_def_obj.header.object.prev_version));
	}
    }

    printf("PATH AND FILE ASSOCIATION ref: %08x\n", nwos_ref_to_word(&assoc_class_ref));
    printf("FILE 001 ref: %08x\n", nwos_ref_to_word(&file_001_class_ref));
    printf("FILE 002 ref: %08x\n", nwos_ref_to_word(&file_002_class_ref));
    printf("FILE 003 ref: %08x\n", nwos_ref_to_word(&file_class_ref));

    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("num_refs: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj));

	    nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));

	    if (!is_void_reference(&file_obj.header.object.prev_version) | !is_void_reference(&file_obj.header.object.next_version))

	    {
		nwos_read_variable_sized_object_from_disk(&assoc_obj.path, kludge, sizeof(kludge), &get_path_object_size);

		/* remember ptr_path_obj points to the kludge buffer */

		for (j = 0; j < ptr_path_obj->count; j++) name[j] = ptr_path_obj->storage[j];

		name[j] = '\0';

		printf("%s", name);

		if (!is_void_reference(&file_obj.header.object.prev_version))
		{
		    nwos_get_object_class_without_update(&file_obj.header.object.prev_version, &prev_class);
		}
		else
		{
		    void_reference(&prev_class);
		}

		if (!is_void_reference(&file_obj.header.object.next_version))
		{
		    nwos_get_object_class_without_update(&file_obj.header.object.next_version, &next_class);
		}
		else
		{
		    void_reference(&next_class);
		}

		printf("  assoc ref: %08x\n", nwos_ref_to_word(&ref_list->references[i]));
		printf("   path ref: %08x\n", nwos_ref_to_word(&assoc_obj.path));
		printf("   file ref: %08x\n", nwos_ref_to_word(&assoc_obj.file));
		printf("  file prev: %08x   class: %08x\n", nwos_ref_to_word(&file_obj.header.object.prev_version),
		       nwos_ref_to_word(&prev_class));
		printf("  file next: %08x   class: %08x\n", nwos_ref_to_word(&file_obj.header.object.next_version),
		       nwos_ref_to_word(&next_class));

		if (is_same_object(&prev_class, &assoc_class_ref))
		{
		    nwos_read_object_from_disk(&file_obj.header.object.prev_version, &bad_assoc_obj, sizeof(bad_assoc_obj));
		    copy_reference(&file_obj.header.object.prev_version, &bad_assoc_obj.file);
		    nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);
		    nwos_overwrite_object_to_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));

		    nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));
		    nwos_get_object_class_without_update(&file_obj.header.object.prev_version, &prev_class);
		    printf("   fix prev: %08x   class: %08x\n", nwos_ref_to_word(&file_obj.header.object.prev_version),
			   nwos_ref_to_word(&prev_class));
		}

		if (is_same_object(&next_class, &assoc_class_ref))
		{
		    nwos_read_object_from_disk(&file_obj.header.object.next_version, &bad_assoc_obj, sizeof(bad_assoc_obj));
		    copy_reference(&file_obj.header.object.next_version, &bad_assoc_obj.file);
		    nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);
		    nwos_overwrite_object_to_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));

		    nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));
		    nwos_get_object_class_without_update(&file_obj.header.object.next_version, &next_class);
		    printf("   fix next: %08x   class: %08x\n", nwos_ref_to_word(&file_obj.header.object.next_version),
			   nwos_ref_to_word(&next_class));
		}

		printf("\n");

		num_files++;
	    }

	    //    memcpy(file_ref, &ptr_path_obj->file, sizeof(*file_ref));
	}
    }

    printf("Number of files: %d\n", num_files);

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    nwos_terminate_objectify();

    return 0;
}


