=head1 NAME

BerkeleyDB - Perl extension for Berkeley DB version 2 or 3

=head1 SYNOPSIS

  use BerkeleyDB;

  $env = new BerkeleyDB::Env [OPTIONS] ;

  $db  = tie %hash, 'BerkeleyDB::Hash', [OPTIONS] ;
  $db  = new BerkeleyDB::Hash [OPTIONS] ;

  $db  = tie %hash, 'BerkeleyDB::Btree', [OPTIONS] ;
  $db  = new BerkeleyDB::Btree [OPTIONS] ;

  $db  = tie %hash, 'BerkeleyDB::Recno', [OPTIONS] ;
  $db  = new BerkeleyDB::Recno [OPTIONS] ;

  $db  = tie %hash, 'BerkeleyDB::Queue', [OPTIONS] ;
  $db  = new BerkeleyDB::Queue [OPTIONS] ;

  $db  = new BerkeleyDB::Unknown [OPTIONS] ;

  $status = BerkeleyDB::db_remove [OPTIONS]

  $hash{$key} = $value ;
  $value = $hash{$key} ;
  each %hash ;
  keys %hash ;
  values %hash ;

  $status = $db->db_get()
  $status = $db->db_put() ;
  $status = $db->db_del() ;
  $status = $db->db_sync() ;
  $status = $db->db_close() ;
  $hash_ref = $db->db_stat() ;
  $status = $db->db_key_range();
  $type = $db->type() ;
  $status = $db->status() ;
  $boolean = $db->byteswapped() ;

  ($flag, $old_offset, $old_length) = $db->partial_set($offset, $length) ;
  ($flag, $old_offset, $old_length) = $db->partial_clear() ;

  $cursor = $db->db_cursor([$flags]) ;
  $newcursor = $cursor->c_dup([$flags]);
  $status = $cursor->c_get() ;
  $status = $cursor->c_put() ;
  $status = $cursor->c_del() ;
  $status = $cursor->c_count() ;
  $status = $cursor->status() ;
  $status = $cursor->c_close() ;

  $cursor = $db->db_join() ;
  $status = $cursor->c_get() ;
  $status = $cursor->c_close() ;

  $status = $env->txn_checkpoint()
  $hash_ref = $env->txn_stat()
  $status = $env->setmutexlocks()

  $txn = $env->txn_begin() ;
  $db->Txn($txn);
  $txn->Txn($db1, $db2,...);
  $status = $txn->txn_prepare()
  $status = $txn->txn_commit()
  $status = $txn->txn_abort()
  $status = $txn->txn_id()
  $status = $txn->txn_discard()

  $status = $env->set_lg_dir();
  $status = $env->set_lg_bsize();
  $status = $env->set_lg_max();

  $status = $env->set_data_dir() ;
  $status = $env->set_tmp_dir() ;

  $BerkeleyDB::Error
  $BerkeleyDB::db_version

  # DBM Filters
  $old_filter = $db->filter_store_key  ( sub { ... } ) ;
  $old_filter = $db->filter_store_value( sub { ... } ) ;
  $old_filter = $db->filter_fetch_key  ( sub { ... } ) ;
  $old_filter = $db->filter_fetch_value( sub { ... } ) ;

  # deprecated, but supported
  $txn_mgr = $env->TxnMgr();
  $status = $txn_mgr->txn_checkpoint()
  $hash_ref = $txn_mgr->txn_stat()
  $txn = $txn_mgr->txn_begin() ;

=head1 DESCRIPTION

B<NOTE: This document is still under construction. Expect it to be
incomplete in places.>

This Perl module provides an interface to most of the functionality
available in Berkeley DB versions 2 and 3. In general it is safe to assume
that the interface provided here to be identical to the Berkeley DB
interface. The main changes have been to make the Berkeley DB API work
in a Perl way. Note that if you are using Berkeley DB 2.x, the new
features available in Berkeley DB 3.x are not available via this module.

The reader is expected to be familiar with the Berkeley DB
documentation. Where the interface provided here is identical to the
Berkeley DB library and the... TODO

The B<db_appinit>, B<db_cursor>, B<db_open> and B<db_txn> man pages are
particularly relevant.

The interface to Berkeley DB is implemented with a number of Perl
classes.

=head1 ENV CLASS

The B<BerkeleyDB::Env> class provides an interface to the Berkeley DB
function B<db_appinit> in Berkeley DB 2.x or B<db_env_create> and
B<DBENV-E<gt>open> in Berkeley DB 3.x. Its purpose is to initialise a
number of sub-systems that can then be used in a consistent way in all
the databases you make use of the environment.

If you don't intend using transactions, locking or logging, then you
shouldn't need to make use of B<BerkeleyDB::Env>.

=head2 Synopsis

    $env = new BerkeleyDB::Env
             [ -Home         => $path, ]
             [ -Server       => $name, ]
             [ -CacheSize    => $number, ]
             [ -Config       => { name => value, name => value }, ]
             [ -ErrFile      => filename or filehandle, ]
             [ -ErrPrefix    => "string", ]
             [ -Flags        => number, ]
             [ -LockDetect   => number, ]
             [ -Verbose      => boolean, ]

=over 5

All the parameters to the BerkeleyDB::Env constructor are optional.

=item -Home

If present, this parameter should point to an existing directory. Any
files that I<aren't> specified with an absolute path in the sub-systems
that are initialised by the BerkeleyDB::Env class will be assumed to
live in the B<Home> directory.

For example, in the code fragment below the database "fred.db" will be
opened in the directory "/home/databases" because it was specified as a
relative path, but "joe.db" will be opened in "/other" because it was
part of an absolute path.

    $env = new BerkeleyDB::Env
             -Home         => "/home/databases"
    ...

    $db1 = new BerkeleyDB::Hash
	     -Filename = "fred.db",
	     -Env => $env
    ...

    $db2 = new BerkeleyDB::Hash
	     -Filename = "/other/joe.db",
	     -Env => $env
    ...

=item -Server

If present, this parameter should be the hostname of a server that is running
the Berkeley DB RPC server. All databases will be accessed via the RPC server.

=item -Cachesize

If present, this parameter sets the size of the environments shared memory
buffer pool.

=item -Config

This is a variation on the C<-Home> parameter, but it allows finer
control of where specific types of files will be stored.

The parameter expects a reference to a hash. Valid keys are:
B<DB_DATA_DIR>, B<DB_LOG_DIR> and B<DB_TMP_DIR>

The code below shows an example of how it can be used.

    $env = new BerkeleyDB::Env
             -Config => { DB_DATA_DIR => "/home/databases",
                          DB_LOG_DIR  => "/home/logs",
                          DB_TMP_DIR  => "/home/tmp"
                        }
    ...

=item -ErrFile

Expects either the name of a file or a reference to a filehandle. Any
errors generated internally by Berkeley DB will be logged to this file.

=item -ErrPrefix

Allows a prefix to be added to the error messages before they are sent
to B<-ErrFile>.

=item -Flags

The B<Flags> parameter specifies both which sub-systems to initialise,
as well as a number of environment-wide options.
See the Berkeley DB documentation for more details of these options.

Any of the following can be specified by OR'ing them:

B<DB_CREATE>

If any of the files specified do not already exist, create them.

B<DB_INIT_CDB>

Initialise the Concurrent Access Methods  

B<DB_INIT_LOCK>

Initialise the Locking sub-system.

B<DB_INIT_LOG>

Initialise the Logging sub-system.

B<DB_INIT_MPOOL>

Initialise the ...

B<DB_INIT_TXN>

Initialise the ...

B<DB_MPOOL_PRIVATE>

Initialise the ...

B<DB_INIT_MPOOL> is also specified.

Initialise the ...

B<DB_NOMMAP>

Initialise the ...

B<DB_RECOVER>



B<DB_RECOVER_FATAL>

B<DB_THREAD>

B<DB_TXN_NOSYNC>

B<DB_USE_ENVIRON>

B<DB_USE_ENVIRON_ROOT>

=item -LockDetect

Specifies what to do when a lock conflict occurs. The value should be one of

B<DB_LOCK_DEFAULT> 

B<DB_LOCK_OLDEST>

B<DB_LOCK_RANDOM>

B<DB_LOCK_YOUNGEST>

=item -Verbose

Add extra debugging information to the messages sent to B<-ErrFile>.

=back

=head2 Methods

The environment class has the following methods:

=over 5

=item $env->errPrefix("string") ;

This method is identical to the B<-ErrPrefix> flag. It allows the
error prefix string to be changed dynamically.

=item $txn = $env->TxnMgr()

Constructor for creating a B<TxnMgr> object.
See L<"TRANSACTIONS"> for more details of using transactions.

This method is deprecated. Access the transaction methods using the B<txn_>
methods below from the environment object directly.

=item $env->txn_begin()

TODO

=item $env->txn_stat()

TODO

=item $env->txn_checkpoint()

TODO

=item $env->status()

Returns the status of the last BerkeleyDB::Env method.

=item $env->setmutexlocks()

Only available in Berkeley Db 3.0 or greater. Calls 
B<db_env_set_mutexlocks> when used with Berkeley DB 3.1.x. When used with
Berkeley DB 3.0 or 3.2 and better it calls B<DBENV-E<gt>set_mutexlocks>.

=back

=head2 Examples

TODO.

=head1 THE DATABASE CLASSES

B<BerkeleyDB> supports the following database formats:

=over 5

=item B<BerkeleyDB::Hash>

This database type allows arbitrary key/value pairs to be stored in data
files. This is equivalent to the functionality provided by other
hashing packages like DBM, NDBM, ODBM, GDBM, and SDBM. Remember though,
the files created using B<BerkeleyDB::Hash> are not compatible with any
of the other packages mentioned.

A default hashing algorithm, which will be adequate for most applications,
is built into BerkeleyDB. If you do need to use your own hashing algorithm
it is possible to write your own in Perl and have B<BerkeleyDB> use
it instead.

=item B<BerkeleyDB::Btree>

The Btree format allows arbitrary key/value pairs to be stored in a
B+tree.

As with the B<BerkeleyDB::Hash> format, it is possible to provide a
user defined Perl routine to perform the comparison of keys. By default,
though, the keys are stored in lexical order.

=item B<BerkeleyDB::Recno>

TODO.


=item B<BerkeleyDB::Queue>

TODO.

=item B<BerkeleyDB::Unknown>

This isn't a database format at all. It is used when you want to open an
existing Berkeley DB database without having to know what type is it. 

=back


Each of the database formats described above is accessed via a
corresponding B<BerkeleyDB> class. These will be described in turn in
the next sections.

=head1 BerkeleyDB::Hash

Equivalent to calling B<db_open> with type B<DB_HASH> in Berkeley DB 2.x and
calling B<db_create> followed by B<DB-E<gt>open> with type B<DB_HASH> in
Berkeley DB 3.x. 

Two forms of constructor are supported:

    $db = new BerkeleyDB::Hash
                [ -Filename      => "filename", ]
                [ -Subname       => "sub-database name", ]
                [ -Flags         => flags,]
                [ -Property      => flags,]
                [ -Mode          => number,]
                [ -Cachesize     => number,]
                [ -Lorder        => number,]
                [ -Pagesize      => number,]
                [ -Env           => $env,]
                [ -Txn           => $txn,]
                # BerkeleyDB::Hash specific
                [ -Ffactor       => number,]
                [ -Nelem         => number,]
                [ -Hash          => code reference,]
                [ -DupCompare    => code reference,]

and this

    [$db =] tie %hash, 'BerkeleyDB::Hash', 
                [ -Filename      => "filename", ]
                [ -Subname       => "sub-database name", ]
                [ -Flags         => flags,]
                [ -Property      => flags,]
                [ -Mode          => number,]
                [ -Cachesize     => number,]
                [ -Lorder        => number,]
                [ -Pagesize      => number,]
                [ -Env           => $env,]
                [ -Txn           => $txn,]
                # BerkeleyDB::Hash specific
                [ -Ffactor       => number,]
                [ -Nelem         => number,]
                [ -Hash          => code reference,]
                [ -DupCompare    => code reference,]


When the "tie" interface is used, reading from and writing to the database
is achieved via the tied hash. In this case the database operates like
a Perl associative array that happens to be stored on disk.

In addition to the high-level tied hash interface, it is possible to
make use of the underlying methods provided by Berkeley DB

=head2 Options

In addition to the standard set of options (see L<COMMON OPTIONS>)
B<BerkeleyDB::Hash> supports these options:

=over 5

=item -Property

Used to specify extra flags when opening a database. The following
flags may be specified by logically OR'ing together one or more of the
following values:

B<DB_DUP>

When creating a new database, this flag enables the storing of duplicate
keys in the database. If B<DB_DUPSORT> is not specified as well, the
duplicates are stored in the order they are created in the database.

B<DB_DUPSORT>

Enables the sorting of duplicate keys in the database. Ignored if
B<DB_DUP> isn't also specified.

=item -Ffactor

=item -Nelem

See the Berkeley DB documentation for details of these options.

=item -Hash

Allows you to provide a user defined hash function. If not specified, 
a default hash function is used. Here is a template for a user-defined
hash function

    sub hash
    {
        my ($data) = shift ;
        ...
        # return the hash value for $data
	return $hash ;
    }

    tie %h, "BerkeleyDB::Hash", 
        -Filename => $filename, 
    	-Hash     => \&hash,
	...

See L<""> for an example.

=item -DupCompare

Used in conjunction with the B<DB_DUPOSRT> flag. 

    sub compare
    {
	my ($key, $key2) = @_ ;
        ...
        # return  0 if $key1 eq $key2
        #        -1 if $key1 lt $key2
        #         1 if $key1 gt $key2
        return (-1 , 0 or 1) ;
    }

    tie %h, "BerkeleyDB::Hash", 
        -Filename   => $filename, 
	-Property   => DB_DUP|DB_DUPSORT,
    	-DupCompare => \&compare,
	...

=back


=head2 Methods

B<BerkeleyDB::Hash> only supports the standard database methods.
See L<COMMON DATABASE METHODS>.

=head2 A Simple Tied Hash Example

## simpleHash

here is the output:

    Banana Exists
    
    orange -> orange
    tomato -> red
    banana -> yellow

Note that the like ordinary associative arrays, the order of the keys
retrieved from a Hash database are in an apparently random order.

=head2 Another Simple Hash Example

Do the same as the previous example but not using tie.

## simpleHash2

=head2 Duplicate keys

The code below is a variation on the examples above. This time the hash has
been inverted. The key this time is colour and the value is the fruit name.
The B<DB_DUP> flag has been specified to allow duplicates.

##dupHash

here is the output:

    orange -> orange
    yellow -> banana
    red -> apple
    red -> tomato
    green -> banana
    green -> apple

=head2 Sorting Duplicate Keys

In the previous example, when there were duplicate keys, the values are
sorted in the order they are stored in. The code below is
identical to the previous example except the B<DB_DUPSORT> flag is
specified.

##dupSortHash

Notice that in the output below the duplicate values are sorted.

    orange -> orange
    yellow -> banana
    red -> apple
    red -> tomato
    green -> apple
    green -> banana

=head2 Custom Sorting Duplicate Keys

Another variation 

TODO

=head2 Changing the hash

TODO

=head2 Using db_stat

TODO

=head1 BerkeleyDB::Btree

Equivalent to calling B<db_open> with type B<DB_BTREE> in Berkeley DB 2.x and
calling B<db_create> followed by B<DB-E<gt>open> with type B<DB_BTREE> in
Berkeley DB 3.x. 

Two forms of constructor are supported:


    $db = new BerkeleyDB::Btree
                [ -Filename      => "filename", ]
                [ -Subname       => "sub-database name", ]
                [ -Flags         => flags,]
                [ -Property      => flags,]
                [ -Mode          => number,]
                [ -Cachesize     => number,]
                [ -Lorder        => number,]
                [ -Pagesize      => number,]
                [ -Env           => $env,]
                [ -Txn           => $txn,]
                # BerkeleyDB::Btree specific
                [ -Minkey        => number,]
                [ -Compare       => code reference,]
                [ -DupCompare    => code reference,]
                [ -Prefix        => code reference,]

and this

    [$db =] tie %hash, 'BerkeleyDB::Btree', 
                [ -Filename      => "filename", ]
                [ -Subname       => "sub-database name", ]
                [ -Flags         => flags,]
                [ -Property      => flags,]
                [ -Mode          => number,]
                [ -Cachesize     => number,]
                [ -Lorder        => number,]
                [ -Pagesize      => number,]
                [ -Env           => $env,]
                [ -Txn           => $txn,]
                # BerkeleyDB::Btree specific
                [ -Minkey        => number,]
                [ -Compare       => code reference,]
                [ -DupCompare    => code reference,]
                [ -Prefix        => code reference,]

=head2 Options

In addition to the standard set of options (see L<COMMON OPTIONS>)
B<BerkeleyDB::Btree> supports these options:

=over 5

=item -Property

Used to specify extra flags when opening a database. The following
flags may be specified by logically OR'ing together one or more of the
following values:

B<DB_DUP>

When creating a new database, this flag enables the storing of duplicate
keys in the database. If B<DB_DUPSORT> is not specified as well, the
duplicates are stored in the order they are created in the database.

B<DB_DUPSORT>

Enables the sorting of duplicate keys in the database. Ignored if
B<DB_DUP> isn't also specified.

=item Minkey

TODO

=item Compare

Allow you to override the default sort order used in the database. See
L<"Changing the sort order"> for an example.

    sub compare
    {
	my ($key, $key2) = @_ ;
        ...
        # return  0 if $key1 eq $key2
        #        -1 if $key1 lt $key2
        #         1 if $key1 gt $key2
        return (-1 , 0 or 1) ;
    }

    tie %h, "BerkeleyDB::Hash", 
        -Filename   => $filename, 
    	-Compare    => \&compare,
	...

=item Prefix

    sub prefix
    {
	my ($key, $key2) = @_ ;
        ...
        # return number of bytes of $key2 which are 
        # necessary to determine that it is greater than $key1
        return $bytes ;
    }

    tie %h, "BerkeleyDB::Hash", 
        -Filename   => $filename, 
    	-Prefix     => \&prefix,
	...
=item DupCompare

    sub compare
    {
	my ($key, $key2) = @_ ;
        ...
        # return  0 if $key1 eq $key2
        #        -1 if $key1 lt $key2
        #         1 if $key1 gt $key2
        return (-1 , 0 or 1) ;
    }

    tie %h, "BerkeleyDB::Hash", 
        -Filename   => $filename, 
    	-DupCompare => \&compare,
	...

=back

=head2 Methods

B<BerkeleyDB::Btree> supports the following database methods.
See also L<COMMON DATABASE METHODS>.

All the methods below return 0 to indicate success.

=over 5

=item $status = $db->db_key_range($key, $less, $equal, $greater [, $flags])

Given a key, C<$key>, this method returns the proportion of keys less than 
C<$key> in C<$less>, the proportion equal to C<$key> in C<$equal> and the
proportion greater than C<$key> in C<$greater>.

The proportion is returned as a double in the range 0.0 to 1.0.

=back

=head2 A Simple Btree Example

The code below is a simple example of using a btree database.

## btreeSimple

Here is the output from the code above. The keys have been sorted using
Berkeley DB's default sorting algorithm.

    Smith
    Wall
    mouse


=head2 Changing the sort order

It is possible to supply your own sorting algorithm if the one that Berkeley
DB used isn't suitable. The code below is identical to the previous example
except for the case insensitive compare function.

## btreeSortOrder

Here is the output from the code above.

    mouse
    Smith
    Wall

There are a few point to bear in mind if you want to change the
ordering in a BTREE database:

=over 5

=item 1.

The new compare function must be specified when you create the database.

=item 2.

You cannot change the ordering once the database has been created. Thus
you must use the same compare function every time you access the
database.

=back 

=head2 Using db_stat

TODO

=head1 BerkeleyDB::Recno

Equivalent to calling B<db_open> with type B<DB_RECNO> in Berkeley DB 2.x and
calling B<db_create> followed by B<DB-E<gt>open> with type B<DB_RECNO> in
Berkeley DB 3.x. 

Two forms of constructor are supported:

    $db = new BerkeleyDB::Recno
                [ -Filename      => "filename", ]
                [ -Subname       => "sub-database name", ]
                [ -Flags         => flags,]
                [ -Property      => flags,]
                [ -Mode          => number,]
                [ -Cachesize     => number,]
                [ -Lorder        => number,]
                [ -Pagesize      => number,]
                [ -Env           => $env,]
                [ -Txn           => $txn,]
                # BerkeleyDB::Recno specific
                [ -Delim           => byte,]
                [ -Len             => number,]
                [ -Pad             => byte,]
                [ -Source          => filename,]

and this

    [$db =] tie @arry, 'BerkeleyDB::Recno', 
                [ -Filename      => "filename", ]
                [ -Subname       => "sub-database name", ]
                [ -Flags         => flags,]
                [ -Property      => flags,]
                [ -Mode          => number,]
                [ -Cachesize     => number,]
                [ -Lorder        => number,]
                [ -Pagesize      => number,]
                [ -Env           => $env,]
                [ -Txn           => $txn,]
                # BerkeleyDB::Recno specific
                [ -Delim           => byte,]
                [ -Len             => number,]
                [ -Pad             => byte,]
                [ -Source          => filename,]

=head2 A Recno Example

Here is a simple example that uses RECNO (if you are using a version 
of Perl earlier than 5.004_57 this example won't work -- see 
L<Extra RECNO Methods> for a workaround).

## simpleRecno

Here is the output from the script:

    The array contains 5 entries
    popped black
    shifted white
    Element 1 Exists with value blue
    The last element is green
    The 2nd last element is yellow

=head1 BerkeleyDB::Queue

Equivalent to calling B<db_create> followed by B<DB-E<gt>open> with
type B<DB_QUEUE> in Berkeley DB 3.x. This database format isn't available if
you use Berkeley DB 2.x.

Two forms of constructor are supported:

    $db = new BerkeleyDB::Queue
                [ -Filename      => "filename", ]
                [ -Subname       => "sub-database name", ]
                [ -Flags         => flags,]
                [ -Property      => flags,]
                [ -Mode          => number,]
                [ -Cachesize     => number,]
                [ -Lorder        => number,]
                [ -Pagesize      => number,]
                [ -Env           => $env,]
                [ -Txn           => $txn,]
                # BerkeleyDB::Queue specific
                [ -Len             => number,]
                [ -Pad             => byte,]
                [ -ExtentSize    => number, ]

and this

    [$db =] tie @arry, 'BerkeleyDB::Queue', 
                [ -Filename      => "filename", ]
                [ -Subname       => "sub-database name", ]
                [ -Flags         => flags,]
                [ -Property      => flags,]
                [ -Mode          => number,]
                [ -Cachesize     => number,]
                [ -Lorder        => number,]
                [ -Pagesize      => number,]
                [ -Env           => $env,]
                [ -Txn           => $txn,]
                # BerkeleyDB::Queue specific
                [ -Len             => number,]
                [ -Pad             => byte,]


=head1 BerkeleyDB::Unknown

This class is used to open an existing database. 

Equivalent to calling B<db_open> with type B<DB_UNKNOWN> in Berkeley DB 2.x and
calling B<db_create> followed by B<DB-E<gt>open> with type B<DB_UNKNOWN> in
Berkeley DB 3.x. 

The constructor looks like this:

    $db = new BerkeleyDB::Unknown
                [ -Filename      => "filename", ]
                [ -Subname       => "sub-database name", ]
                [ -Flags         => flags,]
                [ -Property      => flags,]
                [ -Mode          => number,]
                [ -Cachesize     => number,]
                [ -Lorder        => number,]
                [ -Pagesize      => number,]
                [ -Env           => $env,]
                [ -Txn           => $txn,]


=head2 An example 

=head1 COMMON OPTIONS

All database access class constructors support the common set of
options defined below. All are optional.

=over 5

=item -Filename

The database filename. If no filename is specified, a temporary file will
be created and removed once the program terminates.

=item -Subname

Specifies the name of the sub-database to open.
This option is only valid if you are using Berkeley DB 3.x.

=item -Flags

Specify how the database will be opened/created. The valid flags are:

B<DB_CREATE>

Create any underlying files, as necessary. If the files do not already
exist and the B<DB_CREATE> flag is not specified, the call will fail.

B<DB_NOMMAP>

Not supported by BerkeleyDB.

B<DB_RDONLY>

Opens the database in read-only mode.

B<DB_THREAD>

Not supported by BerkeleyDB.

B<DB_TRUNCATE>

If the database file already exists, remove all the data before
opening it.

=item -Mode

Determines the file protection when the database is created. Defaults
to 0666.

=item -Cachesize

=item -Lorder

=item -Pagesize

=item -Env

When working under a Berkeley DB environment, this parameter

Defaults to no environment.

=item -Txn

TODO.

=back

=head1 COMMON DATABASE METHODS

All the database interfaces support the common set of methods defined
below.

All the methods below return 0 to indicate success.

=head2 $status = $db->db_get($key, $value [, $flags])

Given a key (C<$key>) this method reads the value associated with it
from the database. If it exists, the value read from the database is
returned in the C<$value> parameter.

The B<$flags> parameter is optional. If present, it must be set to B<one>
of the following values:

=over 5

=item B<DB_GET_BOTH>

When the B<DB_GET_BOTH> flag is specified, B<db_get> checks for the
existence of B<both> the C<$key> B<and> C<$value> in the database.

=item B<DB_SET_RECNO>

TODO.

=back

In addition, the following value may be set by logically OR'ing it into
the B<$flags> parameter:

=over 5

=item B<DB_RMW>

TODO

=back


=head2 $status = $db->db_put($key, $value [, $flags])

Stores a key/value pair in the database.

The B<$flags> parameter is optional. If present it must be set to B<one>
of the following values:

=over 5

=item B<DB_APPEND>

This flag is only applicable when accessing a B<BerkeleyDB::Recno>
database.

TODO.


=item B<DB_NOOVERWRITE>

If this flag is specified and C<$key> already exists in the database,
the call to B<db_put> will return B<DB_KEYEXIST>.

=back

=head2 $status = $db->db_del($key [, $flags])

Deletes a key/value pair in the database associated with C<$key>.
If duplicate keys are enabled in the database, B<db_del> will delete
B<all> key/value pairs with key C<$key>.

The B<$flags> parameter is optional and is currently unused.

=head2 $status = $db->db_sync()

If any parts of the database are in memory, write them to the database.

=head2 $cursor = $db->db_cursor([$flags])

Creates a cursor object. This is used to access the contents of the
database sequentially. See L<CURSORS> for details of the methods
available when working with cursors.

The B<$flags> parameter is optional. If present it must be set to B<one>
of the following values:

=over 5

=item B<DB_RMW>

TODO.

=back

=head2 ($flag, $old_offset, $old_length) = $db->partial_set($offset, $length) ;

TODO

=head2 ($flag, $old_offset, $old_length) = $db->partial_clear() ;

TODO

=head2 $db->byteswapped()

TODO

=head2 $db->type()

Returns the type of the database. The possible return code are B<DB_HASH>
for a B<BerkeleyDB::Hash> database, B<DB_BTREE> for a B<BerkeleyDB::Btree>
database and B<DB_RECNO> for a B<BerkeleyDB::Recno> database. This method
is typically used when a database has been opened with
B<BerkeleyDB::Unknown>.

=item $ref = $db->db_stat()

Returns a reference to an associative array containing information about
the database. The keys of the associative array correspond directly to the
names of the fields defined in the Berkeley DB documentation. For example,
in the DB documentation, the field B<bt_version> stores the version of the
Btree database. Assuming you called B<db_stat> on a Btree database the
equivalent field would be accessed as follows:

    $version = $ref->{'bt_version'} ;

If you are using Berkeley DB 3.x, this method will work will all database
formats. When DB 2.x is used, it only works with B<BerkeleyDB::Btree>.

=head2 $status = $db->status()

Returns the status of the last C<$db> method called.

=head1 CURSORS

A cursor is used whenever you want to access the contents of a database
in sequential order.
A cursor object is created with the C<db_cursor>

A cursor object has the following methods available:

=head2 $newcursor = $cursor->c_dup($flags)

Creates a duplicate of C<$cursor>. This method needs Berkeley DB 3.0.x or better.

The C<$flags> parameter is optional and can take the following value:

=over 5

=item DB_POSITION

When present this flag will position the new cursor at the same place as the
existing cursor.

=back

=head2 $status = $cursor->c_get($key, $value, $flags)

Reads a key/value pair from the database, returning the data in C<$key>
and C<$value>. The key/value pair actually read is controlled by the
C<$flags> parameter, which can take B<one> of the following values:

=over 5

=item B<DB_FIRST>

Set the cursor to point to the first key/value pair in the
database. Return the key/value pair in C<$key> and C<$value>.

=item B<DB_LAST>

Set the cursor to point to the last key/value pair in the database. Return
the key/value pair in C<$key> and C<$value>.

=item B<DB_NEXT>

If the cursor is already pointing to a key/value pair, it will be
incremented to point to the next key/value pair and return its contents.

If the cursor isn't initialised, B<DB_NEXT> works just like B<DB_FIRST>.

If the cursor is already positioned at the last key/value pair, B<c_get>
will return B<DB_NOTFOUND>.

=item B<DB_NEXT_DUP>

This flag is only valid when duplicate keys have been enabled in
a database.
If the cursor is already pointing to a key/value pair and the key of
the next key/value pair is identical, the cursor will be incremented to
point to it and their contents returned.

=item B<DB_PREV>

If the cursor is already pointing to a key/value pair, it will be
decremented to point to the previous key/value pair and return its
contents.

If the cursor isn't initialised, B<DB_PREV> works just like B<DB_LAST>.

If the cursor is already positioned at the first key/value pair, B<c_get>
will return B<DB_NOTFOUND>.

=item B<DB_CURRENT>

If the cursor has been set to point to a key/value pair, return their
contents.
If the key/value pair referenced by the cursor has been deleted, B<c_get>
will return B<DB_KEYEMPTY>.

=item B<DB_SET>

Set the cursor to point to the key/value pair referenced by B<$key>
and return the value in B<$value>.

=item B<DB_SET_RANGE>

This flag is a variation on the B<DB_SET> flag. As well as returning
the value, it also returns the key, via B<$key>.
When used with a B<BerkeleyDB::Btree> database the key matched by B<c_get>
will be the shortest key (in length) which is greater than or equal to
the key supplied, via B<$key>. This allows partial key searches.
See ??? for an example of how to use this flag.

=item B<DB_GET_BOTH>

Another variation on B<DB_SET>. This one returns both the key and
the value.

=item B<DB_SET_RECNO>

TODO.

=item B<DB_GET_RECNO>

TODO.

=back

In addition, the following value may be set by logically OR'ing it into
the B<$flags> parameter:

=over 5

=item B<DB_RMW>

TODO.

=back

=head2  $status = $cursor->c_put($key, $value, $flags)

Stores the key/value pair in the database. The position that the data is
stored in the database is controlled by the C<$flags> parameter, which
must take B<one> of the following values:

=over 5

=item B<DB_AFTER>

When used with a Btree or Hash database, a duplicate of the key referenced
by the current cursor position will be created and the contents of
B<$value> will be associated with it - B<$key> is ignored.
The new key/value pair will be stored immediately after the current
cursor position.
Obviously the database has to have been opened with B<DB_DUP>.

When used with a Recno ... TODO


=item B<DB_BEFORE>

When used with a Btree or Hash database, a duplicate of the key referenced
by the current cursor position will be created and the contents of
B<$value> will be associated with it - B<$key> is ignored.
The new key/value pair will be stored immediately before the current
cursor position.
Obviously the database has to have been opened with B<DB_DUP>.

When used with a Recno ... TODO

=item B<DB_CURRENT>

If the cursor has been initialised, replace the value of the key/value
pair stored in the database with the contents of B<$value>.

=item B<DB_KEYFIRST>

Only valid with a Btree or Hash database. This flag is only really
used when duplicates are enabled in the database and sorted duplicates
haven't been specified.
In this case the key/value pair will be inserted as the first entry in
the duplicates for the particular key.

=item B<DB_KEYLAST>

Only valid with a Btree or Hash database. This flag is only really
used when duplicates are enabled in the database and sorted duplicates
haven't been specified.
In this case the key/value pair will be inserted as the last entry in
the duplicates for the particular key.

=back

=head2  $status = $cursor->c_del([$flags])

This method deletes the key/value pair associated with the current cursor
position. The cursor position will not be changed by this operation, so
any subsequent cursor operation must first initialise the cursor to
point to a valid key/value pair.

If the key/value pair associated with the cursor have already been
deleted, B<c_del> will return B<DB_KEYEMPTY>.

The B<$flags> parameter is not used at present.

=head2 $status = $cursor->c_del($cnt [, $flags])

Stores the number of duplicates at the current cursor position in B<$cnt>.

The B<$flags> parameter is not used at present. This method needs 
Berkeley DB 3.1 or better.

=head2  $status = $cursor->status()

Returns the status of the last cursor method as a dual type.

=head2 Cursor Examples

TODO

Iterating from first to last, then in reverse.

examples of each of the flags.

=head1 JOIN

Join support for BerkeleyDB is in progress. Watch this space.

TODO

=head1 TRANSACTIONS

TODO.

=head1 DBM Filters

A DBM Filter is a piece of code that is be used when you I<always>
want to make the same transformation to all keys and/or values in a DBM
database. All of the database classes (BerkeleyDB::Hash,
BerkeleyDB::Btree and BerkeleyDB::Recno) support DBM Filters.

There are four methods associated with DBM Filters. All work
identically, and each is used to install (or uninstall) a single DBM
Filter. Each expects a single parameter, namely a reference to a sub.
The only difference between them is the place that the filter is
installed.

To summarise:

=over 5

=item B<filter_store_key>

If a filter has been installed with this method, it will be invoked
every time you write a key to a DBM database.

=item B<filter_store_value>

If a filter has been installed with this method, it will be invoked
every time you write a value to a DBM database.


=item B<filter_fetch_key>

If a filter has been installed with this method, it will be invoked
every time you read a key from a DBM database.

=item B<filter_fetch_value>

If a filter has been installed with this method, it will be invoked
every time you read a value from a DBM database.

=back

You can use any combination of the methods, from none, to all four.

All filter methods return the existing filter, if present, or C<undef>
in not.

To delete a filter pass C<undef> to it.

=head2 The Filter

When each filter is called by Perl, a local copy of C<$_> will contain
the key or value to be filtered. Filtering is achieved by modifying
the contents of C<$_>. The return code from the filter is ignored.

=head2 An Example -- the NULL termination problem.

Consider the following scenario. You have a DBM database that you need
to share with a third-party C application. The C application assumes
that I<all> keys and values are NULL terminated. Unfortunately when
Perl writes to DBM databases it doesn't use NULL termination, so your
Perl application will have to manage NULL termination itself. When you
write to the database you will have to use something like this:

    $hash{"$key\0"} = "$value\0" ;

Similarly the NULL needs to be taken into account when you are considering
the length of existing keys/values.

It would be much better if you could ignore the NULL terminations issue
in the main application code and have a mechanism that automatically
added the terminating NULL to all keys and values whenever you write to
the database and have them removed when you read from the database. As I'm
sure you have already guessed, this is a problem that DBM Filters can
fix very easily.

## nullFilter

Hopefully the contents of each of the filters should be
self-explanatory. Both "fetch" filters remove the terminating NULL,
and both "store" filters add a terminating NULL.


=head2 Another Example -- Key is a C int.

Here is another real-life example. By default, whenever Perl writes to
a DBM database it always writes the key and value as strings. So when
you use this:

    $hash{12345} = "something" ;

the key 12345 will get stored in the DBM database as the 5 byte string
"12345". If you actually want the key to be stored in the DBM database
as a C int, you will have to use C<pack> when writing, and C<unpack>
when reading.

Here is a DBM Filter that does it:

## intFilter

This time only two filters have been used -- we only need to manipulate
the contents of the key, so it wasn't necessary to install any value
filters.

=head1 Using BerkeleyDB with MLDBM

Both BerkeleyDB::Hash and BerkeleyDB::Btree can be used with the MLDBM
module. The code fragment below shows how to open associate MLDBM with
BerkeleyDB::Btree. To use BerkeleyDB::Hash just replace
BerkeleyDB::Btree with BerkeleyDB::Hash.

    use strict ;
    use BerkeleyDB ;
    use MLDBM qw(BerkeleyDB::Btree) ;
    use Data::Dumper;
 
    my $filename = 'testmldbm' ;
    my %o ;
     
    unlink $filename ;
    tie %o, 'MLDBM', -Filename => $filename,
                     -Flags    => DB_CREATE
                    or die "Cannot open database '$filename: $!\n";
 
See the MLDBM documentation for information on how to use the module
and for details of its limitations.

=head1 EXAMPLES

TODO.

=head1 HINTS & TIPS

=head2 Sharing Databases With C Applications

There is no technical reason why a Berkeley DB database cannot be
shared by both a Perl and a C application.

The vast majority of problems that are reported in this area boil down
to the fact that C strings are NULL terminated, whilst Perl strings
are not. See L<An Example -- the NULL termination problem.> in the DBM
FILTERS section for a generic way to work around this problem.


=head2 The untie Gotcha

TODO

=head1 COMMON QUESTIONS

This section attempts to answer some of the more common questions that
I get asked.


=head2 Relationship with DB_File

Before Berkeley DB 2.x was written there was only one Perl module that
interfaced to Berkeley DB. That module is called B<DB_File>. Although
B<DB_File> can be build with Berkeley DB 1.x, 2.x or 3.x, it only provides
an interface to the functionality available in Berkeley DB 1.x. That
means that it doesn't support transactions, locking or any of the other
new features available in DB 2.x or 3.x.

=head2 How do I store Perl data structures with BerkeleyDB?

See L<Using BerkeleyDB with MLDBM>.

=head1 HISTORY

See the Changes file.

=head1 AVAILABILITY

The most recent version of B<BerkeleyDB> can always be found
on CPAN (see L<perlmod/CPAN> for details), in the directory
F<modules/by-module/BerkeleyDB>.

The official web site for Berkeley DB is F<http://www.sleepycat.com>.

=head1 COPYRIGHT

Copyright (c) 1997-2001 Paul Marquess. All rights reserved. This program
is free software; you can redistribute it and/or modify it under the
same terms as Perl itself.

Although B<BerkeleyDB> is covered by the Perl license, the library it
makes use of, namely Berkeley DB, is not. Berkeley DB has its own
copyright and its own license. Please take the time to read it.

Here are few words taken from the Berkeley DB FAQ (at
F<http://www.sleepycat.com>) regarding the license:

    Do I have to license DB to use it in Perl scripts?

    No. The Berkeley DB license requires that software that uses
    Berkeley DB be freely redistributable. In the case of Perl, that
    software is Perl, and not your scripts. Any Perl scripts that you
    write are your property, including scripts that make use of Berkeley
    DB. Neither the Perl license nor the Berkeley DB license
    place any restriction on what you may do with them.

If you are in any doubt about the license situation, contact either the
Berkeley DB authors or the author of BerkeleyDB.
See L<"AUTHOR"> for details.


=head1 AUTHOR

Paul Marquess E<lt>Paul.Marquess@btinternet.comE<gt>.

Questions about Berkeley DB may be addressed to E<lt>db@sleepycat.comE<gt>.

=head1 SEE ALSO

perl(1), DB_File, Berkeley DB.

=cut
