/*                                                                
**  Copyright (C) 1996,2007,2010,2019  Smithsonian Astrophysical Observatory 
*/                                                                

/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 3 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License along */
/*  with this program; if not, write to the Free Software Foundation, Inc., */
/*  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.             */
/*                                                                          */

/* pfile.c
**/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "list.h"
#include "pfile.h"
#include "ptemplat.h"
#include "parameter.h"
#include "putil.h"
#include "find.h"
#include "range.h"
#include "iraf.h"
#include "rdb.h"

void c_paramwrite (ParamFile pf);

int parerr;			/* Global error indicator       */
paramfile PFList;		/* Global list pointer          */
paramfile PFFile;		/* Global file pointer          */


char paramToolName[PARAMTOOLNAMELEN];	/* Global file name taken from arg[0] */


ParamFormat ParamFormats[3] = {
  {IRAFParOpen, ".par", paramfind},
  {RDBParOpen, ".rdb", paramfind},
  {0, 0, 0}
};

struct _ParamList _ParamFileCant;
ParamList *ParamFileCant = &_ParamFileCant;

static char pfile_last[129];


static char *
ParamFName (char *filepath)
{
  char *filename;

  if (NULL != (filename = strrchr (filepath, '/')))
    return ++filename;
  else
    return filepath;
}


static char *
ParamFPath (char *filepath)
{
  char *filename;

  if (NULL != (filename = strrchr (filepath, '/')))
    {
      *(++filename) = '\0';
      return filepath;
    }
  else
    return NULL;
}

/* c_minmax_check()
**
** This routine checks to ensure that the min/max values for a parameter (if
** provided) are valid. Its primary purpose is to prevent the possibility of 
** entering an infinite loop when a query would always fail due to invalid 
** range specifications. 
*/
static void
c_minmax_check (ParamFile pf)
{
  Parameter **params;
  int mode, perr;

  if (pf == NULL)
    return;

  mode = pf->mode;		/* store mode- add HHMODE to prevent queries while */
  pf->mode |= HHMODE;		/* cecking for range errors- reset value when done */
  perr = parerr;		/* store parerr - want to keep track of local errs */
  parerr = PARNOERROR;

  for (params = pf->psets->param; *params; params++)
    {
      if ((TypeIndex ((*params)->ptype) != CommentType) &&
	  ((*params)->pmin.value && (*params)->pmax.value))
	{
	  Value vmn, vmx;
	  vmn.value =
	    CXCNewString (VConvert (&((*params)->pmin), NULL, StringType));
	  vmx.value =
	    CXCNewString (VConvert (&((*params)->pmax), NULL, StringType));
	  vmx.type = vmn.type = StringType;
	  vmn.indir = vmx.indir = NULL;

	  if (parerr != PARNOERROR)
	    c_paramerr (1, pfile_last, (*params)->pname);
	  else if (VIsIndirect (&vmn))
	    VAccess (pf, (*params), StringType, &vmn);
	  if (parerr != PARNOERROR)
	    c_paramerr (1, pfile_last, (*params)->pname);
	  else if (VIsIndirect (&vmx))
	    VAccess (pf, (*params), StringType, &vmx);

	  if (parerr != PARNOERROR)
	    c_paramerr (1, pfile_last, (*params)->pname);
	  else if ((vmn.type == RangeType) || !VIsNotRange (&vmn))
	    {
	      /* The parameter contained an enumerated range and a max */
	      parerr = PARRANGEMAX;
	      if (vmn.value)
		free (vmn.value);
	      if (vmn.indir)
		free (vmn.indir);
	      if (vmx.value)
		free (vmx.value);
	      if (vmx.indir)
		free (vmx.indir);
	      c_paramerr (1, pfile_last, (*params)->pname);
	    }
	  else
	    {
	      double vminbuf[SZ_PFLINE / sizeof (double) + 1];
	      double vmaxbuf[SZ_PFLINE / sizeof (double) + 1];

	      VConvert (&vmn, vminbuf, (*params)->ptype);
	      VConvert (&vmx, vmaxbuf, (*params)->ptype);
	      if (!VCompar ((*params)->ptype, vminbuf, vmaxbuf, PRange_LE))
		{
		  /* The parameter's max value was below its min value */
		  parerr = PARMAXGTMIN;
		  if (vmn.value)
		    free (vmn.value);
		  if (vmn.indir)
		    free (vmn.indir);
		  if (vmx.value)
		    free (vmx.value);
		  if (vmx.indir)
		    free (vmx.indir);
		  c_paramerr (1, pfile_last, (*params)->pname);
		}
	    }
	  if (vmn.value)
	    free (vmn.value);
	  if (vmn.indir)
	    free (vmn.indir);
	  if (vmx.value)
	    free (vmx.value);
	  if (vmx.indir)
	    free (vmx.indir);
	}
    }
  pf->mode = mode;		/* reset mode/parerr to their original values */
  parerr = perr;
}


paramfile
c_paramopen (filename, argv, argc, filemode)
     const char *filename;
     char *argv[];
     int argc;
     const char *filemode;
{
  int i;
  ParamFile pf;
  ParamList *plist = NULL;
  ParamList *prefr = NULL;

  char *filepath = NULL;
  char frontmode[10];
  char pbackmode[10];
  char fullname[256];
  char mode[10];
  char *fmode;
  char *defaults = NULL;
  int plister = 0;

  if (!(fmode = getenv ("PDEFAULTS")))
    {
      fmode = filemode ? (char *) filemode : "rw";
    }

  for (i = 0; i < 10 && fmode[i] && fmode[i] != '\n'; i++)
    mode[i] = fmode[i];
  mode[i] = '\0';

  if (fmode[i] == '\n')
    defaults = &fmode[i + 1];
  else
    defaults = NULL;

  strcpy (frontmode, mode);	/* Build up the search modes            */
  strcpy (pbackmode, "r>");
  strcat (frontmode, "<");

  parerr = 0;

  memset (paramToolName, 0, PARAMTOOLNAMELEN * sizeof (char));
  if ((argv != NULL) && (argv[0] != NULL))
    {
      strncpy (paramToolName, ParamFName (argv[0]), PARAMTOOLNAMELEN - 1);
    }



  /* if filename is not specified, we take it from the arg list */
  if (argv && argc >= 2 && argv[1][0] == '@' && argv[1][1] == '@')
    {
      /* filename = ParamFName(&argv[1][2]); */
      filename = &argv[1][2];
      argv++;
      argc--;
    }
  else if ((filename == NULL) || (filename && *filename == '\0'))
    {
      
      {
	if ((argv != NULL) && (argv[0] != NULL))
	  filename = ParamFName (argv[0]);
      }
    }

  if (argc == 2 && !strcmp (argv[1], "+"))
    {
      plister++;
    }

  if (filename != NULL && *filename != '\0')
    {
      /* save for errors */
      strncpy (pfile_last, filename, 128);
      pfile_last[128] = '\0';

      /* Look to see if the file is already open.
       */
      if (!strpbrk (mode, "f"))
	{
	  char *checkpath = mode;	/* Any path will do     */

	  if (strpbrk (mode, "R"))	/* unless Reference     */
	    checkpath = pbackmode;
	  else if ((pf = ParamPSetLookup (filename)))
	    return pf;


	  if ((pf =
	       ParamFileLookup (c_paramfind
				(filename, checkpath, PARAMEXTN, PARAMPATH))))
	    return pf;
	}

      /* Try each parameter file parser in turn.
       */
      parerr = PARNOERROR;
      for (i = 0;
	   (plist == NULL) && (prefr == NULL) && ParamFormats[i].open != NULL;
	   i++)
	{

	  /* If the open search mode is Reference or the pfile can't be found
	   * in the front part of the search path.
	   */
	  if ((plist =
	       (*ParamFormats[i].open) (filename,
					(*ParamFormats[i].find) (filename,
								 frontmode,
								 ParamFormats
								 [i].extn,
								 PARAMPATH),
					mode)) == NULL || strpbrk (mode, "R"))
	    {

	      /* Try to find the file in the back path
	       */
	      char *refrname, *refrbase = NULL;

	      refrbase = (char *) malloc (strlen (filename) + 1);	/* Kill any path */
	      strcpy (refrbase, filename);
	      refrname = ParamFName (refrbase);

	      prefr =
		(*ParamFormats[i].open) (refrname,
					 (*ParamFormats[i].find) (refrname,
								  pbackmode,
								  ParamFormats
								  [i].extn,
								  PARAMPATH),
					 pbackmode);
	      if (parerr != PARNOERROR)
		prefr = NULL;

	      free (refrbase);
	      refrbase = NULL;
	    }
	}
    }

  if ((plist == NULL) && (prefr == NULL))
    {				/* Can't find any pfile */
      if (!defaults)
	{
	  parerr = PARNOFILE;
	  return NULL;
	}
      else
	{
	  prefr = IRAFStrOpen (filename, defaults);
	}
    }

  /* If reference is needed make sure it's ok
   */
  if (strpbrk (mode, "R") && ((prefr == NULL) || prefr == ParamFileCant))
    {
      parerr = PARNOREF;
      return NULL;
    }

  if (plist == ParamFileCant)
    {
      /* Something is wrong with the parameter file 
       */
      if (strpbrk (mode, "R"))
	{
	  /* Bogus user copy but the reference is OK.
	   */
	  plist = NULL;
	}
      else
	{
	  parerr = PARCANTPARSE;
	  return NULL;
	}
    }

  if (prefr)
    {				/* If prefr was accessed then we are in R mode
				 * or plist was not found
				 */
      if (plist)
	{
	  /* Open existing pfile from reference;    "R" mode
	   */
	  fclose (prefr->file);	/* Get rid of the reference file's fp   */

	  prefr->file = plist->file;	/* put user file's fp into ref's plist  */
	  prefr->mode = CXCNewString ("w");
	  plist->file = NULL;

	  PFree (plist);	/* Toss out the users plist             */
	  plist = prefr;
	}
      else
	{
	  char *pathbase = NULL;
	  /* Open new pfile from reference;         "R" mode or found only in back
	   */

	  /* The front path search failed but a reference copy was opened
	   *
	   * If no path was given in the input file name
	   * look through front path for a writable directory
	   */
	  if (filename == NULL || !strpbrk (mode, "w"))
	    {
	      filepath = NULL;	/* prefr is from a defaults string 
				   w/no pfile */
	    }
	  else
	    {
	      pathbase = (char *) malloc (strlen (filename) + 1);
	      strcpy (pathbase, filename);

	      /* if no path is given ... */
	      if ((filepath = ParamFPath (pathbase)) == NULL)
		{
		  /* look for a writable directory in the "front" */
		  filepath = cxcFind (".", "wr<", NULL, PARAMPATH);
		  if ((filepath == NULL) || !strcmp (filepath, "."))
		    {
		      /* if no writable, look for any directory in the "front" */
		      filepath = cxcFind (".", "r<", NULL, PARAMPATH);
		      /* no directories at all => don't write */
		      if ((filepath == NULL) || !strcmp (filepath, "."))
			{
			  filepath = NULL;
			}
		      /* if we found directories, but not writable, that's bad */
		      else
			{
			  parerr = PARBACK2FRONT;
			  free (pathbase);
			  pathbase = NULL;
			  return NULL;
			}
		    }
		  else
		    {
		      /* Nuke the dot */
		      if (filepath[strlen (filepath) - 1] == '.')
			filepath[strlen (filepath) - 1] = '\0';
		    }
		}
	    }

	  /* Substitute the reference file's file pointer with a newly opened
	   * one in a user writable directory
	   */
	  plist = prefr;	/* The reference list is the new file   */
	  if (filepath == NULL || plist->filename == NULL)
	    {
	      /* ensure that we do not write the results out */
	      plist->mode = CXCNewString ("r");
	    }
	  else
	    {
	      /* copy the reference file to the writable directory */
	      plist->mode = CXCNewString (mode);
	      strcpy (fullname, filepath);
	      if (fullname[strlen (fullname) - 1] != '/')
		strcat (fullname, "/");
	      strcat (fullname, ParamFName (plist->filename));
	      fclose (plist->file);
	      /*open the file in the writable directory */
	      if ((plist->file = fopen (fullname, "w")) == NULL)
		{
		  parerr = PARBACK2FRONT;
		  free (pathbase);
		  pathbase = NULL;
		  return NULL;
		}
	      /* Write the new guy out
	       */
	      (*plist->writ) (plist->file, plist);
	    }
	  free (pathbase);
	  pathbase = NULL;
	}
    }

  /* Got a plist now make it a ParamFile
   */
  pf = (struct _ParamFile *) malloc (sizeof (struct _ParamFile));
  pf->alist = NULL;
  pf->psets = plist;

  /* Promote the mode parameter to the plist structure
   */
  pf->mode = ParamFileMode (pf) | (strchr (mode, 'H') ? HHMODE : 0) |
    (strchr (mode, 'L') ? LLMODE : 0);

  if (argc > 1)
    {
      if ((pf->alist = ParamFileArgs (pf->psets, &argv[1], argc - 1)) == NULL)
	/* Leak Leak Leak */
	return NULL;
    }
  else
    pf->alist = NULL;


  /* Promote the mode parameter to the plist structure
   */
  pf->mode = ParamFileMode (pf) | (strchr (mode, 'H') ? HHMODE : 0) |
    (strchr (mode, 'L') ? LLMODE : 0);

  /* Push the file on the Big list
   */
  pf->next = NULL;
  PFList = ListPush (PFList, pf);


  if (plister)
    {
      c_paramlist (pf);
      exit (3);
    }

  c_minmax_check (pf);		/* avoid infinite queries from invalid ranges */

  pf->refcount = 1;
  return pf;
}

void
c_paramunlock (pf)
     ParamFile pf;
{
  if (pf == NULL)
    return;

  UnLock (pf->psets->file);
}


void
c_paramclose (pf)
     ParamFile pf;
{

  if (pf == NULL)
    return;

  if (pf->psets && strchr (pf->psets->mode, 'w'))
    c_paramwrite (pf);

  if (--pf->refcount == 0)
    {
      /* Delete the parameter file from the global list */
      PFList = ListDelete ((Listll*) PFList, (Listll*) pf);

      if (pf->psets != NULL)
	PFree (pf->psets);
      if (pf->alist != NULL)
	PFree (pf->alist);
      free (pf);
      pf = NULL;
    }
}


/* returns the path where the current parameter file is loaded from */
char *
c_param_get_path (ParamFile pf)
{
  int len = strlen (pf->psets->filename) + 1;
  char *val = NULL;

  if ((val = malloc (len * sizeof (char))) != NULL)
    {
      strcpy (val, pf->psets->filename);
    }

  return (val);
}



/* Sync with pfile.h */

char *parammessage[] = {
  "parameter error?", "parameter not found", "parameter not unique",
  "parameter file not found", "parameter argument error",
  "positional parameters must come before absolutes",
  "parameter has unknown type", "parameter has bad minimum value",
  "parameter has bad maximum value", "parameter below minimum",
  "parameter above maximum", "parameter pointer missing",
  "cannot convert parameter value", "parameter has no value",
  "command arguments missunderstood", "no parameter file",
  "no parameter converter", "bad value for parameter", "bad subfield name",
  "can't specify range and max field", "bad range specification",
  "parameter type doesn't support range", "value must be in range",
  "too many positional arguments", "error reading paramfile",
  "enumerated value not unique", "can't evaluate indirect parameter",
  "can't open parameter list file", "invalid enumerated value",
  "can't copy reference parameter file to user directory",
  "can't find reference parameter file", "can't open parameter set",
  "positional parameter with no reference list",
  "can't evaluate cyclic indirect parameter",
  "can't evaluate self-redirection",
  "parameter's max value must be >= its min value."
};


static int exit_on_paramerror = 1;

paramerrvector errvector = NULL;

paramerrvector
paramerract (paramerrvector newact)
{
  paramerrvector oldact;

  oldact = errvector;
  errvector = newact;
  return oldact;
}

int
set_paramerror (int new_val)
{
  int old_val;
  old_val = exit_on_paramerror;
  exit_on_paramerror = new_val;
  return (old_val);
}


void
c_paramerr (int level, const char *message, const char *name)
{
  if (errvector == NULL)
    {
      fprintf (stderr, "%s: %s : %s\n", message, parammessage[parerr], name);
      fflush (stderr);

      if (level && exit_on_paramerror)
	exit (level);
    }
  else
    {
      (*errvector) (level, message, name);
    }

  parerr = PARNOERROR;
}


char *
c_paramerrstr (void)
{
  char *message = parammessage[parerr];

  parerr = 0;
  return message;
}


int
ParamFileMode (ParamFile pf)
{
  int hmode = HMODE;
  int *pmode;

  if (0 != (pmode = ParamGetX (pf, "mode", ModeType, &hmode)))
    return *pmode;

  if ( parerr == PARNOTFOUND ) {
    
    pf->psets = PListAdd (pf->psets, ParamCreate (str_duplicate ("mode"),
						  StringType, &hmode,
						  str_duplicate ("ql"), NULL));
    parerr = PARNOERROR;
    return QMODE | LMODE;
 }
  else {
    
    c_paramerr (1, pfile_last, "mode");
  }

  return QMODE | LMODE;
}


void
c_paramwrite (ParamFile pf)
{
  if ((pf->psets == NULL) || (strchr (pf->psets->mode, 'w') == NULL))
    return;
  if (pf->psets->writ)
    (*pf->psets->writ) (pf->psets->file, pf->psets);
}


ParamList *
ParamFileArgs (plist, argv, argc)
     ParamList *plist;
     char *argv[];
     int argc;
{
  return IRAFParArgs (plist, argv, argc);
}


ParamFile
ParamFileLookup (char *fname)
{
  ParamFile pf;

  if (fname == NULL)
    return NULL;
  if (*fname == '\0')
    return PFList;

  for (pf = PFList; pf; pf = pf->next)
    if (pf->psets->filename && !strcmp (fname, pf->psets->filename))
      break;

  if (pf)
    pf->refcount += 1;
  return pf;
}

ParamFile
ParamPSetLookup (const char *psetn)
{
  ParamFile pf;

  if (psetn == NULL)
    return NULL;
  if (*psetn == '\0')
    return PFList;

  for (pf = PFList; pf; pf = pf->next)
    {
      if (pf->psets->psetname && !strcmp (psetn, pf->psets->psetname))
	break;
    }

  if (pf)
    pf->refcount += 1;
  return pf;
}


char *
c_paramfind (const char *name, char *mode, char *extn, char *path)
{
  char *retname;

  if (NULL != (retname = cxcFind (name, mode, extn, path)))
    return retname;

  retname = locatepfile (name);
  return retname;
}


void
c_paramlist (ParamFile pf)
{
  Parameter **params;
  char cbuf[SZ_PFLINE];
  int len;
  int tlen;

  char class[1024];		/* Only I could do it */

  if (pf == NULL)
    return;

  *cbuf = '\0';
  len = 0;
  fprintf (stdout, "\nParameters for %s\n\n", pf->psets->filename);

  for (params = pf->psets->param; *params; params++)
    {

      Value pvalue;
      Value *xvalue;
      char dv[SZ_PFLINE];
      char *dvp;

      if (TypeIndex ((*params)->ptype) == CommentType)
	{
	  tlen = strlen ((char *) (*params)->pvalue.value) + 1;
	  if (len + tlen < SZ_PFLINE)
	    {
	      strcat (cbuf, (char *) (*params)->pvalue.value);
	      strcat (cbuf, "\n");
	      len += tlen + 1;
	    }
	  continue;
	}

      if (((*params)->pmode & HMODE) != 0)
	{
	  *cbuf = '\0';
	  len = 0;
	  continue;
	}

      if (*cbuf)
	{
	  fprintf (stdout, "%s", cbuf);
	  *cbuf = '\0';
	  len = 0;
	}

      dv[0] = '\0';


      pvalue.value = NULL;
      pvalue.indir = NULL;

      if (IndirectClassing (pf, (*params), &pvalue, class))
	{
	  strcpy (dv, class);
	  strcat (dv, ":");
	  dvp = VConvert (&pvalue, class, StringType);
	  strcat (dv, class);
	}
      else
	{
	  dvp = VConvert (&(*params)->pvalue, dv, StringType);
	  VNewValue (&pvalue, (*params)->pvalue.value,
		     (*params)->pvalue.type);
	}

      /* If the PVALUE of the value is an indirection lookup the
       * PWORTH of it. 
       */
      if (dvp && VIsIndirect (&pvalue))
	{
	  strcat (dv, " -> ");
	  if (NULL != (xvalue = VAccess (pf, *params, StringType, &pvalue)))
	    {
	      char *foo;
	      strcat (dv, foo = QuoteValue (pvalue.value, NULL));
	      free (foo);
	      foo = NULL;
	    }
	  else
	    strcat (dv, "INDEF");
	}
      if (pvalue.value)
	{
	  free (pvalue.value);
	  pvalue.value = NULL;
	}
      if (pvalue.indir)
	{
	  free (pvalue.indir);
	  pvalue.indir = NULL;
	}

      fprintf (stdout, "%14s = %-16s %s\n", (*params)->pname, dvp ? dv : "",
	       (*params)->pprompt ? (*params)->pprompt : "");
    }

  *cbuf = '\0';
  len = 0;
  for (params = pf->psets->param; *params; params++)
    {

      Value pvalue;
      Value *xvalue;
      char dv[SZ_PFLINE];
      char name[SZ_PFLINE];
      char value[SZ_PFLINE];
      char *dvp;

      if (TypeIndex ((*params)->ptype) == CommentType)
	{
	  tlen = strlen ((char *) (*params)->pvalue.value) + 1;
	  if (len + tlen < SZ_PFLINE)
	    {
	      strcat (cbuf, (char *) (*params)->pvalue.value);
	      strcat (cbuf, "\n");
	      len += tlen + 1;
	    }
	  continue;
	}

      if (((*params)->pmode & HMODE) == 0)
	{
	  *cbuf = '\0';
	  len = 0;
	  continue;
	}

      if (*cbuf)
	{
	  fprintf (stdout, "%s", cbuf);
	  *cbuf = '\0';
	  len = 0;
	}

      dv[0] = '\0';
      dvp = VConvert (&(*params)->pvalue, dv, StringType);


      pvalue.value = NULL;
      pvalue.indir = NULL;

      if (IndirectClassing (pf, (*params), &pvalue, class))
	{
	  strcpy (dv, class);
	  strcat (dv, ":");
	  dvp = VConvert (&pvalue, class, StringType);
	  strcat (dv, class);
	}
      else
	{
	  VNewValue (&pvalue, (*params)->pvalue.value,
		     (*params)->pvalue.type);
	}

      /* If the PVALUE of the value is an indirection lookup the
       * PWORTH of it. 
       */
      if (dvp && VIsIndirect (&pvalue))
	{
	  strcat (dv, " -> ");
	  if (NULL != (xvalue = VAccess (pf, *params, StringType, &pvalue)))
	    strcat (dv, pvalue.value);
	  else
	    strcat (dv, "INDEF");
	}
      if (pvalue.value)
	{
	  free (pvalue.value);
	  pvalue.value = NULL;
	}
      if (pvalue.indir)
	{
	  free (pvalue.indir);
	  pvalue.indir = NULL;
	}

      strcpy (name, "(");
      strncat (name, (*params)->pname, SZ_PFLINE - 2);
      name[SZ_PFLINE - 1] = '\0';
      if (dvp)
	{
	  strncpy (value, dv, SZ_PFLINE);
	  value[SZ_PFLINE - 1] = '\0';
	}
      else
	*value = '\0';
      strcat (value, ")");
      fprintf (stdout, "%14s = %-16s %s\n", name, value,
	       (*params)->pprompt ? (*params)->pprompt : "");
    }
  fprintf (stdout, "\n");
}


/* Lookup a Parameter in a plist with exact strcmp
 */
Parameter *
ParamLookup (ParamList * plist, const char *pname)
{
  Parameter **param;

  if (plist == NULL)
    return NULL;

  for (param = plist->param; *param; param++)
    {
      if (!strcmp ((*param)->pname, pname))
	return (*param);
    }

  return NULL;
}

/* Lookup a Parameter in a plist with uniq match
 */
char *
ParamMatch (ParamList * plist, char *pname)
{
  Parameter *a = NULL;
  Parameter *b = NULL;
  Parameter **param;
  char name[256];
  char *pn;

  if (plist == NULL)
    return CXCNewString (pname);

  /* don't try to look up extension part */
  strcpy (name, pname);
  if (NULL != (pn = strrchr (name, '.')))
    {
      if (!strcmp (pn, ".p_mode") ||
	  !strcmp (pn, ".p_value") ||
	  !strcmp (pn, ".p_min") ||
	  !strcmp (pn, ".p_max") || !strcmp (pn, ".p_prompt"))
	{
	  *pn = '\0';
	}
    }

  for (param = plist->param; *param; param++)
    {

      if (!strcmp ((*param)->pname, name))
	return pname;

      if ((strstr ((*param)->pname, name) == (*param)->pname))
	{
	  if (a && (b == NULL))
	    b = *param;
	  else
	    a = *param;
	}
    }

  if (a && b)
    {
      fprintf (stderr, "parammatch : parameter name %s not uniq (%s, %s)\n",
	       pname, a->pname, b->pname);

      parerr = PARNOTUNIQ;
      return NULL;
    }

  if (a == NULL)
    {
      fprintf (stderr,
	       "parammatch : can't find parameter %s in reference list\n",
	       pname);
      return NULL;
    }

  free (pname);
  pname = NULL;

  pname = CXCNewString (a->pname);
  return pname;
}


/* Parameter structure contruction routines */

void
PFree (ParamList * plist)
{
  Parameter **param;

  if (plist == NULL)
    return;

  for (param = plist->param; *param; param++)
    ParamFree (*param);

  free (plist->param);
  if (plist->file)
    fclose (plist->file);

#define free(x) { free((x)); (x) = NULL; }

  if (plist->filename)
    free (plist->filename);
  if (plist->psetname)
    free (plist->psetname);
  if (plist->mode)
    free (plist->mode);
  free (plist);

}


void
ParamFree (Parameter * param)
{
  if (param->pname)
    free (param->pname);
  if (param->pvalue.value)
    {
      if (param->pvalue.indir && (param->pvalue.value != param->pvalue.indir))
	free (param->pvalue.indir);
      free (param->pvalue.value);
    }

  if (param->pmin.indir && (param->pmin.indir != param->pmin.value))
    free (param->pmin.indir);
  if (param->pmin.value)
    {
      if (param->pmin.type == RangeType)
	{
	  /* pmin value realloced to hold ranges- delete original memory
	   ** allocated for range specification string */
	  if ((((Range *) param->pmin.value)->str) != NULL)
	    free (((Range *) param->pmin.value)->str);
	}
      free (param->pmin.value);
    }

  if (param->pmax.indir && (param->pmax.indir != param->pmax.value))
    free (param->pmax.indir);
  if (param->pmax.value)
    free (param->pmax.value);
  if (param->pdefault)		/* free default text string if allocated */
    {
      free (param->pdefault);
      param->pdefault = NULL;
    }

  if (param->pprompt)
    free (param->pprompt);

  free (param);
}

#undef free

static void
PListExtend (ParamList * plist, size_t n)
{
  plist->nalloc += n;
  plist->param =
    (Parameter **) realloc (plist->param,
			    sizeof (Parameter *) * plist->nalloc);
}

ParamList *
PListCreate (void)
{
  ParamList *plist = (ParamList *) calloc (1, sizeof (ParamList));
  plist->mode = NULL;
  plist->nalloc = 10;
  plist->nparam = 0;

  plist->param = (Parameter **) calloc (plist->nalloc, sizeof (Parameter *));

  return plist;
}


ParamList *
PListAdd (ParamList * plist, Parameter * param)
{
  if (param == NULL)
    return NULL;

  if (plist == NULL)
    plist = PListCreate ();

  if (plist->nparam + 1 >= plist->nalloc)
    PListExtend (plist, PFILE_SZ);

  plist->param[plist->nparam++] = param;
  plist->param[plist->nparam] = NULL;

  return plist;
}

ParamList *
PListDel (ParamList * plist, Parameter * param)
{
  Parameter **p = plist->param;

  while (*p && *p != param)
    p++;

  if (*p)
    {
      int n = plist->nparam - (++p - plist->param);

      memmove (p - 1, p, n);
      plist->param[--plist->nparam] = NULL;
    }

  return plist;
}



int
c_paccess (ParamFile pfile, const char *pname)
{
  if (ParamLookup (pfile->psets, pname))
    {
      return 1;
    }
  else
    {
      parerr = 0;
      return 0;
    }
}

char *
pfilelast (void)
{
  return (pfile_last);
}


/* evaluateIndir()
**
** This routine takes in a paramfile descriptor, the name of a parmater
** and an indirection value. It returns the value that the indirection
** would equate to for the named parameter. its the user's responsibility
** to free the memory returned by this routine
*/

char *
evaluateIndir (paramfile pf, char *name, char *val)
{
  Parameter *param;
  Value vale;
  Value *value = &vale;
  char *ret = NULL;

  if (pf && name && val && (val[0] == ')'))
    {
      ParamList *plist = ((ParamFile) pf)->psets;
      param = ParamLookup (plist, name);
      vale.value = val;
      vale.indir = NULL;
      vale.type = StringType;

      VDirect ((ParamFile) pf, param, value);

      if (value->value)
	{
	  ret = (char *) malloc (strlen (value->value) + 1);
	  strcpy (ret, value->value);
	}
    }

  return (ret);
}
