#!/usr/bin/perl
use strict;
use warnings;
use Test::More 'no_plan';
use Bio::Phylo::IO qw'parse unparse';
use Bio::Phylo::Util::CONSTANT qw':namespaces :objecttypes';

my $seq = _DATUM_;
my $counter = 0;
parse(
   -format => 'fastq',
   -type   => 'sanger', # to indicate how phred scores are scaled
   -handle => \*DATA,
   -flush  => 1, # don't store record, flush and move on
   -handlers => {
   
	   # a handler that is executed on each newly created datum
	   $seq => sub {
		   my $seq = shift;
		   my $name = $seq->get_name;
		   my @char = $seq->get_char;
		   my @anno = @{ $seq->get_annotations };
		   $counter++;
		   my @outside = grep { $_ < 0 || $_ > 93 } map { $_->{phred} } @anno;
		   ok( scalar @outside == 0, "no phreds outside range in $name");
		   ok( @char == @anno, "a phred for every base in $name");
	   }
   }
);
ok($counter == 584,"seen $counter records");


__DATA__
@EOUL1:00014:00026
GAAGACCCTATAGATGACAATTCTTGATCCAAATATTCCGTGGCATATGCATTGTTGGGGTGACAAGAAA
+
EC@CECC>BBA@BBBBAAB@C@CC@C;;;=?A>AAB@CADCC?BBCC::;CC>::5:@@5@BBBB?BCCC
@EOUL1:00027:00035
GAAGACCCTACCTGGTTATCTGGGCCTTCAGTCCCAGGGCATCAAACTTGATGCGTAGGTATTGCCAGATTGTCATTACCTAAATGGCTAATTTAGCCGTAATATAGAGGTATGCGCCAGTTTGACCGAGTGCGCAGGGAGTTACGAATTGATAGGGGGTTGGGGTGACAAGAAA
+
CDACAAC?CCDBBB?E@@>>???:@=@?BBBBBA>BCCBAAAACC@CC>AAABBBBA?;><<:><:=>;;8<?>??>?A?4//,//,/8@>DD?BBB@BBCACBBAAAH@<<<CDBB?BBBB<BCC?BBCBBBCCCCB>B::8:>AA=A466=66666,7:<<<6?BCDC@CCCC
@EOUL1:00041:00019
GAAGACCCTATAGATCTTGATACCGACAACTTAGTACCTGGTCTTGAGGAATGGTGGCAACTCATAGCGCTTGGAACAGCTCGAAAGATCCTACAAGACAGTATGGATATGGACACTGTAGCTTTGATAGATCCTGAATATAGAGTGCAAATGAATTTGTGTAATCGAAGAAGTCTTGTGCAGTACACCAACGAAAGACCAGCAACCATCTATACACAGCAAACGGATGGTACTAATGGTTATAACGGTTGGGGTGA
+
DC@CBCC?BDBB??@AA@ACB@@>@@@C@CCABAAAC@CB@CCC?BBB?B@CC@CB@BB;<<CCCBCCCCB?B@C@CABAAB:9:5:@AB?BBCB@C@?@BBBBC>@@@@@>@A??CCAAAAAA:??===???;<<<:<<<<???>><;9498438B:>???C?AA;95;@=ABCCACCBC@@@??A@=A>:=<B@GDCACCCC@C@BCEBB<;;B@@@DCC<?>598848???A@BC?A>A@@=AA>A>@AA4<=?
@EOUL1:00026:00125
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAA
+
BHECCCC?CCA@?998;>@=?AA;:48>8>7;;:@C=A>?@>@@@??A?C@@?=@?BBBCJA@=?A:;@=@@<BBDCABCBCGCBB@BB?B<<<BBA@B@CC>
@EOUL1:00020:00186
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATGAAAT
+
>EADBBB@BDCCCCC@BBB?ABBCC@CC@D?BB?BC@B@CC?B?BBCC>C?BB@C@CCCBAAA=??=?A?AC@BAA@=@@:>>@@@=@@=>@@@>@@8<99A<@@@99919919=@4;<=@@AABCAAA:@?4746676;??773444-444678>AAC?B===?@@;<<A;?==-3384722---+-
@EOUL1:00004:00236
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
<A?BBCC?AAACDCCCDEACCC<<:<?BBBC=?;?A??6B@@FDDCC?DDCCAA?BCFCAA<<=@AAC:;;B>AAA7;99B>;>>>>
@EOUL1:00025:00226
GAAGACCCTATAGAGCCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAAG
+
AA>ABBC@BBBB>>98588/833399967:::6:>@@FB5B?>>;79909@@>>B?C@>@<@;<<CFCCACA@=@BB7>@;>?-///+/
@EOUL1:00047:00220
GAAGACCCTATGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCAACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAAGAGA
+
DCEDCCE>CBCCC;;9<==>ABAA@CCBBCC>ABB99;;09AAAC@CEGACBBBBBECB?BC@CCB;:<<@@@BBB=B<<;CCACBCC@AACCB>?@@=ACCCCCAA8;;BBB?BC?BAA?A?B>B?@?BBC?BBC?AB::6:A=??AC?C@C@BCBAAADADCBCCC@@@>@C???:??<2<9===;><<<
@EOUL1:00006:00298
GAAGACCCTATAGACAATAACAATGATAATTTCAC
+
44244;;-99=B@8777331/4,.....,392331
@EOUL1:00012:00251
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
4954999199499<;C@C<B;;:A<;::AA?4741677.8>>?4799044444;:;299049??@@AA>9;;8=@@4;;BB@>BB;;
@EOUL1:00014:00369
GAAGACCCTATAGAGAGGGTTGGGGTGACAATAAA
+
AB@DAAAAABBA@@@?AA<@:>@A6@?BAAABCCC
@EOUL1:00020:00401
GAAGACCCTATGAGCATTGA
+
CD@DBBB@BBA0*****)((
@EOUL1:00009:00492
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
FE@BACHDCBCC@@BCCC>CCCBB?A>55//.0:;>>>6>BBB:8<CCCCCCCE@CACC?CCBBBB?=:@@?<?AB5<<CCDACDDB
@EOUL1:00007:00521
GAAGACCCTATAGAGAGGTTGGGGTGACAAGAAA
+
A?=ABCEDCCCBB?<@B@C@CBB8AA<?;76666
@EOUL1:00048:00564
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAAA
+
DC@B?@@<A@=BCA@@CDAC@BDE8A;;8<>==BB
@EOUL1:00004:00633
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
:@BDCCCBABBC@CB@?B=AABCB@B>?@BB<9888885@=BC?CBB8;;@@CCEC@CDACCCB<<;B>@@@=@CC9BBDCC><;<9
@EOUL1:00011:00613
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
998:A@C>C<<<;978;;6;B?@@@@???:747:<<<<5@ABBCBBB:<<<<<C?E@AA<AABHAAADACAA=@@@6@CC@@===00
@EOUL1:00080:00047
GAAGACCCTAGTGGAAATCTTCGACCTGCTGTGCGGGATGTCTACGGGTTTCATAGACAAGCTGATCAGTAGGAAGAGGAAGTTCAAGCAGGAGGTCCGGGAGACCGGCGAGAGGCTCAGGAACGAGCTCGATGTCCTTGAGAAGCAGCTATTCACCACCAAGGTGGAGACGGAGACGCTCAAGGATCGAC
+
?A@EBBA>BCAA@=@@<ACCDCCB@=@@;:;BBBBEBEBCBCABBB@<AB:;<CB;;;B8;;BAA@@@BBBB?B?AAC?C?BB>::9;@?@???6:;>?A<BAABCD@CBBBDC@AA???>A><==???@AA???@5838;<:6;=;////733;=<==:;8>=@C??=>AA>@@@A>:711+,+,.****
@EOUL1:00089:00041
GAAAGACCCATGATTTCATTGGGCCATTGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCG
+
944-4::@:A99499-4:;9=B@B?BGCHBBFBBA<<9>?9998884=8487<>>;99>;0-,00,,,//+4
@EOUL1:00074:00056
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAGA
+
FCAEBBB?BBBBAAB@B@B@AA??=6::6;?CC@C?>???@?B?A???>A@AA??@<<?BCCA>>@=AA=A@<@BBB;<<;<CBBB@BB?BCCCCCBBB@BB?BBBBBB?CG=B?B;<;GEBCABBBCC<AC?A>ACCBBBCCC>==>A??A??>A==;=?????>>>8838;@1>888A:==?D
@EOUL1:00050:00129
GAAGACCCTGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAAGAA
+
CC@B??@<@@@@CFEBA>BB@C;:8;BBBBCC?BB@BBBB@BBAAA;@@C=??CCA>::8:899//.0;=>@;@=@@A@@@=:;;==:::=<<<999::99=<=::8:=BBBBBDDDD?@==685;A?ACCCC@CBBBBC@@@499999*9@B>>@>4888.7=?77477>
@EOUL1:00087:00194
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAGA
+
CC@CBCC>BB@A@BCAB>B@CCCBB9997;?AC?AA@C>?;;?>A@@@;C@CI?B?BBC@@?@>@@<<A@DD>ADBC@CCACCAAA?CB@CBDCCB????CC@BBB@@A=AB@C??>BA@BA@@BCA@@;AA>>=ACCCBBCBCAAA:<??BAA>A@@<??AA==<A@@@=@AA6AA@@@;==@A
@EOUL1:00095:00152
GAAGACCCTATGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCAACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAA
+
BE?>?>C?CC?>:111568<>>@B@D?@@CC?BBBAABH@C;=<AACAB?DDCAD:;;B@CC@CC@AACCCBBAAB?BBACCC@EACABAAA:;//11,1;;:@A@B9;:<;DCCC?BBB@D@C?A@@:::A?BBD@CCBB?BC?CCCC@D;::<GCBCBE=BA:700:00,0>:;>=@@@599@AAD
@EOUL1:00051:00210
GAAGACCCTCATCGATCCATCACAAGCCTC
+
CD@DBBC>CAAA>>=>=:-,-,**),,*,,
@EOUL1:00053:00242
GAAGACCCTACACCACTTGCTCTTTACGCCCCAGGAGCACACTCACTTGTAGTCGGCGCAGAAAGTTTGCTGCTGCGTCTGTGTACACGGGCTCGTAAGGCTGGTCTGTGCGGGGCG
+
/1,///1,//@AB>B<:48@A99919>=>?A4@@=<;;AB>?@A;:<=@<;;CCD@CC>?999199919999>BBAAA@;998B;;:ACC>??:::69?;>A@5998944/...*,,
@EOUL1:00093:00240
GAAGACCTATAGAGCTTTACTCTTATTATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGTAAAG
+
DB?B992449>@BBABBBCC?=<:00,0.663:>@>@A7?A@?=7990999ACB?ABBB?@1111/885/1111:808874424///+.
@EOUL1:00099:00209
GAAGACCCTATAGACCAGCGGTACTGGTTTTGGCAGACATTATATTTCACACATTGTAATGACAGCCTTGTAAACCGATAAGCACAGGGGAGTGTCACTAGTTGAGTATATTGAGTATATCGATGTTTGTCAAAAGCCAACGCAAAAAGAAAGGGGAGAAAAGAATTGAAAATAACAGTAACTCCACATCTAGCAAGAGATGATCTGCTCCATTTCTATTTGTCGTTGTTGGGTGACAAGAGA
+
A?9?>>>198A@@@C@AABC@BBA:::;EE9C@CAAACCC@???CC?BCCCCBB?BDCBCBDBCCC@D?B;;;6;>?ABB=@@;;;BBB7@@A;::ADBBBD@CCCCBB>?5:9???888>8888880:?@@CC8BBBG?AA88888,88??BCC9AAAKK9BB7;500??+00,.1418<CCC>@>001--,00,..3.11699<@AAA@>>A>AACCB5::::7=>>444-3840-22--.
@EOUL1:00091:00292
GAAGACCCTGACAGGCTGAGGTTCGAGTTTACACTCAGTTTATACTTGGTTTACACTTTGTTAGATTAAGTTTATATTTGGTTTACACTATGTTTTCATTTTATTGCGATGTATTTGCTCTTAGCCTCACCAACTTGAAGATTCAAACAGATTGAAACTCATTGAATCCATCCACTGACAAGTT
+
?A=AECC@;<BAAA>BB??>6959@@8886;AABBD@??B>;;<AA@C>AA<:::@AA9??;?:::>A??AC:44444-424;:CCBBBAABCC?27.///+00,-:=>?BC???9<789C@BA@=@@AB?B60/9;9,///5///,//-1101545:=4884//,/7478877..,,,5+,+0
@EOUL1:00094:00274
GAAGACCGAGAAACCTGCTGCCAAGAAAGTGGCTACCAAGAAAGCGAAGTAATTGCTTCGGCATGGTTTTAATCTTGACTTGATGTACACAAGGCAGCGTTTGCTGCCTTTTTTATTTCTGTTTTATTCTTTCTAGCATGGTCAAACTCACTGCAGCCGCCACCGTCTCAATTCCACGCATTGTGATCTTTGCGTTGACCCTCGTTTATGGTTTAGCGGGTCTCTTTGGTCGTGATCCTTGGAAGAATGAAGACTCCATTGGTTTTGGGGTG
+
???DED@BA;44-9<999?;;9<@CCDABCC?BAAA>A?ACE>CBBB@CBB=?=?CC?AA@CCDC?@@@4=;>??<?CBC??==?@BCCCC=@>=??B@>=7;?::8<@@@@@-FBA<=>;;;;1;?;??@178??==AB?BAAA18:;ABBEA??B?AB<?A?A@99:C@C@B?BBBC@A>??=@@DCF>BBBB>>;9;6<888C?AA@?@@;C:;:C<??ACCC>B?AACCBCBC?C@C?B?A@488?5888884873:5:EE8CBB9>=
@EOUL1:00051:00320
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGTGTTGCTTACA
+
DC@CBAB=BBBBBAA>B?A>AB@AADCDAC@CB?CC@C=??@CACCCC<B?BC?A>BBBABBA>>B@AB?BCBECBB@CCACCB?>=@C@CCCCCCBCC@CCABCBBBC<>>0959;9848<>>>>877-3376779===AB??<<A48/944,..20110
@EOUL1:00055:00370
GAAGACCCTATAGATCCCTTATACTGAACAAGTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGTAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCTGACATAATACATTTGTTGAGAGGGAAGAAGCAAAG
+
442:AABBAAABAAACKBCDCCBBBBC=@@>@???5868<?=>;;;>.0/>==<@=?:838:;9:;8<<7=AAAABA==9<<<<<CBBCCBBB>@@=@A?AAAAB=BCHDBA88867-7?>>AA<<9<@;999@094848>>>@@@@AA??:?=?:?CAAAA?>==?:==AAC>A===BCDDCCCC;;;@CCC@8::>:===2:4434:::7:947:;646//,.2339@@=;;9>=@9>>=?;::CC=A?AA:::CDAC
@EOUL1:00069:00356
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAGACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTGCATATACAGTTGGGGTGACAATAAA
+
AG?GBCB@?@ACBBB?B@C=@@CAA?@G@C@BB@CC@C@CC=A:99:,/..3..,-,-26;@@@?AA?@B?CCBGAAAACE@CCCCC?BC>A@AAAACCEADC@A?>99699-4242437793333311,3316>>>?888=CA===;=@8;8=??7737;444;;;<==>ACC6?44..,-111
@EOUL1:00080:00474
GAAGACCCTATAGACAATAACAATGATAATTTCACAGACATTGCCCTACAAGACCGTGTTTCCTTTTTTAGTAAATGGACTATCAATCGAAAATCTGGAAAAGGAATTTAACATCGTTTCGCGTTAATTTGTATGAAGATCGTTGGGGTGACAAGAGA
+
BEACCAA><<9@DCBB@AA?BB?C@?>D48808>>@BCDAA?ACC>@?@C@CBB@AA?<:4=477777*7>77759@=@AB@:993440300+049=4777,164266,0/04448;@68.777,/-.4445=<=@;999>>><AA@5@55160----
@EOUL1:00088:00464
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
CD@CBBB>BBBA@?????:<<<@B@B==B?<:677661-8888<89919@@@ABBICAB<@@@<;<BA>B<<;<BB8<=><E@BCDD
@EOUL1:00098:00513
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAGA
+
DAAC@@A>AHCCCB@@AB>B?BCD7A??=?5;:?=
@EOUL1:00078:00573
GAAGACCCTGAAAACTTTGTCAGTAGATTGGTGCTGGGTTATCGATACATGTATGGGGTTGGGGTGACAAGAGA
+
CECEBBB@BCCFF:DCC>@@ACCCCBCC@C?A??BBB>B>@?@@A@B::<AAAACJGBGDBDC9CBDBC@CC@?
@EOUL1:00071:00662
GAAGACCCTATAGATCGCTTTACTCTTA
+
CCA><?J@EBBBB7944433,3//..,.
@EOUL1:00075:00664
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
995;B@@>ACAC@@@BAA<ABB;;47777763?::CCA7BCDDAABB<@;<<@A=@69919>>>>99919;;>CDC9CC@AA>A?>?
@EOUL1:00099:00684
GAAGACCGCGAAAGAGGTTTTTGGGGTGACTGTGAAGGATGCCAAGGGTAACCTGGTGATGGTCGATATTTGGGATCACCCTGGCGCCCCTGCCATTGACCCTACATTTATCTGGTCGAAGGGCATCCTGCAGCATCTGCTGCTGTCGCAGGCTACGGGCGAGAATCTCTGGTTCGGGGGTGACAAGAAA
+
:::<CC?BDCEEBCCE@CCCC3EEH9BBBCCCDBC@CACCCB@B@BC>9;8;@BB@CCAAA?CCHHHAAA<@A<;;<BBB=BB<?8888.99969@?@99919DABCE?>==AA=CHHCDDE@DBDHCGCBCBBBBCA<@?FECCCC@@@EBBBBBBB7;;B>?<@99999;>;>@ABB2@@>>>;8777
@EOUL1:00056:00706
GAAGACCCTATAGAGGGGGTTGGGGTGACAAGAAA
+
D@?@944-48844;@<@A0;9<CC7BBA??77833
@EOUL1:00078:00743
GAAGACCCTATGAGACCTTCAGAGGTGTCTGGTGTATTAACTACATGCCCATAAGTGCAACCCTCGTCGAGTATGGGCGATGAATCAAATATCATAGTTGATGAAATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCAACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAAGAGA
+
CC@CCDC@C@//11//54,00>AA>BAA??@?BEDCC@C?B@=:88>?>8?AA>AAA??=?C>B@@?BB@@@CABB>C;;;AA=ABBB<BBA???777:400::;,0,/:644244294:9=?C@CCCCC@A>ABDDCCBC?BBAA89919<>;8;;BDDJB@B@BA@=:73777648888+335:=<A@<<
@EOUL1:00087:00824
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
EEACBBB>BBBBBBB@B@A@A??@C@CC=A@BN@A>>A@DA@@>BAAA<D?BB>AAC?B>>>>999>>B@BB=A@AB@BB>BBBBB@BC?ACCCCBBCC@CC@CCCCCCBCCCB?B@CC@CCBBBBCCC>BB;?:?99:BBB?@>@@>@A@A@@?A9868@<<<@@?;443444-4;;;;37003
@EOUL1:00089:00813
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGGATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCCTGACATAATTACATTTGTTGAGAGGGAAGAAGCAAAGA
+
CCCDCBC>BB@@A@=@@<?@AB@@@CC@CC@CCAA>A>?:?:9::00,0,-/--24443323:<==A?AA:@@AA977;;8;;;<<@@@?999969959>>>;;;C>C@CCCAAABBB<BBABCBCDAAA?<=;B>C:<9;B;;;BA?7:77;>>9??ABB@@:>@:<8<844;388;?77777777==<;>4437:=8:8::16//,/..7742<;..------+00042:4,.,.77?C;==<=:44640843887339933
@EOUL1:00092:00887
GAAGACCCTATAGATATTACAAGACCTGTACTACCAATTGGATTTTTACCACAAAAAAACAATGTTGTGTAACTTTCAATTGTCCAAGTATAAAATAATGAATCACAACCAATATATGAATAAGATAATGGACCTGACGTTGGGTTTGTTATTGTTAAACAATTACAATTACCATTTGTTGGTGTTGGTGTTCTTGTTGGTGTGATAGTAGGTGTTGGTGTTGGTGTTGGGGTGACAAGAAA
+
CD@DCBB?CCBBBBCBB?BBB=AAC@BCAAAB9>262:57=>@AAA2AB@B;;;9888';?>A@@?BAA?9:433+44/05<?C>@:<;?8888.8?<@AB@CCBBBBC@C:::@??::57?<7;9<8:;87467>BC==9=A<@A9==;=A>AHC@@?@AA?=B;:5;BCB@CB@:774747=<3020449773166689?AA??>A?8838;;9>=<===@=@@@<@@@4<<@@@:88>;
@EOUL1:00107:00036
GAAGACCCTATAGAGCGTTTGACTCTATATATCAATAAAACTATCGAAAGTGTATTAAGTTTAGAGAAGTTTC
+
A?<>99919=9?@@88883,3.//....26....4....*.11116:B:;;@99424244;79///,*,220,
@EOUL1:00117:00034
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
BBACCCC@CCCCCCB@B>A@BCCCC?AA=>?BB?CA@C@BA>A>?BBB=?@B@>>>>@CB@>@>>A>@C@CC?DCCD@CBBCC??A@BB?BBBBBB@AB@CB@BB@???=@B?BAB@BB?BBCC@@@CB6<>473777=>AAAA??B>==>8848<<<8<AA@?@88://./9;1;9==B9<<@@
@EOUL1:00132:00030
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
CDFDCFCDCCBCDGDBAA>ABA@@<<<677747=676748>?B98=CACB===DCCCCCCBBBCCBBB=BBC?BBB9@@@ACACBBB
@EOUL1:00106:00091
GAAGACCCATAGATTGGTACTTGAAGGACTATTTGGATCCGTTGGAAAGTTATTACAACTAAAGAACTCATGGATGCAAAACACTTGGCTGAGTTCAAGATCAAATGTTTATTACTACGCCACCCTGAATCCATATGTCAAGCCGCCAAAAACTTCACATATCAGCAAGAGATTGAGTATCTTGTACTCAACGATTCTAGAAACAAATTCATTACTAACCTCTCGATATCGCTTGAGGGAAACATCTTGGTCTTATATCAATATGTTGAC
+
<<;>CCC>BCBBBA?A@CCCC@BB?B?ABAAAA<@=@;1,1<-/,/8577:1969==<?=::64959AAABDCCAAACCC9CCCDACBCBAA;;8<B@?>@@@@<;;;C;<<?BBBC<<<?AAB><;;>AB:???@@@<<:<BB@@59999,9969>8;<B?>;:00,00>>D@?B@@BA=:88////8878:::.88<<<A:>=@7>;@@A>@CBB?B?BBA??@BBBB@@A@CB@B6:86:AABB@F@CBB>@:::00,111<?=AAA
@EOUL1:00128:00051
GAAGACCCTATAATACAGTACTAGGGCCAATCTAACTTGTTGTTATCCGTTGTTAGACGAGGCCAAACTCTAACAACGAATAGCGTGTTGTTAGATTTTTGACGATCGAAACGTACCAAATCTATCATTCGAAGTTTGGCTTTTTTACGTATTAAATGGCCATATTTGTTAGAATAGAAATATATCCCTTACAAAAGGGTGGGCTAAGGACATCATTTCCCCGTTGGGGTGACAATAAA
+
CD@CAAB?BCCC=A<<<CB?>8444-379<BBCB@CCACCCECABBDFID;<<=@>>9911117>>,118885??@ABA?:=;BBB<<:<C@AAACDCC3A?@A@AA@@@<AAAAA>>><998?@>>?<>884866,13133333(37316<>@@=AA?@>@@BBC>CD?C=<877766,17;=744,3/711>>8??9<8;27//,.5..179777,111,03/000+0446=:88;;
@EOUL1:00138:00087
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTAACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCCTGACATAATACATTTGTTGAGAGGGAAGAAGCAAA
+
DD>AAAC>B@AABBADC@C@DCBBBBC@CB@@BBB>>998?:??<=A7:6;B@@>??@@@>???@AA?AC<A;<=AA@<=:<===<AAACCC9<69><?ABAA@CD=B<=<@??@AAA>@A;;==AAACA?B@@@<@?B@CCC@@@;;;@AB<A?A<?=@77757;;=588@?@8<<@8;;??CB>><7747=<:001>=>>@6@@@=????<?>>@B<<;DDEE>BC???@@=@A@@AC=BB@?@=433/069;;/0098
@EOUL1:00132:00147
GAAGACCCTACGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCAACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAATAAA
+
DEADEDB=?@=>>110::::99C>>?????@?B@<<?AB>BCDCCBBAC>BBB@BBBAB>@B>AABCCCBCCB@@@:?::;AA?BAAABECCDCCBB?@BBBBBBBB@B=>@C@@@=@CC@C>B>BBBB@@D:::B????B?BB@CBBB?B?B?BB??;::09A>>=?59:5:B@AB?A@@5A???A6==CH
@EOUL1:00137:00144
GACCCTTGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACATCTTGGGTCAAGGCCGCA
+
EDEEBC@B999-99=8=@<@AG=CBCCAAC<@;=899?44///64489777:43=;??@:<71166676/74333+--,-+01---.228,44-39998:@B68:>?4::;9BB::>;B;??BCBCBB::;:9::6:@<,0,/,.10
@EOUL1:00138:00160
GAAGACCCTATAGACGTCTTACTCTA
+
A?<B@>>:999;;1.////,,,///.
@EOUL1:00110:00227
GAGACCCTATAGACGCCCGGATTGGTATAACTCTGAGTAAAAGGAACCGGAACAATGGATGAATATCCAGTGGCAGGCAGGGTGCGGATTAAAACGTTTTTCTCGTTGTACAATTCCCATTCCATTCTGTATTGATTGTGGAACCAATCGAAACATCACTGAAGGTAACTGTGAATACATCGGCTGAGCCGGGTGTCTTGCAGAATTTGGTTTGAGAAGCACTAATCTGGGCTATGGCAGTTGGGGTGACAAGAAA
+
G<<<CBAAB@@@@@B@CBB>@A=@<?AAC?DCECCBAACED8A>A@C@C;::<C?@C@;:<C>A@A@=::9A=ADCBCCCD>BBAA>@=;=CC6???===1888@@=>A=;<77,//,//,-2279:<>?@@<?>=:=;;8;;=;;:;444:4447B?;;<?9<8;;:345:;@>ACCB?@?>AAA?==377.////-25544<<CC>C;==7<==?=?==><;8;?;?@=@BBB@::9//,/7757:C@@==<@@
@EOUL1:00132:00221
GAAGACCCTATAGATCCCTTATACTGAACAAGTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCCTGACATAATACAT
+
??;?9998<AABA@ACC>@=ACCC?A@=A@@BBBC8;:<@C@>>>==.0/=;=:B:?:89<;;;<<9<B:??AAAABAA699@@6;<BB<<<A<>>=<99?@88;09=>>@@<<<ANAC@@@B>A@@@A=A@BA=B?B@CCCC@@@C:::@=@5:5:@@@AA=?@@A>AABAC=@9:8;DB???;:7=@67:?@8::B4777.4::344409=9>@@7379344,0084:<?4707,,
@EOUL1:00135:00212
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAAACGCGTTGCTTAC
+
BB>ABBC?AACCC??<><;<@ABBC;<;8<BBB@B=999:64:=?@@B=B?BC@C=@@CAAAB>??=@C@CHBCCA@>AB>BCAAAACB@BBBBBCCCC@CB?CBBBBB>BB>B@C?BA@CCB@@BC?C;BB5607>>777=774444-44404:3..,-1
@EOUL1:00136:00211
GAAGACCCTATAGAAATGGGTCGAGGCAGTTTTCGCCGCTAGCGCGCATTATCAGAACGACGACGAAGCCAGCATCTCGGAAATCCCAGATTCCTCAACCATCGCTGAAATGATGCTCTCGGCTATTCCGTAAAAGCCATCGACTACCTCATGTCCAACAGGTACGATAGACGACCCGACCGAAGGGAGTATGGTGAAGTCCGGGTTGGGGTGACAGAGA
+
IC@EA@@<@<9;A@B<@@@:>@>9959;>BBC8AAC@ACAAAD;::B@@:<<<???A=??BB@??A=873:AAAAACCC78818==:?@@@;=;=AA?A;==@@@@@@@7;<=<<7676//,.9;;=A894444,4:8=?@AAA===>@?;<888=@@BAC@@===@@<<;?;437-44;7:6.//,////;;98//,///,/96;9:==4844...333
@EOUL1:00102:00313
GAAGACCCTATAGAGAAACAAACCGCGACACTGATGCGTGCCCGAGTCCTCTACAATCTCGGAATGAAAGATCTGGGTCAGAAGTTGGTCGAGACGATCGACAAGAGTCCTTACAGCTTGACTGAAGAGGAGCGCAGAGTTCAATCGAGAAGATCAAAGAACTTAAGGACGAAAAGACTCATTGTCATCTGATGTGTTAAACTTGCTTTTGAAAAGGAACTCAAACAGACCCTACAATCTCGAAAGCATCAAGAT
+
995499919;;@9:;8803776><??AB@??ABA??>BA99909?999<>@ABAB?<99AB=><?99909@@@9<918>>@A>@@=@=@@BAA@AA==120=>;>8B?>,0,0:::;B?BAA?@@@=;<88444;;4486469,///-132400055358<@C?B??37//000+0/7777<>??737330003000/00+144../7769:::6665366111,111000+03312...3334+-111160115
@EOUL1:00121:00328
GAAGACCCTGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATCAAG
+
CE@D;;9:<@?;;<BBB?BC@A@CACCCB<;=<::4;AB@=@?7661:@ACCBBBA>CC?A;98//0;=;@;@?@@@A<<<CCCB777@@?CA:=????@?>AA@=@@C>@=11111:::>,1//66
@EOUL1:00144:00380
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATTATACAGTTTGGGGTGACAATAGA
+
AB?A9<9-4?@@@@B>A>A=@;;<B@CC>C;31;::,0,...+.,17778;:;;:;=879:::<:6636=59919449544-49999;@>>?BBAAABBA@@@=>999>>:>B<@=@>BB?B@AB;:7:9/731313776111...45+//6657:>8:<<4445655553442444.50004/0031
@EOUL1:00104:00448
GAAGACCCTATCCGCCGCAATCCGCAAAGCCGAAATCAGCGGAGCCTCATTGCTGTATCGAATCAGCACAGGCATCCTCATGCGGAGTATACGCGACTCATTCACAGCAGACGAACACGCGGCCAACGACTGGCAAGTGTACTACCAGAGCCGTTCATTGGCATAGT
+
CC@BBBHACBBB@CC@CCB>:74888808?<>@A<AAACCC@<<;@BB@C>A@@@;<<BBC@AAACCCCCC@;:;B?B@@@@@@;@@B@@@CC<<<CCCBB=@>>>9;;???@@?@>>@::6:?B>D???BC?CBDFCCC@0111,0:7::9,,+,77+,+,,...0
@EOUL1:00131:00423
GAAGACCCTATAGAGAGGGTTGGGGTGACAATAAA
+
CD?A999-4;9@ABDDDD@C@CCC7@<=9;;8???
@EOUL1:00100:00488
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
EB?BBBC?BBBBBAA:::6:@??B?@777==:==>>>A5@888=79999AADCDBDBCH>CAA=<<CC>BAA=@CC9CCCCC>@@@@
@EOUL1:00121:00450
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGGTGACAATAAA
+
?:9@>?>:999;9CCCCC::8>9=90//:B>>C@>AAA6@?@C?BBC>B>944424:;;6911////75668,////*4439A?BBBB
@EOUL1:00108:00538
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAGA
+
CD@DBBC@CCBBBBBBBB?C?BAA8AABBC?BBB@
@EOUL1:00102:00588
GAAGACCCTATCAGACCAGGACTACTTTATTTGGTCGTTCCTTGAGACTTGCGCCTTCGAGCATCACCCAGTGAGTGAGGAGCGGCGCGCTGGAGGGACCACGTTGGGGTGACAAGAGA
+
CC@CAAA<@CAABB996<F=B@B@@==<?AA<B@AAAD=@=@?BBBBBB?AAAA>@>@C<<<CCCCBB>BAAAA@::99-.//06:;A<=9C>>==78838B@@>AAA5@AA==;=A@@
@EOUL1:00107:00586
GAAGACCCTATAGAGCTTACTCTTATATATTAATAAACTATTGAAAGTGTATTAATTTAGAGGAGTTTCGTTGGG
+
DB?BCCC@CC<<9>>884,0-,,*,,,71101016>:88:3;;<<8;11/66,1,1>588//,////+///,/88
@EOUL1:00148:00558
GAAGACCCTATAGACCAGCGGTACTGGTTTTGGCAGACATTATATTTCACACATTGTAATGACAGCCTTGTAAACCGATAAGCACAGGGGAGTGTCACTAGTTGAGTATATTGAGTATATCGATGTTTGTCAAAAGCCAACGCAAAAAGAAAGGGGAGAAAAGAATTGAAAATAACAGTAACTCCACATCTAGCAAGAGATGATCTGCTCCATTTCTATTTGTCGTTGTTGGGGTGACAAGAAA
+
CCAFCBB=BBCBBBCABBCC@AAAAC?BCC9C?AAACCCC796<<;6:;<BCBB?BBC?BBA>>>A?A?AABA>C@CBDEBBBBCCCCC9BBBBCBBB???E@CB??=BBC=BA@@??>>448888808?9999/9@;@=@AA@@@@/88808??2?7777.7><9<?>@@8<A=??<<A>CBC@@?@@B>878737:7,,,,,,.63337;679166?98??<=><7737>@6AA@CC:==BA
@EOUL1:00120:00603
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATATAGAGATGTTTCGGTTGGGGTGAACAA
+
FDBD><GCCBBGCA@788/88837176<<?B@C@CFDC9AAAA>ABB@BCCC?C>@-111//9449:<>;=<:<,1//*//46699
@EOUL1:00118:00658
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAAA
+
CCDDCCC?CCDBCBB???<@@BEE8CCCBB?BCDE
@EOUL1:00140:00746
GAAGACCCTATAGAGCTTTATATTAATTATTTTAATTTTATGTATATAATTTTTGATTATAATATTTATTATTTTGTTGGGGTGACAAGAAA
+
DC?CCCCACBBCBABBBB>888C?C@B>CACC5:5;BB2<<>AAA@B=>@@@A1?AA?BCBAB@@C<@<;;::6.84233508=>AB@BCCC
@EOUL1:00128:00755
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAAC
+
CCBCCCC@CCBBBCCBA?A?ABABB?BB>C@>>=@C@C@??>?;;>?@=C@888:?BCCBB>>B>>99@@DD?DCCB=?>:@CC@?>@B@CCCCBCC8:9<CBCCNABB?AC@D@B>AB@EC@@@BBBB:??5737>@>>;7,,+**)*
@EOUL1:00105:00847
GAAGACCCTATAGAGAGGGTTGGGGTGACAATAAG
+
CC?BBBC?CACACBBCCC?C@CAA7BB<<7+,,*+
@EOUL1:00132:00802
GAAGACCCTATAGAAGGTGGGATGACAAGAA
+
A@@C;9919@>9984313.63...344,...
@EOUL1:00106:00872
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGT
+
CCADCCC@DBBBBCC@B?A?AB??@6::6;@BB@A?@A@??>?>?@@?<>>>A@A><ABBB>@@@B?BDBCJA=<=BBEH@CCCBC@CC@BBBBBAAAB<@@=@>>99909>9?=@=;;9;CAABCCCC>@@6;6:>>>>@@//,//,/8:8:>>A@@>@?==<;4/../,.77./
@EOUL1:00129:00932
GAAGACCCTATAGTGTTGAATTTTCTTAAATCATATTTATAAGGTGCATAGTTACCATGCCAGGCTACAACATCAAATGGGGAATGTTATTGATTATATTCAAAAAGCTCACCTGAAAATTTGTTAAAAATTGTCCATGAACTTTGATCATCTT
+
DCBCBBBBBBBCC@@@@CCCDCC9??;>>8==>?888/8>>?BCCA;<BCACAAA=A=<<ACDBCDDDB@@AABB?:@CCC9C=@@?=?B5::;@CCBA;?ACCC2@@@BBA>A<<;;1:;:::7::::.:;>BB>:88884433./+0+/-10
@EOUL1:00135:00912
GAAGACCCAGAATAGCAAGTACAATCACAGTATCACCTCCCCAAAATCTCCACATGCTCAGATGCACAAGACGAAGTGGATTGCCTGAATCAGTACTACCAAGTGGCGGTTGGGGTTGACAAGAGA
+
CC=@A@95888544:9:<://77,//66779877331/68804440...33336@?>?@;669:4499244>::9:44249:4464399778>977611013100211012233*-,---345<<;
@EOUL1:00139:00952
GAAGACCCTATAGACATTGCGCGACTGAATGGGATGCCGTGGATGGTGATGAGGAAGTGGATGGAGGATGCGCCTGAGAGGATGCGCGAGTGGGAGTTGGCGAAGAGGTGTTTTGCGGACGGGTTGATATGGAGCGGGTTGAGTGCGGCGCGTGATGCGGACATTGAGAGTGTTCAGGTAGCGAAGTTGCAGGCGGACATATTCCATAAGACGGCGGCGAAGATGTCGAGGGATGAGTGGGGTGGACAA
+
99594496=88=>?>>77388::10..8,//;;B=>A?AAA@BBEABB;<<BB?B@BBD?BAA<;;789?A<;8<CCBBEEA<<<@<9;<;9@>B<<;>EABC>=<C?BBBCC9CCC>@9990959DD88<7;@>@C>D?BAAABAA5:/81,,,,,,002933200004>>@<??C>@A:98@<>><?@>@?8838==888;@9<=?>?>BB>;84889;?>??>9998808887?@BBC8::8:@::
@EOUL1:00147:00984
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAAA
+
ECB@999-898>?A@@AA<A?BGE7BBBBC8:;DD
@EOUL1:00166:00015
GAAGACCCGTACGGGCATCGACGAGCGATGCCTGAATCATAAGAAATCACCTGCAAGATGCTTTTCAACTCCGCATCCTTTCTAATTTTTCTGCCAGTCGTTTTTTGTGCTGTACTGGTTTGTGTGTGGCCGGAATGTCCGGAGTCAGAACTGTTGCTGCTGGCCTCAAGTTATGTCTTCTATGGTTGGGGTGACAAGAGA
+
AB=A9>?:;99@@B<AAAGCCDC;<<CA78;:<77:;;?;>:111,16<?@A@??<<845=A==4<><:33200333/00+035;====.=<65233++++++++(++++2.3377;:<B<A?<==>A=9=><96698857=@A80/;:,0...,.;?688D<?9==A<979.---100-----,-0333+0348::555=
@EOUL1:00175:00033
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAAA
+
:::BDCB>BB<;<A@ABB?B?ABB8BCC?631777
@EOUL1:00190:00037
GAAGACCTAAAATCATATCGTTTACGGGGAAATGAACATCCGCTTTCAGCTCGGCGAAAAAATTCAGGACTAAAACATCTTTGAAAGACGCTGTTCCAGAATTT
+
495499;>=AA6@>>>94449;5==???7BB@B@?::444249:::=99944824444444(424882444444,444444-44:-439:9976039731011,
@EOUL1:00162:00098
GAAGACCCTATAAGCTTAACTGTTACTATAGGTAGAAGCACAGGAGGAAGAGGAGTAGTAAAAAGTGGAGACTGTAAACAGAAAGATGAAAAACCAAGACTCAGAGTCAA
+
<<:>BBB@A>>>>11199988@CBCBCC<<;@BBBB?B???AB?ACAD?A?<9<<::;:::::0<>@=A???@@@A@<DCBBB>BB?01111+198+,,,**,,,...63
@EOUL1:00187:00097
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
BA?BBBC?CBBBA;=78;6;?@CC@DC@???997919948?AB8;<DA@>?B992;8>DAAAAAA@BB?B<<:;BC8BBBBA<9999
@EOUL1:00187:00148
GAAGACCCTATAGAGCGTTGACTCT
+
AA?ECEC?CDB:;11..,*,,,///
@EOUL1:00175:00195
GAAGACCCTACGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCGACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAAAGAAA
+
@A@CBCB?BCB>@;;<69>?BA@B?BB?@:947===;@A<ACCBBDCBB=BBB?BBDCCADD>><<CCDCDCCCGC@AAAA@C=@;;<?>>@<99<<<>BBCCDDECBCCCBC@DC@B;;9;=@=@AA@@;A:<10,0<:>=@CDCNDE?B@B@<<<BCCC>A@C>?>;8848>>;63666,1171//+--22
@EOUL1:00173:00230
GAAGACCCTATAGAGCTTTACTTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
CBA@999-4998498488/0/1,00,.449B@@A?8<==.999><7>A:>>>@996<?GB?AAAAAAED>@@@5999,48899>?@BB
@EOUL1:00196:00315
GAAGACCCTATGGAGCTTACTCTTATATATAATAAAACTATGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAA
+
992449404?6<:<;//,///4><@98333373111,/8/1///+/499?C@C?DD=?@@@>?@@:>?>6999/99@@B?999
@EOUL1:00187:00394
GAAGAACCCTATAGATCCCAATGCTCCTTTCAAGTAAGCCTGTCCCAGTTCCAAAGCTATATAATCCTCCTGCGGTTTGAAGTCCTGCAGATGCTATGTAAAGACTTCTGTTAGTGTACCAATTTGTAATGGTACTATTATCCGTCCAAAGTATTTGTTGTTCCAGTTGTTGTCAACGAATTAAA
+
,-*,,*,0000/44444;-42494442//+/,*,,,*,9244444-4442424:-44==??AD>?=<99244442//+/87///,.664473373333333,1...5.,,000,/.66647144-3777..,.44...33355566011,11117997967217711016777110611010111
@EOUL1:00160:00411
GAAGACCCTATGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAAGAGA
+
;9@A9990944;?AB;;;<B=@AD@BBD@@@B:<;?BBAC?CCBBB<BA??????@@BB?A?==7747<>>::8<=>@;000999;;AAAC@AA>??=?AAA=@8838<AACCCDCCCCCCC?D=;;59@@@A@@ACCDB@A@499999*9@=@AAC?BBB8BBBBB?@@@A
@EOUL1:00163:00415
GAAGCTAGACCGCCTAGGGCCTCACGAAACATGTTCTCTGCACTTGCGATCTTGCCCGAACAGGTGCAACTCATCAATAAGGGTGCCTACAGACTTCTTACCGCCCACCGTGTTCTGGTCAGCAGCTAGAACCTTTAGCGTCGCATTGCTTTCTCTGTGCGTTATGGTCTGATGTGCGTCTGAACCTGCATCAATAC
+
ABADCAAA=A699@BBBCBC@HBA@ABB8;:B?@=???88766977?@=777:776,110165244:;?<=<@@<<:=A>AC?C@@>?===<<::8;;9443444-4;788::4804/39>?@@???>??<?=?@:>>><<<@?==9;44417>>@@AAAA@>;//,///-12--006333=:7948...242,433
@EOUL1:00156:00470
GAAGACCCTATAGAGCTTACTGTATATATAATG
+
AA?ACCDBBCBBA=??>4../..4..3911011
@EOUL1:00199:00524
GAAGACCCTATAGTGGTACGACAACCGCAAAGCCAAGGCCGAGCGGGAGGCATGCAGGCCAACTTTGCCCAAGAGGTGGACCGCGACTATTTGGCTGCGATCAGCAACACAACGATACCGCGCGAGTGGATACAAGCCTCGGTGGATGCACACCTTGACTGGGACGACTGGGAGGCACCTAACGTGTGGATGGCTGGCCTTGACGTTGCAGATGGCGGGGTTGACCGCAATGGTCTGGCACTCAGGCAGTCTGTGATCT
+
>A?B999-9:;<<BB>ABCAAACFG?BBCDDED@A?A8;9;BBBCC@BA@BBAAA9<8<?B@CCC@CEC;<:<BC?BCACB?B<<<B@@@B<@<@>>99<>>A@AAABGBBAC<<<;;CABABBDCBB?A??@B>>=?>894:=8?AAAAB>>=:::11116-7?;43033+30/0-33-+*+00++-012688<@455.422---,-1117::43433+388;;867730-22--.,.3315=;<<661100000403
@EOUL1:00170:00594
GACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
ADEKDEAACBA@BCC>CGCBBCA@88;;7;ACCEG9EAA>?99919A@@AA=@@CC<@@A<<;AA<ABB>BCC9CDBAAA9999
@EOUL1:00175:00554
GAAGACCCTCAATCATGGAGAGTATGTAGATTGGACGATTTCCGTTGTCTGAAG
+
BECAAABA94/,/////,//4444443333424244444:-4233139334423
@EOUL1:00177:00635
GAAGACCCTATAAGCTTAACTGTTACTATAGGTAGAAGCACAGGAGGAAGAGGAGTAGTAAAAAGTGGAGACTGTAAACAGAAAAGATGAAAAACCAAG
+
;95<??=9@ABC?BCC;<;<CBB?CCBBBAC>@B@BAB7::::5:C?BACB@>A<;8430000+4;;<@AA@@@<<:4>>9888.8A@00000+0,.,.
@EOUL1:00163:00730
GAAGACCCTATAGAGAGAGTTGGGGTGACAAGAAA
+
ELBCBCC@CCDCCA@??@<;6:BA8CCAAB@CBBB
@EOUL1:00185:00765
GAAGACCCTATGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAATAAA
+
9951111+1//66/9;B@<<99CB>CCBBABB=AA=@99858@???8=@@???BB=;BC<>>>:0/0:8766:-//4-04;998=;655555555422-++*+++*+4;@CCC;<<@999@5958848433313444;@;>?=@BCCB-BB@888?>?CC>C:::A;@>@@
@EOUL1:00174:00834
GAAGACCCTGAAAACAACTGGTGTGGTCCTGTGGTTTGTGACTACCCGTACTTGGCAGCGCTGGAGGACGACTGGCTGAACGCCTTCGATGTGTGCATTGATATGTACACAGAGCAGTCCAGCCGCTCCGTGTACGCTGCTTTAGCCAGGGCTATCCTAACCAGGCGCACCAGCCGCGTCTTGTCCTTTCATGCTGCTGTCAACGGAGACAACGCCACGTCGGTGTGGAATTTCGTGGATCCAG
+
ACBDBBB>A@@GG7BEBHCECCC;:8;C>AABB@CC8::B@AACCC?CD<<<?BBCCCB>ABB@CCGABBBA::7:A@@?BAA?B=@:::??BABB@@=@=<<=?8:=CCCBCBBBCCC8::A>;9;@@CBBDAAABAA?@@<???<:::6;B@BC@??;?7::>BC:<;?DBF@A@@0//,//,+,1115,,/6//330364//,.74>BBBBCBCDDDBB@B;;:5747?.7:??>A@A?AA
@EOUL1:00183:00834
GAAGACCCTATGGAGCTTTAATTAACTATTCCAAAAGTTAAACAACTCAACACAAAGGGATAAAACATAACTTAACATGGACTAGCAATTTCGGTTGGGGTGACAATAAA
+
CB=DBBC@CCBB?ABCCC@CAC?B?ABBB@B@BBB6>><?@<??;7>=@7<<CCDCED>CCCCC:CCCCDEG?B@DDDC>@ACCCBC@BC>@=;=5888/9888A>88;8
@EOUL1:00188:00866
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
AACCCCC@CCCB?B@A@B=BCCBBACB<;;<=?;?@CC9CCCB?BB?:A>==?C>ACCC@D;;9@9990999<@?@49889924444
@EOUL1:00173:00978
GAAGACCCACGTTGCACCTAGGTAAGAGGCTACACTTGACATGGGTGCGAAGCTAAAACATTGATCATCGGGTGACGCTGCTAGTGCAGACATTGTGACGGCAGCAGCAGCGAAGAGGAGGTGTGTTGATGATTGAACAACTGAAGAAGTTGTAGTTTTTTTCATAGTTTGTAATTTGGGGTGACAATAAA
+
BB?ADEC>BB996;@@?=@CC@CC@BBB@CBAAAA?;@ABB>>?<??A@B?BABA@?5?:94:CCA??AAA<@@@@@@CCC@?A>=?CCCCBC?BCACBB?BB@@=AAAA@@@>@@@>?@?AA=:<;<<<<<??AA=@@>ADBB?BCACC@CCBACKCCCC(CAAACCG@BCC@CDABCC9BBABB@DCCF
@EOUL1:00197:00998
GAAGACCCTATGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCAACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAATAAA
+
IB>B@?@<@@;<811/6864>>AA@ABBCA>=?A@@<??<@BBBBBCBC=BBB@BDB::::G?CBBBBBBCC@@@C>CBBCBA?@AB?@BEABBC@AA@CCCAA@AA?BABAC?AA@CCC>A=?>@B?B?775:@@=AABD@CC>CCCC>C?B@BBBABAC=BBB@??;?@;::7??4444-4:=@A=BBCE
@EOUL1:00190:01050
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCCTGACATAATACATTTGTTGAGAGGGAAGAAGCAAA
+
CC@CCCCACB?BBE@CC>B?BB88<CCCCC@BC==;?@C?C?CBBAC>??BB@??:>7838>>44838818A@>@@<<<;===>4<<BAB@9969<8<BBBFFEC@B@BBBCAAAAA<ABBBC@CCCCC@CBBB>B@C@C@@@BBBEEA?B>BAC?CCCAAA8?<;;8<877?=8788BAABCCCDAABEE?@?=A@?5:77.4442448@<@@@?A?@;;487,4;=78;?ACCDAAA:??:;?:88@;@<??AA@AA9
@EOUL1:00202:00018
GAAGACCCTATAGAAGAATATGAAGACACCACAGTTACTCAGTGGAAAACAGCCGGCTATGATATGTCAAAGTTGGGGTGACAAGAAA
+
BCACCCNCDC<;<C?BB=>@BBB@DCCDE@CB:;;?ACCB@@;;7;B@2?=7747379ABCBBBDBBBCD?CB=A@@6@A@@@=9999
@EOUL1:00210:00084
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGGCGCTGTCCCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGG
+
AE@ACCC=B@>99919A@<BB@B@CACCCC;<<@CBB<<=<<AAC>B<<9@8980858?=811-01..6659@@B@599198548::>A;>8858=>?:A@?=6770,0:00,/00:=;<@4789=76/,,5+***,+,+
@EOUL1:00220:00057
GAAGACCCTATAGTGTTGCAGGCGGTAGTACCACCCCCGACGGTAACTCCGATGGGAACGGGAAGTGGGCGGACGTGGTAAGACCAACTGTACTGGTAAAAATAGTGCTCAAAGTGGCACTGTCCCAGCAGACCATAAGACCACCATCGGAATCGACAGGGTTGGTGCTGGCAGGAAGAGACGAAGTAACAGGAACGGAAATTGAAGGAACGGAAAAACAGGAAGAAACGGGTGCAGTTGGGGTGACAAGAGA
+
:=BHHCC@CBBBBCBC@BAAB?BC@BBBCCC?BACDD2<:;C@BB?@@@GCCBBCAC>A@@:@:==>>:;;,00...577:7<<;=>A:97885;96=@@@077<<<<@BAA=A@@>@?@@A?::68:8///;;;;@>@@@@CD9<=AA;=9<<<;==@A9=9=>?===>=;/0.0;;;;;=>=====BDHD@C>BCBDC@C?BB8;7;<>>48888,8>><>:77757>?97::4492444,4999:24444
@EOUL1:00221:00147
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
9?;9944-444:;@9B@C<<:CCC><;:@?@>A7;:>?5@999;88904>@99959=@@:AB@99999199958>>4>BBBB=@@;;
@EOUL1:00238:00144
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAAATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCCTTGACATAATACATTTGTTGAAGAGGGA
+
CA?ACCB;;;@AAB@BIBC@;<;?@?@?A@@@@@B?A@C?B?BB???5:47777;;??@@>A@?=;?244-49:;=?>?;;/00;6<@B99989@BB;=<@AA99909<>@@>??88808888;?A@ABA>B;;908595;@??@>?>B7374736;>>>=??9?;?A>A??//,////088<>=;;;6736610.../,...*...3-00023468888?4224,/,/44;?AAC??@B@:99,//,/777+*
@EOUL1:00244:00132
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
9959444-4::C@BBBDD?BBBCC?A;;;;8<?=77777+9@>=7<>>084999B242:@694499444-44454??6>BBA==A@@@
@EOUL1:00202:00171
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAAATGTGACAGTCTGACAAGTGGCACAAAGGGGTCTTGTGTTCTCTATGAGATC
+
4?<ABCB>CBBA@@=@B>@>BBBCBBC?AA6;;;;=@@DBC@E;::<,0,/:<<;>;<7749==?BA>AA<@???AAAAA>A99:>CCCCCC<<59969<<<CCLE?CB<;;A@::;B6;<331;>>11/,////+/,,*//../....33;97;;577669726611013314727///0:>>?A>>5773...446111,17729@B5:7<;8;83034444--+
@EOUL1:00221:00182
GAACGACCTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTAGTTAAAAAAGTTGCGTTGGGGTGACAATAAA
+
**)***,*,06:?EC?BB@CDBCECDDCCC@CC@CCA>==?:8:6;ABBCCCDD>BB?B?B=778:>BBABA=???<<@===AABBAAA<@A>@?;<<;;9>8838=@@AA>>=IFBBBB69599>@@?11,0<<9942444<=AABB-:999BBB@BBB8@>@??777@@
@EOUL1:00201:00203
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAGA
+
CC@HDBB<ABA999>833,8<??>8@>76:=@@>?
@EOUL1:00229:00233
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
,,*,8:B;?CBABCBB@A6;;;;B788AA876:=@>>B6<<:>>BCC?CBBAAB=A=AD@AAACEA@@<@;;7;@@/;;;9A69:BB
@EOUL1:00217:00336
GAAGACCCTATGCATCTTTACTGTACTCTTGTCTCTTTGCGGCCTAAAGGCTTTGCGTATCAGTTACCAACTGCGGTGCCGCAAGCCACTCTCTACCTTGAGCGGTTGGGGTTGACAAGAAA
+
EJDDBBB?BAAACBECBC?AAABBCCCCC@BC:8;CC?BBB?C@DCC@B?@??<@DDCCCCC::8:A>A@??<??<<@@>BBB?BE?AB@@@AAAA>@>@?;;7995677+/,-1434.111
@EOUL1:00231:00310
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTGTTCCTGAAGCAACGTCATAAACAGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAATAGA
+
CC@CCBDACCA@BBC@B@B@BBBBB@CC=B?BC@CC@C?B@?A@@B?C=?@BB@B@B=@@BCB?BB=?ABDC@BABB@CDBCC@A?A@B?BBBBCCCCC@CC@AAABBCC@B@BCB@BBB?;<;AAAAC<@@BD@CCA@@@A??>::7:ACCCC=@;:6<@@@@AAAA@@<ABB6BDBA@<@??=
@EOUL1:00221:00387
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAA
+
BEBCCCB?A@BDCCA9998:CCCC@C;:;BB?B?CCEE7<<<D?BCC?DCCC<<9<CBBACA@@BB@B=BBCBCCC9CCCBC?;89
@EOUL1:00207:00612
GAAGACCCTATAGAGCATTACTCTATATATTAATAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGATTGGGGTGACAAGAAA
+
HDAFBBC@CBB@>;<00.,,,,...3367737570<6=?10,.6656444442:=;@@>::/11685:111;:;:6@??BA?BBAA
@EOUL1:00217:00667
GAAGACCCTATAGACGACGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATGAA
+
CDBEDDD>ABB@>;4///////,/858;;::CC@B<??@C@CB??>B@DC?A>BBBCA@@@>>?7;8<C?CCCC@EDACCAAA@BB?BCDBBBBAA>AB?BBBB>A<>?<@@C?BB@CCCC;<<CC>BC?B?BBCBAAACB@@@;B;<???;83334;==<==A@@@<@AA6BAA;;56///
@EOUL1:00235:00674
GAAGACCCAGAATAGTGCAAGACAATCAACAGTCACACCTCCCAAAATCTCCACATG
+
EDFFCCC>@@>6///////,/6/4237999993333310379,111,1111311000
@EOUL1:00245:00720
GAAGACCCAAACAAAAGCTGTAGATGAGGCAAAAAAACTTTCTTGAGGAGGAGTTGTCTAGCGGTAACTAAGCCAGTTAACGACATCAAGGACGCAGCAAAGAATGCTGGTAATT
+
9969AAA<AB:@@@@6>@BA@@?@@?6631111111&111,11766121100002334555172-/./;;9@@<BBB@D?BBB;;;=?;>5::@???@??6::56==77*,*)*)
@EOUL1:00224:00866
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGCGGCGA
+
DA?BCCC@CBA@@@<>@A28;6;@B>BCCCB@@=@@ACCBBCC<<:BBCC@@@@@=@B?A==;<00,77010/643:;84664..,..136<@@@@@@??;883300440335564233...4333.
@EOUL1:00234:00870
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATAA
+
BB?BCCC@CA@@BBB@B@C@CA@@=677/8<?>;@A@D?AA@C@CCBC>FCBC@B@BBB<A??:?B57<@B;:=::9;=@?BBCBB??@<=99;;<;BCACC@DCAA@B8;;8;>?>BB=@@AB??000,1<,0,////:@ABB@@A====B@A;@<<8;==;<<?=;;;8;;;09;;994400
@EOUL1:00218:00961
GAAGACCCTATAGTGGTTTATGTTATGTGTAAATATCGTTACACTTCTGATGGTTTCTTAGCTTTGCCTGTTTCTGGGTTGGGTAGAGAATATATCGTTGCTACATATAACGACATGGGTTGGGGTGACAAGAGA
+
BC@C@@@>BBCB?==@BC@BA>?=@CCB@@@@<?@?@??<??BBB?BBBCCC@CC>BB?BBBCC>888:@@C>CBBC>C?A<;A:::BC?CCC@??@@=@@@BA488AA>A<<=====;<::::4:===<:;<<<
@EOUL1:00236:00999
GAAGACCCTGACAGGCTGAGGTTCGAGTTTACACTCAGTTTATACTTGGTTTACACTTTGTTACATTTAGTTTATATTCGGTTTACACTATGTTTTCATTTTTATTGCAACGTATTTGCTCTTAGCCTCACCAAGATTCAAACAGATTGAAATTCATTGAATCCATCCACTGACAAGGTTCTTCACGTCGCTCGCGTGTCACAGGTTATTGTTGGGGTGACAATAAA
+
;=C<<<DBCDCDEC@CCCDCAB?@A>8;:6;ABCDECCCB?AABBA@A6::5:?@??A<483::::D@EBCCB?<??;;<9;=:=:::CCCCAAA5?@ACCC2?97800,.7:7;<9=@@@@>A<<:???;:=::442444-4449823840323;;89;=4838==:=?<<<@@=?>A699>@????889??;;//.386:A@6;8:B@DB@ABB:AB??@>9999
@EOUL1:00238:00963
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATAA
+
;:8<BBBBBCBBBBC=>=?@BBBBB@EC?C@CB?AA:?=?B>@>CAAA>B@CC@C?BBBCCC@>?A?8:8<C>CCCCCDC>AB;;;>AA>BBBCCBBBCBCC@BB:;;A<@@<AAC?BBADDBAAACCEABB=?=?@@AB?????==:=;;8888:=;;;:994444433379<4<@>;;2400
@EOUL1:00203:01027
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCACCGCCATATTGCATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCC
+
FDB@?@D>BBCCCCBCCC@BA=A@B@AA?@BCC@C?@@E>888//-1><??@CCCC@CC@B>=>=000<<8?AAAA>BB@BD?CCC@@??BBB?B?AC>B8:;?<8838?==<A?@@CA@@688>844:////8;8:9;;9@@@=@:=;=AAA<?A>A?<<9;=>A@@@@<<<DD@C@C?CCB@BBABBBCACL@C<<;9;A@>CCAA@BCC@C?@@>9888;9CB?ABB=???774
@EOUL1:00247:01027
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATAAA
+
CB@ABBC>BCBBBCCAB>A?AB88;@CCCC998@DK:=;@A>A@CCCC<B?BB?C@AAACCAA?@>>>B?BC@CECBAEB=BB@@@@@@=@BBBCCC@A?ACACCCCCG>@@<@?A8<<@BBBCC@??D>??477.7=<=?@<:9BCC@A@A=<;;>>=;?@???AAAA@A>@@@4;8;4424033
@EOUL1:00204:01058
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
A?<?9996<BBCBAA@B@B@BABBC?AA>B@CE@BB?B?AB=?@A99:4:>AA?B?CCBBB?@;<?@BC@CC@CAAABCD>BBAAA@@@=@AAABBACB@BBACCC@@@<?A<B?CBBB?BCCCCC>>;679,/,0000;@>>;7442499334;A??<?@AA@@?A???>AAA5A@@??388==
@EOUL1:00228:01068
GAAGACCCCAAAATCATATCGTTTACGGGGAAATGAACATCCACTTTCAGCTCGGCGAAAAAATTCAGGACTAAAACATCTTTTGAAAGACGCTGTTCCAGATTATCTAGTGAGATAAGGGTATGTTTTATTTCATTTCGAATTACTCAACGGATC
+
AACA?@@@/999/9@?>@@@@@@9@@@AA8BC?CCDBCBBBACDCE>AABAD@@B@999999)9<@>>=@CAAAA6A@@9999/9BB8<<CCB???7;7<;A@@CCF;<;?CCAAA?;==7=;86666+/,563336655;,/,////;244////
@EOUL1:00252:00019
GAAGACCCTATAGCATTCTTGAAAGAGCAGGGTATAGATGTTGACGACGTGGTGTTGAACTTCACATCTTTCCCTATGATAACTCTCGCCGACGGGGTGTTCTCTTGCCCTGAAGGCAAATTAGGCAGCGAGTTTGAAGTAATCATTATCAACCGGCGCAAACAATTTTTGTTTAAAGCTGATCTGGGTCGTGATAAGGATCCTATATTGATGTACTCCAACGATAAAATTCCACGATGCAGACGGTCTACTTATGCAGACCACATCGCTGAG
+
DCACCBC@CCCEBB@@;=>=@?ABB;:<BCE>DAB?AAAAB;;;BC?<?BB>:::???8?A<==?<888D:>?<?=<=??@>@CCCC8858@??AA6@AABC???888B<A;<9<;88:;;;AA?AA@@@@@AA5889;>;<=?<=:=DAA>@?@>@<AA=:88/4444,4:449<;=859A@<<<@:7611/..834433427488<=@<A7362--,-22-+.677+/--,-++..34?AA@A=@<<=9=;=A?986//,/789?;;70,,
@EOUL1:00252:00030
GAAGACCCTATAGAGGTTGGGGTGACAAGAAA
+
DEDCABCACBABB999:@BED7ABCCB?A@CB
@EOUL1:00275:00004
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACTGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGATGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCTGACATAATACATTTGTTGAGAAGGGAAGAAGCAAA
+
;<9>GCC@CCCBAA>@CA=;>=00100,.<<8377363;:=<77:??:=<?:::77<?7737>::99///,/88//-4.4/44498@BCBCCCA?AB?ABCAAACNACCAAAAAAABEACBB@A=@AABB@CECB:B5;5;@>>AB@@?@@?5867477=777-677=:<:8:A;A??9;;AA000000/,//99;::7@CCC7A?<:6333/00007=:43..,.11-016061666628633://,/0,.4420-1162
@EOUL1:00257:00112
GAAGACCCTACGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCAACATTTAATTCGTACAAAGATGCGGATGTCAGTTGGGGTGACAATAAA
+
DBD<<<C>AABA99<<@CBBCBCB@A@A@CB>????BBB=BBBB@@@BC=BBC@CBC;:8;B<ABBCBBBBCCBBB=BB;;;B=;;;@@CCABAC<=@ACBCBBB;;8;ABBC?BB?BCD?B?B?BBBBBBB@CCBAB@<<>:748@AA=BAEABBB@AAECEC;;1;,110...33799957661104999
@EOUL1:00292:00105
GAAGACCCCATCACCGGCGACGAGGCTAAACACGCCTGCGGGAAGTGAAAACCAGCTTTCTTTGAGAGAGCTGCAGCAGATCAGGGAGATGATGGAAACGTTGAGAGCAAGAACGCCTGATGAAGGAGAGAAGTCGCAGATGGTGGAAGCACAACCCGTGGTGCCTCCTAGAGATAGACTCGCACCCTTGTCCTTTCTCACTTTGAGCACGGTTGAAGGAATAGTTGGGGTGACAAGAAA
+
CCBBBAAB9BBBCC?C@CCBBBBB@AABB?BCDCC@DBBCBACBA@;;;;49?@ABBC?DCE?BBBB;:;AABBCCC?@@CCBBB>AABAA@AC>@@<888>A@@A@BA<>>?>??>CBA9::494:??;A<??BBBCA???>@A9=9;>>78488/8>=:?AA=AB=??CCA?????AC@??AAA;B?BBC?BB>?@C???C7==@AAAC@F?BB8:8:>A@AB?ABB5@==>?<?:;9
@EOUL1:00262:00175
GAAGACCCTATGGAATGGTATCTACTTACATAGATAACTTTGCCATTGACAAATAATAGAGTCCAGGTTTGTTGGTGGTT
+
::AABBE>C@@>;?<99599449:8>59444////1,///+/;888244444-4989449::1:;8588+//,/,/8,/,
@EOUL1:00268:00206
GAAGACCCTATAGAGCTTTAATTAACTATTCCAAAAGTTAAACAACTCAACACAAAGGGATAAAACATAACTTAACATGGACTAGCAATTTCGGTTGAGGTGACAAGAAA
+
CA?CGCDADECCBBBCBB>C?B?A>887701;?@B8:928>/731711;,1//;859=594944,8994=@B5424499;=BBBBDC@CC>BC=>;>99594==@@CCCC
@EOUL1:00272:00217
GAAGACCCTATAGAGCGTTTGACTCTATATATCAATAAACTATCA
+
@@=BAA?>BCBBB=8100.+../////7773333774424....,
@EOUL1:00267:00263
GAAGACCCACGACGTGAAAGGTAAAAGTGAAAGAGGCAAGT
+
94299991////////44-478;;;2::444-33313.3..
@EOUL1:00250:00401
GAAGACCCTATAGTCATCATC
+
@F649@:59@A--,,,,,***
@EOUL1:00261:00434
GAAGACCCACACGTAAGTAAATGAAAGAGGAAGTAATCCACACAAAAGAGCTGAAAATAATGCAAACGCCTTCTAGCTTAATGTGCTGTCGCATCTCTTCTTCGTAATGAGAAGGCATTGGAGGCACGTCAGACGAGTTTGTTGGGGTGACAATAA
+
BE@A==@<@?@AB@@<>>>>:>99919@@=@AADECCC@CCAAACC9AAA:::EAA7:8>@@BBC:;;C@C@BAAABD?A>====ABBC:::700000,0;9;?@@<88:;@<?;>>=;;8::387B==;;;;533399;:<;=>>5<=@<;8;;;
@EOUL1:00273:00429
GAAGACCCTACGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCGACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAAGAGA
+
EG?BBBD>DEDC<<=A=@@BCB;;8;@??BB@CCCBBBB>DCCCB@;;;:<BBCCCCBCADG=B@@@@@BCCBBBB@@>>>;969@>>>BA@<<9<<A=A@CCABBB788>885<@=@CL?B?B@::77==>=@B@<@@:96:::ABCC@B@B@AA99999099@>@@<??761.333344*....0/0003
@EOUL1:00288:00464
GAAGACCCACACGAGACCATCAGCGCATCAAGCGCCAGCTGCTGACACACGCATTCAAGGAGGTTGGTATTGAGTTCATGGACACTATTCGCACTTGATTCCCGT
+
??<>?>=5111////449:::;;??:7772*,,,,2///444;9998843999C@BB@B@94231.,...,.647:>>@@<;97744424944442444244-4,
@EOUL1:00250:00506
GAAGACCCTATAGTTCCCTTATT
+
9:@BCCD@A@>8-))))((((((
@EOUL1:00252:00521
GAAGACCCTATAGTGAAGCAATTGGTTGGGGTGACAATAGA
+
DDADDDDCCCCCBBBB@BBB@BAB?B?ABB8BBBBC@BBBA
@EOUL1:00259:00522
GAAGACCCTGCGAGTGCGGATGGTCGCATAGTTCCAAGCTGCCAAACGACGGGCCA
+
BBABAAB:949449;:4424449<97<>997?==9>?>8666011,111777,0/.
@EOUL1:00289:00560
GAAGACCCTCGATACGCAGGTACTCTTCGACGCCGAGCTTCTTCTTCTCCGCTTCAGCATCGGCATCCGTCTTAGACAAGTCCGGCAAAGGCCTATCAACGCCCCATAAAACCTGGCGGCGTAGTAGTCTGCCGCCCAGTGACCTTTGGATAAGAGGCTTCTTTGGTTTGG
+
88:>CCC?;?=CCEBBAB@>A@999@5117-,,*,0,,429=<?B@BBBCCBB@CBB8::AC@BC:;58?885;>==B:::9=?=???;??B?A???A>?=>AA089A@@5@=??698;??///9///::6@000?6:///30/34285<==<@>>203/000+0/00+04
@EOUL1:00295:00609
GAAGACCCTGGGGAAAGAGTGAGTGAGGGAGGTGAAACGGACTGGGGGGGATGTAGGCTGGCAGAGCTGAGGTGTTTGAGTGAGAGTGGTTGAGGGTGGTTGAGGGTGGTTGAGGGTGCGTGGGTGGAAGGGCAAGATATGATTGTTCATGGCTGGCGGTGGCGCTGGCAGTTTGGTGTGGTTGGG
+
99:C99919999/9BBC@AACCCBB;::688=?7<<;<CADCCBCCCCC(C;;;AB<??A=ABBBBBABBBBBBBC>CAAABB???A=>A@=@<@:7438;::;>:>>7:79444-499:444-49/4244+0426:==A@@@?AA?;::B????>@A=???A::9?=?;?<926333..44,--+
@EOUL1:00271:00662
GAAGACCCTATAGAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAGA
+
BADCBCB@BBBA996:<><A@A@B=BB<A>BBADB@D@CCBC?ABBB>B>BC=>=@BAAACC@BB>:96;A<@BCB=AA=ABACCBCB?BCCCC::;@=AB@CCCCCC>AB>A@C=AA@CCCCB@@@@/88;>;?:::CB?>?744-7;;>=?:;;;344;;;;<@444/444,4:;4838>>>
@EOUL1:00284:00728
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACGACTGGACTTCGACACCTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAAGGGTCTTGT
+
/8=11/8+///44954=:@@BBBCA><480,1..547355?3;6;67011=>668888800-1--,-,-3323<::?<?=<;9@99<99>>=AAAEHACACCC@CCCBAA::::B<CC;;<ACCCCC@@@@@1969<88;BBBBBBBCC=?<??777===.7/887////977=;9711611166667711001111,11,144/00
@EOUL1:00286:00784
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGGAATAACTACTTTCCCGGAAGCTGCGTCATAAACGGAACGATGTGCAGGAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
CC?ABCCBCCCCCBB>A>@@BA@AA@BB>B?BB@BB@@@@@?@?@BCC>C@CC@B?BBBDBBB>?A?BC@DC?CFDJ@BB<AA999;@@=ACDDCBB?B@CC@CBBBCC=AA=A?B?BCBCBBBCB@??/885747776??CCCA?A?988:97;@@@>BAB@@A=<<<:79;90;@B@;74444
@EOUL1:00282:00977
GAAGACCCTATGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAAGAAA
+
CC@CCDEAECBBBCCDC?BB?@:;9;BBB?@A@BB@CDCB?@96111:?@?BDA==;AD@?<<;448=?=>@@====@@A@@@A==<====@@>??<?@@A=@@A=<<@BBCDDDBA;:68,/,1;:<>CCC@ABBAAAABB>AAAAA,@A=;;;A@CCC9BBBBB>AAAB
@EOUL1:00272:01011
GAAGACCCTATAGAGCTTTAATTCATTAATGCAAATAAAAACTCAAACAAGCCTAGAGGCCCTAGCCTACTATCCCTGCATTAAAAAATGTGGTTGGGGATGACAATTAGA
+
CC@CDBC?DCBB@@@@BA<B?B@CCB?B?BBBBB>BCCCC2CCB@@:==89;4444::7:;7;;:;8888=>88<999===<AAAAA-888;8:3444-4:::/.,-,-+*
@EOUL1:00274:01137
GAAGACCCTATAGACGAGGTGGCACA
+
CCCCCCC@CAA=><;1/*000)**,,
@EOUL1:00331:00019
GAAGACCCTATAGTGGTTTATGTTATGTGTAAATATCGTTACACTTCTGATGGTTTCTTAGCTTTGCCTGTTTCTGGGTTGGGTAGAGAATATATCGTTGCTACATATAACGACATGGGTTGGGGTGACAAGAGA
+
CCACCCC?BCBCCBC=@?7;;BB>BC@@@BAA<A@@???@BCCCB???=>9969908?<@AA@@::;7<B??8?877.7>BDECCC@@????????<=>@EDCBB@A??<???AAAAA<A?AAA6><<33249::
@EOUL1:00332:00081
GAAGACCCTAAAACGAAAATGCAAAAACCCAACAGAATGCAGGGGAAGCAACAACCATCCAATAATGGTGCGGAAAAACAAAAGCGAATTTACATGTGGTTCAAGTGGCCACCGCACCCACCCAACACAGTCCACACTGAGTCCTCGTGGACACGACGTCTTGTGGTTGTGGTTGTTGTTGTTGTTGTTGTTGCTGTTGTTGATGTTGTTGATGATGATGTTGGGGTGACAATAA
+
BDACCCC>AAA@9ABAAA7@999>>>/>>9?=?ABB9<<@CCED8B>CCC@BB?B?CCC>A?AC@BB?BC??;?CCC2B>>>6>>11-0>>>?BBBA@?A;==:=DC@H@::89@@?A=<<<7;9;8////85677><==A@@<??AA@A>A===AAACGGAH==<=ACBCDC@BB@CC?BB>::5:@>>@=?:<<B?BC>AACCB@BB8;;@99;;;;@B?BBB8BCFB@7888
@EOUL1:00348:00067
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
DCBDCAA>ABCCBAA=><>@CBBBC@CC=C?BC@>?=@@CC@C@B@@B>B?AA>A@CBBABCA@??>>BEACBCCEDACC>AAA;72,,*,,206=BBB?BBBCCBB>>6;;09<@7;<>@BABAAAAA<>A=?9::CCCBCBBB@@<BACCCA=@@A=@????>===<<8<@>4@C???>AAAA
@EOUL1:00302:00119
GAAGACCCTATGAGACCATCAGAGGTGTCTGGTGTATTAACTAC
+
//,///>,1/;4494424487;;B@B>>71424///.,.,/7//
@EOUL1:00327:00106
GAAGACCCTAACAAAAGCTGTAGATGAGGCAAAAAACTTCTTGAGGAGGAGTTGTCTAGCGGTACTAAGCCAGTTAACGACATCAAGGACGCAGCAAAGAATGCTGGTATTTCATGGGCGTCTATACGCAGAGCCAAGGACACATGTTTATTACCGTCAAAAAAGGCAGTGAAGCTGGGAAATGGACGCTTATTCAAATCAAATAACGATACAAAT
+
4???>?@<@=?BCCC:BAACCBBBBBC>::::744*//,0=:>?@;B@><?A>@@@?????>@@@@A<?A=BBB@DBCBB@@@C@@C@AABB@@@CC<@B?:::?>;=777.7???C>C?@????<<<84444:7:2838=;;;888;5;A8<<994333300(02440003204444-/8566-0..666970111,1677,11011111...+.
@EOUL1:00327:00134
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAGA
+
DCBAAAFCCCCCBA@BCG=@=@CC9CCCB?=>???
@EOUL1:00306:00157
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGGAGTTCG
+
CC@CBBB?BCCBCBBBCC>CBBBC?A?@>=?>B::::94:???@:996;B@@ABEFABB;>800**)*(
@EOUL1:00307:00170
GAAGACCCGCTTTTGAAGATGGTGCAC
+
//,//44-44444,499:;94244//6
@EOUL1:00322:00154
GAAGACCCTATAGTGCAGAGAGAAGTATGTCTCAAACTCCCTGAAGTGCTTGGACATCTTGTTCTGTCCACGATCTTTCCACTTTGGAATGTTTGGGAGGTCAGGTATCATTTTGATGGTGATCTCCCCTCCTCTCATGCTCCTCTGGGCAAGGATGGGTTCAAAGTGCTGGGGCCTCAGGTCCGTTACTGTGTAGAAACTGACTGACCATGGGACACAGCACCTGTGTTGACAAGCCCATGAGGCACAATGTATGTGG
+
CC?C@@@<>><<<<<FGBBCC@@=?BBBBBBBAA?9????9?AB:<=BC@=AAAACCCDFCC@CCC@@<@@>>>@@@<A?BCDD?C?B?B@>?.77.7>477A@;?:::BDDC9B?@@@??A@94///+//,---322:ABA<??88708;9;;<::9,/,..0+..971388,3137760131331169>?>>=744-7>===>@??7:<DD@C<<;?>=77476;;=:?;;825;535=?=;9:///.//;;00/87
@EOUL1:00328:00166
GAAGACCTATAGTCAGCTCATAGAAGCAACTAGCTGCGTAAAAAGGTGGAACGCCACGAT
+
1/5///,///68444884448:1/,///5444444449@BCCC2@;4/,/,//.,,,///
@EOUL1:00301:00333
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCCTGACATAATTACATTTGTTGAGAGGGAAGAAGCAAAG
+
4424:9B>AA=>>>59908<BBCCCCB?BC@737<4737:=:==<7736017;;AC?@?=:?==?<<9<=7=>==8876=;===;8@999911;;11,/888:3333,424?BBBCCID=;;:?@?@@>@?5999=0859<>>@300>>00.5,.,.,17167A>A<?>9300384333387@?B???@==47776...4,...*.384;==;;=5...4410000+3840-.3,.,.536688685>?8:=46534/0000+0
@EOUL1:00337:00320
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAAA
+
??=B999-4::>ACCBCC>B?BBB9CEBBA=????
@EOUL1:00346:00349
GAAGACCCTAGAAGCTTAACTGTTACTATAGGTAGAAACACAGGAGGAAGAGGAGTAGTAAAAAGTGGAGACTGTAAACAGAAAGATGAAAACCAAGAC
+
CD@DCCC>CCCC@CCC@D@BBAC@CBB@@@?=B@@@B@BCBACEAA?@<777:>::::9====/=>><??=?>>744-94;@B>B491///*/,/,///
@EOUL1:00344:00534
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATTAGCCCGCTGCCAAAGTTATCTCTTCACC
+
94:9944-4;:?>>:>21-/8+/,/79;9;9AA>@@99999998440-,,,228::B::/..6..,..44-48:4469A799:3334.,*,,,*
@EOUL1:00308:00562
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
AAAACDD@B>AAAAA???<BBACBBA?@?@C?@>BA@@8AC@99344-9>@BAA?ABHDBCBB<;;@@<@BB@BCC:CBBCB@;;9;
@EOUL1:00338:00554
GAAGACCCTCAAGACCATCGAATGGGCGCTGACGACCAACACAGTCAAGAAGCTCATCGGAGAGGCATCTCTCCACAGGCAAA
+
9429AA=>CCCBEDC<<<A@@=A@@<AA@>>>=111:;54=.831316110166113??567??5:688613-242-/,-222
@EOUL1:00305:00640
GAAGACCCTGAAATCCGAAATCGTTCCAAAATGCTGACGACACGCTCCGCAAGTCGAAAAACAAGGTGTGGGTCACATACCTTCCGAAGACAACTCCATCTTGACGAAGAGAAAACCCCACAAACCAAG
+
CC@CCBC>CCBB?CB@ABB>BCCC@B@BAB6ABBBBABBBCDDCBCB@B?A=@@@CDCCD2CC8:5:@@@B>BCCCBCCC@C@CACC?BBCC?CBC?;>><:7788747???>>.777-7<<;,10171
@EOUL1:00334:00733
GAAGACCCTATAGACAAGGAATGCAATGA
+
CE@DBBB=@=@@<::79>9:,..,,*,,,
@EOUL1:00348:00703
GAAGACCCTCATAACATGCGACAATTGAGAGAAAAAAGGCTGCTTTCTGACACTGCAATCTAGACAACCATGCAAGGGGAGTCCTTGAGGTGCTTTGCATTGTTCCCTCCAGTGCTTCAGATTCCTCAAAGTGCATGCTGCATGTTTTCTAAGCTTCAAGCCAAGCCAAGTAAATCCATGGGGTACAATGTGCCTCGCCGGGTGTGCGGCAGGTGAGGTCTCGTCTGTGGGGTTGGGGTGACAAGAAA
+
@A?A?==:@BBCD@CCBBBBBBB>?>ACBB???777*7;@BCCCC?BCDBBBCCCBB@ECCBBGBD?C@AACCC?BAA9ACCC@C@CBC@CCCCC@DCCC?BB@CC@CC@CBBBBB@BBB@ABA@@@AC>CA@BBCCAAA@<>9999/1<<-22<:<@11;?B=BB@CFCDCC@@@;>9999/9>89499111,1<;::@@<@BBBBB=@@<,1111,11..7??31167726867721111101111
@EOUL1:00307:00851
GAAGACCCTATAGATCTATCAGCAAATAATCTTAATCCATTGTTAGATGAATATTGGCAATTAATAGCGCTTGGCACGAGCCGGAAGATACTTCAAGAC
+
4424999-4999@CBBC>1111/.689999>B7;8<D?CCBCD>@A@@?@@>77063379917566;9ABA@A?A??:44<;88;:......,.44496
@EOUL1:00311:00883
GAAGACCCTATAGATCCTCTTAC
+
77:>AAB><:1888--*,,,**,
@EOUL1:00327:00879
GAAGACCCTATGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGC
+
CDBBBCBABCBB::<DDACC?CBC>BAB???B?CBACB@@=@AAAA<>?=:::0..,..,.1..
@EOUL1:00305:00932
GAAGACCCTATAGACCAGAGCGTAC
+
CCC<=;@<=999911,0-,,,,,,/
@EOUL1:00345:00966
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCCGGTTGG
+
4424>>>1;9??AA<==C@CB?A:<69@>9999<>ACCC<<<<111118??>3337331100,....8859;???>?B>@A;@<@:717;677,1010.4
@EOUL1:00345:01000
GAAGACCCTCAATTCGTTCATCCGCCGCAAGGCACGTTCCGGATTGCCGTGCTTCAGCATCAGATCCAGAAAGATGCGCTGCGCCTCGGGGTCGTCGCGCATGTCGTCGTCAATCAGTCCAGATTGGCCGCGATCAACCGCATGATATTGGGGTGACAAGAAA
+
BA>A@@?<@CD>A>ACC@BCBC@BC>BAB@C@CCCAA=@>A@CCABB@BCCBB@CBBBCBABCCBC=?@BB=??;?@@<;;<<=8:8;;;0;;;9==<<99=@===AAA@@@=A<88438486;5<5623-57./.596:;>://.4594:B4><585<7745
@EOUL1:00321:01087
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTCCCGAGACGCGTACTAACGAACATGT
+
AB=@>?>5994=>994:@@5;;A@@=@B<B?AA<70,.,.0,.444576A<:?<?<??==?;:67B;7<:9A>999?<99699CBB?@@111//=1.3424656////,/8+//443444444442444244///
@EOUL1:00310:01111
GAAGACCCTATAGACAGGTGGCGGTAC
+
DDACBBB>BCBBB@@B>;><,/.,...
@EOUL1:00312:01122
GAAGACCCTGGGAAATGCTGTCGACTACACCAGGTGGCGGAGACGATCATCGGACTCAAGCACTCTACAAGATCCAGGAGATCGGGTTCGATCAGCGCAATGCCGGGGACGGTTGAGCAGATGCTGGAGGAAGGCGGGGTTCACGCTGGTGGATGAGCCGCAAACGGTGGTCACCTACAACGCGCCAATGAAAAGGAGGTGAGGCAATGGTGAAGC
+
//,/4?A:899044-4;;?;99<B;;:???=;;7;C=?>B>313>188?B?>=@@-/,5,,,/.3164;9:94424924>999991499884//////87666,///*...,.,./44443378686,,*,00*,099598435//./7577;11.,,*,,,,*,,*,65689964/78,/11:66;944444,49:9994486//,//,//4:4/
@EOUL1:00320:01109
GAAGACCCTAGGGGAAAGGAGTGGAGTG
+
?@=A@AA?@9899,44-/,//.0,...4
@EOUL1:00329:01114
GAAGACCCAGAGGAGGATGAAGGTGGTTGGGGTGACAAGAAA
+
<9@?11/,////68;?A889=>>66010111,1799924444
@EOUL1:00339:01101
GAAGACCCTTGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGG
+
8877494-42449@=:?=>AA:A>B@C=B@CCHDDDAB<11<?11/1100....6665-/.6,..../8577=;;<?@<@B8731.//,..3,.,
@EOUL1:00325:01160
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
EKBB@@C?CEDB=@@AAB@BBCCD@B>>96137;7777.9@@?;@A@<ABBBCC?B?BB>CBA>>99909>>;>AA7<;>CCBDDGD
@EOUL1:00354:00042
GAAGACCCTATAGATGCTGTATTTCGGGCATTGTGGCAGAGGAGCTACCGGCCACCCTCCATAGTGCCAGAAGCATACCCTTACAGTTCTTTGGGCTTGTCGGCTTAGCATGCATAGCGTCTATGGCGTCCCTTACTGTATTTTGGTAGATTGAAGCCCAACATACAGCAAGTGACGCTTTTCTCGCTCAAGACGCCAAGTTGATATGGCGAGAAGAGCGCCCGTTACAGCACTTTCGGGCGATGGA
+
?@=B@@=:@AABBBBBDCB?9998;A?:ADC?BBC?AAABCACCCCCC@B?B?B@B=;;8;@BBBDBACCC@DBB@;;;6:D@?@DC=??B>CC>CCCC?@@@BA?:::@????@=<<<<<=@=@>000>>6;:@>;444444-47:448;789,///,/468;??989@<@@884:4444-4;;;==@;9;8///587/4344899:5;-/-2,--223353538>=7576100,++00+++3722
@EOUL1:00365:00038
GAAGACCCTATAGAGCTCCCAGGCGTGGCCGTGAATCTGGAATATCTCGACAAGGTCATTGACGAGTCAAATGCGTATCTGG
+
LH?A@@>:?948==?BCBB<::68?774:?AACJACC@=;=>@@@@>44777373660385555;;;444-4044444400/
@EOUL1:00373:00031
GAAGACCCTGAAAAAGGCATGGCACTGAGATGAAGTACATCAAATAGCTTGACAACATAAAAGCTACTATCGCAAGAGATGTAGTATTGGTTTACCCAGATTATTTGCAAGAATTTGAGATATACACTGATGACTCGCCTAAACAACTTGGTGCAGTCATGACTCAGTGTAATAAGCCAATAGCGTTCTTCGGTAGAAAAACCGACAGAAATGCAACATACAGTGTGGCCAACATTGAACGACTGGCCA
+
=BBDBBB>@>>AA>/?><;::B?@AA?><DCAA?ACCBCCCCE?;;<;969??=9;93888.78389888=@?@=@BBCCB??????<?<??88880611601..+.//,,2222,/1;;:@@9:;999@999@<;;A?CBD@CCA<<<;ABB:;::A?BB:<<BBCC@A@=774784747=779?:6?47666400000+13640338<4@@9@==9;4;9>8::7991126//,/:,/...47065:
@EOUL1:00356:00143
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAGCGCTGTTGCTTACATATACAGTTGGGGTGACAAGAGA
+
94>BECF@CDCA@@;7824237788233/39991310131601;:::;+0,.4655>66<><A?AC>BC;<;09A9959919?B@@6995899=@999C@EDACCDC@>-11-1-1,/16=;@BB;;;C<AA8;89>@BBCCAA@>384;;:9959>>;>BA@A@BBB@>4888.:A@DC@<<<B
@EOUL1:00377:00110
GAAGACCCTATATAGAAACAAACCGCGACACTGATGCGTGCCCGAGTCCTCTACAATCTCGGAATGAAAGATCTGGGTCAGAAGTTGCTCGAGACGATCGACAAGAGTCCTTACAGCTTGACTGAAGAGGAGCGCAGAGTTCAATTCGAGAAGATCAAAGAACTTAAGGAACGAAAAGGACTC
+
:EDAAAC>?@B@>9889099919<>?@AAAAAACBCCC><<@:>A===@DCDCCC@C<<<D@C?AABC@CCCC::7/8?999=>?=BAABHDGHDBBA;<<A@?9;<AA>?>B<<<A@<A;991@-111:<000>>7666019595999=>799:BCB<494..,.65,.,44489/4943..
@EOUL1:00392:00117
GAAGACCCTATAGCCTTCGTACTCAAAGACGCCGCGTAAGAGCTGTGTGGTAGAAACATCAACATCGACAATCAAAATCAATGGAGGTTGAGAAAATGTTTGACGCATGAGTGTATACACCGAGTGAATAAGCGCATTAATTCCCTCTAATGACGATAACACTAGGTTCATGTCTGATACCGTATGCTCTCGTTTAGAATTGGGTA
+
BC@CHHH@AACABBD>?>>113?CFGADCDBB?BCCCD::>DCBBCCCCCCCCBC@B@AAC@C<<<CCEE?BA=>>/99>;9959;?A@CDDDCC8@?BB@BFBD@@@@@FBBBB???::46=<?<=9<C=?BB@87,/,/,-1+--11478373<?;9:/..//,/6;:8//8;/////-/<<<@A@=8744;444/51115+,,
@EOUL1:00354:00195
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
ACAEECC?CCBB??@CBB>BCB?A?ABBA?@@A>AA??7BBBB=?AA;BBBBBC@C@DD>B;;9BBCC>BB@@BCB7?@CCC@BBBC
@EOUL1:00381:00176
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATAAA
+
AA?ACCC@BCCCCBB?B=@=@CCCC@BB>BBCC?BB=@6;:=A?BECC=B?AA@C@BBCA>>@>>>>?B@CB?BCDE?AA=ABAAA?;;7;BBBBBBCC@CB@CCCA@?08809<@69;>@>?888>>>8>>484:?BCBBAAABAA<???AAADC<<9<>@<;<?888=;=>?4?>?;;34444
@EOUL1:00386:00181
GAAGACCCTAATGGACATCTTCGACCTGCTGTGCGGGATGTCTACGGGTTTCATAGACAAGCTGATCAGTAGGAAGAGGAAGTTCAAGCAGGAGGTCCGGGAGACCGGCGAGAGGCTCAGG
+
88,/994-49244244:444;>==BB@<<7=>?@BB>B@@@AAA;<;-9;-34=>>:11,////64.33338<?=AAC?B8:;@:;9:B>>477467:<<6<9>>;;75://--,-11---
@EOUL1:00397:00189
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCACCGCCATATTGCATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTC
+
BDAB999-9>>?BB>BGB?BB?D@C@CCFBBCC@CDCCC?<<;11,1119BCCBBA<8;470:,00..885>A:992//+//,.44639577701261,1>667@5444-++*+++----4203866633456:4777545...,-33,++--,--340042355=??@@ABCBD<@=@=>>?<A@@@@CCACC>CA@@<999>A?BB?B>>==,1110008>>>:9000,/77::
@EOUL1:00362:00231
GAAGACCCTATAGACGCGGACTCGCACTCAGACTCGTACTCAGACACCGACTCAGACTCAGACTCTGACTCGGACTCAGACTCTGACTCGGACTCAGACTCCACATACAACAAGACTGAGACGCGGTCAGGCACAGTGTGGTGGGTCACGAACAGGACAGCCACAGAAGGTAAGGCCTCGGCTAGGGGTGAAGGTGAGGACCTGATGCCCAGATCAGCACCCCGAGGCATGAGCGGCAGGGGCTGG
+
@A==CCC@CGCCC;<=CCDDCBBDBCBBAA??9<<<@@D<<<AAABCICBAA@;998;;;;AECDAA=<9BC;<<ACDBD<<;ABB<=<DCC==<BBA@AA=@A>>>8848858>=?AC==<ABGDCCCCBCA???:87>:777.7???7711110167883433..,.5..,.33,117133;;;>7=CC=A=A:;:?BA<@@;;:EL=BCB887><6344,7=>37777777?:865550113,
@EOUL1:00388:00270
GAAGACCCTATGGTCTCGCTTTATAAACGAGTCGACATGTACCGTAACCCACCAGGACACACATTTATGAGCGATCGCGACGCATGGGCAAGTTTCTTAGATAATGGCAAATCCTTGGCGTATGTACTACAACTGAGGATTGGGCAACACGTGTCACCGGTAACTGCAGAGGTACTAGTAAACCTATTAGCAGTGCAGTTTATGCAGCAAATTATGTTTCGCAACTA
+
HEBDCEE@CBBB@CBABBBBB>>999094@@ACBA@@;;;;;@CA@=@B=::8;A>AB@@@CCCC>???---8-,,,0,16ACBBAD@CC:::?9??;>>>77477:=<@8??6200*++++0443997;;997777519<9=6@@=;9:>BACBBB@BBAA>BBAA@@=:;=>>>98;BA=B@B@=?9748;86././/+..6378BBE:944....+.3370,,,
@EOUL1:00384:00340
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
BD@DBBB>CCDC9;<A@@>ADCBB@BB>;;;;@;@CDD9CCCC@BBB>BBBB::8:?BB>CBBBBBBB=BB@=@@@6?>@>><9999
@EOUL1:00357:00428
GAAGACCCTATATTTGGGCTAAGTAGTATGGATTCGGACTCGCCACCGCTAGATTCTGTGGCAGTATGGAATAGCTTGCTGGAGGCTCGCCCGCCGAATAACGACCGAA
+
BCCCAAA:<99<<EBCB<911,10.....13888==@=@@BAA>AA=@?::77934::00,.4433660358=77:@BAAB7:::=>=???<A@=99;--,--++0++*
@EOUL1:00387:00529
GAAGACCCTATAGAGCTTTACTCTTATTATATT
+
9964444-4444498333,37779931261170
@EOUL1:00374:00587
GAAGACCCGTGTGACTAGGATTCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAATAAAC
+
BA?BB>86;<1//1:88@@AA=?>?@CEC?@A@CC?CBBC?@AA@@<ABBBAACDCABB??;?;//0:/0/>@<?@@??@@@@AAC@@@A>@@>@A>@A@?=???<?@ECCCDCBCBCCBB?B?BC@CCBBB@ABBBCB>AA;>>>>>+>>;88;><@?A2?7770,///,-
@EOUL1:00395:00577
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCAGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAA
+
BCADEHNBC<9<A?B696;588BBB?BC>C>AA>::6:;??AC@AAAA=C<??47376667/0,..55:;CC>BBAA<==699>@@>@@>AA<;;;;8;:<A<@@999@<>A>A>A>;;78BB@@BBBC:::<@5::::A::;@@CC>@@ABCCACABCA<<;@@@;10?9:::+0,,--+-//
@EOUL1:00384:00617
GAAGACCCTATAGAAGGTAATGACTGAATGATATGGAACTAGTTAAAACGATTGAACAGATAGTTGTTACAATCAGCAGATGCTGACATTGGGGT
+
DDACDCC?EDEDCB@;110,000188:8;74;733137444442444,48,,*,,*,,,/44887777744464../..3......6..,.550.
@EOUL1:00360:00729
GAAGACCCTCCAATCATGGGAGAGATGTAGATTGGGACGTTTTCCGTGTCCTGAGTGGCTGCGATCTGATACGCCTCACCCGCCTCGCTTTTAAGCATTTCTGCATCTGCGGGGCCAGATGCGGGCCAGATGGAAGGCCAGACCATTCTCCAAGGCATCGAGCCAGCGATACGGGATGTCCGCCGTCTCACCATTCGTAGGTTGCGCTC
+
>:CEFCCAAB>B?BAABKKACCBBBCBB???A>AA<ABBCCC:CCBBBAECCCBBCCDBBGDDBBCCBBACAA@=>@99919?<>>@>@>>4>699>AB11011/6666=9=969:4449999919244?B<<<7713137162769@:8:<;<;==<667337800043066775..--,-2,-03448886=5336--,-2---346
@EOUL1:00374:00754
GAAGACCCTGGCTGAATGCTGCGTGTGTGAGGTCGCCTTTGAGTTGTAGTTCCAAGCGTGATAGACGCAATAAAGTTTTTTGACCTTGCCAAAGGCGTCTCTGCAAAGCATGTTGCCTCGATGTGCGCAACGGTTGACAACGACATGAAGCGAATTGTCTTCGGCGCGCGTCAGCACCACAGACTGTTCACCAATATAAGTGTTTT
+
F@=9999-49244BA@BCBB@@@>@9880.666.0/54524.710117610.,.44.-.222278735:28;=999====*84:/3/03/44-443403>;;;;??9;444>>38:7<;B@A===:11/,.::6:;;<A<?>9999??544/1,1,///.64444...45..9;<<A8::??:::AB:<>>4858;>75,,,,,,*
@EOUL1:00393:00823
GAAGACCCTATAGATGCGATGTCGGTGCAGAGCAATATAGCTTGCTTCTTTTCAACGAACTAGAAGTAAATAGCAGTACGTTTGTTTGATCTTGCTTGCCATGCAACTCAAAAAGATTCTAACATACTTTCAATGCTTTAGGGACTGGTAAGCAAATCGCACTTGCTTGCGGGAACTAAGAAGACTAAGACTTGAGCTTTGCGTGCGATTTAAGAAACTGAACAAGCATGTCAAGTTTTCTTGGACCTGAAC
+
BB?BBBC?B<;<;;@CCCGDD@@ADBBACCC@@@=??=:00.3...63111,1799>C@ABBAA@AA@@;@88:A@=99///,/660-.../74BBABB>B?@AB5:;@@>>@/999>999599>999>99959@99903772<77711686777,333;77<=>AA>??::8664..,..44388>=9663230068?;9>?877AB@?<B=<<>9;;7736600099=;==<===>5=;844--,---,-
@EOUL1:00375:00850
GAAGACCCTATAGAGACATCCCAGGCGT
+
//,////+/699494=8..11,.4,.,,
@EOUL1:00394:00948
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAAACGATGTGCAAGAAACGCTAGTTGCTTACATATACAGTTGGGGTG
+
BBAC<;;8<A@AC;9;;@C@BBBA@=@C=B?BG@CA@A=??>B?AABD>B?BB?B@BBBAA??:<<;:98<B>B=D=DDC?AABBA>BB>BBBBBCC@A=@B=99999919B9@>A8;9<?>;:;@B>=8::,00+0....1111017>8<=>==888548/03440000000/000+00
@EOUL1:00381:00951
GAAGACCCTATAGAGCTTTACTCTTATATATCATAAAT
+
CC@CCCCABCBBBBBAAA=CACCB@>7.070004000*
@EOUL1:00371:01033
GAAGACCCTATAGTGCGAGGAGAGAAGTAATGTCTCAAACTCC
+
CDCHBDECCCA@ACB<<<C@CCABB?BB>:<0,,,,++*+...
@EOUL1:00397:01042
GAAGACCCTATAGACAATAACAATGATAATTTCACAGACATTGCCCTACAAGACCGTGTTTCCTTTTTTAGTAAATGGACTATCAATCGAAAATCTGGAAAGGAATTTAACATCGTTTCGCGTTATTTGTATGAAGATCGTTGGGGTGACAAGAAA
+
AA@ECCBBCC@@@BBC@CC9<<BCBA@C@CE>CAAABBA??;::7.4>773:@>=@;;;;-74:BCCF-CAACC@C@><99@===9==??CC6@===>AA<A>@=??:?<??A@CAA;??=;;8;==7>?@?@@>@@@>>;2444-49;;;<7784
@EOUL1:00368:01118
GAAGACCCTATAGAGCTCCCAGGCGTGGCCGTGAATCTGGAATATCTCGACAAGGTCATTGACGAGTCAAATGCGTATCTGGCACC
+
CD@B???<@BBBBBABBCE@@@=?B??=@>AACE>==?@>@?A>>78447?;8;>@@@A=@>=888;8;=:=>>>;:78//,--7,
@EOUL1:00355:01152
GAAGACCCTATAGAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
CD@CBBB:?BCBBC?A@B??:;<A?@B>DACC@CB?A?AA?B=??BB=B?AADC@CCCB>>?@@C=>A?AA=BAAAACCBICABA=@C?BCBBBCC::;<C@BDDCBB7::6;@BBFB?BBBBBB>>>7==,0,0>;>?CCBA?@@>@A@@ABEA??<?@<<<@?==>>;>@@1@;///,.-23
@EOUL1:00406:00036
GAAGACCCTATAAAAATCGACTATACAGAGCCGTCATTAGTAC
+
>=><BBB7<9=>>>>/>@@@??BC;;;?000,.....,.7116
@EOUL1:00425:00043
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
CC@DAAA@ACAAABBBBC:??@;;5;ABACB@C@CBBC9CB>?=89918;;9=@=A>AC>ABCCCCCC;<<B>AAB6AABBB?BECC
@EOUL1:00405:00064
GAAGACCCTATAGAGGGTTGGGGTGACAATAGA
+
BB?CCCE?CCBB???B?CABCC9CCBBB?AB@?
@EOUL1:00411:00055
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
CC@CCCCACCBBB@CC??:<<?BBAC88;?@?B?BBBB8CBCC<ABC?DCA@AA=A?BC<AACBBAAA>ACCABBB8B@ACB@==>=
@EOUL1:00418:00061
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCTGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACCGTCTTGGGTCAAGGCCGCACCGCCCATATTGCATGATGCGCAAATGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCC
+
::>ABEE?CB@A@A<@BB@CB>C>B;9:CDCCC;@<DE=<=BC<<;B@@>?@>??;7717>?<?><69544;;8=244-43133337999?:99244-49949;AD?BBBECAAAA<<;B?AA<;9@@111/,///8:B>:;;8777<;70..,...+.5572767<<690/,277+-.33355/116613133,16772766;9==;900/670**-,,,.77677533,.....-
@EOUL1:00428:00138
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCCGTAGTTTAGGGACATATGCAATAACTACTTTCCTGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACGGTTGGGGTGACAATAGA
+
9424449149998;;8;@B>BACDDABD=A=94,..,.,..,.3668<;@@CD@C@CC:::?::::,01,/8+///8,444,499444-444-4999A9:8C@GC<<<CAA@<=;88;82448=99:@=CBC>@@@B>===9=6///,//,///944249;:6//////6656966609333313.44
@EOUL1:00434:00108
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTACAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
DB@@?=C@CC;9=@B?A>A>ABABDACC@C@CC@CA>C???;?=????=@@CCAB?BBBABBC?CC=AB@CHBCBBCACC>BBCDD@BB?BCBBCC@@@?@@=AAACCCACE<@=>;9958???;;;BC>CD?B@BCB???CA?<<@:@<=@BB>@@?<?988BB4487737CC5@@;;7,/...
@EOUL1:00407:00211
GAAGACCCTATAGTGGTTTATGTTATGTGTAAATATCGTTACACTTCTGATGGTTTCTTAGCTTTGCCTGTTTCTGGGTTGGGTAGAGAATATATCGTTGCTACATATAACGACATGGGTTGGGGTGACAAGAGA
+
BB@BBBCBBABABBC@EE@BBAC@CB???BB><>@@@@B>BBC?@=@BCDCB@BC@BC?AAACC@CC@BCBB=CABB=A?AD>@BB??@=<?@@@AC@?BBBCCAAA??B>::@@===;=;=<<5?@@<B=<???
@EOUL1:00420:00248
GAAGACCCTATAGATCTTGATACCGACAACTTAGTACCTGGTCTTGAGGAATGGTGGCAACTCATAGCGCTTGGAACAGCTCGAAAGATCCTACAAGACAGTATGGATATGGACACTGTAGCTTTGATAGATCCTGAATATAGAGTGCAAATGAATTTGTGTAATCGAAGAAGTCTTGTGCAGTACACCAACGAAAGACCAGCAACCATCTATACACAGCAAACGATGGTACTAATGGTTATAACGGATTGGGGTGA
+
BB=;99919999;;BAA;>999958>>><??=@CAA@=77477=:=?@=BD<<:<A<@@=9<;AAA@@BBC@C?A?@:778>877.7>>?@::;?687@>=????=?BCDD?AAC:::@??<?@7<<@??888><<<>@?@=898???BC9==<=@@:==;:>6<<AA?BB>BB@A=@B@@@CBBA@@3747>>>:>>@<>>995959>>89;:@A==>:::6--,--*..699;772677633..,..,.55057:
@EOUL1:00424:00315
GAAGACCTTGAACCATGAAGGCAGGCTATGACGTGATCTATCAACT
+
9:244;:;29924:444424244;6444444//////,2,//65/4
@EOUL1:00416:00364
GAAGACCCTATGGAGCTTTACATCTTATATATTAATAAAATCTACTTGAAGTGTAATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
9959999194485941//+////67768;;@;>>A::::+00...,*./,////37799;9<:;776999?>AA@>BCC8787>=9;9>9
@EOUL1:00434:00393
GAAGACCCTATAGCCATTTATGTATTAGTAATGGACTGCTGCTATACAATAAAGCAACACAACCTATCTTGCCTTCTATACAATGAAGCATTTGTAGCTACACTATGTTGTTGTTTACTAACAACATGTCA
+
BB>BA@A<@??@;<9<CB?CA<:;DBCBBB?@@9;;@AB@A@CCCCCC@@@@:???@BB??;?;==:::,/0,.69<;887>=@7:11888?8???=BA?=?868800/33444434.--2.,*+2----2
@EOUL1:00430:00458
GAAGACCCTGAAAACTTTGTCAGTAGATTGGTGCTGGGTTATCGATACATGTATGGGGTTGGGGTGACAAGAGA
+
CB@D99419>?AA4BAB<ABCCCDDAAA@B=@@998>092:AAA@AA::<@F@9999/1-1//*///020,...
@EOUL1:00411:00573
GAAGACCCTATAGTTTCCCTTG
+
B?A>99A:944;444-44-/6/
@EOUL1:00440:00615
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGAATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACA
+
>>64999-4B4<;B?A@<A8;7;9CCG@EB?B:::6;=?=@7:7>>>3:26:<;00,/:77;>>884788083443333423004:9;9??A:;::C@CEBBCB@@<=:;11..,.4445<ABAB?BAA@E@BCDDBB@B@?>>330>>>>>B=?:?:777766,167700003323666;;7
@EOUL1:00411:00737
GAAGACCCTATACGTGAATTAGGCCGTTGGTGATTTGTGTTTGCACTGGGTCAGCCTACGATGGACCTTGATGTTAAGTGCGCGTTTGAAGCCAGCGGTCACCCGGGTTATTGCCCCGGTGGCAGCTTTGCTTTGCGTCGAGGTTTGACCCCTAACGCAGTGACGTGGACCGGAGCTACAGGTGTCGTTATCTGTTCCTTGTATTTCTACCCACGCGGTGATTTCTTTATCGGAACACTTGTCATATGTGT
+
9969@A@>AB;9=CDCC@B@BC@C?CC?B:;;BCC>CAAAC@CAAA;;;6;AACFD><9>>9924989?DBACC@C=@@A@?@877.7@;@@@BBAB><@>BC?CDAB??=:7777.:=BCAB??79904444-488998=;:;;:>>AC@8CA5:7?:119696>@888@C>9:647.355.///..77668>@=@?A;==>>9677166,00119=99:;;:<;>9;=47/48::8101-.----0000
@EOUL1:00403:00933
GAAGACCCTATCCAATACTGATGACAGTGACAAGGGTTCCATCATGAGTGGAGTCGGTGAGTATGTGGTCCGGTTTACCATCCTCCAGCACTATAGCAATGTCACTAGCCTGCACTGAGATGGGGATGTGAAAAGTTAGAGTAGCTCCTACGGAGCTGTTGCCAGAAGTTCAAGTCTATCTGCTAGTTTGGGGTG
+
BDDDCBC@CCBB?C?CCCCBBCBCBAAABC@>=?B>B@C?BBB???>>>>>;99>>>>>>>><<;>A>=A=A@BB>;97;<959B=<==CBAAB=><BCCBCCCCCCCCCCDCHHAAABCBBBB9B@@@BCBC9BB?BB@B;;;??A>BBAC?A:;;AA<AB=B===7:5:@<@BBCBB@???83111,111-11
@EOUL1:00428:00907
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATA
+
9959999-9;>A@AC@C@C@BCCBB>BB8;698138131173706===9:;>>.0/;;;5?C?AA>AB?BB@A?91,11-1<111:;=:;=>>>9989<>>;>@@99<8;A9><@599;=><ABBBCB9884843311169=>>?B=A?;7./,+++*+++++
@EOUL1:00406:00972
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
DD@BBEC?CAB@=@@@@A>ACCBC?BCBBB@B?=9>??8BABC?BBC@CCCECCBC@CC@CBB?@@@@>@DD@CCC9BAD9959999
@EOUL1:00406:01021
GAAGACCCTATGTCCACCCTCCTCGTAGTTTGCATGGGCAAAGTCTTGCAGTTGCTTGATTGCAGTAGTCATGTCACTCTCCTTCAGTTGTTGTTACGATCTCAATTAAAAAGCCAAGGCAAACACCCGATGCAACAAGGCAAAGTTTGTGCAAGCCACCATCAC
+
DCCCCCCBCCA;;;=ABB>CC@BBBBCBB@>BBBBCC?BCC>AABD?DCCCC@AAA@DDFCCCCCCCCD@AACCBBCFFDDAB>BA998;@;>B>BAABABA?<:<:<===/<><A@B?BED<A::7.7774=B?7:59:8=>:761,145./,.4431064300
@EOUL1:00409:01067
GAAGACCCCAAAATCATATCGTTTACGGGGAAATGAACATCCGCTTTCAGCTCGGCGAAAAAATTCAGGACTAAAACATCTTTTGAAAGACGCTGTTCCAGATTATCTAGTGAGATAAGGGTATGTTTTATTTCATTTCGAATACTCAACGGATCGAATCCTTTCTTTGTTCAGAGCGCGAAGTCGTTGCTTTGTCTTTCCTGAATCATTCTTTTTTGT
+
DC@BAA@@4>>>4>>@?>>B@@A<ABBLD:CCA;;<@CCDC@BBCHBEA@@@@@=@CJJCCC-C@CCC@BCCBDD:@A?BBBC9BBB???:;;A::9;?BBCC@BCBB:::??A@<<9<@:;;;4444-44=:@AAB?@@@9===@@B<883788849;;244+..-+--,-4200362113...4443377../:;;6;;@;97:4442/,6666+5+
@EOUL1:00446:01091
GAAGACCCCATAGATGCTTTTACTTCTTATATATTAATTAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGGAAA
+
49244444,4///////...*3985..,....6601279>@@7@@9626=A:@AA9996959=9=:;9999C=@@AA;;;10.;6:,0,.66
@EOUL1:00443:01144
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
??;==?;:4;9;@?:===-00>==6:7//0.,/://..*.776:799149949:@C@BC?A>@?@=@B<A@@>A@@7<??@?<>777
@EOUL1:00437:01192
GAAGACCCTGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAAGAGA
+
A?;@@>@<A<<<CGCBBABA?BBC@CC===@B@BB@AACCBDBBBC>BBBBBAAAA;???A::9//0;;;;;?=??BAAAA@??===A@@>@B>@@>?AAA>A==;=ADBBDDF=><B<<<1111;<;AAA?CCBBBC@CC?BAAAA-BB?BBCFDBBB8BDCECC;:;@
@EOUL1:00454:00082
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAGA
+
A@7:94A?A<:=CCBB?A<@@CCB8ABBBC?BBAA
@EOUL1:00460:00067
GAAGACCCTATGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCAATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCGACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAAGAAA
+
CDCCCBC@A@>=1111;;;:9:BB=@@A???>ABB@?@@<@;<=CCCCCACCC?@@C;9;;C>CC@@@@ABBDBBC?BB?B::6:AA>???AAAABBC@BA@CCBCC?BBCAC??=:?AC5::;@BAAAB@C<=<B@B@AA@CCDCCCC?B@C@BBB<<<C@CCCDCC@BC@BB@>>3777.488::83444
@EOUL1:00482:00056
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAAA
+
//,///1-17@ABCCCCC?C@CAA8A?CBB8::??
@EOUL1:00457:00116
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGATATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACCAGTCTGACAAGTGACACA
+
AA@==<B>B<;;AA<>>9A8;;CFDCC@CC?BBBB>A>@<A?@@?==:=;===@A@<@DB?CBBB@A>AB>A@;9999>>;>;;9?AB;;;AAA<A@=@@9999991959>>99;@>D=C<<;E?AAAAB?BCGCBDAC@AA@;;;A@@@@?=??@=:;;B?B:<:00,//6//,333/..,.6533611101111..--
@EOUL1:00462:00148
GAAGACCCAAAAACGAAAATGCAAAAACCCAACAGAATGCAGGGGAAGCAACAACCATCTAATAATGGTGCGGAAAAACAAAAGCGAAATTTACATGTGGTTCAAGTGGACCACC
+
:=CDDCC@CCCA+49999/9?>>>>>/;;,1,11.;;==B<???5?;AAA@BB@B@BCB=@=??;??;????;????,7777-63311,176=<==<<C>C88/./.;//+5500
@EOUL1:00453:00179
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
CD@FFDD>BCBA>BBBBC@BCBBB@B?;>7769:?BDD8CCCC<<;C@CCAAAB?BADD@DCC<;<CCACBB@CCC9@@B<<:=BCB
@EOUL1:00469:00198
GAAGACCCTATAGAGCCAGGATCTACGTTGATGATTCACTTCTATGTGCGCCATGTTCACAGATGAAGCGATTACTAAGATCAATGATAGAGCGATATTCATCCTATGTGGTCGGCCAGATATTGCCAGGAGACCATGTCCTCTAGCTCAAGACAAATGGGGTGGGGAGTCAAGTCTGTGGGGTGTGAATATGTCGTTGTGCGGACACAGTG
+
CCBDBBB>AAAA<;;@>;87;?777?::3:??>>A77:::@BCBBB>110677300/04C;=>@???A@AAB@B@@@?BA@A@>::;9::?AAACDCC;??6647>777=;=:9242788=@C:<<=8824:8978430-37.-.--224/3444940000+0000+00000/0000549;=4:0/0;56488346606.--2200--.320
@EOUL1:00469:00205
GAAGACCCTCATCGATCTCAACGCCAAAATCTCCCAAATGTAGTGCCCAAGGCCGTCTACCTTCAGTCGTAACACATTGAAATAACGATGAACTGACGGTTGGGGTGACAAGAGA
+
CCACCCCACCCEABBC;<<C=AACBDDD:BGBCD===:8999>8449-95958<=>@B@A>AACC@??A884:>??A;??@9??<?DDCCC?CCCBCCAC@CCC9BBBBB?BBAA
@EOUL1:00481:00283
GAAGACACAGCTCAAAAACGAGAGAATTGGGGTTCAAGTTTCATGAGTTGCTTGTTCGGGGGTCAATCGCTGAACACTAAGTTGCTGGCCTGAGTGCATCGACGACGATTCCGGTCGTAGTGACATTAGCACCTGGGAAGGAA
+
DDCAAAA@@;<<BBDBE2@;87>76383;77,10131333,3677>?A?@@@@CC@CAAAA084;893-----,-22---.4245.../..-497.87/-18/-.355;>@;=<=/////---244263004/000+023444
@EOUL1:00489:00272
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
DB@?ABHD<><BA@@88:6:??CCACECAAA@B@BBBB9AAAB@;;;8<AABAA?BACI@CCB;;<@>:>AA?ACC9DACCC@DCBB
@EOUL1:00459:00471
GAAGACCCTATAGCATCGC
+
4::=949-4444>9944//
@EOUL1:00496:00539
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGA
+
CD@BBCE@CCCBCBB@B@BABBBBB?BB>B@999;C@CB@?ACACBCC=B?BB?B>B?@@@@?>@A@CC@DC>A;::24409B?998;A<BDDAADDCC@CCCFBBBBB>@@9@8;9;><@@8800---+**)*)***
@EOUL1:00499:00573
GAAGACCCTATAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAAGCGACAACCTTTTACGTCGCTGAGACA
+
BBCB@@A<ACBBBBC@BBBBBCB@A>>A??;?@==::75;B@ABBBCC?BBICCCAA?@CC<B;;;ABAAABCBCBCABBBB?BB@@@CCBBA<<<BABDBBCDCC@@@AAB@DD@BBC?A?A<>>=10,000++*-,,0029;;2;3333331331133
@EOUL1:00476:00644
GAAGACATCTACACAAAACTGAAGGAGGCCGCTGGTAGCCGCAAGGCATCGGCCTTGGTTCGGGATGCCATCACCATGATCATCGAAGGCGATGACTCCTTCAACGGAGCTACAACAAAGCAGTACGGGACATGATCGCCCATGTCCAGCAAGACCAGTGGTGCCAGTCCCTAGCTGTGGTTGGGGTGACAATAAA
+
CC@C@BCCB>9==9999/9>>A@D@BB@B@CBAB=@;:;>B;;9;?CBCBBAC@C@C@C@:::5:>=?:@@@@B>ABCCC:;:BAA=A:?:7:@@:::>B?BB>?>?111@AAB=AAB<===BAACCC>@======@AAA<@@@@@?AAAA>AA@=<<<?9;;A=999DC:@<<:::44242444+031123...-
@EOUL1:00494:00625
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
@?;?999-4999AAB@AA<@@@CDABBBB??;?:==@A6ABBB>@AA<@@@<<=ADACC?DAA@AB@@=@BB=BCC599==;94444
@EOUL1:00483:00673
GAAGACCCTCACGAAGTCCTGCTGGTGCTAGATGCATCCACCGGACAAAATGCCTTTGAGCAAAGCCCAGACAAACACGTACTCTACATCGATCTGACAGCTCT
+
EEBCBBD@CB:71/,111>@DCBBACDCB@AAACABBC=AA>AB@>AAA6@AB=A@:>@9999-911,11////+///////77344444444433333374..
@EOUL1:00498:00655
GAAGACCCTATAGACAGGAGGGATG
+
AA@A999-499>?99371...+...
@EOUL1:00488:00775
GAAGACCCGTTGGTGGCTCATTCGTGACTTTGATCGACTTTTGCAAGAGCGCCATGGAAAAATTCACATGCGGCTCAGTTGTCAAGAACAGTCGAAAGTCTGGGTGAAGAGTTATGGGTTTGGGGGTGACAAAGAAA
+
CC@B:;;89D;A@ADBHBBBB@B?@@CBBB>A@@@A@CCDE7BBB?BB:;:?>B@@59888-9@BCCBBCBC@BBBBDC@::;D@AA;;;CCCDDB>BBBBCB>B?;6>600/0333+00+044;.300111,1677
@EOUL1:00455:00946
GAAGACCCTATAGACCAGCGGTACTGGTTTTGGCAGACATTATATTTCACACATTGTAATGACAGCCTTGTAAACCGATAAGCACAGGGGAGTGTCACTAGTTGAGTATATTGAGTATATCGATGTTTGTCAAAAGCCAACGCAAAAAGAAAGGGGAGAAAAGAATTGAAAATAACAGGTAACTCCACATCTAGCAAGAGATTGGATCTGCTCCATTTCTATTTGTCGTTGTTGGGGTGACAATAGA
+
BB;=9;;;=CCCCAA?ABBC@BB<AA@CAA7=;=CBBBCC@A???C>CCCDCCB:<<D@CCBCCCD?A?AAAA>B@DCDC=@@@BB@BB7BCCCCBBACBBBBBAAA888CCBBCCC??===@@@CCACDBBBB9BC@D@CA78888,88CADFC9CCCBB7::567666?570,00/,--,.77066<46.....5441662656>@@?AA?7::@<?:::B977::737=:888.88832,..55
@EOUL1:00465:00986
GAAGACCCTAGAGAGATCACGAGAAAAG
+
//,/4495==>8:8/........444*.
@EOUL1:00467:00990
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGC
+
AA?ADCH@@@BBBCCBC?A<;?@A@=AC?>=>>;///5557:7;:886>47:;?@CBBBDBB>??:<>59989A@@6<919>98;@CD@CCCBBCCBCACBAAB>>;;-11,1::,/.6663333111,36360117161742---+---
@EOUL1:00472:00987
GAAGACCCTACTCGTGGACCTTTAGACCCAGTTCGCACAGTTAATGAGCTCGCCTCACGTGGTGCATATGGAGTTACCTTTCACGACGATGAATTTGATTCCTATCGGTTAGTGACGATGCAACTCGACGCAGTCACATTGATTCGTTTCAAGAAGGCACTTGCGGAATACCGGAATGAAAGTTCGATGGCGAC
+
BB@B999-4:::@BBB?BB>@>8>8777,117016788@<><C=@@=443087=<?@BAA@?A@BBBA@?@BBBABBABC>BBA:::==:4838B:;=>3424:///,/,/777AA@@@@@?=898:8//+000...16361734300+3;378>@>@@>A;888,/,///,/7;9>?:544://.8800..-2
@EOUL1:00459:01017
GAAGACCCTATAGACAATAACAATGATAATTTCACAGACATTGCCCTACAAGACCGTGTTTCCTTTTTTAGTAAATGGACTATCAATCGAAAATCTGGAAAGGAATTTAACATCGTTTCGACGTTATTTGTAT
+
BEADCCEACDCBCCCC@CB@BB@>?ADD?AA>AC@?ABABC>AAD>BBCC>ABB?A?AAA:=377777*7==?B>?=:@@@=???499??CC6AA==9<A6879244-8>?@B;5755:.//.3015633:64
@EOUL1:00488:01043
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCACCGCCATATTGCATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCC
+
DDAAAAC?BBCDCB>BBB?BC?CACACA@?ACDABBBCE>=>>113>?C@@ABDDAEEAC=;9:0001=8;<<CC@CB@CE@@@@C@?<?A?4747=6=?A@CB884884=<AA@===@@;;;;0/0@;;?@@>;:7::-4449;788;30056648834246=?@?CCFGCCC@DCC?BAC>AAA??@B@BB@C<<;099969@@?>@@@?B?CABB@;8;CA?=@BB=?:<00,
@EOUL1:00497:01002
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAAA
+
@A?ACDC@AABB@>97776:AABC@BBC@@@?A@A><>7>;760:;;7<BBBBIAC=@@:>>>>@@AB>BBB?BCC9E@@@@?98888
@EOUL1:00478:01071
GAAGACCCTATAGAGCTTTACTCTTATGTATTAATAAAACTATTGAAGGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
4@594495999ABA@@EE@AA@00,00,2..,.4457:6:976014248999992497>+////:144-44/5;=;0;;0//6888:
@EOUL1:00451:01107
GAAGACCCTACACCTGTTCCTGAAGTCTGGCTGGCGGTACATTGACCCCAACCCATATATGGACAACTGGCATCTGGAAGCCATTGTGGAGCATCTAGAGGCGGTCAAGAATGGCGAGATCAAGCGCCTGATCATTAACCAGCCGCCGCGCACATCCAAGTCCTCTATGCTGGTGGCGTTCGATCCGTGGGTCTGGGCGCAGCAAGAAGTGTCCGACACATCCGGCCCGGCCGTGCAGTTCCTCCATGCGTCCTACGCTC
+
CC@CBBBBCCC?><@@@=@@BBB@AABCC?:::?BA?BBECC@B9944-4;@B>BD@AABB?BCD@CCC@@@@<;;AB7;<=?B?AAA=@@;;;BAAABC@BD>A::::B@CC@?@@?BBBB?A???>AA===AA?A?@>@AA?AA@D<<=@@A==:=<@?@9<<A888@@A=AA899A>A@@@@;==<CA@899A688AAA@;;;>0//7::5==?AA?>=:;3//,/,/,/666;<;8;><<9=?=?@<<9;==:888
@EOUL1:00459:01100
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGTCTTGTGTTCGTCA
+
442:@AA<A@999>;>><@>AEDDCCC?CB?B????A>A?C?ABCCB6:37=>448=?>?<;773037984;:862---101940/494447787//,/678=BCC>C;<;:<;:;:C?BBBCCACCCCC@;99>0949=;;;BBBBB===<377,6:<??CCD>@?@@>AAA@A:<<?=@AA:;;@::::>@??<6???::96//.,.++0*+++--
@EOUL1:00474:01156
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAACGCTGTTGCTTACATATACAGTTGGGGTGAC
+
CE;<<A@>;99BAAA=@<@@CBCBB?BC>B8<;>@@CD@BB@C@C:875:>AA@CA???B???>BB?ABCGF?DCDCABA<A<<<;?ABACCBCCBCBC=AA>??999919919<>>BCGCCDDDB???9?A;?;=777777444877.----,-1133.4555<==:<8866.5310
@EOUL1:00451:01214
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCTGTAGTTAGAACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
CC@DCCC@CFCCCCC@C@C@A;78>483,3779>?B>B???<?DBA??<B@CC?B????ACBB?BC@BB?BC;===<ACBDCDHDCADBB?AACC=<=DAKL@CCBBA?08808;><@B>A@<<<BBCCAAA=@>@;:7@@;>A>;;7:BAAAA?A;;48BA;;;BA===:===.76677179BB
@EOUL1:00473:01243
GAAGACCCTATAGACATGCTGGACAGCTGGCGCG
+
//,///1-//448?@@BAB?;,11.,0,,*,,,2
@EOUL1:00474:01213
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
ECABABB>ABBCBBBBBE>CCCCB@>>>@A?3637777-8C?A?ABB=CCCCCEAD?BB9;;B@BBCCBCAA>ACC9CCBAA<8888
@EOUL1:00481:01230
GAAGACCCTGAATAGCAAGACAATCACAGTACACCTCCCCAAAATCTCCACATGCTCAGATGCACAAGACGAAGTGGATTGCCTGAATCAGTACTTCCAAGTGGCGGTTGGGGTGACAATAGA
+
AA<?>@@=ADCAD@AAA@CFCC?BBCCCCEBB==88444-999/9>@A@CCCCCDECCCCCCCCCC@CCCCDAB=:94859><ABB?BCECAACC?B?A=AAB?BCACACCC9<<???:??@@
@EOUL1:00510:00033
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAAAA
+
AEBDBCE?CDCCCCA@@C<A=???5@BAB?=77777
@EOUL1:00525:00027
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCATATAA
+
ED@CCCC?@>AACCB@A@B@BAABB?BB>B?BB?CB?B?@?>?>?@@@;B@BD?@>>?A@BC?<<<B@C@BC@CDDC@CC>DC:;:@BC?C;<<CCBBB@CCACA?9999;C@IAC>8820*))((((
@EOUL1:00502:00093
GAAGACCCTATAGCAACACAACTGGAAGCCTCCTCCTCTAACCTCCTCCTCTTAATAGTACGGGAAGATGGGATACTGAGAGGTTTGAGCACGAAATTGCTTCACAATATTATCATTAAAATGCAATCCCACCCCACCACCACCAGCAAGGACGGGGGGAGGGGGAGTTTTCCATGCTGTTGCTCAAGTTGGGGTGACAATAAA
+
CA?BCDDBCBC;<<C@B999-018885763..,..,.3333601101101110101116:?AD@D?@A@CA>ABDB@@<::C<<<:784887888-477=@?AA@@:?AA7;:???;<<<-91=?=888/3333,399@@?C?733..,.,...5777+76:77/33333+0/0003030444000/00/04;0;555><=<==
@EOUL1:00529:00105
GAAGACCCTATGAGACCATCAGA
+
AA@CCCC>CA:;:1//,/6000.
@EOUL1:00533:00115
GAAGACCCTATAGAGAGGGGTTGGGGTGACAAGAGA
+
995<CBB@CDDCBBCAA?>5@>BBB8BABAB>BCCC
@EOUL1:00545:00104
GAAGACCCTATGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCGACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAATAATCA
+
AA@C@A@>ABA<81//,/9>?:BB@CBB?::67>>>?@?<@ABBBBCCC>BBC@B@AAA=>A<BBB@@ABCBB@@@>ADCCBA@CBCCCCAAACBB@A=ABCBBBBB?B;;;A>AB>@@A:?=?>A@??A:;6;AA>BAAB@C@@@ABC?BCG@DBBEBCC@CCCCBC?BB@::;A?:=??5?=84/711-,,,
@EOUL1:00529:00207
GAAGACCCTAATTCAGGTGCGGTGGTGTAAGGTCAATGCGCCACCCGGTCTTGCCCGTCGCACAACTCATCGAGCACTAGCGCATCGGGCTCTAAGCCACGCTCCAGTACACCATCAGCACGATGATTTTGAGGTCATCCAGTGACATGGGGTGACCGGGCACTGCCATCGCGCGGTCGAGCACAATTTCGCGCATATGGGGTGACAATAAA
+
CDACBFH>CC@C@BCD?BBBB?BB9;;AA>B@CCC@CCBBB?AAB9@=<<;?CCCACFCCCBBB@CCCCFCC:;<ACCCBCB@@@ABA<>@>8858848>??8<7:??>A??<?ABBBCCD???@:;88.7??=@C???ABCDBBBBBCCC9BBBB>AA<??AAA?<AAAAA@B@=B@<<<@:::344-4:::@???AAAA6B?AAC;=<@?
@EOUL1:00509:00296
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTCGTTGCTTACATA
+
996499;:AAB@@;:8;=?488784239,317313>:=3??;?>ABBB1;79?@B=?=?;;1637<:<B=@AABECEACB8999996;B>ACCCAA??>599699999919@:@5949><>>?@BA=>;6:?,0,.54./::BC?;;>@;80../,///7/7340
@EOUL1:00520:00289
GAAGACCCTATAGTGTTGCAGGCGGTAGTACCACCCCCCGACGGTAACTCCGATGGGAACGGGAAGTGGGCGGACGTTGTAAGACCAACTGTACTGGTAAAAATAGTGCTCAAAGTGGCACTGTCCCAGCAGACCATTAGACCACCATCGGAATCGACAGGGTTGGTGCTGGCAGGAAGAGACGAAGTAACAGGAGCGGAAATTGAAGGAACGGAAAAACAGGAAGAAACGGGTGCAGTTGGGGTGACAAGAGA
+
AA>ACCB?CCCCBB@@7<;CB@CE@:;;B>>=877777)8999:AC@ABB<7>777-7:<>@<B@@@@C>BBAA??A5:6?===:;<:<AA@?@CB?;<;;;0;>?@AAAAAA<AAA>???@?;;?9;;//0::>9=;=A@>9=<9<=@@>@?A@@@A@??:?=?=@A@@@@BDD@C=@@@@DBB?BBB?BCC?BB;;68>8><@B?B@C@KD<????2?BD@DBB@?9?>>4:<?787=@@>5A@88:5:66?
@EOUL1:00505:00359
GAAGACCCTATAGATTCAGCCCGCTTGAGACAGGCCCCCGTAAGGAACTTCGGAAGAATATTAAAATCCTGAAGACCATTGAGTACATTATGGATTCACCTGAGTCTAACGAGGACCATCTAACCTGTCGTATGCGCCCATGTTGACTTTAGTATGTACTTTAATAGGAAACAGAACTATCGTTGGGGTGACAAGAGA
+
CC@FEHHBFCBBF@@?@@99919@@69;A<<9DADCDC2;:;9;9;???<;84959954:;5999/499ACE@CCE<<<@GCCFCDCC@99969969>><>B<<=CBA?D==<@BD@CB>81,1,1////68;;;444-33334;<@AB<BBA>=99999@9?;?;;7;C>A@@@><<;9AC@=@@>4>989@>B<<<
@EOUL1:00530:00465
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAAACGATGTG
+
4;749:B:BA<<<@B@C@CAC;79@6991424423399137;76;677,37771313761117<774:959D:>9;;7;B>BB9448482448=8>A;989B6<;@ACD@CC?C@C@@>499988777963713723331661
@EOUL1:00507:00580
GAAGCGAACAAAGAGAAGCTATCGGCGTATGACAAGGCCAAGCGTGATTCCCGATACAAAGAAAAGCTATTCGTTTTCAAAACTCAGTTTGTGATCGATGTACAGCTACGTACGTCTGAGCATCGATCGATGTACAGCTACGTACGTCTGAGCATCGATCGATGTACAGCTAC
+
CC@C;99<99918@AB@CCEDDCC@CBBBCB?@@7;59;>599>>?>@=@C<@@A@99919@B@7CDBBB@BBCCB4<999/999A99919@>>9<9><<:11/68///////6...077;8:<:87;//.:;;./-/-22;/...@86>@=44444>@@;3300033341--
@EOUL1:00516:00644
GAAGACCCTATAGATGCGTGTGTTAGCTAATCCGTTGTTAGCAAAATGATCTCTTGCCAGATATTAGGAGTGTGAGTTGGGGTGACAAGAAA
+
@A=ABABABCCCC@AADCBBA@@?B@@@CBBB?BC@CC@ECCCCC9BCC<<<CC@CC@BBABCC@CB?AAABBCCCCABBC9BCBBB>BBBC
@EOUL1:00528:00608
GAAGACCCTATAGACCAGCGGTACTGGTTTGGCAGACATTATATTTCACACATTGTAATGACAGCCTTGTAAACCGATAAGCACAGGGGAGTGTCACTAGTTGAGTATATTGAGTATATCGATGTTTGTCAAAAGCCAACGCAAAAAGAAAGGGGAGAAAAGAATTGAAAATACAGTAACTCCACATCTACAAGAGATGATCTGCTCCATTTCTATTTGTCGTTGTTGGGGTGACAATGAA
+
AA=ACB?:@AC@@@C@@899683333137266611111101111778=>A;99<>995;BA?BCA@C:88CD?DADDCB=@@BBBBCC7AAAAABBBABAC@BAAA;::C?BBBCB@?@??<;;DD:<<CGGD:BB?B>BCBDCCC0;8?/88;4<CDEL9CC6747777.777=>>;;@@@>@?=>>?>88587>>::11/..3777666267666899969=??=@ED9B@A??;9767
@EOUL1:00531:00632
GAAGACCCTGACGAACAACCTAAACCTGATCCAAATCCCATTATCCAATCCATCTCGGCGGCTATCCAACGAATGGTCATCGCTGCCTCCTTCATATATACCAAACAACCATCTCGAATCGAACAGATATTACGTCAGGTCTATCCGGCAACGATGCCAACGTCGATGCCGAACTAGTCGAATCGATTCAAACTC
+
@C?F?>A:44DD@?8942968=>:@@CF@@B699094;:@@5:;96;<BB@A@>@994996999995969>><9959<<111>;182499:<BBEE===9;7;A>ACBC@BBD@;;;<??:<69@999@@>AABB<=<;<<B@<<:>111::/43331717=@;>@@@>;=?;:4499411,/////,/665/44
@EOUL1:00511:00670
GAAGACCCTTGTCTGTCCAAATTGTACATAGTAGGATTTGTGATAATTTATTAGGTAGTTTATTTATTTATAATCTAGTTAGAATGCCCCACCCCCAGTCCGCAGCACC
+
?@@DDCC>A?ACCBHCC@ED?BBCBBBB@@<=9@<BCCCCCCCBC?BC?BCABC@C:<<CADDD>CCDCBB@?BBA;;;>A@A@@@?B?.77777+1422222....33
@EOUL1:00509:00761
GAAGACCCTAGAGCTTAACTGTTACTATAGGTAGAAGCACAGGAGGAAGAGGAGTAGTAAAAAGTGGAGACTGTAAACAGAAAGATGAAAACCAAGA
+
DD@::;:52,,,,,228;;DCB@@AACBBB?BBBB9<:CCCC=@?@B?BCD???@@A====;1@<<9=@@?@AAAA<CCBBB<???>9==10)))((
@EOUL1:00526:00772
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTT
+
<<8=CB@>?AAAAGCAC@DACBB996;A<B@BC@<<8;@CCAB>@:::5;?BBELADA:::::47?>=@@AA<AEDCFCFBBBAAB?BB=@@@@@@AAB>BBACC;<9@1998;BB?CD?;;:777;<>7>957533969=@===@@89><7..44----211..34433/
@EOUL1:00502:00829
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
BDDBCBCA??ABCBBAACBBCBBA@:79A>B?B?BC?@8BB<77344-8=>>@A=AACIAECC@@@BDACCA@CCD9;<CBB?BFCC
@EOUL1:00546:00912
GAAGACCTGTGACTAGGATTGCAATCAGCATTGTTG
+
A@<@9954@9>@CB;;;<C@BCGACA>900*)*00*
@EOUL1:00527:00966
GAAGACCCTATATAGAACAGAAAAGGAAAAACCTACCTCCGCAGGCGGTGGAAGCAGGGGAGGAAACCCCTTCGATAGGGAACCGGACCCTTTGACGGATGCCACCCCATACAAATACGACATGGTCCTGTTTTTCTTCTATTGTGCGTTAGTTGGGGTGACAAGAAA
+
ECBCA=C@CC;:<@A@ADA9888,9=BBGN3BA<<<@@A=@CBA?BA<AC888;CCCCB8BC@CC>???4?<???<<<?:?;=9@5;;;9;B>CAA@<>=7747>>>27776;?>@CCA=844434:3444444+73767;@=??A<<A>9743444-4;?7:8;BCC
@EOUL1:00502:01025
GAAGACCTTGAATCGCTTGTGGCGGTCAGCCCCTATGTGGATTACCGAACCAAGGATGCGCGGGAATGGCGTGATTCTCAACGCATGACTGGAAAGGTCATCGCCACCGAGGAAGAGGTCAGCATGGCGAAAGCCTGTGCCGAGGTGTTCCGCGAGGACTACGCCCAGAAGTTTCAAGCCAAGACCCAGAACGAGGTCGCGGTCTTTGGACGGATTGGGGTGACAATAGA
+
>876/1-/78;,<<AACBCEE>@>=@ECCCCC:BBBAAA>AB;?A=AA?BAC@C@CCCB>??C>@?BB?@=<<@B>=448?AAABBCCCBA:881636;???=A=@@>BBB?BBCBD?@??>=8;;8:;CC>CC@C111??101@>98AA?098<<8;;<;=<;<6;@A:==C9=<9<;2434444-4;;9>A:838>>;;94000+0237=;;;/333+39>>@?A888
@EOUL1:00527:01015
GAAGACCCTGCAGAAGAGGCAAGGAAATTGGCTGCTGCGTCAGGGTTGGGAGCACCGAGCGGGCCTTGTCCATATTGTGGCGGACCTGGGCATAGGATCACAGAGTGCCCTAAACTTGATCGTGATCGCAGGCAGATGACTGGAGTGAAAAAGGACTTTATCGGGCACGGTGGTGGGGAGTGGTAAAATGGTTGGGGTGACAAGAAA
+
CB@CBBB>BBCCBB?BBB@BB@B@BA@B@A@??@CCCCCBBBBA=A?CB?BB@@@@CFCCCC@B@D@CBAACCCC@BCC@BCBEB?BCC?C@AAC@C<=<BAAA<<<CCACBCACC?BBBBBCBBBC???C?BBC???CCCB?AAAAADDE2B>A@?@9:::AA:>?@C?AA@CDCC7A;;8:444804;;=<@@@5@88:F>CBBB
@EOUL1:00531:01065
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAAA
+
DC@C;9;;<ABABB?>@@>?@ECC7@BC@?8:?:8
@EOUL1:00539:01053
GAAGACCCTATAGATCTTAAAGGTTCACCATGGACGATGGCAGGGCCTCTGAGACGTTAGAAGAATCTGGGGAATTGCCGTTCCCGGCGCAAATCGTGCGGCCTCCTAGTCGGGGCCGCATTTTGCGTCAATATGAGTTGATTGATCGCATTCGGGCTATGACCCCCACTGTCGAA
+
/1,/66<8=8>>>999929969@B=>@C?CBA77376110444;5=;>?@A777<11-/11,/;:444499,15::11,//,/656:;BCCC@BBBB;;;>@;>>599;;BCCJ:C@B@9999/444>=@=@BB@ABB=@AC@CDCA@??A>77:5:?;?@<<=<<18311-----
@EOUL1:00500:01108
GAAGACCCTGGCCGGATCCGTGCGGTGCGCGGTGCTGGCATCGATGACGCAGGTGTTCGGGTTGGTCACCAGGGCAGCCGACTCCTTGGCCGCCACATCGGGCAGGCACAGAAAAGCCACATCAGCGGCGTTGAGCAGGCGTGCGCGCTCGCCTGCGTCCTTGCGGCGCTCGGGGTCGATATGCAGCATCTCGATGTCCGCGCGCTGCGCCAGCACCTCGTGGATGCGCAGACCGTGGTGCTCCTGACC
+
A>?B>>><<;8<@C=AA@=999>ACGBBCCB@BBBAA@A?@@CCBBCCCBCADC===?@@;B?=>@>==:444-4444;@AAA@=@?CACAED?;;;AAA@=?@@?AA7777:7.7?>?:;:BDBCA?B;;6;A@@BC@A@@@<==?<<<<=9<@@8889<8<@@8<<?;99=:-44;;;;>>=888?>?A88/55*)()(((++-?AAABBBBBB?BAAAC@CCDDCAA@<969?88----,*,,,31
@EOUL1:00511:01181
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCACCGCCATATGCATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCC
+
<;>ACCC>BDCBCC?CCCADC@B>A=@@AB@@@@BBBBB<<1111/::9?7:9>A;;B<?::9:0/.:854:9>979=:;;;77><7637>>3737<-706;:374899;;;<??CAAAA698<===>888@AA@A>@A<???;=9;///.-,--,------044>?>@BBBD?B@@<<<@=@A::<BCDBC>C@@@<@@=<?BBD@CCC@C@;;;:78>A???<9884888784
@EOUL1:00558:00108
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCACCGCCATATTGCATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCC
+
GCADCCC<@>>9<<:=BC@BD?C@D@?AABAA@@CC<<=BB>>;;99@@@@<<<CB?AA@B@=;>110?>>ABBBBABC@GB@@@@<;;=?=?;?:=?.:5:BCCCC@A:88;>88888>>588>@889A=<<@@?><>@4:9;3838@A888>?=9;440/0039=@99:BBCC<@=@<@@@=@BBB@@@=AA9<;ABA<;;69>>>;AAA>C?<;;;;;8@@9;9<AA>AAA786
@EOUL1:00570:00153
GAAGACCCTAATAGGAAGAGA
+
CC@DCCC@1/----*,*,,,,
@EOUL1:00554:00209
GAAGACCCTATAGACAATAACAATGATAATTTCACAGACATTGCCCTACAAGACCGTGTTTCCTTTTTTAGTAAATGGACTATCAATCGAAAATCTGGAAAGGAATTTAACATCGTTTCGCGTTATTTGTATGAAGATCGTTGGGGTGACAAGAAA
+
CCABBBCBCDCBB@BB@C?>A?;<<<<<>??;BB??BB>992::?9?BBB<A::377=994=477777*7>AAC>AA;;:??=@A>A?==AA6A==A=>>>A@;266-3/44888AC<@@????@@A<AAA???@@?@A@>/00008;?AA?ABCC
@EOUL1:00570:00238
GAAGACCCTACAACGTTCTGCTACTTAGAGGTTTATCCCTCTAGTACTAAACCAGTGTGGCAAACGCGGCCCACACTTCGAGCTACGCTCCGCGAATTCGCCTCCCTCCTCATCAAGCGATCCTCCGGCTGCACTCTTCTCCAAGGCCCACTTGCGATCCCTCCGACGGTGGCATTGGCAAGGATCCTTT
+
CDBFCCCABCCC?DECBDBBBBBBB@ABBB>BB=BBCC>B@@@AA?C???;>>>999>>>>>>;;;?@?@@;BBC:7::6//0;000BBA>BEA?<?<?AA>BBC<BB?BACB??;:::=<?=?<9<>@>>?@????8:8?=;7877828//,./-;8886:978///,/9-8/;>=./.330111,--+
@EOUL1:00597:00221
GAAGACCCATGATTTCATTGGGCCATTGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCACCGCCATATTGCATGATGCGCAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTC
+
<<59@?>:@@@@BDBBBBAECBC@CC@CCDEDC?AABBBBAA<<<;@@99;A>@?<>><?::9;0../85;=>=><=?<??@?>?;88;<;<:<;>>:@5::A7455-/,.7776;;4439/4356<<;???????A?@?:843/38:79983/73447.-,.149:=491111,/,/5649===::79;244-44;A;CBB=@>8848884848448:<?=>@>487;:;;>;;
@EOUL1:00568:00274
GAAGACCCTAGAAGCTTAACTGTTACTATAGGTAGAAGCACAGGAGGAAGAGGAGTAGTAAAAAGTGGAGACTGTAAACAGAAAGATGAAAAACCAAAGACT
+
CB?G@=@<@CCB@BCB=?<@@@@<@CCCACB=B7;:26=ACFC?@@>?>?==<<=C=A65533.84327;88;=9;94>>888/8??:::::/:,11,..*(
@EOUL1:00583:00330
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
GCCCA@@<ACCBAAA;::6;BBBE@BBBE;:5:;==??7@CCC=@AC@BBBA<969699:>B@@AAAA<ACC@DCD9CE@9959999
@EOUL1:00556:00360
GAAGACCCTCAACCCCCAAACAGTCGAGGAGCTCACCCAGGAGACTTTCTCCACAATCTCAGATGAAAGAAAGATGGTCAGAGAGGAACTGGCCAGAAAGTCACGCACCCTTGCCTCGAGCGGCATGCGCTACTACAACATCTTCTGGGTGAAAGCCACTCGACGACGCCGAGAGCACAGTACTATGTTCTGCGTAAGTCGAACGGCCTCAGCCACGAGGAGGCCATCAAGAAGGTGTGGGGTGGACAATAAA
+
CC?BBCD>CBF@EDDD3DC?B;;;CCA@=@CB<<<DDCCC@EDBBBA=>@@@CCCBCCAAACBB9990111,111>=>>=>CCCC@C@@A@=@@B???::78>>==99919699A999>>8848?@9<:CBB@@@;;59>>9961110,00..+..,3166997771110135:/..11128?@????>??A@A@B?A=774744249;;=;;794779::6,0,--110-0/02333555.3;530040000
@EOUL1:00583:00413
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
>C@DBBC>BAABBAAABD@D<<;C=?;;8870637:BB5CC@>;79908999999<BBB?<;;=?@=@:99429>?4;C889;444;
@EOUL1:00586:00482
GAAGACCCTATGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGCACTTCGACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAAAGAAT
+
BB?EBBC?A@<<<1//,/669>CC@BBB@??>ACCABBN@BBGBACACCBC<<8;AGBCCDC@BCBA@?@@AACDF@CCCCCCACB@??A<<<BC<<@BCEBCCC@@=@CBAA?CB@BBB@C@C?CCC;;0=>>0007071333334;>@@B=??@@@?B>BCBA@@7:;;?>>>>;???2=>8///,++*,
@EOUL1:00581:00540
GAAGACTGGTCACCGGTCGCTG
+
9959994::;44424894449/
@EOUL1:00580:00616
GAAGACCCTATAGAGCTTTAACTAACCAACCCAAAGAGAATAGATTTAACCATTAAGGAATAACAACAATCTCCATGAGTTGGTAGTTTCGGTTGGGGTGACAAGAGA
+
DDAFFDC@DCCBCB?>BB>B>@BB?A<?47807>6==?A>A?:=8<7<<???A>C>@>@>@@>AA>????AAA>CCDDBBAC?ABBAA<?>;>48884:BCEDABCAD
@EOUL1:00586:00637
GAAGACCCCACCCTTGATGTCGAGCGCAAGGGCCGACAAATTCGGCGCGGAGAGGAAGATGCGGTCGAGGCGGAGGCGGTCTCGTCGGAGGCGGAGAATTGGGTGTGCTGCGGGCTGCTGACTTCCGGCAAGAACTTCCGACTTGCCTGCCCATGGTTGCAAATGGAAGCGCGGGCGTGGTGAGGGAAGCCGCC
+
?ABDCDCA849499=@BCCCEBA@@@<;;;CAC=<=CBB8<;<C@BACA?AABB?C@BBAA995:???@=79711,1<:;>CCCDCG@BA?;;7;4;<>599048434;;444-444444497779959959?>BB>B>961101101661,1..,.7...443355-6677553----,-334403233/00/
@EOUL1:00557:00708
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCGACCGCCATACTTGCATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCC
+
@A=CCCB>CBBBBB>BDDCDDDB@C@CCBBBBB@CCCCC>>>>113??GBCC@@@@CCBG=;:;00/1/+4448999=<??@BBBBCC?B@?47477.737<;69=73333/033<66:A3/.5//-21-0080004480000/0/0440030232223...4..3BC?==::444-49;84442/66.../,4:::ABB=?@B@BABA?BBB@B76:1/...;;99=;775996670
@EOUL1:00565:00816
GAAGACCCTGATGCTGGTGTCTGCCGGTGGCACGACTTTGCCACTTTGGATGGTATTGGGGCCTTTGAAGACTTTTTGGTAGTCGGCTTGGGCGTAGATGCTGGAGTTGGGCAGTACCGCGCCTTGGTTGCGCTGGGTTTCGGTGTAGCCGGGTGTGCAGCTTTCGCGTCCGCAGTGTCGGGACGACGGAC
+
<DCBCCE@CAAACCCC@CBB:;;?<?;?=:::::8>@C>A?:>777.7:<=;:;AC@@BB9>>><811,///88:;/;;?999?B@BB>=B<@A@ABAB;:;A>@>@7<;6;A??@D?BBB?5;7969@@A<<;CB>BC?BB?AA>?=B@BB:?;::<<>=??<?C???C?A:;4:;:444-3422--,-0
@EOUL1:00574:00994
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
9969@@@<BCCCCCAABB>BCCCC@CB::7659=A@@A6?;;:479A:??>>995<ABKAJCB<9999099><ABC8B@@>B>@B@@
@EOUL1:00591:01000
GAAGACCCTATAATACAGTACTAGGGCCAATCTAACTTGTTGTTATCCGTTGTTAGACGAGGCCAAATCTAACAACGAATAGCGTGTTGTTAGATTTTTGACGATCGAAACGTACCAAATCTATCATTCGAAGTTTGACTTTTTTACGTATTTAAATGGCCATATTTGTTAGAATAGAAATATATCCTT
+
CC?CBBC@BBBB@C@@ACCB@B;;8/8<@ABDCC@AA;996<A>BCFCCE@CC?B=9?B9;6<BBDACBBB>@@=@BEADCFAAAAA=@DDBBBCCCC2CCCDBBA;;909@>>><??:?AAABBAC?BCB=??@:=3....//(0..---10394;:8893334848<<==>9?9000+000003242
@EOUL1:00553:01077
GAAGACCCTATAGCCATTTATGTATTAGTAATGGACTGCTGCTATACAATAAAGCAACACAACCTATCTTGCTTCTATACAATGAAGCATTTGTTGCTACACTATGTTGTTGTTTTACAACACATGCAAATATATCGCTTGACAAGATTTTTGAGATTGGTATAACATTGGGGTGTTGGGGTGACAAGAAA
+
CC@DECC>CCBB:::<EB>BBAABBABBBB?CA@CBBBB>??>>9==@>@AA=A??>?A@@><865;;=38;C>AACCBBB?BB@>??BBC>BB=ABCCB@<===<<8;;4;==?5@AA>>==ACC===<AA8<<>?>><????>@@AAAA1@@A@@>@>@@@@478?<=>@1;;92744-4;==?>@@@@
@EOUL1:00557:01080
GAAGACCCGGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGGCCAGTCTTCTCAGGGATGGCTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATCAAGTGTAAGCATGTTGTTAAAAAAAGTTGCGTTGGGGTGACAAGAAA
+
BB?C??@<@>AB<=>A@@@@89;><A@AAABC?AA@CBBB=>>>?>:>==@????D@CB?B=9;,,+,270,+-,.3433304/1,//877;CDACC@ABGC@@@?@CBC::>FADCG?>>?122>;;;?A@>><<;B@<>>4999999&9?<77391333,1111666666
@EOUL1:00594:01093
GAAGACCCTATAGATAGCGGCACTGGCGCGGAGGAGCCCTCCTCAAGCGCGAGAAAGGGGACTTTATTAGCACAGATGTGGTCACGCAGCATAGTCGCAAGG
+
9=;?444-49899@ABBBB@BA@@B?????7::==?AB;A@?AAB?BB@@?AABBCBLK6@::7.760104433---2--,.3455==<885330--+0*+*
@EOUL1:00566:01191
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGTTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCCTGACATAATACATTTGTTGAGAGGGAAGAAGCAAAG
+
>B@B@@@<>@?99969918>@@AB@@@@CB>BCAB?A@CBE???<774444::9;;==><?=@=;;344-4999;<<=><=8488AA?@A99959954>=8==?A=B?C::::7<<>5668??7<<BAA>BAAA<@>@;AA<;;BB>8;;:57=@=:::<<=6<1334444..,-311<@@A@B7779631??=>@>?4777.7=:8<<;==;44777694000+034438:>B71677-69::0/0/0,/979-448836
@EOUL1:00577:01174
GAAGACCCTATAGATCTTAAAGGTTCACCATGGACGATGGCAGGGCCTCTGAGACGTTAGAAGAATCTGGGGAATTGCCGTTCCCGGCGCGAAATCGTGCGGCCTCCTAGTCGGGGCCGCATTTGCGTCAATATGAGTTGATGATCGCATTCGGGCCTATGACCCCATGTGAATGAAGATGCAATCAACCGCGCTTACGTGTTTGCGATGCAGCGCCATGGCAGCCAGTTGCGGGCTCTGGCGAT
+
=@>999><A@@AA@>30,.6+.,.,..35/66=>@>BBC@AAADADCAB==99444424:;:442444444,48889A@4964;6:899444;5889944424944249;49999/9=B@99904:::>>?A8>>>70,./,,,,,45444333+1003343333+0499992:9B699B=946:@98=@@@@BC>>>>>888,3333674/6...,.4777713111133...+..-22358<<
@EOUL1:00552:01214
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
CGACBCC?BBBB@@@@@@<@CCCCACC888<=98@@AA7ACCA>AAD?CBBCAA>@>BB>BCCA@@@>9>998;FF:DCDCDA;888
@EOUL1:00570:01226
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGACACAAGGGGTCTTGTGTTCTCATGAGATCCCTGACATAATACATTTGTTGAGAGGGAAGAAGCAA
+
AA<A==><@@AA@@?@EAC@>@@BBBA>@>;==7740:<:<.10;;>,/,.44/8846///;84:;344-499;@@>>?599?A>CAAACAAA@995<>>>@;;;:<@BCBB@@@AC>@AABD;<=CBA<A;;=8;::,11@@@=@B??::5:4747==744,4>>A>AAA@@;=888:;@BB?BB::;;=@::////,/991;98./7;@;>=A9;/..42885=@A<;?=:7::::B>CC??:;88818=@@>@C@@
@EOUL1:00580:01240
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAGA
+
:@<@?@><ABBCC@@>?@B@B=?@A>ADADABB?BA?C=?@?B@B???<A@CC@C@C???CB@@?<;<CAFD?CAAA?CB>BBBA@@;;7;BC;<;BBB@CC@CAB:9908@9@?B;<;@EBBBBCCCB<?>3737==?B@???>=>6<<;<@@>A==:<@@=>::////./9=4;8;78./.:9
@EOUL1:00564:01251
GAAGACCCTCAAAATCCTCAACCAAGTTGTTGCGACAACCTTGGATACCTCTTCAAAAACCTCGCTGTTACGACAACTGAGCAAAAACGAAACCTTACCTACGGGTATCGTAAAACCT
+
44249>?:>8999/9D>BCCAC=@7;9;>@6;;@@ABIC?B<?599137>?=;=>>>>/>68838><<8<?@BC>:;000......*-33303.4/04/44730+0330003;;-423
@EOUL1:00617:00031
GAAGCAGCAGCGCTTCATCACGGAGGACATGATTGGAGGGTTGATGACAGGCATGACCAAGCTTCTCGTGAACAAGGTCAATGAGTTGGTGCAGCCATTGGTGGAGCGCGTCACGACCTTGGAACAGCGGCTCGATATGCTACTGAAGGCTACGCAGACCGGGCAGAACAACAAGGAACCATCGGCAACAAGGGCGACGCGAACGAGACATGGTCGAGCCCCACGACCTCTACGTCGGGGCGCGCGTGAGCTCCGACCAGCCCTGACCGCGGTTAACGAA
+
@@?BDCCBBBCCCC@BBA@@CC=@@?BA@A@??;?;??A<B=A888??AA>BAA?<<<=<?@@=@@@@AAA>@B9=9<=>8;7888<?<AA???B<==999==AAA@C?<?A889AA>@=@@C>@@@AA?A????@<<<??AAACA=@=@AAA=8=AAA9<<7;;::34/6//,/46,/./664978587588,//8::8<<>===>=??AA>@@@;4444-4:;:;8<==<<==@@CC599000.666../756//,4444,7::=<?==<?24/444:
@EOUL1:00625:00086
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACACTGATGAAGCAAAGGAAGATGAGCAGACTTGGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCTGACATTAATAC
+
CB?B@@@<>>?>A@;AB<@=@A???8748?<>>;<5:7:@CA??;::,0,0><>?DA=<B;8888<@>AB?A@=888=:;8<788<BCBCBCAD=<989@@ABEDC?C@BBBECCBBA:?@;;;CCCCCB@DCDC>C?B?BDCB???>>888477-9B?8=??B;<>>A<;>;6:59437;@:::A;;B@@<?;;579;<;9@B5:::@CCCC@DCBC??777>>;6117370717,,
@EOUL1:00630:00088
GAAGACCCTATGGAACCATTGGAACAAGACGAGTTGATCGTTTCCATCCCTGGTCATACCGAAATGATCGGGGATATTCGCCAAGCAGCGCGCAATCGAGTACAACAATGGGCGGAACGGGTCGTAATGATCACAGGAATAGTCAAGCTATCATC
+
@A=99;>:>?@B@D?C@=<6;9;94969:33;;?:6>@?949-42449A?CC>@AHBBA@AAB<999444=>6=BBB=;7750,.65./....68?>?9666//,/777::53314248;-44937777;;<@@?@>;211111677111033..
@EOUL1:00608:00152
GAAGACCCTATAGATGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGGAGTTTCGTTGGGGTGAC
+
:429999199=>>>>=444-444489977733135617>5;7331144-44444=<=<BB;==94244885774,...*.4..
@EOUL1:00635:00298
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAAA
+
9969999-444;8>@@@?:?9??A47617701177
@EOUL1:00642:00347
GAAGACCCCAAAATCATATCGTTTACGGGGAAATGAACATCCGCTTTCAGCTCGGCGAAAAAATTCAGGACTAAAACATCTTTTGAAAGACGCTGTTCCAGATTATCTAGTGAGATAAGGGTATGTTTTATTTCATTTCGAATACTCAACGGATCGAATCCTTCT
+
;;7<BBBB9BBB9BCCBBB<<<C@CGCCC:FC@CA@=AC@><>99919@BCBBD?AAACDCC-C<>>E?BBCCCC:CBB@@@@4?@B19<>999AB>ADGCBB=??><??;::BBBC@DB>??;:7>>2=??<888C<=<=9<<?=99530234000/00/3424
@EOUL1:00647:00391
GAAGACCCGGTGACTAGGATTGCAATCAGCATTGTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCATATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAATAGA
+
CCADGAA>B@ECFCDCB@?@=A@@=ACCCCCB@CBCCCDB@@BBB=BCC@@@BBC>CA@C>>>//1;?>>@B?AACA??>AA=<:??@A>??>8737?==:>>>9==CCECFDDDIDCCC@C@CC@CCCBC@CBCCCC7;;=>@ABC-CEABBBE@BBB:CAABB50000
@EOUL1:00604:00429
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGACGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTG
+
AA=AB@=<@=ABCC>BBB=@@<@>A?BCCCCDC@CDBBEBCBABCEC??>@:;9@>;AB7;1<:<000<;6=@@@@=@C;BB797;??A?@???B?CC<C=?@?CCB@?:37/3877///.99;7/87473>99000*+
@EOUL1:00614:00460
GAAGACCCTGAGCCGCCGCTTCTTCGTAGCCGGTCGTCCCGTTGATTTGCCCGGCAAAAAATAGACCCTCGATGGATTTGGTTTCGAGACTTGATTTCAATGCCCGAGGATCAAAGTAGTCGTACTCGATGGCATAACCCGGCCTCAGGATGTGGGCTTGTTCCAAGCCCAGCATGGAGTGGACGGCCGCGATCTGAATGTCTAAGGGAAGGGACGTGGATATCCCGTTGGGTGACAAGAA
+
@C@DGCD@CC:>>BBB>ECC@DC@BB@;<<?B?BBB;;<:;BACBBA>BCC>C?CCCCCC-BBAAAA<ABABAD@<;;7;?BC>CDFFDCC@<<?DACC>BADDBNBF?BBCCC?BBCCBBBBCCBBBC@@=;<<A=BB8<8<?ABCC@BAA99919959A@B?B@BBC@CCAAAC@CECDCDBB?B;<<CC@@@@@=???@AB@BC?CBEC@CBCCC@AAADCC>>:9:;,,,,,,+,89
@EOUL1:00617:00548
GAAGACCCTATGGAGCTTACATCTTTATATATTAATAAAACTATTGTAAAAGTGTATTAATTTAGAGATTTCGTTGGGGTGACAAGTAAGA
+
EC?A@@959911,1///,////4//+/....6,3136662444;943333,3777;@?@>@@>876;899594424992999442499899
@EOUL1:00601:00573
GAAGACCGACTACCACAACCGGTTCGGACTGCAGGCGGTCGAGAAGGGATATGTGGTGTTCGCCCCGGTCATGATGAACGCGGGCAAGAAGCGCGAATGGCTCGACCGCAAGGCGATCATGGTCGGGCAGCGGATGCAGGGGCTGGAGCAGTTCAAGATGATTCGCGCCGTCGACTACCTGGCGTCGCGCGCGATGTCG
+
>?=?9929@944@=99;69<?@B@CC>;;7778><>><>?7=>>477.761=;;=<874:BCCCC9C?B@BDECBDH=<<CBB>==;=DBCC=<=C@BB@BBBC;;98?><?<>?A::<A?<?99908883?;97116677-66601>7437334798???>;;7,++*++++1171624<59==43.26//---110-
@EOUL1:00618:00573
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
DEFCCBC@C?A?@CCCCC@B@==B@BB>>>B<<<<BBB9CCCC?ABB>BA<<<C@B8<<8<BB@@@A?9>9949BB6ABCDC=@A>>
@EOUL1:00632:00562
GAAGACCCTATAGAGCTCCCAGGCGTGGCCGTGAATCTGGAATATCTCGACAAGGTCATTGACGAGTCAAATGCGTATCTGGCACAC
+
DB?EBEECCBBABBBCA;;<??<??>>3:6::A@@CC@@>@>888@=4477:4;>BBCE@DA@@@@@@AA=BA???@8200*,**((
@EOUL1:00638:00589
GAAGACCCTCAGCAGCAACAGGGGGGGCATCAGTCCCGGAATTCTCCTTATCCTCAATAGAGGGATTCTCTAGGTGCACTTCTGGACAAGCTGACTTTTAAACGCTAACCTCGCTTTCTTGGCTTCTGATCGAAAGTTTGCCTTGAGCGTGCGGAGTTCCGGTTGGGGTGGACAATAAA
+
9959:=@>CBCBE@=<B?99999994&49@BBAAAD9;8;?B=@AB=?=??@<@?@;=>>99919968A?@11-111.6;9?BB@@AA@CCBB?9999/99199@9969<>>>99909@<A?CBBCCCCCCDCB<::;68><AABCCCC@@>>121:9;--+/8=770--*///,/444
@EOUL1:00649:00585
GAAGACCCTAGCGGACTGTGGCGATGATCCAGTTGATGAGGGGGATGGCGATTTCGACACCGATGAGGGCGATGATGGTCATTTCGAGGGTAGTCGAACGGACCATGTCGAGGCGGTCGAGAATCATTTTCTGACTGCTC
+
;;7=ECC@BBBCC?CBB@@@=A@BBBBA?599959@A@99999-9>>;@ABBDD@?@CCC@@@>999>9>9<=BJDE@BBBCDAC3=2?11122@;@A@A?@B>?>?:;7=>B775:961?@?A?997=-6..,,*****
@EOUL1:00610:00621
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGTCTTGTGTTCTCATGAGATCCTGACATAATACATTTGTTGAGAGGGAAGAAGCAAA
+
9954@@A:@A>999=@C?BABB>==DCCCC@C@??;?;?=@>AA===4736;<66:=@@@4848?@@@@D<@A??=;;4434====CFCAAA<<69?<99;;;;BD>C;;377339?B8?BBBDAECCCDAC<<<9;BB>AA?>>>801100,.5577BBAAA<A>9444---44408/0//.55<8<:B888C@CBB?:::6:/,+****)*+037466633/3555<>@>B@FAA?<;84870003,49;;988//,
@EOUL1:00620:00653
GAAGACCCTATAGAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATAAA
+
A=;@@@=:@?@@9448595999>>?@B>A@AGDCD@C=@@BB?CCCE>C@<<8;>B?@@AAC@BB?BDBIICAABBACD>AA;9959969@@@A@@@A@CCACCCCCC<@@<A>A>BB?BBBBBB??=6?@5747771;;??99<C<A88:=@>@??<?AAAAA9;;//./66.85;;9/3000
@EOUL1:00646:00824
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCAACAAGGGTC
+
9959AA=>BCBCCC?BB>B?BAAABBB8;:????C?C@C@C:11;=@,01>>:;A>ABB>@??===?AA<=<<?<<<=@<??AA><@@DDA>@><<5<@@@?@BB<@@@CCBBB???9?@>AA?AAADC@CCCD;<<;@BBAA@A:;::7A>?6;6:A>=<=7<<AA>AA@AA9<;;8::??>A??;::<>;8,/0,0/,,5055
@EOUL1:00625:00956
GAAGACCCTATAGAGCTCCCAGGCGTGGCCGTGAATCTGGAATATCTCACAAGGTCATTGACGAGTCAAATGCGTATCT
+
CG@BBBC>A@@@@@?@?AA=::5:@@@<@=@::8;==?A:<:<D??:8<FC@E?@@AA>@=<<7873444:8684;44/
@EOUL1:00636:01065
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
DC@BBBNDBBBD?@<<DH>DCCBB?A>>:0/,0:=>@?6CCKN@AA@<>?@@9969=@B?AAA==<BB><<<=ACC9BBCBA<@999
@EOUL1:00644:01072
GAAGACCCTACGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCGACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAAGA
+
BBBC@@@>@ABEC:<<@DB>BC;;;;BBE?==?A??;?@<ABBDFCCCC>CCBBC@@@>;>=8==000/;;<@@@@<@AACCC@CCCCCBCECCCAAABAA@BACBA>??@@@@CB@CCC@C?A:?:::AAC@DAA?A::;?A????@@<@@CD@E@@@BA><;<B@B>::7:A??;3888-44961++,
@EOUL1:00616:01115
GAAGACCCTATAGAGCTTACTCTA
+
BB?BCCC@BB@;::0..,,,,,,,
@EOUL1:00628:01176
GAAGACCCTATACAAGAAGAATTACTGGGATGCCGTGCGAGGCGATGAGCTGCCGGGCGGTGTGGATTATTGTGTGTTTGACATCGCCGTGAACTCCGGCGTCTCTCGCGCAATCCGCCTGATGCAGCGGTCTTTGGCCTACGGGAGGACGGCGTCATTGGGCCCGTGACCATGGGCAAGGTTCTTGCTGCAAATCCGCGAGAATTAGTGCAAGATATTAGCGAGAAACGCTGGGCTTTCTGCAGTCGCTGGCAACTTGGCCTACC
+
EHCCC@A@AB@@@AABD@AA>A?AACCDAB;;;@CBBBBBCAB9;;CCDCDBC@BB>BCACEBB@CC9<;@B@@@A@@<@?>==?==:???ABBBCBCACCAABBBBBBBBB=@@>AA=@??????@::57>??.73:>AAAKEA??<?@CADA?;988;;-//,/78//,//6616:6=;=9<;;=;;;034-88488767B?A?B??=473100882866,,,,00,,++++*.4557;;<=<@@98548733348;:>;:77,
@EOUL1:00644:01157
GAAGACCCTATGCGACAGGTGACGTTTTTCCCTGAGACCGGGGAAACACAGCTGCGGCCAAAAGCAGAGTGGGTCGGCCTCGCATCTCCTGGACAAAAGAATGTTGCAAGCATGCGGTAGCAAGAAAAAATAGGCGAACCGGGTGTGTACAGCA
+
4959ABG>AACCB>>000:;55<>?AAC2A@>ACFEAA==@;6@@9>899;A>>AC@C?BBA.111111111,11101;<<<=ABBBC@CD@C?==<089,///,./.,---32//,-222<::;;;;=-<@?98849;799444004000311
@EOUL1:00631:01233
GAAGACCGCATAAAGCAGGAGCGGCCCGACTGGGTGGTGCTTCTGCCTGGAATCTGAAAGAAGAGCTTGTAGCCAACTGTGCTATGTCAGCGAATGGGGTGGCAGATTCGTGACGGCAGTGCCCTCTCTAGTGAGAACATGAAGCCCCGCATCTACTACACAAAGCCTTCCATTAACCGAACGAG
+
;<<<=BBCBB@??>ABCCABCAC@CC@ABC;;;6;B?BCD>;>=10,0<;>:>>CCCD@CC@@@?AA;>>>9959@A?BCGCEFCDCCDBBA@<@@@@7ACACC;<;?BAA:;:C;??>>43717699:???>===;<<<<@>@CCC9CCCA==@B@@===@B;A@>@?=987,/,/,-0*++--
@EOUL1:00648:01226
GAAGACCCGGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAAGAAA
+
CB@CCCC@C@CCCCBCB@EB?BBB?ABBBB@A@BB@BBBA?CBACA@@BB?<<?@=<;;>B?=84437>;=>??=@@@@AAA===A@@@=A?AA>AA=A=?@>B==;=AAAACCCCCCCABB8:8;C@CBBBC@;<<DCB<@?488888)8?<ABAA?ACC9CBBAA34444
@EOUL1:00606:01264
GAAGACCCTATAGAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGGAGA
+
CA>==ABBBBCB=97<4928334C@CD?C@ACADDAC?BB@C?BBBB=A@B>?B?BB@@@B?>AA@?A?BD@CBBB=?=<@C=@@@::8;BBBCBBBC@BB@B??4447:?9@?C7;;=;;;AAACCB<AA:=:=7::@A@A@@@C<A@@<<==@@A>AAA?=<<=><<:=CC6A@@@A583866
@EOUL1:00611:01261
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGTAAGAACGCTGTGCT
+
CHADCCB>A@9999868@B@CAABC@CD>B@CB>??<??AC@C@C?>>7@47:6;7:CCCCDD;==<?DCDD@CCBB8990999AA>BB?ABA@A@@@@?A@>AB;;;B=BB?B?B@BBACDGD@????.7737;=7:::4999484,--//-----
@EOUL1:00655:00126
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCACCGCCATATTGCGTGATGCGCCAAATTAGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCCC
+
GCEDCCC>@@@;;;;=DE@CB?B@C@CCB@AAB7<<BAAAAB<<<<AAA@>?A;<58?=?::9;0./::;B??BB@CC@CC@CAABCB?=7747:==6=;7::776><43486888899C:=<<=<8888889999244-49;=A>A>AA9948>;;>83/7;8559885./,.7,/,--013>;88=@>AB?CB<@@@A9@>>898>>4;;CACADCC@??877>>;?BA;>>?777.
@EOUL1:00689:00131
GAAGACCCTATAGCCTCATCGAACTGCTTGCTCGCGTCCGATCTATTATTCGTCGGTCGGCAGCCACGGCACGAACCAAGTTGCAGTGGGCCCTTTGGTAATCGACCTGGGTTCTAAAACGATTACTTC
+
9969494-4;;1/171///4..,0A??@CDBDCA??:;4887=;6:50.,....46;68:8779888B@B>B@=;>3736999>??<==;=@;BC::8:@=9<;B@<111,/,/////*///4243313
@EOUL1:00677:00191
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
?A=<<<C@CCCCDBB<@@8<>>887;BB@BCAC;@@CC9CBBB;?@@<@ABBCC@C?BB<@>@@@911,111-1<<011.//,/847
@EOUL1:00679:00190
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTGCATATACAGTTGGGGTGACAAGAAA
+
IEAEBBC<@ACDDBB?B@B@ABBCC=@@<@@BB@B@>@>>>>?;;>??;B:<<@@????C???>AA@=@=@C@CECC@CC>DBCCC@??=?CBBB?>@C@CC@B>=@@=:=>9>;@=@@?BBBB@@?>>98857:=>A@@>>77444-44478:5866,.----13464454==4:<:33/0444
@EOUL1:00691:00177
GAAGACCCTATAGACATCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGTCTACATATACAGTTGGGGTGACAATAGA
+
AA?BCCC@BA>A11.....,..58>@BD=C@CC@CB>@>994:=>AAA>CAJLAC:;;F@A::8;C@BCABB>BA@@=??8?@88;?BC@EDC@@@CAA?AB?A?????6;?4:5:=AB=A@BCCCC@@9774424;@@AAA???@D<@8/.,,+,,,,,5005898@A@<???4==>78466//
@EOUL1:00657:00227
GAAGACCCATGATTTCATTGGGCCAATGCATGTTGCAGACAGTCGCGCTGTCTCGGTGGCACATATCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGTACGGCGACCTGCTGCAGGCGGTGATAGACGTCTTGGGTCAAGGCCGCACCGCCATATTGCATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCC
+
FEBDBBCCCGCCCC>CBB@CC@A@A@ACCBBBBCA@AEEDCBBCCDCCA@@@CBB?BA><77888777-6<>>><@C=CC<<;<ICC@B@>=?377.73777A??=@BBB?BAB@>=;;79;79;==888<AB8;.//,/88,/,/776::8998;;=B;==<<<;@?@@<;<6;?F=A?@=?AA???@?AA<A;880;>>=@AAA@CCC=@>@ABB??@CBBC?@???AABBC@
@EOUL1:00668:00285
GAAGACCCTACTAACCGACATCGATGATTGTACGCCTTTCCCTTGTAGCCACCCCGCCGATGAGCAGATCGAGGACGGTGGACATGTGGTTTGTACGGCGTGTGGCCTTGTACGGGCTGTATGTACGAGAGGCCTATATTTGTTTTACCAAACATGGTAATGACTGTAAAGCGCAAGCACTTTTACCGGTCGGAGGCTTATCTCAAGACACACACTCAAACGGCTTGG
+
?:29999-49449>A><>>11199;>AA?@9999959@19911-1<--,22222*,,00,//1948A>=?9993...,..34//9>>AA?B>B@@875:B@?AA?>764////;;9;EBBAA@A<;9>?;858=998888/87>>5BC@CD>???C@CC?BBCB=<=AA7<:::A<??AA@@@488389<;A?;;7443999>>>;ABB;;<:000/00,.791.,.,
@EOUL1:00684:00301
GAAGACCCTATATGGTTAAAGAACGAAGCCCCGAAGACTTGGGCGTATTTAGAAAAGAATGGCGAGCGTTTGGGGCAAAGAAAAAGCTCAATTTACCGATCTCAACCCAAATATGCTGTCTTTGGTATTGGTGAATATTCATTTGTCTCCTACAAGATTGCCGTGGCGGGTTTGTATAAAAACTTGAATTTCGATTGATCGAACCGGTAATGTC
+
CBBDGCB@BCC99;=@59903><98944799,34244449@@6::CDBDA99999/8@59;@BA==<<<B<BCC7BBB<@>@@>1@@;;8=??8??<?B@?=66011,177>@=:AACCCCC>B:6613671117714/4=>>76166731334/00424@<@992989-44-48844444+43777133,33166611111010101.4...-
@EOUL1:00686:00307
GAAGACCTATAGGGGATGATGTGCCACAGGTGCCGGCCATGATTGTGTGCGACATACACTGTTGAAATCGTGGTTCAACCACATCTTGATTGTGATTCTGAACGGATGAATGCAGCGCATGTTTGTAAGAGATCGCATCTATGCTTGTCTTTGTGGTGGTGTTTTTGGCAAGCCTGGTCT
+
BBAAAAAA@>8844/9>>>999>>43799@@AA<@<>;?667>76876336666?<==@BDD@CCC@CB@>>5859@<?=?8....,...34468888996799216<=:??BBBBC;<<FCG>BAD@DA><100/7:;=DACCC<<7766,38716737?CB??,73/.,.-,-.,.-3
@EOUL1:00658:00383
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGTAACGCCGC
+
AC?BBBB>BCA@@A@=@@C@CBBBHCCC@A=@@@8;5:7>=;@;??;B8@6::<>=;:61<7747>=;==<CAA<<9<>?19;@@@?AA;@>>999=;>?BC6;4849904;-424/,///,..
@EOUL1:00681:00381
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
CD@CDBC?BBBB???B??;?BBBC>??@AAB?BA????7@B><<@@B@BCCECC?CAFI@CAAACCCE>BBB?BBB9?@@??;8899
@EOUL1:00670:00412
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGGCGCTGTCTCGGTGGCACCATAGCCCACTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGGC
+
//,///88>@CCB@:>9924;-959<>A@@@@A?BBCEFC@@@@@=@AA::;@@??;8848985873344-434:;2//+//,/646376;A>;>:=?9?3;@;/0/52***)++*+
@EOUL1:00679:00451
GAAGACCCTATAGTGCAGAGAGAAGTATGTCTCAAACTCCTGAAGTGCTTGGACATCTTGTTCTGTCCACGATCTTTCCACTTTGGAATGTTTGGGAGGTCAGGTATCATTTTGATGGT
+
AFAD:;;;<C@@ABBCCCBBBBB@BBBB???B>==9=00,0:<:>BBAB@A;<>CBGGDBB@<;9;>=999??88;6<=??DE>@=@@CCCBACC>::5;A??>A7113...*.333,.
@EOUL1:00655:00505
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
BB?GDDDACBBBBA@?@?AABB<?BBDB9:=<D@FC@D;;;?B?BBBC?B?BC@C?CBB@??C59:=:;7<B@<<>B@DA>CA999;@A>BCCCCBB@@=AA@?@@99919@:A?D@CC?B<<<@BDCJ?BC?B?A?@@@@CAA?@A<A88:?=:::;;=;;=A@@>=::3444-4;;::34444
@EOUL1:00672:00562
GAAGACCCTATAGAGCATGCTGAGTAGTGAGATGGGACGATAGTTTGCCTGGTTGTCTGGGTCTCCATTTTTGTAGAGGGACACCACCCGGGCCCTCGTCAATGCCTCCTCCCAGTCCTGTGTTCAATGATTCTATGAGAATTCCCAGCAACTTGTCTAGATTTGTTTCATCCAATCTTTTTCATTTCT
+
DDDDCCB@CCABBBBAB@@@BAABBBBCB???AA?9????==>CC=AA>@@4738>=??A<@=>=>ACCDC2BA??BCC=BAAA>@@@<DCBCD@CC@@>?9://,,22,,,*,.-3455:9=;675776>>A?843300/4233,33777;;5777344====?BB;==44231333333+3344-44
@EOUL1:00681:00650
GAAGACCCTATAGTGGTTTATGTTATGTGTAAATATCGTTACACTTCTGATGGTTTCTTAGCTTTGCCTGTTTCTGGGTTGGGTAGAGAATATATCGTTGCTACATATAACGACATGGGTTGGGGTGACAAGAAA
+
IEABBCHBAAAA@AA>BC>BCCC?BB@@@NGECBBCBAA?CBBBC@C??::<=@C>CC?CCBBB=@AI@@@D?@@@C>B?BB>BBCCBCCBC???><<8<CBBBCAAA@>@@@@>;;8497999-4/688,//66
@EOUL1:00667:00914
GAAGACCCTATGAGACCATCAGAGGTGACTGGTGTGTAAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCAACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAAGAAA
+
@@=@AAA>B@<<:1//688;>>CC@;:;???:777?7999/9A@>>@;9808??>@@ACC@BB<@?@@:;<BBBCCE>BBAAA?6:;;;;AB?@A<<<BAH@@@;;;A?C:<<B@CE@CGC8:7:>?>>:::>;>>?=?;;;CDD?ABCC?DAB?@@@ABBC;<<CB<<:00,0:000,44407:=>A=??AA
@EOUL1:00666:01036
GAAGACCCTATAGTGACACAGCAGCGAAGGGTGTCCTCAAAGACATCCTGGATGGCTACTTCCCATATGTGCTCAAGCAGGAGTTCCCTGATGGTGTTCCCATAAAGGCAGGTGTTGGACTGGCTGGTTGGTTGGTTGGTTAAGGTGGTTGGATGTCAGTTGGGGTGACAAGAGA
+
CCABBBC@CBCCC?BBCCBB>C=77//-0>6=?@?;BDDD>CB@@@@>@@8???;=;;:;8;;6;=>?899CDCD@CCCC@CCC@CC>CBBBB@DCB@BC<AACC>B@ABB@BBC@CB=<<A=@998;;>;>4848;?<?@C@CD>>>@??;?11/>>@@;>>>6?<??B?;;9A
@EOUL1:00672:01091
GAAGACCCTAGGCAAGGCAGGCCGGATCGCGCAGTCGCTCTTGCGAACGGACCTCGTGATCCTCGATGAGTTGGGGTGACAATAAA
+
ABACDECCCCC@BC<<69AC>A?C@CEBCCBC;<<CDAAAB>:::>=<>:11121>@BBEBB@?@A>@@::67FB8;;:??<CCCC
@EOUL1:00679:01082
GAAGACCCTATATAGAAACAAACCGCGACACTGATGCGTGCCCGAGTCCTCTACAATCTCGGAATGAAAGATCTGGTCAGAAGTTGGTCGAGACGATCGACAAGAGTCCTTACAGCTTGACTGAAGAGGAGCGCAGAGTTCAATTCGAGAAGATAAAAGAACTTAAGGACGAAAAGGACTCATTGTCATCTGATGTGTCAAACTTGCTTTTGAAAAGGAATCTCAACAGTACCCTACAATCACGAAAG
+
CC@DCCC@CCB@>>737/99919@@@AE><<CBBBCCBBBB@<BB<<=@DDCCCEBDA@@B=@=ACCCA@@?A::6;ABBDACC@B@CCCBBCCABB@@@@A??AAAA>A=AAAAAB888<<??<??@>@@AA===@@?;==2424:=<;8:;4444-4:7=F6:689<9?;??4;.//8///9;>>888?>?=??@888=7=@<888AA04744,028/0003325;33366333400++++00003
@EOUL1:00690:01052
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAT
+
>@<9994-4998=@BACC>CC>00,00/01636=1677,3?7<:<>@1994?9959?BA:<<9;99:::611,/88*.../.,.
@EOUL1:00690:01090
GAAGACCCTATAGAGCTTTACTCTTATAATATTAAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
49599=B>BDCDA@@BBGAAAAA?:?0/,093,66,1111,37767144-4:444:?@<AA<?=BA@@@@:>9969@@7ACABA=A;@A
@EOUL1:00658:01102
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAGA
+
BBACDDC>CDCEBBB@B@B@BBAAA>AC<@=@C?@==?:;;@C?B=994?@??=??BABCB@@<<<>8A>AI@CCBA@BC@CC@@>@BB@?ABBBBBBB@CC@CA=48809>9A@D>@@?BBEBBBCEC>:74747>>>AAC@@@CD?@9:9=<<?@@<;;;;888<<<>>AA@5@A===.0/::
@EOUL1:00672:01128
GAAGACCCTATAGAGAGTCGTAGACACGCAGGATGAGGGTGACAAGAGA
+
CBBAAAC@DCDBA@A>=00099996=<BDCEAA?88880888>@?ACCC
@EOUL1:00691:01131
GAAGACCCTGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAATTGTAAGCATGTTGTTAAAAAAAGTTGCGTTGGGGTGACAAGAGA
+
A@<@999-4449>@@>?=@?4387437@@ABC@CC@CCBB8:;@?D>DAAA?@AAC?@@=AA@@77434;::32743:8:???AAA@@@@?AA?==9<>;97;8838<@==BBAAA@CBBA81,/;,0,00><;====:8882444444&33777;@?AAA7A?==?@;<<6
@EOUL1:00676:01206
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTG
+
CC@BBAA=>BBBC9888?B?BCCBB=>?;@>BB=?@@C@CB?B>@===;@;???CDC>AA@@A?@=<=DADDBCAAA=??<@B?===9748?AAAABBD?BH@CCBBBB=CC>C?B?BB@CCBBABA?=9<<3737>>>><<==<AC;?==;33/48848>7433.----,-
@EOUL1:00686:01240
GAAGACCCTATAGAGCATGCTGAGTAGTGAGATGGGACGATAGTTTGCCTGGTTGTCTGGGTCTCCATTTTTGTAGAGGGACACCACCCGGGCCCTCGTCAATGCCTCCTCCAGTCCTGTGTTCAATGATTCTATGAGAATTCCCAGCAACTTGTCTAGATTTGTTTCATCCAATCTTTTCATTTCTGCCCTGAAGCCGTCAGGTCCTGACGCTTGTGTCGCTTGAGTGAACGGAGCACACTTGTTAATTCCGCTGCGGTT
+
CC@BBBC=BBCBBBCCB?>BCBBBB@@?CCB?<<@7<<A@@@???7<<=??>@:<;@;;;188;?;=??@@1@AACA@A<????=?AC>@@09919B?>?@@C:0,0<,00,.:/0,711185555855886<@@@@@@;@=@B<>893778;;;;BA:9:D>??B>A=@B79488888.89BDA@@99919<<;=@,//4477;B<B999:3317;;7;?=?;<;;777>><<@699?AAB@?>2626867769977;@3
@EOUL1:00683:01275
GAAGACCCTATAGCTCCTTTATCGTAG
+
CB69;@>:>@999106444+.,,,,,.
@EOUL1:00749:00081
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGGAA
+
AC?ADCC@CABABCC@B@B@BBCBA>AC>C?AA@BA?BA@@?CAC?<:8?>A?;?>@CBBCC@@AC;:?=@@<@CDC?BB>BCCBCBA@=ADCCC@@@B@AA?BBBBCC>AA=B?C>@@=@AACBBCCC7::<?57:??BBBBCBCC;@A@?<<9<<<=@AAA@A=====;=CD5A<835+,+,5
@EOUL1:00729:00247
GAAGACCCTATAGACAGGTGCGGTA
+
AAAEDEE@CCBCB9.063..//,/.
@EOUL1:00719:00353
GAAGACCCCCAATTCGTTCATCCGCCGCAAGGCACGTTCCGGATTGCCGTGCTTCAGCATCAGATCCAGAAAGATGCGCTGCGCCTCGGGGTCGTCGCGGCATCTCGTCGTCGATCAGTCCAGATTGGCCGCAATCAACCGCATGATATTGGGTGACAATAA
+
BC?BCCCCE3A?B?BBB=@@BB>@@:<;C@C@C;:<<7<@CABDBEECDBBC?9;;A@;<<B@@@C?@AAD>BBCBBCC@@@A?;90000+11000.,,*,,,,.577/88<::>44403779660101016>>?>7=>,0/.33.000/00+034667111
@EOUL1:00706:00428
GAAGACCCTATGACCCCACAGTTTGCTCAAGTATCTATTGCATTGGTTTAAACGCTTGTGATATGATTCGTTCATGCACTAGCAATAGTGCTTCACCGAACAGTCCGGTATAGGTTGTCGAGGCGAACA
+
9964@@B:;<;9:BBB7CCA@@C=9;<BB?BCCCA<<<@ABAC@B=?@<@A<BDEG?BA??:::>?B@BB@@<::::77:C@?@@CBC<<;?BB:;7:B=??=;;3601111044/3333177703231
@EOUL1:00730:00498
GAAGACCCTATAGCTTGGGACAAGTTGCTGGAGAATACACGTCAGCGCAACAGAGGTGGGCGTACTGCCGCTGGTAACATATTCGAGTGCTTTAGAATCAACTTGGGTAGTGTGGTCATGTTCAAAAGCACTACAGCCAAGTACTATATAAGAAGTACAATGCCAAGAGTGTATTAGATCCTACTGCTGGTTGGGGTGACA
+
<929999-99>>?AACBB>CDC@CC@>ABC@CBA?CCBCDBB@?@AECE@CABBC@BAB;??????AC>AA@?=773:BA?C?99;;;8BBD???@<??BA@@=@C>B@@@CBB@CCDCBC=@@CC9BCCCCC@??<50,0<00/;?==954844::==888A<=>>??AA@??;444::7484;;@88//,///+.--11
@EOUL1:00721:00536
GAAGACCTTGAAGCCATGAAGGCAGGCTATGACGTGATCTATCAAGCTTTTCTTTCGCGGGATGGATTTCTGGGTTTCGCCGATTCTGGTGAAAGTACCTGGTGCGTCGGAAGCTTGGGGGATTATTACTACGAGGCTTGGGATACCAAGCCTTTCCA
+
4959442499:24424;;@?>5<11-/666/9444998@@>;::,/////*///+/8;;>>973<844-8988/77/44778//,///,//44-444@>A7766,,,,,*,*.1?@@;@;+448999=@?=A>=;933149-44//,/5//,/858,,
@EOUL1:00744:00833
GAAGACCCTGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAAGAAAG
+
B:9@988-4;;;CCBBB@CC@F@@=?B?=::<;;;GCCCC?A?961,7>>==395326694466//,/768::?8;;=;;<<<<@BA@@@@8;;8::7:6//,/8879=AFDCAAAHCCABB7;7;A=@AABB?BCCCCC@CC@CCCCC-BB?B???:===.7777/7///,-
@EOUL1:00709:01045
GAAGACCCTATAGTTCCCTTATACTGAACAAGCCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACATTCAATGTGACAGTCTGACAAGTGGGCAC
+
@>@@@>959A@@AB>BA<@<?AA@@@?=?=5::;?=@=B=?;?<71636316666?@AAC:??>8447996<==>??@@A=====?BCBBCDBB8<<=@A>@A??@<@?BF<<;B???:?@>>@69<<@>=A@@@<@>A@CCCBCCBC;<;G@CBD;::>444-4//879//-/;46;8770000353333/0000+044
@EOUL1:00714:01081
GAAGACCCTATAGACAATAACAATGATAATTTCACAGACATTGCCCTACAAGACCGTGTTTCCTTTTTTAGTAAATGGACTATCAATCGAAAATCTGGAAAAGAATTAACATCGTTTCGCGTTATTTGTATGAAGATCGTTGGGGTGACAAGAGA
+
A@=?>@@<;99CC@A@?BA8;8@C?<88488/6=>@BCC?>=777.4>>>;7714364=A5?:?>>>>,??@@?;>?388=====:;444::.:::;/300+00/0/056468;::4:60002---+---455448::8899==5=9==A<?754
@EOUL1:00733:01148
GAAGACCCTATAGAGCTTTATATTAATTATTTTAATTTTATGTATATAATTTTTGATTATAATATTTATTATTTTGTTGGGGTGACAAGAGA
+
;=ACCBC?CBBBCCCCCC>@BBC@C?CA????2848<<4<=;;;<<<@<?CHC1A@@?>AA=A===9=B>AA@@47738@@09:ADA>A@??
@EOUL1:00742:01199
GAAGACCCTATAGTGTTGCAGATGGTAGTACCACCCCCGACGGTAACTCCGATGGGAACGGGAAGTGGGCGGACGTTGGTAAGACCAACTGTACTGGTAAAAATAGTGCTCAAAGTGGCACTGTCCCAGCAGACCATAAGACCACCATCGGAATCGACAGGGTTGTGCTGGCAGGAAGAGACGAAGTAACAGGAACTGAAATGAAGGAGCGGAAAAACAGGAAGAAACGGAGGTAGTTGGGTGACAAGAGA
+
AC@CCBC?CFCBCC>>=@CB<>?A;?????C?BBCCD2@==A>@@>AAC;999AA@A=@AC9=9===A@BC>@@8834844344838=@899AADA<84444,4=@ABBA???;??><?A===BBA;?@A@@AA<?>?;78/789,////-7?@;;;:66616,//867?=A;;6;=@@BBCCCAC<;9;AA>==><<<:,00::<:;==<?CDDD2:989;8;>>9::9//,//88,//,088335556@
@EOUL1:00749:01214
GAAGACCCTGTGACTAGGGATTGCAATCAGCAATTTGTTGCCGAATCGTCCCT
+
99648890984344/.00+..,..333.116627B:77;631377116666,.
@EOUL1:00762:00198
GAAGACCCAGACGAAGACGCAACAGTGTCTCAAGAAGATGTAAAACACGGACATAGTATGACGATGAATGAAGCTCAAATGTGGGCCGATTTTAATAAACTTTACGGTACTGCCCGTTTGCAAAGTGATAATTTAATGACACTTCGCCCAGAACGTAAAGCATTATCAAAAGAAAGTTTTTGAATTTTCCAGCT
+
CDEB;9909999CB;<<CCA@>AC<=;AACBB@BC8;8?B8888.7?BC?BBCBBBB???@?<<<0/,000,//00076067@?;@>:::@?2=:777.1;6,03484946388-4;@189@C<AA998;;=;9>./.66.00/003440032308<<498<<;6<;:;;-444,30000+0323880453330
@EOUL1:00776:00164
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
CB@BCCC?A=?=AABBB?;<<BCD@?A<@8?:?>@ABB7BCCC@BBB>BC@@@B@EACC>BAAAEBDB>B@@>ADC:CCBBB>BBBA
@EOUL1:00795:00236
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAAA
+
CC?BA@@?ACCCCB????:?=ACD9CCCBC?BBCC
@EOUL1:00763:00252
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGACTAG
+
ECBDDCEBCCC;;;89?@;=@:@;>;@<99@@@<@@<<=BB>?<<8<BABBDCGBC@?:9:--+--,*,20/11;;;BC=AA<77>7831133,.,..+.,,,,,.3333565488>;<8@3/.7/////.*
@EOUL1:00777:00346
GAAGACCCTATAGAGCCAGGATCTACGTTGATGATTCACTTCCTATGTGCGCCATGTTCACAGATGAAGCGATTACTAAGATCAATGATAGAGCGATATTCATCCTATGTGGTCGGCCAGATATTGCCAGGAGACCATGTCCTCTAGCTCAAGACAAATGGGTGGGAGTCAAGTCTGTGGGGTGTGAATATGGTCTTGTGCGGACACAGTGAGAGACGGATATGGTTGTGACAGTGATGTGAAT
+
CDFCBBC?<9;A@@AA@AA?A;::;::B5::CCCB@B@>;,0,....168?<;<;;<;49:?B=9=87///..,..47768<<8?BE::0>00.9:>=??AAAC@C:::??:<::3424889;;89@8<<:<<<<=@@<<<::66///:245;:33:4444+43305=8<<;<@@<;:;==5;<;884/++,+,50033++,.,,6,,.:534:???A>AA9::./,/77:666;000>B@@>@
@EOUL1:00756:00432
GAAGACCCATGATTTCATTTGGGCCAAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTG
+
AB84;@?698994969111+1=+/,//+/88;:9929@@?A@BBC<;;C;<8@::00,.0,/-0,..../8599=9759A;;?A<;?C8857==476:K@E@CAAAA77//-.,-7;;@AAA4//9////866:<63101
@EOUL1:00790:00569
GAAGACCCTATAGACACACCCTTTCATTTCCCTTAGGATAGCCCTCAACATCATGACCTCTACTCCTTCCTCCACTCGCTGGTTCCACTTCTCCCAGAACAACAGCGGCGGCTCGTTTGACTACAACGAGCCGCCGGTATCACCCATCACGTTGTCATTGAGCTGAAGACGCTGCTCACGC
+
IDCDCCCAAAACCCCA@AAC@CC=BBCC>BB<A=AB??@A:::8;B?<@;A=DDCCC@CDCCCCB7959;9959@A@@@EB=@@?699?<?@AA<AAA8;<>??@;;70>9;;0000,/9=ABB>>?>11/5579>;??<:<@A;@BB;77>47787;;000...54///88;498:...3
@EOUL1:00799:00563
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
BB?BBB@=BBAABBAADC>CBBA@=?=??8?;<;==??5>>???BCCCCCAAACAB7;969999>@>><@@@=@CC8A@BCC:9:B?
@EOUL1:00759:00696
GAAGACCGGAAAGGAAGGGAAGCGTCACGATGATGCTCACGGCGATGGAGACTTTGACGGCGAGGACGAAGGAGCTTACGACCGATGCGAACGCGATGATGCGCGCATCGATGACGCCGATGCGGTTACGACCCCGAAACGCCACCCAACAACCGAGATAATGCCGCAAGCCGCCTACAGGCGGAATTGGAACGTACGACAAAGACCTGCTATCATCAT
+
99<@9959<>>9>7=<;DAD=@@>9:7A@?7:;99>>88:@>@>AAA<>>>99919@@@=ACA@=@@@@=A=?>A@>BA<<<:212>BBCCEAAAB@>@<;<>>A?>>>8999>888?@@?CBB?B@CDBB@@B6;;73../,-,000227+-*,,,2699?;997..,.3,..3/33;B578?:::9633,..../6===B;977;;;7766750,,*
@EOUL1:00751:00735
GAAGACCCTAAGAGCAGTACGAAGACGGTGCCGACAAATGCGATTGAGATTGCTAGACCCACGATCAGGATGAGTCGTCCTTTGATTTCTTTCGTTTGGATAGTCGTGCGCGCAATTTCATGAGCAATC
+
DD?AABC>CC@CCCCCA?ACBB@??BB?B@B?BBCBC>CCCA:720;:00,00:=7/0/+..-..100/033303363324=48666,100+03330320.------22403365643441111--,-3
@EOUL1:00783:00757
GAAGACCCTATAGCATTTTCTCGAAGAGCGTGCGACCTTCTC
+
C=8>999099;4448844,4443397>>?:../...,.544.
@EOUL1:00768:00831
GAAGACCCCCAATTCGTTCATCCGCCGCAAGGCACGTTCCGGATTGCCGTGCTGCAGCATCAGATCCAGAAAGATGCGCTGCGCCTCGGGGTCGTCGCGGCATGCCGTCGTCGATCAGTCACGA
+
DA@ACCCFF3C=A=A@?6:;BC=@@=??B4:3:719=5929=@@5<;;>@@;;;@A@AAABB;;<B?BBCE?CBAAA??@BBAB?AABBB7:;7774..,.04449;==9837777734.....
@EOUL1:00784:00827
GAAGACCCTATAGAGAGGGTTGGGGTGACTATAAA
+
CB>A999-4;;BBBBCCC>B@CAA7@@@>>>4444
@EOUL1:00781:00972
GAAGACCCTATAGCACAGAGTCAGAAGAAAATGAACGCACAAACCTTCAAACTCCAGAACAAAAC
+
EEA<<;=:@=ABCCC@>@CA:79;989;77,666017;ABBC>A=A>:77,4:;340420000+0
@EOUL1:00784:01034
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
=B@CCCE@CDBBBBC?B?C=@@AAC@CD>B?BCABB@CBCC?C>A<<=5?:==:=5::?@@?@<7434A?AIBEDDCACCACDCCD@BB@ACCCBCBAA@;958>>98909>9><?>AB?BBCC?@?DB.77477.7777>@?=>;>>:=6?::68;4737A<;;88///777774:7;;;./.::
@EOUL1:00797:01043
GAAGACCCTATAGACAATAACAATGATAATTTCACAGACATTGCCCTACAAGACCGTGTTTCCTTTTTTAGTAAATGGACTATCAATCGAAAATCTGGAAAAGAATTTAACATCGTTTCGCGTTATTTGTATGAAGATCGTTGGGGTGACAAGAGA
+
BA@CCCC@CAB@BBCB@>?:<<@BBAAA>?><AC<=?AACC@B?@<@@@@?A=<;@;??::A=ACCCC-ABCEE:88>???AAAA=?;;<==5==4448@>1>B38758;=?@AAAA<@@@AA>AAB<??@AAA>@??@992444-4;@AA>====
@EOUL1:00761:01073
GAAGACCCTATAGTTCCCTTG
+
@ABB@BC898=1/1,..+,*,
@EOUL1:00761:01135
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGATGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAATAAA
+
AAACCCE?CC<;;@C@CBC?ABBC?:??6707>37.,..+.-,-222344>=>A?B@BC@;5<<3749:@=AA:<=CGIGCBBBDFA=@C?ABCC@@@CC@BB?B@A@@@>@C=B<@8;;=@AAACDA@@:==4747//./-.4.4/39994555556:68<<455;:686=8555.54:::24444
@EOUL1:00768:01121
GAAGGCCGAAAATTCACTCGCAGAACTCAAAGCACTTGCTGAGACTGCAGGTTCAGAAGTCTTAGAGGGTTTGATTCAACGCAGAGATAAGCCAGATCCTGCTACCTATATCGGTTCAGAAAGTGATTGAGCTGCGACAACTTGTTGTATCTACGTGCCGACACGGTGATTTGTGATGGCGAACTCTCTCCTGCACTCACGCAGCTTGTAGAAGA
+
CB?B@BDCBBC9B?BBBBCC?>AA:@@???9??=??;????==<<<@=??>@>@A@@=@=<<?==;88,//,///,/6588877?>@?@>??>@@AAA>=<<>;<8<8:<;::;;77//--+-22;=555=88;6,,.,+,03889;A<<;:..**..)./89;??@@@BA<A===?A<<<=:=@AAB;6..,,0667/,+3335:68666;../
@EOUL1:00778:01112
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
BA?BBBB>BBCC@>@CBC>?ABA@?@87:AA=A?BBDC9BB@?>:;;9;ABCFC@CACD@CBBAAABB>CCC@CCC9GCCBA>ADEE
@EOUL1:00765:01175
GAAGACCCTATAGAGACTTTACTCTTATTATATTAATAAACTAATTGAAAGTGTATTAATTTAGAGAGT
+
CB::999-499:9=00/77778@@?==>07777010...+...,.3.44-44494424244-488/,,0
@EOUL1:00780:01172
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
BB@BBBB>BBBBAABAAA<A??=@@AA::8>>??@@@@5BBB@>@DB>DDCECB@B@CCBBBB=?===:=;;?BCC9CCBCC?BCCC
@EOUL1:00753:01210
GAAGACCCTATAGAAAGGGTTGGGGTGACAAGAAA
+
BA>A;;;:9@@?833/83,84711,1667747611
@EOUL1:00769:01236
GAAGACCCTTGAGTATCAGGACCAGGCGTTCCACAAGATGTTGGTGGCGTTGGGCGCTGACCGCACGCTGGGTGGCGATGTTGTGTCCTCGGATGTTGAGGAGGACGTGTCTGCTGATGTGTATCGCACGGAGAGTGGTCAGGTGGTTGGGGTGACAATAAA
+
DC@CBBC>BACCCCCCCBB@?@CAD@CCB@?@CCC@BBAAA@C=BB@CA?:<<-34;;AAA?AA???@BCC<BB@C?BBB?>AAAAA;==?>ABAA>===>AA>A@@@@@@@BAAAAA@???==<>>?>?:<<@@<=;=@A?:==3428=?4@BABD=4444
@EOUL1:00817:00135
GAAGACCCTTGTAAGCACTGTGGGTGGAAACTAGCAAGGAGCCGACACGGTCGGCGAAACTGGCAGGAGGCAACA
+
IBABA@@:><98;9:994333397955442....33,.,.16661115554399/./777344464/00/00231
@EOUL1:00835:00132
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
FKBHBBBBBCCCB@>7:8,8?>;@@CD:;:8AB:??AA6;;;CCCCCCCC=<<DAC@CC@BAA9<9>?99994444,4?>?AA:88D
@EOUL1:00825:00224
GAAGACCCTATCAACGAGAAAACCCTGCTCAAGCTTCAGCTCAAAGCTAAGAACCTGCAATCACCAATATGCTTGCAATCCGTGGCTATAGCAAGCAGCTAGGCAAACTCAAACTAAATCTCTATGCGCCTACACCAGTGGTTAGAGTATCAGAACTCAACGCTCGTCTTACTACAGCTACAAGCTTGCAGCACTACAACCTATCGACTCGCTTCAGCAAAACTCCTTGCGTTGGGTGACAATAA
+
AA?ACCB>CJCCC@CC<<@EE:CC@DCDD=<7;AA=@@@@ACAC@A>>=;11,1,/6AA@CC@@=?<@??@BDBCCE@CCADCFCCFECDCCC@CCBBBBAA=AGE=AA?@@<@@?C?CBAABB@A@::6:B?<<>@A@@?@>A@@@CA???A@3888834==;48444244/67//.45539A;8577//.27464;79788=777;5,,,,++501666.4889=;<<6A343,3/.--,--2
@EOUL1:00803:00278
GAAGACCCTATAGAGAGGGTTGGGGTGACAAGAGA
+
;959999-499@;;<BBB<A>ACC9DBBFD@C;;;
@EOUL1:00805:00295
GAAGACCCTATAGTGCAGAGAGAAGTATGTC
+
;:8<BAB@AABBBA@93.0....,.366111
@EOUL1:00830:00289
GAAGACCTATAGAGGGGGGGGCATGATCCGGAATGAGCGCCGGGGGGGCACAGGTGGCGGCATGGGGCATCGCGGAGTAGCGCCATGGTTGCGCGTTTTGTTATTTTTCTGCTTCTTGTTTTGTGGTTGGGGTGACAATAAA
+
?A=?//,/7768@@@@@A@B%BBBCCBBAB@E@BBBBCBE@CCCCCC(AA??>?AA>==;=@ACCA6A=>=?AA>@@???@AA?BAA?B:==??AAAA6??<?@CCC1:99?>;==887::1/..6666@D1,,.33+,+00
@EOUL1:00845:00296
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
DC=A@BA>BBBB@CDBBB>BCABB@?><@@@<<<66164;==><89909@BBBC?B@EHCCCEAAADC@CCC?BBB5;;BBB?CCCC
@EOUL1:00805:00304
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATAAA
+
BCCDCCC?BBA>>99595958>>@@?@EAC;;;=??:A?AB@C?BABD8;8;C@A=@@<;<DC?BA>BB@BC@CCBB?BJACCAA@@BB=@ANECDBABAAA=ABC@AA<AB1;:<;;;DCBACBCBBB<A?4747=<===?>@?<<6=;:?8858;;98877589=>?;::<<4<787>387=?
@EOUL1:00817:00335
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATATGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATAGA
+
CCAAAABABCCBCA@696:>A@ABCABB?BBBD?B>>C@CA:8;:CCC>C@CC?A=ACCBBBB@@@??A?ACBDCBBACCBBBBCC?@?<@@@@ABBBB?CCBCBBBCC@CC>B=@=@A>>@@@::8==7>>475;BCDCDBAAACC>B???A=;=??=@@@@???AA??<?CD6@???A;==AA
@EOUL1:00834:00423
GAAGACCCTCAAAAT
+
4;:49990442220,
@EOUL1:00822:00479
GAAGACCCTATAGACGATAAGTTTTACGAGAAGGATCTTGCAGTCGTTGTTGGCGGAAATAGCGGCGATCAGCTTGAAAACGAGGATGATGGTGGCGAAACAAGTCCAAGTGATGACGATACCCAATATGATGACGCCGTTGGTGACGTGTGTGAGAGTGATGAGGATACTCATGCGGGTGATGACGAGGTTGACAATGGCGAGGACGAGGACGAGGACGATGGCGTTGATGTTGATGTTGGGGTGACAAGAAA
+
B?>BBBA?BECCCB8<<CB?BFBB8ABBBBB?A??;;<8;<BCDDBE@CC?B?BB@BC?C;<<D@A;;:::@AA?BBCC9AAAEEC@:://./97;>>=:8838@A?;9;7///:::8<;<<<7=>@@@@@????@@>@A><><?BBCBBCCCCC;;;;999<;CAC<>=CBBDCCAF>B?@;;<CCCC@C@C@@A@CC>CAAA>AC??@@B97::54:=;;<:?DA??=988?A@<==>AAA6===??>AAAA
@EOUL1:00832:00698
GACCCTATAGAGCTTTACTACTTATTATTATTAATAAAACTTATTGGAAAAGTGTAATTAATTTAAGTAGAGTTTCGATTGGGGTGACAATAAATACAACCCTGGA
+
////+///66=9444-44434;>?;?AA899=971...*//,//,/,49929;9442497778565//////44-444423772333442444-4877133,..,.
@EOUL1:00800:00806
GAAGACCCTACAACCTACAACGGTCACCAGACCGACTGCCCGCGACGG
+
A?<@@@@<A@@CAC@BA??;??<?????BBAA>A?77777.747/,+*
@EOUL1:00844:00946
GAAGACCCTATAGAAGCTTTACTCTTATATATTAATAAAAACTATTGACAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
442444;:44/6/.,.655263779999@@86:3,/....*...33349499@DBB@B@AA=A:44:444-49>;>==2<733166111
@EOUL1:00805:00971
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAACGGAACGATGTGCAGAAC
+
AA>ACCC?AA@CC@?<?<?<?AACC@CC:B=??:=>;?3:7;>:=033,6016>A@@BB???B??979@59919@>><@@1;;@@@?CC?CBCAAA@@AADD?B?>99909@:@<?58878>BB???B@::59:;6;966666/,++
@EOUL1:00840:01047
GAAGACCCTCACCGAGGTCTTCTTGTTGGCCAAGAAGGCCTTGGACCAAGACATGCTGGCGCTGACGCGGTCCTTGATCAGGTAGTTGGAGTTCTGCAGCACTTCTTCGTTGGGGTGACAATAGA
+
AA@C999:<AAD><:=ACDC?CDBCC==<=?B?BCAGACACBDDBD@EFCCCCDBBCB;<<BBBCCAA@@CD@C@CCCC;;7;?=?>B?BBC@BB@ABAAA==9;>>@8848@@6A@@@@>@ABA
@EOUL1:00804:01081
GAAGACCCTACGAGACCATCAGAGGTGACTGGTGTGTAAACTGCATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATTATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAAGCGGCACTTCAACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGGACAATAAA
+
AA>BAAA?BB>>;100:9001>??>BB@A:657==9:??:@BBBEBBCC@CCC@DCB=?==><@BBBCCCC?A;;;09>@=@@>BBBCCC@@<>999969>>>>?;87;@877400,1--00*,*,...3344-4498444424;<888;-424983379?A?><44..,,,0,,.553111.00/04429===
@EOUL1:00834:01066
GAAGACCCTATAGAGCCAGGATCTACGTTGATGATTCACTTCTATGTGCGCCATGTTCACAGATGAAGCGATTACTAAGATCAATGATAGAGCGATATTCATCCTATGTGGTCGGCCAGATATTGCCAGGAGACCATGTCCTCTAGCTCAAGACAAATGGGTGGGAGTCGAAGTCTGTGGGGTGTGAATATGTCTTGGTGCGGACACATGTGAGAGACGGACTATGTTTGGTGACAGTGATGTGA
+
@@?CCCC?BAA@;:;;>B@<??BBAAA@?A@@C@@??==@?BB???5///96:>@A>AJDCCAFCJ@BBBCE@CC;;7:BA??8;:A?@0/0/.?=AB?:::?>@A?<<A8///,-,-344:=<9=:=?<??AA>AAABA>:;;.0088779:004,400+444,0==;<=>??=>;-.-.-...=>=?@@@<<;8;348=?=@@>:87/...-003..-.33+./;;9;?99==<<:800066;
@EOUL1:00837:01094
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCGACCGCCATACTGCTATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCC
+
BCACAAA=?@@@BB>BEE@CB<@9959?B@ACCBGEDDC>>>>113>>A?;;;?@=@C@C>>:>0..6853:9>>@CD>??<<76177711136011,166314/--1--,-0003333/00/00066:<=AA@@>AD?887;=8:97----,-000++++++-303046A999B??@B@889@BCBBBBC@CI>AACCBBBB@@@AC@CCC@C?<;99988898@<?>A?BBBBB?
@EOUL1:00808:01170
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGTGACAAGGAAA
+
DD@B?BB?BBBBBBB?A@C@@AAAC@CD>C?CB?BC@C@C@>A<><;;7?=?B>A>BB@?<<=<<::7;59?:>>@B>B@<A@ABD?BB?BBB@@@@BB=A@CBBB@??9>>6?<@@CC@CCC????=>97747477=488?AADBB=@>=:44249979>@@@@@@=99/00+3449A6824::
@EOUL1:00837:01235
GAAGACCCTATAGAGCGTTTGACTACTTA
+
;;7<?@>9;;9BBB@??>7-/.2,,,,*,
@EOUL1:00846:01215
GAAGACCCTAGAAGCTTAACTGTTACTATAGGTAGAAGCACAGGAGGAAGAGGAGTAGTAAAAGTGGAGACTGTAAACAGAAAGATGAAAAACC
+
DC>B@BBBBBBB=B@B?B@@??A<A:;:??@;??BA?<<==;;9<>;=>@<<9844;:4444-3;;/75A<<;??=4>>777.7=::7777.7+
@EOUL1:00814:01264
GAAGACCCTATAGACAACCCGATACATCCCCTACATGAAGAGAGTGTATATATTAGACTGGCGCCACTATAGCCTCCCCCGAGCAACCACCACCCACTCTTAGAATAGACATATAAATATATCTCTTACAAAAGGGTGGGCTAAGGACATCATTTCCCGTTGGGGTGACAATAAA
+
CJBECCD?DDCCCCC?@@@??AABBA@A@@5??AACCB@BBA:;:DE@=?@7737>><=<@CCB388DA@@?A?CCBCC2CBCBB?B>BB?@@C>==?:747@@<ACCDA@@>777.7>=9:8844:;4444-44-4AC<@@@?C>@@?==444-44-/8,///+/.//.74434
@EOUL1:00851:00122
GAAGACCCTATAGAGAAACTCATGAATTATGTTACATCCCCCTCCTTTGGAAACAAACCGTGGCTCCTACTAATGCCTCAATGGGTGCATAAAAAAGATTATTACTTGAATGCCATCTCGAAACAAAAGAACATACGATGGCAGCAAAGCGACAACTAAAAGGGAATCCTCAGAACCCATTTTACATTGTACCAAGGAAGCGATACGTTTACCTTCACCGGCGAATTTTCGTGAAAAGAAGGATAGCGACGTGCACAAGAAAAGTTCTCCGTTTGTTAGCATGTGGTATTGTT
+
CBCCCCGACC@?@BADE>CCBAABB>A@CCBBAABBCCCCB2CDAEEBCACC>CCCAE?BAA@ABCCCBBBD@AAAACCE::<C@DCBBCBCCCC-CCB8;;=998=895<A@;>?>>?<>>88888.8?;>?:7999=>;>@;@BB=BC;::?=?7777.7?9>=AA;@744;7;;358;=49??B?6;0/,/,.,.,..3963?7970..,-,-.3366<<:::7507..++++*+-,-0----=?<<?A99917:;;??5<@=BBBABAE>AB<::;:@@@B@B@@5:7:
@EOUL1:00875:00147
GAAGACCCTATAGCCAGGTATAGACCTTCCTATAAATCAGGTCGCTGGGATGGTAGTATTAGTTTTTTTGGTCTAGGTGCACCACCTATCTTTCAATGCTTCCGCTAACGTTCTAAGAATTTTTT
+
449:@@@:9949;A>BC@BBBCBAA?A>A=@@A@A=AABD@BABBBCB<AAC@BBB>AA>BBBBBBCE(C571000,00..5579>7779453./,///.,3133331339976713.,00000(
@EOUL1:00869:00150
GAAGACCCTAGTGGACATCTTCGACCTGCTGTGCGGGATGTCTACGGGTTTCATAGACAAGCTGATCAGTAGGAAGAGGAAGTTCAAGCAGGAGGTCCGGGAGACCGGCGAGAGGCTCAGGAACGAGCTCGATGTCCTTGAGAAGCAGCTATTCACCACCAAGGTGGAGACGGAGACGCTCAAGATCGACAAGAGGTATTTCGAAGAGCAGGTGAGCGAGCTGCAGAAGAAGAACCAGGAGATGATATCTAACAAGTCGAAGAAACATAAT
+
AA?A949-4;;>@?BCBBBB?CCCC;<<BDC@B@>@:?AAACCCBBB>BGA@?<;;@BB?DABBBCCCCB>>B?>?@A>;88:;=A?BAAB@CE?AA=??9?B@?9=9=@@AA@?AA=99;=>@=>?889??@???9;8<=<;8<@?????@<898;==3858=@@>9::AC=??@=<;@87249;889AA?CAA>===@;@@A;==A@@@>AAA@@<<<ABBB===>8738@>99::.//8776/44:111,,+,,00355754.-..,-
@EOUL1:00857:00230
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAAA
+
BB?AACC@FBBBCCBDB?B@CDDCC?BB>BBCC@CC?C;;;@C@CBBB>B@??=>>>>>=>@A@??>AA?BC@CCFB594-9@@@ABCB@DCCB?BBCC>BB@EC@@=>08809?A>BB@A@@BCCCCB6884737777BAABB?CA?CA@CAA>A9838?@;;<>?778<?@@6A@@@@;==@A
@EOUL1:00870:00280
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
<95<@=9:@AABBBCCCC?AAACC@CAA?BC?C@A>>>7;<997?=@<>C9;<A?AADHBCBBBCCCC@C@A@ADD:CCABB>BBBB
@EOUL1:00855:00353
GAAGACCGAGGCCGAGTCTGCCCGTTGTCACGCTGTACTCCATCGAGCCTGGGTCCTCCATTTCCCATGCATTCATCCGTACGTGACGTGAGGAATGAAGAAGGG
+
CC?<11,//6,/,//666849809969>>>4447387749999713113111,177770111,.522...8:577,,*----1344<?67742423444.-,-33
@EOUL1:00874:00376
GAAGACCCTATAGAGCTCCCAGGCGTGGCCGTGAATCTGGAATATCTCGACAAGGTCATTGACGAGTCAAATGCGTATCTGGCACCCAGCTCGGTGGAATGGCAATGAATCC
+
<::=BBB<AADBBBCCCCC=<?>A???=><==7734;44/348489;87:BC@A=@??@BCA===????C<???AA@@?AC<@884-4;444;8;?>B25/..-31..,-22
@EOUL1:00899:00367
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
DB@CBBC@BCCBB?<7775:BBBC@C@??AA>A?BCCC9CCCC@BCB=@@AB<<==BCC?DFAABBGG@GBC@@@A8@AACC@BAAA
@EOUL1:00879:00479
GAAGACCCTGACGGCAACCGTAGGAAGTTTGGTTCAGGAAAGCCAGGCAACCAACACGGCGGCTGCGGTTGGCGAGTTTCTGGGCGGCATCATCAAAGCCGCGAAGTGATTTCCCTGCGCCCGTCGTGTTTCCCGTCTCGCACGCGGGTTGACT
+
BCBBBAB=BABAB?AAAC<<<AA=@;<<CAC>A?A;;9<IACC@BB7::<@;=:=>==:7:6:EDCC>B69599@9998<@DB<<<7;BB<<=BBA<BC?BEBB@BBBBHEBDGB@@A999099944>CB>BB=BBB@@?;66/000,0,.4.,
@EOUL1:00852:00524
GAAGACCCTTGTAACGGTAATGTAGTTTATCTTGGCTCTTGCTATAAAACATGGGCTGCATTAAAAGACTTGCGGTATTTTAAATATTATT
+
,,*,,22022//4888888777<78776900.,/,/.3313777;>@9266788,1;111>99??566660117770000+0424352342
@EOUL1:00892:00537
GAAGACCCTGCAACATGCGGCTGCCAGCAAAGGGGTATTCCAGATGCAGCGCGTCGATCGCCCGCATCAGCGCAGATCGGCGGCAGGAACCGCGCGGGC
+
CC?BBDC@BBBC=@@@<99ACEEC@DC@==0999/99C59594333687<<>?@;::ABBAB<A???>?77777==4372382803/3/0207.9996/
@EOUL1:00890:00641
GAAGACCCTATAGAACGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCCGGAAAGCC
+
DEBAA@C@BBA;1,+--*,,,,28::;;;;@BCC@?;<<@BBAAAA<@=>@>B?BBCCDBB>???@C@CB>DABA>BB>BBCCC@CC@CBB@C@@@B>A?;>>999914;;,42//+/4,
@EOUL1:00860:00870
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
;;8;BCC>DABB@;;<BB<B;;;?=@7::>>:=30000+111/9/9979999AA=@@BCABBCBBBGC>B::8;BB7BBBBA@BBCC
@EOUL1:00862:00885
GAAGACCCTGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAAGAAA
+
CB:;9446<@ACCBBBBDAA?CCCBCNCBB>A@BB@BBBB>BBC?>9AACB@@???>CC?@@A?=<<;;;8888;@;<<?<<<=@>>?<<8:;84858A;9>;/0//;>CDDDHDBBBCCC8;99A@BCCCB?:<<BBB@CB@CCCCC-AA=:;:?=?CF9BBBBA<@>DD
@EOUL1:00883:00887
GAAGACCCTATAGAGGTAGCTGCGACCACCTCTGCCGCAAAAGGGTTGTCCTGGCCAAAGGTCAGCAACAGGGGCTGTATGATGTGGTGAGGCTGCACCGTCTGCACGCG
+
/1,494;5@=BAA@E@CCBBBAA999699<<;;CCABAABB/9919;>8847999488,136=A?????<5;;.337611677164;<<==;;::/.+000+++----00
@EOUL1:00872:00926
GAAGACCCTATAGATTGGGACAAGTTGCTGGA
+
CB<@989-/881/0::888:1-+0:3...,*,
@EOUL1:00856:01006
GAAGACCCTATAGACACGACACTAGCGAGCTCCGTCG
+
GE?B@BBBB>>4483376?=7:;337400..,*,,,,
@EOUL1:00857:01000
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
:929>>><@AAA=;=<CD@CKB?>=;717664737?BD7BB@?=88809A999<8<@GE@CGGAAACC?@AA=AGG9B@BBB?BBBB
@EOUL1:00854:01061
GAAGACCCCACCTTGATGTCGAGCGCAAGGGCCGACAAATTCGGCGCGGAAAGGAAGATGCGGTCGAGGCGGAGGCGGTCTCGTCGGAGGCGGAGAATTGGGTGTGCTGCGGGCTGCTGGACCTTCCGGCAAGAACTTCCGACTTGCCTGCCATGGTTGCAAATGGAAGCGCGGGCGTG
+
@@;;/1//*11,/688;ABBCC;<<@@=@C?CB;:<CC<@>@B@CCBB?CC>C8:5:??ABB???@AA;>=3;979C6<<AACAAAACC@BB?DDE;<9<B:??;801218:8700:.0,.430288;-//745566767B;::7354455444?:>9>784,49/4/./--243---3
@EOUL1:00886:01084
GAAGACCCTATCTATTCACTACAGGGGCACCTGATTTACGGCTTTCCTTCCGGCATCTGGATGGCTTTGACCACGGCCCATCTGAGAGGCGGACGCACAACCATTGATGGTGTGCGGGGAAATGATCTGCAAGAAAGTTCCCGCT
+
A@?B@BA<@A@;79@<999BBACCBB8>?A>;;6:7,7<<379@095;:A>A;;9BBCB>BAA>ABG@CCC@CBB@BC>CAA@>A?9959?<?@;;:@?3?8896;88<<<??9????5?@<B@CDEDAAA???>01/,/65/..
@EOUL1:00883:01144
GAAGACCCTGACTTAGCGGTCAAGTTCCTGTCCAAGCGCAAGCTGACTGACGATTTATTAGAGGAGGCCAAGCAGACTGGCCCGAGCCTACGACGCACCGACTGTGCTGTGCACACTGTTCGATGCTATCATTGTGAAGCGTAACCAAAGGAGAAAGGAAGGATGACGATGCCGAGATTT
+
<<8<>>>9>11--+,,//56887;;77:;;;;<<88BB7751663316111677@;???83-13--,-,-,--36334<8<=5587?8833383336<89864.--2---44<::;==?<=;444////-344488:234;;=;=:7:6:6///;999:533465344403023333330
@EOUL1:00948:00237
GAAGACCCTATCAATCAAGGCAAAATTGCGAGAAGTTCTAGAGGCTATTCTCCAGACTGATGTTCACGTTATTCTTAACCCGAAACATATCCATCGACCTTCTCTCGTTGGGGTGACAATAAA
+
BB?B@@@<@<;<C?BBA>?>9999,49:AAAAICBB?B@@?<<8;@ACBFIE@CCBBCCCBBE@AA@CC@AB?AC>B5:75:>>:@@@A??;?@???CBD@CBA?=@@@CLD9BB<;;@CDDG
@EOUL1:00941:00347
GAAGACCCTATAGAGGGTGGTTGGGGTGACAATAAA
+
<97<DAA>CCDBBBBC<>B:77>>>2>@@BB?BCDC
@EOUL1:00943:00316
GAAGACCCAGAATAGCAAGACAATCACAGTACACCTCCCCAAAATCTCCACATGCTCAGATGCACAAGACGAAGTGGATTGCCTGAATCAGTACTACCAAGTGGCGGTTGGGGTGACAAGAAA
+
BC@CBBC===B@ECCBC?CCCCACBBBBCBBB=994444-9??4ABCC@CCCCCBBBCCBCCCCCC@DBCEC@BBBCDC>???BDBADAA:;:BCCC?C=AAA?AA=?3444,4;@BA>????
@EOUL1:00945:00322
GAAGACCCTATAGAGCTTTACTCTTATATATAATAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
49649:;:=A=@@@>>?>0888337667....,.553988:=8==541111/7::;:58444444;699;;;;:59..//,/.42
@EOUL1:00937:00395
GAAGACCCTCATCGATCCCAACGCCAAAATCTCCCAAATGTAGTGCCAAGCCGCCCACCTTCCGTCGTAACACATTGAAATAACGATGAACTGACGGTTGGGTGACGAAGGAGA
+
IDBBBBC?CCBDBAB@DC@C@CCB=@@@6>>>?@:@B?<<9??>87431331111,397713117771701776=85?A9;?:?;934;/716300/0/00+-33-./75547<
@EOUL1:00904:00448
GAAGACCCTGGCGTAGTAGTTGCCCGAGATGACAAAGCCTGGTTTTGGGTGTGTGTATGGGGGGGAGAGAGAGTGTGTGTGTATGGGGGAGAGGGAGGAGTGTGTTGGGTGTTGAGAGTGTGTGTTTATGTGGTTGTTGGGGTGACAATAGA
+
A??B>>><@@=@CDCCBBBB?@AB?BCCBBBC;>>611,1/,///*4;:ABBCDHA@@@===@='@DBBBBCCBBBCBCCBBBBCCCC2AAACA;===?BCA@AA>?=:;3724;;48889@@AAA:866//,/7;;;>;>4=BA@A>@@88
@EOUL1:00931:00472
GAAGACCCTATAGATGCTGTATTTCGGGCATTGTGGCAGAGGAGCTACCGGCCACCCTCCATAGTGCCAGAAGCATACCCTTACAGTTCTTTGGGCTTGTCGGCTTAGCATGCATAGCGTCTATGGCGTCCCTTACTGTATTTTGGTAGATTGAAGGCCCAACATACAGCAAGTGACGCTTTCTCGCTTCAAGACGCCAAGTTGATATGGCGAGAAGAGCGCCCGTTACAGCACTTGGCGATGGAGCTCGTAAGGTGACCGATAT
+
FDABABC@CCCCAAAACCAB>;?:??B:@CB?C<<;<CBBCA@<<<CB=B:<;;A@:<<;<B@@ACC?BBB?CCBBEEE>C@BBAAB@???=?C>CCA???B<??>?:::AA>@??<<9994777:9;999-424@=?A@AAA6A588::64//,/44404985688<;?@<;;58899;;69668//-/8567/78=A=@AA@<;<8838;;;@>@BA;444-44888955,,,++++..06.//.7743;2;78434294610
@EOUL1:00917:00532
GAAGACCCTATAGACACATTACAT
+
9964AAA?B@;;?-,,,,,*,/..
@EOUL1:00945:00504
GAAGACCCTATCTATTCACTACAGGGGCACCTGATTTACGGCTTTCCTTCCGGCATCTGGATGGCTTTGACCACGGCCCATCTGAGAGGCGGACGCACAACCATTGATGGTGTGCGGGGAAATGATCTGCAAGAAAGTTCCCGCTAAATGCCACATGCTTGCCGGTAGATCGGCACAATTCGATCAAACTGTATGTCAATTCCGGTGTACATACGCGCACCGGGACTGACTTCGATACGATCGGAGCGCCTGGCAGTATC
+
9954>>A:<99@A@?=>A;40=46>>+1.4,...8@8>=?=79>8<FCAHACACDBB<9:<AA>BDDBDCC?AABBCCACCCCC><=D@@>;11100;;643.10133313799<<AC7>?9779BBBBCC@CBC;<<CHB998811+//.,.00,,,,*,8565>9949;;7;==8;>@999D?BA<?>>==:78>>;>4841711177777866696629B<??<:866268666766111010111211,-+++2--
@EOUL1:00947:00563
GAAGACCCTGCACAGGGCTGAAGGTTCGAGTTACACTCGA
+
BC@CBBC@C;89>899-44331377774...,,,22,,,.
@EOUL1:00909:00694
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTGAGCATCGATCATG
+
D@?>999-4;;;9<@CC@C?BB<;;>>;>?;700:,0,0,/::97??5:7?;==@?@@@?@A@?>B33314:::;894438:<;><==AAF99699699>>>88809=@99988>>>7:1111-167//,1@;;;B69677777;791668=>>A@C@613/033----11--.-1
@EOUL1:00904:00772
GAAGACCCTCACGTAAGTAAATGAAAGAGGAAGTAATCCACACAAAAGAGCTGAAAATAATGCAAACGCCTTCTAAGCTTAATGTGCTGTCGCATCTCTTCTTCGTAATGAGAAGGCATTGGAGGCACGTCAGACGAGTTTGTTGGGGTGACAATAA
+
<;8<AAA<CBCBBBBACBB@<>99;09@@587;@@=@D@??=7777.177==99991;>485EDC<=@975+,,,*+++*+*+0238:/878//0;;:?=@@=<9=A>@@@@@>@>A=<8;=8769@BB<<<88887=;;9997:991484;;9444
@EOUL1:00910:00890
GAAGACCCTATAGATATCACATACAGCACTGCAACAGTAACTAGTATTGGTGCTATGGGTCTAACGATGTATGTGAAGATAAGAGTCTGGTGCAATATTGTTCCAGGTGTGGGAGCAATATAATTATTAGTGGTAAAACTTTGGGCT
+
CC?CCBB>AB;998>943;;@@AABBBAA???@=@<<<>=@@<<888=;<;;A>>>>?087=A<;:::77BCA@BA<??>:9:00,,,,+,3377BCG@?>,0,00,0>00/+8336378<:363163666<=4444-044+00+00
@EOUL1:00919:00879
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAAGT
+
CC@;;;C><9;CCBB?C@B;@1/09996266..,..
@EOUL1:00945:00937
GAAGACCCTGACATTTTGTACGTACACGGTGAAGATAGTGCTACAGGTCGTGCTACATATACACCAGACTCTTGTCTTTGACATTGCAGGCTCAGATGGGCTCTCGAG
+
@@A===D799BAACDD9BBBBABCABCG><<@<>><<4>=@BBB@A><99;BA?<;;CAA@:::::::0/.4574000+3333313@@9:8=?A97479,33331.,,
@EOUL1:00928:01012
GAAGACCCTAGTAGCACTGACCGCTAGTGCTGATAATATGTGGATGTTCGATGTCATGAACGATGCGTCCACTACAGATAACACCCTGATCGCCCATGTTGCTCCGAATCTGACAAATATTAATAATGCTACTGGTGGAGAAATCTATGTAGGTGACATTCTATCAGATGCTCCTCCTGCTGCTGTGACGTTGGCTACTGGTGGGAATGCTTCTGTGGCGTGTCATACTTCACCCAT
+
@A>BBBB=BCCBBBCABCAABBAAAA>>>>>1//:::B=?=A=?<<<;=;<====A???>@AAAA@AAA>A???@@@@@@?AAAD=@???A@@<;<?AA>AC??<?<;?@AA@<=87<?><@>8889CB>@A8868A?@?=A:@<9==;<4747>???@?@???@AA@@@@AA>@@;==A@@?A??@AA==:=>@???@@=???<?>AA@?<6,,,*.3,/0@=;;@@@?AAAB;?A
@EOUL1:00904:01066
GAAGACCCATGATTTCATTGGGCCATTGCACGTTGCAGACAGTCGGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGACTAGAC
+
?B@CBBB>@@@=@@<@@C?BD?C@CC@ABABBC@CACBCCA;B;;8;@@?BCBADDDFDA:0/9:0//1;5<?@BB;A@<:7579??::37==37377-7360366656633233348===566,//.--220.
@EOUL1:00946:01092
GAAGACCCTATAGACAATAACAATGATAATTTCACAGACATTACCTACC
+
CB?ECCF@CCCBBABB@C@>BB@BCCA>8889???CC?;;<,//,+++*
@EOUL1:00915:01136
GAAGACCCTAAAACGAAAATGCAAAAACCCAACAGAATGCAGGGGAAGCAACAACCATCCAATAATGGTGCGGAAAACAAAAAGCGAATTTACATGTGGTTCAAGTGGCC
+
A@=@B==:=?@>4>>>>@6@BABDDE2CC8:8:?A@?BBBCBCC9C>?AC@AC<?<?A<<@=??@==>=A=9/333+44444,4;;;244-4488774;949;5//+63.
@EOUL1:00916:01129
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAAACGATGTGCAAGAACGCT
+
BB<@99;09@@@A;:58;=:??ABB?BB>B?BB?@B5644:3424444+:@BCAC@C@>?AAA;;;87959D:@@>>59918899>273:7:7>7:76?;8=943378=9?@6;=B897>?778@@>7>;7:26>;99;:<473300000+++
@EOUL1:00930:01140
GAAGACCCTATAGAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGG
+
BB@CBBD?BB>9;6,.,0,...1B@CB>BAA@=@C@G@CC>A>ABCC>C@CC?B?ABBBBBB?BB?BCACFEDCCC@CC>BB;:;ABB?BCCCBBB?><ACACCCCCC=@@6><?7;;>BC?@@B?B>677+,+,,,+06<=<?AC?BAA@=<8956566///./3303233
@EOUL1:00931:01142
GAAGACCCTATGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCGACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAATAAA
+
BB?@99979>>>:;97>;;??::::<=>?:;463390;;6;>>AAAAAB>BAA>A?@@B?BC>CAA;::AB????D=BBBBBB?BBCBCAAACCCB@@=@CCC@@@CAC?@@A?AB?CCC>B>BABB?>>BC@B::5:?<<:?@@::<C?C@D@<<;@@AB<@@CCBC=;:7:@===9<;:1444===CCCC
@EOUL1:00910:01228
GAAGACCCTATGATCTTGTGTTGACGAGACGGGGCACTCTTACACAGGAAGAGTTGACGGAGACTCCTGAACACGGAAGATGGCTGAATCCTACCTACCTTGTTCTCGAGTGCTGACTTCACTCGCACTGGCTGTGCCAACTGTGAGCTCCACGTAACAATGTTGGCACGCACGTCGCTCTGCACCACCTCGTGGTGCCGCTTCCGCTGCTCGAAGTGGAATCAACCTTTCC
+
CA>A999-9::?@911,.6..,.55.68CCHCH:CCCCBB@CCCBBB5718877477==;?;?<<@=@<=>CCCC@F?BCA@=@AAA>AA>???>A@@=@>@>=;:::7>>=;<;;;;8<@@@@B@===A>>;;8//,/769444<<;>@==848>2447877:;:;??@A@@@BBBAB??=?77477:==;::34/74///,/78888?===@?A@A?C??>>::244,,+
@EOUL1:00921:01200
GAAGACCCTCGAGATCGCATCCTGGCTCGACCAGATGGTTCCTTCCGACGGCAAGGGTCAGGAGTTGATGAAGAAGTCCAAGTCCAAGAAGCCGCCCAAGGACCTGAAGTTCAAGGACGTCTACGAGGGTTGGGGTGACAAGGAAA
+
A@<@4849;CCCDCBBCAAAC@CC@CBBBAA@C@>B@=8;?877;@CBBA@BB>>>;99>C?@??=?@A@A?@@>:889<:==?=;8;B38:A@@D7<8:93437492588<8=?=@@8883304333+0/447-3=?B=5558A@
@EOUL1:00928:01208
GAAGACCCTATGCAGCACCTTTTTCTCCAGCAAATATCGATAGCCCGAGATGACCTGCTCCATGATAGTTACGGCTGTGCTTTCCGCAAGCTTCTCGTGCTTCTTGATGAAGTTCTCGAGAGTCCCTTGGGAACAGTACTCGAGCACAAAATAATAAAAGTAGCCTTTTCGATGATGCGATGGCATTTGATGATGTTGGGGTGACAAGAGA
+
@=8=8;98<BAABCB@??49999-9@@?CCBCDAHDCCAAA@A@C@CCCCCCABC=1111-1@?@@A9959@A>BCCCCCDD>B?A@@=AAB>BBDDCCBBAECABBAA@>BC>ABBBBAD@AAC>C?CC=B@C;<<CCBB@A@BBDDKK9CB>???=2777773488.40074::=BBC??:;6;8/8?7;:BBB>BCB9B@@@A>@B<<
@EOUL1:00989:00283
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
CC=B@@=5999;?CBDCC>B@CBB@B<BBBB@@>@C@C7??AA489909?@ADEBCBCDADCCBBBCCB<;;@BEG9CAADCACC@@
@EOUL1:00970:00319
GAAGACCCTATAGATCG
+
AA?BAAA<<91--**((
@EOUL1:00979:00441
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
EC@CC>@6::11,///:;;;?C@@>BCBBBC@C??<??6@@@@?:;;79@999969?BB?CAAABBBDAAAA@DCC:CBCCB>9999
@EOUL1:00959:00496
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
CC@BCCCACBBCACCBBB=BCBAA>ACA>?<6671111,9?@>?<<<8<BBCCC@CBDC>CCCAAABB@BBB>BB?6;@CCC@BCCI
@EOUL1:00966:00483
GAAGACCCTAGAAGCTTAACTGTTACTATAGGTAGAAGCACAGGAGGAAGAGGAGTAGTAAAAAGTGGAGACTGTAAACAGAAAAGATGAAAA
+
A@=@@@><@CCB@CCC?B?BB::5;BBAABC=?=??36??BCC??>9=:==@?@@@A?;;;;;0:@B9==@?@;<<>:??8888.8>>0----
@EOUL1:00972:00450
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACAGACCTCGTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCT
+
AABCCDICC?A?>@=@@<?@?:;<@CC@DCABBBB@C?B?B=@?CCC7:;A?==D:?=C;>=A????CC?@A988>=:846////,/8449944:4429=;9<<:?879737<<BABD@;<<@A?A;;<B@CBCEACBC9;<@@:99;@<775::<=>@??@@;?>?B?884++0++++,.11116
@EOUL1:00985:00537
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAA
+
4954??=<@B@>>??<?<?@CCCCC?BA>B@CC@B@?@>AA?C@B??A=B@CC>>=ACB@>???@B?=A>AHACECB>AA<A@@BB@<<;;CCCCB@@@@BB@B?=99908A0;7<:<;@@@@A@>7:7.8>474:@AAACDAC=@@>@AA=<;:>7737@;;;?=;;<;;=AA5?>;;9,///
@EOUL1:00992:00597
GAAGACCCTATAGAGCTCCCAGGCGTGGCCGTGAATCTGGAATATCTCGACAAGGTCATTGACGAGTCAAATGCGTATCTGGACACAC
+
DC@CCCE>BC@@@BBC@@@<::6:CCB@C?BBBB><<==<<;==??>78:@>;==@@@@>@@888>>=>@;?@A@?>=;96,,,****
@EOUL1:00976:00646
GAAGACCCTATGAGCTTACGTGCTATAT
+
CB?B9;909@9;<;;17//.,,,,,...
@EOUL1:00984:00616
GAAGACCCTATAGCTTGGGACAAGTTGCTGGAGAACACTAGACAGCGTAACAGGGGTGGACGTACTGCCGCTGGTAACATATTCGAGTGCTTTAGGATCAACCTGGGTAGCGTGGTCATGTTCAAAAGTACCACAGCCAAGTACCTGTATAAGAAGTACAATGCTAAGTCAGTATTGGACCTAACTGCTGGTTGGGGTGACAAGAAA
+
CC?CBBB@;<<CCCA@BB=@BB?BB5::;;7<;C@BBBBCCBBAB@CA@>===CC6@@=@AAAAA898?<8888>>=@AAA?>?AC?><BBC;@@=<==A>A8<<?:<;;??:8766//-101222....36>?@@@>@>===@>????AA;==>A?;;;0/0>A@<ACA?><<<<?<>A?<<:=?>=?@;=:===1<;<@B8<<<;
@EOUL1:00961:00844
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTGG
+
?@=ABBC@CCCBBB==@>=AA<@@B@ABBBAAA@DDEFC>>>>113D?DBBBBAA=@A=?779:0..//+37=>@@BD=A@>ABCCBB>A==4747?:@@A??==<2733423>??A=<=?=>=000;887655,,,+
@EOUL1:00965:00814
GAAGACCTAAAATCATATCGTTTACGGGGAAATGAACATCCGCTTTCAGCTCGGCGAAAAAATTCAGGACTAAAACATCTTTGAAAGACGCTGTTCAAGATTATCTAGTGAGATAAGGGTATGTTTTATTTCAATTTCGAATACTCAACGGATCGAATCCTTCTTTGTTTCAGAGCGCGAAGTCGTTGCTTTGGTCTTTCCTGAATTCATTCTTTTCGATGTCAAGGCTCACAATAGTTCAGCTTCGATTAACTTAGAATAAATCATGTATTCAAT
+
>?;<<;9:;::0=211//98;>;331995662434249:;>DDDCAEAAACBB=;;;>>>>*9599858;;;CC9BB?<;;6;@@<@A@>6716016868@>@AA=<<888;;=;:=>188988::.,,51,,+,0000444;.88953555;::><>>;=:=;;144;7303::;=5444...1101133035343344,--1,-,.9;7;;;;196664440++*++0++-23648856566568:883488-..8578;9;://+.++++++,
@EOUL1:00979:00837
GAAGACCCTATGAGACCATCAGAGGTGGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCAACATTTAATTCGTACAAAGATGCGGTGTCAGTTGGGGTGACAAGAGA
+
CA>A;9409>1111/.3...479:90,*,0,,*.44707801@<1108>>9>EC?BBDCCABA@BC:;:A<;;B??>:?>>?>?=@@AA?@AC@<<<@@@CCBCBBBC=??CA@=@B47::=@>B=@@@:74>;=7:4:BBD@CC@B;991968<?88;BBB=@;:67:6///75447<==5A?6373798:
@EOUL1:00957:00897
GAAGACCCTATAGAGCATGCTGAGTAGTGAGATGGGACGATAGTTTGCCTGGTTTGTCTGGGTCTCCATTTTTGTAGAGGACACCACCGGGCCTCGTCAATGCTCCTCCC
+
BCABBBCACBBBCCCCCBBBABAC@???>AAB@@A<???@@@@BC<A?989244-4898=A58300/00000+04380.,.2--,--,48-424433377111.4,...+
@EOUL1:00977:00879
GAAGACCTGCTTAAGGTTTCACGGGAGTTGTAGATTTGGAC
+
A@@<;<88>@?5954788:>?>CB<@;63111/...+.,..
@EOUL1:00952:00975
GAAGACCTATAGATGACTTTACTA
+
@;249=24484439849662-,-,
@EOUL1:00980:01032
GAAGACCCTATAGACGTTCTACGCTATGCCGCAGTCTCTGGTATTGACCATGTGGACGGGCAAGCTATATCTGTGACTATCCAAGGACAAGGAGGATACTAACCATGGCAACAAAGAAAAAAAGCAGCAAAGAAGGCGGCAAAGAAGGTAGCACCTAAGCAGTCGCAGA
+
=C@BBBC>AA;:<??B@=?@::::00/7::7A::6338==<???=?;68<888A=>=@?:==:><<<?B<;;;;88778>?>A?A?>>>?C<@@@A===??>A>@:969A588D74444444&4:=??@A<AA>@:;;8444,4;:;3844438/00233830000--1
@EOUL1:00957:01063
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
CC@B99;:;AFB>@AA@A9??A;:78?=77:37??@AB7BCBB>;;;8<AABDD@CBDDC;<;;89>@<>>@>@BB8@AA@@=AA@A
@EOUL1:00970:01076
GAAGACCCTGAGAGTCAATATCAGGTATCTCCTTCACTACTATTCCGTCATATATCAGGTCGCCCCCTGAAAATAACGGATTGCTTGCGCCCCGTGTCCACGCTTCACGCTGTGCTGCCGTGATAGTGGTATCGGCTTCAGGTCGCGGAAGCACTGCGAGTCGGTATACAATCACATACCACCTCTCGTCCTCGCTGGTGCGAATCGGGCCTGATCTTCGGTGAGGCGAGTG
+
9B?B@@><@CAABBBCC@C@>>997;A@@CC@C?BCCBCCC<@>@>=AC=99=<@=@A>@==<ACH1C===<4<A:=?;84/0348::4444-4;448>@@?@@;=<>>=788<>?@@>AACA===AC=@@;::9//,.//,/8;8959;=000;:>69///8./.6;;;../98==<<2832--...799:5/..444558855355355:<=;==?A:;55,,+,,11.5
@EOUL1:00981:01267
GAAGACCCTAGAAGCTTAACTGTTACTATAGGTAGAAGCACAGGAGGAAGAGGAGTAGTAAAAAGTGGAGACTGTAAACAGAAAGATGAAAAACCAAG
+
CD?BBBC?DBBB@CBB?A@BBAA@CCB8;:@=@:;:>?B:83337737=AAB=@AAA?30003-7>@?AAAA<<<AA@CD???<???>7777.7**)*
@EOUL1:00998:01254
GAAGACCCTATATAGAAACAAACCGCGACACTGATGCGTGCCCGAGTCCTCTACAATCTCGGAATGAAAGATCTGGGTCAGAAGTTGGTCGAGACGATCGACAAGAGTCCTTACAGCTTGACTGAAGAGGAGCGCAGAGTTCAATTCGAAGAAGATCAAGAACTTAAGGAACGAAAAGGACTCATTGTCATCTGATGTGTCAAACTTGC
+
CB@BBBCAC@A@BBBBB>BCHAC@@@AEA@@@@>>@@<88BB<9<;;;=>>>?985;@@@B=A<<;BB:888>877-6<=8=?AB?B?BBB@@@BBBCDD<;:@A@:::::9:///::6;>>AAA>?@@588<<?=@??;966+,+,,,+,;77;@;;878338845:4445====4<9<@BB==;@?=<;?>777;=334/74222-+
@EOUL1:01019:00282
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
AAACCCC@CCBB?=?:77/8A?@@ICBCB::79>B@BC9BBBF=?AB?CCGDFB?BACC@EBB<;;BB=BCB@CCC:BAAA@=@BAA
@EOUL1:01022:00366
GAAGACCCTATAGCCCACAGTTGCTCAAGTATCTATTGCATTGGTTTAAACGCTTGTGATATGATTCGTTCATGCACTAGCAATAGTGCTTCACCGAACAGCCCGGTATACGGTTGTCTGAGGCGAAACCAAGGGAAGCTT
+
DC?B;;;:9@>>99904993877>@=A@CDCCBCBC@C@??>?477.77.761637>>?AA??=<8;=>8<;>@<<=ACCBB@BAAAAAA?CAA=AB>AAAAA@A>??;>;;=>00/88/.,+,555...,555+,++022
@EOUL1:01030:00370
GAAGACCCATGATTTCATTGGGCCATTGCACGTTGCAGACAGTCGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCACCGCCATATTGCATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCCA
+
F@::444-9D@>BD?BCB?CB@87/1,/8;;=B?DCDCC>>=:111><888837843?;600;>0//1:899:BA?BF>BB>@<>===:=7759ACD>C?B??5=<:;;;?8;;;787<<<=@<@<<<?<86;;//.0@6:;;;;;==B???=:875,,,*+48;>BBBC<;;B8:6:>;?=:;8<>>8959@8:;AA=<<;>B@AC?ACC=@=@CB@@@@@@CC@BACBBBC??<:
@EOUL1:01044:00365
GAAGACCCTACGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCGACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAAG
+
DCBA@@C><94:999?;<?@;;;B?@=<?:7479?=?BC<AAA<<<B@@:>9858?>>>688/9>?8;;CAA@CBD@BDCCEC>A@A87<<9@A@<<<=@ABCD<<<?AAACEBBB?B@@<?@?47777778477811=A?=BCAABLCAC@C?B@@@>>>9988877=773:66770000+0040--.
@EOUL1:01008:00420
GAAGACCCTCATCGATCCCAACGCCAAAATCTCCCAAATGTAGTGCCAAGCCGCCCACCTTCAGTCGTAACACATTGAAATAACGATGAACTGACGCTGGGGTGACAATAAA
+
;<;=CCCCCAAECAA@CC9<<=BBABCC9;9377,9904:B@@>>?,1,0.4457::C@A@AA??ACCB@CCCBB@CCC:<;?A@@AAC?@>>;000;>>>.67>?A>ABDD
@EOUL1:01033:00488
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
CC@@ABB@B=?BABAAAB>BBBAA=?11::::?@CCDC9DAA?:?==:@BDAAA?BBHH@C@@@CCEG>A@C=@@A8@CCBCABDBD
@EOUL1:01035:00463
GAAGACCCTATGACCGGCGCTTTCAATCGGCGCTATGCCGATCAGATGCACCGTGATTTGATTCGCTTGGAAGACGCCTCACCCTCGGGGGGCTTGTTGCTGATTGACATTGATCACTTCAAACAAGTGAACGATCAGTTTGGGCATCCGGCCGGAGACGCGGTGATTGTGGAAACGGTTCGGCGCATCTATTTGGTGCTGCGCGAATCTGACAGCGTGGCGCGTTGGGGTGACAATAAA
+
FGCC???=@CCBCB@B@BBBEE?@@=@CA8<<CC@@@B@CCCBBBBCBCCC?BCBCDC@BBB@CCBCACAC@CC<><@CCCCC>B@@>>@@-BB@DCACCCCBB=@@?AA>;<<B@>?>99918?<??@DBEBCBBBCFD>BC@@A@JCD@C=?=?@>>>7744477:=7747=6;67=:9:5:??AADC@??;A:???@@@A????BA?A?<88::7:=BBBCD?CBB7BDCCC>::::
@EOUL1:01004:00514
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGA
+
CC@CCCC>BBDCDCB@B@B@BABBB?BB?BBDC@CB?A=@?@B@CAA@;?>AB@C@BCBB6////57@?CC;AAAB@CC?CBDBBACCACCBBBAABC@DC@CCCCCC?BB=B@C;<<?BA@@?B===6>=374//...,
@EOUL1:01022:00552
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATGAGA
+
B@>A@@=<@@@@CCCAB@C?BBCBE?BB=B@CC?BB?CA>A?B@CCCB:@<7737<>@AAACC@CCB@A?ADACBAB59909D@@@=<<:<CCCCC@@@@CC@CDC?>=<?@<@?C6;:>;:;;;ABBA<A?6747??<?@D@AABA?@===?A<?<<:<@@@AAAAA??<:>>4=,..6..++59
@EOUL1:01030:00568
GAAGACCCTGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGGCAGCCAGTCTTCTCAGAGATGACTGTAGCCGCGATCCTGAAGCTGTGCGCGTATCAATCAAGTGGTAAGCATGTTGTTAAAAAAAGTTGGCGTTGGGGTGACAAGAAA
+
//,////+/;@999@91,1464..,.6.,001,/6,.473779;9986:4.//-332./-300/0443/35;68<==<447;73.4822--++*+--22425532344444::99488//,/,,*,,,*,,*.//;4424424AAABB(=:231777?BDB8877<:=9=>>
@EOUL1:01003:00641
GAAGACCCTATAGAGCTTTACTCTTATATAATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
CCBDCCD@ECBBBCCCBE>AAA;;47>;00,06443733*.377319904>999424294-49889>AA9A??4999/8??AB=;8:>
@EOUL1:01016:00648
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGGCACTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCGACCGCCATATTGCATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCC
+
AABDDCCABCA99978>@<AA>A?B@<>?DCCCDAAADCCABBBBBDJDCC@AADDCCB?@>@?>110>@;BBBCC@ED@CB?BBCCCEBCB@<@:==6=477?9?;3;713/3;88=A==9<@?889?<<<A?===38858@@>?>>7//..-289/777733==9;;@<<<BCB>C?A>C@A@ADCDBBDCCA<@9999/99969999>BA@=?7<9AAACCCAAC7;:9<@>AA@<
@EOUL1:01020:00629
GAAGACCCTATGGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
888?<;<7<BAB?BBDCJ@CDCDC?ABCA??;?>????6?B@@=@@@<@CCCCCBDEFFACA@;;;@B=BCCBCCC9<@CGDA=@<C
@EOUL1:01011:00752
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAAACGATGTGCAGAAACGCTAGTTGCTTACATATACAGTTGGGGTGACAATAAA
+
CC@DBBB=BBBBB@@??@B@DCCBB?BC@C@@@>BC@C:;:?C=?9667;?BC@C@BBAA>>?@@A@8::=H>CBBD8;919B?>@@BB>BCCCBBB@A>BCABCB;4408=9>>C>CB@AA@@@ABCB6>=477.777>ABECC?>>9;;://.;9;;>099???C@@@==8<9=4<9???38033
@EOUL1:01015:00771
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
4959@>=<AA@ACCCCBC<AACCD@C::;??:?=????6@BBB@CCC?CCAABIBEBFD?AACACACC?BBE?BBB:BA?@@59333
@EOUL1:01023:00753
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTAGTTGCTTACATATACAGTTGGGGTGACA
+
EC?CDCC>BBBBBBB?A?@@BB@>>>?B>B>AB?B?>?=>>>?>>>>>;?>>>>>;????<<<@@@>8::<F>BDDC@CCBBBABA@CC@CCCCCBCCC@BB@BCBBBC@CCCB?B@CBACB@@@@??C8;:>B<?::;CBCCC;??9;;8////46//,33883333004266:4:9433
@EOUL1:01020:00835
GAAGACCCTACAAACCCACGACAGTGCCGCGATGGCGACGGGCCTATTGGAGGACGGTAGCAACGATGACTCAGGCCCAGTTCTTCCTCTCCGCTGCCTCCACAAACACACAGCGGAACATCCGCCTCATTTGCAGCCCACAAACAGCCGAGTCTGAGGAGGACTGGGCCCTATCCAATGATGATGCGTCCGGAAGCGAGCCTCGGCGAAGTATGCAAGAAACCGACACACCGTCCTGAATATGCCAGCACGTGCCGCGGATCCCCAAGACGGAA
+
ACCCCCC@CBBAB>BC>BCCBB9;;??;?A:<<C@BCCCCC>><A;;9;@AA>B@@=AA@AB@<=<AC;99A;84;B<ABD@EGCD?@@?B?BC@@@@BB>BAAA<ABBBBCCCB?B?B:88?@;7:4444,34;8;;7<=5500,+,*+++-43344/30/08344,7?6>?@@?B@CAABBBBADCKD?A7:67>>>A@?:665555+,++0.---.432233..3366<43133-.,..-3;;89<?=;6655544430333,022211,-,
@EOUL1:01025:00872
GAAGACCCTGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCTTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAATAGA
+
CC@CDBB=B?@@BBDFB@BB@BCC@B?BBCCC@@@?DDDC@@?:::8?@@@@?;885;@;=:;;//07::;9===BA@<<<A888@>==?<>>9<<<>>>??;>?>=>CECCEDDCB>111,1-1;:<>DDC@BCCCCCBEE=A@@@>,AB9<;CC>AAA7AAAAA?AB??
@EOUL1:01004:00947
GAAGACCCTATAGCGGAGGGATGCCGGCAGGCATTCAGTACGCCATTCCATACTTTAGTGAATCATTTTTTATGTCGATGATTCCAGCCCTGTGTTGTGTGGGATTGTTGCACGAGTGTTCCAGTCGACCAGGCTTCAGGTGGTATTGGAGCGACTGGTTACTGTTGGTGGTATGCCTGGTTGCGGCGATCACGTGTAAGATTACCACTGGTTTTGGGCTCGTTTGCTGGCTGGTCTG
+
DD@B@@B@AAA?AA?>??B>BBAA>AA?8?=??@@CCCBCCCC>BB@BABBB@@@;@ACCC=A?BBCCCC-::;CFBDC@AA@ABCCEC?DCDBB>B@@@@@:@@?BB@BCCCCCBAAB?B?CAACBAB@CC?BB????<??>ACC@C@BBABBDBA?A?AAC<<8<@CCACCCDC@@@<?>A??<@C:::CAAABB@DC;;9;B@?A@>4888.7<7===77757>>;9:0/,0007
@EOUL1:01014:00946
GAAGACCTATAGAGAGGGTTGGGGTGACTAAGTAGA
+
AC??9954849400...+.,...*.54.168889<=
@EOUL1:01027:00920
GAAGACCCTATAGAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGGAAA
+
CC@CCDB=?@@61.,0,1,/../6589:==AA@EC@C?AA@C?????9@>@@=B@CDCCCB?@@@>8:8;D?CCDC@CB>BB<<<@BB?CBBBBBB@@=@C?@@@BBB<BB<@=@9<8<A@;;;ABBB<@@474777:???BA??>:=;48782;;<8<@@?888>;99/333+04;;>6855==
@EOUL1:01014:00994
GAAGACCTATAGAGCTTTACTCTTATATATTAATAAAGAACT
+
:878//,/8887AA@??9?CCCC=?:::@?=<90//,--,--
@EOUL1:01024:00968
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
CC@AABC?BBCCCAAAAA=ACCCC?AAAA9:47:=::91=777=79914@@@BC?B>BA<A@@AAACC@CCB>AAA7ACCCC998EC
@EOUL1:01005:01014
GAAGACCCTATAGACCAGCGGTACTGGTTTTGGCAGACATTATATTTCACACATTGTAATGACAGCCTTGTAAACCGATAAGCACAGGGGAGTGTCACTAGTTGAGTATATTGAGTATATCGATGTTTGTCAAAAGCCAACGCAAAAAGAAAGGGGAGAAAAGAATTGAAAATAACAGTAACTCCACATCTAGCAGAGATTGATCTGCTCCATTTCTATTTGTCGTTGTTGGGGTGACAATAAA
+
CB@CCDB?BBCBCBB?BBBB?AA????BBB7B?ABAABCC???@@C<?CCCCBA>B;;6<DBCCCG@B>A@@@<@@AABB?;;;B@AAB8ACCCCCBBBB@?>B????@@@@@<<==@=<<=AC@>@=@=@@EE>CB@B?BCCCCC>/611,666-:BCCC8AA>A>?=;;14434::;?=BDC=?<?>>00000/,7,/,-...06<A@<<?C>A888C;9AB?;8;:7:==1>@?;;:7777
@EOUL1:01022:01125
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCCTGACATAATACATTTGTTGAGAGGGAAGAAGCAAA
+
;<FECCBBCBB@@@68808=??ECCCB>C@@C:::>A>A@C@???@A?888;<442444/4744;;/44+3339>@?@@9888@>DBBBI>B>CGEBBA@A;;;?:@@CCBACCDCC9:;B<<:=CBAB@A@??0858;>@BCBBA@@B::6:4744::444-4;::8;==:;8;<<<???@A@@;;::7:443888?>@AA5>??:988A>;;8733004000+00005::544?===;==?=:3038,423:78;;A6
@EOUL1:01027:01250
GAAGACCCGGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCT
+
<<;>9990;?4;<CBBA@AA;=975793313469999944233311,11116995658>;1000000211--+
@EOUL1:01066:00377
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAA
+
DC@CBCCBBCBB??;B??<BBB@><<96:?@=?=A?AA7AB>>;:996;BBBBC?B@CC@CCCBBBAB=?@@?BBB9BCCCC@CCCC
@EOUL1:01072:00397
GAAGACCCTATGGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAAGAGA
+
<>AHDBC@D>=<ABB=@=@>A@@BB>ACAB@CCBBB?B?BB?C@C@??>C@CC@B?BBB:??C@DCA@@=@B=AABA8<<8<ABBBAAA@AECCBBBBB@CDCCAA;;;8;B?C@C?BBACCBBABCCB8@@5848877@B@A@???9;;8CCC@DAA<@@AA===AAAA>ACD6???AA899BB
@EOUL1:01078:00459
GAAGACCTATGGGATCTTCTCCGGATCAATCTTGAACTTCAGATGGGTAACTGACATTGGGGATCTCCGAAGGTGAAAGGTTATGAGAGGGATCGGTAAGCTCTCGGATCATGTAGGTTTCTTTGTCGAAGTAGAGCTACCTCTGGGCCAGTAGTGGCAGCGAAGCGGTTCTTCTTAACGACAGGTTGGTGGTGTTACGCTCTATCGG
+
//7;11-////1,/@>?=<<<AGADBCC@BBA@CCA@@=@@999>@<@A>@AAAB<<:;BB7BBBCDACC@C?CDDH=B?B9<<AB<9;B999;?;??>;;<DDDD?BBBE@A@?@?@E>A@A<AAA??>?@A@AA?8:;7///675:@@@@?@@@>@;8//.,.-;9;:==;<@?C3///.----,.7.-,-77,++++-3669<88
@EOUL1:01054:00543
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
IECAAAC>CAAACCCCCC=BBBBB?:::::=;><;???6@@CC@@AA>AAAAAA=A?CC>CCBBFCCC@A@@@CDD7BACCC=@@AA
@EOUL1:01091:00529
GAAGACCCTATAGAGAAACTCATGAATTATGTTACATCCCCCTCCTTTGGAAACAAACCGTGGCTCCTACTCATGCCTCAATGGGTGCATAAAAAAGATTATTACTTGAATGCCATCTCGAAAAGAAGAACATGCGACGGCAGCAAAGCGACAACAAAAGGGAATCCCCCGAACCCATTTACATTGTACCAAGGAAGCGATACGTGTACCTTCCACCGGCGAATTTTCGTGAAAAGAAAGATAGCGAACGTGCATAAGAAAAGTTCTCCGTTTGTTAGCATGTGGTATTGTTGG
+
CC@C@?@@ACCCCAC?B>AABBA@@:?:????<>@999999-99?AC?C@BC?AAA<>@@99798C<9;;;A@@@A?CIC@ECC>BBBAEAHDHN-C@@=;;89@B>AC@ACCBGCD<<=GEG8A@<8847:;BCECCC=@>999909@?ABB@CCCB9DE>A<77777,7>699088807>>A=??FC@C;;8;@B@:<<>>;;;:11,09:::>;?@CBB@BCB5AB:::CD9DBA=B=7668//,/978?A??:=>>>.::=?;;8==A;@@?C:::@;832400/03204
@EOUL1:01053:00600
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAAACGCTGTTGCTTACATATACAGTTGGGTGACAAGAGA
+
EC@C@BB?BCBBBBB@A@C?BBCBB?BD>C?BB@B?>>>?@>@@A@@@;A@@@@@><>AB@>B?BB?AH@CC?CCCD@CC>BB;;;@BB?BCECBBCBB@CCBC@@@@>9>B<B@C@BB?@@@AACBBA<;:47:=777>;;//,///,.7736?447;8;=??55;:304244-0004<<==<;
@EOUL1:01073:00611
GAAGACCCTATAGAAGACGT
+
997<AAA>BB@7--+,,**(
@EOUL1:01087:00624
GAAGCAACACAAACACCCACAAACACAAACACACGCGGACACACTCACTTGTCCCCGAGTGCGCTGCTGATGTGGCGCTCCCACATTGCTGAGCTGTCAACAGCATCCACCAGCTTTGACTTGAACACAGCACGTGTTGACCCAAGGCGCGACGCCGCCGCTCATGCTGTCACCACTCACACCAGCAGCCAGCCACGACAACACTCCCTGAACCGCCTGCAGGCAGCCGAGTCGT
+
CB@BCC?AABCC=A;;;;;CCECBBBB>;<<<==BBC@BABAA@@@CCC=@??A?5ACCCFCB=<?CCCCA@@@?B?ACCC=?::<?<;;CD??@A@@@@DDBCBBC@BB@BB:881:@BC@BB@CA???>===888<?AAA<A>A>@00/7:753334..-.-3-34>54.54.//....9;:66934235;5688<6858//.6604--,13;<:?<<?;-..44,..1-.--
@EOUL1:01092:00628
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAACGCTGTTGCTTACATATACAGTTGGGGTGACAATAAA
+
DC@CCCC>C@@BBBB=>=><@BCCB?BCAC@??@@B=A@@==??BBCB?B?BB>>;>?@@@=>?@A?<@>AD@DBCB@CH>AAABB>BC@CCBBCC@@@@CCBCCCEAB=BCAC@CACC@BAA@@@BBC:==,0,000/:=?=@>@A;==?<<=:;4937?>><==@AAC?ACC6@??>>8<;;=
@EOUL1:01097:00626
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACCGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGCATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGTCTTGTGTTCTCATGAGATCCCTGACATAATACATTTGTTGAGAGGGAAGAAGCA
+
:<;=CCC><99;449;?687:DE@@@C@DCC:6163101010.....,.,.3311:86<<:898?==8<<7<<<A<;;<=8;;4439??><<<C?DB@CDFECBCA<A999>>>833766;=;=;9444=;==BB=B>=<9;7;:??B>@@B@C?B@>><777,6044/4.997--.311116610000044555;55:56647--21---,---000000444000000165664222+--,---04948;33/011
@EOUL1:01092:00667
GAAGACCCTACGAGACCATCAGAGGTGACTGGTGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCAACATTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGTGACAAGAGA
+
ECEDCBB>DCCB@:=;@CECBB??>@@B@@@@>AB?:::6<@ABBBBBC?CCB@ABB==<@@:?CCBBBCBBBAAA<BBBBCC@CCC@@@BBBB@AAC@CCBBCBCC=>>>@@=@C@BB@@C@C=@@B???@?CCDADBBB?BB@::<C:;8<@CCCCCCC@BBCCBB6;:7:BBDC@ACC4@?@@@?@@@D
@EOUL1:01071:00742
GAAGACCTGCACGGTAAGGT
+
B@?@99544444/,//,/,/
@EOUL1:01073:00751
GAAGACCCTATAGAAAGGGTTGGGGTGACAAGAGA
+
C@>@999-9;?@@>>-78/84888.879;;:<CDC
@EOUL1:01088:00810
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAA
+
CC@CBBD>BBCBBBB;;;67@@?@<??=777<A<>@@A6??@>=7<919A9<=FBD@CDBDHBAABBC@CBB?BCC9DBB>?=888
@EOUL1:01053:00945
GAAGACCCATGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGGCCCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGATAGACGTCTTGGGTCAAGGCCGCGACCGCCATATTGCATGATGCGCAAATTGGCGAAGAGCGCTTGGGCTAAATCTTCATCCTGTTCCAGAGCGCTAGCCAGAAGCGTCCC
+
DCACCBC<@>>99919?B7>>19596<BCCGCCACBBECAAABBC@CC>A@@A?8848?<=<<:<1////+4444424969=>?BBABB@;7717477,601119=9;>>16218634=8986>?88888778=>=<5;943332044443000234/0033/049@BB999999B>B@E@E;<;<C;;;BB?BC@CCCCAFFD@CBBBA88858699>8778=><?<??@;>==777-
@EOUL1:01064:00937
GAAGACCCTGTGACTAGGATTGCAATCAGCATTGTTGCGAATGTCCCTCAGAGTGTTCGGATGCAGCCAGTCTTCTCAGAGATGACTGTAAGCCGCCGATCCTGAAGCTGTGCGCGTATCAATTCAAGTGTAAGCATGTTGTTAAAAAAGTTGCGTTGGGGTGACAAGAAA
+
CB@A9997<AABECCCB@BB?BAB?BCCFBAA?ACGCCCC?BBCCC>BBBA?;;;;8?@=?@A=8788::;::387==>????B@@>8888;<;==;<<<=;=;;8;?BDCFDDDEDC@@@5959><??@@DACCCBBH@A@499999*9>=@AAB@CDD8@@BAA=??AA
@EOUL1:01076:00937
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAGCCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACATATACAGTTGG
+
CGACBBBACBCCCCB>C=@8:;A??=@@<@B::5;>4747737;@=?<?H@@?<?>BB@??:<?A==?B@BD@CCDD@CC>CABBB@ECCDECBBBCCB?CC@CCCBBBCBC<A;?5;8<@@;;:AA??:773747><???>>>@A@<@>?>883/:858://///77//,/.
@EOUL1:01077:00936
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAGA
+
CAEACCC@BBBBBABBCC??@@@?;?AAACCAD@BBA<5?;??=@CD?CCDCCB?A=@@:>@A;;<BB>BBB?CCC9CCCCC=@@CC
@EOUL1:01093:00986
GAAGACCCTGATAAGATGATGGGTGTGGACTACTCAAAACTGGTCCCAATGCTGGTCAAGGAAATTCAATCATTACGTGCCCGCGTGGTACAACTAGAAGGAAACTAAGACATGGCAATCACAAACACTTGGAACGTCGTTCAGAATGGACGCATACCCAGAACTGGACGGCGAGACAGACGTAGTATTCACCGTGCACTGGACATTGAAATGGTACGGATGGGGCATACGCAGGTAGCGTATA
+
9=;?BACACCCCC@CGBBBBBB<AA::7;A:7:@@>>>2>=?5::B:@?BA???=@@?=@>@B:@<842444:78?@@99909=489EBBCEABB?995489959=A@BCCCCDBB>:<11111,169=?C@B;;:==70,02-,*,22,,,,,.7777@<;:688558586534466>>78444444/077778144:44244//,//858;;;>>A899B??8744444//65......111
@EOUL1:01096:01014
GAAGACCCTATAGCTTGGGACAAGTTGCTGGAGAATACACGTCAGCGCAACAGAGGTGGGCGTACTGCCGCTGGTAACATATTCGAGTGCTTTAGAATCAACTTGGGTAGTGTGGTCATGTTCAAAAAGCACTACAGCCAAGTACCTATATAAGAAGTACAATGCCAAGAGTGTATTAGATCCTACTGCTGGTTGGGGTGACAATAAA
+
<;8<BCB@CCCBBBB@BB<@BB>>A@?BB?ACCA?DAAABCACBB?@@C@CBAA?>@@A=A@?@@@A@>@A@?=@A?A????=?ABACA???7<<<:>@C@@@@EE>B;;;BA??AAC@?<;77777,7???@A@:::;?;?CDCAAA@@::7//,/55*,+,55+,..5,,.0//9;9==A>A@@@?>==8:2444-44:9:9;::;
@EOUL1:01059:01092
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGTAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTGTTCTCATGAGATCCCTGACATAATACATTTGTTGAGAGGGAAGAAGCAAAGAT
+
998<BCBABCCCCB@BC>A????CAA?=??;??BCAB?B@C?C@?@@378;=;;<====?AAAA@@>@@:777;;;;@@>A@AA?D>==ABBBB;<8<@AAA@CCAD>BBCBAA@??6;=>>@>@<<9A=@99919<@@CB@??:777010,0:>:1;<BBEB??AB@CBBBB?A>??A=<;:8/////76//..889,///+///797.7/403///.987?A;AC=<=?@8::;444-44347789;././979///,/2-
@EOUL1:01070:01055
GAAGACCTAGTGGACATCTTCGACCTGCTGTGCGGGATGTCTACGGGTTTCATAGACAAGCTGATCAGTAGGAAGAGGAAGTTCAAGCAGGAGGTCCGGGAGACCGGCGAGAGGCTCAGGAACGAGCTCGATGTCCTTGAGAAGCAGCTATTCACCACCAAGGTGAGACGGAGACGCTCAAGACGACAAGCCCCCTTTAAAACCCCGGGGTTTAAACCCCGGGTTTTCCCCTTGGGAAAGGGGCCAAAATTTCCCCCGGGAATTTCCCGGGAA
+
11,111,/6611,/499;B?AABF>???:;;?B@C=@@D?@??=;6,11,133366??88:76999?AAAD@A>899?D>?@58;60<<87//-/>>;=9@@@@>C>@999>=<;0//75:9::;;@@@C=;337,/,/66:<@8879833233995331010-.-233449=A>?@54458733./.944755.77477747774433*--+--+-344==;<<8.533.1,--+--+-22.10111.1344775.54-.--3222011+--
@EOUL1:01071:01052
GAAGACCCCTGCGGATGAAAAGAGCTTGATATTTGGCTGCGGCAGTTCAACCGGGACGGGACACCCTATTTTGCTCCAGACGGAGACTTACCCGGCTCTACAGGTGCAATCTCAATCATTAAAAGCACAGGCTCGACCAA
+
@;944999,4;@?=??8899/4337?@BB@;;;19>ABCCEBCA@@@CC@B@DC>B:::68??@BB>@@??61///55..4=88=@<=8<?;172-11----2222:=<8834A>;=;A>A==5=86553,++0+++*+*
@EOUL1:01072:01090
GAAGACCCTGATCCAGAGCCTCCTCCACCAAGAACAGTCGTTTT
+
7878444-49949>:;;88694:;@8;9?@64:9:44///,,,*
@EOUL1:01067:01118
GAAGACCCTGATTTCATTGGGCCAATGCACGTTGCAGACAGTCGGCGCTGTCTCGGTGGCACCATAGCCCGCTGCCAAAGTTATCTCTTCACCGGTTTGGCGCACGGCGACCTGCTGCAGGCGGCGGATAGACGTCTGGGTCAAGGCCGT
+
CB=@449-8449969@B>AB>A?B@AAABABCCDCACCCBB;;969988@A@@????BD?0,+,,,***),,,4437?;7;5668888>86636000+0/033300/03332300000044222.-,.4833455.-.221110131034
@EOUL1:01058:01158
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTTCCGGAAGCCGCGTCATAAACGGAACGATGTGCAGAAACGCTGTTGCTTACAAATACAGTTGGGGTGACT
+
CB@CBCC?CBCCCCB@C@B?AAC@?=?E>D@BB=B@@C=A;<?<?===<A@@@775:?=??<<8<>;>B@CD@AACA=@@<@BBBB@CE@CCCBBBB<?<BB=><<7889/8=?>F@DB@AA:;<C?CK=>>,0,0/0/8>@===?C@?<<<;:8;::799994999;9;832400+++*
@EOUL1:01084:01201
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAAG
+
CB@CBBB>BCEBBBC@??<?A@@@>@::;BB?B@AAAA6C:::@?@@<@B>A>@<@59919D@@@;999;BB@?>>9>@88=+-,*,
@EOUL1:01085:01239
GAAGACCCTATAGTGGTTTATGTTATGTGTAAATATCGTTACACTTCTGATGGTTTCTTAGCTTTGCCTGTTTCTGGGTTGGGTAGAGAATATATCGTTGCTACATATAACGACATGGGTTGGGGTGACAATGAAT
+
CCBCDCCACCBBA@@=@E>AABC?BBBBB???;@BBCCA??A?AA>@<=C??>AA<AB?CCCCD>CC?AAAB>BCCC<A;;;6;==>@@<???888AA>A@@===@@?A>@999@AAA<A@???498877,/-,*+
@EOUL1:01091:01238
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGCTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAAACGATGTGCAGAAACCGCTAG
+
@@>BCCC?CBABACC<>=@=??BCC?BC>B=??<@@:?8?==@<??AC<B?BBBC@BAB?@@==@@?8<8=EAD@AA@CC=BBAAA@CC@CCCCBDDCB?BF;>>889908?9A?C>AA@?????88?7266:;<6777046631122+.,---11
@EOUL1:01067:01269
GAAAGACCCTATAGTTCCCTTATACTGAACAATTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACATCAATGTGACAGTCTGACAAGTGGCACAAGGGGTCTTGTCGTTCTCATGAGATCCCTGACATAATACATTTGTTGAGAGGGAATGAAGCAAAGACGG
+
444-444419>>@;:58D/84;??=BDB@CB?C?CC@C===A==;<@@-111;::>7?::5;C?>>>477.7====?;8:5:AAA?CCBBBDAA=AC?BBBEBACD@D>BBCCCAAAC<?@A?@=ABCCE@CBBB8;8;@CAAAB:::?@??:?>><;:=88818??@>@@???<??A???AAAA@@?<<:;=;9////,///+,5555//.53335<444866:=134:4315:0066442335333444+0/033;=BBB9:..-1
@EOUL1:01121:00503
GAAGACCTGGCTGAATGCTGCGTGTGTGAGGTCGCCTTTGAGGTTGTAGTTCCAAGCGTGATAGACGCAATAAAGTTTTTTGACCTTGCCAAAAGGCGTCTCTGCAAAGCATGTTGCCTCGTAT
+
@?=111-//:<1/11<A?@B?:8;BCB@@@:????=A?;@@@@@=A@?=:34347:>@@@@BCA@@@@@>=@A<??AAAA->??9;79;2444-48:88<777==@@9999888,.*)*,,,,,
@EOUL1:01134:00548
GAAGACCCTATAGAGCTCCCAGGCGTGGCCGTGAATCTGGAATATCTCGACAAGGTCATTGACGAGTCAAATGCGTATCTGGACCACACG
+
496<99BA<<=AACC>>>98//.0:9:,:8:A@A;<<<=;;889////./@999//.?>:=?===CACCC188AA<<<;:65,*)*****
@EOUL1:01119:00593
GAAGACCCTATAGAGCTTTAACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGTGACATAGA
+
DC?@==A69;9=:?;4;4+0,.2,,*,.,,..555../0+0<97=79904449998=;9@?>BECBCBB;;8>6991399443733
@EOUL1:01144:00590
GAAGACCCTATAGCTTGGGACAAGTTGCTGGAGAATACACGTCAGCGCAACAGAGGTGGGCGTACTGCCGCTGGTAACATATTCGAGTGCTTTAGAATCAACTTGGGTAGTGTGGTCATGTTCAAAAGCACTACAGCCAAGTACCTATATAAGAAGTACAATGCCAAGAGTGTATTAGATCCTACTGCTGGTTGGGGTGACAATAAA
+
CC@CA@=>B@@@BBB?CD@CDC@BA?ABAA?BA@=B@ACCCBBBBBBCA?CB79:=?9;7<>>?AAA@?A=;<:<=;<<<==>A=9=@<===;:::;<=B@<<;BB>B?>@@@@>@@A<??=777:4:AABCCC:::?B?BCFD@C?@?=@<7868A??A688?9;8<=:;;//0;;8<=@=@@>=@?88343444-444;;84444
@EOUL1:01100:00633
GAAGACCCTATAGAGCTTTACTCTTATTATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
@B?@994-4;;@A@@?BB<BB>88477400..,.,..5754336069@<>999995959D@B9949444-4882444,433442477;
@EOUL1:01140:00628
GAAGACCCGTCTTTTTGACAAGCTCCTGCAACTGCCTGGCTTCGCTGAGATCAAACGTGGCGGGTTTGTCCGAAAGCACGTGGAAACCGTTCTCTAAGAGCAAGTTTTGCCGGTGGAAAGTGGACGTGGTTGGGGTGACAAGAAA
+
A?=ABBB>CCDCCCC3ECCC?CB@>;@B;;7;CBA?BC>BB@BCCCCBB@@99919@@@<99919B:@?A?@AA<AB@AA==<<C>EDCC@A<;11,/886==?AAB9@@=@?CB?CCAKCDABAA884857?B5AAABB?BAEE
@EOUL1:01130:00664
GAAGACCCTATAGCTTCATCTGCTGCGTTCATAAACTCAAGTTCGTCGTTTGCTTGCACGCCGTCCGAGTTCAAACCCTCTGATGTTTTGAACACAAATGCTTTGGCAGACGGCGCCGTAATAACTCCTGTGTAAGTTGGGTTTGCTACAAGTGCTTTTGTTTGCTAACTCTGTATCTACGTAGTTGGGGTGACAAGAGA
+
DC@@@AA?BBBB?@@?BCCB?>BBBB>B@BBACA=>@@@@??;?@@A=<:85558<@@@AA?=999<C??=BBE>CC>===???AAAA6@?=>>?AC<?>87717;?<?<8858@AAABB<@?<??@=@@???A;=A<>>;==;==<<;=<<<9<===4;;<;<===;=<<<999<?7::::../,/..+/77;<:====
@EOUL1:01115:00722
GAAGACCCTATGAAACTTTACTGTAGCCTGTAGCTGCGTCGCGAAATTGAGTATATAGCGTATGTGGGAGCCGTCGAGACCAAGTCGTCAGATTTGGTGGAGGCAACAATGTAACACCACATATTTAATTTTACGAAGCTTACTTGTTTTACAAGGACACCTATTGGTGGACAGTTTGGTTGGGGTGACAATAAA
+
EC@CCCC?BBBBCCABBB?BBCBBABA@ECBAACECC==?CBBCD>@??@AB>B@?><<BA??@??A689<ADCBA?<<=@A=@@A@?>>?<<<7=>@A>@A=AA??@>@?@?>@??<??>B777-48:==5>@@A:=<?;=;;7:<;:1<BA>BB<<;@?BBC@C=@@>ABCCGD@CDD@CFF9FFDCC@;;::
@EOUL1:01147:00701
GAAGACCCTATACGGCATCCCTATAAATGGCATCAAACATATCTGTCAGCTTCAACAATGACCCCTGGAAGGGATCCTCAATGGTATCACCAATCTGATACACCCCAAACATGAGCCAAGCCACCAGACCAGTAAACAACACCAATTACATCCATTAGACCAAATGGAAACTTAAGGCTCCAACACTCAAAGTTCTTAATAACATGTCATGCATAATTCAACTGCATGCCATTTGATGTGA
+
DC?CCCC@BBBBB?99478808???@>AAACBBBDD>EBA???????DAAAA@@>?A>AB@@@?1=A4869C<??B;::4:;;;=;;;??=@>BA@??>784::;1<:6;=>;:;;;;...356,87;;;;>;<<7<@9<=;;;586////8468:;;;68>A9==>AA<A@>@9<8345047//++0000+.,--,-03=:@A444///57888.837>;=8715,++*+303.3///66
@EOUL1:01146:00773
GAAGACCCTATGAGACCATCAGAGGTGACTGATGTGTAAACTACATGCCCATAAGTACAACCCTCGTCGAGTATGGGCGATGAATCAGACATCATAGTTGCTGATATTCTATCCTGGATAACCAATCTATGAAGCGGCACTTCGACAGTTAATTCGTACAAAGATGCGGTGGTCAGTTGGGGGTGACAATAAAA
+
FEADBECBCBAAA9;;=@@@BCCD@C??CBBB;:7?:<;9=CB@=?@>?<@CD@CA@=?488037>;?BCCBCBAB>BCCBAA?ABBB8:=CCCBBCC@C@@@AC>@=?::;ABCCACCC>@>??BAAABAA>A??:??:;8;CCCBBB?@5:5::4>>@@9?;===>;>@?884::34444,4:::;799999
@EOUL1:01111:00821
GAAGACCCTGAAGGCTATTCGTGAAGGGAAGTTAGCCTTTATAAAACCTGTTATGATAGTTACTGATGGCCTCTGCGTTGGGGTGACAAGAAA
+
<;7=AAA<A@B8<;<CCC>@ADBB6;;089?@>?<;377,7>:990744..-/557:==??<8;;444534334459956:603300026449
@EOUL1:01118:00872
GAAGACCTATGGAGCTTTACTCTTATATTATTAATAAACTATTGAAAGTGTATTAATTTAGAGAAGTTTCGTTGGGGTGACA
+
,70611,/7//,14489-4333313887011010111,4443778;6966/1/,/778+//8//,/685/,,0000*/..//
@EOUL1:01127:00856
GAAGACCCTATAGAGAGGTTGGGTGACA
+
DC@B??@A@AAA>;;00,.,.5+.,0,.
@EOUL1:01138:00873
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAATAAA
+
CC@CCCD=BBB:88>88807?@?@>?????@=@;?<;<2=>=>;?@@<@BABBB@@59919@@@@;;919@A>BCC7@@@A?;??>>
@EOUL1:01125:00983
GAAGACCCTATAGTTCCCTTATACTGAACAAGTCAACCAACCACTGAACGACATGGACTTCGACACCTTTGACACCAGACCTCTGGAGCGAGATTCAAGATGCTCCCGGTGAGATCTTTGACATTGATGAAGCAAAGGAAGATGAGCAGACTTGGAATGAGTTTGTCAACAGCAATGTGACAGTCTGACAAGTGGCACAAGGGTCTTGTGTTCTCATGAGATCCCTGACATAATACATTTGTTGAGAGGGAAGAAGCAAAG
+
BA>@999-49949>@DD?B?BBBCBBB?BB>AA@@<@<?<?@?;;=;344444999;==388>===37814::4424334/4;=;:;<;;;99@;@A>BBCDCCCCAC?CCBA??>??::<;CD@CCCBB>AA@>9>5850000.333344627279744000+00488:<<=@>@@A@CBBDC:;;=@;7;8424:;;344-49979:442000033040000+0----.553374:=489<9=9===;=<9999=::;=
@EOUL1:01136:00959
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCCGGAAGCCGCGTCATAAACGGAACGATGTGCAAGAAACGCTGTTGCTTACATATACAGTTGGGGTGACAATTAAA
+
DC?CAAB?BAAAAA?@??>=BB@BB>AA>A?BB?BC?CABB@B>A?A@<B@CC?B?BC@<<<?>AA>8A>AIACC@@=AC>CCCBB@CB@CBBBCAA>><AB@AA@?@@9>>9@?B?BB@CCBB?@@DB?==3737777><9//,///,/7/.34/4:::>A???@???@@>@BA5?///8,/,---
@EOUL1:01136:01043
GAAGACCCTATTAAGCGCGAAGTAATTTCGGTAATCTGATCCATAAATTTAGTTAAAAAAATACATAATAAGCAAGGCTTTAAGGATTGTAAGCACTTTTTCTCTGGTGTGATGCGCAAAAAAATCAATTAAAAATGGTAAAGCGATGCCAATTACTTCTATTCCGGGCCAGTTGGCGTAATTGTTTGCGCTTATCATGCAGCTATGCTGTCTGCGGTTGGGGTGACAAGAGA
+
BBADCCK@CBA@C@BABBBC@CCC@CC=@?<?@5::A?<AB?BBCC>B=;:33/444444&048:?C;;:@ACC@C@===9<?A@==;=@@=@CC@BBBB2@==@C@C???CB?>>7777777&7;=:>>BBBB2BA<?AC?AA?????@C=@BACC@CCCC>CACC>@=BBB@A?A@@?<=933717==?A>A;;;;;//0;;::;@989AAAAA=A=ACC6?:54<;<<<=
@EOUL1:01106:01070
GAAGACCCTATAGTAACCGGAGTACCAAAGGTAATGGTTGTTCCAGAAACCGTTCCTACTATGGTGGTAAGGGTAGTTCCCGTAGTTAGGACATATGCAATAACTACTTTCCTGGAAGTCGCGTCATAAACGGAACGATGTGCAAGAACACTGTTGCTTACATATACAGTTGGGGTGACAATAAA
+
CD@A@A@<A@BCBBBAB?B@BCA<@=?@7@>??=?C>A>>?@C@<9884?=AA@C?BBB@B@@@==:4:8;C?B@A@>BB>BBBCB@CC@CCCA@@EBB?BC@A?A;8808<?A<A=ABAADDAADCBA>AA=A@CBBBBBBBB@AA=@AB;;<?ABB?BAAACAABAAC@CCC7AAACC?????
@EOUL1:01112:01070
GAAGACCCTATAGAAAGGAAGGGCGCTGGATGGTGACGAGGATTTCTTGGCGCAACACCAGCAGAGTCTTATTCATCGCGAAAACACTTCTCTCCAGCGTAAGCGCTGCCCATAACGCAGCATGTCCAAGTCTGAAAGCCTGCCTGCCAACCAGAATTGCCCCCCGCTGCAGCTCAAGACCA
+
@A@BAA>:?>493880399:>>;ACCCC8:;@=@CCCDCC@:::59B;><?@A@>A@@=@=<;>===<<9?B=@?<<;==>==08888=??=??373<====???@@@@C<???>@989AA=7544-/-.4334<=;=A8==96:/0./777777442388888*340444993434400/-
@EOUL1:01139:01127
GAAGACCCTATAGAGCTTTACTCTTATATATTAATAAAACTATTGAAAGTGTATTAATTTAGAGAGTTTCGTTGGGGTGACAAGAGA
+
CB:C889-8888BA@@BD>BBBCC@:7/000,.4----*.17;>ADDAEC><<><>69919>>99999-4999<AA7@>9;;:3:7=
@EOUL1:01145:01114
GAAGACCCTATAGCCCCACAGTTTGCTCAAGTATCTATTGCATTGGTTTAAACGCTTGTGATATGATTCGTTCATGCACTAGCAATAGTGCTTCACCGAACAGCCCGGTATACGGTTGGTCTGAGGCGAAACCAAGGGAGCTGGGG
+
9959>>>:@>>>8888,377=?F>BBBBDBCEBAAACC@CBBB====@>BC=A@===@?>==;;<;<9<=@@C???ABA????<:?CAAAAC?:88>AA=>=444-4:::8;;;:;4873003465::<=9984759,/.-----*
@EOUL1:01112:01152
GAAGACCCTAGAAGCTTAACTGTTACTATAGGTAGAAGCACAGGAGGAAGAGGAGTAGTAAAAAGTGGAGACTGTAAACAGAAAGATGAAAA
+
CC@CCCC>BB@=;@;:5:=?:::>B?<;;BD@CBAA@???@@A?8868>===<<:??@@=AAA19;;./0;;=;;<:4>=777.6>>00000
