package Graphics::PLplot;

=head1 NAME

Graphics::PLplot - Perl interface to the PLplot plotting library

=head1 SYNOPSIS

  use Graphics::PLplot qw/ :all /;

  @x = (0..5);
  @y = map {$_ * $_} @x;

  plsdev ("xwin");
  plinit ();
  plcol0 (1);

  plenv (-0.5, 5.5, -1, 26, 0, 0);
  plline (\@x, \@y);

  plend ();

=head1 DESCRIPTION

This module provides a perl interface to the PLplot plotting library
available from http://www.plplot.org. The interface is very similar
to the C interface except that:

 - Arrays are passed in by reference
 - If the number of elements in an array is required by the C function
   the perl interface calculates this automatically [eg plline]
 - Return values are returned and not supplied as arguments

=cut

use vars qw/ $VERSION %EXPORT_TAGS /;
$VERSION = '0.03';

require DynaLoader;
require Exporter;
use base qw/ Exporter DynaLoader /;

# Setup export tags
# Simple to generate from XS:
# cat grep c_pl PLplot.xs | awk -F\( '{print $1}' | sort | uniq
%EXPORT_TAGS = (
		
		'all'=>[qw/
			   pl_setcontlabelformat
			   pl_setcontlabelparam
			   plP_getinitdriverlist
			   plP_checkdriverinit
			   pladv
			   plaxes
			   plbin
			   plbop
			   plbox
			   plbox3
			   plcalc_world
			   plclear
			   plcol0
			   plcol1
			   plcpstrm
			   plend
			   plend1
			   plenv
			   plenv0
			   pleop
			   plerrx
			   plerry
			   plfamadv
			   plfill
			   plfill3
			   plflush
			   plfont
			   plfontld
			   plgchr
			   plgcol0
			   plgcolbg
			   plgcompression
			   plgdev
			   plgdidev
			   plgdiori
			   plgdiplt
			   plgfam
			   plgfnam
			   plglevel
			   plgpage
			   plgra
			   plgspa
			   plgstrm
			   plgver
			   plgvpd
			   plgvpw
			   plgxax
			   plgyax
			   plgzax
			   plhist
			   plimage
			   plinit
			   pljoin
			   pllab
			   pllightsource
			   plline
			   plline3
			   pllsty
			   plmkstrm
			   plmtex
			   plpat
			   plpoin
			   plpoin3
			   plpoly3
			   plprec
			   plpsty
			   plptex
			   plreplot
			   plschr
			   plscmap0
			   plscmap0n
			   plscmap1
			   plscmap1l
			   plscmap1n
			   plscol0
			   plscolbg
			   plscolor
			   plscompression
			   plsdev
			   plsdidev
			   plsdiori
			   plsdiplt
			   plsdiplz
			   plsesc
			   plsetopt
			   plsfam
			   plsfnam
			   plsmaj
			   plsmin
			   plsori
			   plspage
			   plspause
			   plsstrm
			   plssub
			   plssym
			   plstar
			   plstart
			   plstripa
			   plstripc
			   plstyl
			   plsurf3d
			   plsvpa
			   plsxax
			   plsyax
			   plsym
			   plszax
			   pltext
			   plvasp
			   plvpas
			   plvpor
			   plvsta
			   plw3d
			   plwid
			   plwind
			   plxormod
			   plgFileDevs
			   plgDevs
			   plClearOpts
			   plResetOpts
			   plParseOpts
			   plSetUsage
			   plOptUsage
			   plgfile
			   plsfile
			   plgesc
			   plGetCursor
			   PARSE_FULL
			   PARSE_QUIET
			   PARSE_NODELETE
			   PARSE_SHOWALL
			   PARSE_NODASH
			   PARSE_SKIP
			   MAG_COLOR
			   FACETED
			   SURF_CONT
			   BASE_CONT
			   DRAW_SIDES
			   pl_Alloc2dGrid
			   pl_MinMax2dGrid
			   /],
	       );

# plstrl is not exported since it is not part of the API but needed
# by Starlink::AST. Do not make this part of the public API.

Exporter::export_tags('all');

bootstrap Graphics::PLplot $VERSION;


=head1 HELPER FUNCTIONS

The following perl helper functions are also available:

=over 4

=item B<pl_Alloc2dGrid>

Create a perl 2D array (technically an array of 1-D arrays)
suitable for use in C<plimage>. Returns a reference
to the initialised 2D array. All elements are set to 0.0.

  $ref = pl_Alloc2dGrid( $nx, $ny );

=cut

sub pl_Alloc2dGrid ($$) {
  # Create a 2D perl "array" and initialize with zeroes
  my ($nx, $ny) = @_;
  my @z = ();
  for my $i ( 0 .. ($nx -1 ) ) {
    $z[$i] = [];
    for my $j ( 0.. ($ny - 1 ) ) {
      $z[$i]->[$j] = 0.0;
    }
  }

  return \@z;
}

=item B<pl_MinMax2dGrid>

Given an array of arrays (e.g. like one generated by pl_Alloc2dGrid),
return the min and max data value.

  ($min, $max) = pl_MinMax2dGrid( \@z );

=cut

sub pl_MinMax2dGrid {
  my $z = shift;

  # Make first guess
  my $min = $z->[0][0];
  my $max = $z->[0][0];

  for my $i (0..$#$z) {
    my $row = $z->[$i];
    for my $j (0..$#$row) {
      my $val = $row->[$j];
      next unless defined $val;
      $min = $val if $min > $val;
      $max = $val if $max < $val;
    }
  }
  return ($min, $max);
}

=back

=head1 EXAMPLES

This module is distributed with Perl versions of many of the C example
files that are distributed as part of PLplot itself. They can be
used to learn the Perl interface.

=head1 TODO

Most PLplot functions are implemented (albeit inefficiently for the
functions requiring 2D data) but documentation needs to be improved.

=head1 SEE ALSO

The L<PDL::Graphics::PLplot> module (distributed with L<PDL|PDL>
is more suitable for plotting large data arrays. This module exists
primarily for cases where a dependency on PDL is not desirable.

The PLplot library is very similar to the PGPLOT library (see
the L<PGPLOT|PGPLOT> module).

The L<Starlink::AST|Starlink::AST> module provides a graphics
interface to this PLplot module.

=head1 AUTHOR

Tim Jenness E<lt>t.jenness@jach.hawaii.eduE<gt>

Copyright (C) 2004 Tim Jenness. All Rights Reserved.

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful,but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc., 59 Temple
Place,Suite 330, Boston, MA  02111-1307, USA

=cut

1;
