#! /usr/bin/env false

use v6.d;

use JSON::Fast;
use File::Temp;

unit module App::Rakuman;

#| Render a local file's Pod6 document.
multi sub rakuman (
	Str:D $path,
	Bool:D :$skip-pager = False,
	Bool:D :$try-resolving where !*,
	--> Int
) is export {
	# Extract only the relevant parts
	my ($filename, $filehandle) = tempfile;

	for $path.IO.lines {
		state $pod = False;

		if ($_.trim.fc eq '=begin pod') {
			$pod++;
			$filehandle.say('=begin pod');
			next;
		}

		if (!$pod) {
			next;
		}

		if ($_.trim.fc eq '=end pod') {
			$filehandle.say('=end pod');
			$filehandle.close();
			last;
		}

		$filehandle.say($_);
	}

	my $command = "'$*EXECUTABLE' -Ilib --doc=Rakuman '$filename'";

	if (!$skip-pager) {
		my Str $pager = %*ENV<PAGER> // ($*DISTRO.is-win ?? 'more' !! 'less -r');

		$command ~= " | $pager";
	}

	shell $command;

	0;
}

#| Render an Pod6 document of an installed module.
multi sub rakuman (
	Str:D $module,
	Bool:D :$skip-pager = False,
	Bool:D :$try-resolving where *,
	--> Int
) is export {
	my $path = rakuman-resolve($module);

	# Ensure a module was resolved.
	if (!$path) {
		note "No file found for $module";
		return 3;
	}

	# Run rakuman with the resolved file.
	samewith($path.absolute, :$skip-pager, :!try-resolving);
}

sub rakuman-resolve (
	#| The name of the module that needs to be resolved to its file on disk.
	Str:D $module,
	--> IO::Path
) is export {
	my $file = rakuman-list-installed-compunits(){$module};

	return IO::Path unless $file;

	$*REPO
		.repo-chain
		.grep(*.?prefix.?e)
		.map({
			.?prefix // .path-spec.?path
		})
		.map(*.child("sources/$file"))
		.grep(*.f)
		.first
}

sub rakuman-list-installed-compunits (
	--> Associative
) is export {
	# Most of this massive chain is taken with a lot of inspiration from the zef
	# module, specifically the code that handles the locate functionality.
	$*REPO
		.repo-chain
		.grep(*.?prefix.?e)
		.map({
			.?prefix // .path-spec.?path
		})
		.map(*.child('dist'))
		.grep(*.e)
		.map( *.IO.dir.grep(*.IO.f).Slip)
		.map({
			my %provides = $_.slurp.&from-json<provides>;

			%provides
				.keys
				.map({
					$_ => %provides{$_}.first.value<file>
				})
				.Slip
		})
		.Hash
}

=begin pod

=NAME    App::Rakuman
=AUTHOR  Patrick Spek <p.spek@tyil.work>
=VERSION 0.1.0

=head1 Synopsis

=item1 rakuman [--skip-pager] <file>
=item1 rakuman [--skip-pager] <module>

=head1 Description

Read the main Pod6 documentation of a Raku file or module.

=head1 Examples

Invoke C<rakuman> on a file or installed module.

    rakuman App::Rakuman
    rakuman program.raku

You can make tell it not to open the output in a pager by adding
C<--skip-pager> to the command.

    rakuman --skip-pager App::Rakuman

=head1 Configuration

Rakuman allows configuration through environment variables. These all start
with C<RAKUMAN>.

=head2 Code block highlighting

Code block highlighting is enabled by default if you have C<pygmentize>
installed and executable from your C<$PATH>. If you prefer to not have
highlighting, or are experience slow performance in files with many highlighted
code blocks, you can disable this through C<$RAKUMAN_PYGMENTIZE>.

    RAKUMAN_PYGMENTIZE=0 rakuman …

=head2 Tab sizes

Tab sizes default to B<8> characters wide. This can be altered through the
C<$RAKUMAN_TAB_SIZE> variable.

    RAKUMAN_TAB_SIZE=4 rakuman …

=head2 Line length

The lines get wrapped at B<79> characters by default. The
C<$RAKUMAN_LINE_LENGTH> variable allows you to alter this.

    RAKUMAN_LINE_LENGTH=120 rakuman …

=end pod

# vim: ft=perl6 noet
