use BigRoot;
use Test;

BigRoot.precision = 20_076;

my Duration $sqrt2-duration;

plan 2;

subtest 'It calculates roots with high precision' => {
    plan 2;

    my Instant $start = now;
    ok BigRoot.newton's-sqrt(2) eq sqrt2, 'First 20,076 sqrt(2) digits are correct';
    $sqrt2-duration := now - $start;

    ok BigRoot.newton's-sqrt(3) eq sqrt3, 'First 20,076 sqrt(3) digits are correct';
}

subtest 'Cache works as expected' => {
    plan 2;

    constant $MAX_TRIES := 100;
    my Instant $start = now;
    for ^$MAX_TRIES {
        BigRoot.newton's-sqrt: 2;
        BigRoot.newton's-sqrt: 3;
    }
    ok (now - $start) < $sqrt2-duration, 'Cached results are not calculated again';

    BigRoot.use-cache = False;
    $start = now;
    BigRoot.newton's-sqrt: 2;
    BigRoot.newton's-sqrt: 3;
    ok (now - $start) > $sqrt2-duration, 'Do not use cache entries when cache is disabled';
}

done-testing;

sub sqrt2 {
    # From: https://apod.nasa.gov/htmltest/gifcity/sqrt2.1mil
    # Note that last digit (20_076) is rounded up
    return q:to/SQRT2/.subst( / \s+ | '0'+ \s* $ /, ｢｣, :g )
          1.4142135623730950488016887242096980785696718753769480731766797379907324784621
        07038850387534327641572735013846230912297024924836055850737212644121497099935831
        41322266592750559275579995050115278206057147010955997160597027453459686201472851
        74186408891986095523292304843087143214508397626036279952514079896872533965463318
        08829640620615258352395054745750287759961729835575220337531857011354374603408498
        84716038689997069900481503054402779031645424782306849293691862158057846311159666
        87130130156185689872372352885092648612494977154218334204285686060146824720771435
        85487415565706967765372022648544701585880162075847492265722600208558446652145839
        88939443709265918003113882464681570826301005948587040031864803421948972782906410
        45072636881313739855256117322040245091227700226941127573627280495738108967504018
        36986836845072579936472906076299694138047565482372899718032680247442062926912485
        90521810044598421505911202494413417285314781058036033710773091828693147101711116
        83916581726889419758716582152128229518488472089694633862891562882765952635140542
        26765323969461751129160240871551013515045538128756005263146801712740265396947024
        03005174953188629256313851881634780015693691768818523786840522878376293892143006
        55869568685964595155501644724509836896036887323114389415576651040883914292338113
        20605243362948531704991577175622854974143899918802176243096520656421182731672625
        75395947172559346372386322614827426222086711558395999265211762526989175409881593
        48640083457085181472231814204070426509056532333398436457865796796519267292399875
        36661721598257886026336361782749599421940377775368142621773879919455139723127406
        68983299898953867288228563786977496625199665835257761989393228453447356947949629
        52168891485492538904755828834526096524096542889394538646625744927556381964410316
        97983306185201937938494005715633372054806854057586799967012137223947582142630658
        51322174088323829472876173936474678374319600015921888073478576172522118674904249
        77366929207311096369721608933708661156734585334833295254675851644710757848602463
        60083444911481858765555428645512331421992631133251797060843655970435285641008791
        85007603610091594656706768836055717400767569050961367194013249356052401859991050
        62108163597726431380605467010293569971042425105781749531057255934984451126922780
        34491350663756874776028316282960553242242695753452902883876844642917328277088831
        80870253398523381227499908123718925407264753678503048215918018861671089728692292
        01197599880703818543332536460211082299279293072871780799888099176741774108983060
        80032631181642798823117154363869661702999934161614878686018045505553986913115186
        01038637532500455818604480407502411951843056745336836136745973744239885532851793
        08960373898915173195874134428817842125021916951875593444387396189314549999906107
        58704909026088351763622474975785885836803745793115733980209998662218694992259591
        32764236194105921003280261498745665996888740679561673918595728886424734635858868
        64496822386006983352642799056283165613913942557649062065186021647263033362975075
        69787060660685649816009271870929215313236828135698893709741650447459096053747279
        65244770940992412387106144705439867436473384774548191008728862221495895295911878
        92149179833981083788278153065562315810360648675873036014502273208829351341387227
        68417667843690529428698490838455744579409598626074249954916802853077398938296036
        21335398753205091998936075139064444957684569934712763645071632791547015977335486
        38939423257277540038260274785674172580951416307159597849818009443560379390985590
        16827215403458158152100493666295344882710729239660232163823826661262683050257278
        11694510353793715688233659322978231929860646797898640920856095581426143636310046
        15594332550474493975933999125419532300932175304476533964706627611661753518754646
        20967634558738616488019884849747926404506544489691004079421181692579685756378488
        14989864168549949163576144840470210339892153423770372333531156459443897036531667
        21949049351882905806307401346862641672470110653463493916407146285567980177933814
        42404526913706660977763878486623800339232437047411533187253190601916599645538115
        78884138084332321053376746181217801429609283241136275254088737290512940733947943
        30619439569367020794295158782283493219316664111301549594698378977674344435393377
        09957134988407890850815892366070088658105470949790465722988880892461282816013133
        70102908029099974564784958154561464871551639050241985790613109345878330620026220
        73724716766854554999049940857108099257599288932366154382719550057816251330381531
        46577907926868500806984428479152424275441026805756321565322061885751225113063937
        02536292716196825125919202521605870118959673224423926742373449076464672737534796
        45988191498079317180024238554538860383683108007791824664627541174442500187277795
        18164383451463461299020763343017968554385631667723518389336667042222110939144930
        28796381283988931173130843004212555018549850652945563776603146125590910461138476
        82823595924772286290426427361632645854433928772638603431498048963973633297548859
        25681149296836126725898573833216436663487023477302610106130507298611534129948808
        77447311122954265275165366591173014236062652586907719821703709810464436047722673
        92829874152593069562063847108274082184906737233058743029709242899481739244078693
        75284401044399048520878851914193541512900681735170306938697059004742515765524807
        84473621441050162008454441222559562029847259403528019067980680983003964539856859
        30458625260637797453559927747299064888745451242496076378010863900191058092874764
        72075110923860595019543228160208879621516233852161287522851802529287618325703717
        28574067639449098254644221846543088066105802015847284067126302545937989065081685
        71371656685941300533197036596403376674146104956376510308366134893109478026812935
        57331890551970520184515039969098663152512411611192594055280856498931958983456233
        19836834948808061715624391128663127978483719789533690152776005498055166350197855
        57110140555297633841275044686046476631832661165182067501204766991098721910444744
        03268943641595942792199442355371870429955924031409171284815854386600538571358363
        98163094524075570093251682434416824083619792733728252154622469615332170268299509
        79089034594858878349439616204358422497397187113958927305092197054917176961600445
        58089942787888036916943289459514722672292612485069617316380941082186004528610269
        65475763043102560271523139694821355198214097165490973199928349256740974903922971
        26348693414574933198041718076111963902278664075922434167762466236238913110270343
        30457636814112832132630858223945621959808661293999620123415617631817431242008901
        49838485604808798646083935964923665142968125773143229145687168276219961182782695
        31574983802624651759054103976181287604216386134502213262727756612441133610775195
        55774950865636067378665062318564069912280187574178549466125327599769796059776059
        07564891066610158384172028185304321190446577525542775437987260548817361982675816
        86283295260789932226683602838513512281059318591028641508157056319717315183136250
        24359041463212239217663398268936825315053005989154702909537193266207341123494743
        36788469020139049784285216341442921458955828784766939464642678122190497856363552
        63368278051860098699248937786002398769169807656621943898544370805946433362333810
        58745816235475600136592435242657143083465545768002370814675732525470255074763747
        16350678515991736937932510326827606286459146182047214863703707719269268236233347
        20379245964691810526139153086280291440965482563873092730426544662929045896063751
        91871146934536197332478957270703153093090192119919999361576500350398405406742538
        79275279227247335667706078379113844889362613676570602636003151329520953952028548
        97384486256134924414708607086602676349978793420875836121947116994223848482595914
        30452810706260150896913530301772006271705440209066951491527459771970594769547409
        52102878725578568800221937177435581107939308833845586482772910086295545661413067
        21230848740227121058686323388237413884428938155444647105755651468435702946635062
        89387356986868837648032651952841465351739530273612013742030098673983851432190043
        60289826982935293994141292305803845650227072168151619410114498263013649008770483
        98488386090653368599054583895203185648041493272142390865164999431659207965953569
        43072311291162928679751715668890543932203569129332457020806719444049730494398140
        82278296027994245410831666759214248351827238172050410392742888015562233807961475
        12433514731021284545944899444996000752437519570116683417447490795882099517836768
        02323651767497230148745774272599476096219843271483529861119027287358490521797590
        83741974860267060537462315300393752123678677528486921958571375542696848278363178
        61109933680143915905974842858054516130230143979057016108898627779610750673332676
        04865492925139978139053588227689373220494148394013556035656044214017612060513180
        68919899626061848318534018362378217266375804552471962661749254228528045714420485
        78342113228008528704205488992341278554812367615377071042544698685219911228354266
        34999712748366076246241820736466617128394748473280474430403344107200428727127567
        02795675824292627194545805300266648996507956977817862194217200523716536946770419
        51119127046248360511302890464377511486948878496151188414719100012558838366606772
        08411235153558811267789571558590412576261601067513153580212427331871000635824954
        50409957940725479890031682651237311905566829151943053708489307869197428290490386
        03723116099283424317122250994547150192866648787107951995180054633883844315481724
        63548024451803084527343100062137103462573306001234973744355818096567846464153390
        51465691932456235314057791936989884236471835253758052577133112007971040683154926
        65402026046806818391437827214769063242469517128636738443139833371176159418699934
        66262345373452356794012416809229116360956372167452839170990914664850739205151605
        60473787106154702169960746569309794426121469256159342564940191229895147325447151
        81263258368897282262833295240359700727863364604594707124174729468775705958157349
        96284809956783925547424044899188707106967524250774520122936081057414265323472406
        41621410333533405511045212617503590284037454591864504727624342071770929793540102
        14096464502836834180407586081001407216192477179809859681115404464437285689592868
        31977797786934641598469745133917741537904877880830022058335046746555323028587325
        83515708599649068672875967295038725475708791695547366917087012413339221484668517
        43706661548819529332272737436041082542596603039869326542235052369108595126300831
        84675550345975839550584035670155887977736443804818213870700344023618041200211483
        72794227407873789331627081013626498289629272562445805397134142214511099995445821
        42923783881026483948233951418767468967831862868178827255582573193951815531695164
        50149435726310604569492967098625204339385207822076221910034469269663342590853058
        16044978025776325448937080062677873179548529856683948694673356963001402931314190
        25780775816945815272529343422590519791831662164448751781696775276770913043157342
        56405492293818739511084416683092491115978577332736388414185073793630026392180680
        01949823966647123131719025237031990587719774100071324075192041812214132425327294
        91860004200841548511547411573059872196212988541663720877522483769485974767293301
        86839052250014869038261084824819816759310777270264882620907238477529058765040326
        67275848252185162310745449887588274656780949712308766144264148241579035703933122
        56518933356281836185405746706380618398489466284245736564564213907216305295529359
        28487755524275455951338277150017840165530548544228501198836557568015934645055899
        44248496274127118698831580476918141567961853216571696452225945947124693199571164
        19861884797789121142681164383772384836318673186075647785369993038705466322969807
        56758468212302807726100696917407820247949882109547334301126545442170195852375880
        78534800373724711876111000877190355388157319225133384249477450311881194745595365
        33660920641929344003507856422343292324929727084724823557671740589500126876360081
        24521124487564342809465931336185643241485578079193115126509729589160529930307710
        56352454514834572092245519848905889042198065439733537575998248580375463927365376
        41967480626968382712920014349566748522472414548636036211584723231736998061719936
        42113631458071198839681295705611588124620588579665056221507482089747764177083787
        05292420288029004400248068681254220757905942434704644895754402387369360474013086
        03607599174387615635296776058018334930879646627071160805073761071800221552519199
        37962007091613832272801773133201900597804820796075803249946223853858035734780187
        13802840398120046812370790924572728576545104897170310237054867879336437815780740
        07677474215280311849815576981656151626115720204540264412993161170773312538461289
        36763791838537050094206306091032540258476822203676824927947340006177512952630726
        56378530973686420007766665889932845661224650730022095628772726222780803954834038
        10962805764928974651843631949840261299761890046781909273709647827872435775220668
        46540002468330746087835876558905305694257499098903922046300471457205905371209131
        42758865376931480400008717913845690993629987847885421778154073505170625320509514
        47822066725260862041079962227034808180138006610071922681402919768354884243991628
        09803618597719358892265485872816327690542861746632308136287764990073775993244175
        21476776046936962233215175926450556452563840546700404521580075454379681038435585
        14794309229635219785228329574545727156479318504188960701280594922959218359493707
        45803903214104366016376509554894454190263391196074110066949778024695409365628127
        53849632360106258465366705076517702969513039685870236791287541358806440263423823
        56806076407451761190883337120914157628056522379012735641935345652676296244026602
        28245426196034228352400205032950531908532014968045135643341034313292235896972831
        08739569438131809431666913390526489148332879882762852563045120637614900045218642
        71711150897628275286714663611738982858742531721659624764332384003490049629878948
        70010518844941186604397391074937573495289347707396386659332554385899935379941438
        40662422102268328511662511368344732896613210526750893794834463493035278532130127
        82115268594298437565174510930399249586646094238684700215355018037800018701111315
        19378754010914958890807647334550026409805683214381160075146182788449046812481468
        93097430000109019843208666309225138112111599481279636783908122243781910187779940
        34076527406038234150532717416278674888085754101214286674663103610880018188435401
        82368653221687750411978076525811538417365621835675013034456595936590974690077656
        30951563662834863997975493756384052967232835634030315916549588611222995999686827
        01428407239146230016173544083143643804589220554110179535135588527134798493787613
        37910756559954145289177701575813487576801862429222297766621154224971133417396031
        96763909350512320947616642753474388333888699979916463836750324186324862841878469
        96096380827512996338173937422095347586163221630520270351703749029856852559581419
        29549951765525821234731081976633013417508151236775231516073208818295640726347645
        05887576136189361870128904026792264704949678723740258130083476397564463263354967
        52857495370151271006944644206246175364428949860492052182321338432627533519882949
        20864907096059216545737690959513893789976626287708683235059640980501885698199740
        56600544153213840734978154080943540759681561338964320840415315102432432476506355
        82409785346811510563238980401380381484973528744429069393443733818010901017885920
        56406907697263909339111613684666699318068382346740389223672292551660244685974607
        63582903728529497158369063729095033685951182387240386566803844098591359996588300
        62279975291368494561705199329159949232434041372725380763684029518736981797353665
        70959942204751059644075907078002039362324718238041377992837396735880963895473938
        47128950623044844732470443823390251314913859447521279271461067225268355552093101
        40982503241041356811888934417063480181887903852437284345041399526750839074929315
        59489927997402060166861010605738362034369613992370502059136912247881448219700455
        64629606180152957267746654025224032152010626805924692941841465169269429703164474
        89225533568194701055860753950312574877927182201980680506551347189262650998704038
        72393615262809117150163983918382080371076644472311255942979308415748575497128495
        67707689130531391519283160749372260464837412112410527404580769077499032167615319
        96566097430089028478792209894555140349566776329368418912928228889379139256579030
        61704219517464266717628600737654825489490823670490417898279469481337100543757352
        29262593956809537167977177738428166119599319878150374402232529401665135964883989
        18771266667644592828072477419805118340357726301941505622270926688815100874081021
        63045511193689703398758991634367665524596900229663906182345992244371615681887167
        85019552192690477008887628817012435907238421885909432302524008728363411346002474
        63505407631743028561032883144639525995577714162497515992886034410104753346774530
        43727868857611962268589481389788435122516690676187913234462077242638989111751935
        75536755089771736080779854992933748575879407969489011853826051113623591734039139
        86090018722454028726512923507251346336039947797211253440796964196584329248583896
        15370786254462405273418372961658712809962156751416778888521820178268579475088605
        61917614334505307242257944215044011899380328211769427535155005035938402619271248
        40735344805764151349209066433260871886931878391137249135429063143277314147565234
        42769892641072919296477831522676530963377190788702121017362889280133206655384688
        75227098214169947453467839730618843380636885678875093483712812994594714167402106
        47944623047509596911213284185737450768802174200091903786114928999321369828255050
        43941252342938789152929448806729045337155868589391194058679926796801975192946353
        13212046058273013652463549197477178431255147195610894481716873695950097551490905
        80423770550765831660455263178819159288580151410990335159992764926020916753796585
        65407172149027277207207953304640949267929698014564740758616841751827035541915232
        85901319918975644427209195806647378539654749435033660984556942205412322091494769
        85226606686931349412846052436006261919200954559599299203576635844725208888438770
        10984850961455366250564822233108277487712496459239403441038488045655720915372083
        69237042203903081669215344336555529659147737595207945959705914921302438333795709
        37471630364094522401198254550375439726080376366587365259895269116799601027835888
        11157158411574479474035286890009482413391845137805999225189847359411654219009436
        69850291800726152708954832476910790547502395766594197888184410520182887164116705
        28264469474464709886880658944170090145701739592379888063120134295083414410967004
        60069706630113988380654410359590363854288905339597947613555539309235350010227464
        02573996549260318712100543951693151027362514695808436694837491133853153237804262
        47949617769295622506132578266616587528170615484688178446049185158500223578507688
        44467787401475044226257510173648618326337325335419213096310532213256241946140930
        27893358253117355483121861881858082744966425868045457888041604096100038498733107
        55633890347244188014704960144521546446352768745300463429370465787461230411428584
        13931547069644559886270234135533696146889413057952434482857881384291313340044438
        82694562774572044648927503718896002502148102432266232474466759222209575796803987
        58330118809412348594105539793115016614982410947734045594477362140729819067342271
        22132982820189355181480711612936047539354519164488519197681989432468826658469990
        81034874330571542521148982924940641352048699331708524365460290800650242152985981
        19251209628332680767252801429242638106620922006716242390535394287757500791358700
        37680650406932904623670199806464238166278074558454279728136364961281560063361281
        91171807774985721284909654045638060252797929001411671808882167276232695731445299
        53790935999635720195691053823183323228796036615493435619556357778872038051407147
        02862748666577272849638060682090874337475577893122071661832909979680774069583054
        55058169463910538025471298095019164059177292147061205379737583188169040345737308
        92287048764298832763431061565604142353102111260771779308733171317133834055399077
        14874603987863410018154894306021276414517536855678124104876278918445986488750725
        69190707056126189065115415374343317047710735120897496645858712323260612277676488
        01741904170532621901129299413424068594779706484545123043270036204444603070019114
        50415420812728658532659862596583412919318743374681425409919891908009325715225950
        25697576362583785420687607875336423890245352422314781169503074229968826623618943
        79754634314869404737054660220523658535224334918934149537318546177563908506301498
        05930744131272144306096868121549167999336020911246342068160464235634526699224711
        SQRT2
}

sub sqrt3 {
    # From: https://apod.nasa.gov/htmltest/gifcity/sqrt3.1mil
    # Note that ultimate and penultimate digits (20_075 && 20_076) are rounded up
    return q:to/SQRT3/.subst( / \s+ | '0'+ \s* $ /, ｢｣, :g )
          1.7320508075688772935274463415058723669428052538103806280558069794519330169088
        00037081146186757248575675626141415406703029969945094998952478811655512094373648
        52809323190230558206797482010108467492326501531234326690332288665067225466892183
        79712270471316603678615880190499865373798593894676503475065760507566183481296061
        00947602187190325083145829523959832997789824508288714463832917347224163984587855
        39766795806381835366611084317378089437831610208830552490167002352071114428869599
        09563657970871684980728994932964842830207864086039887386975375823173178313959929
        83007838702877053913369563312103707264019249106768231199288375641141422016742752
        10237299427083105989845947598766428889779614783795839022885485290357603385280806
        43819723446610596897228728652641538226646984200211954841552784411812865345070351
        91650016689294415480846071277143999762926834629577438361895110127148638746976545
        98245178855097537901388066496191196222295711055524292372319219773826256163146884
        20328537166829386496119170497388363954959381457576718533736331259108996554246248
        34787197605235997769192323570220305302840385915414971072429559206706202509520175
        96318587276635997528366343108015066585371064732853862592226058222051040368027029
        75047987280794616581004170526819400190957334621759438936702493204226910343698124
        63720111185261084268910299720311202100063507176374582405203847555197279933797614
        90610789498554422332600401885130363156114488684728158928816324518726506664538487
        75991625766428721112408420680167635171001029431807155151909616424609070394081292
        16903517492961364004139670431041253632327030922577327960292376597745537095469115
        74214042423078199232761740190642451245487751686269610533369421621360539460424565
        41401285330078136334498567364067039773422298119610429255345016014059404795471545
        34548407271737656262366549166640233006013265744070107836858468452313160467754480
        50040224063991197036221860292023886715071101716940029686875966350004089531621423
        34252279568340670134701859020283607167621477434934495635958080821304425864694685
        22610908263353008756612603460672195404055984128912975994810000772057440230047673
        25880009151437148947544487915719129465908357087396151553779764026206837084804607
        29693827195856897597596261041591526575777907823349805678400229015320521389353737
        75536566427046826874289963441395743666073744445583086477893212985302148197395341
        47817051661495255176329199369956574452263911251909354138698936681743093822642473
        69262020729909678311541319464843779154599159239282877146951492740264092136456540
        41644581490201945749419305269002613972646081065071439603206077510594187798284793
        98619524996416521313971529359942189741664707518723578862946610856017042886960579
        83940529064074308118333886778815626358671560083967602453492299439388670597543154
        42943430957258470988215463111260766774067864571578060647447499750354544559313286
        54918984933657274762629741473823568691483783136336128362790382484016380667160717
        98487285558429313492260932405659575536511367546443878342833134666445541803908218
        98983294626345016171122016929619460169320621033039744866231656003566781813443100
        89656870674296565817744557268501563159441250736119806347288646624132719366942671
        29514503975189283134861928413032551884311690146714237943747283343446524820437960
        59732870616274910941596862625302558889768470417502003140262284691048046984032437
        90660673774962710251271060206086266284366605019289089383832000609122869188183684
        23616939035231269303686137968115269469978919266352843899270165198255022529235793
        66281634568108913074112678378199449772765448294920438801301783546499213574032791
        57750778600229642350427599699762691873841484739477385156355209122096331583903456
        71894152943570406530760852901742497681855028852763449495774491104225024537231560
        63618103504871759158032791798367510067136100066217566049573766777112925287302479
        92672015899641893776201611411828989968295590251165639142236499506285003069681297
        35072517279185983011152775422988518853117635947860031230637260983698503083108521
        45595748359555715739356633680864821078791247051031631569581604348748829614894154
        58535556197499929687795509928552742359025181032376686160959255254431995490742067
        97391570726142453289504280973510379677366601359175226270612323125894506976348131
        77754456066603576220483006904225389494867729521325719769245927670660765816558298
        67505657289465886091902964438128505785975943750640244606212145608560854524925136
        83287548434387337202016923116304327056480601375879368567215350878168440780232920
        64623163811291772297752725531075161383682134604577799854625622081960631636019206
        33096851621623166068346886311662150132385428739130449512315521636562129518747553
        09765706399550298804256473405577851913495330398886430714317850520079862424684813
        82976340802176297056457213162198306829364628682222279923816416353172598273213079
        71419267628691568118183996651510030701082670312866401591831271960403892354068259
        15045957649183685526148034651092898965599618297164084981357029227296221402637505
        95519947501618105854474019764971593766342388569701982214045410209642908388248860
        98721887649480083594449193447893133108077774083803560189222853711009389809120485
        32329904555985240968891805044970438384150455861501309029070913061564696859589692
        28276473764261083791700284689036668670497679170399954436330425855797794077269935
        28895083782711435712578063128272934988338504592711060952606287905971505494590727
        98416817516870415050729577233652010032638314784794107946076565063693326630382060
        21719551471952041208479627248539111589302866641079696991524811276979149995102660
        57453112388139393491970892694073573849194429196300728095033865430213781233945614
        24724245868250054905097821684198951540776630466486602532118215170585464816990103
        51020406040779053482395457081103662802419307833539518597508847118544793150345001
        58038470944785460209216193440279845891247205294177676250649796185823477655628082
        49405444376505977794390801979243001549599367104079839130331199968665350178723089
        95874345981050410546853618716669008602651872709506307892209449850428619343629495
        42931097209940859558589679600042582295942694390957411016678479809079289075408433
        87989975855690094719632489043233660363728874204771798977873196747893346400992387
        95928413280479076582258181183540591455395810029729132679714194354057353161415847
        98029441186203132251905409158901912225531965373739285631039078775792050343944361
        05607282048657439146472804807233659273201766562230057011146855912144509955053649
        85643365509525473974996502828824922856959093188033794992264659862068823401174394
        49629313198598588025419934709620103894917197706225865726670761872835390233480692
        50834356495374456283107575871868775381593970854006988992201660544250263128328807
        89794907045544329548492870187596100762027461311058672024649539970560477439601070
        67742057472358757075902472651853228891022363037446200471964240075854163676572689
        20371635392194774286906858888158097042483582151491356992810750723078264474067187
        34809484544537264931062922580337258517145144805462076646086736427250483015812043
        42175186931670856000833529593442674587299873507162517030057115713817530130240701
        10296602509558624225602890646741319080006821289845940660914873475988613959100493
        96065914517216530677681935572187655860065307738893120925049062804376001777854452
        26501520621850502043831495061372102708013920377075629783065360777573663259286473
        65858953011561800178170369022806607100764410062540003168840377529981249455547190
        04248954149482241055947835787572003255966447123768440333531554321404257417457980
        06029830829177130827142554246923266644414836279852571797525822468122047195648467
        17398920410879193844507863357697868887939147231768168152167068102145555981333521
        26063672655392136952255122818916717426718903555974152910467258069243862506098603
        87135886722931428923916203433273931039257964358665558134253017251042384960681029
        02193241247883304519746316017074652260342989164540709484562290741791740090112338
        62395384545294973224293397306200271688265880011208985226125337807028455581448577
        75181193533455812353801843543229621528483696299891926156571641697746616824452031
        14487162780669497265548371919914676148107974684952144144971912661242582093340384
        08435153619698962218737471928474471924073026312829794461570159696747688148437922
        57557350327894448960301265759219980029877201229053965423363179815036216968859918
        38704958970325070591396571344311334355007767294918902707686149548339946677241627
        65514731195167511681718998875787982178391089760065710008376769429534350552888108
        65891170675872119515936148911282308230684332119607205021863531839922109971482661
        51083690657111006301415107040757110854813735703344847363861289738071706512644964
        78839340250121137258483942385699581395465756563747606366397154802563573736399649
        84725543285885870949544699063168525511859987797107454415882875639038668562404562
        73771403576082251948946485327585096461353516108643806062047390528236040530147069
        12323032636557053851755375864010248567111408373108906937774684792995636886177071
        18866407738814622948371281801917103215592415006079870572291435439125195736876822
        38088982448959746974767810227262035149129049368822889115077490463869723017882528
        88706432486798314996711993510261548209307384397532266800587384309407578948075198
        16381685610451417837442864004156135774111269129932938895969229009604053100323183
        38218577787993482184022913347370689218898207143551298643155087785079937015352295
        70998843782958817370884720700044671618700284314375344174419498974828148386643492
        74181798045102179529895291152953586620103822337718791749869553228316430615664593
        60934573676220209533838311337455435570693333523605410975808062515075690820184887
        74733352689124246378746329829720703684819427992012273665336822499743175584189601
        56204057648029657377064116392836006561242745279302427187819071828565144846989589
        89307844166763219885733320866403491603258218771672650046563027922746573906794446
        54885688846995270196061119942760195715001069960613187691725593898690715131894766
        44102291326233979622874705194527713735168356202507452756649823587191001253392401
        10313240293530727058453878955596620356560863445813722609269451698856155856727245
        91846723237672388902404630147827209056907327435750851263388215320381801796701230
        40197297235768013011490015457614217179825051971048381345959246757550605679516494
        16187383995284330819433744891905629838955891842366997243941630336015977981770968
        09974708327966458536136421237326941806493170021835351063688698576140997580819323
        30847261595601525267956348299334481308629361503106879444508385679193835245291250
        04597160435444282261550962322402373162900504373772781347828505597613576092283525
        70952081898316651640195570340999938598521575963837145303974921800116556289159210
        81798747421772358180102472619511867612536900635158015260450258969797082593563488
        90598694105955063792026162707067718087390644306123423600395630088194249193272120
        05058541456290963769904961706255909705061317054161128163777764306237708256586091
        57957860714713129960383072121968988558913944800051551070657103597615184980822670
        43802144533530877161517260965721980312582046661604058657756565427575470268165624
        58055301547087993695034082328447526589852789049425455215707595591494511315673452
        31019514972644070523372502238107672725740745323712871045393071669109930838566953
        45292768997264646352120149994950915423333145340014022110124806824899314675171865
        40862268722311415802443408161054616103417854992338644639905205027551291197833729
        40001948273293151494275317027983047411116749845308147697994355281332944153394484
        71915440910253099019906508522402430753192448792335637216884052823997728810616189
        62990128635978281538394611177847509031613815920769703157712922588367174861032589
        62773273497744748653456547763753161205920465010852421670208343306227879375568798
        35661707948539805672405348813332863107983491253431573486176091981021427979229101
        87450802582026363623256193096243553231587285348829873765869344524453602183855862
        83876933716334021064222181216481081580441485509849214522327811962167176954788915
        98057099820333683236400840543660560960213397731321134370041157779279416953733290
        55568984754686013179259713833914745762809485408199711482358077319436460313898339
        02931572295796059957379968808921987506251393367494282432897155702772725687330155
        93562060936647420322995103302578000433139634271097336646661490863286786373981691
        63930833679425191320719427043977823333656549271542838405415482842951244186019213
        69248641679823968859510560570821359074137535970094832491067822895707371493685828
        61767705691151789837945472481786809925833497564131050431929905071074735720210361
        90442176913825407722587709825866537712336311454672809622686285751790365617135626
        28483687299337248388696417430030951465688580731060786063381361141065783808579168
        24332787866051017755602745219297787276714818643586326926397144107440076379893600
        92786507634449861332261318570349393965292581031258997975668211352874690069920869
        24610447030799487393878361095899242875483059159169081893559272285413466102882858
        38279190331761854589620529793379972937540785366652909730692348052308968960238907
        69450388425300795015348493690850523854493714546234377923124787198041802950912040
        66149153809431982313012870281388464689830668083636573533907984621737376365215305
        55810107953833590149109533688942698656282762958961307540977007112160560798993295
        07813586738566490035746080812067091993588223203775931420023557565111140385238879
        40393963217897264417310285646864433256204938803170315155930960154816215460489186
        15035403233749543076452816087928819594429051611973748958001216251344462661015527
        67507385143884877260387501305912440737786875552087993017069940230049438751945135
        67660849886562081427259302839694402542771559031182830559294667461594644206614773
        91285073828270312641298117000677610212224092504509581437044356956442795847267426
        92575136416909652474525651542917048134651493242804768514155617344246171002265205
        37557993050944909047378741545955857119476509358400493338980173224956674213495942
        27762088042443689291676237878470678272504359258059600870021318903848323784744251
        99268419192170016794673539032615667971820415559522300956095276545700706153115657
        32489383484045390755194298376166223163774785594777332404376433330225887967989009
        72897817625827182709465857106368465281386322352299967783910650919388133511815108
        40625792700141726219404643872065341494864810645224661417773908453016085890445099
        20896192973111693291140914977698518504440667745165302927814789580750088381321283
        56821349425177511216469764855395440142471268399165112143727355201194782764445993
        74071653522646130415885577461858334332170042288550818280319636798634501983044574
        56772369829780644653869213064192201879603979164515437303783396311569677892631092
        77235980770834371606727813820103909176876013687920071757462097439748508491330592
        61232489728816159412286175289504563958819931939701822986003770641163788053317134
        90773215344838726470534926089769457435645840719224252638964284617896938348119225
        11483429307533115473746469182857775141252643845544976256842191379987356275063359
        16455535977596570960941017564348959222418747988579147047519831786507988341085244
        13168473158734076640164187829488524960216928024130069224948727268853622379624028
        14505942688905368716755391946427703539639315975421206980288723469535162309646225
        02386938238646084614888861227348957457780774981921650697917045601166533662667814
        29712731815066329599906107462017591354741585452653424803517707161650071952590358
        89050547510317857046811052327150417210214328888810284703730549581195361696109888
        39734796022335815458204651123459926191604946146743366119579038273065425569609765
        66379750800059381634265783139701676937663170697583971389759038480430931145433552
        49475659788306946532567411865225008837152756171648861479781848603575891436440433
        24866921011959249436977074495463488997118051028309022926853664402250842445615256
        64123002951765048776974863131129482608641870811018783713415329452942923899383757
        81744935321383258964188412694034608767596871938114266315314691939848160895420606
        68522903576222132984337920693831194696132449515702829424546025879634778359347248
        47632309294423188824420389498238758478560404195437541236481138391989400817204441
        73727269054722039939559357773196032606652425740389903202496224670544437964811213
        97442542073848272813998358924273860508659652819973478979438139142586181065558071
        33628086341570257263763605627566850927027376551310552112444807186627527822558757
        79034637289025412019019708664796419158437289944719047308735742096150814737761472
        67554137121811210568818840995458711550328274598416594034353064117047606251305701
        83681792792095439082732014564137929984930622205906418076188569014197203218366939
        16136781813969556017117721855015686264234700067419917233984820802105551798814449
        28336943965845632911386578898942370761542258917359214671829109853559308127299807
        65760377173794303496770546979204857691617495750447480554551405839392777956007570
        05323166385413033324059868839751280826011762565617090658606573612488732540304200
        70660363890597704694308784962149652927906284353282001913802890262330481000608649
        41681999127291187320199492731772717287701160953565230746685111243307042686038123
        46320048970831203321710396591963440580168315858335355535413579770085828573125609
        68171144744328702641395582484648553027091325723979903972838994326978148524852717
        41263769559428258587801695381475991892401580043303429253596087749817321065393494
        04021393856783866451144621075317413164507898667075228932034562815574855150334536
        99207971656489736564925293109242549342998210348758542616289953951411268872811967
        14642643096269495821618368027531029746038878757753575676994784668216016726569958
        50488736403571446106070133253787835509166640118586177427904029259127346066378627
        47202868577642326309428985480618719526940597693261077656542572179823236080279677
        60993062833501672522746079351740883175598039547375925858167714482056787582506511
        39005595665258955613405318511423398241880381578125861196881933010373091924613848
        42612929059176993540730775746653829217041637502382018635315096283041613681682409
        18836634281300010076232357784501927901743451563278486234707671011726233756476034
        08933721603862829931184059838043604162565133539010228488078753616056569770805296
        67681431709023562255966911667873416643613503178023486359815352505353514174646639
        50020800757515947479683417495334907390658376542385090682185419133211245241149966
        28699612586776530460989119740428191948336060178966278825204356891693268472960493
        11954102720548794297412675648777444104925233794868861149188866653240083519394541
        61115613858067437578273959881350329526114618348720777573531566311149578407904659
        28936449779422509357053540431610083385241307711286388655584093155255016852796469
        21600977251991529406474295235693044744654553619384507394283738779388645663297069
        61847768058833953054470510580836013692545844702573381591778528434143249289601639
        80824098071419916048855624910769499637896552328855908293808558597854685644965083
        70548170291499115192724689506050298582130521505935316510193507999808880656837746
        49130519241709911003764764399819496420239460500355095381322324419717874078809512
        32960370221012715113564316523537617211540047389436623718648059919987210194117358
        93439518302823943392491379172491524615847478973142705800830569702656708402712802
        97702813557410418678137721716568289721622213003386565877235942783367111984421543
        39671515247032300933517215089947443780075958948815700805999801403901632344264893
        77127086242622501926448704608043342012488513882270320128938185444144078019438850
        50880507628481013040345802221198149521541464283541075881515919967148666865472660
        75826211198927186066495416919813588193670522818002121146439553451306258518941434
        45391024170621610457493742666127264938266640567314851007639276915520490667301193
        46551953536681842248665077725233101966665882857512233437481884217057663756091314
        26279899748364474874367321655365209864848523538610080515157257113582042571933514
        00341252240830764323691986568394788244707804741961891564760821625354166559318513
        98107589278956322932613518953200397071907121398486492105191282226834541425757106
        70404465259448362691439822182314017139572365512964135439813859210699115099046427
        73553118732948749554978939372477622992483142920069679075601499325151050405770350
        30664296443970482570211736581157300801266045467657519887930932273798704595443123
        30603969304256694412741992101068471096204900685534047507704200986521101478772421
        20599078204500088787705638432815221389015815828141161340707088853967582098398990
        SQRT3
}
