#ident "$Id: raid6x86.h,v 1.1 2002/11/16 08:17:18 hpa Exp $"
/* ----------------------------------------------------------------------- *
 *   
 *   Copyright 2002 H. Peter Anvin - All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, Inc., 53 Temple Place Ste 330,
 *   Bostom MA 02111-1307, USA; either version 2 of the License, or
 *   (at your option) any later version; incorporated herein by reference.
 *
 * ----------------------------------------------------------------------- */

/*
 * raid6x86.h
 *
 * Definitions common to x86 and x86-64 RAID-6 code only
 */

#ifndef LINUX_RAID_RAID6X86_H
#define LINUX_RAID_RAID6X86_H
#if defined(__i386__) || defined(__x86_64__)

typedef struct {
	unsigned int fsave[27];
} raid6_mmx_save_t;

/* N.B.: For SSE we only save %xmm0-%xmm7 even for x86-64, since
   the code doesn't know about the additional x86-64 registers */
/* The spillover area is so we can actually align the stores */
typedef struct {
	unsigned int sarea[8*4];
	unsigned int pad[3];	/* Spillover area */
	unsigned int cr0;
} raid6_sse_save_t /* __attribute__((aligned(16))) */;

#ifdef __KERNEL__

/* Real code */

static inline raid6_before_mmx(raid6_mmx_save_t *s)
{
	preempt_disable();
	if (!test_thread_flag(TIF_USEDFPU))
		asm volatile("ctls");
	asm volatile("fsave %0 ; fwait" : "=m" (s->fsave[0]));
}

static inline raid6_after_mmx(raid6_mmx_save_t *s)
{
	asm volatile("frstor %0" : : "m" (s->fsave[0]));
	if (!test_thread_flag(TIF_USEDFPU))
		stts();
	preempt_enable();
}

static inline raid6_before_sse(raid6_sse_save_t *s)
{
#ifdef __x86_64__
	unsigned int *rsa = &s->sarea;
#else
	unsigned int *rsa =
		(unsigned int *)((((unsigned long)&s->sarea)+15) & ~15);
#endif
	preempt_disable();
	asm volatile("movl %%cr0,%0 ; "
		     "clts ; "
		     "movaps %%xmm0,%1 ; "
		     "movaps %%xmm1,%2 ; "
		     "movaps %%xmm2,%3 ; "
		     "movaps %%xmm3,%4 ; "
		     "movaps %%xmm4,%5 ; "
		     "movaps %%xmm5,%6 ; "
		     "movaps %%xmm6,%7 ; "
		     "movaps %%xmm7,%8"
		     : "=rm" (s->cr0),
		     "=m" (rsa[0]),
		     "=m" (rsa[4]),
		     "=m" (rsa[8]),
		     "=m" (rsa[12]),
		     "=m" (rsa[16]),
		     "=m" (rsa[20]),
		     "=m" (rsa[24]),
		     "=m" (rsa[28]) );
}

static inline raid6_after_sse(raid6_sse_save_t *s)
{
#ifdef __x86_64__
	unsigned int *rsa = &s->sarea;
#else
	unsigned int *rsa =
		(unsigned int *)((((unsigned long)&s->sarea)+15) & ~15);
#endif
	asm volatile("movaps %%xmm0,%1 ; "
		     "movaps %%xmm1,%2 ; "
		     "movaps %%xmm2,%3 ; "
		     "movaps %%xmm3,%4 ; "
		     "movaps %%xmm4,%5 ; "
		     "movaps %%xmm5,%6 ; "
		     "movaps %%xmm6,%7 ; "
		     "movaps %%xmm7,%8 ; "
		     "mov %0,%%cr0"
		     : : "rm" (s->cr0),
		     "m" (rsa[0]),
		     "m" (rsa[4]),
		     "m" (rsa[8]),
		     "m" (rsa[12]),
		     "m" (rsa[16]),
		     "m" (rsa[20]),
		     "m" (rsa[24]),
		     "m" (rsa[28]) );
	preempt_enable();
}

#else /* ! __KERNEL__ */

/* "Simulation" code to test in userspace */

static inline raid6_before_mmx(raid6_mmx_save_t *s)
{
	asm volatile("fsave %0 ; fwait" : "=m" (s->fsave[0]));
}

static inline raid6_after_mmx(raid6_mmx_save_t *s)
{
	asm volatile("frstor %0" : : "m" (s->fsave[0]));
}

static inline raid6_before_sse(raid6_sse_save_t *s)
{
#ifdef __x86_64__
	unsigned int *rsa = &s->sarea;
#else
	unsigned int *rsa =
		(unsigned int *)((((unsigned long)&s->sarea)+15) & ~15);
#endif
	asm volatile("movl $0xf00ba9,%0 ; "
		     "movaps %%xmm0,%1 ; "
		     "movaps %%xmm1,%2 ; "
		     "movaps %%xmm2,%3 ; "
		     "movaps %%xmm3,%4 ; "
		     "movaps %%xmm4,%5 ; "
		     "movaps %%xmm5,%6 ; "
		     "movaps %%xmm6,%7 ; "
		     "movaps %%xmm7,%8"
		     : "=rm" (s->cr0),
		     "=m" (rsa[0]),
		     "=m" (rsa[4]),
		     "=m" (rsa[8]),
		     "=m" (rsa[12]),
		     "=m" (rsa[16]),
		     "=m" (rsa[20]),
		     "=m" (rsa[24]),
		     "=m" (rsa[28]) );
}

static inline raid6_after_sse(raid6_sse_save_t *s)
{
#ifdef __x86_64__
	unsigned int *rsa = &s->sarea;
#else
	unsigned int *rsa =
		(unsigned int *)((((unsigned long)&s->sarea)+15) & ~15);
#endif
	asm volatile("movaps %%xmm0,%1 ; "
		     "movaps %%xmm1,%2 ; "
		     "movaps %%xmm2,%3 ; "
		     "movaps %%xmm3,%4 ; "
		     "movaps %%xmm4,%5 ; "
		     "movaps %%xmm5,%6 ; "
		     "movaps %%xmm6,%7 ; "
		     "movaps %%xmm7,%8"
		     : : "rm" (s->cr0),
		     "m" (rsa[0]),
		     "m" (rsa[4]),
		     "m" (rsa[8]),
		     "m" (rsa[12]),
		     "m" (rsa[16]),
		     "m" (rsa[20]),
		     "m" (rsa[24]),
		     "m" (rsa[28]) );
}

#endif /* ! __KERNEL__ */

#endif
#endif
