/* Source Installer, Copyright (c) 2005,2006,2007 Claudio Fontana

 dialog.c - dialogs for displaying messages and getting simple input

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#include "global.h"
#include "gui.h"
#include "dialog.h"

/* this function is called by many other dialogs to display a message
   box with buttons, and possibly a scroll for text2 */

extern GtkWidget*
dialog_create_msgbox_scroll(char* wintitle, char* title, 
			    char* text1, char* text2, char* text3,
			    GtkMessageType type, GtkButtonsType buttons);

static GtkWidget*
dialog_create_msgbox(char* wintitle, char* title,
		     char* text, char* opt, 
		     GtkMessageType type, GtkButtonsType buttons) {

  return dialog_create_msgbox_scroll(wintitle, title, text, 0, opt,
				     type, buttons);
}

extern GtkWidget*
dialog_create_msgbox_scroll(char* wintitle, char* title,
			    char* text1, char* text2, char* text3,
			    GtkMessageType type, GtkButtonsType buttons) {
  GtkWidget* msgbox;

  switch (buttons) {

  default:
  case GTK_BUTTONS_NONE:
  case GTK_BUTTONS_OK:
    msgbox = gtk_dialog_new_with_buttons(wintitle, GTK_WINDOW(w_main[W_MAIN]),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_OK,
					 0);
    break;
  case GTK_BUTTONS_CLOSE:
    msgbox = gtk_dialog_new_with_buttons(wintitle, GTK_WINDOW(w_main[W_MAIN]),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
					 0);
    break;
  case GTK_BUTTONS_CANCEL:
    msgbox = gtk_dialog_new_with_buttons(wintitle, GTK_WINDOW(w_main[W_MAIN]),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					 0);
    break;
  case GTK_BUTTONS_YES_NO:
    msgbox = gtk_dialog_new_with_buttons(wintitle, GTK_WINDOW(w_main[W_MAIN]),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_YES, GTK_RESPONSE_YES,
					 GTK_STOCK_NO, GTK_RESPONSE_NO,
					 0);
    break;
  case GTK_BUTTONS_OK_CANCEL:
    msgbox = gtk_dialog_new_with_buttons(wintitle, GTK_WINDOW(w_main[W_MAIN]),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_OK,
					 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					 0);
    break;
  }

  {
    GtkWidget* vbox, *hbox, *title_txt, *img;
    
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(msgbox)->vbox), hbox, 
		       TRUE, TRUE, 0);

    img = gtk_image_new_from_stock(type == GTK_MESSAGE_ERROR ?
				   GTK_STOCK_DIALOG_ERROR :
				   type == GTK_MESSAGE_WARNING ?
				   GTK_STOCK_DIALOG_WARNING :
				   type == GTK_MESSAGE_QUESTION ?
				   GTK_STOCK_DIALOG_QUESTION :
				   GTK_STOCK_DIALOG_INFO, 

				   GTK_ICON_SIZE_DIALOG);

    gtk_box_pack_start(GTK_BOX(hbox), img, FALSE, FALSE, bw);
				   
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
    
    title_txt = gtk_label_new(NULL);
    gtk_box_pack_start(GTK_BOX(vbox), title_txt, FALSE, FALSE, bw);

    title = srcinst_strjoin("<b>", title, "</b>", 0);
    gtk_label_set_markup(GTK_LABEL(title_txt), title);
    free(title);

    if (text1) {
      GtkWidget *txt1;
      txt1 = gtk_label_new(text1);
      gtk_box_pack_start(GTK_BOX(vbox), txt1, FALSE, FALSE, bw);
    }
    
    if (text2) {
      GtkWidget* scroll, *txt2;

      scroll = gtk_scrolled_window_new(NULL, NULL);
      txt2 = gtk_label_new(text2);

      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				     GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
      gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), txt2);

      gtk_box_pack_start(GTK_BOX(vbox), scroll, TRUE, TRUE, 0);
    }

    if (text3) {
      GtkWidget *txt3;

      txt3 = gtk_label_new(text3);
      gtk_box_pack_start(GTK_BOX(vbox), txt3, FALSE, FALSE, bw);
    }
    
    {
      GtkWidget *space;
      space = gtk_label_new(NULL);
      gtk_box_pack_start(GTK_BOX(hbox), space, FALSE, FALSE, bw / 2);
    }
  }

  { 
    GdkGeometry geo;
    geo.max_aspect = 8.0 / 4.0;
    geo.min_aspect = 3.0 / 4.0;

    gtk_window_set_geometry_hints(GTK_WINDOW(msgbox),
				  0, &geo, GDK_HINT_ASPECT);
  }

  gtk_widget_show_all(GTK_DIALOG(msgbox)->vbox);
  return msgbox;
}


/* display an error dialog, with window title WINTITLE,
   dialog title TITLE, additional TEXT and an optional 
   additional text argument OPT. Use NULL to avoid. Then quit. */

void dialog_error(char* wintitle, char* title, char* text, char* opt) {
  GtkWidget* msgbox;

  msgbox = dialog_create_msgbox(wintitle, title, text, opt,
				GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);
  gtk_dialog_run(GTK_DIALOG(msgbox));
  gtk_widget_destroy(msgbox);
  exit(1);
}

/* Same as the dialog_error function, but do not abort the program. */

void dialog_warning(char* wintitle, char* title, char* text, char* opt) {
  GtkWidget* msgbox;

  msgbox = dialog_create_msgbox(wintitle, title, text, opt,
				GTK_MESSAGE_WARNING, GTK_BUTTONS_OK);
  gtk_dialog_run(GTK_DIALOG(msgbox));
  gtk_widget_destroy(msgbox);
}

/* Asks the user for confermation. This function implies a YES|NO response,
   which is returned. Text2 and Text3 are optional. */

int dialog_confirm(char* wintitle, char* title, 
		   char* text1, char* text2, char* text3) {
  GtkWidget* msgbox;
  gint response;

  msgbox = dialog_create_msgbox_scroll(wintitle, title,
				       text1, text2, text3,
				       GTK_MESSAGE_QUESTION,
				       GTK_BUTTONS_YES_NO);

  response = gtk_dialog_run(GTK_DIALOG(msgbox));
  gtk_widget_destroy(msgbox);
  return response == GTK_RESPONSE_YES;
}

/* Display a message to the user, and wait for an OK. */

void dialog_message(char* wintitle, char* title, char* text, char* opt) {
  GtkWidget* msgbox;

  msgbox = dialog_create_msgbox_scroll(wintitle, title, text, opt, 0,
				       GTK_MESSAGE_INFO, GTK_BUTTONS_OK);

  gtk_dialog_run(GTK_DIALOG(msgbox));
  gtk_widget_destroy(msgbox);
}

static void dir_cursor_changed(GtkTreeView* dir_list, GtkWidget* widget) {
  GtkFileSelection* f;
  GtkTreeSelection* sel;
  GtkTreeModel* model;
  GtkTreeIter iter;
  char* value;

  f = GTK_FILE_SELECTION(widget);

  sel = gtk_tree_view_get_selection(dir_list);
  if (gtk_tree_selection_get_selected(sel, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &value, -1);
    gtk_entry_set_text(GTK_ENTRY(f->selection_entry), value);
    free(value);
  }
}
    
static void dir_row_activated(GtkTreeView* dir_list, GtkTreePath* arg1, GtkTreeViewColumn* arg2, GtkWidget* widget) {

  GtkFileSelection* f = GTK_FILE_SELECTION(widget);

  gtk_entry_set_text(GTK_ENTRY(f->selection_entry), "");
}

/* prompt the user for a filename (file or dir), existing or not */

char* dialog_input_opendir(char* title, char* spath) {
  char* filename, *startpath;
  GtkWidget* f, *paned;
  gint response;

  f = gtk_file_selection_new(title);
  
  gtk_file_selection_set_select_multiple(GTK_FILE_SELECTION(f), FALSE);

  if (spath) {
    startpath = srcinst_strjoin(spath, "/", 0);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(f), startpath);
    free(startpath);
  }

  paned = gtk_widget_get_ancestor(GTK_FILE_SELECTION(f)->dir_list,
				  GTK_TYPE_HPANED);

  gtk_widget_show_all(f);
  gtk_widget_hide(GTK_FILE_SELECTION(f)->fileop_del_file);
  gtk_widget_hide(GTK_FILE_SELECTION(f)->fileop_ren_file);

  if (paned) {
    gint max_position;
    g_object_get(G_OBJECT(paned), "max-position", &max_position, NULL);
    gtk_paned_set_position(GTK_PANED(paned), max_position);
  }

  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(f)->dir_list),
		   "cursor-changed", G_CALLBACK(dir_cursor_changed), f);
  
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(f)->dir_list),
		   "row-activated", G_CALLBACK(dir_row_activated), f);

  response = gtk_dialog_run(GTK_DIALOG(f));

  if (response == GTK_RESPONSE_OK) {
    filename = srcinst_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(f)));
    srcinst_rstrip(filename, '/');

  } else {
    filename = 0;
  }
  
  gtk_widget_destroy(f);
  return filename;
}

/* returns the index of the selected string, or -1 to cancel */

int dialog_input_select_string(char* title, char* text, struct srcinst_string_list* s) {
  GtkWidget* dialog, *vbox, *msg;
  GtkWidget** elements;
  gint response; int i;

  if (s->count == 0)
    return -1;

  dialog = gtk_dialog_new_with_buttons(title, GTK_WINDOW(w_main[W_MAIN]),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_OK,
				       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				       NULL);
  vbox = GTK_DIALOG(dialog)->vbox;
  msg = gtk_label_new(text);
  
  gtk_box_pack_start(GTK_BOX(vbox), msg, FALSE, FALSE, 0);

  elements = srcinst_malloc(sizeof(GtkWidget*) * s->count);
  
  elements[0] = gtk_radio_button_new_with_label(NULL, s->strings[0]);
  gtk_box_pack_start(GTK_BOX(vbox), elements[0], FALSE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(elements[0]), TRUE);
  
  for (i = 1; i < s->count; i++) {
    elements[i] = gtk_radio_button_new_with_label_from_widget(
      GTK_RADIO_BUTTON(elements[i - 1]), s->strings[i]);
    
    gtk_box_pack_start(GTK_BOX(vbox), elements[i], FALSE, FALSE, 0);
  }

  gtk_widget_show_all(dialog);
  response = gtk_dialog_run(GTK_DIALOG(dialog));

  if (response != GTK_RESPONSE_OK) {
    i = -1;
  } else {
    for (i = 0; i < s->count; i++) {
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(elements[i])) == TRUE)
	break;
    }
  }

  gtk_widget_destroy(dialog);
  free(elements);
  return i;
}

int dialog_input_package2(char* title, char** filename, char** packagename) {
  int i; int rv = 0;
  GtkWidget* dialog, *frame, *vbox, *label;
  GtkWidget* default_name, *custom_box, *custom_name, *custom_entry;
  GtkFileFilter* f;
  char* filters[] = { 
    "gzip-compressed tar archive (.tar.gz)", "*.tar.gz",
    "gzip-compressed tar archive (.tgz)", "*.tgz",
    "bzip2-compressed tar archive (.tar.bz2)", "*.tar.bz2",
    "bzip2-compressed tar archive (.tbz2)", "*.tbz2",
    "compressed tar archive (.tar.Z)", "*.tar.Z",
    "tar archive (.tar)", "*.tar",
    "zip archive (.zip)", "*.zip",
    "gzip-compressed shell archive (.shar.gz)", "*.shar.gz",
    "bzip2-compressed shell archive (.shar.bz2)", "*.shar.bz2",
    "compressed shell archive (.shar.Z)", "*.shar.Z",
    "shell archive (.shar)", "*.shar",
    0
  };

  *filename = *packagename = 0;

  dialog = gtk_file_chooser_dialog_new(title, GTK_WINDOW(w_main[W_MAIN]),
				       GTK_FILE_CHOOSER_ACTION_OPEN,
				       "Choose", 10,
				       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				       NULL);
  
  if (state.o.add_directory) {
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog),
					state.o.add_directory);
    gtk_file_chooser_add_shortcut_folder(GTK_FILE_CHOOSER(dialog),
					 state.o.add_directory, NULL);
  }

  frame = gtk_frame_new(NULL);
  gtk_file_chooser_set_extra_widget(GTK_FILE_CHOOSER(dialog), frame);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), bw);
  gtk_container_add(GTK_CONTAINER(frame), vbox);

  default_name = gtk_radio_button_new(NULL); /* default name */
  gtk_box_pack_start(GTK_BOX(vbox), default_name, FALSE, FALSE, 0);

  label = gtk_label_new(NULL);
  gtk_label_set_markup(GTK_LABEL(label), "<small>Assign default package name</small>");
  gtk_container_add(GTK_CONTAINER(default_name), label);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(default_name), TRUE);

  custom_box = gtk_hbox_new(FALSE, bw);
  gtk_box_pack_start(GTK_BOX(vbox), custom_box, TRUE, TRUE, 0);

  custom_name = gtk_radio_button_new_from_widget( /* custom name */
				 GTK_RADIO_BUTTON(default_name));
  gtk_box_pack_start(GTK_BOX(custom_box), custom_name, FALSE, FALSE, 0);

  label = gtk_label_new(NULL);
  gtk_label_set_markup(GTK_LABEL(label), "<small>Custom package name:</small>");
  gtk_container_add(GTK_CONTAINER(custom_name), label);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(custom_name), FALSE);

  custom_entry = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(custom_box), custom_entry, FALSE, FALSE, 0);

  /* filters */

  f = gtk_file_filter_new();
  gtk_file_filter_set_name(f, "All known package formats");
 
  for (i = 0; filters[i]; i += 2) {
    gtk_file_filter_add_pattern(f, filters[i + 1]);
  }
  
  gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), f);

  for (i = 0; filters[i]; i += 2) {
    f = gtk_file_filter_new();
    gtk_file_filter_set_name(f, filters[i]);
    gtk_file_filter_add_pattern(f, filters[i + 1]);
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), f);
  }

  gtk_widget_show_all(dialog);

  if (gtk_dialog_run(GTK_DIALOG(dialog)) == 10) {
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(custom_name)) == TRUE) {
      char* text;
      if (*(text = gtk_entry_get_text(GTK_ENTRY(custom_entry))) != '\0')
	*packagename = srcinst_strdup(text);
    }
    
    *filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    rv = 1;
  }

  gtk_widget_destroy(dialog);
  return rv;
}

int dialog_input_package(char** filename, char** packagename) {
  int rv = 0;
  GtkWidget* f, *vbox, *frame, *label;
  GtkWidget* default_name, *custom_box, *custom_name, *custom_entry;

  char* title = "Choose source package or directory";
  *filename = *packagename = 0;

  f = gtk_file_selection_new(title);
  gtk_file_selection_set_select_multiple(GTK_FILE_SELECTION(f), FALSE);

  if (state.o.add_directory) {
    char* startpath;
    startpath = srcinst_strjoin(state.o.add_directory, "/", 0);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(f), startpath);
    free(startpath);
  }

  gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(f));
  vbox = GTK_FILE_SELECTION(f)->main_vbox;
  frame = gtk_frame_new(NULL);
  gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
  /*   gtk_container_set_border_width(GTK_CONTAINER(frame), bw); */

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), bw);
  gtk_container_add(GTK_CONTAINER(frame), vbox);

  default_name = gtk_radio_button_new(NULL); /* default name */
  gtk_box_pack_start(GTK_BOX(vbox), default_name, FALSE, FALSE, 0);

  label = gtk_label_new(NULL);
  gtk_label_set_markup(GTK_LABEL(label), "<small>Assign default package name</small>");
  gtk_container_add(GTK_CONTAINER(default_name), label);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(default_name), TRUE);

  custom_box = gtk_hbox_new(FALSE, bw);
  gtk_box_pack_start(GTK_BOX(vbox), custom_box, TRUE, TRUE, 0);

  custom_name = gtk_radio_button_new_from_widget( /* custom name */
				 GTK_RADIO_BUTTON(default_name));
  gtk_box_pack_start(GTK_BOX(custom_box), custom_name, FALSE, FALSE, 0);

  label = gtk_label_new(NULL);
  gtk_label_set_markup(GTK_LABEL(label), "<small>Custom package name:</small>");
  gtk_container_add(GTK_CONTAINER(custom_name), label);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(custom_name), FALSE);

  custom_entry = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(custom_box), custom_entry, FALSE, FALSE, 0);

  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(f)->dir_list),
		   "cursor-changed", G_CALLBACK(dir_cursor_changed), f);
  
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(f)->dir_list),
		   "row-activated", G_CALLBACK(dir_row_activated), f);

  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(f)->file_list),
		   "cursor-changed", G_CALLBACK(dir_cursor_changed), f);

  gtk_widget_show_all(f);

  if (gtk_dialog_run(GTK_DIALOG(f)) == GTK_RESPONSE_OK) {
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(custom_name)) == TRUE) {
      char* text;
      if (*(text = gtk_entry_get_text(GTK_ENTRY(custom_entry))) != '\0')
	*packagename = srcinst_strdup(text);
    }
    
    *filename = srcinst_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(f)));
    srcinst_rstrip(*filename, '/');
    rv = 1;
  }

  gtk_widget_destroy(f);
  return rv;
}

/* show configuration dialog, return choices in the passed string list.
   Return 0 if the user cancelled the operation.
 */

int dialog_configure(struct srcinst_configure_sets* opts, char **prefix,
		     struct srcinst_string_list* s) {
  GtkWidget* dialog, *vbox, *notebook;
  GtkTooltips* tips;
  char* title = "Package Configuration";
  gint response;
  int i, j;

  struct {
    char* title;
    struct srcinst_configure_options* set;

    GtkWidget** check_buttons;
    GtkWidget** entries;

  } columns[6];

  dialog = gtk_dialog_new_with_buttons(title, GTK_WINDOW(w_main[W_MAIN]),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_OK,
				       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				       NULL);

  {
    gint width, height;
    GdkScreen* screen;
    screen = gdk_screen_get_default();
    width = -1;
    height = gdk_screen_get_height(screen) / 2;

    gtk_window_set_default_size(GTK_WINDOW(dialog), width, height);
  }

  vbox = GTK_DIALOG(dialog)->vbox;

  notebook = gtk_notebook_new();
  gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
  tips = gtk_tooltips_new();
  
  columns[0].set = &(opts->feats); columns[0].title = "Features";
  columns[1].set = &(opts->packs); columns[1].title = "Packages";
  columns[2].set = &(opts->envs); columns[2].title = "Environment";
  columns[3].set = &(opts->olds); columns[3].title = "Features and Packages";
  columns[4].set = &(opts->res1); columns[4].title = "Uncategorized";
  columns[5].set = &(opts->base); columns[5].title = "Directories";

  for (i = 0; i < 6; i++) {
    GtkWidget* vertical, *lbl, *scroll;

    if (columns[i].set->count == 0)
      continue;

    columns[i].check_buttons = srcinst_malloc(sizeof(GtkWidget*) * 
					      columns[i].set->count);
    columns[i].entries = srcinst_malloc(sizeof(GtkWidget*) *
					columns[i].set->count);

    scroll = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				   GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    
    lbl = gtk_label_new(columns[i].title);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), scroll, lbl);

    vertical = gtk_vbox_new(FALSE, 0);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll),
					  vertical);

    for (j = 0; j < columns[i].set->count; j++) {
      GtkWidget* horizontal, *label; char* text;
      char* key, *param, *comment;

      horizontal = gtk_hbox_new(FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vertical), horizontal, FALSE, FALSE, 0);
      
      columns[i].check_buttons[j] = gtk_check_button_new();
      columns[i].entries[j] = gtk_entry_new();
      gtk_box_pack_end(GTK_BOX(horizontal), columns[i].entries[j],
		       FALSE, FALSE, 0);

      gtk_entry_set_width_chars(GTK_ENTRY(columns[i].entries[j]), 
				i == 5 ? 20 : i == 2 ? 15 : 10);

      if (i == 5 && j == 0 && state.o.prefix) {
	/* pre set prefix option value */
	gtk_entry_set_text(GTK_ENTRY(columns[i].entries[j]), state.o.prefix);
      }

      gtk_box_pack_start(GTK_BOX(horizontal), columns[i].check_buttons[j],
			 FALSE, FALSE, 0);

      label = gtk_label_new(columns[i].set->options[j].key);
      gtk_container_add(GTK_CONTAINER(columns[i].check_buttons[j]), label);

      key = columns[i].set->options[j].key;
      param = columns[i].set->options[j].param;
      comment = columns[i].set->options[j].comment;

      text = srcinst_strjoin(key, param ? "  [" : "", param ? param : "",
			     param ? "]\n" : "\n", comment ? comment : "", 0);

      gtk_tooltips_set_tip(GTK_TOOLTIPS(tips), columns[i].check_buttons[j],
                           text, 0);

      free(text);
    }
  }
  
  gtk_widget_show_all(dialog);
  response = gtk_dialog_run(GTK_DIALOG(dialog));

  for (i = 0; i < 6; i++) {
    if (columns[i].set->count == 0)
      continue;

    for (j = 0; j < columns[i].set->count; j++) {
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(columns[i].check_buttons[j])) == TRUE) {

	char* key; const char* value;
	key = columns[i].set->options[j].key;
	value = gtk_entry_get_text(GTK_ENTRY(columns[i].entries[j]));
	
	if (i == 5 && j == 0) {
	  /* prefix is handled specially */
	  
	  if (*value) 
	    *prefix = srcinst_strdup(value);
	  else
	    *prefix = 0;
	  
	} else {
	  char *string;
	  
	  if (*value)
	    string = srcinst_strjoin(key, "=", value, 0);
	  else
	    string = srcinst_strdup(key);
				 
	  srcinst_add_string_list(s, string);
	  free(string);
	}
      }
    }
    
    free(columns[i].check_buttons);
    free(columns[i].entries);
  }
  
  gtk_widget_destroy(dialog);
  gtk_object_destroy(GTK_OBJECT(tips));
  
  return response == GTK_RESPONSE_OK;
}

/* input a filename for saving */

char* dialog_input_export_bin(char* title, char* defaultname, SRCINST_COMP fmt) {
  GtkWidget* dialog;
  gint response;
  char* filename = 0;

  if (defaultname) {
    defaultname = srcinst_strjoin(defaultname, ".tar", srcinst_compression_format_to_ext(fmt), 0);
  }
  
  dialog = gtk_file_chooser_dialog_new(title, GTK_WINDOW(w_main[W_MAIN]),
				       GTK_FILE_CHOOSER_ACTION_SAVE,
				       "Export", GTK_RESPONSE_OK,
				       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				       NULL);
  
  if (defaultname) {
    gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), defaultname);
  }

  response = gtk_dialog_run(GTK_DIALOG(dialog));

  if (response == GTK_RESPONSE_OK) {
    filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
  }

  if (defaultname)
    free(defaultname);

  gtk_widget_destroy(dialog);
  return filename;
}

static void export_format_toggled(GtkToggleButton* button, GtkWidget* dialog, char* ext) {
  char* filename, *fileext, *newname;

  if (gtk_toggle_button_get_active(button) == FALSE)
    return;

  if (!(filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog))))
    return;

  if (*(fileext = srcinst_file_ext(filename)) == '\0')
    return;

  *fileext = 0;
  newname = srcinst_strjoin(filename, ext, 0);
  free(filename);

  gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), 
				    srcinst_basename(newname));
  free(newname);
}

static void export_format_toggled_txt(GtkToggleButton* toggle, GtkWidget* dialog) {
  export_format_toggled(toggle, dialog, ".txt");
}

static void export_format_toggled_xml(GtkToggleButton* toggle, GtkWidget* dialog) {
  export_format_toggled(toggle, dialog, ".xml");
}

static void export_format_toggled_lst(GtkToggleButton* toggle, GtkWidget* dialog) {
  export_format_toggled(toggle, dialog, ".lst");
}

char* dialog_input_export_info(char* title, char* defaultname, SRCINST_EXPORT* fmt) {
  GtkWidget* dialog, *vbox, *mybox, *frame;
  GtkWidget* txt, *xml, *lst;
  gint response;
  char* filename = 0;

  if (defaultname) {
    defaultname = srcinst_strjoin(defaultname, ".xml", 0);
  }

  dialog = gtk_file_chooser_dialog_new(title, GTK_WINDOW(w_main[W_MAIN]),
				       GTK_FILE_CHOOSER_ACTION_SAVE,
				       "Export", GTK_RESPONSE_OK,
				       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				       NULL);

  gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), defaultname ?
				    defaultname : "all-packages.xml");


  vbox = GTK_DIALOG(dialog)->vbox;

  frame = gtk_frame_new("Export as");
  gtk_file_chooser_set_extra_widget(GTK_FILE_CHOOSER(dialog), frame);
  /*  gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0); */

  mybox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(frame), mybox);

  txt = gtk_radio_button_new_with_label(NULL, "Plain text (.txt)");
  xml = gtk_radio_button_new_with_label_from_widget(
	    GTK_RADIO_BUTTON(txt), "XML (.xml)");
  if (defaultname)
    lst = gtk_radio_button_new_with_label_from_widget(
	      GTK_RADIO_BUTTON(txt), "File List (.lst)");
  
  gtk_box_pack_start(GTK_BOX(mybox), txt, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(mybox), xml, FALSE, FALSE, 0);
  if (defaultname) gtk_box_pack_start(GTK_BOX(mybox), lst, FALSE, FALSE, 0);

  g_signal_connect(G_OBJECT(txt), "toggled",
		   G_CALLBACK(export_format_toggled_txt), dialog);

  g_signal_connect(G_OBJECT(xml), "toggled",
		   G_CALLBACK(export_format_toggled_xml), dialog);

  if (defaultname)
    g_signal_connect(G_OBJECT(lst), "toggled",
		     G_CALLBACK(export_format_toggled_lst), dialog);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(xml), TRUE);
  gtk_widget_show_all(dialog);
  response = gtk_dialog_run(GTK_DIALOG(dialog));

  if (response == GTK_RESPONSE_OK) {
    filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(txt)) == TRUE) {
      *fmt = SRCINST_EXPORT_TXT;
    } else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(xml)) == TRUE) {
      *fmt = SRCINST_EXPORT_XML;
    } else if (defaultname && 
	       gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lst)) == TRUE) {
      *fmt = SRCINST_EXPORT_LST;
    }
  }

  if (defaultname)
    free(defaultname);

  gtk_widget_destroy(dialog);
  return filename;
}
