/* text-finger.c -- How to do textual GNU fingering. */

/* Copyright (C) 1988, 1990, 1992  Free Software Foundation, Inc.

   This file is part of GNU Finger.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <config.h>
#include <packet.h>
#include <pwd.h>
#include <general.h>

/* Tell CONNECTION to send us the info on USER, and then print that
   info out. */
void
text_finger (user, connection)
  char *user;
  int connection;
{
  int out_fd, in_fd;
  FILE *out_stream, *in_stream;

  /* This buffer is used only to fragment the data; it is not a limit. */
  char line[256];

  out_fd = dup (connection);
  in_fd = dup (connection);

  if (!(in_stream = fdopen (in_fd, "r"))
      || !(out_stream = fdopen (out_fd, "w")))
    return;

  fprintf (out_stream, "%s\r\n", user);
  fflush (out_stream);

  while (fgets (line, sizeof line, in_stream))
    {
      int n = strlen (line);

      if (n > 1 && line[n - 2] == '\r')
	{
	  line[n - 2] = 0;

	  fputs (line, stdout);
	  putc ('\n', stdout);
	}
      else
	fputs (line, stdout);
    }
  
  fflush (stdout);

  close (in_fd);
  close (out_fd);

  fclose (in_stream);
  fclose (out_stream);
}


/* Produce long finger information for USER.  Print the textual output
   to STREAM. Perform net-local finger if LOCALFLAG is nonzero, otherwise
   perform net-global finger. */
int
long_finger (user, stream, localflag)
  char *user;
  FILE *stream;
  int localflag;
{
  FINGER_PACKET **get_finger_data ();
  struct passwd *entry = getpwnam (user);
  char *pw_real_name (), *strindex ();
  int entries_printed = 0;
  extern int mail_list ();
  extern int maybe_special_target ();


  if (!entry)
    {
      char *t;

      if (maybe_special_target (user, stream, 'l')
	  || maybe_special_target (user, stream, 'x'))
	return;
      
      /* Here is protection against busted YP implementations. */
#if defined (BROKEN_YP)

      {
	char **user_list = (char **)NULL;
	int user_size, user_index;

	user_size = user_index = 0;

	setpwent ();
	while (entry = getpwent ())
	  {
	    if (matches_username (user, pw_real_name (entry))
		|| matches_username (user, entry->pw_name))
	      {
		if (user_index + 1 >= user_size)
		  user_list = (char **)
		    xrealloc (user_list, (user_size += 50) * sizeof (char *));

		user_list[user_index++] = savestring (entry->pw_name);
		user_line[user_index] = (char *)NULL;
	      }
	  }
	endpwent ();

	if (user_index)
	  {
	    fprintf (stream, "Users who have `%s' in their real names:\r\n", user);

	    for (user_index = 0; user_list[user_index]; user_index++)
	      {
		entry = getpwnam (user_list[user_index]);

		if (entry)
		  display_long_info (entry, stream, localflag);

		free (user_list[user_index]);
	      }
	    free (user_list);
	  }
	else
	  if (!mail_list (user, stream))
	    return 0;
      }
#else /* !BROKEN_YP */

      setpwent ();

      while (entry = getpwent ())
	if (matches_username (user, pw_real_name (entry))
	    || matches_username (user, entry->pw_name))
	  {
	    if (!entries_printed)
	      fprintf (stream, "Users who have `%s' in their names:\r\n", user);

	    display_long_info (entry, stream, localflag);
	    entries_printed++;
	  }
      endpwent ();
#endif /* !BROKEN_YP */

      if (!entries_printed)
	if (!mail_list (user, stream))
	  return 0;
    }
  else
    display_long_info (entry, stream, localflag);

  fflush (stream);
  return 1;
}
