/*
 * TileSet.cs - Load tile sets for display in Mahjongg.
 *
 * Copyright (C) 2003  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Mahjongg
{

using System;
using Qt;

// This class loads the images for a set of tiles.  The layout 
// of the image file must be the same as that used by KMahjongg,
// except that we use transparency-based pngs instead of bmps:
// http://www.kde.org/kdegames/kmahjongg/

public sealed class TileSet : IDisposable
{
    // Internal state.
    private QImage[] tiles;
    private QImage normal;
    private QImage highlight;

    // Useful constants.
    private const int TileFaceWidth  = 34;
    private const int TileFaceHeight = 50;
    private const int TileWidth      = 40;
    private const int TileHeight     = 55;
    private const int TileOverlap    = 4;

    // Constructor.
    public TileSet(String filename)
	    	{
				// Load the main tileset.
				QImage image = new QImage(filename);

				// Extract all of the tiles and create smaller images.
				Tile tile;
				QPoint top;
				tiles = new QImage [((int)(Tile.Max)) + 1];
				for(tile = Tile.Min; tile <= Tile.Max; ++tile)
				{
		    		top = findTile(tile);
					tiles[(int)tile] = image.Copy
						(top.X(), top.Y(), TileFaceWidth, TileFaceHeight);
				}
		
				// Extract the tile blocks for normal and highlighted.
				top = new QPoint(5 + 6 * 40 + 35, 4 * 56);
				normal = image.Copy(top.X(), top.Y(), TileWidth, TileHeight);
				top = new QPoint(5 + 6 * 40 + 35 + 40, 4 * 56);
				highlight = image.Copy(top.X(), top.Y(), TileWidth, TileHeight);
		
				// Dispose the main image, which we no longer need.
				image.Dispose();
    		}

    // Destructor.
    ~TileSet()
	    	{
				Dispose();
	    	}

    // Implement IDisposable.
    public void Dispose()
	    	{
				if(tiles != null)
				{
		    		Tile tile;
		    		for(tile = Tile.Min; tile <= Tile.Max; ++tile)
		    		{
						tiles[(int)tile].Dispose();
		    		}
		    		normal.Dispose();
		    		highlight.Dispose();
				}
				tiles = null;
				normal = null;
				highlight = null;
	    	}

    // Locate the top-left corner of a particular tile.
    private QPoint findTile(Tile tile)
	    	{
				int row, col;
				if(tile >= Tile.Char_1 && tile <= Tile.Char_9)
				{
		    		row = 0;
		    		col = ((int)tile) - ((int)(Tile.Char_1));
				}
				else if(tile >= Tile.Wheel_1 && tile <= Tile.Wheel_9)
				{
		    		row = 1;
		    		col = ((int)tile) - ((int)(Tile.Wheel_1));
				}
				else if(tile >= Tile.Bamboo_1 && tile <= Tile.Bamboo_9)
				{
		    		row = 2;
		    		col = ((int)tile) - ((int)(Tile.Bamboo_1));
				}
				else if(tile >= Tile.Season_1 && tile <= Tile.Wind_West)
				{
		    		row = 3;
		    		col = ((int)tile) - ((int)(Tile.Season_1));
				}
				else
				{
		    		row = 4;
		    		col = ((int)tile) - ((int)(Tile.Dragon_Green));
				}
				return new QPoint(5 + col * 40, 1 + row * 56);
	    	}

    // Get the size of the tiles on-screen.
    public int Width
	    	{
				get
				{
					return TileWidth;
				}
	    	}
    public int Height
	    	{
				get
				{
					return TileHeight;
				}
	    	}
    public int Overlap
	    	{
				get
				{
					return TileOverlap;
				}
	    	}

    // Draw a tile.
    public void DrawTile(QPainter painter, int x, int y,
			 			 Tile tile, bool highlighted)
	    	{
				if(highlighted)
				{
		    		painter.DrawImage(x, y, highlight);
				}
				else
				{
		    		painter.DrawImage(x, y, normal);
				}
	    		painter.DrawImage(x + 5, y + 1, tiles[(int)tile]);
	    	}

}; // class TileSet

}; // namespace Mahjongg
