// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "server.h"
#include <getopt.h>
#include <sys/wait.h>

#ifdef	_POSIX_MEMLOCK
#include <sys/mman.h>
#endif

int mainpid;
int tgipipe[2];
static int *tgipids = NULL;
static Resolver *resolver = NULL;

#ifndef	HAVE_SETENV
static setenv(char *name, const char *value, int overwrite)
{
	char strbuf[256];

	sprintf(strbuf, "%s=%s", name, value);
	putenv(strbuf);
}
#endif

static void rt(void)
{
	int min, max;
	int pages = keythreads.getPages();
	int pri = keythreads.getPriority();
	int policy = keythreads.getPolicy();
	char memory[1024 * pages];

#ifdef	_POSIX_PRIORITY_SCHEDULING
	struct sched_param p;
	
	sched_getparam(0, &p);
	pri += p.sched_priority;
	if(!policy)
		policy = sched_getscheduler(0);

	min = sched_get_priority_min(policy);
	max = sched_get_priority_max(policy);
	if(pri < min)
		pri = min;
	if(pri > max)
		pri = max;
	p.sched_priority = pri;
	sched_setscheduler(getpid(), policy, &p); 
#else
	nice(-pri);
#endif

#ifdef	MCI_CURRENT
	if(pages)
		mlockall(MCI_CURRENT | MCI_FUTURE);
#endif
}

static void tgi(void)
{
	int argc;
	char *argv[65];
	char buffer[512];
	int cfd, vpid, pool, stat;
	int count = keythreads.getGateways();
	tgipids = new int[count];
	sigset_t sigs;
	tgicmd_t cmd;
	TGI *tgi;
	int status;

	if(canModify(keypaths.getRunfiles()))
	{
		strcpy(buffer, keypaths.getRunfiles());
		strcat(buffer, "/bayonne.ctrl");
	}
	else
		sprintf(buffer, "%s/.bayonne.ctrl", getenv("HOME"));

	pipe(tgipipe);

	for(pool = 0; pool < count; ++ pool)
	{
#ifdef	__FreeBSD__
		tgipids[pool] = vfork();
#else
		tgipids[pool] = fork();
#endif
		if(!tgipids[pool])
			break;
	}
	
	if(pool == count)
	{
		close(tgipipe[0]);
		return;
	}

	close(tgipipe[1]);

	setgid(keyserver.getGid());
	setuid(keyserver.getUid());
	sigemptyset(&sigs);
	sigaddset(&sigs, SIGINT);
	signal(SIGINT, SIG_DFL);
	sigprocmask(SIG_UNBLOCK, &sigs, NULL);

	slog(SLOG_DEBUG) << "tgi: initialized; uid=" << getuid() << " pid=" << getpid() << endl;
	nice(-keythreads.priGateway());

	sleep(10);
	cfd = open(buffer, O_RDWR);
	while(cfd > -1 && cfd < 3)
		cfd = dup(cfd);
	slog(SLOG_DEBUG) << "tgi: buffer=" << buffer << "; cfd=" << cfd << endl;

	while(read(tgipipe[0], &cmd, sizeof(cmd)) == sizeof(cmd))
	{
		argc = 0;
		slog(SLOG_DEBUG) << "tgi: cmd=" << cmd.cmd << endl;
		tgi = getInterp(cmd.cmd);
		if(tgi)
		{
			if(!cmd.detach)
				sprintf(buffer, "wait %d %d\n", cmd.port, getpid());
			status = tgi->Parse(cfd, cmd.port, cmd.cmd);
			if(!cmd.detach)
				sprintf(buffer, "exit %d %d\n", cmd.port, status);
			continue;
		}
		vpid = vfork();
		if(vpid)
		{
			if(!cmd.detach)
			{
				sprintf(buffer, "wait %d %d\n", cmd.port, vpid);
				write(cfd, buffer, strlen(buffer));
			}

#ifdef	__FreeBSD__
			wait4(vpid, &stat, 0, NULL);
#else
			waitpid(vpid, &stat, 0);
#endif
			if(!cmd.detach)
			{
				sprintf(buffer, "exit %d %d\n", cmd.port, WEXITSTATUS(stat));
				write(cfd, buffer, strlen(buffer));
			}
			continue;
		}

		sprintf(buffer, "%d", cmd.port);
		setenv("PORT_NUMBER", buffer, 1);
		argv[argc++] = strtok(cmd.cmd, " \t");
		while(NULL != (argv[argc] = strtok(NULL, " \t")))
		{
			if(!strnicmp(argv[argc], "digits=", 7))
			{
				setenv("PORT_DIGITS", argv[argc] + 7, 1);
				continue;
			}

			if(!strnicmp(argv[argc], "dlid=", 5))
			{
				setenv("PORT_DLID", argv[argc] + 5, 1);
				continue;
			}

			if(!strnicmp(argv[argc], "clid=", 5))
			{
				setenv("PORT_CLID", argv[argc] + 5, 1);
				continue;
			}

			if(!strnicmp(argv[argc], "query=", 6))
			{
				setenv("PORT_QUERY", argv[argc] + 6, 1);
				continue;
			}
			++argc;
		}
		argv[argc] = NULL;
//		slog.close();
		close(0);
		close(1);
//		close(2);
		open("/dev/null", O_RDWR);
		dup2(cfd, 1);
//		dup2(cfd, 2);
		close(cfd);

		sprintf(buffer, "%s/%s", getenv("SERVER_LIBEXEC"), *argv);
		getInterp(buffer, argv);
		execvp(buffer, argv);
		slog(SLOG_ERROR) << "tgi: exec failed; " << buffer << endl;
		exit(-1);
	}
	exit(SIGINT);
}

static RETSIGTYPE final(int sig)
{
	int count = keythreads.getGateways();
	int i;

	if(getpid() != mainpid)
	{
		kill(mainpid, sig);
		ccxx_sleep(~0);
	}
	signal(SIGINT, SIG_IGN);
	signal(SIGABRT, SIG_IGN);
	signal(SIGTERM, SIG_IGN);
	signal(SIGQUIT, SIG_IGN);

	network.Stop();
	scheduler.Stop();
	stopServers();
	endMaps();

	if(resolver)
		delete resolver;

	if(driver)
		driver->Stop();

	if(sig)
		slog(SLOG_WARNING) << "exiting: reason=" << sig << endl;
	else
		slog(SLOG_NOTICE) << "normal shutdown" << endl;

	if(tgipids)
	{
		for(i = 0; i < count; ++i)
			kill(tgipids[i], SIGINT);
		tgipids = NULL;
	}
	exit(sig);
}

static void initial(int argc, char **argv)
{
	static struct option long_options[] = {
		{"background", 0, 0, 'D'},
		{"foreground", 0, 0, 'F'},
		{"daemon", 0, 0, 'D'},
		{"help", 0, 0, 'h'},
		{"priority", 1, 0, 'p'},
		{"driver", 1, 0, 'd'},
		{"node", 1, 0, 'n'},
		{"test", 0, 0, 't'},
		{"trace", 0, 0, 'T'},
		{"version", 0, 0, 'V'},
		{"gui", 0, 0, 'G'},
		{"display", 1, 0, 'X'},
		{0, 0, 0, 0}};

	static bool daemon = false;
	static bool usage = false;
	static bool gui = false;

	mainpid = getpid();
	char prefix[256];
	char *pp;
	sigset_t sigs;
	int opt, opt_index;
	char *opt_script = NULL;
	int ports;
	bool test = false;
	bool schedule = true;

	sigemptyset(&sigs);
	sigaddset(&sigs, SIGTERM);
	sigaddset(&sigs, SIGQUIT);
	sigaddset(&sigs, SIGINT);
	pthread_sigmask(SIG_BLOCK, &sigs, NULL);
	slog.level(SLOG_NOTICE);

	while(EOF != (opt = getopt_long(argc, argv, "GTDFX:hd:p:d:tn:", long_options, &opt_index)))
		switch(opt)
		{
		case 'X':
			setenv("DISPLAY", optarg, 1);
		case 'G':
			gui = true;
			plugins.setValue("debug", "gui");
			break;			
		case 'V':
			cout << VERPATH << endl;
			exit(0);
		case 'T':
			daemon = false;
			slog.level(SLOG_DEBUG);
			plugins.setValue("debug", "trace");
			break;			
		case 't':
			schedule = false;
			test = true;
			daemon = false;
			slog.level(SLOG_DEBUG);
			getcwd(prefix, sizeof(prefix) - 1);
			pp = strrchr(prefix, '/');
#if defined(have_montecarlo_h) || defined(HAVE_MONTECARLO_H)
			strcpy(pp, "/drivers/pika/pika.ivr");
#elif defined(HAVE_VPBAPI_H)
			strcpy(pp, "/drivers/vpb/vpb.ivr");
#elif defined(HAVE_LINUX_TELEPHONY_H)
			strcpy(pp, "/drivers/phonedev/phonedev.ivr");
#else
			strcpy(pp, "/drivers/dummy/dummy.ivr");
#endif
			plugins.setValue("driver", prefix);
			strcpy(pp, "/modules/functions/string.fun");
			plugins.setValue("functions", prefix);
			strcpy(pp, "/modules/auditing/append.aud");
			plugins.setValue("auditing", prefix);
			strcpy(pp, "/modules/translators/english.tts");
			plugins.setValue("languages", prefix);
			if(gui)
				strcpy(pp, "/modules/gui/gui.dbg");
			else
				strcpy(pp, "/server/test.dbg");
			plugins.setValue("debug", prefix);
			strcpy(pp, "/data/script");
			keypaths.setValue("scripts", prefix);
			strcpy(pp, "/data");
			keypaths.setValue("prompts", prefix);
			keypaths.setValue("libexec", prefix);
			break;
		case 'n':
			keyserver.setValue("node", optarg);
			break;
		case 'd':
			plugins.setValue("driver", optarg);
			break;
		case 'p':
			keythreads.setValue("priority", optarg);
			break;
		case 'D':
			daemon = true;
			break;
		case 'F':
			daemon = false;
			break;
		default:
		case 'h':
			usage = true;
		}

	if(optind < argc)
	{
		if(test && !gui)
		{
			strcpy(pp, "/modules/auditing/trace.dbg");
			plugins.setValue("debug", prefix);
		}
		keyserver.setValue("default", argv[optind++]);
		schedule = false;
	}

	if(usage || optind < argc)
	{
		clog << "use: bayonne [-options] [defscript]" << endl;
		exit(-1);
	}

	endKeydata();

	mkdir(keypaths.getDatafiles(), 0750);
	chdir(keypaths.getDatafiles());
	mkdir("tmp", 0770);
	mkdir("prompts", 0770);
	mkdir("cache", 0700);
	mkdir("maps", 0700);

	if(!getuid())
	{
		chown(keypaths.getDatafiles(), keyserver.getUid(), keyserver.getGid());
		chown("tmp", keyserver.getUid(), keyserver.getGid());
		chown("prompts", keyserver.getUid(), keyserver.getGid());
	}
	
	if(daemon)
	{
		close(0);
		close(1);
		close(2);
		pdetach();
		mainpid = getpid();
		slog.open("bayonne", SLOG_DAEMON);
		slog(SLOG_NOTICE) << "daemon mode started" << endl;
	}

	sprintf(prefix, "%s/tgi", keypaths.getLibexec());
	setenv("SERVER_PLATFORM", plugins.getDriverName(), 1);
	setenv("SERVER_LIBEXEC", prefix, 1);
	setenv("SERVER_SOFTWARE", "bayonne", 1);
	setenv("SERVER_VERSION", "2.0", 1);
	setenv("SERVER_TOKEN", keyserver.getToken(), 1);
	setenv("PATH", keypaths.getTgipath(), 1);

	slog(SLOG_INFO) << "TGI VERSION 2.0";
	slog() << "; driver=" << plugins.getDriverName();
	slog() << "; prefix=" << prefix << endl;

	slog(SLOG_DEBUG) << "Loading TGI plugins..." << endl;
	try
	{
		plugins.loadTGI();
	}
	catch(DSO *dso)
	{
		slog(SLOG_CRITICAL) << dso->getError() << endl;
		final(-1);
	}
	tgi();
	sleep(1);
	rt();

	try
	{
		slog(SLOG_DEBUG) << "Loading DSO plugin images..." << endl;
		if(!test)
			plugins.loadExtensions();
		plugins.loadDebug();
		plugins.loadDriver();
		plugins.loadFeeds();
		plugins.loadModules();
		plugins.loadFunctions();
		plugins.loadTranslators();
		plugins.loadAuditing();
		plugins.loadManagers();
		driver->getImage();
	}
	catch(Driver *drv)
	{
		slog(SLOG_CRITICAL) << "multiple drivers loaded or driver failure" << endl;
		final(-1);
	}
	catch(Socket *sock)
	{
		slog(SLOG_CRITICAL) << "socket interface binding failure" << endl;
		final(-1);
	}
	catch(Dir *dir)
	{
		slog(SLOG_CRITICAL) << keypaths.getScriptFiles(); 
		slog() << ": no script directory" << endl;
		final(-1);
	}
	catch(DSO *dso)
	{
		slog(SLOG_CRITICAL) << dso->getError() << endl;
		final(-1);
	}
	catch(InetAddress *in)
	{
		slog(SLOG_CRITICAL) << "protocol resolver failed" << endl;
		final(-1);
	}
	catch(Thread *)
	{
		slog(SLOG_CRITICAL) << "service failure" << endl;
		final(-1);
	}
	catch(...)
	{
		slog(SLOG_CRITICAL) << "unknown failure" << endl;
		final(-1);
	}

	if(!driver)
	{
		slog(SLOG_CRITICAL) << "no driver loaded" << endl;
		final(-1);
	}

	keyserver.loadGroups();
	if(canAccess(ETC_PREFIX "bayonne.clid"))
		new Map(ETC_PREFIX "bayonne.clid", "clid");

	if(canAccess(ETC_PREFIX "bayonne.dnis"))
		new Map(ETC_PREFIX "bayonne.dnis", "dnis");

	plugins.loadMaps();

	sync();
	ports = driver->Start();
	if(ports)
		slog(SLOG_NOTICE) << "driver started " << ports << " port(s)" << endl;
	else
	{
		slog(SLOG_CRITICAL) << "no trunk ports activated" << endl;
		final(-1);
	}

	if(keythreads.getResolver())
		resolver = new Resolver();

	startServers();

	if(schedule)
		scheduler.Start();

	network.Start();

	signal(SIGTERM, final);
	signal(SIGQUIT, final);
	signal(SIGINT, final);
	signal(SIGABRT, final);
	pthread_sigmask(SIG_UNBLOCK, &sigs, NULL);
	slog("bayonne", SLOG_DAEMON);
	slog(SLOG_NOTICE) << "normal startup; " << keyserver.getLast("config") << endl;
}

int main(int argc, char **argv)
{
	char buffer[256];
	char *p;
	int i;

	initial(argc, argv);
	if(debug->DebugTest())
		final(0);

	if(canModify(keypaths.getRunfiles()))
	{
		strcpy(buffer, keypaths.getRunfiles());
		strcat(buffer, "/bayonne.ctrl");
	}
	else
		sprintf(buffer, "%s/.bayonne.ctrl", getenv("HOME"));

	fifo.open(buffer);
	slog(SLOG_DEBUG) << "fifo: path=" << buffer << endl;
	if(!fifo.is_open())
	{
		slog(SLOG_WARNING) << "fifo: open failed" << endl;
		sleep(-1);
	}

	chown(buffer, keyserver.getUid(), keyserver.getGid());

	for(;;)
	{
		fifo.getline(buffer, sizeof(buffer) - 1);
		if(fifo.eof())
		{
			slog(SLOG_CRITICAL) << "fifo failure" << endl;
			final(-1);
		}
		p = buffer + strlen(buffer) - 1;
		while(p > buffer)
		{
			if(*p == ' ' || *p == '\t' || *p == '\n' || *p == '\r')
				*(p--) = 0;
			else
				break;
		}
		p = buffer;
		while(*p == ' ' || *p == '\t')
			++p;
		if(!*p)
			continue;
		fifo.Command(p);
	}
	fifo.close();
	final(0);
}

