// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"
#include <sys/ioctl.h>

PhonedevRecord::PhonedevRecord(PhonedevTrunk *trunk) :
Service((Trunk *)trunk, keythreads.priAudio()), AudioFile()
{
	stopped = false;
	reset = false;
	driver = trunk->getDriver();
	dev = trunk->getDevice();
}

PhonedevRecord::~PhonedevRecord()
{
	char buffer[12];
	struct stat ino;
	int trim;

	Terminate();
	sprintf(buffer, "%ld", getPosition());
	trunk->setSymbol(SYM_RECORDED, buffer);
	trim = tobytes(getEncoding(), atoi(trunk->getSymbol(SYM_TRIM)));
	stat(data->record.name, &ino);
	truncate(data->record.name, ino.st_size - trim);
	chown(data->record.name, keyserver.getUid(), keyserver.getGid());

	Close();
	if(reset)
	{
		ioctl(dev, PHONE_REC_STOP); 
		dspReset();
	}
}

void PhonedevRecord::Initial(void)
{
	int codec, framing, len, count = 0;
	char buffer[32];
	audioinfo_t info;
	char *ext = strrchr(data->record.name, '.');
	int ctrl;

	trunk->getName(buffer);
	info.format = AUDIO_FORMAT_RAW;
	info.encoding = MULAW_AUDIO_ENCODING;
	info.order = 0;
	info.annotation = trunk->getSymbol(SYM_ANNOTATION);
	info.rate = 8000;
	
	if(!ext)
		ext = trunk->getSymbol(SYM_EXTENSION);

	if(!stricmp(ext, ".al"))
		info.encoding = ALAW_AUDIO_ENCODING;
	else if(!stricmp(ext, ".au"))
	{
		info.format = AUDIO_FORMAT_SUN;
		info.order = __BIG_ENDIAN;
	}
	else if(!stricmp(ext, ".wav"))
	{
		info.format = AUDIO_FORMAT_RIFF;
		info.order = __LITTLE_ENDIAN;
	}

#ifdef	POSIX_PHONEDEV
	ctrl = data->record.volume * 10;
	ioctl(dev, PHONE_REC_VOLUME, &ctrl);
#else
	ioctl(dev, PHONE_REC_VOLUME, data->record.volume * 10);
#endif
	if(data->record.append)
		Open(data->record.name);
	else
		Create(data->record.name, &info);	
	if(!isOpen())
	{
		slog(SLOG_ERROR) << data->record.name << ": cannot open" << endl;
		Failure();
	}
	if(data->record.append)
		setPosition();

	switch(getEncoding())
	{
	case MULAW_AUDIO_ENCODING:
		codec = ULAW;
		framing = 20;
		samples = 24000;
		break;
	case ALAW_AUDIO_ENCODING:
		codec = ALAW;
		samples = 24000;
		len = 80;
	default:
		slog(SLOG_ERROR) << buffer << ": unsupported codec required" << endl;
		Failure();
	}

	reset = true;
#ifdef	POSIX_PHONEDEV
	ctrl = phivr.getAudioBuffers();
	ioctl(dev, PHONE_REC_DEPTH, &ctrl);
#else
	ioctl(dev, PHONE_REC_DEPTH, phivr.getAudioBuffers());
#endif

	switch(driver)
	{
#ifdef	IXJ_PHONEDEV
	case ixj_driver:
		if(framing == 20)
			framing = 30;
		break;
#endif
	}

#ifdef	POSIX_PHONEDEV
	for(;;)
	{
		ctrl = framing;
		ioctl(dev, PHONE_FRAME, &ctrl);
		if(framing == ctrl)
			break;
		if(++count > 4)
			break;
		Sleep(20);
	}
#else
	while(framing != ioctl(dev, PHONE_FRAME, framing) && count++ < 5)
		Sleep(20);
#endif

	if(count > 4)
	{
		slog(SLOG_ERROR) << buffer << ": cannot set framing" << endl;
 		Failure();
	}

#ifdef	POSIX_PHONEDEV
	if(ioctl(dev, PHONE_REC_CODEC, &codec))
#else
	if(ioctl(dev, PHONE_REC_CODEC, codec))
#endif
	{
		slog(SLOG_ERROR) << buffer << ": cannot set codec" << endl;
		Failure();
	}
	samples = (samples * framing) / 3000;
	bufsize = tobytes(getEncoding(), samples);
}

void PhonedevRecord::Run(void)
{
	char name[33];
	char buffer[bufsize];
	audioerror_t status = AUDIO_SUCCESS;
	int len = 0;

	trunk->getName(name);
	if(ioctl(dev, PHONE_REC_START))
	{
		slog(SLOG_ERROR) << name << ": failed channel record" << endl;
		Failure();
	}

	while(status == AUDIO_SUCCESS && !stopped)
	{	
		len = ::read(dev, buffer, bufsize);
		if(len < bufsize)
			status = AUDIO_READ_INCOMPLETE;
		else
			status = putSamples(buffer, samples);
	}
	if(status == AUDIO_WRITE_FAILURE)
		slog(SLOG_ERROR) << name << ": failed record" << endl;
	ioctl(dev, PHONE_REC_STOP);
	reset = false;
	Success();
}

bool PhonedevTrunk::recordHandler(TrunkEvent *event)
{
	unsigned short mask;

	switch(event->id)
	{
	case TRUNK_DTMF_KEYUP:
		mask = (1 << event->parm.dtmf.digit);
		if(!(mask & data.record.term))
			return false;
	case TRUNK_SERVICE_SUCCESS:
		TrunkSignal(TRUNK_SIGNAL_STEP);
		handler = &PhonedevTrunk::stepHandler;
		return true;
	case TRUNK_SERVICE_FAILURE:
		setSymbol(SYM_ERROR, "record-failed");
		TrunkSignal(TRUNK_SIGNAL_ERROR);
		handler = &PhonedevTrunk::stepHandler;
		return true;
	case TRUNK_ENTER_STATE:
		debug->DebugState(this, "record");
		flags.dsp = DSP_MODE_VOICE;
		status[id] = 'r';
		if(!flags.offhook)
			setHookState(true);
		if(data.record.term)
			setDTMFDetect(true);
		else
			setDTMFDetect();
		thread = new PhonedevRecord(this);
		thread->Start();
		setTimer(data.record.timeout);
		return true;
	}
	return false;
}

