// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne 
// with the Aculab telephony libraries to produce a executable image
// without requiring Aculab's sources to be supplied in a free software
// license long as each source file so linked contains this exclusion
// and the unalrtered Aculab source files are also provided.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include "driver.h"

AculabConfig::AculabConfig() :
Keydata("/bayonne/aculab")
{
	static KEYDEF defkeys[] = {
	{"brifirmware", "ets_usr.ram"},
	{"prifirmware", "ets_bnet.ram"},
	{"prosody", "sp30a.smf"},
	{"briconfig", "-cNE"},
	{"priconfig", "-cNE"},
	{NULL, NULL}};

	Load(defkeys);
}

unsigned AculabConfig::getMaxPorts(void)
{
	const char *val = getLast("maxports");
	int v;
	if(!val)
		return MAXPORT;

	v = atoi(val);
	if(v > MAXPORT)
		return MAXPORT;

	return v;
}

AculabDriver::AculabDriver() :
Driver(), AculabConfig(), Thread(NULL, keythreads.priService())
{
	running = false;
	port_count = 0;
	unsigned long mask;
	int port, ts;
	AculabTrunk *trunk;

	status = AculabTrunk::status;
	memset(status, ' ', sizeof(AculabTrunk::status));

	if(InitNetworks())
		if(InitProsody())
			port_count = call_nports();

	if(port_count < 0)
		port_count = 0;

	if(!port_count)
	{
		slog(SLOG_ERROR) << "Aculab driver failure; no ports" << endl; 
		trunks = NULL;
		groups = NULL;
		maps = NULL;
		return;
	}

	maps = new AculabTrunk *[port_count * MAXCHANNELS];
	trunks = new AculabTrunk *[port_count * MAXTIMESLOTS];
	groups = new TrunkGroup *[port_count * MAXTIMESLOTS];
	memset(trunks, 0, sizeof(AculabTrunk *) * port_count * MAXTIMESLOTS);
	memset(groups, 0, sizeof(TrunkGroup *) * port_count * MAXTIMESLOTS);
	memset(maps, 0, sizeof(AculabTrunk *) * port_count * MAXCHANNELS);

	call_signal_info(&siginfo[0]);
	for(port = 0; port < port_count; ++port)
	{
		mask = 1L;
		for(ts = 0; ts < MAXTIMESLOTS; ++ts)
		{
			if(mask & siginfo[port].validvector)
			{
				trunk = new AculabTrunk(port, ts);
				trunks[port * MAXTIMESLOTS + ts] = trunk;
			}
			mask <<=1L;
		}
	}			

	slog(SLOG_INFO) << "Aculab driver loaded; capacity=" << MAXPORT * MAXTIMESLOTS << endl;
}

AculabDriver::~AculabDriver()
{
	if(running)
		Terminate();

	Stop();

	if(trunks)
		delete trunks;

	if(groups)
		delete groups;

	if(maps)
		delete maps;
}

interface_t AculabDriver::getInterface(unsigned port)
{
	if(port >= port_count)
		return INVALID_PORT;

	return interfaces[port];
}

void AculabDriver::setChannel(AculabTrunk *trunk)
{
	if(trunk->lcn >= MAXCHANNELS)
		return;
	maps[trunk->port * MAXCHANNELS + trunk->lcn] = trunk;
}

void AculabDriver::clrChannel(AculabTrunk *trunk)
{
	if(trunk->lcn >= MAXCHANNELS)
		return;
	maps[trunk->port * MAXCHANNELS + trunk->lcn] = NULL;
}

bool AculabDriver::InitProsody(void)
{
	struct sm_download_parms sm_download_parms;
	int rc;
	int this_module;
	int total_modules = sm_get_modules();

	slog(SLOG_INFO) << "Aculab loading " << total_modules << " prosody modules" << endl;

	for(this_module = 0; this_module < total_modules; ++this_module)
	{
		if(sm_reset_module(this_module))
		{
			slog(SLOG_ERROR) << "Aculab module " << this_module << " failed, exiting..." << endl;
			return false;
		}
		sm_download_parms.module = this_module;
		sm_download_parms.id = 0;
		sm_download_parms.filename = (char *)getProsody();
		slog(SLOG_INFO) << "Aculab module " << this_module << " downloading..." << endl;
		if(sm_download_fmw(&sm_download_parms))
		{
			slog(SLOG_ERROR) << "Aculab module download failed; exiting..." << endl;
			return false;
		}
	}
	return true;
}

bool AculabDriver::InitNetworks(void)
{
	int total_switches;
	int this_port, total_ports;

	struct restart_xparms restart_xparms;

	int rc;

	pri_count = bri_count = 0;
	total_switches = sw_get_drvrs();
	if(total_switches < 1)
	{
		slog(SLOG_ERROR) << "Aculab Failure; no switch drivers" << endl;
		return false;
	}

	total_ports = call_nports();
	for(this_port = 0; this_port < getMaxPorts(); ++this_port)
		interfaces[this_port] = INVALID_PORT;

	for(this_port = 0; this_port < total_ports; ++this_port)
	{
		if(call_line(this_port) == L_BASIC_RATE)
		{
			interfaces[this_port] = BRI_PORT;
			++bri_count;
			restart_xparms.filenamep = (char *)getBriFirmware();
			restart_xparms.config_stringp = (char *)getBriConfig();
		}
		else
		{
			interfaces[this_port] = PRI_PORT;
			++pri_count;
			restart_xparms.filenamep = (char *)getPriFirmware();
			restart_xparms.config_stringp = (char *)getPriConfig();
		}
		if(call_is_download(this_port))
		{
			slog(SLOG_INFO) << "Aculab(" << this_port << "): loading " << restart_xparms.filenamep << " to " << this_port << endl;
			restart_xparms.net = this_port;
			rc = call_restart_fmw(&restart_xparms);
			if(rc)
			{
				slog(SLOG_ERROR) << "Aculab(" << this_port << "): firware load failure; " << restart_xparms.filenamep << endl;
				interfaces[this_port] = FAILED_PORT;
			}
		}
	}
	if(system_init())
	{
		slog(SLOG_ERROR) << "Aculab: system init failure" << endl;
		return false;
	}
	return true;
}

int AculabDriver::Start(void)
{
	int count = 0;

	if(active)
	{
		slog(SLOG_ERROR) << "driver already started" << endl;
		return 0;
	}

	slog(SLOG_INFO) << "driver starting..." << endl;

	if(!running)
		Thread::Start();

	active = true;
	return count;
}

void AculabDriver::Stop(void)
{
	if(!active)
		return;

	if(trunks)
		memset(trunks, 0, sizeof(AculabTrunk *) * port_count * MAXTIMESLOTS);

	active = false;
	slog(SLOG_INFO) << "driver stopping..." << endl;
}

Trunk *AculabDriver::getTrunkPort(int id)
{
	if(id < 0 || id >= port_count * MAXTIMESLOTS)
		return NULL;

	if(!trunks)
		return NULL;

	return (Trunk *)trunks[id];
}

// this should dispatch events thru putEvent!

void AculabDriver::Run(void)
{
	struct state_xparms event_xparms;
	struct detail_xparms detail_xparms;
	struct cause_xparms cause;
	struct in_xparms in_xparms;
	int port, ts, channel;
	unsigned long mask;
	int lcn, rc;
	char buf[1];
	AculabTrunk *trunk;
	TrunkEvent event;
	
	for(;;)
	{
		event_xparms.handle = 0;
		event_xparms.timeout = 500L;
		memset(&event, 0, sizeof(event));
	
		if(call_event(&event_xparms))
		{
			slog(SLOG_ERROR) << "Aculab event failure" << endl;
			Sleep(10000);
			continue;
		}
	
		if(!event_xparms.handle)
			continue;

		port = call_handle_2_port(event_xparms.handle);
		channel = call_handle_2_chan(event_xparms.handle);

		if(port >= MAXPORT)
			continue;

		if(channel >= MAXCHANNELS)
			continue;

		trunk = maps[port * MAXCHANNELS + channel];
		if(!trunk)
			continue;

		switch(event_xparms.state)
		{
		case EV_INCOMING_CALL_DET:
			// accept processing entirely in port thread???
			event.id = TRUNK_CALL_DETECT;
			break;
		case EV_CALL_CONNECTED:
			event.id = TRUNK_CALL_CONNECT;
			break;
		case EV_IDLE:
			cause.handle = event_xparms.handle;
			cause.cause = 0;
			cause.raw = 0;
	
			// call release and openin happens in
			// trunk state handler rather than here???
			event.id = TRUNK_CALL_RELEASE;
			break;
		}
		if(event.id)
			trunk->putEvent(&event);
	}		
}

AculabDriver aculabivr;
