/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * try_correlator.cc -- feed it some data and let it output it's answers
 */

#include <gr_mmse_fir_interpolator.h>

#include <unistd.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>


void
usage (const char *name)
{
  fprintf (stderr, "usage: %s [-F <filename>] [-M <mu>]\n", name);
  fprintf (stderr, "   -F <filename>: file contains floats\n");
  
  exit (1);
}


int 
main (int argc, char **argv)
{
  int			fd;
  char			*base;
  struct stat		stat;
  off_t			file_size;
  unsigned long 	nentries; 	// # of data entries
  int			c;
  bool			float_p = false;
  float			mu = 0;
  

  char *filename = 0;

  while ((c = getopt (argc, argv, "F:M:")) != EOF){
    switch (c){
    case 'F':
      filename = optarg;
      float_p = true;
      break;

    case 'M':
      mu = strtod (optarg, 0);
      break;
      
    default:
      usage (argv[0]);
    }
  }
    
  if (argc - optind != 0 || filename == 0)
    usage (argv[0]);


  fd = open (filename, O_RDONLY);
  if (fd < 0){
    perror (filename);
    exit (1);
  }

  if (fstat (fd, &stat) < 0){
    perror ("stat");
    exit (1);
  }

  file_size = stat.st_size;	// file size in bytes

  if ((base = (char *) mmap (0, file_size, PROT_READ, MAP_SHARED, fd, 0)) == MAP_FAILED){
    perror ("mmap");
    exit (1);
  }

  gr_mmse_fir_interpolator	interp;
  
  float *fdata = (float *) base;

  nentries = file_size / sizeof (float);
  nentries -= interp.ntaps ();

  for (unsigned long i = 0; i < nentries; i++){
    float r = interp.interpolate (&fdata[i], mu);
    fwrite (&r, sizeof (float), 1, stdout);
  }

  munmap (base, file_size);
  close (fd);
  exit (0);
}
