/*
   GNU xhippo: a GTK-based playlist manager.
   Copyright 1998, 1999, 2000 Adam Sampson <azz@gnu.org>

   Please report bugs to bug-xhippo@gnu.org.

   This file is part of GNU xhippo.

   GNU xhippo is free software; you can redistribute and/or modify it
   under the terms of that license as published by the Free Software
   Foundation; either version 2 of the License, or (at your option)
   any later version.
   
   GNU xhippo is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with GNU xhippo; see the file COPYING. If not, write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
   MA 02111-1307 USA, or see http://www.gnu.org/.

*/

#include "xhippo.h"

#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <signal.h>
#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <time.h>
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <sys/stat.h>
#include <glib.h>
#ifdef USEGNOME
#include <gnome.h>
#endif
#ifndef NOGETOPTLONG
#include <getopt.h>
#endif

/* If the user doesn't have snprintf, use sprintf instead. 
   This is a GNU cpp extension. */
#ifndef HAVE_SNPRINTF
#define snprintf(s, l, f, a...) sprintf(s, f, ##a)
#endif

gchar *types[MAXFILETYPES], *actions[MAXFILETYPES];
gint numtypes = 0;

#define STRBUFSIZE 1024
gchar strbuf[STRBUFSIZE];

GtkTooltips *tooltips;
GtkWidget *window, *restartbutton, *stopbutton, *buttons_box, 
  *all_box, *savebutton, *sortbutton, *addbutton,
  *filelist, *statusbar, *nextbutton, *pausebutton, *loadbutton,
  *minicheckbox, *randomcheckbox, *fileselector=NULL, *list_box,
  *scrollbar, *upbutton, *downbutton, *deletebutton, *popupmenu,
  *up_item, *down_item, *delete_item, *info_item;
GtkAdjustment *vadj;
GtkAccelGroup *accel_group;
GList *list;
FILE *f;
pid_t childpid;
guint contextid, 
  listcount = 0, 
  playing = 0, 
  paused = 0, 
  donotstopflag, 
  mode_play_on_start = 0, 
  mode_scroll_catchup = 0, 
  mode_left_scroll = 0,
  mode_must_play_all = 0, 
  mode_one_time = 0, 
  mode_show_pid = 0,
  mode_read_id3_tags = 0,
  mode_save_window_pos = 0,
  mode_start_mini = 0,
  mode_play_ordered = 0,
  mode_strip_extension = 0,
  mode_hide_player_output = 0,
  mode_sort_on_load = 0,
  mode_demangle_names = 0,
  mode_playlist_title = 0,
  mode_title_basename = 0,
  mode_no_check_files = 0,
  mode_write_playing = 0,
  attribute_mini = 0;
gchar *attribute_playlist_dir = NULL;
gint attribute_xpos = 100, 
  attribute_ypos = 100, 
  attribute_width = 0, 
  attribute_height = 0,
  last_played = -1,
  last_row_hit;

/* Information about a song in the playlist. */
struct _songinfo {
  gchar *name;             /* Its filename. */
  gchar *display_name;     /* The name shown in the list. */
  char played;             /* Whether it has been played. */
  char inforead;           /* Whether the information that comes 
			      from stat (mtime, size) has been read. */
  time_t mtime;            /* When the file was last modified. */
  off_t size;              /* The size of the file. */
  GtkWidget *info_window;  /* The information window; NULL if not up. */
};
typedef struct _songinfo songinfo;

/* Copy a songinfo structure. */
songinfo *copy_songinfo(songinfo *src) {
  songinfo *dest = (songinfo *)malloc(sizeof(songinfo));

  dest->name = strdup(src->name);
  dest->display_name = strdup(src->display_name);
  dest->played = src->played;
  dest->inforead = src->inforead;
  dest->mtime = src->mtime;
  dest->size = src->size;
  dest->info_window = src->info_window;

  return dest;
}

/* Destructor for songinfo, and therefore for the row as well. We can
   safely decrement the list count when this gets called. */
void destroy_songinfo(gpointer sinfo) {
  songinfo *s = (songinfo *)sinfo;
  free(s->name);
  free(s->display_name);
  if (s->info_window) gtk_widget_destroy(s->info_window);

  free(sinfo);
  --listcount;
}

/* Insert a row into the list based on a filled-in songinfo structure. 
   Call as insert_row(X, listcount); to append to the end. */
void insert_row(songinfo *sinfo, gint location) {
  gchar *row[1];
  
  row[0] = sinfo->display_name;
  gtk_clist_insert(GTK_CLIST(filelist), location, row);
  gtk_clist_set_row_data_full(GTK_CLIST(filelist), location, sinfo,
			      destroy_songinfo);
  listcount++;
}

/* Get a pointer to a song's songinfo. */
songinfo *get_songinfo(gint number) {
  return (songinfo *)gtk_clist_get_row_data(GTK_CLIST(filelist), number);
}

/* Move a row from one location to another within the list. */
void move_row(gint src, gint dest) {
  songinfo *new;

  gtk_clist_freeze(GTK_CLIST(filelist));

  new = copy_songinfo(get_songinfo(src));
  gtk_clist_remove(GTK_CLIST(filelist), src);
  insert_row(new, dest);
  gtk_clist_thaw(GTK_CLIST(filelist));
}

/* Type of file referred to. 0=unknown/nothing, 1=file, 2=dir. */
gint filetype(char *file) {
  struct stat st;

  if ((!file) || (!*file)) return 0;

  if (stat(file, &st) < 0) return 0;

  if (S_ISREG(st.st_mode)) return 1;
  if (S_ISDIR(st.st_mode)) return 2;
 
  return 0;
}

/* Show a message on the status bar */
void status(char *message) {
  gtk_statusbar_pop(GTK_STATUSBAR(statusbar), contextid);
  gtk_statusbar_push(GTK_STATUSBAR(statusbar), contextid, message);
}

/* Stop playing. */
void stop_playing() {
  if (!playing) return;
  if (!childpid) return;

  /* Wake up a paused player. */
  if (paused) {
    kill(childpid, SIGCONT);
    paused = 0;
  }

  playing = 0;
  kill(childpid, SIGTERM);

  /* Pause (to avoid the formation of zombies) until the child
     dies. childpid is reset in the SIGCHLD handler. */
  while (childpid) pause();
}

/* Toggle paused status. */
void pause_playing() {
  if (!playing) return;
  if (!childpid) return;

  if (paused) {
    kill(childpid, SIGCONT);
    paused = 0;
  } else {
    kill(childpid, SIGSTOP);
    paused = 1;
  }
}

#ifdef NOSTRCASECMP
/* The system doesn't provide strcasecmp, so we will instead. */
int strcasecmp(char *a, char *b) {
  char *la, *lb, *p;
  int r;

  la = strdup(a);
  lb = strdup(b);
  p = la; 
  while (*p) {
    *p = tolower(*p);
    p++;
  }
  p = lb;
  while (*p) {
    *p = tolower(*p);
    p++;
  }

  r = strcmp(la, lb);

  free(la);
  free(lb);

  return r;
}
#endif

/* Start playing song number num. Stop the old one first. */
void start_playing(gint num) {
  gchar *name, *type, *thisact;
  pid_t cpid;
  gchar args[16][STRBUFSIZE], *argptr[16];
  int argcount, i, j;
  gchar statbuf[STRBUFSIZE];

  if (num < 0 || num >= listcount) return;

  last_played = num;
  name = get_songinfo(num)->name;

  /* Stop the previous song. */
  if (donotstopflag) {
    donotstopflag = 0;
  } else {
    stop_playing();
  }

  /* name is the filename */
  type = name + strlen(name);
  while ((type > name) && (*type != '.') && (*type != '/')) {
    --type;
  }
  thisact = NULL;
  if (*type == '.') {
    ++type;
    for(i=0; i<numtypes; i++) {
      if (!strcasecmp(type, types[i])) thisact = actions[i];
    }
  }
  if (!thisact) {
    status(STRGUESSTYPE);
    return;
  }

  j=0;
  argcount = 0;
  argptr[0] = args[0];
  for(i=0; i<strlen(thisact); i++) {
    if (thisact[i] != ' ')
      args[argcount][j++] = thisact[i];
    else {
      args[argcount][j] = '\0';
      argcount++;
      argptr[argcount] = args[argcount];
      j=0;
    }
  }
  args[argcount][j] = '\0';
  argcount++;
  
  strcpy(args[argcount], name);
  argptr[argcount] = args[argcount];
  argcount++;

  argptr[argcount] = NULL;
  
  cpid = fork();
  if (cpid) {
    childpid = cpid;
    if (attribute_mini) {
      /* Put the name of the song playing in the status bar. */
      snprintf(statbuf, STRBUFSIZE - 1, "%s", get_songinfo(num)->display_name);
    } else {
      /* Put the "Playing with" string in the status bar. */
      if (mode_show_pid) {
	snprintf(statbuf, STRBUFSIZE - 1, STRPLAYINGWITHPID, argptr[0], childpid);
      } else {
	snprintf(statbuf, STRBUFSIZE - 1, STRPLAYINGWITH, argptr[0]);
      }
    }
    status(statbuf);
    playing = 1;
    paused = 0;
  } else {
    /* This is the child process */

    if (mode_hide_player_output) {
      int fd;

      fd = open("/dev/null", O_RDWR);
      dup2(fd, 1); /* stdout */
      dup2(fd, 2); /* stderr */
      close(fd);
    }

    execvp(argptr[0], argptr);
    fprintf(stderr, STRNOSTART, argptr[0]);
    execvp("echo", argptr);
    _exit(20);
  }

  if (mode_write_playing) {
    gchar csname[STRBUFSIZE];
    FILE *f;

    /* Write the title of the song that's currently playing to the
       ~/.xhippo/current_song file. */
    snprintf(csname, STRBUFSIZE - 1, "%s/.xhippo/current_song", 
	     getenv("HOME"));
    if ((f = fopen(csname, "w"))) {
      fprintf(f, "%s\n", get_songinfo(num)->display_name);
      fclose(f);
    }
  }
}

/* Pick and play a random song. */
void pick_next() {
  int newitem, first, i;

  if (listcount) {
    if (mode_play_ordered) {
      newitem = last_played + 1;
      if (newitem == listcount) {
	newitem = 0;
        if (mode_one_time) {
          playing = 0;
          return;
        }
      }
    } else {
      newitem = rand()%listcount; /* FIXME: This is a terrible randomiser. */
      if (mode_must_play_all) {
	first = newitem;
	while (get_songinfo(newitem)->played) {
	  newitem++;
	  if (newitem==listcount) newitem=0;
	  if (newitem==first) {
	    for (i=0; i<listcount; i++) get_songinfo(i)->played = 0;
            if (mode_one_time) {
              playing = 0;
              return; /* do this after setting played back to 0 */
            }
	  }
	}
      }
    }
    if (mode_scroll_catchup 
	&& !gtk_clist_row_is_visible(GTK_CLIST(filelist), newitem)) {
      gtk_clist_moveto(GTK_CLIST(filelist), newitem, 0, 0.5, 0.0);
    }
    gtk_clist_select_row(GTK_CLIST(filelist), newitem, 0);
    /* ... which sends a message anyway and gets handle_list_select called. */
  } else {
    status(STRNOITEMS);
  }
}

/* Routine to read the ID3 tag from an MP3 file, originally supplied
   by Craig Knudsen. */
static gchar *read_mp3_tag ( char *filename ) {
  int fd;
  gchar *trackname = NULL;
  gchar data[100];
  int ret;
  
  fd = open(filename, O_RDONLY);
  ret = lseek(fd, -128, SEEK_END);
  if (ret > 0) {
    ret = read(fd, data, 3);
    if (ret == 3) {
      data[3] = '\0';
      if (strcmp(data, "TAG") == 0) {
        ret = read(fd, data, 26); /* FIXME: Find the spec. */
        if (ret > 0) {
          trackname = strdup(data);
        }
      }
    }
  }
  close(fd);

  return trackname;
}

/* Fill in entries in a songinfo structure that need a stat first. Use
   the information in the provided struct stat if it's not NULL. */
void read_song_info(songinfo *sinfo, struct stat *st) {
  struct stat stnew;

  /* If we've already done this, then don't bother. */
  if (sinfo->inforead) return;

  if (!st) {
    st = &stnew;
    if (stat(sinfo->name, st) < 0) return;
  }
  sinfo->size = st->st_size;
  sinfo->mtime = st->st_mtime;
  sinfo->inforead = 1;
}

/* Called to add a new file to the playlist. */
void add_file(gchar *name) {
    gchar *basename, *dispname, *tag;
    songinfo *sinfo;
    struct stat st;

    if (!mode_no_check_files) {
      /* Note: this uses stat rather than filetype() because we need the
	 other information from the stat structure and this avoids
	 having to stat every file in the playlist twice. */
      if (stat(name, &st) < 0) {
	g_print(STRNOFIND, name);
	return;
      }

      if (!S_ISREG(st.st_mode)) {
	g_print(STRNOTFILE, name);
	return;
      }
    }

    basename = name + strlen(name);
    while((basename > name) && (*basename != '/')) --basename; 

    tag = NULL;
    if (mode_read_id3_tags) tag = read_mp3_tag(name);
    if (tag) {
      dispname = strdup(tag);
    } else {
      gchar *p;

      if (basename[0] == '/') ++basename;
      dispname = strdup(basename);

      if (mode_demangle_names) {
	/* Replace underscores with spaces. */
	while ((p = strchr(dispname, '_'))) *p = ' ';

	/* Replace '%20's with spaces. */
	while ((p = strstr(dispname, "%20"))) {
	  *p = ' ';
	  p += 2;
	  while (*(++p - 1)) *(p - 2) = *p;
	}
      }

      if (mode_strip_extension) {
	/* Strip off the extension. */
	p = dispname + strlen(dispname);
	while (p > dispname && *p != '.') --p;
	if (*p == '.') *p = '\0';
      }
    }

    /* Build a songinfo struct. */
    sinfo = malloc(sizeof(songinfo));
    sinfo->name = strdup(name);
    sinfo->display_name = dispname;
    sinfo->played = 0;
    sinfo->inforead = 0;
    sinfo->info_window = NULL;
    if (!mode_no_check_files) read_song_info(sinfo, &st);

    insert_row(sinfo, listcount);

    if (tag) free(tag);
}

/* Save the window position. */
void savewindowstate() {
  char filename[STRBUFSIZE];
  FILE *f;

  snprintf(filename, STRBUFSIZE - 1, "%s/.xhippo/winstate", getenv("HOME"));
  if ((f = fopen(filename, "w"))) {
    fprintf(f, "%d\n%d\n%d\n%d\n%d\n", attribute_xpos, attribute_ypos, 
	    attribute_mini, attribute_width, attribute_height);
    fclose(f);
  } else {
    g_print(STRNOWINSTATE);
  }
}

/* Read the window position. */
void readwindowstate() {
  char filename[STRBUFSIZE];
  char line[STRBUFSIZE];
  FILE *f;

  snprintf(filename, STRBUFSIZE - 1, "%s/.xhippo/winstate", getenv("HOME"));
  if ((f = fopen(filename, "r"))) {
    fgets(line, STRBUFSIZE, f);
    attribute_xpos = atoi(line);
    fgets(line, STRBUFSIZE, f);
    attribute_ypos = atoi(line);
    fgets(line, STRBUFSIZE, f);
    attribute_mini = atoi(line);
    fgets(line, STRBUFSIZE, f);
    attribute_width = atoi(line);
    fgets(line, STRBUFSIZE, f);
    attribute_height = atoi(line);
    fclose(f);
  } /* else fail silently, it happens all the time */
}

/* Clear the playlist. */
void clearplaylist() {
  gtk_clist_clear(GTK_CLIST(filelist));
  last_played = -1;
}

/* Sorting function for playlist entries by display name. */
int sort_compare_name(const void *a, const void *b) {
  return strcmp((*((songinfo **)a))->display_name,
		(*((songinfo **)b))->display_name);
}

/* Sorting function for playlist entries by mtime. */
int sort_compare_mtime(const void *a, const void *b) {
  return (*((songinfo **)a))->mtime - (*((songinfo **)b))->mtime;
}

/* Sort the playlist by the display name using the specified function,
   and getting the song info first if required. */
void sortplaylist(int (*func)(const void *a, const void *b),
		  char needinfo) {
  songinfo **sl;
  gint i, lc = listcount;

  if (needinfo) {
    for (i=0; i<lc; i++) read_song_info(get_songinfo(i), NULL);
  }

  /* Copy the songinfo structs from the playlist into a new list,
     and then sort that. */
  sl = (songinfo **)malloc(lc * sizeof(songinfo *));
  for (i=0; i<lc; i++) sl[i] = copy_songinfo(get_songinfo(i));
  qsort(sl, lc, sizeof(songinfo *), func);
  
  gtk_clist_freeze(GTK_CLIST(filelist));
  clearplaylist();
  for (i=0; i<lc; i++) insert_row(sl[i], i);
  gtk_clist_thaw(GTK_CLIST(filelist));

  free(sl);
}

/* Read a playlist. */
void readplaylist(gchar *name) {
  FILE *f;

  /* Prevent the list from being updated while we load the list. */
  gtk_clist_freeze(GTK_CLIST(filelist));

  /* Check that the specified name is actually a file. */
  if (filetype(name) == 1) {

    if (!(f = fopen(name,"r"))) {
      g_print(STRNOPLAYLIST, name);
    } else {
      char line[STRBUFSIZE], fullname[STRBUFSIZE], dirname[STRBUFSIZE], 
	*slash, *filename;

      /* Work out the dirname if we've got one. */
      slash = strrchr(name, '/');
      if (slash) {
	*slash = '\0';
	snprintf(dirname, STRBUFSIZE - 1, "%s/", name);
	*slash = '/';
      } else {
	*dirname = '\0';
      }

      while (fgets(line, STRBUFSIZE, f)) {
	/* Remove \n if it's present. It will be. :) */
	if (line[strlen(line)-1]=='\n') line[strlen(line)-1]='\0';

	/* Work out the full filename including the extra path. */
	filename = (*line == '!') ? line + 1 : line;
	if (*filename != '/') {
	  snprintf(fullname, STRBUFSIZE - 1, "%s%s", dirname, filename);
	} else {
	  snprintf(fullname, STRBUFSIZE - 1, "%s", filename);
	}

	/* Allow inclusion of a playlist from a playlist. */
	if (*line == '!') {
	  readplaylist(fullname);
	} else {
	  add_file(fullname);
	}
      }

      fclose(f);
    }
  }

  if (mode_sort_on_load) sortplaylist(sort_compare_name, FALSE);

  /* Set the window title if required. We do this at the end of this
     function so that the name of the highest nested playlist is
     used. */
  if (mode_playlist_title) {
    gchar *s, *t, *prefix = "xhippo " VERSION ": ";

    if (mode_title_basename) {
      s = name + strlen(name);
      while (s > name && *s != '/') s--;
      if (*s == '/') s++;
    } else {
      s = name;
    }

    t = malloc(strlen(s) + strlen(prefix) + 1);
    sprintf(t, "%s%s", prefix, s);
    gtk_window_set_title(GTK_WINDOW(window), t);
    free(t);
  }
  
  gtk_clist_thaw(GTK_CLIST(filelist));
}

/* Read a config file, named in name. Pass 0 is before the 
   GUI is shown, pass 1 is after. */
void readconfig(gchar *name, gint pass) {
  FILE *f;
  gchar configopt[STRBUFSIZE], opta[STRBUFSIZE], optb[STRBUFSIZE], *c, *s;

/* Ack bletch. However, I can't think of a more efficient way.  Today
   19991002 at work I was described as an "obfuscation junkie" (which,
   in retrospect, was quite impressive, given that I'd only been
   working there two weeks!). I can't really argue with that, given
   the code below!

   Actually, there is a more efficient way; use strtok. However,
   that's deprecated according to the glibc docs. */

#define SCANS(x) c = (x);\
                 while(*s && ((*s) != ':')) *c++ = *s++;\
                 *c = '\0';\
                 if (*s) s++;
#define SAME(a,b) (!strcmp((a),(b)))

  if ((f = fopen(name, "r"))) {
    while (fgets(strbuf, STRBUFSIZE - 1, f)) {

      /* Remove \n. It will have one, but we check anyway. */
      if (strbuf[strlen(strbuf)-1]=='\n') strbuf[strlen(strbuf)-1]='\0';

      if ((*strbuf) && ((*strbuf) != '#')) {
	s = strbuf;
	SCANS(configopt);
	SCANS(opta);
	SCANS(optb);

	if (pass == 0) {
	  if (SAME(configopt, "type") && numtypes < MAXFILETYPES) {
	    types[numtypes] = strdup(opta);
	    actions[numtypes] = strdup(optb);
	    numtypes++;
	  }
	  if (SAME(configopt, "autostart")) mode_play_on_start = atoi(opta);
	  if (SAME(configopt, "scroll")) mode_scroll_catchup = atoi(opta);
	  if (SAME(configopt, "leftscroll")) mode_left_scroll = atoi(opta);
	  if (SAME(configopt, "mustplayall")) mode_must_play_all = atoi(opta);
	  if (SAME(configopt, "onetime")) mode_one_time = atoi(opta);
	  if (SAME(configopt, "showpid")) mode_show_pid = atoi(opta);
	  if (SAME(configopt, "readid3")) mode_read_id3_tags = atoi(opta);
	  if (SAME(configopt, "savewinstate")) mode_save_window_pos = atoi(opta);
	  if (SAME(configopt, "startmini")) mode_start_mini = atoi(opta);
	  if (SAME(configopt, "ordered")) mode_play_ordered = atoi(opta);
	  if (SAME(configopt, "stripextension")) mode_strip_extension = atoi(opta);
	  if (SAME(configopt, "hideplayeroutput")) mode_hide_player_output = atoi(opta);
	  if (SAME(configopt, "sortonload")) mode_sort_on_load = atoi(opta);
	  if (SAME(configopt, "playlistdir")) attribute_playlist_dir = strdup(opta);
	  if (SAME(configopt, "demanglenames")) mode_demangle_names = atoi(opta);
	  if (SAME(configopt, "playlisttitle")) mode_playlist_title = atoi(opta);
	  if (SAME(configopt, "titlebasename")) mode_title_basename = atoi(opta);
	  if (SAME(configopt, "nocheckfiles")) mode_no_check_files = atoi(opta);
	  if (SAME(configopt, "writeplaying")) mode_write_playing = atoi(opta);
	} else {
	  if (SAME(configopt, "load")) readplaylist(opta);
	  if (SAME(configopt, "exec")) system(opta);
	}
      }
    }

    fclose(f);
  }
}

/* Read config files. */
void read_config_files(gint pass) {
  gchar rcname[STRBUFSIZE];

  readconfig(SYSTEMXHIPPOCONFIG, pass);
  snprintf(rcname, STRBUFSIZE - 1, "%s/.xhippo/config", getenv("HOME"));
  readconfig(rcname, pass);
  readconfig("xhippo.config", pass);
}

/* Handle selections from the list. */
void handle_list_select(GtkWidget *gtklist, gint row, gint column,
			GdkEventButton *event, gpointer data) {
  get_songinfo(row)->played = 1;
  start_playing(row);
}

/* Handle right clicks on the list. */
void handle_list_click(GtkWidget *gtklist, GdkEventButton *event, 
		       gpointer data) {
  gint state = 1;

  /* Return if this isn't an RMB click. */
  if (event->button != 3) return;
  
  /* Translate the coordinates into row and column numbers. */
  last_row_hit = -1;
  gtk_clist_get_selection_info(GTK_CLIST(filelist),
			       event->x, event->y,
			       &last_row_hit, NULL);

  /* If the click isn't on the list, disable the menu items. */
  if (last_row_hit < 0) state = 0;

  gtk_widget_set_sensitive(GTK_WIDGET(info_item), state);
  gtk_widget_set_sensitive(GTK_WIDGET(up_item), state);
  gtk_widget_set_sensitive(GTK_WIDGET(down_item), state);
  gtk_widget_set_sensitive(GTK_WIDGET(delete_item), state);

  gtk_menu_popup(GTK_MENU(popupmenu), NULL, NULL, NULL, NULL, 
		 event->button, event->time);
}

/* Handle the "Cancel" button in the file selector. */
void handle_fileselector_cancel(GtkWidget *widget, gpointer data) {
  gtk_widget_destroy(fileselector);
  fileselector = NULL;
}

/* Handle the "OK" button in the Load Playlist file selector. */
void handle_load_ok(GtkWidget *widget, gpointer data) {
  clearplaylist();
  readplaylist(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileselector)));
  gtk_widget_destroy(fileselector);
  fileselector = NULL;
}

/* Handle the "OK" button in the Save Playlist file selector. */
void handle_save_ok(GtkWidget *widget, gpointer data) {
  FILE *f = fopen(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileselector)), "w");
  int i;

  if (!f) {
    status("Unable to save playlist.");
    return;
  }

  for (i=0; i<listcount; i++) fprintf(f, "%s\n", get_songinfo(i)->name);
  fclose(f);

  gtk_widget_destroy(fileselector);
  fileselector = NULL;
}

/* Handle the "OK" button in the Add file selector. */
void handle_add_ok(GtkWidget *widget, gpointer data) {
  add_file(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileselector)));
  gtk_widget_destroy(fileselector);
  fileselector = NULL;
}

/* Called when "Next" is pressed. */
void handle_next(GtkWidget *widget, gpointer data) {
  pick_next();
}

/* Called when attribute_mini has changed, to alter the window size. */
void set_mini() {
  if (attribute_mini) {
    gtk_widget_hide(list_box);
    gtk_widget_set_usize(window, -1, -1);
  } else {
    gtk_widget_show(list_box);
    gtk_widget_set_usize(window, attribute_width, attribute_height);
  }
}

/* Called when the "Mini" checkbox is toggled. */
void handle_minitoggle(GtkWidget *widget, gpointer data) {
  attribute_mini = 0;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(minicheckbox))) {
    attribute_mini = 1;
  }
  set_mini();
}

/* Called when the "Random" checkbox is toggled. */
void handle_randomtoggle(GtkWidget *widget, gpointer data) {
  mode_play_ordered = 1;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(randomcheckbox))) {
    mode_play_ordered = 0;
  }
}

/* Called when "Restart" is pressed. */
void handle_restart(GtkWidget *widget, gpointer data) {
  start_playing(last_played);
}

/* Called when "Stop" is pressed. */
void handle_stop(GtkWidget *widget, gpointer data) {
  stop_playing();
  status(STRSTOPPED);
}

/* Called when "Pause" is pressed. */
void handle_pause(GtkWidget *widget, gpointer data) {
  pause_playing();
  if (playing) {
    if (paused) {
      status(STRPAUSED);
    } else {
      status(STRRESUMEPLAY);
    }
  }
}

/* Handle the "delete_event" event on an info window. data points to
   the songinfo structure. */
gint handle_info_delete_event(GtkWidget *widget, GdkEvent *event, 
			      gpointer data) {
  return(FALSE); /* Close the window, as FALSE */
}

/* Handle the "destroy" event on an info window. data points to the
   songinfo structure. */
void handle_info_destroy(GtkWidget *widget, gpointer data) {
  /* Clear the pointer to the window. */
  ((songinfo *)data)->info_window = NULL;
}

/* Add a row to the info window. */
void add_info_window_row(GtkWidget *table, gint pos,
			 gchar *text1, gchar *text2) {
  GtkWidget *label, *text;

  label = gtk_label_new(text1);
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
  gtk_table_attach(GTK_TABLE(table), label,
		   0, 1, pos, pos + 1,
		   GTK_FILL, GTK_FILL, 4, 0);
  gtk_widget_show(label);

  text = gtk_label_new(text2);
  gtk_label_set_justify(GTK_LABEL(text), GTK_JUSTIFY_LEFT);
  gtk_table_attach(GTK_TABLE(table), text,
		   1, 2, pos, pos + 1,
		   GTK_EXPAND, GTK_EXPAND, 4, 0);
  gtk_widget_show(text);
}

/* Handle the items in the popup menu. */
void handle_menu_info(GtkWidget *widget, gpointer data) {
  songinfo *sinfo;
  GtkWidget *table;
  gchar buf[STRBUFSIZE], *p;

  sinfo = get_songinfo(last_row_hit);

  /* If the window already exists, don't bother. */
  if (sinfo->info_window) return;

  /* Make sure we've got the information we need. */
  read_song_info(sinfo, NULL);

  sinfo->info_window = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_title(GTK_WINDOW(sinfo->info_window), STRINFOTITLE);
  gtk_signal_connect(GTK_OBJECT(sinfo->info_window), "delete_event",
		     GTK_SIGNAL_FUNC(handle_info_delete_event), sinfo);
  gtk_signal_connect(GTK_OBJECT(sinfo->info_window), "destroy",
		     GTK_SIGNAL_FUNC(handle_info_destroy), sinfo);

  table = gtk_table_new(4, 2, FALSE);
  gtk_container_add(GTK_CONTAINER(sinfo->info_window), table);

  add_info_window_row(table, 0, STRINFONAME, sinfo->display_name);
  add_info_window_row(table, 1, STRINFOFILENAME, sinfo->name);
  snprintf(buf, STRBUFSIZE - 1, "%s", ctime(&sinfo->mtime));
  /* ctime appends a carriage return; strip it off. */
  while ((p = strrchr(buf, '\n'))) *p = '\0';
  add_info_window_row(table, 2, STRINFOLASTMODIFIED, buf);
  snprintf(buf, STRBUFSIZE - 1, "%lu", sinfo->size);
  add_info_window_row(table, 3, STRINFOFILESIZE, buf);
  gtk_widget_show(table);
  gtk_widget_show(sinfo->info_window);
}

void handle_menu_up(GtkWidget *widget, gpointer data) {
  if (last_row_hit > 0) move_row(last_row_hit, last_row_hit - 1);
}

void handle_menu_down(GtkWidget *widget, gpointer data) {
  if (last_row_hit < listcount - 1) move_row(last_row_hit, last_row_hit + 1);
}

void handle_menu_delete(GtkWidget *widget, gpointer data) {
  gtk_clist_remove(GTK_CLIST(filelist), last_row_hit);
}

/* Get the directory in which the user wants to store playlists. */
void get_playlist_dir(gchar *buf, int buflen) {
  /* If attribute_playlist_dir is set, use that. */
  if (attribute_playlist_dir) {
    snprintf(buf, buflen - 1, "%s", attribute_playlist_dir);
    return;
  }
  
  /* If $HOME/.xhippo/playlists is a directory, use that. */
  snprintf(buf, buflen - 1, "%s/.xhippo/playlists/", getenv("HOME"));
  if (filetype(buf) == 2) return;

  /* Else use nothing. */
  strcpy(buf, "");
}

/* Called when "Load" is pressed. */
void handle_menu_load(GtkWidget *widget, gpointer data) {
  gchar name[STRBUFSIZE];

  get_playlist_dir(name, STRBUFSIZE);
  if (fileselector) return;
  fileselector = gtk_file_selection_new(STRPICKPLAYLIST);
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileselector), name);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->ok_button),
		     "clicked", (GtkSignalFunc) handle_load_ok,
		     fileselector);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->cancel_button),
		     "clicked", (GtkSignalFunc) handle_fileselector_cancel,
		     fileselector);
  gtk_widget_show(fileselector);
}

void handle_menu_save(GtkWidget *widget, gpointer data) {
  gchar name[STRBUFSIZE];

  get_playlist_dir(name, STRBUFSIZE);
  if (fileselector) return;
  fileselector = gtk_file_selection_new(STRSAVEPLAYLIST);
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileselector), name);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->ok_button),
		     "clicked", (GtkSignalFunc) handle_save_ok,
		     fileselector);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->cancel_button),
		     "clicked", (GtkSignalFunc) handle_fileselector_cancel,
		     fileselector);
  gtk_widget_show(fileselector);
}

void handle_menu_add(GtkWidget *widget, gpointer data) {
  if (fileselector) return;
  fileselector = gtk_file_selection_new(STRSAVEPLAYLIST);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->ok_button),
		     "clicked", (GtkSignalFunc) handle_add_ok,
		     fileselector);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->cancel_button),
		     "clicked", (GtkSignalFunc) handle_fileselector_cancel,
		     fileselector);
  gtk_widget_show(fileselector);
}

/* Handlers for the sort menu items. */
void handle_menu_sort_name(GtkWidget *widget, gpointer data) {
  sortplaylist(sort_compare_name, FALSE);
}
void handle_menu_sort_mtime(GtkWidget *widget, gpointer data) {
  sortplaylist(sort_compare_mtime, TRUE);
}

void handle_menu_clear() {
  clearplaylist();
}

/* Handle the "delete_event" event. */
gint handle_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data) {
  /* Save the window position, if necessary. */
  if (mode_save_window_pos) savewindowstate();

  return(FALSE); /* Close the window, as FALSE */
}

/* Handle the "destroy" event. */
void handle_destroy(GtkWidget *widget, gpointer data) {
  /* Stop the player. */
  stop_playing();

  /* Then quit the interface. */
  gtk_main_quit();
}

/* Handle "configure" events (moves & resizes) of the main window. */
void handle_configure(GtkWidget *widget, gpointer data) {
  /* Don't bother if we're making the window small */
  if (attribute_mini) return;
  gdk_window_get_root_origin(widget->window, 
			     &attribute_xpos, &attribute_ypos);
  gdk_window_get_geometry(widget->window, NULL, NULL,
			  &attribute_width, &attribute_height,
			  NULL);
}

/* A timeout function to be called after we've had a SIGCHLD. This
   does stuff that we can't do in the signal handler. */
gint sigchld_timeout(gpointer data) {
  donotstopflag = 1;
  pick_next();
  if (!playing) { /* reached end of one time list? */
    donotstopflag = 1;
    status(STRDONE);
  }
  return FALSE; /* don't do the timeout again */
}

/* Called when the child process quits. */
void handle_sigchld() {
  /* Wait for the correct child to die. */
  if (wait(NULL) != childpid) return;

  childpid = 0;

  /* If we're in play mode, get the timeout called next time we hit
     the main loop. */
  if (playing) gtk_timeout_add(0, sigchld_timeout, NULL);
}

#ifdef USEGNOMEMENUS

static GnomeUIInfo file_menu[] = {
  { GNOME_APP_UI_ITEM, "Load playlist...", NULL, handle_menu_load, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 'l', 0, NULL },
  { GNOME_APP_UI_ITEM, "Restart", NULL, handle_restart, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 'r', 0, NULL },
  { GNOME_APP_UI_ITEM, "Stop", NULL, handle_stop, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 's', 0, NULL },
  { GNOME_APP_UI_ITEM, "Random", NULL, handle_next, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 'n', 0, NULL },
  { GNOME_APP_UI_ITEM, "Quit", NULL, handle_destroy, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXIT, 'Q',
    GDK_SHIFT_MASK, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

void handle_about(GtkWidget *widget, void *data) {
  GtkWidget *about;
  static const gchar *authors[] = { "Adam Sampson", NULL };
  
  about = gnome_about_new("GNU Xhippo", VERSION, "(c)1998-2000 Adam Sampson",
			   authors, "Xhippo is a generic music player"
			   " for UNIX systems.", NULL);
  gtk_widget_show(about);
}

static GnomeUIInfo help_menu[] = {
  { GNOME_APP_UI_ITEM, "About...", NULL, handle_about, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT, 0, 0, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

static GnomeUIInfo main_menu[] = {
  { GNOME_APP_UI_SUBTREE, ("File"), NULL, file_menu, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_SUBTREE, ("Help"), NULL, help_menu, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

#endif

/* Handle files dropped on our window. */
static void handle_dnd_drop(GtkWidget *widget, GdkDragContext *context, gint x, gint y,
		    GtkSelectionData *data, guint info,
		    guint time)
{
  if (data->data) {
#ifdef USEGNOME
    /* Use the GNOME functions to parse the list. */
    GList *urls = gnome_uri_list_extract_uris(data->data);
    gchar *s;

    while (urls) {
      s = (gchar *)urls->data;
      if (strlen(s) > 5 && !strncmp(s, "file:", 5)) add_file(s+5);
      urls = urls->next;
    }
    gnome_uri_list_free_strings(urls);
#else
    /* The format of data->data is a list of URIs, seperated by
       and terminating in \r\n. We're only interested in the file:
       URIs. */
    gchar *s = strdup(data->data), *p, *q;

    p = s;
    while ((q = strchr(p, '\r'))) {
      *q = '\0';
      if (!(*++q == '\n')) break;
      *q = '\0';
      if (strlen(p) > 5 && !strncmp(p, "file:", 5)) add_file(p+5);
      p = q + 1;
    }
    
    free(s);
#endif
  }   
}

static GtkTargetEntry target_table[] = {
  { "text/uri-list", 0, 0 },
};

void print_version() {
  g_print("GNU xhippo %s\n", VERSION);
  g_print("Copyright (c) 1998-2000 Adam Sampson
GNU xhippo comes with NO WARRANTY, to the extent permitted by law.
You may redistribute copies of GNU xhippo under the terms of the
GNU General Public License. For more information about these matters,
see the file named COPYING.\n");
}

GtkWidget *add_button(gchar *label, gchar *tip, GtkSignalFunc callback,
		       GtkWidget *container, gchar *name, gint accel) {
  GtkWidget *new;

  /* Create and bind the Load button. */
  new = gtk_button_new_with_label(label);
  gtk_signal_connect(GTK_OBJECT(new), "clicked",
		     GTK_SIGNAL_FUNC(callback), NULL);
  gtk_box_pack_start(GTK_BOX(container), new, TRUE, TRUE, 0);
  gtk_tooltips_set_tip(tooltips, new, tip, NULL);
  if (accel) {
    gtk_widget_add_accelerator(new, "clicked", accel_group,
			       accel, 0, GTK_ACCEL_VISIBLE);
  }
  gtk_widget_set_name(new, name);
  gtk_widget_show(new);

  return new;
}

GtkWidget *add_menu_item(gchar *label, GtkSignalFunc callback, gint accel) {
  GtkWidget *item;

  item = gtk_menu_item_new_with_label(label);
  gtk_menu_append(GTK_MENU(popupmenu), item);
  gtk_signal_connect_object(GTK_OBJECT(item), "activate",
			    GTK_SIGNAL_FUNC(callback), NULL);
  if (accel) {
    gtk_widget_add_accelerator(item, "activate", accel_group,
                               accel, 0, GTK_ACCEL_VISIBLE);
  }
  gtk_widget_show(item);

  return item;
}

void add_menu_separator() {
  GtkWidget *item;

  item = gtk_menu_item_new();
  gtk_menu_append(GTK_MENU(popupmenu), item);
  gtk_widget_show(item);
}

int main(int argc, char **argv) {
  struct sigaction act;
  gchar rcname[STRBUFSIZE], wintitlestring[STRBUFSIZE];
  int c, j; /* for getopt */

  /* Check that HOME is available. */
  if (!getenv("HOME")) {
    fprintf(stderr, STRSETHOME);
    exit(20);
  }

  /* Seed RNG. */
  srand(time(0));

  /* Handle arguments. */
#ifdef USEGNOME
  gnome_init("xhippo", VERSION, argc, argv);
#else
  gtk_init(&argc, &argv);
#endif

  /* Read the GtkRC file. */
  snprintf(rcname, STRBUFSIZE - 1, "%s/.xhippo/gtkrc", getenv("HOME"));
  gtk_rc_parse(rcname);
  gtk_rc_parse("xhippo.gtkrc");

  /* Set our response to SIGCHLD. */
  act.sa_handler = handle_sigchld;
  sigemptyset(&act.sa_mask);
  act.sa_flags = SA_NOCLDSTOP;
  sigaction(SIGCHLD, &act, NULL);

  /* Create the acceleration group. */
  accel_group = gtk_accel_group_new();

  /* Create the tooltips object. */
  tooltips = gtk_tooltips_new();

  /* Create a window. */
  snprintf(wintitlestring, STRBUFSIZE - 1, "xhippo %s", VERSION);
#ifdef USEGNOME
  /* In GNOME mode, window is actually a GNOME app, but the
     differences in use are very little. */
  window = gnome_app_new("xhippo", wintitlestring);
#ifdef USEGNOMEMENUS
  gnome_app_create_menus_with_data(GNOME_APP(window), main_menu, window);
#endif
  gtk_widget_realize(window);
#else
  /* Standard GTK initialisation. */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), wintitlestring);
#endif
  gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, TRUE);
  gtk_signal_connect(GTK_OBJECT(window), "delete_event",
		     GTK_SIGNAL_FUNC(handle_delete_event), NULL);
  gtk_signal_connect(GTK_OBJECT(window), "destroy",
		     GTK_SIGNAL_FUNC(handle_destroy), NULL);
  gtk_signal_connect(GTK_OBJECT(window), "configure_event",
		     GTK_SIGNAL_FUNC(handle_configure), NULL);

  /* Listen to drag-and-drop. */
  gtk_signal_connect(GTK_OBJECT(window), "drag_data_received",
		     GTK_SIGNAL_FUNC(handle_dnd_drop), NULL);
  gtk_drag_dest_set(window,
		    GTK_DEST_DEFAULT_MOTION |
		    GTK_DEST_DEFAULT_HIGHLIGHT |
		    GTK_DEST_DEFAULT_DROP,
		    target_table, 1,
		    GDK_ACTION_COPY | GDK_ACTION_MOVE);

  /* Create the container for everything. */
  all_box = gtk_vbox_new(FALSE, 0);
#ifdef USEGNOME
  gnome_app_set_contents(GNOME_APP(window), all_box);
#else
  gtk_container_add(GTK_CONTAINER(window), all_box);
#endif
  gtk_widget_set_name(all_box, "all_box");

  /* Create the status bar. */
  statusbar = gtk_statusbar_new();
  gtk_box_pack_start(GTK_BOX(all_box), statusbar, FALSE, FALSE, 0);
  contextid = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar), "MyContext");
  gtk_statusbar_push(GTK_STATUSBAR(statusbar), contextid, STRWELCOME);
  gtk_widget_set_name(statusbar, "statusbar");

  gtk_widget_show(statusbar);

  /* Create the upper buttons container. */
  buttons_box = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(all_box), buttons_box, FALSE, FALSE, 0);
  gtk_widget_set_name(buttons_box, "buttons_box");

  /* Create the buttons and checkboxes in the upper bar. */
  restartbutton = add_button(STRRESTART, STRRESTARTTIP, handle_restart,
			      buttons_box, "restartbutton", 0);

  stopbutton = add_button(STRSTOP, STRSTOPTIP, handle_stop,
			   buttons_box, "stopbutton", GDK_KP_Divide);

  pausebutton = add_button(STRPAUSE, STRPAUSETIP, handle_pause,
			    buttons_box, "pausebutton", GDK_KP_Add);

  nextbutton = add_button(STRNEXT, STRNEXTTIP, handle_next,
			   buttons_box, "nextbutton", GDK_KP_Subtract);
  gtk_widget_add_accelerator(nextbutton, "clicked", accel_group,
			     GDK_KP_Multiply, 0, GTK_ACCEL_VISIBLE);
  
  minicheckbox = gtk_check_button_new_with_label(STRMINI);
  gtk_signal_connect(GTK_OBJECT(minicheckbox), "toggled",
		     GTK_SIGNAL_FUNC(handle_minitoggle), NULL);
  gtk_widget_set_name(minicheckbox, "minicheckbox");
  gtk_box_pack_start(GTK_BOX(buttons_box), minicheckbox, FALSE, FALSE, 0);
  gtk_widget_show(minicheckbox);

  randomcheckbox = gtk_check_button_new_with_label(STRRANDOM);
  gtk_signal_connect(GTK_OBJECT(randomcheckbox), "toggled",
		     GTK_SIGNAL_FUNC(handle_randomtoggle), NULL);
  gtk_widget_set_name(randomcheckbox, "randomcheckbox");
  gtk_box_pack_start(GTK_BOX(buttons_box), randomcheckbox, FALSE, FALSE, 0);
  gtk_widget_show(randomcheckbox);

  /* Show the buttons boxes. */
  gtk_widget_show(buttons_box);

  /* Create an hbox into which the list and its scrollbar fit. */
  list_box = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(all_box), list_box, TRUE, TRUE, 0);
  gtk_widget_set_name(list_box, "list_box");

  /* Create the popup menu. */
  popupmenu = gtk_menu_new();
  info_item = add_menu_item(STRINFO, handle_menu_info, 0);
  up_item = add_menu_item(STRUP, handle_menu_up, 0);
  down_item = add_menu_item(STRDOWN, handle_menu_down, 0);
  delete_item = add_menu_item(STRDELETE, handle_menu_delete, 0);
  add_menu_separator();
  add_menu_item(STRADD, handle_menu_add, GDK_a);
  add_menu_separator();
  add_menu_item(STRLOAD, handle_menu_load, GDK_l);
  add_menu_item(STRSAVE, handle_menu_save, GDK_v);
  add_menu_item(STRSORTNAME, handle_menu_sort_name, GDK_o);
  add_menu_item(STRSORTMTIME, handle_menu_sort_mtime, GDK_t);
  add_menu_item(STRCLEAR, handle_menu_clear, GDK_c);
  
  /* Read the config file for the first time. */
  read_config_files(0);

  /* Parse arguments. */
  while (1) {
#ifndef NOGETOPTLONG
    static struct option long_options[] = {
      {"autostart",0,0,0}, /* -a */
      {"scroll",0,0,0}, /* -s */
      {"leftscroll",0,0,0}, /* -l */
      {"mustplayall",0,0,0}, /* -m */
      {"showpid",0,0,0}, /* -p */
      {"readid3",0,0,0}, /* -i */
      {"savewinstate",0,0,0}, /* -w */
      {"startmini",0,0,0}, /* -t */
      {"version",0,0,0}, /* -V */
      {"help",0,0,0}, /* -h */
      {"ordered",0,0,0}, /* -o */
      {"stripextension",0,0,0}, /* -S */
      {"hideplayeroutput",0,0,0}, /* -q */
      {"sortonload",0,0,0}, /* -O */
      {"onetime",0,0,0}, /* -1 */
      {"playlisttitle",0,0,0}, /* -T */
      {"titlebasename",0,0,0}, /* -b */
      {"nocheckfiles",0,0,0}, /* -c */
      {"writeplaying",0,0,0}, /* -W */
      {0,0,0,0}
    };
    j = 0;
    c = getopt_long(argc, argv, "aslmpiwtVhoqOd1TbcW", long_options, &j);
#else
    c = getopt(argc, argv, "aslmpiwtVhoqOd1TbcW");
#endif
    if (c==-1) break; /* out of the while; done with the args */

#ifndef NOGETOPTLONG
    if (c==0) switch(j) {
    case 0: c = 'a'; break;
    case 1: c = 's'; break;
    case 2: c = 'l'; break;
    case 3: c = 'm'; break;
    case 4: c = 'p'; break;
    case 5: c = 'i'; break;
    case 6: c = 'w'; break;
    case 7: c = 't'; break;
    case 8: c = 'V'; break;
    case 9: c = 'h'; break;
    case 10: c = 'o'; break;
    case 11: c = 'S'; break;
    case 12: c = 'q'; break;
    case 13: c = 'O'; break;
    case 14: c = 'd'; break;
    case 15: c = '1'; break;
    case 16: c = 'T'; break;
    case 17: c = 'b'; break;
    case 18: c = 'c'; break;
    case 19: c = 'W'; break;
    };
#endif

    switch(c) {
    case 'a': /* -a: automatic play on start */
      mode_play_on_start = 1;
      break;
    case 's': /* -s: scroll catchup */
      mode_scroll_catchup = 1;
      break;
    case 'l': /* -l: left scrollbar */
      mode_left_scroll = 1;
      break;
    case 'm': /* -m: must play all */
      mode_must_play_all = 1;
      break;
    case 'p': /* -p: show PIDs */
      mode_show_pid = 1;
      break;
    case 'i': /* -i: read ID3 tags */
      mode_read_id3_tags = 1;
      break;
    case 'w': /* -w: save window position */
      mode_save_window_pos = 1;
      break;
    case 't': /* -t: start minified */
      mode_start_mini = 1;
      break;
    case 'S': /* -S: strip extension */
      mode_strip_extension = 1;
      break;
    case 'o': /* -o: play ordered */
      mode_play_ordered = 1;
      break;
    case 'q': /* -q: hide player output */
      mode_hide_player_output = 1;
      break;
    case 'O': /* -O: sort on load */
      mode_sort_on_load = 1;
      break;
    case 'd': /* -d: demangle names */
      mode_demangle_names = 1;
      break;
    case 'T': /* -T: playlist title */
      mode_playlist_title = 1;
      break;
    case 'b': /* -b: playlist title basename */
      mode_title_basename = 1;
      break;
    case 'c': /* -c: don't bother stat()ing files on startup */
      mode_no_check_files = 1;
      break;
    case 'W': /* -w: write playing song to a file */
      mode_write_playing = 1;
      break;
    case '1': /* -1: one time */
      mode_one_time = 1;
      break;
    case 'V':
      print_version();
      exit(0);
      break;
    case 'h':
      print_version();
      g_print("\nUsage: xhippo [options] [playlist [playlist ...]]
-a  --autostart    Start playing automatically once the playlists have
                   been loaded.
-s  --scroll       Automatically scroll the list when a random song
                   is picked.
-l  --leftscroll   Place scrollbar on the left side of the window.
-m  --mustplayall  When picking a random song, pick one that has not
                   already been played if possible.
-1  --onetime      Stop once the entire list has been played.
-p  --showpid      Show the PIDs of player processes in the status bar.
-i  --readid3      Attempt to read ID3 tags from files and use the titles
                   stored in the list.
-S  --stripextension
                   Strip extensions from files shown in the playlist.
-d  --demanglenames
                   Replace underscores and %%20s in names with spaces.
-w  --savewinstate Save the window position and state upon exit to
                   $HOME/.xhippo/winstate.
-T  --playlisttitle
                   Make the window title the name of the loaded playlist.
-b  --titlebasename
                   If -T is set, use the basename of the playlist.
-t  --startmini    Force xhippo to start with its window in the \"mini\"
                   state (with the list hidden).
-q  --hideplayeroutput
                   Redirect player stdout and stderr to /dev/null.
-O  --sortonload   Sort the playlist immediately after loading it.
-W  --writeplaying Write the name of the currently-playing song to
                   $HOME/.xhippo/current_song.
-V  --version      Print xhippo's version and exit.
-c  --nocheckfiles Don't bother checking whether the files in the playlist
                   actually exist.
-o  --ordered      Play the files in the order specified in the playlist,
                   rather than in random order.
-h  --help         This help.

Report bugs to bug-xhippo@gnu.org.\n");
      exit(0);
      break;
    default:
      exit(20);
      break;
    }
  }

  /* Create the list. We have to do this after we've read the options,
     because we need to know the state of mode_left_scroll. */
  filelist = gtk_clist_new(1);

  gtk_clist_set_selection_mode(GTK_CLIST(filelist), GTK_SELECTION_SINGLE);
  gtk_clist_column_titles_hide(GTK_CLIST(filelist));
  gtk_clist_set_column_width(GTK_CLIST(filelist), 0, 200);
  gtk_clist_set_shadow_type(GTK_CLIST(filelist), GTK_SHADOW_ETCHED_IN);

  gtk_signal_connect(GTK_OBJECT(filelist), "select_row",
		     GTK_SIGNAL_FUNC(handle_list_select), NULL);
  gtk_signal_connect(GTK_OBJECT(filelist), "button_press_event",
		     GTK_SIGNAL_FUNC(handle_list_click), NULL);
  gtk_widget_set_name(filelist, "filelist");

  vadj = (GtkAdjustment *)gtk_adjustment_new(0, 0, 0, 0, 0, 0);
  gtk_clist_set_vadjustment(GTK_CLIST(filelist), GTK_ADJUSTMENT(vadj));
  scrollbar = gtk_vscrollbar_new(GTK_ADJUSTMENT(vadj));

  if (mode_left_scroll) { 
    gtk_box_pack_start(GTK_BOX(list_box), scrollbar, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(list_box), filelist, TRUE, TRUE, 0); 
  } else { 
    gtk_box_pack_start(GTK_BOX(list_box), filelist, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(list_box), scrollbar, FALSE, FALSE, 0); 
  } 

  /* Deal with non-options */
  while (optind < argc) {
    readplaylist(argv[optind++]);
  }
  
  /* Show the list and scrollbar. */
  gtk_widget_show(filelist);
  gtk_widget_show(scrollbar);

  /* Show the list box. */
  gtk_widget_show(list_box);

  /* Update the checkbox state. */
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(randomcheckbox),
			       1 - mode_play_ordered);

  /* Show the everything box. */
  gtk_widget_show(all_box);

  /* Read the config file for the second time. */
  read_config_files(1);

  /* Set the status. */
  snprintf(strbuf, STRBUFSIZE - 1, STRNUMFILES, listcount);
  status(strbuf);

  /* Read the window attributes as appropriate. */
  if (mode_save_window_pos) {
    readwindowstate();
  }
  if (mode_start_mini) attribute_mini = 1;

  /* Attach the acceleration group to the window. */
  gtk_accel_group_attach(accel_group, GTK_OBJECT(window));

  /* Set the window attributes and show it. */
  gtk_widget_set_uposition(window, attribute_xpos, attribute_ypos);
  if (attribute_width && attribute_height)
    gtk_widget_set_usize(window, attribute_width, attribute_height);
  if (attribute_mini) {
    attribute_mini = 1;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(minicheckbox), 1);
    set_mini();
  }
  gtk_widget_show(window);

  /* If mode_play_on_start, start playing. */
  if (mode_play_on_start) pick_next();

  /* Main loop. */
  gtk_main();

  /* And exit cleanly. */
  exit(0);
}
