/*
 * GNU POC (passwords on card) - manage passwords on smartcards
 * Copyright (C) 2001 Henning Koester <henning@crackinghacking.de>
 *
 * Please report bugs to bug-poc@gnu.org
 *
 * This file is part of POC.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* $Id: format.c,v 1.6 2001/08/18 16:10:47 henning Exp $ */

#include <config.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "poc.h"
#include "poc_types.h"
#include "lang.h"
#include "card.h"
#include "misc.h"

/******************************************************************************
 *
 * Function    : format_card
 *
 * Description : This function formats a card for use with POC.
 *
 * Input       : [1] port (u16)
 *                   Com-Port
 *
 * Return      : POC_ERROR or POC_SUCCESS
 *
 *****************************************************************************/
bool 
format_card(const u16 port) {
  u8 *buffer;          /* Buffer which will be written to the card. Filled
			  with zeros. */
  u16 ctn = 0;         /* Card-terminal handle. */
  int card_size;       /* Card's size. */
  char ret;            /* Returns of CT-API calls. */

  /* Initialize & reset card terminal. */
  if ( (card_init_terminal(ctn, port) != POC_SUCCESS) ||
       (card_reset_terminal(ctn) != POC_SUCCESS) ) {

    /* If initialization fails, or resetting, we will return to the caller. */
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* Request ICC and check whether a memory card is inserted. */
  if (card_request_icc(ctn) != MEMORY_CARD) {
    print_err(STR_NO_MEM_CARD);
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* Ask for the card's size. */
  printf(STR_ENTER_CARD_SIZE);
  scanf("%i", &card_size);

  /* Ask whether we really want to format card. */
  getchar();
  printf(STR_ASK_FORMAT);
  if (tolower(getchar()) != 'y') {

    /* No, we don't format the card. */
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* Allocate zero initialized memory. */ 
  if ( (buffer = calloc(card_size, sizeof(u8))) == NULL) {
    print_err(ERR_PRFX_NM);
    perror("");
    return(POC_ERROR);
  }

  /*
   * As explained in poc's manual, the card's memory structure is organized
   * the following way:
   * The first 2 byte of the card contain the card's size. Byte 3 and 4
   * contain the data size (the data size is the number of bytes of the
   * data area which are used.) From the 5'th byte starts the data area,
   * which is the area of the card where the passwords+descriptions are
   * stored. The data area is encrypted.
   */

  /* Store the card's size in the first 2 bytes of the card. */
  buffer[0] = card_size >> 8;
  buffer[1] = card_size & 0x00ff;

  /* Select card's memory for writing. */
  if (card_select_file(ctn) != POC_SUCCESS) {
    card_close_terminal(ctn);
    return(POC_ERROR);
  }

  /* Now we will write the prepared buffer to the card. */
  if ( (ret =
        card_write_data(ctn, 0, card_size, buffer)) != POC_SUCCESS) {
    if (ret == POC_MEM_ERR) {
      print_err(ERR_PRFX_NM);   
      perror("");
    }
    card_close_terminal(ctn);
    return(POC_ERROR);
  } 

  card_close_terminal(ctn);  /* Communication is over... disconnect. */

  free(buffer);              /* Free allocated memory. */

  return(POC_SUCCESS);
}
