/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: RoutingDebug.java
 *
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 */
package com.sun.electric.tool.user.ui;

import com.sun.electric.database.geometry.EPoint;
import com.sun.electric.database.hierarchy.Cell;
import com.sun.electric.database.topology.RTNode;
import com.sun.electric.technology.Layer;
import com.sun.electric.tool.routing.SeaOfGates;
import com.sun.electric.tool.routing.SeaOfGates.SeaOfGatesOptions;
import com.sun.electric.tool.routing.seaOfGates.SeaOfGatesEngine;
import com.sun.electric.tool.routing.seaOfGates.SeaOfGatesEngine.NeededRoute;
import com.sun.electric.tool.routing.seaOfGates.SeaOfGatesEngine.SOGBound;
import com.sun.electric.tool.routing.seaOfGates.SeaOfGatesEngine.SearchVertex;
import com.sun.electric.tool.routing.seaOfGates.SeaOfGatesEngineFactory;
import com.sun.electric.tool.routing.seaOfGates.SeaOfGatesEngineFactory.SeaOfGatesEngineType;
import com.sun.electric.tool.user.Highlight;
import com.sun.electric.tool.user.User;
import com.sun.electric.tool.user.dialogs.EModelessDialog;
import com.sun.electric.util.TextUtils;
import com.sun.electric.util.math.DBMath;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;

/**
 * Class to handle routing mode.
 */
public class RoutingDebug
{
	private static RoutingDialog debugDialog = null;

	/**
	 * Method to do an interactive routing.
	 */
	public static void startDebugging()
	{
		// enable routing mode in click/zoom/wire listener
		User.setRoutingMode(true);

		debugDialog = new RoutingDialog();
	}

	private static void endDebugging()
	{
		// disable routing mode in click/zoom/wire listener
		User.setRoutingMode(false);

		if (debugDialog != null)
		{
			// remove dialog
			debugDialog.setVisible(false);
			debugDialog.dispose();
			debugDialog = null;
		}
	}

	public static boolean isActive() { return debugDialog != null; }

	public static void setRouteDescription(String desc)
	{
		if (debugDialog != null)
			debugDialog.routeDescription.setText(desc);
	}

	public static void debugRoute(NeededRoute nr)
	{
		if (debugDialog != null)
		{
			debugDialog.nr = nr;
			System.out.println("Routing from " + nr.getAtoBDirection().getFromPortInst().getNodeInst().describe(false) + ", port " +
				nr.getAtoBDirection().getFromPortInst().getPortProto().getName() +
				" at (" + nr.getAtoBDirection().getFromX() + "," + nr.getAtoBDirection().getFromY() +
				"), M" + (nr.getAtoBDirection().getFromZ() + 1));
			System.out.println("Routing to   " + nr.getAtoBDirection().getToPortInst().getNodeInst().describe(false) + ", port " +
				nr.getAtoBDirection().getToPortInst().getPortProto().getName() +
				" at (" + nr.getAtoBDirection().getToX() + "," + nr.getAtoBDirection().getToY() +
				"), M" + (nr.getAtoBDirection().getToZ() + 1));
		}
	}

	public static void identifyNewDebugPoint(SearchVertex sv)
	{
		if (debugDialog != null)
			debugDialog.identifyNewDebugPoint(sv);
	}

	public static void endProcessingSV(SearchVertex sv, String details)
	{
		if (debugDialog != null)
		{
			SVState svs = debugDialog.svInfo.get(sv);
			if (svs != null)
				svs.details = details;
		}
	}

	public static void mouseMoved(MouseEvent evt)
	{
		if (debugDialog == null) return;
		EditWindow wnd = (EditWindow)evt.getSource();
		Point2D dbClick = wnd.screenToDatabase(evt.getX(), evt.getY());

		// findObject handles cycling through objects
		double bestDist = Double.MAX_VALUE;
		SearchVertex bestSV = null;
		for(SearchVertex sv : debugDialog.svInfo.keySet())
		{
			double off = sv.getZ() * debugDialog.layerOffset;
			double dX = sv.getX() + off - dbClick.getX();
			double dY = sv.getY() + off - dbClick.getY();
			double dist = Math.sqrt(dX*dX + dY*dY);
			if (dist < bestDist)
			{
				bestDist = dist;
				bestSV = sv;
			}
		}
		if (bestDist < 1)
		{
			SVState svs = debugDialog.svInfo.get(bestSV);
			debugDialog.model.clear();
			if (svs.details == null)
			{
				String msg = "Did not get propagated. Cost=" + bestSV.getCost();
				if (bestSV.getLast() != null)
					msg += ", previous point at (" + TextUtils.formatDouble(bestSV.getLast().getX()) + "," +
						TextUtils.formatDouble(bestSV.getLast().getY()) + ",M" + (bestSV.getLast().getZ()+1) + ")";
				debugDialog.model.addElement(msg);
			} else
			{
				String[] parts = svs.details.split("/");
				for(int i=0; i<parts.length; i++)
				{
					String part = parts[i];
					if (part.indexOf('|') >= 0)
					{
						String [] subParts = part.split("\\|");
						int firstLen = subParts[0].length();
						String leading = "";
						for(int j=0; j<firstLen; j++) leading += " ";
						debugDialog.model.addElement(subParts[0]);
						for(int j=1; j<subParts.length; j++)
							debugDialog.model.addElement(leading + subParts[j]);					
					} else
						debugDialog.model.addElement(parts[i]);
				}
			}
		}
	}

	private static class SVState
	{
		Highlight label;
		String details;

		SVState(Highlight l) { label = l; }
	}

	/**
	 * Class to handle the "Routing control" dialog.
	 */
	private static class RoutingDialog extends EModelessDialog
	{
		private SeaOfGatesEngine router;
		private NeededRoute nr;
		private EditWindow wnd;
		private Cell cell;
		private boolean runningToEnd;
		private int svOrder;
		private List<Highlight> tempHighlights = new ArrayList<Highlight>();
		private Map<SearchVertex,SVState> svInfo = new HashMap<SearchVertex,SVState>();
		private Map<String,Highlight> anchorHighlights = new HashMap<String,Highlight>();
		private Map<String,Integer> anchorHeight = new HashMap<String,Integer>();
		private double goalWidth = 0.005;
		private double layerOffset = 0.04;
		private JLabel routeDescription;
		private JList list;
		private DefaultListModel model;

		/** Creates new form Debug-Routing */
		public RoutingDialog()
		{
			super(TopLevel.getCurrentJFrame());
			runningToEnd = false;
			svOrder = 1;

			wnd = EditWindow.getCurrent();
			wnd.getRulerHighlighter().clear();
			cell = wnd.getCell();

			// fill in the debug dialog
			getContentPane().setLayout(new GridBagLayout());
			setTitle("Debug Routing");
			setName("");
			addWindowListener(new WindowAdapter()
			{
				public void windowClosing(WindowEvent evt) { endDebugging(); }
			});
			GridBagConstraints gbc;

			JButton next = new JButton("Step");
			next.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent evt) { step(); }
			});
			getRootPane().setDefaultButton(next);
			gbc = new GridBagConstraints();
			gbc.gridx = 0;   gbc.gridy = 0;
			gbc.insets = new Insets(4, 4, 4, 4);
			gbc.weightx = 0.5;
			getContentPane().add(next, gbc);

			JButton run = new JButton("Run");
			run.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent evt) { run(); }
			});
			gbc = new GridBagConstraints();
			gbc.gridx = 1;   gbc.gridy = 0;
			gbc.insets = new Insets(4, 4, 4, 4);
			gbc.weightx = 0.5;
			getContentPane().add(run, gbc);

			JButton showEndBlockage = new JButton("Show End Blockage");
			showEndBlockage.addActionListener(new ActionListener()
			{
				public void actionPerformed(ActionEvent evt) { showEndBlockages(); }
			});
			gbc = new GridBagConstraints();
			gbc.gridx = 0;   gbc.gridy = 1;
			gbc.gridwidth = 2;
			gbc.insets = new Insets(4, 4, 4, 4);
			gbc.weightx = 0.5;
			getContentPane().add(showEndBlockage, gbc);

			routeDescription = new JLabel("");
			gbc = new GridBagConstraints();
			gbc.gridx = 0;   gbc.gridy = 2;
			gbc.gridwidth = 2;
			gbc.fill = GridBagConstraints.HORIZONTAL;
			gbc.insets = new Insets(4, 4, 4, 4);
			gbc.weightx = 1;
			getContentPane().add(routeDescription, gbc);

			model = new DefaultListModel();
			list = new JList(model);
			list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			JScrollPane Center = new JScrollPane();
	        Center.setMinimumSize(new Dimension(700, 50));
	        Center.setPreferredSize(new Dimension(700, 300));
			Center.setViewportView(list);
			gbc = new GridBagConstraints();
			gbc.gridx = 0;   gbc.gridy = 3;
			gbc.gridwidth = 2;
			gbc.fill = GridBagConstraints.BOTH;
			gbc.insets = new Insets(4, 4, 4, 4);
			gbc.weightx = gbc.weighty = 1;
	        getContentPane().add(Center, gbc);
			list.addMouseListener(new MouseAdapter()
			{
				public void mouseClicked(MouseEvent evt) { listClick(evt); }
			});

			pack();
			finishInitialization();
			setVisible(true);
		}

		private void listClick(MouseEvent evt)
		{
			int index = list.getSelectedIndex();
			System.out.println("CLICKED ON INDEX "+index);
		}

		private void showEndBlockages()
		{
			router = SeaOfGatesEngineFactory.createSeaOfGatesEngine(SeaOfGatesEngineType.defaultVersion);
			SeaOfGatesOptions prefs = new SeaOfGatesOptions();
			prefs.getOptionsFromPreferences();
			displayEndBlockages = true;
			router.setPrefs(prefs);
			SeaOfGates.seaOfGatesRoute(cell, router);
		}

		private void run()
		{
			router = SeaOfGatesEngineFactory.createSeaOfGatesEngine(SeaOfGatesEngineType.defaultVersion);
			SeaOfGatesOptions prefs = new SeaOfGatesOptions();
			prefs.getOptionsFromPreferences();
			router.setPrefs(prefs);
			SeaOfGates.seaOfGatesRoute(cell, router);
			if (nr == null) return;

			runningToEnd = true;
			for(;;)
			{
				// mark the next point at the start of the wavefront
				SearchVertex sv = nr.getAtoBDirection().getNextSearchVertex();
				if (sv != SeaOfGatesEngine.svExhausted) startProcessingSV(sv);

				// advance the wavefront
				SearchVertex result = advanceOneStep();
				if (result != null) break;
			}
			for(String anchorLoc : anchorHeight.keySet())
			{
				Integer height = anchorHeight.get(anchorLoc);
				if (height.intValue() <= 0) continue;
				String [] anchorCoords = anchorLoc.split("/");
				double x = TextUtils.atof(anchorCoords[0]);
				double y = TextUtils.atof(anchorCoords[1]);
				double off = height.intValue() * layerOffset;
				Highlight h = wnd.getRulerHighlighter().addLine(new EPoint(x, y), new EPoint(x+off, y+off), cell, false, false);
				h.setColor(Color.BLACK);
			}
			wnd.getRulerHighlighter().finished();
		}

		private void step()
		{
			for(Highlight h : tempHighlights) wnd.getRulerHighlighter().remove(h);
			tempHighlights.clear();

			SearchVertex result = advanceOneStep();
			if (result != null) return;

			// put "X" at best next point
			SearchVertex sv = nr.getAtoBDirection().getNextSearchVertex();
			if (sv != SeaOfGatesEngine.svExhausted)
			{
				startProcessingSV(sv);
				double off = sv.getZ() * layerOffset;
				Highlight h = wnd.getRulerHighlighter().addLine(new EPoint(sv.getX()+off-9, sv.getY()+off-9),
					new EPoint(sv.getX()+off+9, sv.getY()+off+9), cell, true, false);
				h.setColor(Color.RED);
				tempHighlights.add(h);
				h = wnd.getRulerHighlighter().addLine(new EPoint(sv.getX()+off-9, sv.getY()+off+9),
					new EPoint(sv.getX()+off+9, sv.getY()+off-9), cell, true, false);
				h.setColor(Color.RED);
				tempHighlights.add(h);
				System.out.println("AT ("+TextUtils.formatDouble(sv.getX())+","+TextUtils.formatDouble(sv.getY())+",M"+(sv.getZ()+1)+")C="+sv.getCost());
			}
			wnd.getRulerHighlighter().finished();
		}

		private SearchVertex advanceOneStep()
		{
			// advance the wavefront
			SearchVertex result = nr.getAtoBDirection().advanceWavefront();

			// stop now if complete
			if (result != null)
			{
				String resA;
				if (result == SeaOfGatesEngine.svAborted) resA = "Aborted by user"; else
				if (result == SeaOfGatesEngine.svExhausted) resA = "Examined all possibilities"; else
				if (result == SeaOfGatesEngine.svLimited) resA = "Stopped after " + router.getPrefs().complexityLimit + " steps"; else
				{
					showGoalSV(result);
					resA = "Success!";
				}
				System.out.println("Result: " + resA);
			}
			return result;
		}

		protected void escapePressed() { endDebugging(); }

		private void showGoalSV(SearchVertex sv)
		{
			identifyNewDebugPoint(sv);
			SVState svs = svInfo.get(sv);
			if (svs != null)
			{
				Highlight.Message msg = (Highlight.Message)svs.label;
				msg.setInfo("!!GOAL!!");
			}
			for(;;)
			{
				SearchVertex svLast = sv.getLast();
				if (svLast == null) break;
				if (sv.getZ() != svLast.getZ())
				{
					int lowZ = Math.min(sv.getZ(), svLast.getZ());
					int highZ = Math.max(sv.getZ(), svLast.getZ());
					double lowOff = lowZ * layerOffset;
					double highOff = highZ * layerOffset;
					Highlight h = wnd.getRulerHighlighter().addLine(new EPoint(sv.getX()+lowOff, sv.getY()+lowOff+goalWidth),
						new EPoint(sv.getX()+highOff-goalWidth, sv.getY()+highOff), cell, true, false);
					h.setColor(Color.WHITE);
					h = wnd.getRulerHighlighter().addLine(new EPoint(sv.getX()+lowOff+goalWidth, sv.getY()+lowOff),
						new EPoint(sv.getX()+highOff, sv.getY()+highOff-goalWidth), cell, true, false);
					h.setColor(Color.WHITE);
				} else
				{
					double off = sv.getZ() * layerOffset;
					if (sv.getX() != svLast.getX())
					{
						// horizontal line
						Highlight h = wnd.getRulerHighlighter().addLine(new EPoint(sv.getX()+off, sv.getY()+off-goalWidth),
							new EPoint(svLast.getX()+off, svLast.getY()+off-goalWidth), cell, true, false);
						h.setColor(Color.WHITE);
						h = wnd.getRulerHighlighter().addLine(new EPoint(sv.getX()+off, sv.getY()+off+goalWidth),
							new EPoint(svLast.getX()+off, svLast.getY()+off+goalWidth), cell, true, false);
						h.setColor(Color.WHITE);
					} else
					{
						// vertical line
						Highlight h = wnd.getRulerHighlighter().addLine(new EPoint(sv.getX()+off-goalWidth, sv.getY()+off),
							new EPoint(svLast.getX()+off-goalWidth, svLast.getY()+off), cell, true, false);
						h.setColor(Color.WHITE);
						h = wnd.getRulerHighlighter().addLine(new EPoint(sv.getX()+off+goalWidth, sv.getY()+off),
							new EPoint(svLast.getX()+off+goalWidth, svLast.getY()+off), cell, true, false);
						h.setColor(Color.WHITE);
					}
				}
				sv = svLast;
			}
		}

		private void startProcessingSV(SearchVertex sv)
		{
			SVState svs = svInfo.get(sv);
			if (svs != null)
			{
				if (sv.getLast() != null)
				{
					Highlight.Message msg = (Highlight.Message)svs.label;
					msg.setInfo(svOrder + "");
					svOrder++;
				}
			}
		}

		public void identifyNewDebugPoint(SearchVertex sv)
		{
			if (sv.getLast() == null)
			{
				// show starting vertex
				double x = sv.getX(), y = sv.getY();
				double off = sv.getZ() * layerOffset;
				Highlight h = wnd.getRulerHighlighter().addMessage(cell, "START", new EPoint(x+off, y+off));
				SVState svs = new SVState(h);
				svInfo.put(sv, svs);
				return;
			}

			if (sv.getZ() != sv.getLast().getZ())
			{
				// draw white line at angle showing change of layer
				int lowZ = Math.min(sv.getZ(), sv.getLast().getZ());
				int highZ = Math.max(sv.getZ(), sv.getLast().getZ());
				double lowOff = lowZ * layerOffset;
				double highOff = highZ * layerOffset;
				Highlight h = wnd.getRulerHighlighter().addLine(new EPoint(sv.getX()+lowOff, sv.getY()+lowOff),
					new EPoint(sv.getX()+highOff, sv.getY()+highOff), cell, true, false);
				h.setColor(Color.WHITE);
			} else
			{
				// draw line in proper metal color showing the motion
				double off = sv.getZ() * layerOffset;
				Highlight h = wnd.getRulerHighlighter().addLine(new EPoint(sv.getX()+off, sv.getY()+off),
					new EPoint(sv.getLast().getX()+off, sv.getLast().getY()+off), cell, false, false);
				Layer lay = router.getMetalLayer(sv.getZ());
				h.setColor(lay.getGraphics().getColor());
			}

			// show black anchor line to the proper location
			String anchorLoc = TextUtils.formatDouble(sv.getX()) + "/" + TextUtils.formatDouble(sv.getY());
			Highlight anchorHigh = anchorHighlights.get(anchorLoc);
			Integer height = anchorHeight.get(anchorLoc);
			int lowZ = Math.min(sv.getZ(), sv.getLast().getZ());
			if (height == null) height = Integer.valueOf(lowZ); else
			{
				int lowest = Math.min(height.intValue(), lowZ);
				if (!runningToEnd && anchorHigh != null && lowZ < height.intValue())
				{
					wnd.removeHighlight(anchorHigh);
					anchorHighlights.remove(anchorLoc);
				}
				height = Integer.valueOf(lowest);
			}
			anchorHeight.put(anchorLoc, height);
			if (!runningToEnd && height.intValue() > 0)
			{
				double off = height.intValue() * layerOffset;
				anchorHigh = wnd.addHighlightLine(new EPoint(sv.getX(), sv.getY()),
					new EPoint(sv.getX()+off, sv.getY()+off), cell, false, false);
				anchorHigh.setColor(Color.BLACK);
				anchorHighlights.put(anchorLoc, anchorHigh);
			}

			// now add the cost text
			double x = sv.getX(), y = sv.getY();
			double off = sv.getZ() * layerOffset;
			Highlight h = wnd.getRulerHighlighter().addMessage(cell, "M" + (sv.getZ()+1), new EPoint(x+off, y+off));
			SVState svs = new SVState(h);
			svInfo.put(sv, svs);
		}
	}

	/************************************* DEBUGGING BLOCKAGES AT START AND END OF ROUTE *************************************/

	private static Color[] allColors = new Color[]{Color.RED, Color.GREEN, Color.BLUE, Color.CYAN, Color.MAGENTA, Color.YELLOW};
	private static Map<Integer,Color> netColors;
	private static Map<Integer,Integer> netAngles;
	private static int colorsAssigned;
	private static int angleAssigned;
	private static boolean displayEndBlockages;

	public static boolean isDisplayEndBlockages() { return debugDialog != null && displayEndBlockages; }

	public static void showGeometryAtRouteEnds(NeededRoute nr)
	{
		displayEndBlockages = false;
		if (debugDialog == null) return;

		double examineSurround = 10;
		colorsAssigned = 0;
		angleAssigned = 45;
		netColors = new HashMap<Integer,Color>();
		netAngles = new HashMap<Integer,Integer>();
		double lowestX = Double.MAX_VALUE, lowestY = Double.MAX_VALUE;
		debugDialog.wnd.getRulerHighlighter().clear();

		Rectangle2D headBound = new Rectangle2D.Double(nr.getAtoBDirection().getFromX()-examineSurround,
			nr.getAtoBDirection().getFromY()-examineSurround, examineSurround*2, examineSurround*2);
		for(int i=0; i<debugDialog.router.getNumMetals(); i++)
		{
			Layer layer = debugDialog.router.getMetalLayer(i);
			RTNode metalTree = nr.getMetalTree(layer);
			if (metalTree == null) continue;
			for (RTNode.Search sea = new RTNode.Search(headBound, metalTree, true); sea.hasNext();)
			{
				SOGBound sBound = (SOGBound)sea.next();
				showGeometryPiece(sBound);
				if (sBound.getBounds().getMinX() < lowestX) lowestX = sBound.getBounds().getMinX();
				if (sBound.getBounds().getMinY() < lowestY) lowestY = sBound.getBounds().getMinY();
			}
		}
		Rectangle2D tailBound = new Rectangle2D.Double(nr.getAtoBDirection().getToX()-examineSurround,
			nr.getAtoBDirection().getToY()-examineSurround, examineSurround*2, examineSurround*2);
		for(int i=0; i<debugDialog.router.getNumMetals(); i++)
		{
			Layer layer = debugDialog.router.getMetalLayer(i);
			RTNode metalTree = nr.getMetalTree(layer);
			if (metalTree == null) continue;
			for (RTNode.Search sea = new RTNode.Search(tailBound, metalTree, true); sea.hasNext();)
			{
				SOGBound sBound = (SOGBound)sea.next();
				showGeometryPiece(sBound);
				if (sBound.getBounds().getMinX() < lowestX) lowestX = sBound.getBounds().getMinX();
				if (sBound.getBounds().getMinY() < lowestY) lowestY = sBound.getBounds().getMinY();
			}
		}

		// show key
		double pos = lowestY - 3;
		for(Integer netIDI : netColors.keySet())
		{
			Color color = netColors.get(netIDI);
			Integer angle = netAngles.get(netIDI);
			Rectangle2D bound = new Rectangle2D.Double(lowestX, pos, 4, 2);
			showStripedRect(bound, color, angle.intValue());
			debugDialog.wnd.getRulerHighlighter().addMessage(debugDialog.cell, "Net " + netIDI.intValue(),
				new EPoint(lowestX+5, pos+1));
			pos -= 3;
		}
		debugDialog.wnd.getRulerHighlighter().finished();
	}

	private static void showStripedRect(Rectangle2D bound, Color color, int angle)
	{
		Highlight h = debugDialog.wnd.getRulerHighlighter().addArea(bound, debugDialog.cell);
		h.setColor(color);
		double gapX = 0.5, gapY = 0.5;
		double lX = bound.getMinX(), hX = bound.getMaxX();
		double lY = bound.getMinY(), hY = bound.getMaxY();
		if (angle == 0)
		{
			// horizontal lines
			for(double y = gridValue(lY, gapY); y < hY; y += gapY)
			{
				h = debugDialog.wnd.getRulerHighlighter().addLine(new Point2D.Double(lX, y), new Point2D.Double(hX, y),
					debugDialog.cell, false, false);
				h.setColor(color);
			}
			return;
		}
		if (angle == 90)
		{
			// vertical lines
			for(double x = gridValue(lX, gapX); x < hX; x += gapX)
			{
				h = debugDialog.wnd.getRulerHighlighter().addLine(new Point2D.Double(x, lY), new Point2D.Double(x, hY),
					debugDialog.cell, false, false);
				h.setColor(color);
			}
			return;
		}

		// prepare for angled lines
		double s = DBMath.sin(angle*10);
		double c = DBMath.cos(angle*10);
		gapX = gapX / Math.abs(s);
		gapY = gapY / Math.abs(c);
		if (angle <= 45 || angle >= 135)
		{
			// lines that are closer to horizontal
			double pos, length;
			if (angle < 90)
			{
				pos = gridValue(lX-gapX, gapX);
				length = (hX - pos) / c;
			} else
			{
				pos = gridValue(hX+gapX, gapX);
				length = (lX - pos) / c;
			}
			for(double y = gridValue(lY - bound.getWidth() - gapY, gapY); y < hY + bound.getWidth(); y += gapY)
			{
				Point2D pt1 = new Point2D.Double(pos, y);
				Point2D pt2 = new Point2D.Double(pos + c * length, y + s * length);
				boolean gone = DBMath.clipLine(pt1, pt2, lX, hX, lY, hY);
				if (gone) continue;
				h = debugDialog.wnd.getRulerHighlighter().addLine(pt1, pt2, debugDialog.cell, false, false);
				h.setColor(color);				
			}
		} else
		{
			// lines that are closer to vertical
			double pos = gridValue(lY - gapY, gapY);
			double length = (hY - pos) / s;
			for(double x = gridValue(lX - bound.getHeight() - gapX, gapX); x < hX + bound.getHeight(); x += gapX)
			{
				Point2D pt1 = new Point2D.Double(x, pos);
				Point2D pt2 = new Point2D.Double(x + c * length, pos + s * length);
				boolean gone = DBMath.clipLine(pt1, pt2, lX, hX, lY, hY);
				if (gone) continue;
				h = debugDialog.wnd.getRulerHighlighter().addLine(pt1, pt2, debugDialog.cell, false, false);
				h.setColor(color);				
			}
		}
	}

	private static double gridValue(double v, double gap)
	{
		int iv = (int)Math.round(v / gap);
		return iv * gap;
	}

	private static void showGeometryPiece(SOGBound sBound)
	{
		Integer netIDI = Integer.valueOf(sBound.getNetID());
		Color color = netColors.get(netIDI);
		Integer angle = netAngles.get(netIDI);
		if (color == null)
		{
			netColors.put(netIDI, color = allColors[colorsAssigned]);
			colorsAssigned = (colorsAssigned + 1) % allColors.length;
			netAngles.put(netIDI, angle = Integer.valueOf(angleAssigned));
			angleAssigned = (angleAssigned + 41) % 180;
		}
		showStripedRect(sBound.getBounds(), color, angle.intValue());
//		System.out.println("    "+sBound.getBounds().getMinX()+"<=X<="+sBound.getBounds().getMaxX()+" AND "+
//			sBound.getBounds().getMinY()+"<=Y<="+sBound.getBounds().getMaxY()+" ON NET "+sBound.getNetID());
	}
}
