/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: SeaOfGatesEngine.java
 * Routing tool: Sea of Gates routing
 * Written by: Steven M. Rubin
 *
 * Copyright (c) 2007, Oracle and/or its affiliates. All rights reserved.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 */
package com.sun.electric.tool.routing.seaOfGates;

import com.sun.electric.database.EditingPreferences;
import com.sun.electric.database.Environment;
import com.sun.electric.database.geometry.EGraphics;
import com.sun.electric.database.geometry.EPoint;
import com.sun.electric.database.geometry.ERectangle;
import com.sun.electric.database.geometry.Poly;
import com.sun.electric.database.geometry.Poly.Type;
import com.sun.electric.database.geometry.Poly3D;
import com.sun.electric.database.geometry.PolyBase;
import com.sun.electric.database.hierarchy.Cell;
import com.sun.electric.database.hierarchy.Export;
import com.sun.electric.database.hierarchy.HierarchyEnumerator;
import com.sun.electric.database.hierarchy.Nodable;
import com.sun.electric.database.hierarchy.View;
import com.sun.electric.database.network.Netlist;
import com.sun.electric.database.network.Network;
import com.sun.electric.database.prototype.NodeProto;
import com.sun.electric.database.topology.ArcInst;
import com.sun.electric.database.topology.Connection;
import com.sun.electric.database.topology.Geometric;
import com.sun.electric.database.topology.NodeInst;
import com.sun.electric.database.topology.PortInst;
import com.sun.electric.database.topology.RTBounds;
import com.sun.electric.database.topology.RTNode;
import com.sun.electric.database.variable.EditWindow_;
import com.sun.electric.database.variable.VarContext;
import com.sun.electric.database.variable.Variable;
import com.sun.electric.technology.ArcProto;
import com.sun.electric.technology.DRCTemplate;
import com.sun.electric.technology.Layer;
import com.sun.electric.technology.PrimitiveNode;
import com.sun.electric.technology.SizeOffset;
import com.sun.electric.technology.Technology;
import com.sun.electric.technology.technologies.Generic;
import com.sun.electric.tool.Job;
import com.sun.electric.tool.drc.DRC;
import com.sun.electric.tool.routing.Routing;
import com.sun.electric.tool.routing.SeaOfGates;
import com.sun.electric.tool.user.ErrorLogger;
import com.sun.electric.tool.user.ui.RoutingDebug;
import com.sun.electric.tool.util.concurrent.runtime.taskParallel.ThreadPool.PoolWorkerStrategyFactory;
import com.sun.electric.tool.util.concurrent.utils.ElapseTimer;
import com.sun.electric.util.TextUtils;
import com.sun.electric.util.concurrent.ElectricWorkerStrategy;
import com.sun.electric.util.math.DBMath;
import com.sun.electric.util.math.GenMath;
import com.sun.electric.util.math.Orientation;

import java.awt.Color;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.Semaphore;

import javax.swing.SwingUtilities;

/**
 * Class to do sea-of-gates routing. This router replaces unrouted arcs with
 * real geometry. It has these features:
 * > The router only works in layout, and only routes metal wires.
 * > The router uses vias to move up and down the metal layers.
 * > Understands multiple vias and multiple via orientations.
 * > The router is not tracked: it runs on the Electric grid
 * > Favors wires on full-grid units
 * > Tries to cover multiple grid units in a single jump
 * > Routes power and ground first, then goes by length (shortest nets first)
 * > Prefers to run odd metal layers on horizontal, even layers on vertical
 * > Routes in both directions (from A to B and from B to A) and chooses the one that completes first
 * > Serial method alternates advancing each wavefront, stopping when one completes
 * > Parallel option runs both wavefronts at once and aborts the slower one
 * > Users can request that some layers not be used, can request that some layers be favored
 * > Routes are made as wide as the widest arc already connected to any point
 * > User preference can limit width
 * > Cost penalty also includes space left in the track on either side of a segment
 * 
 * Things to do:
 *  Make layer-change cost go up as you get farther from destination layer
 *  At the end of routing, try again with those that failed
 *  Detect "river routes" and route specially
 *  Ability to route to any previous part of route when daisy-chaining
 *  Lower cost if running over existing layout (on the same net)
 *  Ability to connect to anything on the destination net
 *  Rip-up
 *  Global j * routing(?)
 *  Sweeping cost parameters (with parallel processors) to find best, dynamically
 *  Forcing each processor to use a different grid track
 *  Characterizing routing task (which parallelism to use)
 */
public abstract class SeaOfGatesEngine
{
	/** True to display the first routing failure. */			protected static final boolean DEBUGFAILURE = false;
	/** True to debug "infinite" loops. */						static final boolean DEBUGLOOPS = false;
	/** true to use full, gridless routing */					private static final boolean FULLGRAIN = true;

	/** Percent of cell size that route must stay inside. */	private static final double PERCENTLIMIT = 7;
	/** Number of steps per unit when searching. */				private static final double GRANULARITY = 1;
	/** Size of steps when searching. */						private static final double GRAINSIZE = (1 / GRANULARITY);
	/** Cost of vertical=odd (M1), horizontal=even (M2). */		private static final int COSTALTERNATINGMETAL = 20;
	/** Cost of changing layers. */								private static final int COSTLAYERCHANGE = 8;
	/** Cost of routing away from the target. */				private static final int COSTWRONGDIRECTION = 15;
	/** Cost of running on non-favored layer. */				private static final int COSTUNFAVORED = 10;
	/** Cost of making a turn. */								private static final int COSTTURNING = 1;
	/** Cost of having coordinates that are off-grid. */		private static final int COSTOFFGRID = 15;

	public static SearchVertex svAborted = new SearchVertex(0, 0, 0, 0, null, 0, null);
	public static SearchVertex svExhausted = new SearchVertex(0, 0, 0, 0, null, 0, null);
	public static SearchVertex svLimited = new SearchVertex(0, 0, 0, 0, null, 0, null);

	/** number of metal layers in the technology. */			private static int numMetalLayers;
	/** true if this is first failure in route (debug) */		protected static volatile boolean firstFailure;

	/** Cell in which routing occurs. */						protected Cell cell;
	/** Cell size. */											private Rectangle2D cellBounds;
	/** Technology to use for routing. */						private Technology tech;
	/** R-Trees for metal blockage in the cell. */				private Map<Layer, RTNode> metalTrees;
	/** R-Trees for via blockage in the cell. */				private Map<Layer, RTNode> viaTrees;
	/** Maps Arcs to network IDs in the R-Tree. */				private Map<ArcInst, Integer> netIDs;
	/** metal layers in the technology. */						private Layer[] metalLayers;
	/** via layers in the technology. */						private Layer[] viaLayers;
	/** arcs to use for each metal layer. */					private ArcProto[] metalArcs;
	/** favoritism for each metal layer. */						private boolean[] favorArcs;
	/** avoidance for each metal layer. */						private boolean[] preventArcs;
	/** vias to use to go up from each metal layer. */			private MetalVias[] metalVias;
	/** worst spacing rule for a given metal layer. */			private double[] worstMetalSurround;
	/** minimum spacing between the centers of two vias. */		private double[] viaSurround;
	/** cache of connectivity between pure-layer nodes */		private Map<Cell,Map<NodeInst,Network>> nodeExtractions;
	/** the total length of wires routed */						protected double totalWireLength;
	/** true to run to/from and from/to routing in parallel */	protected boolean parallelDij;
	/** for logging errors */									protected ErrorLogger errorLogger;
	/** preferences */											private SeaOfGates.SeaOfGatesOptions prefs;

	/************************************** CONTROL **************************************/

	/**
	 * This is the public interface for Sea-of-Gates Routing when done in batch mode.
	 * @param cell the cell to be Sea-of-Gates-routed.
	 * @param arcsToRoute a List of ArcInsts on networks to be routed.
	 */
	public int routeIt(Job job, Cell cell, List<ArcInst> arcsToRoute)
	{
		// initialize information about the technology
		if (initializeDesignRules(cell, prefs)) return 0;

		EditingPreferences ep = cell.getEditingPreferences();

		// user-interface initialization
		prefs.theTimer = ElapseTimer.createInstance().start();
		if (!RoutingDebug.isActive())
		{
			Job.getUserInterface().startProgressDialog("Routing " + arcsToRoute.size() + " nets", null);
			Job.getUserInterface().setProgressNote("Building blockage information...");
		}

		// create an error logger
		errorLogger = ErrorLogger.newInstance("Routing (Sea of gates)");

		// get all blockage information into R-Trees
		metalTrees = new HashMap<Layer, RTNode>();
		viaTrees = new HashMap<Layer, RTNode>();
		netIDs = new HashMap<ArcInst, Integer>();
		nodeExtractions = new HashMap<Cell,Map<NodeInst,Network>>();
//ArcInst ai = arcsToRoute.get(0);
//NodeInst ni = ai.getHeadPortInst().getNodeInst();
//AffineTransform x = ni.transformIn();
//Point2D goIn = new Point2D.Double();
//infoAreaX = ai.getHeadLocation().getX();
//infoAreaY = ai.getHeadLocation().getY();
//x.transform(ai.getHeadLocation(), goIn);
//infoAreaXLow = goIn.getX();
//infoAreaYLow = goIn.getY();
//System.out.println("EXAMINING POINT ("+infoAreaX+","+infoAreaY+") WHICH, INSIDE CELL, IS ("+infoAreaXLow+","+infoAreaYLow+")");
		BlockageVisitor visitor = new BlockageVisitor(arcsToRoute);
		HierarchyEnumerator.enumerateCell(cell, VarContext.globalContext, visitor);
		addBlockagesAtPorts(arcsToRoute, prefs);

//		for (Iterator<Layer> it = metalTrees.keySet().iterator(); it.hasNext();)
//		{
//			Layer layer = it.next();
//			RTNode root = metalTrees.get(layer);
//			System.out.println("RTree for " + layer.getName() + ":");
//			root.printRTree(2);
//		}

		// make a list of all routes that are needed
		List<NeededRoute> allRoutes = new ArrayList<NeededRoute>();
		int numBatches = arcsToRoute.size();
		RouteBatches[] routeBatches = new RouteBatches[numBatches];

		this.makeListOfRoutes(numBatches, routeBatches, allRoutes, arcsToRoute, ep);

		if (RoutingDebug.isDisplayEndBlockages())
		{
			RoutingDebug.showGeometryAtRouteEnds(allRoutes.get(0));
			return 0;
		}

		if (RoutingDebug.isActive())
		{
			ElectricWorkerStrategy electricWorker = new ElectricWorkerStrategy(null);
			electricWorker.setUi(Job.getUserInterface());
			PoolWorkerStrategyFactory.userDefinedStrategy = electricWorker;

			RoutingDebug.debugRoute(allRoutes.get(0));
			return 0;
		}

		// now do the actual routing
		boolean parallel = prefs.useParallelRoutes;
		parallelDij = prefs.useParallelFromToRoutes;

		firstFailure = true;
		totalWireLength = 0;
		int numberOfProcessors = Runtime.getRuntime().availableProcessors();
		if (numberOfProcessors <= 1) parallelDij = false;
		
		int numberOfThreads = numberOfProcessors;

		// user's input overrides thread-count computation
		if (prefs.forcedNumberOfThreads > 0)
		{
			System.out.println("Forcing use of " + prefs.forcedNumberOfThreads + " threads");
			numberOfThreads = prefs.forcedNumberOfThreads;
		}
		if (parallelDij && numberOfThreads > 1) numberOfThreads /= 2;
		if (!parallel) numberOfThreads = 1;
		if (numberOfThreads == 1) parallel = false;
		
		// show what is being done
		System.out.println("Sea-of-gates router finding " + allRoutes.size() + " paths on " + numBatches + " networks");
		if (parallel || parallelDij)
		{
			String message = "NOTE: System has " + numberOfProcessors + " processors so";
			if (parallel) message += " routing " + numberOfThreads + " paths in parallel";
			if (parallelDij)
			{
				if (parallel) message += " and";
				message += " routing both directions of each path in parallel";
			}
			System.out.println(message);
		}

		// do the routing
		Environment env = Environment.getThreadEnvironment();
		startRouting(numberOfThreads, allRoutes, routeBatches, env, ep, job);

		// finally analyze the results and remove unrouted arcs
		int numRoutedSegments = 0;
		int totalRoutes = allRoutes.size();
		for (int a = 0; a < totalRoutes; a++)
		{
			NeededRoute nr = allRoutes.get(a);
			if (nr.winningWF != null && nr.winningWF.vertices != null)
			{
				routeBatches[nr.batchNumber].numRouted++;
				numRoutedSegments++;
			} else
				routeBatches[nr.batchNumber].numUnrouted++;
		}
		int numFailedRoutes = 0;
		for (int b = 0; b < numBatches; b++)
		{
			if (routeBatches[b] == null) continue;
			if (routeBatches[b].numUnrouted == 0)
			{
				// routed: remove the unrouted arcs
				for (ArcInst aiKill : routeBatches[b].unroutedArcs)
					if (aiKill.isLinked()) aiKill.kill();
				cell.killNodes(routeBatches[b].unroutedNodes);
			} else
			{
				numFailedRoutes++;

				// remove arcs that are routed
				List<PortInst> orderedPorts = routeBatches[b].orderedPorts;
				for (ArcInst aiKill : routeBatches[b].unroutedArcs)
				{
					int headPort = -1, tailPort = -1;
					for (int i = 0; i < orderedPorts.size(); i++)
					{
						PortInst pi = orderedPorts.get(i);
						if (aiKill.getHeadPortInst() == pi) headPort = i; else
							if (aiKill.getTailPortInst() == pi) tailPort = i;
					}
					if (headPort >= 0 && tailPort >= 0)
					{
						boolean allRouted = true;
						if (headPort > tailPort)
						{
							int swap = headPort;
							headPort = tailPort;
							tailPort = swap;
						}
						for (NeededRoute nr : allRoutes)
						{
							if (nr.batchNumber != b) continue;
							if (nr.routeInBatch - 1 < headPort || nr.routeInBatch - 1 >= tailPort) continue;
							if (nr.winningWF == null || nr.winningWF.vertices == null) allRouted = false;
						}
						if (allRouted && aiKill.isLinked()) aiKill.kill();
					}
				}
			}
		}

		// clean up at end
		errorLogger.termLogging(true);
		prefs.theTimer.end();
		Job.getUserInterface().stopProgressDialog();
		System.out.println("Routed " + numRoutedSegments + " out of " + totalRoutes +
			" segments; total length of routed wires is " + TextUtils.formatDistance(totalWireLength) + "; took " + prefs.theTimer);
		if (numFailedRoutes > 0)
			System.out.println("NOTE: " + numFailedRoutes + " nets were not routed");
		return numRoutedSegments;
	}

	/**
	 * Method to set the preferences in this SeaOfGatesEngine.
	 * @param p Preferences to use for routing.
	 */
	public void setPrefs(SeaOfGates.SeaOfGatesOptions p) { prefs = p; }

	/**
	 * Method to return the preferences in this SeaOfGatesEngine.
	 * @return the preferences to use for routing.
	 */
	public SeaOfGates.SeaOfGatesOptions getPrefs() { return prefs; }

	/**
	 * Method to return the number of metal layers being considered in routing.
	 * @return the number of metal layers being considered in routing.
	 */
	public int getNumMetals() { return numMetalLayers; }

	/**
	 * Method to return the metal Layer associated with a layer number.
	 * @param layNum a layer number, from 0 to getNumMetals()-1.
	 * @return the metal Layer being used on the layer number.
	 */
	public Layer getMetalLayer(int layNum) { return metalLayers[layNum]; }

	/**
	 * Method to return an R-Tree of blockages on a given metal Layer.
	 * @param lay the metal Layer to examine.
	 * @return an RTNode that is the top of the tree of blockages on that Layer.
	 */
	public RTNode getMetalTree(Layer lay) { return metalTrees.get(lay); }

	/**
	 * Method to do routing.
	 * @param numberOfThreads number of processors to use.
	 * @param allRoutes List of all routes that need to be done.
	 */
	protected void startRouting(int numberOfThreads, List<NeededRoute> allRoutes,
		RouteBatches[] routeBatches, Environment env, EditingPreferences ep, Job job)
	{
		ElectricWorkerStrategy electricWorker = new ElectricWorkerStrategy(null);
		electricWorker.setUi(Job.getUserInterface());
		PoolWorkerStrategyFactory.userDefinedStrategy = electricWorker;

		if (numberOfThreads > 1)
		{
			doRoutingParallel(numberOfThreads, allRoutes, routeBatches, env, ep, job);
		} else
		{
			doRouting(allRoutes, routeBatches, job, env, ep);
		}
	}

	protected abstract void doRoutingParallel(int numberOfThreads, List<NeededRoute> allRoutes,
		RouteBatches[] routeBatches, Environment env, EditingPreferences ep, Job job);

	/**
	 * Method to do the routing in a single thread.
	 * @param allRoutes the routes that need to be done.
	 * @param routeBatches the routing batches (by network)
	 * @param job the job that invoked this routing.
	 */
	protected void doRouting(List<NeededRoute> allRoutes, RouteBatches[] routeBatches, Job job,
		Environment env, EditingPreferences ep)
	{
		int totalRoutes = allRoutes.size();
		for (int r = 0; r < totalRoutes; r++)
		{
			if (job != null && job.checkAbort())
			{
				System.out.println("Sea-of-gates routing aborted");
				break;
			}

			// get information on the segment to be routed
			NeededRoute nr = allRoutes.get(r);
			Job.getUserInterface().setProgressValue(r * 100 / totalRoutes);
			String routeName = nr.routeName;
			if (routeBatches[nr.batchNumber].segsInBatch > 1)
				routeName += " (" + nr.routeInBatch + " of " + routeBatches[nr.batchNumber].segsInBatch + ")";
			Job.getUserInterface().setProgressNote("Network " + routeName);
			System.out.println("Routing network " + routeName + "...");

			// route the segment
			findPath(nr, env, ep, job);

			// if the routing was good, place the results
			if (nr.winningWF != null && nr.winningWF.vertices != null)
				nr.createRoute();
		}
	}

	/************************************** NEEDEDROUTE: A ROUTE TO BE RUN **************************************/

	/**
	 * Class to hold a route that must be run.
	 */
	public class NeededRoute
	{
		String routeName;
		int netID;
		double minWidth;
		int batchNumber;
		int routeInBatch;
		Rectangle2D routeBounds;
		double minimumSearchBoundX;
		double maximumSearchBoundX;
		double minimumSearchBoundY;
		double maximumSearchBoundY;
		Rectangle2D jumpBound;
		Wavefront dir1, dir2;
		PortInst fromPi, toPi;
		protected Wavefront winningWF;
		SeaOfGates.SeaOfGatesOptions prefs;

		NeededRoute(String routeName, PortInst from, double fromX, double fromY, int fromZ, PortInst to,
			double toX, double toY, int toZ, int netID, double minWidth, int batchNumber,
			int routeInBatch, SeaOfGates.SeaOfGatesOptions prefs)
		{
			this.routeName = routeName;
			this.netID = netID;
			this.minWidth = minWidth;
			this.batchNumber = batchNumber;
			this.routeInBatch = routeInBatch;
			fromPi = from;
			toPi = to;
			this.prefs = prefs;

			cellBounds = cell.getBounds();
			minimumSearchBoundX = downToGrain(cellBounds.getMinX());
			maximumSearchBoundX = upToGrain(cellBounds.getMaxX());
			minimumSearchBoundY = downToGrain(cellBounds.getMinY());
			maximumSearchBoundY = upToGrain(cellBounds.getMaxY());
			if (PERCENTLIMIT > 0)
			{
				double maxStrayFromRouteBounds = Math.min(cellBounds.getWidth(), cellBounds.getHeight()) * PERCENTLIMIT / 100;
				routeBounds = new Rectangle2D.Double(Math.min(fromX, toX) - maxStrayFromRouteBounds,
					Math.min(fromY, toY) - maxStrayFromRouteBounds,
					Math.abs(fromX - toX) + maxStrayFromRouteBounds * 2,
					Math.abs(fromY - toY) + maxStrayFromRouteBounds * 2);
				minimumSearchBoundX = routeBounds.getMinX();
				maximumSearchBoundX = routeBounds.getMaxX();
				minimumSearchBoundY = routeBounds.getMinY();
				maximumSearchBoundY = routeBounds.getMaxY();
			}
			jumpBound = new Rectangle2D.Double(Math.min(fromX, toX), Math.min(fromY, toY),
				Math.abs(fromX - toX), Math.abs(fromY - toY));

			// make two wavefronts going in both directions

			if (RoutingDebug.isActive())
				RoutingDebug.setRouteDescription("Debugging search from (" + TextUtils.formatDouble(fromX) + "," +
					TextUtils.formatDouble(fromY) + ",M" + (fromZ + 1) + ") to (" +
					TextUtils.formatDouble(toX) + "," + TextUtils.formatDouble(toY) + ",M" + (toZ + 1) + ")");
			dir1 = new Wavefront(this, from, fromX, fromY, fromZ, to, toX, toY, toZ, "a->b");
			dir2 = new Wavefront(this, to, toX, toY, toZ, from, fromX, fromY, fromZ, "b->a");
		}

		/**
		 * Method to return the Wavefront that starts at the "from" point and progress toward the "to" point.
		 * @return the Wavefront that starts at the "from" point and progress toward the "to" point.
		 */
		public Wavefront getAtoBDirection() { return dir1; }

		/**
		 * Method to return the Wavefront that starts at the "to" point and progress toward the "from" point.
		 * @return the Wavefront that starts at the "to" point and progress toward the "from" point.
		 */
		public Wavefront getBtoADirection() { return dir2; }

		/**
		 * Method to return an R-Tree of blockages on a given metal Layer.
		 * @param lay the metal Layer to examine.
		 * @return an RTNode that is the top of the tree of blockages on that Layer.
		 */
		public RTNode getMetalTree(Layer lay) { return metalTrees.get(lay); }

		/**
		 * Method to create the geometry for a route. Places nodes and arcs to make
		 * the route, and also updates the R-Tree data structure.
		 */
		protected void createRoute()
		{
			Wavefront wf = winningWF;
			PortInst lastPort = wf.to;
			PolyBase toPoly = wf.to.getPoly();
			if (toPoly.getCenterX() != wf.toX || toPoly.getCenterY() != wf.toY)
			{
				// end of route is off-grid: adjust it
				if (wf.vertices.size() >= 2)
				{
					SearchVertex v1 = wf.vertices.get(0);
					SearchVertex v2 = wf.vertices.get(1);
					ArcProto type = metalArcs[wf.toZ];
					double width = Math.max(type.getDefaultLambdaBaseWidth(), minWidth);
					PrimitiveNode np = metalArcs[wf.toZ].findPinProto();
					if (v1.getX() == v2.getX())
					{
						// first line is vertical: run a horizontal bit
						NodeInst ni = makeNodeInst(np, new EPoint(v1.getX(), toPoly.getCenterY()),
							np.getDefWidth(), np.getDefHeight(), Orientation.IDENT, cell, netID);
						makeArcInst(type, width, ni.getOnlyPortInst(), wf.to, netID);
						lastPort = ni.getOnlyPortInst();
					} else if (v1.getY() == v2.getY())
					{
						// first line is horizontal: run a vertical bit
						NodeInst ni = makeNodeInst(np, new EPoint(toPoly.getCenterX(), v1.getY()),
							np.getDefWidth(), np.getDefHeight(), Orientation.IDENT, cell, netID);
						makeArcInst(type, width, ni.getOnlyPortInst(), wf.to, netID);
						lastPort = ni.getOnlyPortInst();
					}
				}
			}
			for (int i = 0; i < wf.vertices.size(); i++)
			{
				SearchVertex sv = wf.vertices.get(i);
				boolean madeContacts = false;
				while (i < wf.vertices.size() - 1)
				{
					SearchVertex svNext = wf.vertices.get(i + 1);
					if (sv.getX() != svNext.getX() || sv.getY() != svNext.getY() || sv.getZ() == svNext.getZ()) break;
					List<MetalVia> nps = metalVias[Math.min(sv.getZ(), svNext.getZ())].getVias();
					int whichContact = sv.getContactNo();
					MetalVia mv = nps.get(whichContact);
					PrimitiveNode np = mv.via;
					Orientation orient = Orientation.fromJava(mv.orientation * 10, false, false);
					SizeOffset so = np.getProtoSizeOffset();
					double xOffset = so.getLowXOffset() + so.getHighXOffset();
					double yOffset = so.getLowYOffset() + so.getHighYOffset();
					double wid = Math.max(np.getDefWidth() - xOffset, minWidth) + xOffset;
					double hei = Math.max(np.getDefHeight() - yOffset, minWidth) + yOffset;
					NodeInst ni = makeNodeInst(np, new EPoint(sv.getX(), sv.getY()), wid, hei, orient, cell, netID);
					ArcProto type = metalArcs[sv.getZ()];
					double width = Math.max(type.getDefaultLambdaBaseWidth(), minWidth);
					makeArcInst(type, width, lastPort, ni.getOnlyPortInst(), netID);
					lastPort = ni.getOnlyPortInst();
					madeContacts = true;
					sv = svNext;
					i++;
				}
				if (madeContacts && i != wf.vertices.size() - 1) continue;

				PrimitiveNode np = metalArcs[sv.getZ()].findPinProto();
				PortInst pi = null;
				if (i == wf.vertices.size() - 1)
				{
					pi = wf.from;
					PolyBase fromPoly = wf.from.getPoly();
					if (fromPoly.getCenterX() != sv.getX() || fromPoly.getCenterY() != sv.getY())
					{
						// end of route is off-grid: adjust it
						if (wf.vertices.size() >= 2)
						{
							SearchVertex v1 = wf.vertices.get(wf.vertices.size() - 2);
							SearchVertex v2 = wf.vertices.get(wf.vertices.size() - 1);
							ArcProto type = metalArcs[wf.fromZ];
							double width = Math.max(type.getDefaultLambdaBaseWidth(), minWidth);
							if (v1.getX() == v2.getX())
							{
								// last line is vertical: run a horizontal bit
								PrimitiveNode pNp = metalArcs[wf.fromZ].findPinProto();
								NodeInst ni = makeNodeInst(pNp, new EPoint(v1.getX(), fromPoly.getCenterY()),
									np.getDefWidth(), np.getDefHeight(), Orientation.IDENT, cell, netID);
								makeArcInst(type, width, ni.getOnlyPortInst(), wf.from, netID);
								pi = ni.getOnlyPortInst();
							} else if (v1.getY() == v2.getY())
							{
								// last line is horizontal: run a vertical bit
								PrimitiveNode pNp = metalArcs[wf.fromZ].findPinProto();
								NodeInst ni = makeNodeInst(pNp, new EPoint(fromPoly.getCenterX(), v1.getY()),
									np.getDefWidth(), np.getDefHeight(), Orientation.IDENT, cell, netID);
								makeArcInst(type, width, ni.getOnlyPortInst(), wf.from, netID);
								pi = ni.getOnlyPortInst();
							}
						}
					}
				} else
				{
					NodeInst ni = makeNodeInst(np, new EPoint(sv.getX(), sv.getY()), np.getDefWidth(),
						np.getDefHeight(), Orientation.IDENT, cell, netID);
					pi = ni.getOnlyPortInst();
				}
				if (lastPort != null)
				{
					ArcProto type = metalArcs[sv.getZ()];
					double width = Math.max(type.getDefaultLambdaBaseWidth(), minWidth);
					makeArcInst(type, width, lastPort, pi, netID);
				}
				lastPort = pi;
			}

			// now see how far out of the bounding rectangle the route went
			Rectangle2D routeBounds = new Rectangle2D.Double(Math.min(wf.fromX, wf.toX), Math.min(wf.fromY,
				wf.toY), Math.abs(wf.fromX - wf.toX), Math.abs(wf.fromY - wf.toY));
			double lowX = routeBounds.getMinX(), highX = routeBounds.getMaxX();
			double lowY = routeBounds.getMinY(), highY = routeBounds.getMaxY();
			for (int i = 0; i < wf.vertices.size(); i++)
			{
				SearchVertex sv = wf.vertices.get(i);
				if (i == 0)
				{
					lowX = highX = sv.getX();
					lowY = highY = sv.getY();
				} else
				{
					if (sv.getX() < lowX) lowX = sv.getX();
					if (sv.getX() > highX) highX = sv.getX();
					if (sv.getY() < lowY) lowY = sv.getY();
					if (sv.getY() > highY) highY = sv.getY();
				}
			}
		}

		public void cleanSearchMemory()
		{
			dir1.searchVertexPlanes = null;
			dir1.searchVertexPlanesDBL = null;
			dir1.active = null;
			dir1.inactive = null;
			if (dir1.vertices != null)
			{
				for (SearchVertex sv : dir1.vertices) sv.clearCuts();
			}

			dir2.searchVertexPlanes = null;
			dir2.searchVertexPlanesDBL = null;
			dir2.active = null;
			dir2.inactive = null;
			if (dir2.vertices != null)
			{
				for (SearchVertex sv : dir2.vertices) sv.clearCuts();
			}
		}
	}

	/************************************** WAVEFRONT: THE ACTUAL SEARCH CODE **************************************/

	/**
	 * Class to define a routing search that advances a "wave" of search coordinates from the starting point
	 * to the ending point.
	 */
	public class Wavefront
	{
		/** The route that this is part of. */							NeededRoute nr;
		/** Wavefront name (for debugging). */							String name;
		/** Active search vertices while running wavefront. */			private TreeSet<SearchVertex> active;
		/** Used search vertices while running wavefront (debug). */	private List<SearchVertex> inactive;
		/** Resulting list of vertices found for this wavefront. */		List<SearchVertex> vertices;
		/** Set true to abort this wavefront's search. */				boolean abort;
		/** The starting and ending ports of the wavefront. */			PortInst from, to;
		/** The starting X/Y coordinates of the wavefront. */			double fromX, fromY;
		/** The starting metal layer of the wavefront. */				int fromZ;
		/** The ending X/Y coordinates of the wavefront. */				double toX, toY;
		/** The ending metal layer of the wavefront. */					int toZ;
		/** Count of the number of wavefront advances made. */			int numStepsMade;
		/** Search vertices found while running the wavefront. */		Map<Integer, Set<Integer>>[] searchVertexPlanes;
		/** Gridless search vertices found while running wavefront. */	Map<Double, Set<Double>>[] searchVertexPlanesDBL;
		/** minimum spacing between this metal and itself. */			private Map<Double, Map<Double, Double>>[] layerSurround;

		Wavefront(NeededRoute nr, PortInst from, double fromX, double fromY, int fromZ, PortInst to,
			double toX, double toY, int toZ, String name)
		{
			this.nr = nr;
			this.from = from;
			this.fromX = fromX;
			this.fromY = fromY;
			this.fromZ = fromZ;
			this.to = to;
			this.toX = toX;
			this.toY = toY;
			this.toZ = toZ;
			this.name = name;
			this.numStepsMade = 0;
			active = new TreeSet<SearchVertex>();
			inactive = new ArrayList<SearchVertex>();
			vertices = null;
			abort = false;
			searchVertexPlanes = new Map[numMetalLayers];
			searchVertexPlanesDBL = new Map[numMetalLayers];
			layerSurround = new Map[numMetalLayers];
			for (int i = 0; i < numMetalLayers; i++)
				layerSurround[i] = new HashMap<Double, Map<Double, Double>>();

			SearchVertex svStart = new SearchVertex(fromX, fromY, fromZ, 0, null, 0, this);
			svStart.cost = 0;
			setVertex(fromX, fromY, fromZ);
			if (RoutingDebug.isActive())
				RoutingDebug.identifyNewDebugPoint(svStart);
			active.add(svStart);
		}

		public PortInst getFromPortInst() { return from; }

		public PortInst getToPortInst() { return to; }

		public double getFromX() { return fromX; }

		public double getFromY() { return fromY; }

		public int getFromZ() { return fromZ; }

		public double getToX() { return toX; }

		public double getToY() { return toY; }

		public int getToZ() { return toZ; }

		public SearchVertex getNextSearchVertex()
		{
			if (active.size() == 0) return svExhausted;
			return active.first();
		}

		/**
		 * Method to tell whether there is a SearchVertex at a given coordinate.
		 * @param x the X coordinate desired.
		 * @param y the Y coordinate desired.
		 * @param z the Z coordinate (metal layer) desired.
		 * @return true if there is a SearchVertex at that point.
		 */
		public boolean isVertex(double x, double y, int z)
		{
			if (FULLGRAIN)
			{
				Map<Integer, Set<Integer>> plane = searchVertexPlanes[z];
				if (plane == null) return false;
				Set<Integer> row = plane.get(new Integer((int)Math.round(y * DBMath.GRID)));
				if (row == null) return false;
				boolean found = row.contains(new Integer((int)Math.round(x * DBMath.GRID)));
				return found;

//				Map<Double, Set<Double>> plane = searchVertexPlanesDBL[z];
//				if (plane == null) return false;
//				Set<Double> row = plane.get(new Double(y));
//				if (row == null) return false;
//				boolean found = row.contains(new Double(x));
//				return found;
			}

			Map<Integer, Set<Integer>> plane = searchVertexPlanes[z];
			if (plane == null) return false;
			Set<Integer> row = plane.get(new Integer((int) (y * GRANULARITY)));
			if (row == null) return false;
			boolean found = row.contains(new Integer((int) (x * GRANULARITY)));
			return found;
		}

		/**
		 * Method to mark a given coordinate.
		 * @param x the X coordinate desired.
		 * @param y the Y coordinate desired.
		 * @param z the Z coordinate (metal layer) desired.
		 */
		public void setVertex(double x, double y, int z)
		{
			if (FULLGRAIN)
			{
				Map<Integer, Set<Integer>> plane = searchVertexPlanes[z];
				if (plane == null)
					searchVertexPlanes[z] = plane = new HashMap<Integer, Set<Integer>>();
				Integer iY = new Integer((int)Math.round(y * DBMath.GRID));
				Set<Integer> row = plane.get(iY);
				if (row == null)
					plane.put(iY, row = new HashSet<Integer>());
				row.add(new Integer((int)Math.round(x * DBMath.GRID)));

//				Map<Double, Set<Double>> plane = searchVertexPlanesDBL[z];
//				if (plane == null)
//					searchVertexPlanesDBL[z] = plane = new HashMap<Double, Set<Double>>();
//				Double iY = new Double(y);
//				Set<Double> row = plane.get(iY);
//				if (row == null)
//					plane.put(iY, row = new HashSet<Double>());
//				row.add(new Double(x));
				return;
			}

			Map<Integer, Set<Integer>> plane = searchVertexPlanes[z];
			if (plane == null)
				searchVertexPlanes[z] = plane = new HashMap<Integer, Set<Integer>>();
			Integer iY = new Integer((int) (y * GRANULARITY));
			Set<Integer> row = plane.get(iY);
			if (row == null)
				plane.put(iY, row = new HashSet<Integer>());
			row.add(new Integer((int) (x * GRANULARITY)));
		}

		/**
		 * Method to determine the design rule spacing between two pieces of a given layer.
		 * @param layer the layer index.
		 * @param width the width of one of the pieces (-1 to use default).
		 * @param length the length of one of the pieces (-1 to use default).
		 * @return the design rule spacing (0 if none).
		 */
		public double getSpacingRule(int layer, double width, double length)
		{
			// use default width if none specified
			if (width < 0) width = metalArcs[layer].getDefaultLambdaBaseWidth();
			if (length < 0) length = 50;

			// convert these to the next largest integers
			Double wid = new Double(upToGrain(width));
			Double len = new Double(upToGrain(length));

			// see if the rule is cached6
			Map<Double, Double> widMap = layerSurround[layer].get(wid);
			if (widMap == null)
			{
				widMap = new HashMap<Double, Double>();
				layerSurround[layer].put(wid, widMap);
			}
			Double value = widMap.get(len);
			if (value == null)
			{
				// rule not cached: compute it
				Layer lay = metalLayers[layer];
				DRCTemplate rule = DRC.getSpacingRule(lay, null, lay, null, false, -1, width, length);
				double v = 0;
				if (rule != null) v = rule.getValue(0);
				value = new Double(v);
				widMap.put(len, value);
			}
			return value.doubleValue();
		}

		/**
		 * Method to advance a wavefront by a single step.
		 * Takes the first SearchVertex in the WaveFront (the one with the lowest cost) and expands it in 6 directions
		 * (+X, -X, +Y, -Y, +Z, -Z), creating up to 6 new SearchVertex objects on the WaveFront.
		 * @return null for a successful advance, non-null to terminate wavefront (could be a completion of the route
		 * or an error).
		 */
		public SearchVertex advanceWavefront()
		{
			numStepsMade++;
			if (numStepsMade > prefs.complexityLimit) return svLimited;

			// get the lowest cost point
			SearchVertex svCurrent = getNextSearchVertex();
			if (svCurrent == svExhausted) return svCurrent;

			active.remove(svCurrent);
			inactive.add(svCurrent);
			double curX = svCurrent.getX();
			double curY = svCurrent.getY();
			int curZ = svCurrent.getZ();

			String debStr = null;
			if (RoutingDebug.isActive())
				debStr = "AT (" + TextUtils.formatDouble(curX) + "," + TextUtils.formatDouble(curY) +
					",M" + (curZ + 1) + "), Cost=" + svCurrent.cost;

			// see if automatic generation is requested
			if (svCurrent.isAutoGen())
				svCurrent.generateIntermediateVertex();

			// look at all directions from this point
			SearchVertex destinationSV = null;
			for (int i = 0; i < 6; i++)
			{
				// compute a neighboring point
				double dx = 0, dy = 0;
				int dz = 0;
				switch (i)
				{
					case 0:
						dx = -GRAINSIZE;
						if (FULLGRAIN)
						{
							double intermediate = upToGrainAlways(curX + dx);
							if (intermediate != curX + dx) dx = intermediate - curX;
						}
						break;
					case 1:
						dx = GRAINSIZE;
						if (FULLGRAIN)
						{
							double intermediate = downToGrainAlways(curX + dx);
							if (intermediate != curX + dx) dx = intermediate - curX;
						}
						break;
					case 2:
						dy = -GRAINSIZE;
						if (FULLGRAIN)
						{
							double intermediate = upToGrainAlways(curY + dy);
							if (intermediate != curY + dy) dy = intermediate - curY;
						}
						break;
					case 3:
						dy = GRAINSIZE;
						if (FULLGRAIN)
						{
							double intermediate = downToGrainAlways(curY + dy);
							if (intermediate != curY + dy) dy = intermediate - curY;
						}
						break;
					case 4:
						dz = -1;
						break;
					case 5:
						dz = 1;
						break;
				}

				// extend the distance if heading toward the goal
				boolean stuck = false;
				if (dz == 0)
				{
					boolean goFarther = false;
					if (dx != 0)
					{
						if ((toX - curX) * dx > 0) goFarther = true;

//						// if on wrong metal layer, only go incrementally to grid
//						if ((curZ % 2) == 0) goFarther = false;
					} else
					{
						if ((toY - curY) * dy > 0) goFarther = true;

//						// if on wrong metal layer, only go incrementally to grid
//						if ((curZ % 2) != 0) goFarther = false;
					}
					if (goFarther)
					{
						double jumpSize = getJumpSize(curX, curY, curZ, dx, dy);
						if (dx > 0)
						{
							if (jumpSize <= 0) stuck = true;
							dx = jumpSize;
//							if (FULLGRAIN && curX+dx != toX && curY+dy != toY) dx = downToGrainAlways(curX + dx) - curX;
						}
						if (dx < 0)
						{
							if (jumpSize >= 0) stuck = true;
							dx = jumpSize;
//							if (FULLGRAIN && curX+dx != toX && curY+dy != toY) dx = downToGrainAlways(curX + dx) - curX;
						}
						if (dy > 0)
						{
							if (jumpSize <= 0) stuck = true;
							dy = jumpSize;
//							if (FULLGRAIN && curX+dx != toX && curY+dy != toY) dy = downToGrainAlways(curY + dy) - curY;
						}
						if (dy < 0)
						{
							if (jumpSize >= 0) stuck = true;
							dy = jumpSize;
//							if (FULLGRAIN && curX+dx != toX && curY+dy != toY) dy = downToGrainAlways(curY + dy) - curY;
						}
					}
				}
				double nX = curX + dx;
				double nY = curY + dy;
				int nZ = curZ + dz;

				if (RoutingDebug.isActive())
				{
					switch (i)
					{
						case 0: debStr += "/Move X -" + TextUtils.formatDouble(Math.abs(dx));  break;
						case 1: debStr += "/Move X +" + TextUtils.formatDouble(dx);            break;
						case 2: debStr += "/Move Y -" + TextUtils.formatDouble(Math.abs(dy));  break;
						case 3: debStr += "/Move Y +" + TextUtils.formatDouble(dy);            break;
						case 4: debStr += "/Move down layer";                                  break;
						case 5: debStr += "/Move up layer";                                    break;
					}
				}
				if (stuck)
				{
					if (RoutingDebug.isActive()) debStr += ": Cannot Move";
					continue;
				}

				if (nX < nr.minimumSearchBoundX)
				{
					nX = nr.minimumSearchBoundX;
					dx = nX - curX;
					if (dx == 0)
					{
						if (RoutingDebug.isActive()) debStr += ": Out Of Bounds";
						continue;
					}
				}
				if (nX > nr.maximumSearchBoundX)
				{
					nX = nr.maximumSearchBoundX;
					dx = nX - curX;
					if (dx == 0)
					{
						if (RoutingDebug.isActive()) debStr += ": Out Of Bounds";
						continue;
					}
				}
				if (nY < nr.minimumSearchBoundY)
				{
					nY = nr.minimumSearchBoundY;
					dy = nY - curY;
					if (dy == 0)
					{
						if (RoutingDebug.isActive()) debStr += ": Out Of Bounds";
						continue;
					}
				}
				if (nY > nr.maximumSearchBoundY)
				{
					nY = nr.maximumSearchBoundY;
					dy = nY - curY;
					if (dy == 0)
					{
						if (RoutingDebug.isActive()) debStr += ": Out Of Bounds";
						continue;
					}
				}
				if (nZ < 0 || nZ >= numMetalLayers)
				{
					if (RoutingDebug.isActive()) debStr += ": Out Of Bounds";
					continue;
				}
				if (preventArcs[nZ])
				{
					if (RoutingDebug.isActive()) debStr += ": Illegal Arc";
					continue;
				}

				// see if the point has already been visited
				if (isVertex(nX, nY, nZ))
				{
					if (RoutingDebug.isActive()) debStr += ": Already Visited";
					continue;
				}

				// see if the space is available
				int whichContact = 0;
				Point2D[] cuts = null;
				if (dz == 0)
				{
					// running on one layer: check surround
					double width = Math.max(metalArcs[nZ].getDefaultLambdaBaseWidth(), nr.minWidth);
					double metalSpacing = width / 2;
					boolean allClear = false;
					double initNX = nX, initNY = nY;
					for(;;)
					{
						SearchVertex prevPath = svCurrent;
						double checkX = (curX + nX) / 2, checkY = (curY + nY) / 2;
						double halfWid = metalSpacing + Math.abs(dx) / 2;
						double halfHei = metalSpacing + Math.abs(dy) / 2;
						while (prevPath != null && prevPath.last != null)
						{
							if (prevPath.zv != nZ || prevPath.last.zv != nZ) break;
							if (prevPath.xv == prevPath.last.xv && dx == 0)
							{
								checkY = (prevPath.last.yv + nY) / 2;
								halfHei = metalSpacing + Math.abs(prevPath.last.yv - nY) / 2;
								prevPath = prevPath.last;
							} else if (prevPath.yv == prevPath.last.yv && dy == 0)
							{
								checkX = (prevPath.last.xv + nX) / 2;
								halfWid = metalSpacing + Math.abs(prevPath.last.xv - nX) / 2;
								prevPath = prevPath.last;
							} else
								break;
						}
						SOGBound sb = getMetalBlockageAndNotch(nZ, halfWid, halfHei, checkX, checkY, prevPath);
						if (sb == null)
						{
							allClear = true;
							break;
						}
						if (RoutingDebug.isActive())
							debStr += ": Blocked on M" + (nZ+1) + " because proposed " +
								TextUtils.formatDouble(checkX-halfWid) +
								"<=X<=" + TextUtils.formatDouble(checkX+halfWid) +
								" and " + TextUtils.formatDouble(checkY-halfHei) +
								"<=Y<=" + TextUtils.formatDouble(checkY+halfHei) +
								" conflicts with " + TextUtils.formatDouble(sb.bound.getMinX()) +
								"<=X<=" + TextUtils.formatDouble(sb.bound.getMaxX()) +
								" and " + TextUtils.formatDouble(sb.bound.getMinY()) +
								"<=Y<=" + TextUtils.formatDouble(sb.bound.getMaxY());

						// see if it can be backed out slightly
						if (i == 0)
						{
							// moved left too far...try a bit to the right
							double newNX = downToGrainAlways(nX + GRAINSIZE);
							if (newNX >= curX || DBMath.areEquals(newNX, nX)) break;
							dx = newNX - curX;
							if (dx == 0) break;
						} else if (i == 1)
						{
							// moved right too far...try a bit to the left
							double newNX = upToGrainAlways(nX - GRAINSIZE);
							if (newNX <= curX || DBMath.areEquals(newNX, nX)) break;
							dx = newNX - curX;
							if (dx == 0) break;
						} else if (i == 2)
						{
							// moved down too far...try a bit up
							double newNY = downToGrainAlways(nY + GRAINSIZE);
							if (newNY >= curY || DBMath.areEquals(newNY, nY)) break;
							dy = newNY - curY;
							if (dy == 0) break;
						} else if (i == 3)
						{
							// moved up too far...try a bit down
							double newNY = upToGrainAlways(nY - GRAINSIZE);
							if (newNY <= curY || DBMath.areEquals(newNY, nY)) break;
							dy = newNY - curY;
							if (dy == 0) break;
						}

						// if (!getVertex(nX, nY, nZ)) setVertex(nX, nY, nZ);
						nX = curX + dx;
						nY = curY + dy;
					}
					if (!allClear)
					{
						if (RoutingDebug.isActive())
						{
							double checkX = (curX + nX) / 2, checkY = (curY + nY) / 2;
							double halfWid = metalSpacing + Math.abs(dx) / 2;
							double halfHei = metalSpacing + Math.abs(dy) / 2;
							double surround = worstMetalSurround[nZ];
							SOGBound sb = getMetalBlockage(nr.netID, nZ, halfWid, halfHei, surround, checkX, checkY);
							if (sb != null) debStr += ": Blocked"; else
								debStr += ": Blocked, Notch";
						}
						continue;
					}
					if (RoutingDebug.isActive() && (initNX != nX || initNY != nY))
					{
						debStr += " so move only ";
						switch (i)
						{
							case 0: debStr += TextUtils.formatDouble(Math.abs(dx));  break;
							case 1: debStr += TextUtils.formatDouble(dx);            break;
							case 2: debStr += TextUtils.formatDouble(Math.abs(dy));  break;
							case 3: debStr += TextUtils.formatDouble(dy);            break;
						}
					}
					
				} else
				{
					int lowMetal = Math.min(curZ, nZ);
					int highMetal = Math.max(curZ, nZ);
					List<MetalVia> nps = metalVias[lowMetal].getVias();
					whichContact = -1;
					String[] failureReasons = null;
					if (RoutingDebug.isActive()) failureReasons = new String[nps.size()];
					for (int contactNo = 0; contactNo < nps.size(); contactNo++)
					{
						MetalVia mv = nps.get(contactNo);
						PrimitiveNode np = mv.via;
						Orientation orient = Orientation.fromJava(mv.orientation * 10, false, false);
						SizeOffset so = np.getProtoSizeOffset();
						double conWid = Math.max(np.getDefWidth() - so.getLowXOffset() - so.getHighXOffset(), nr.minWidth) +
							so.getLowXOffset() + so.getHighXOffset();
						double conHei = Math.max(np.getDefHeight() - so.getLowYOffset() - so.getHighYOffset(), nr.minWidth) +
							so.getLowYOffset() + so.getHighYOffset();
						NodeInst dummyNi = NodeInst.makeDummyInstance(np, new EPoint(nX, nY), conWid, conHei, orient);
						Poly[] conPolys = tech.getShapeOfNode(dummyNi);
						AffineTransform trans = null;
						if (orient != Orientation.IDENT) trans = dummyNi.rotateOut();

						// count the number of cuts and make an array for the data
						int cutCount = 0;
						for (int p = 0; p < conPolys.length; p++)
							if (conPolys[p].getLayer().getFunction().isContact()) cutCount++;
						Point2D[] curCuts = new Point2D[cutCount];
						cutCount = 0;
						String failedReason = null;
						for (int p = 0; p < conPolys.length; p++)
						{
							Poly conPoly = conPolys[p];
							if (trans != null) conPoly.transform(trans);
							Layer conLayer = conPoly.getLayer();
							Layer.Function lFun = conLayer.getFunction();
							if (lFun.isMetal())
							{
								Rectangle2D conRect = conPoly.getBounds2D();
								int metalNo = lFun.getLevel() - 1;
								double halfWid = conRect.getWidth() / 2;
								double halfHei = conRect.getHeight() / 2;
								if (getMetalBlockageAndNotch(metalNo, halfWid, halfHei, conRect.getCenterX(),
									conRect.getCenterY(), svCurrent) != null)
								{
									if (failureReasons == null) failedReason = ""; else
										failedReason = "layer " + conLayer.getName() +
											" at " + TextUtils.formatDouble(conRect.getMinX()) +
											"<=X<=" + TextUtils.formatDouble(conRect.getMaxX()) +
											" and " + TextUtils.formatDouble(conRect.getMinY()) +
											"<=Y<=" + TextUtils.formatDouble(conRect.getMaxY());
									break;
								}
							} else if (lFun.isContact())
							{
								// make sure vias don't get too close
								Rectangle2D conRect = conPoly.getBounds2D();
								double conCX = conRect.getCenterX();
								double conCY = conRect.getCenterY();
								double surround = viaSurround[lowMetal];
								if (getViaBlockage(nr.netID, conLayer, surround, surround, conCX, conCY) != null)
								{
									if (failureReasons == null) failedReason = ""; else
										failedReason = "cut " + conLayer.getName() +
											" at " + TextUtils.formatDouble(conRect.getMinX()) +
											"<=X<=" + TextUtils.formatDouble(conRect.getMaxX()) +
											" and " + TextUtils.formatDouble(conRect.getMinY()) +
											"<=Y<=" + TextUtils.formatDouble(conRect.getMaxY());
									break;
								}
								curCuts[cutCount++] = new Point2D.Double(conCX, conCY);

								// look at all previous cuts in this path
								for (SearchVertex sv = svCurrent; sv != null; sv = sv.last)
								{
									SearchVertex lastSv = sv.last;
									if (lastSv == null) break;
									if (Math.min(sv.getZ(), lastSv.getZ()) == lowMetal &&
										Math.max(sv.getZ(), lastSv.getZ()) == highMetal)
									{
										// make sure the cut isn't too close
										Point2D[] svCuts;
										if (sv.getCutLayer() == lowMetal) svCuts = sv.getCuts(); else
											svCuts = lastSv.getCuts();
										if (svCuts != null)
										{
											for (Point2D cutPt : svCuts)
											{
												if (Math.abs(cutPt.getX() - conCX) >= surround ||
													Math.abs(cutPt.getY() - conCY) >= surround) continue;
												if (failureReasons == null) failedReason = ""; else
													failedReason = "path cut " + conLayer.getName() +
														" at " + TextUtils.formatDouble(conRect.getMinX()) +
														"<=X<=" + TextUtils.formatDouble(conRect.getMaxX()) +
														" and " + TextUtils.formatDouble(conRect.getMinY()) +
														"<=Y<=" + TextUtils.formatDouble(conRect.getMaxY());
												break;
											}
										}
										if (failedReason != null) break;
									}
								}
								if (failedReason != null) break;
							}
						}
						if (failedReason != null)
						{
							if (RoutingDebug.isActive()) failureReasons[contactNo] = failedReason;
							continue;
						}
						whichContact = contactNo;
						cuts = curCuts;
						break;
					}
					if (whichContact < 0)
					{
						if (RoutingDebug.isActive())
						{
							debStr += ": Blocked because";
							for(int contactNo = 0; contactNo < failureReasons.length; contactNo++)
							{
								MetalVia mv = nps.get(contactNo);
								debStr += "|In " + mv.via.describe(false) + " (rotated " + mv.orientation + ") cannot place " +
									failureReasons[contactNo];
							}
						}
						continue;
					}
				}

				// see if it found the destination
				boolean foundDest = DBMath.areEquals(nX, toX) && DBMath.areEquals(nY, toY) && nZ == toZ;

				// ignore if a large jump on the wrong metal layer
				if (foundDest)
				{
					if (Math.abs(dx) > 10 && (nZ % 2) == 0) continue;
					if (Math.abs(dy) > 10 && (nZ % 2) != 0) continue;
				}

				// we have a candidate next-point
				SearchVertex svNext = new SearchVertex(nX, nY, nZ, whichContact, cuts, Math.min(curZ, nZ), this);
				if (dz == 0 && (Math.abs(dx) >= 2 || Math.abs(dy) >= 2)) svNext.setAutoGen(true);
				svNext.last = svCurrent;
				
				if (foundDest)
				{
					if (RoutingDebug.isActive()) debStr += ": Found Destination!";
					destinationSV = svNext;
					continue;
				}

				// compute the cost
				svNext.cost = svCurrent.cost;
				String costExplanation = "";
				boolean penaltyOffGridX = downToGrainAlways(nX) != nX && nX != toX;
				boolean penaltyOffGridY = downToGrainAlways(nY) != nY && nY != toY;

				double distBefore = Math.sqrt((curX-toX)*(curX-toX) + (curY-toY)*(curY-toY));
				double distAfter = Math.sqrt((nX-toX)*(nX-toX) + (nY-toY)*(nY-toY));
				int c = (int)((distAfter - distBefore) / 5);
				svNext.cost += c;
				if (RoutingDebug.isActive()) costExplanation = " [Dist-to-target=" + c;

				if (dx != 0)
				{
					if (toX == curX)
					{
						c = COSTWRONGDIRECTION / 2;
						svNext.cost += c;
						if (RoutingDebug.isActive()) costExplanation += " Zero-X-progress=" + c;
					} else if ((toX - curX) * dx < 0)
					{
						if (toY == curY)
						{
							c = 1;
							svNext.cost += c;
							if (RoutingDebug.isActive()) costExplanation += " Backward-X-progress-at-dest-Y=" + c;
						} else
						{
							c = COSTWRONGDIRECTION;
							svNext.cost += c;
							if (RoutingDebug.isActive()) costExplanation += " Backward-X-progress=" + c;
						}
					}
					if ((nZ % 2) == 0)
					{
						c = COSTALTERNATINGMETAL * (int)Math.abs(dx);
						svNext.cost += c;
						if (RoutingDebug.isActive()) costExplanation += " Not-alternating-metal=" + c;
					}
				}
				if (dy != 0)
				{
					if (toY == curY)
					{
						c = COSTWRONGDIRECTION / 2;
						svNext.cost += c;
						if (RoutingDebug.isActive()) costExplanation += " Zero-Y-progress=" + c;
					} else if ((toY - curY) * dy < 0)
					{
						if (toX == curX)
						{
							c = 1;
							svNext.cost += c;
							if (RoutingDebug.isActive()) costExplanation += " Backward-Y-progress-at-dest-X=" + c;
						} else
						{
							c = COSTWRONGDIRECTION;
							svNext.cost += c;
							if (RoutingDebug.isActive()) costExplanation += " Backward-Y-progress=" + c;
						}
					}
					if ((nZ % 2) != 0)
					{
						c = COSTALTERNATINGMETAL * (int)Math.abs(dy);
						svNext.cost += c;
						if (RoutingDebug.isActive()) costExplanation += " Not-alternating-metal=" + c;
					}
				}
				if (dz != 0)
				{
					if (toZ == curZ)
					{
						c = COSTLAYERCHANGE;
						svNext.cost += c;
						if (RoutingDebug.isActive()) costExplanation += " Layer-change=" + c;
					} else if ((toZ - curZ) * dz < 0)
					{
						c = COSTLAYERCHANGE + 1;
						svNext.cost += c;
						if (RoutingDebug.isActive()) costExplanation += " Layer-change-wrong-direction=" + c;
					}
				} else
				{
					// not changing layers: compute penalty for unused tracks on either side of run
					double jumpSize1 = Math.abs(getJumpSize(nX, nY, nZ, dx, dy));
					double jumpSize2 = Math.abs(getJumpSize(curX, curY, curZ, -dx, -dy));
					if (jumpSize1 > GRAINSIZE && jumpSize2 > GRAINSIZE)
					{
						c = (int)((jumpSize1 * jumpSize2) / 10);
						svNext.cost += c;
						if (RoutingDebug.isActive()) costExplanation += " Fragments-track=" + c;
					}

					// not changing layers: penalize if turning in X or Y
					if (svCurrent.last != null)
					{
						boolean xTurn = svCurrent.getX() != svCurrent.last.getX();
						boolean yTurn = svCurrent.getY() != svCurrent.last.getY();
						if (xTurn != (dx != 0) || yTurn != (dy != 0))
						{
							c = COSTTURNING;
							svNext.cost += c;
							if (RoutingDebug.isActive()) costExplanation += " Turning=" + c;
						}
					}
				}
				if (!favorArcs[nZ])
				{
					c = (int)((COSTLAYERCHANGE * COSTUNFAVORED) * Math.abs(dz) + COSTUNFAVORED * Math.abs(dx + dy));
					svNext.cost += c;
					if (RoutingDebug.isActive()) costExplanation += " Layer-unfavored=" + c;
				}
				if (penaltyOffGridX)
				{
					c = COSTOFFGRID;
					svNext.cost += c;
					if (RoutingDebug.isActive()) costExplanation += " Off-X-grid=" + c;
				}
				if (penaltyOffGridY)
				{
					c = COSTOFFGRID;
					svNext.cost += c;
					if (RoutingDebug.isActive()) costExplanation += " Off-Y-grid=" + c;
				}

				// add this vertex into the data structures
				setVertex(nX, nY, nZ);
				active.add(svNext);
				if (RoutingDebug.isActive())
				{
					debStr += ": To (" + TextUtils.formatDouble(svNext.getX()) + "," + TextUtils.formatDouble(svNext.getY()) +
						",M" + (svNext.getZ() + 1) + "), Cost=" + svNext.cost + costExplanation + "]";
				}
				if (RoutingDebug.isActive())
					RoutingDebug.identifyNewDebugPoint(svNext);

//				// add intermediate steps along the way if it was a jump
//				if (dz == 0)
//				{
//					if (Math.abs(dx) > 1)
//					{
//						double inc = dx < 0 ? 1 : -1;
	////if (Math.abs(dx) >= 20) inc = -(int)(dx / 10);
//						double lessDX = dx + inc;
//						int cost = svNext.cost;
//						int countDown = 0;
//						for (;;)
//						{
//							double nowX = curX + lessDX;
//							cost++;
//							if (!getVertex(nowX, nY, nZ))
//							{
//								SearchVertex svIntermediate = new SearchVertex(nowX, nY, nZ, whichContact, cuts, curZ, wf);
//								svIntermediate.last = svCurrent;
//								svIntermediate.cost = cost;
//								if (debugRouting != null)
//								{
//									debugRouting.identifyNewDebugPoint(svIntermediate);
//								}
//								setVertex(nowX, nY, nZ);
//								active.add(svIntermediate);
//							}
//							lessDX += inc;
//							if (inc < 0)
//							{
//								if (lessDX < 1) break;
//							} else
//							{
//								if (lessDX > -1) break;
//							}
//							if (countDown++ >= 10)
//							{
//								countDown = 0;
//								inc *= 10;
////								inc += inc;
//							}
//						}
//					}
//					if (Math.abs(dy) > 1)
//					{
//						double inc = dy < 0 ? 1 : -1;
	////if (Math.abs(dy) >= 20) inc = -(int)(dy / 10);
//						double lessDY = dy + inc;
//						int cost = svNext.cost;
//						int countDown = 0;
//						for (;;)
//						{
//							double nowY = curY + lessDY;
//							cost++;
//							if (!getVertex(nX, nowY, nZ))
//							{
//								SearchVertex svIntermediate = new SearchVertex(nX, nowY, nZ, whichContact, cuts, curZ, wf);
//								svIntermediate.last = svCurrent;
//								svIntermediate.cost = cost;
//								if (debugRouting != null)
//								{
//									debugRouting.identifyNewDebugPoint(svIntermediate);
//								}
//								setVertex(nX, nowY, nZ);
//								active.add(svIntermediate);
//							}
//							lessDY += inc;
//							if (inc < 0)
//							{
//								if (lessDY < 1) break;
//							} else
//							{
//								if (lessDY > -1) break;
//							}
//							if (countDown++ >= 10)
//							{
//								countDown = 0;
//								inc *= 10;
////								inc += inc;
//							}
//						}
//					}
//				}
			}
			if (RoutingDebug.isActive())
				RoutingDebug.endProcessingSV(svCurrent, debStr);
			return destinationSV;
		}

		private double getJumpSize(double curX, double curY, int curZ, double dx, double dy)
		{
			Rectangle2D jumpBound = nr.jumpBound;
			double width = Math.max(metalArcs[curZ].getDefaultLambdaBaseWidth(), nr.minWidth);
			double metalToMetal = getSpacingRule(curZ, width, -1);
			double metalSpacing = width / 2 + metalToMetal;
			double lX = curX - metalSpacing, hX = curX + metalSpacing;
			double lY = curY - metalSpacing, hY = curY + metalSpacing;
			if (dx > 0) hX = jumpBound.getMaxX() + metalSpacing; else
				if (dx < 0) lX = jumpBound.getMinX() - metalSpacing; else
					if (dy > 0) hY = jumpBound.getMaxY() + metalSpacing; else
						if (dy < 0) lY = jumpBound.getMinY() - metalSpacing;

			RTNode rtree = metalTrees.get(metalLayers[curZ]);
			if (rtree != null)
			{
				// see if there is anything in that area
				Rectangle2D searchArea = new Rectangle2D.Double(lX, lY, hX - lX, hY - lY);
				for (RTNode.Search sea = new RTNode.Search(searchArea, rtree, true); sea.hasNext();)
				{
					SOGBound sBound = (SOGBound) sea.next();
					if (Math.abs(sBound.getNetID()) == nr.netID) continue;
					Rectangle2D bound = sBound.getBounds();
					if (bound.getMinX() >= hX || bound.getMaxX() <= lX || bound.getMinY() >= hY || bound.getMaxY() <= lY)
						continue;
					if (dx > 0 && bound.getMinX() < hX) hX = bound.getMinX();
					if (dx < 0 && bound.getMaxX() > lX) lX = bound.getMaxX();
					if (dy > 0 && bound.getMinY() < hY) hY = bound.getMinY();
					if (dy < 0 && bound.getMaxY() > lY) lY = bound.getMaxY();
				}
			}
			if (dx > 0)
			{
				dx = downToGrain(hX - metalSpacing) - curX;
				if (curX + dx > toX && curY == toY && curZ == toZ) dx = toX - curX;
				if (curX + dx != toX && FULLGRAIN) dx = downToGrainAlways(hX - metalSpacing) - curX;
				return dx;
			}
			if (dx < 0)
			{
				dx = upToGrain(lX + metalSpacing) - curX;
				if (curX + dx < toX && curY == toY && curZ == toZ) dx = toX - curX;
				if (curX + dx != toX && FULLGRAIN) dx = upToGrainAlways(lX + metalSpacing) - curX;
				return dx;
			}
			if (dy > 0)
			{
				dy = downToGrain(hY - metalSpacing) - curY;
				if (curX == toX && curY + dy > toY && curZ == toZ) dy = toY - curY;
				if (curY + dy != toY && FULLGRAIN) dy = downToGrainAlways(hY - metalSpacing) - curY;
				return dy;
			}
			if (dy < 0)
			{
				dy = upToGrain(lY + metalSpacing) - curY;
				if (curX == toX && curY + dy < toY && curZ == toZ) dy = toY - curY;
				if (curY + dy != toY && FULLGRAIN) dy = upToGrainAlways(lY + metalSpacing) - curY;
				return dy;
			}
			return 0;
		}

		/**
		 * Method to see if a proposed piece of metal has DRC errors.
		 * @param wf the Wavefront being processed.
		 * @param metNo the level of the metal.
		 * @param halfWidth half of the width of the metal.
		 * @param halfHeight half of the height of the metal.
		 * @param x the X coordinate at the center of the metal.
		 * @param y the Y coordinate at the center of the metal.
		 * @param svCurrent the list of SearchVertex's for finding notch errors in the current path.
		 * @return a blocking SOGBound object that is in the area. Returns null if the area is clear.
		 */
		private SOGBound getMetalBlockageAndNotch(int metNo, double halfWidth, double halfHeight,
			double x, double y, SearchVertex svCurrent)
		{
			// get the R-Tree data for the metal layer
			Layer layer = metalLayers[metNo];
			RTNode rtree = metalTrees.get(layer);
			if (rtree == null) return null;

			// determine the size and width/length of this piece of metal
			int netID = nr.netID;
			double minWidth = nr.minWidth;
			double metLX = x - halfWidth, metHX = x + halfWidth;
			double metLY = y - halfHeight, metHY = y + halfHeight;
			Rectangle2D metBound = new Rectangle2D.Double(metLX, metLY, metHX - metLX, metHY - metLY);
			double metWid = Math.min(halfWidth, halfHeight) * 2;
			double metLen = Math.max(halfWidth, halfHeight) * 2;

			// determine the area to search about the metal
			double surround = worstMetalSurround[metNo];
			double lX = metLX - surround, hX = metHX + surround;
			double lY = metLY - surround, hY = metHY + surround;
			Rectangle2D searchArea = new Rectangle2D.Double(lX, lY, hX - lX, hY - lY);

			// prepare for notch detection
			List<Rectangle2D> recsOnPath = new ArrayList<Rectangle2D>();

			// make a list of rectangles on the path
			if (svCurrent != null)
			{
				List<SearchVertex> svList = getOptimizedList(svCurrent);
				for (int ind = 1; ind < svList.size(); ind++)
				{
					SearchVertex sv = svList.get(ind);
					SearchVertex lastSv = svList.get(ind - 1);
					if (sv.getZ() != metNo && lastSv.getZ() != metNo) continue;
					if (sv.getZ() != lastSv.getZ())
					{
						// changed layers: compute via rectangles
						List<MetalVia> nps = metalVias[Math.min(sv.getZ(), lastSv.getZ())].getVias();
						int whichContact = lastSv.getContactNo();
						MetalVia mv = nps.get(whichContact);
						PrimitiveNode np = mv.via;
						Orientation orient = Orientation.fromJava(mv.orientation * 10, false, false);
						SizeOffset so = np.getProtoSizeOffset();
						double xOffset = so.getLowXOffset() + so.getHighXOffset();
						double yOffset = so.getLowYOffset() + so.getHighYOffset();
						double wid = Math.max(np.getDefWidth() - xOffset, minWidth) + xOffset;
						double hei = Math.max(np.getDefHeight() - yOffset, minWidth) + yOffset;
						NodeInst ni = NodeInst.makeDummyInstance(np, new EPoint(sv.getX(), sv.getY()), wid, hei, orient);
						AffineTransform trans = null;
						if (orient != Orientation.IDENT) trans = ni.rotateOut();
						Poly[] polys = np.getTechnology().getShapeOfNode(ni);
						for (int i = 0; i < polys.length; i++)
						{
							Poly poly = polys[i];
							if (poly.getLayer() != layer) continue;
							if (trans != null) poly.transform(trans);
							Rectangle2D bound = poly.getBounds2D();
							if (bound.getMaxX() <= lX || bound.getMinX() >= hX || bound.getMaxY() <= lY || bound.getMinY() >= hY)
								continue;
							recsOnPath.add(bound);
						}
						continue;
					}

					// stayed on one layer: compute arc rectangle
					ArcProto type = metalArcs[metNo];
					double width = Math.max(type.getDefaultLambdaBaseWidth(), minWidth);
					Point2D head = new Point2D.Double(sv.getX(), sv.getY());
					Point2D tail = new Point2D.Double(lastSv.getX(), lastSv.getY());
					int ang = 0;
					if (head.getX() != tail.getX() || head.getY() != tail.getY())
						ang = GenMath.figureAngle(tail, head);
					Poly poly = Poly.makeEndPointPoly(head.distance(tail), width, ang, head, width / 2, tail,
						width / 2, Poly.Type.FILLED);
					Rectangle2D bound = poly.getBounds2D();
					if (bound.getMaxX() <= lX || bound.getMinX() >= hX || bound.getMaxY() <= lY || bound.getMinY() >= hY)
						continue;
					recsOnPath.add(bound);
				}
			}

			for (RTNode.Search sea = new RTNode.Search(searchArea, rtree, true); sea.hasNext(); )
			{
				SOGBound sBound = (SOGBound)sea.next();
				Rectangle2D bound = sBound.getBounds();

				// eliminate if out of worst surround
				if (bound.getMaxX() <= lX || bound.getMinX() >= hX || bound.getMaxY() <= lY || bound.getMinY() >= hY)
					continue;

				// see if it is within design-rule distance
				double drWid = Math.max(Math.min(bound.getWidth(), bound.getHeight()), metWid);
				double drLen = Math.max(Math.max(bound.getWidth(), bound.getHeight()), metLen);
				double spacing = getSpacingRule(metNo, drWid, drLen);
				double lXAllow = metLX - spacing, hXAllow = metHX + spacing;
				double lYAllow = metLY - spacing, hYAllow = metHY + spacing;
				if (DBMath.isLessThanOrEqualTo(bound.getMaxX(), lXAllow) ||
					DBMath.isGreaterThanOrEqualTo(bound.getMinX(), hXAllow) ||
					DBMath.isLessThanOrEqualTo(bound.getMaxY(), lYAllow) ||
					DBMath.isGreaterThanOrEqualTo(bound.getMinY(), hYAllow)) continue;

				// too close for DRC: allow if on the same net
				if (Math.abs(sBound.getNetID()) == netID)
				{
					// on same net: make sure there is no notch error
					if (sBound.getNetID() >= 0)
					{
						boolean notch = foundANotch(rtree, metBound, bound, netID, recsOnPath, spacing);
						if (notch) return sBound;
					}
					continue;
				}

				// if this is a polygon, do closer examination
				if (sBound instanceof SOGPoly)
				{
					PolyBase poly = ((SOGPoly) sBound).getPoly();
					Rectangle2D drcArea = new Rectangle2D.Double(lXAllow, lYAllow, hXAllow - lXAllow, hYAllow - lYAllow);
					if (!poly.contains(drcArea)) continue;
				}

				// DRC error found: return the offending geometry
				return sBound;
			}

			// consider notch errors in the existing path
			double spacing = getSpacingRule(metNo, metWid, metLen);

			// ISN'T THIS A BETTER WAY TO DO IT?
			for(Rectangle2D bound : recsOnPath)
			{
				if (foundANotch(rtree, metBound, bound, netID, recsOnPath, spacing))
					return new SOGBound(bound, netID);
			}

// THIS IS THE BAD WAY TO DO IT
//			if (svCurrent != null)
//			{
//				List<SearchVertex> svList = getOptimizedList(svCurrent);
//				for (int ind = 1; ind < svList.size(); ind++)
//				{
//					SearchVertex sv = svList.get(ind);
//					SearchVertex lastSv = svList.get(ind - 1);
//					if (sv.getZ() != metNo && lastSv.getZ() != metNo) continue;
//					if (sv.getZ() != lastSv.getZ())
//					{
//						// changed layers: analyze the contact for notches
//						List<MetalVia> nps = metalVias[Math.min(sv.getZ(), lastSv.getZ())].getVias();
//						int whichContact = lastSv.getContactNo();
//						MetalVia mv = nps.get(whichContact);
//						PrimitiveNode np = mv.via;
//						Orientation orient = Orientation.fromJava(mv.orientation * 10, false, false);
//						SizeOffset so = np.getProtoSizeOffset();
//						double xOffset = so.getLowXOffset() + so.getHighXOffset();
//						double yOffset = so.getLowYOffset() + so.getHighYOffset();
//						double wid = Math.max(np.getDefWidth() - xOffset, minWidth) + xOffset;
//						double hei = Math.max(np.getDefHeight() - yOffset, minWidth) + yOffset;
//						NodeInst ni = NodeInst.makeDummyInstance(np, new EPoint(sv.getX(), sv.getY()), wid, hei, orient);
//						AffineTransform trans = null;
//						if (orient != Orientation.IDENT) trans = ni.rotateOut();
//						Poly[] polys = np.getTechnology().getShapeOfNode(ni);
//						for (int i = 0; i < polys.length; i++)
//						{
//							Poly poly = polys[i];
//							if (poly.getLayer() != layer) continue;
//							if (trans != null) poly.transform(trans);
//							Rectangle2D bound = poly.getBounds2D();
//							if (bound.getMaxX() <= lX || bound.getMinX() >= hX || bound.getMaxY() <= lY || bound.getMinY() >= hY)
//								continue;
//							if (foundANotch(rtree, metBound, bound, netID, recsOnPath, spacing))
//								return new SOGBound(bound, netID);
//						}
//						continue;
//					}
//
//					// stayed on one layer: analyze the arc for notches
//					ArcProto type = metalArcs[metNo];
//					double width = Math.max(type.getDefaultLambdaBaseWidth(), minWidth);
//					Point2D head = new Point2D.Double(sv.getX(), sv.getY());
//					Point2D tail = new Point2D.Double(lastSv.getX(), lastSv.getY());
//					int ang = 0;
//					if (head.getX() != tail.getX() || head.getY() != tail.getY())
//						ang = GenMath.figureAngle(tail, head);
//					Poly poly = Poly.makeEndPointPoly(head.distance(tail), width, ang, head, width / 2, tail,
//						width / 2, Poly.Type.FILLED);
//					Rectangle2D bound = poly.getBounds2D();
//					if (bound.getMaxX() <= lX || bound.getMinX() >= hX || bound.getMaxY() <= lY || bound.getMinY() >= hY)
//						continue;
//
//					if (foundANotch(rtree, metBound, bound, netID, recsOnPath, spacing))
//						return new SOGBound(bound, netID);
//				}
//			}
			return null;
		}

		/**
		 * Method to tell whether there is a notch between two pieces of metal.
		 * @param rtree the R-Tree with the metal information.
		 * @param metBound one piece of metal.
		 * @param bound another piece of metal.
		 * @return true if there is a notch error between the pieces of metal.
		 */
		private boolean foundANotch(RTNode rtree, Rectangle2D metBound, Rectangle2D bound, int netID,
			List<Rectangle2D> recsOnPath, double dist)
		{
			// see if they overlap in X or Y
			boolean hOverlap = metBound.getMinX() <= bound.getMaxX() && metBound.getMaxX() >= bound.getMinX();
			boolean vOverlap = metBound.getMinY() <= bound.getMaxY() && metBound.getMaxY() >= bound.getMinY();

			// if they overlap in both, they touch and it is not a notch
			if (hOverlap && vOverlap) return false;

			// if they overlap horizontally then they line-up vertically
			if (hOverlap)
			{
				double ptY;
				if (metBound.getCenterY() > bound.getCenterY())
				{
					if (metBound.getMinY() - bound.getMaxY() > dist) return false;
					ptY = (metBound.getMinY() + bound.getMaxY()) / 2;
				} else
				{
					if (bound.getMinY() - metBound.getMaxY() > dist) return false;
					ptY = (metBound.getMaxY() + bound.getMinY()) / 2;
				}
				double pt1X = Math.max(metBound.getMinX(), bound.getMinX());
				double pt2X = Math.min(metBound.getMaxX(), bound.getMaxX());
				double pt3X = (pt1X + pt2X) / 2;
				if (!pointInRTree(rtree, pt1X, ptY, netID, recsOnPath)) return true;
				if (!pointInRTree(rtree, pt2X, ptY, netID, recsOnPath)) return true;
				if (!pointInRTree(rtree, pt3X, ptY, netID, recsOnPath)) return true;
				return false;
			}

			// if they overlap vertically then they line-up horizontally
			if (vOverlap)
			{
				double ptX;
				if (metBound.getCenterX() > bound.getCenterX())
				{
					if (metBound.getMinX() - bound.getMaxX() > dist) return false;
					ptX = (metBound.getMinX() + bound.getMaxX()) / 2;
				} else
				{
					if (bound.getMinX() - metBound.getMaxX() > dist) return false;
					ptX = (metBound.getMaxX() + bound.getMinX()) / 2;
				}
				double pt1Y = Math.max(metBound.getMinY(), bound.getMinY());
				double pt2Y = Math.min(metBound.getMaxY(), bound.getMaxY());
				double pt3Y = (pt1Y + pt2Y) / 2;
				if (!pointInRTree(rtree, ptX, pt1Y, netID, recsOnPath)) return true;
				if (!pointInRTree(rtree, ptX, pt2Y, netID, recsOnPath)) return true;
				if (!pointInRTree(rtree, ptX, pt3Y, netID, recsOnPath)) return true;
				return false;
			}

			// they are diagonal, ensure that one of the "L"s is filled
			if (metBound.getMinX() > bound.getMaxX() && metBound.getMinY() > bound.getMaxY())
			{
				// metal to upper-right of test area
				double pt1X = metBound.getMinX();
				double pt1Y = bound.getMaxY();
				double pt2X = bound.getMaxX();
				double pt2Y = metBound.getMinY();
				if (Math.sqrt((pt1X - pt2X) * (pt1X - pt2X) + (pt1Y - pt2Y) * (pt1Y - pt2Y)) > dist) return false;
				if (pointInRTree(rtree, pt1X, pt1Y, netID, recsOnPath)) return false;
				if (pointInRTree(rtree, pt2X, pt2Y, netID, recsOnPath)) return false;
				return true;
			}
			if (metBound.getMaxX() < bound.getMinX() && metBound.getMinY() > bound.getMaxY())
			{
				// metal to upper-left of test area
				double pt1X = metBound.getMaxX();
				double pt1Y = bound.getMaxY();
				double pt2X = bound.getMinX();
				double pt2Y = metBound.getMinY();
				if (Math.sqrt((pt1X - pt2X) * (pt1X - pt2X) + (pt1Y - pt2Y) * (pt1Y - pt2Y)) > dist) return false;
				if (pointInRTree(rtree, pt1X, pt1Y, netID, recsOnPath)) return false;
				if (pointInRTree(rtree, pt2X, pt2Y, netID, recsOnPath)) return false;
				return true;
			}
			if (metBound.getMaxX() < bound.getMinX() && metBound.getMaxY() < bound.getMinY())
			{
				// metal to lower-left of test area
				double pt1X = metBound.getMaxX();
				double pt1Y = bound.getMinY();
				double pt2X = bound.getMinX();
				double pt2Y = metBound.getMaxY();
				if (Math.sqrt((pt1X - pt2X) * (pt1X - pt2X) + (pt1Y - pt2Y) * (pt1Y - pt2Y)) > dist) return false;
				if (pointInRTree(rtree, pt1X, pt1Y, netID, recsOnPath)) return false;
				if (pointInRTree(rtree, pt2X, pt2Y, netID, recsOnPath)) return false;
				return true;
			}
			if (metBound.getMinX() > bound.getMaxX() && metBound.getMaxY() < bound.getMinY())
			{
				// metal to lower-right of test area
				double pt1X = metBound.getMinX();
				double pt1Y = bound.getMinY();
				double pt2X = bound.getMaxX();
				double pt2Y = metBound.getMaxY();
				if (Math.sqrt((pt1X - pt2X) * (pt1X - pt2X) + (pt1Y - pt2Y) * (pt1Y - pt2Y)) > dist) return false;
				if (pointInRTree(rtree, pt1X, pt1Y, netID, recsOnPath)) return false;
				if (pointInRTree(rtree, pt2X, pt2Y, netID, recsOnPath)) return false;
				return true;
			}
			return false;
		}

		private boolean pointInRTree(RTNode rtree, double x, double y, int netID, List<Rectangle2D> recsOnPath)
		{
			Rectangle2D searchArea = new Rectangle2D.Double(x-0.5, y-0.5, 1, 1);
			for (RTNode.Search sea = new RTNode.Search(searchArea, rtree, true); sea.hasNext(); )
			{
				SOGBound sBound = (SOGBound) sea.next();
				if (sBound.getNetID() != netID) continue;
//if (sBound.getNetID() < 0) continue;
				if (DBMath.isGreaterThan(sBound.getBounds().getMinX(), x) ||
					DBMath.isLessThan(sBound.getBounds().getMaxX(), x) ||
					DBMath.isGreaterThan(sBound.getBounds().getMinY(), y) ||
					DBMath.isLessThan(sBound.getBounds().getMaxY(), y))
						continue;
				return true;
			}

			// now see if it is on the path
			for (Rectangle2D bound : recsOnPath)
			{
				if (DBMath.isGreaterThan(bound.getMinX(), x) ||
					DBMath.isLessThan(bound.getMaxX(), x) ||
					DBMath.isGreaterThan(bound.getMinY(), y) ||
					DBMath.isLessThan(bound.getMaxY(), y))
						continue;
				return true;
			}
			return false;
		}
	}

	/************************************** SEARCH VERTICES **************************************/

	/**
	 * Class to define a vertex in the Dijkstra search.
	 */
	public static class SearchVertex implements Comparable<SearchVertex>
	{
		/** the coordinate of the search vertex. */		private double xv, yv;
		/** the layer of the search vertex. */			private int zv;
		/** the cost of search to this vertex. */		private int cost;
		/** the layer of cuts in "cuts". */				private int cutLayer;
		/** auto-generation of intermediate vertices */	private boolean autoGen;
		/** the cuts in the contact. */					private Point2D[] cuts;
		/** the previous vertex in the search. */		private SearchVertex last;
		/** the routing state. */						private Wavefront wf;

		/**
		 * Method to create a new SearchVertex.
		 * @param x the X coordinate of the SearchVertex.
		 * @param y the Y coordinate of the SearchVertex.
		 * @param z the Z coordinate (metal layer) of the SearchVertex.
		 * @param whichContact the contact number to use if switching layers.
		 * @param cuts an array of cuts in this contact (if switching layers).
		 * @param cl the layer of the cut (if switching layers).
		 * @param w the Wavefront that this SearchVertex is part of.
		 */
		SearchVertex(double x, double y, int z, int whichContact, Point2D[] cuts, int cl, Wavefront w)
		{
			xv = x;
			yv = y;
			zv = (z << 8) + (whichContact & 0xFF);
			this.cuts = cuts;
			cutLayer = cl;
			autoGen = false;
			wf = w;
		}

		public double getX() { return xv; }

		public double getY() { return yv; }

		public int getZ() { return zv >> 8; }

		public SearchVertex getLast() { return last; }

		public int getCost() { return cost; }

		int getContactNo() { return zv & 0xFF; }

		Point2D[] getCuts() { return cuts; }

		void clearCuts() { cuts = null; }

		int getCutLayer() { return cutLayer; }

		boolean isAutoGen() { return autoGen; }

		/**
		 * Method to set the automatic generation of intermediate SearchVertex objects.
		 * When a SearchVertex is placed at a great distance from the previous one,
		 * it may have missed intermediate steps.
		 * The AutoGen value directs creation of an intermediate step.
		 * @param a true to automatically generate intermediate steps.
		 */
		void setAutoGen(boolean a) { autoGen = a; }

		/**
		 * Method to sort SearchVertex objects by their cost.
		 */
		public int compareTo(SearchVertex svo)
		{
			int diff = cost - svo.cost;
			if (diff != 0) return diff;
			if (wf != null)
			{
				double thisDist = Math.abs(xv - wf.toX) + Math.abs(yv - wf.toY) + Math.abs(zv - wf.toZ);
				double otherDist = Math.abs(svo.xv - wf.toX) + Math.abs(svo.yv - wf.toY) + Math.abs(svo.zv - wf.toZ);
				if (thisDist < otherDist) return -1;
				if (thisDist > otherDist) return 1;
			}
			return 0;
		}

		private void generateIntermediateVertex()
		{
			SearchVertex prevSV = last;
			double dX = 0, dY = 0;
			if (getX() > prevSV.getX()) dX = -1; else
				if (getX() < prevSV.getX()) dX = 1; else
					if (getY() > prevSV.getY()) dY = -1; else
						if (getY() < prevSV.getY()) dY = 1;
			if (dX == 0 && dY == 0) return;
			double newX = getX() + dX;
			double newY = getY() + dY;
			int z = getZ();
			for(;;)
			{
				if (dX < 0 && newX < prevSV.getX()) break;
				if (dX > 0 && newX > prevSV.getX()) break;
				if (dY < 0 && newY < prevSV.getY()) break;
				if (dY > 0 && newY > prevSV.getY()) break;
				if (!wf.isVertex(newX, newY, z))
				{
					SearchVertex svIntermediate = new SearchVertex(newX, newY, z, getContactNo(), getCuts(), z, wf);
					svIntermediate.setAutoGen(true);
					svIntermediate.last = prevSV;
					svIntermediate.cost = cost + 1;
					if (RoutingDebug.isActive())
						RoutingDebug.identifyNewDebugPoint(svIntermediate);
					wf.setVertex(newX, newY, z);
					wf.active.add(svIntermediate);
					break;
				}
				newX += dX;   newY += dY;
			}
		}
	}

	/********************************* INITIALIZATION *********************************/

	/**
	 * Method to initialize technology information, including design rules.
	 * @return true on error.
	 */
	private boolean initializeDesignRules(Cell c, SeaOfGates.SeaOfGatesOptions prefs)
	{
		// find the metal layers, arcs, and contacts
		cell = c;
		tech = cell.getTechnology();
		numMetalLayers = tech.getNumMetals();
		metalLayers = new Layer[numMetalLayers];
		metalArcs = new ArcProto[numMetalLayers];
		favorArcs = new boolean[numMetalLayers];
		preventArcs = new boolean[numMetalLayers];
		viaLayers = new Layer[numMetalLayers - 1];
		metalVias = new MetalVias[numMetalLayers - 1];
		for (int i = 0; i < numMetalLayers - 1; i++)
			metalVias[i] = new MetalVias();
		for (Iterator<Layer> it = tech.getLayers(); it.hasNext();)
		{
			Layer lay = it.next();
			if (!lay.getFunction().isMetal()) continue;
			if (lay.isPseudoLayer()) continue;
			int layerIndex = lay.getFunction().getLevel() - 1;
			if (layerIndex < numMetalLayers)  metalLayers[layerIndex] = lay;
		}
		boolean hasFavorites = false;
		for (Iterator<ArcProto> it = tech.getArcs(); it.hasNext(); )
		{
			ArcProto ap = it.next();
			for (int i = 0; i < numMetalLayers; i++)
			{
				if (ap.getLayer(0) == metalLayers[i])
				{
					metalArcs[i] = ap;
					favorArcs[i] = prefs.isPrevented(ap);
					if (favorArcs[i]) hasFavorites = true;
					preventArcs[i] = prefs.isPrevented(ap);
					break;
				}
			}
		}
		if (!hasFavorites)
			for (int i = 0; i < numMetalLayers; i++) favorArcs[i] = true;
		for (Iterator<PrimitiveNode> it = tech.getNodes(); it.hasNext();)
		{
			PrimitiveNode np = it.next();
			if (np.isNotUsed()) continue;
			if (!np.getFunction().isContact()) continue;
			ArcProto[] conns = np.getPort(0).getConnections();
			for (int i = 0; i < numMetalLayers - 1; i++)
			{
				if ((conns[0] == metalArcs[i] && conns[1] == metalArcs[i + 1]) ||
					(conns[1] == metalArcs[i] && conns[0] == metalArcs[i + 1]))
				{
					metalVias[i].addVia(np, 0);

					// see if the node is asymmetric and should exist in rotated states
					boolean square = true, offCenter = false;
					NodeInst dummyNi = NodeInst.makeDummyInstance(np);
					Poly[] conPolys = tech.getShapeOfNode(dummyNi);
					for (int p = 0; p < conPolys.length; p++)
					{
						Poly conPoly = conPolys[p];
						Layer conLayer = conPoly.getLayer();
						Layer.Function lFun = conLayer.getFunction();
						if (lFun.isMetal())
						{
							Rectangle2D conRect = conPoly.getBounds2D();
							if (conRect.getWidth() != conRect.getHeight()) square = false;
							if (conRect.getCenterX() != 0 || conRect.getCenterY() != 0) offCenter = true;
						} else if (lFun.isContact())
						{
							viaLayers[i] = conLayer;
						}
					}
					if (offCenter)
					{
						// off center: test in all 4 rotations
						metalVias[i].addVia(np, 90);
						metalVias[i].addVia(np, 180);
						metalVias[i].addVia(np, 270);
					} else if (!square)
					{
						// centered but not square: test in 90-degree rotation
						metalVias[i].addVia(np, 90);
					}
					break;
				}
			}
		}
		for (int i = 0; i < numMetalLayers; i++)
		{
			if (metalLayers[i] == null)
			{
				System.out.println("ERROR: Cannot find layer for Metal " + (i + 1));
				return true;
			}
			if (metalArcs[i] == null)
			{
				System.out.println("ERROR: Cannot find arc for Metal " + (i + 1));
				return true;
			}
			if (i < numMetalLayers - 1)
			{
				if (metalVias[i].getVias().size() == 0)
				{
					System.out.println("ERROR: Cannot find contact node between Metal " + (i + 1) + " and Metal " + (i + 2));
					return true;
				}
				if (viaLayers[i] == null)
				{
					System.out.println("ERROR: Cannot find contact layer between Metal " + (i + 1) + " and Metal " + (i + 2));
					return true;
				}
			}
		}

		// compute design rule spacings
		worstMetalSurround = new double[numMetalLayers];
		GenMath.MutableDouble mutableDist = new GenMath.MutableDouble(0);
		for (int i = 0; i < numMetalLayers; i++)
		{
			if (DRC.getMaxSurround(metalLayers[i], Double.MAX_VALUE, mutableDist)) // only when found
				worstMetalSurround[i] = mutableDist.doubleValue();
		}
		viaSurround = new double[numMetalLayers - 1];
		for (int i = 0; i < numMetalLayers - 1; i++)
		{
			Layer lay = viaLayers[i];

			double spacing = 2;
			double arcWidth = metalArcs[i].getDefaultLambdaBaseWidth();
			DRCTemplate ruleSpacing = DRC.getSpacingRule(lay, null, lay, null, false, -1, arcWidth, 50);
			if (ruleSpacing != null) spacing = ruleSpacing.getValue(0);

			// determine cut size
			double width = 0;
			DRCTemplate ruleWidth = DRC.getMinValue(lay, DRCTemplate.DRCRuleType.NODSIZ);
			if (ruleWidth != null) width = ruleWidth.getValue(0);

			// extend to the size of the largest cut
			List<MetalVia> nps = metalVias[i].getVias();
			for (MetalVia mv : nps)
			{
				NodeInst dummyNi = NodeInst.makeDummyInstance(mv.via);
				Poly[] conPolys = tech.getShapeOfNode(dummyNi);
				for (int p = 0; p < conPolys.length; p++)
				{
					Poly conPoly = conPolys[p];
					if (conPoly.getLayer().getFunction().isContact())
					{
						Rectangle2D bounds = conPoly.getBounds2D();
						width = Math.max(width, bounds.getWidth());
						width = Math.max(width, bounds.getHeight());
					}
				}
			}

			viaSurround[i] = spacing + width;
		}
		return false;
	}

	protected void makeListOfRoutes(int numBatches, RouteBatches[] routeBatches, List<NeededRoute> allRoutes,
		List<ArcInst> arcsToRoute, EditingPreferences ep)
	{
		this.doMakeListOfRoutes(0, numBatches, routeBatches, allRoutes, arcsToRoute);
	}

	protected void doMakeListOfRoutes(int start, int end, RouteBatches[] routeBatches,
		List<NeededRoute> allRoutes, List<ArcInst> arcsToRoute)
	{
		for (int b = start; b < end; b++)
		{
			// get list of PortInsts that comprise this net
			ArcInst ai = arcsToRoute.get(b);
			Netlist netList = cell.getNetlist();
			Network net = netList.getNetwork(ai, 0);
			if (net == null)
			{
				System.out.println("Arc " + ai.describe(false) + " has no network!");
				continue;
			}
			Set<ArcInst> arcsToDelete = new HashSet<ArcInst>();
			Set<NodeInst> nodesToDelete = new HashSet<NodeInst>();
			List<PortInst> orderedPorts;
			if (RoutingDebug.isActive())
			{
				orderedPorts = new ArrayList<PortInst>();
				orderedPorts.add(ai.getHeadPortInst());
				orderedPorts.add(ai.getTailPortInst());
				arcsToDelete.add(ai);
			} else
			{
				Map<Network, ArcInst[]> arcMap = null;
				if (cell.getView() != View.SCHEMATIC)
					arcMap = netList.getArcInstsByNetwork();
				List<Connection> netEnds = Routing.findNetEnds(net, arcMap, arcsToDelete, nodesToDelete, netList, true);
				orderedPorts = makeOrderedPorts(net, netEnds);
			}
			if (orderedPorts == null)
			{
				System.out.println("ERROR: No valid routes points found on the network " + net.describe(false));
				List<EPoint> lineList = new ArrayList<EPoint>();
				for(ArcInst delAi : arcsToDelete)
				{
					lineList.add(delAi.getHeadLocation());
					lineList.add(delAi.getTailLocation());
				}
				errorLogger.logMessageWithLines("Arcs in " + net.getName() + " not make valid connection: deleted",
					null, lineList, cell, 0, true);

				routeBatches[b] = new RouteBatches();
				routeBatches[b].unroutedArcs = arcsToDelete;
				routeBatches[b].unroutedNodes = nodesToDelete;
				routeBatches[b].numUnrouted = 0;
				continue;
			}
			routeBatches[b] = new RouteBatches();
			routeBatches[b].unroutedArcs = arcsToDelete;
			routeBatches[b].unroutedNodes = nodesToDelete;
			routeBatches[b].orderedPorts = orderedPorts;
			routeBatches[b].segsInBatch = 0;

			// determine the minimum width of arcs on this net
			double minWidth = getMinWidth(orderedPorts, prefs);
			int netID = -1;
			Integer netIDI = netIDs.get(ai);
			if (netIDI != null) netID = netIDI.intValue() - 1;

			// find a path between the ends of the network
			int batchNumber = 1;
			for (int i = 0; i < orderedPorts.size() - 1; i++)
			{
				PortInst fromPi = orderedPorts.get(i);
				PortInst toPi = orderedPorts.get(i + 1);
				if (inValidPort(fromPi) || inValidPort(toPi)) continue;

				// get information about one end of the path
				ArcProto fromArc = null;
				ArcProto[] fromArcs = fromPi.getPortProto().getBasePort().getConnections();
				if (fromPi.getPortProto() instanceof Export)
				{
					Export e = (Export)fromPi.getPortProto();
					Variable var = e.getVar(Export.EXPORT_PREFERRED_ARCS);
					if (var != null)
					{
						String[] arcNames = (String[])var.getObject();
						ArcProto[] arcs = new ArcProto[arcNames.length];
						boolean allFound = true;
						for(int j=0; j<arcNames.length; j++)
						{
							arcs[j] = ArcProto.findArcProto(arcNames[j]);
							if (arcs[j] == null) allFound = false;
						}
						if (allFound) fromArcs = arcs;
					}
				}
				for (int j = 0; j < fromArcs.length; j++)
					if (fromArcs[j].getTechnology() == tech && fromArcs[j].getFunction().isMetal())
					{
						fromArc = fromArcs[j];
						break;
					}
				if (fromArc == null)
				{
					String errorMsg = "Cannot connect port " + fromPi.getPortProto().getName() + " of node " +
						fromPi.getNodeInst().describe(false) + " because it has no metal connection in " + tech.getTechName() + " technology";
					System.out.println("ERROR: " + errorMsg);
					List<PolyBase> polyList = new ArrayList<PolyBase>();
					polyList.add(fromPi.getPoly());
					errorLogger.logMessage(errorMsg, polyList, cell, 0, true);
					continue;
				}

				// get information about the other end of the path
				ArcProto toArc = null;
				ArcProto[] toArcs = toPi.getPortProto().getBasePort().getConnections();
				if (toPi.getPortProto() instanceof Export)
				{
					Export e = (Export)toPi.getPortProto();
					Variable var = e.getVar(Export.EXPORT_PREFERRED_ARCS);
					if (var != null)
					{
						String[] arcNames = (String[])var.getObject();
						ArcProto[] arcs = new ArcProto[arcNames.length];
						boolean allFound = true;
						for(int j=0; j<arcNames.length; j++)
						{
							arcs[j] = ArcProto.findArcProto(arcNames[j]);
							if (arcs[j] == null) allFound = false;
						}
						if (allFound) toArcs = arcs;
					}
				}
				for (int j = 0; j < toArcs.length; j++)
					if (toArcs[j].getTechnology() == tech && toArcs[j].getFunction().isMetal())
					{
						toArc = toArcs[j];
						break;
					}
				if (toArc == null)
				{
					String errorMsg = "Cannot connect port " + toPi.getPortProto().getName() + " of node " +
						toPi.getNodeInst().describe(false) + " because it has no metal connection in " + tech.getTechName() + " technology";
					System.out.println("ERROR: " + errorMsg);
					List<PolyBase> polyList = new ArrayList<PolyBase>();
					polyList.add(toPi.getPoly());
					errorLogger.logMessage(errorMsg, polyList, cell, 0, true);
					continue;
				}

				// determine the coordinates of the route
				Poly fromPoly = fromPi.getPoly();
				EPoint fromLoc = fromPoly.getCenter();
				Poly toPoly = toPi.getPoly();
				EPoint toLoc = toPoly.getCenter();
				double fromX = fromLoc.getX(), fromY = fromLoc.getY();
				double toX = toLoc.getX(), toY = toLoc.getY();
				if (toLoc.getX() < fromLoc.getX())
				{
					toX = upToGrain(toLoc.getX());
					fromX = downToGrain(fromLoc.getX());
				} else if (toLoc.getX() > fromLoc.getX())
				{
					toX = downToGrain(toLoc.getX());
					fromX = upToGrain(fromLoc.getX());
				} else
				{
					toX = fromX = upToGrain(fromLoc.getX());
				}
				if (toLoc.getY() < fromLoc.getY())
				{
					toY = upToGrain(toLoc.getY());
					fromY = downToGrain(fromLoc.getY());
				} else if (toLoc.getY() > fromLoc.getY())
				{
					toY = downToGrain(toLoc.getY());
					fromY = upToGrain(fromLoc.getY());
				} else
				{
					toY = fromY = upToGrain(fromLoc.getY());
				}
				int fromZ = fromArc.getFunction().getLevel() - 1;
				int toZ = toArc.getFunction().getLevel() - 1;

				// determine "from" surround
				double fromMetalSpacing = Math.max(metalArcs[fromZ].getDefaultLambdaBaseWidth(), minWidth) / 2;
				Layer lay = metalLayers[fromZ];
				DRCTemplate rule = DRC.getSpacingRule(lay, null, lay, null, false, -1,
					metalArcs[fromZ].getDefaultLambdaBaseWidth(), -1);
				double fromSurround = 0;
				if (rule != null) fromSurround = rule.getValue(0);

				// see if "from" access is blocked
				SOGBound block = getMetalBlockage(netID, fromZ, fromMetalSpacing, fromMetalSpacing, fromSurround, fromX, fromY);
				if (block != null)
				{
					// see if gridding caused the blockage
					fromX = fromLoc.getX();
					fromY = fromLoc.getY();
					block = getMetalBlockage(netID, fromZ, fromMetalSpacing, fromMetalSpacing, fromSurround, fromX, fromY);
					if (block != null)
					{
						// ungridded center location still blocked: see if port has nonzero area and other places in it are free
						Rectangle2D fromRect = fromPoly.getBounds2D();
						double stepSize = fromMetalSpacing + fromSurround;
						if (stepSize > 0 && (fromRect.getWidth() > 0 || fromRect.getHeight() > 0))
						{
							for(double x = fromRect.getMinX(); x <= fromRect.getMaxX(); x += stepSize)
							{
								for(double y = fromRect.getMinY(); y <= fromRect.getMaxY(); y += stepSize)
								{
									SOGBound stepBlock = getMetalBlockage(netID, fromZ, fromMetalSpacing, fromMetalSpacing, fromSurround, x, y);
									if (stepBlock == null)
									{
										fromX = x;   fromY = y;
										block = null;
										break;
									}									
								}
								if (block == null) break;
							}
						}
						if (block != null)
						{
							String errorMsg = "Cannot Route from port " + fromPi.getPortProto().getName()
								+ " of node " + fromPi.getNodeInst().describe(false) + " at ("
								+ TextUtils.formatDistance(fromX) + "," + TextUtils.formatDistance(fromY)
								+ ") because it is blocked on layer " + metalLayers[fromZ].getName()
								+ " [needs " + TextUtils.formatDistance(fromMetalSpacing + fromSurround)
								+ " all around, blockage is "
								+ TextUtils.formatDistance(block.getBounds().getMinX()) + "<=X<="
								+ TextUtils.formatDistance(block.getBounds().getMaxX()) + " and "
								+ TextUtils.formatDistance(block.getBounds().getMinY()) + "<=Y<="
								+ TextUtils.formatDistance(block.getBounds().getMaxY()) + "]";
							System.out.println("ERROR: " + errorMsg);
							List<PolyBase> polyList = new ArrayList<PolyBase>();
							polyList.add(new PolyBase(fromX, fromY, (fromMetalSpacing + fromSurround) * 2,
								(fromMetalSpacing + fromSurround) * 2));
							polyList.add(new PolyBase(block.getBounds()));
							List<EPoint> lineList = new ArrayList<EPoint>();
							lineList.add(new EPoint(block.getBounds().getMinX(), block.getBounds().getMinY()));
							lineList.add(new EPoint(block.getBounds().getMaxX(), block.getBounds().getMaxY()));
							lineList.add(new EPoint(block.getBounds().getMinX(), block.getBounds().getMaxY()));
							lineList.add(new EPoint(block.getBounds().getMaxX(), block.getBounds().getMinY()));
							errorLogger.logMessageWithLines(errorMsg, polyList, lineList, cell, 0, true);
							continue;
						}
					}
				}

				// determine "to" surround
				double toMetalSpacing = Math.max(metalArcs[toZ].getDefaultLambdaBaseWidth(), minWidth) / 2;
				lay = metalLayers[toZ];
				rule = DRC.getSpacingRule(lay, null, lay, null, false, -1, metalArcs[toZ].getDefaultLambdaBaseWidth(), -1);
				double toSurround = 0;
				if (rule != null) toSurround = rule.getValue(0);

				// see if "to" access is blocked
				block = getMetalBlockage(netID, toZ, toMetalSpacing, toMetalSpacing, toSurround, toX, toY);
				if (block != null)
				{
					// see if gridding caused the blockage
					toX = toLoc.getX();
					toY = toLoc.getY();
					block = getMetalBlockage(netID, toZ, toMetalSpacing, toMetalSpacing, toSurround, toX, toY);
					if (block != null)
					{
						// ungridded center location still blocked: see if port has nonzero area and other places in it are free
						Rectangle2D toRect = toPoly.getBounds2D();
						double stepSize = toMetalSpacing + toSurround;
						if (stepSize > 0 && (toRect.getWidth() > 0 || toRect.getHeight() > 0))
						{
							for(double x = toRect.getMinX(); x <= toRect.getMaxX(); x += stepSize)
							{
								for(double y = toRect.getMinY(); y <= toRect.getMaxY(); y += stepSize)
								{
									SOGBound stepBlock = getMetalBlockage(netID, toZ, toMetalSpacing, toMetalSpacing, toSurround, x, y);
									if (stepBlock == null)
									{
										toX = x;   toY = y;
										block = null;
										break;
									}									
								}
								if (block == null) break;
							}
						}
						if (block != null)
						{
							String errorMsg = "Cannot route to port " + toPi.getPortProto().getName()
								+ " of node " + toPi.getNodeInst().describe(false) + " at ("
								+ TextUtils.formatDistance(toX) + "," + TextUtils.formatDistance(toY)
								+ ") because it is blocked on layer " + metalLayers[toZ].getName()
								+ " [needs " + TextUtils.formatDistance(toMetalSpacing + toSurround)
								+ " all around, blockage is "
								+ TextUtils.formatDistance(block.getBounds().getMinX()) + "<=X<="
								+ TextUtils.formatDistance(block.getBounds().getMaxX()) + " and "
								+ TextUtils.formatDistance(block.getBounds().getMinY()) + "<=Y<="
								+ TextUtils.formatDistance(block.getBounds().getMaxY()) + "]";
							System.out.println("ERROR: " + errorMsg);
							List<PolyBase> polyList = new ArrayList<PolyBase>();
							polyList.add(new PolyBase(toX, toY, (toMetalSpacing + toSurround) * 2,
								(toMetalSpacing + toSurround) * 2));
							polyList.add(new PolyBase(block.getBounds()));
							List<EPoint> lineList = new ArrayList<EPoint>();
							lineList.add(new EPoint(block.getBounds().getMinX(), block.getBounds().getMinY()));
							lineList.add(new EPoint(block.getBounds().getMaxX(), block.getBounds().getMaxY()));
							lineList.add(new EPoint(block.getBounds().getMinX(), block.getBounds().getMaxY()));
							lineList.add(new EPoint(block.getBounds().getMaxX(), block.getBounds().getMinY()));
							errorLogger.logMessageWithLines(errorMsg, polyList, lineList, cell, 0, true);
							continue;
						}
					}
				}

				// schedule the route
				NeededRoute nr = new NeededRoute(net.getName(), fromPi, fromX, fromY, fromZ, toPi, toX, toY,
					toZ, netID, minWidth, b, batchNumber++, prefs);
				routeBatches[b].segsInBatch++;
				allRoutes.add(nr);
			}
		}
	}

	/**
	 * Method to find a path between two ports.
	 * @param nr the NeededRoute object with all necessary information.
	 * If successful, the NeededRoute's "vertices" field is filled with the route data.
	 */
	protected void findPath(NeededRoute nr, Environment env, EditingPreferences ep, Job job)
	{
		// special case when route is null length
		Wavefront d1 = nr.dir1;
		if (DBMath.areEquals(d1.toX, d1.fromX) && DBMath.areEquals(d1.toY, d1.fromY) && d1.toZ == d1.fromZ)
		{
			nr.winningWF = d1;
			nr.winningWF.vertices = new ArrayList<SearchVertex>();
			SearchVertex sv = new SearchVertex(d1.toX, d1.toY, d1.toZ, 0, null, 0, nr.winningWF);
			nr.winningWF.vertices.add(sv);
			nr.cleanSearchMemory();
			return;
		}

		if (parallelDij)
		{
			// create threads and start them running
			Semaphore outSem = new Semaphore(0);
			new DijkstraInThread("Route a->b", nr.dir1, nr.dir2, outSem, env, ep);
			new DijkstraInThread("Route b->a", nr.dir2, nr.dir1, outSem, env, ep);

			// wait for threads to complete and get results
			outSem.acquireUninterruptibly(2);
		} else
		{
			// run both wavefronts in parallel (interleaving steps)
			doTwoWayDijkstra(nr);
		}

		// analyze the winning wavefront
		Wavefront wf = nr.winningWF;
		double verLength = Double.MAX_VALUE;
		if (wf != null)
			verLength = getVertexLength(wf.vertices);
		if (verLength == Double.MAX_VALUE)
		{
			// failed to route
			String errorMsg;
			if (wf == null)
				wf = nr.dir1;
			if (wf.vertices == null)
			{
				errorMsg = "Search for '" + nr.routeName + "' too complex (exceeds complexity limit of " +
					prefs.complexityLimit + " steps)";
			} else
			{
				errorMsg = "Failed in '" + nr.routeName + "' to route from port " + wf.from.getPortProto().getName() +
					" of node " + wf.from.getNodeInst().describe(false) + " to port " + wf.to.getPortProto().getName() +
					" of node " + wf.to.getNodeInst().describe(false);
			}
			System.out.println("ERROR: " + errorMsg);
			List<EPoint> lineList = new ArrayList<EPoint>();
			lineList.add(new EPoint(wf.toX, wf.toY));
			lineList.add(new EPoint(wf.fromX, wf.fromY));
			errorLogger.logMessageWithLines(errorMsg, null, lineList, cell, 0, true);

			if (DEBUGFAILURE && firstFailure)
			{
				firstFailure = false;
				showSearchVertices(nr.dir1.active, nr.dir1.inactive, cell, d1.fromX, d1.fromY, d1.toX, d1.toY, metalTrees, nr.routeBounds);
			}
		}
		nr.cleanSearchMemory();
	}

	/********************************* SEARCH SUPPORT *********************************/

	/**
	 * Method to run search from both ends at once (interleaved).
	 * Used in single-processor systems.
	 * @param nr the NeededRoute to search.
	 */
	protected void doTwoWayDijkstra(NeededRoute nr)
	{
		SearchVertex result = null;
		while (result == null)
		{
			SearchVertex resultA = nr.dir1.advanceWavefront();
			SearchVertex resultB = nr.dir2.advanceWavefront();
			if (resultA != null || resultB != null)
			{
				if (resultA == svExhausted && resultB == svExhausted)
				{
					result = svExhausted;
					break;
				}
				if (resultA == svExhausted) resultA = null;
				if (resultB == svExhausted) resultB = null;
				result = resultA;
				nr.winningWF = nr.dir1;
				if (result == null)
				{
					result = resultB;
					nr.winningWF = nr.dir2;
				}
			}
		}
		if (result == svAborted || result == svExhausted)
		{
			nr.winningWF = null;
			return;
		}

		// dumpPlane(0);
		// dumpPlane(1);
		// dumpPlane(2);
		List<SearchVertex> realVertices = getOptimizedList(result);
		nr.winningWF.vertices = realVertices;
	}

	/**
	 * Class to run search in a separate thread.
	 * Runs in a single direction, from point A to point B,
	 * so two of these threads are needed to run A->B and B->A in parallel.
	 */
	class DijkstraInThread extends Thread
	{
		private Wavefront wf;
		private Wavefront otherWf;
		private Semaphore whenDone;
		private Environment env;
		private EditingPreferences ep;

		public DijkstraInThread(String name, Wavefront wf, Wavefront otherWf, Semaphore whenDone,
			Environment env, EditingPreferences ep)
		{
			super(name);
			this.wf = wf;
			this.otherWf = otherWf;
			this.whenDone = whenDone;
			this.env = env;
			this.ep = ep;
			start();
		}

		public void run()
		{
			Environment.setThreadEnvironment(env);
			EditingPreferences.setThreadEditingPreferences(ep);
			SearchVertex result = null;
			while (result == null)
			{
				if (wf.abort) result = svAborted; else
					result = wf.advanceWavefront();
			}
			if (result != svAborted && result != svExhausted && result != svLimited)
			{
				if (DEBUGLOOPS)
					System.out.println("	Wavefront " + wf.name + " first completion");
				wf.vertices = getOptimizedList(result);
				wf.nr.winningWF = wf;
				otherWf.abort = true;
			} else
			{
				if (DEBUGLOOPS)
				{
					String status = "completed";
					if (result == svAborted) status = "aborted"; else
						if (result == svExhausted) status = "exhausted"; else
							if (result == svLimited) status = "limited";
					System.out.println("	Wavefront " + wf.name + " " + status);
				}
			}
			whenDone.release();
		}
	}

	/********************************* MISCELLANEOUS SUPPORT *********************************/

	/**
	 * Class to hold a "batch" of routes, all on the same network.
	 */
	protected static class RouteBatches
	{
		Set<ArcInst> unroutedArcs;
		Set<NodeInst> unroutedNodes;
		List<PortInst> orderedPorts;
		int segsInBatch;
		int numRouted, numUnrouted;
	}

	private double getMinWidth(List<PortInst> orderedPorts, SeaOfGates.SeaOfGatesOptions prefs)
	{
		double minWidth = 0;
		for (PortInst pi : orderedPorts)
		{
			double widestAtPort = getWidestMetalArcOnPort(pi);
			if (widestAtPort > minWidth) minWidth = widestAtPort;
		}
		if (minWidth > prefs.maxArcWidth) minWidth = prefs.maxArcWidth;
		return minWidth;
	}

	/**
	 * Get the widest metal arc already connected to a given PortInst. Looks
	 * recursively down the hierarchy.
	 * @param pi the PortInst to connect.
	 * @return the widest metal arc connect to that port (zero if none)
	 */
	private double getWidestMetalArcOnPort(PortInst pi)
	{
		// first check the top level
		double width = 0;
		for (Iterator<Connection> it = pi.getConnections(); it.hasNext();)
		{
			Connection c = it.next();
			ArcInst ai = c.getArc();
			if (!ai.getProto().getFunction().isMetal()) continue;
			double newWidth = ai.getLambdaBaseWidth();
			if (newWidth > width) width = newWidth;
		}

		// now recurse down the hierarchy
		NodeInst ni = pi.getNodeInst();
		if (ni.isCellInstance())
		{
			Export export = (Export) pi.getPortProto();
			PortInst exportedInst = export.getOriginalPort();
			double width2 = getWidestMetalArcOnPort(exportedInst);
			if (width2 > width) width = width2;
		}
		return width;
	}

	private boolean inValidPort(PortInst pi)
	{
		ArcProto[] conns = pi.getPortProto().getBasePort().getConnections();
		boolean valid = false;
		for (int j = 0; j < conns.length; j++)
		{
			ArcProto ap = conns[j];
			if (ap.getTechnology() != tech) continue;
			if (!ap.getFunction().isMetal()) continue;
			if (preventArcs[conns[j].getFunction().getLevel() - 1]) continue;
			valid = true;
			break;
		}
		if (!valid)
		{
			System.out.println("Cannot connect to port " + pi.getPortProto().getName() + " on node " +
				pi.getNodeInst().describe(false) + " because all connecting layers have been prevented by Routing Preferences");
			return true;
		}
		return false;
	}

	/**
	 * Method to order a set of connections for optimal routing.
	 * @param net the Network being ordered.
	 * @param netEnds a list of Connections that must be routed.
	 * @return a list of PortInsts to connect which are ordered such that they
	 * form the proper sequence of routes to make. Returns null on error.
	 */
	private List<PortInst> makeOrderedPorts(Network net, List<Connection> netEnds)
	{
		List<PortInst> portEndList = new ArrayList<PortInst>();
		for (int i = 0; i < netEnds.size(); i++)
		{
			PortInst pi = netEnds.get(i).getPortInst();
			if (!pi.getNodeInst().isCellInstance() &&
				((PrimitiveNode)pi.getNodeInst().getProto()).getTechnology() == Generic.tech()) continue;
			if (portEndList.contains(pi)) continue;
			portEndList.add(pi);
		}
		int count = portEndList.size();
		if (count <= 1) return null;
		PortInst[] portEnds = new PortInst[count];
		int k = 0;
		for (PortInst pi : portEndList) portEnds[k++] = pi;

		// find the closest two points
		int closest1 = 0, closest2 = 0;
		double closestDist = Double.MAX_VALUE;
		for (int i = 0; i < count; i++)
		{
			PolyBase poly1 = portEnds[i].getPoly();
			for (int j = i + 1; j < count; j++)
			{
				PolyBase poly2 = portEnds[j].getPoly();
				double dist = poly1.getCenter().distance(poly2.getCenter());
				if (dist < closestDist)
				{
					closestDist = dist;
					closest1 = i;
					closest2 = j;
				}
			}
		}
		List<PortInst> orderedPorts = new ArrayList<PortInst>();
		orderedPorts.add(portEnds[closest1]);
		orderedPorts.add(portEnds[closest2]);
		portEnds[closest1] = null;
		portEnds[closest2] = null;
		for (;;)
		{
			// find closest port to ends of current string
			boolean foundsome = false;
			double closestDist1 = Double.MAX_VALUE, closestDist2 = Double.MAX_VALUE;
			for (int i = 0; i < count; i++)
			{
				if (portEnds[i] == null) continue;
				PolyBase poly = portEnds[i].getPoly();
				PolyBase poly1 = orderedPorts.get(0).getPoly();
				double dist1 = poly.getCenter().distance(poly1.getCenter());
				if (dist1 < closestDist1)
				{
					closestDist1 = dist1;
					closest1 = i;
					foundsome = true;
				}
				PolyBase poly2 = orderedPorts.get(orderedPorts.size() - 1).getPoly();
				double dist2 = poly.getCenter().distance(poly2.getCenter());
				if (dist2 < closestDist2)
				{
					closestDist2 = dist2;
					closest2 = i;
					foundsome = true;
				}
			}
			if (!foundsome) break;
			if (closestDist1 < closestDist2)
			{
				orderedPorts.add(0, portEnds[closest1]);
				portEnds[closest1] = null;
			} else
			{
				orderedPorts.add(portEnds[closest2]);
				portEnds[closest2] = null;
			}
		}
		return orderedPorts;
	}

	/**
	 * Method to sum up the distance that a route takes.
	 * @param vertices the list of SearchVertices in the route.
	 * @return the length of the route.
	 */
	protected static double getVertexLength(List<SearchVertex> vertices)
	{
		if (vertices == null) return Double.MAX_VALUE;
		if (vertices.size() == 0) return Double.MAX_VALUE;
		double sum = 0;
		SearchVertex last = null;
		for (SearchVertex sv : vertices)
		{
			if (last != null)
				sum += Math.abs(sv.getX() - last.getX()) + Math.abs(sv.getY() - last.getY()) +
					Math.abs(sv.getZ() - last.getZ()) * 10;
			last = sv;
		}
		return sum;
	}

	/**
	 * Method to convert a linked list of SearchVertex objects to an optimized path.
	 * @param initialThread the initial SearchVertex in the linked list.
	 * @return a List of SearchVertex objects optimized to consolidate runs in the X or Y axes.
	 */
	List<SearchVertex> getOptimizedList(SearchVertex initialThread)
	{
		List<SearchVertex> realVertices = new ArrayList<SearchVertex>();
		SearchVertex thread = initialThread;
		if (thread != null)
		{
			SearchVertex lastVertex = thread;
			realVertices.add(lastVertex);
			thread = thread.last;
			while (thread != null)
			{
				if (lastVertex.getZ() != thread.getZ())
				{
					realVertices.add(thread);
					lastVertex = thread;
					thread = thread.last;
				} else
				{
					// gather a run of vertices on this layer
					double dx = thread.getX() - lastVertex.getX();
					double dy = thread.getY() - lastVertex.getY();
					lastVertex = thread;
					thread = thread.last;
					while (thread != null)
					{
						if (lastVertex.getZ() != thread.getZ()) break;
						if ((thread.getX() - lastVertex.getX() != 0 && dx == 0) ||
							(thread.getY() - lastVertex.getY() != 0 && dy == 0)) break;
						lastVertex = thread;
						thread = thread.last;
					}
					realVertices.add(lastVertex);
				}
			}
		}
		return realVertices;
	}

	/**
	 * Method to round a value up to the nearest routing grain size.
	 * @param v the value to round up.
	 * @return the granularized value.
	 */
	private double upToGrain(double v)
	{
		if (FULLGRAIN) return v;
		return upToGrainAlways(v);
	}

	/**
	 * Method to round a value up to the nearest routing grain size.
	 * @param v the value to round up.
	 * @return the granularized value.
	 */
	private double upToGrainAlways(double v)
	{
		return Math.ceil(v * GRANULARITY) * GRAINSIZE;
	}

	/**
	 * Method to round a value down to the nearest routing grain size.
	 * @param v the value to round down.
	 * @return the granularized value.
	 */
	private double downToGrain(double v)
	{
		if (FULLGRAIN) return v;
		return downToGrainAlways(v);
	}

	/**
	 * Method to round a value down to the nearest routing grain size.
	 * @param v the value to round down.
	 * @return the granularized value.
	 */
	private double downToGrainAlways(double v)
	{
		return Math.floor(v * GRANULARITY) * GRAINSIZE;
	}

	/********************************* BLOCKAGE CONSTRUCTION *********************************/

	/**
	 * Method to add extra blockage information that corresponds to ends of unrouted arcs.
	 * @param arcsToRoute the list of arcs to route.
	 * @param prefs the preferences to use.
	 */
	private void addBlockagesAtPorts(List<ArcInst> arcsToRoute, SeaOfGates.SeaOfGatesOptions prefs)
	{
		Netlist netList = cell.getNetlist();
		Map<Network, ArcInst[]> arcMap = null;
		if (cell.getView() != View.SCHEMATIC)
			arcMap = netList.getArcInstsByNetwork();

		for (ArcInst ai : arcsToRoute)
		{
			int netID = -1;
			Integer netIDI = netIDs.get(ai);
			if (netIDI != null)
			{
				netID = -(netIDI.intValue() - 1);
				if (netID > 0)
					System.out.println("INTERNAL ERROR! net=" + netID + " but should be negative");
			}
			Network net = netList.getNetwork(ai, 0);
			HashSet<ArcInst> arcsToDelete = new HashSet<ArcInst>();
			HashSet<NodeInst> nodesToDelete = new HashSet<NodeInst>();
			List<Connection> netEnds = Routing.findNetEnds(net, arcMap, arcsToDelete, nodesToDelete, netList, true);
			List<PortInst> orderedPorts = makeOrderedPorts(net, netEnds);
			if (orderedPorts == null) continue;

			// determine the minimum width of arcs on this net
			double minWidth = getMinWidth(orderedPorts, prefs);

			for (PortInst pi : orderedPorts)
			{
				PolyBase poly = pi.getPoly();
				Rectangle2D portBounds = poly.getBounds2D();
				ArcProto[] poss = pi.getPortProto().getBasePort().getConnections();
				int lowMetal = -1, highMetal = -1;
				for (int i = 0; i < poss.length; i++)
				{
					if (poss[i].getTechnology() != tech) continue;
					if (!poss[i].getFunction().isMetal()) continue;
					int level = poss[i].getFunction().getLevel();
					if (lowMetal < 0) lowMetal = highMetal = level; else
					{
						lowMetal = Math.min(lowMetal, level);
						highMetal = Math.max(highMetal, level);
					}
				}
				if (lowMetal < 0) continue;

				// reserve space on layers above and below
				for (int via = lowMetal - 2; via < highMetal; via++)
				{
					if (via < 0 || via >= numMetalLayers - 1) continue;
					MetalVia mv = metalVias[via].getVias().get(0);
					PrimitiveNode np = mv.via;
					SizeOffset so = np.getProtoSizeOffset();
					double xOffset = so.getLowXOffset() + so.getHighXOffset();
					double yOffset = so.getLowYOffset() + so.getHighYOffset();
					double wid = Math.max(np.getDefWidth() - xOffset, minWidth) + xOffset;
					double hei = Math.max(np.getDefHeight() - yOffset, minWidth) + yOffset;
					NodeInst dummy = NodeInst.makeDummyInstance(np, EPoint.ORIGIN, wid, hei, Orientation.IDENT);
					PolyBase[] polys = tech.getShapeOfNode(dummy);
					for (int i = 0; i < polys.length; i++)
					{
						PolyBase metalPoly = polys[i];
						Layer layer = metalPoly.getLayer();
						if (!layer.getFunction().isMetal()) continue;
						Rectangle2D metalBounds = metalPoly.getBounds2D();
						Rectangle2D bounds = new Rectangle2D.Double(metalBounds.getMinX() + portBounds.getCenterX(),
							metalBounds.getMinY() + portBounds.getCenterY(), metalBounds.getWidth(), metalBounds.getHeight());

						// only add blockage if there is nothing else present
						boolean free = true;
						RTNode rtree = metalTrees.get(layer);
						if (rtree != null)
						{
							for (RTNode.Search sea = new RTNode.Search(bounds, rtree, true); sea.hasNext();)
							{
								SOGBound sBound = (SOGBound)sea.next();
								if (sBound.getBounds().getMinX() > bounds.getMaxX() ||
									sBound.getBounds().getMaxX() < bounds.getMinX() ||
									sBound.getBounds().getMinY() > bounds.getMaxY() ||
									sBound.getBounds().getMaxY() < bounds.getMinY()) continue;
								if (Math.abs(sBound.getNetID()) == netID) continue;
								free = false;
								break;
							}
						}
						if (free) addRectangle(bounds, layer, netID);
					}
				}
			}
		}
	}

	/**
	 * Class to define an R-Tree leaf node for geometry in the blockage data structure.
	 */
	public static class SOGBound implements RTBounds
	{
		private Rectangle2D bound;
		private int netID;

		SOGBound(Rectangle2D bound, int netID)
		{
			this.bound = bound;
			this.netID = netID;
		}

		public Rectangle2D getBounds() { return bound; }

		/**
		 * Method to return the global network ID for this SOGBound.
		 * Numbers > 0 are normal network IDs.
		 * Numbers <= 0 are blockages added around the ends of routes.
		 * @return the global network ID for this SOGBound.
		 */
		public int getNetID() { return netID; }

		public String toString() { return "SOGBound on net " + netID; }
	}

	private static class SOGPoly extends SOGBound
	{
		private PolyBase poly;

		SOGPoly(Rectangle2D bound, int netID, PolyBase poly)
		{
			super(bound, netID);
			this.poly = poly;
		}

		public PolyBase getPoly() { return poly; }
	}

	/**
	 * Class to define an R-Tree leaf node for vias in the blockage data structure.
	 */
	private static class SOGVia implements RTBounds
	{
		private Point2D loc;
		private int netID;

		SOGVia(Point2D loc, int netID)
		{
			this.loc = loc;
			this.netID = netID;
		}

		public Rectangle2D getBounds()
		{
			return new Rectangle2D.Double(loc.getX(), loc.getY(), 0, 0);
		}

		public int getNetID() { return netID; }

		public String toString() { return "SOGVia on net " + netID; }
	}

	/**
	 * Method to see if a proposed piece of metal has DRC errors (ignoring notches).
	 * @param netID the network ID of the desired metal (blockages on this netID are ignored).
	 * @param metNo the level of the metal.
	 * @param halfWidth half of the width of the metal.
	 * @param halfHeight half of the height of the metal.
	 * @param surround is the maximum possible DRC surround around the metal.
	 * @param x the X coordinate at the center of the metal.
	 * @param y the Y coordinate at the center of the metal.
	 * @return a blocking SOGBound object that is in the area. Returns null if the area is clear.
	 */
	private SOGBound getMetalBlockage(int netID, int metNo, double halfWidth, double halfHeight,
		double surround, double x, double y)
	{
		// get the R-Tree data for the metal layer
		Layer layer = metalLayers[metNo];
		RTNode rtree = metalTrees.get(layer);
		if (rtree == null) return null;

		// compute the area to search
		double lX = x - halfWidth - surround, hX = x + halfWidth + surround;
		double lY = y - halfHeight - surround, hY = y + halfHeight + surround;
		Rectangle2D searchArea = new Rectangle2D.Double(lX, lY, hX - lX, hY - lY);

		// see if there is anything in that area
		for (RTNode.Search sea = new RTNode.Search(searchArea, rtree, true); sea.hasNext(); )
		{
			SOGBound sBound = (SOGBound) sea.next();
			Rectangle2D bound = sBound.getBounds();
			if (DBMath.isLessThanOrEqualTo(bound.getMaxX(), lX) ||
				DBMath.isGreaterThanOrEqualTo(bound.getMinX(), hX) ||
				DBMath.isLessThanOrEqualTo(bound.getMaxY(), lY) ||
				DBMath.isGreaterThanOrEqualTo(bound.getMinY(), hY)) continue;

			// ignore if on the same net
			if (Math.abs(sBound.getNetID()) == netID) continue;

			// if this is a polygon, do closer examination
			if (sBound instanceof SOGPoly)
			{
				PolyBase poly = ((SOGPoly) sBound).getPoly();
				if (!poly.contains(searchArea)) continue;
			}
			return sBound;
		}
		return null;
	}

	/**
	 * Method to find a via blockage in the R-Tree.
	 * @param netID the network ID of the desired space (vias at this point and on this netID are ignored).
	 * @param layer the via layer being examined.
	 * @param halfWidth half of the width of the area to examine.
	 * @param halfHeight half of the height of the area to examine.
	 * @param x the X coordinate at the center of the area to examine.
	 * @param y the Y coordinate at the center of the area to examine.
	 * @return a blocking SOGVia object that is in the area. Returns null if the area is clear.
	 */
	private SOGVia getViaBlockage(int netID, Layer layer, double halfWidth, double halfHeight, double x, double y)
	{
		RTNode rtree = viaTrees.get(layer);
		if (rtree == null) return null;

		// see if there is anything in that area
		Rectangle2D searchArea = new Rectangle2D.Double(x - halfWidth, y - halfHeight, halfWidth * 2, halfHeight * 2);
		for (RTNode.Search sea = new RTNode.Search(searchArea, rtree, true); sea.hasNext();)
		{
			SOGVia sLoc = (SOGVia) sea.next();
			if (sLoc.getNetID() == netID)
			{
				if (sLoc.loc.getX() == x && sLoc.loc.getY() == y) continue;
			}
			return sLoc;
		}
		return null;
	}

//double infoAreaX, infoAreaY, infoAreaXLow, infoAreaYLow;

	/**
	 * HierarchyEnumerator subclass to examine a cell for a given layer and fill an R-Tree.
	 */
	private class BlockageVisitor extends HierarchyEnumerator.Visitor
	{
		private List<ArcInst> arcsToRoute;
		private boolean didTopLevel;

		public BlockageVisitor(List<ArcInst> arcsToRoute)
		{
			this.arcsToRoute = arcsToRoute;
			didTopLevel = false;
		}

		public boolean enterCell(HierarchyEnumerator.CellInfo info)
		{
			return true;
		}

		public void exitCell(HierarchyEnumerator.CellInfo info)
		{
			Cell cell = info.getCell();
			Netlist nl = info.getNetlist();
			AffineTransform trans = info.getTransformToRoot();
			for (Iterator<ArcInst> it = cell.getArcs(); it.hasNext(); )
			{
				ArcInst ai = it.next();
				int netID = -1;
				Network net = nl.getNetwork(ai, 0);
				if (net != null) netID = info.getNetID(net);
				Technology tech = ai.getProto().getTechnology();
				PolyBase[] polys = tech.getShapeOfArc(ai);
				for (int i = 0; i < polys.length; i++)
					addLayer(polys[i], trans, netID, false);
			}
		}

		public boolean visitNodeInst(Nodable no, HierarchyEnumerator.CellInfo info)
		{
			if (info.isRootCell() && !didTopLevel)
			{
				didTopLevel = true;
				if (arcsToRoute != null)
				{
					Netlist nl = info.getNetlist();
					for (ArcInst ai : arcsToRoute)
					{
						Network net = nl.getNetwork(ai, 0);
						int netID = info.getNetID(net);
						netIDs.put(ai, new Integer(netID + 1));
					}
				}
			}

			Cell cell = info.getCell();
			Map<NodeInst,Network> nodeExtraction = nodeExtractions.get(cell);
			if (nodeExtraction == null)
			{
				nodeExtraction = new HashMap<NodeInst,Network>();
				nodeExtractions.put(cell, nodeExtraction);

				// first fill in all exported pure-layer nodes
				for(Iterator<NodeInst> it = cell.getNodes(); it.hasNext(); )
				{
					NodeInst ni = it.next();

					// must be an exported pure-layer node
					if (!ni.hasExports()) continue;
					if (ni.getFunction() != PrimitiveNode.Function.NODE) continue;

					// ignore if already processed
					if (nodeExtraction.containsKey(ni)) continue;

					// add to the node extraction
					Netlist nl = info.getNetlist();
					Network net = nl.getNetwork(ni, ni.getOnlyPortInst().getPortProto(), 0);
					nodeExtraction.put(ni, net);
//boolean debug = (ni.getBounds().getMinX()-10 <= infoAreaXLow) && (ni.getBounds().getMaxX()+10 >= infoAreaXLow) &&
//	(ni.getBounds().getMinY()-10 <= infoAreaYLow) && (ni.getBounds().getMaxY()+10 >= infoAreaYLow);
//if (debug) System.out.println("IN CELL "+info.getCell().describe(false)+", PLACING EXPORTED NET "+net.describe(false)+" ON "+ni.describe(false)+" AT "+ni.getBounds().getMinX()+"<=X<="+
//	ni.getBounds().getMaxX()+" AND "+ni.getBounds().getMinY()+"<=Y<="+ni.getBounds().getMaxY());
					// look for others that touch
					recursivelyExtract(ni, net, nodeExtraction);
				}

				// next fill in remaining pure-layer nodes
				for(Iterator<NodeInst> it = cell.getNodes(); it.hasNext(); )
				{
					NodeInst ni = it.next();

					// must be a pure-layer node
					if (ni.getFunction() != PrimitiveNode.Function.NODE) continue;

					// ignore if already processed
					if (nodeExtraction.containsKey(ni)) continue;

					// add to the node extraction
					Netlist nl = info.getNetlist();
					Network net = nl.getNetwork(ni, ni.getOnlyPortInst().getPortProto(), 0);
					nodeExtraction.put(ni, net);
//boolean debug = (ni.getBounds().getMinX()-10 <= infoAreaXLow) && (ni.getBounds().getMaxX()+10 >= infoAreaXLow) &&
//	(ni.getBounds().getMinY()-10 <= infoAreaYLow) && (ni.getBounds().getMaxY()+10 >= infoAreaYLow);
//if (debug) System.out.println("IN CELL "+info.getCell().describe(false)+", PLACING REGULAR NET "+netID+" ON "+ni.describe(false)+" AT "+ni.getBounds().getMinX()+"<=X<="+
//	ni.getBounds().getMaxX()+" AND "+ni.getBounds().getMinY()+"<=Y<="+ni.getBounds().getMaxY());
					// look for others that touch
					recursivelyExtract(ni, net, nodeExtraction);
				}
			}

			NodeInst ni = no.getNodeInst();
			if (!ni.isCellInstance())
			{
				Network net = nodeExtraction.get(ni);
				Netlist nl = info.getNetlist();
				AffineTransform trans = info.getTransformToRoot();
				AffineTransform nodeTrans = ni.rotateOut(trans);
				PrimitiveNode pNp = (PrimitiveNode) ni.getProto();
				Technology tech = pNp.getTechnology();
				Poly[] nodeInstPolyList = tech.getShapeOfNode(ni, true, false, null);
				boolean canPlacePseudo = info.isRootCell();
				if (!ni.hasExports()) canPlacePseudo = false;
				for (int i = 0; i < nodeInstPolyList.length; i++)
				{
					PolyBase poly = nodeInstPolyList[i];
					if (net == null && poly.getPort() != null)
						net = nl.getNetwork(no, poly.getPort(), 0);
					int netID = -1;
					if (net != null) netID = info.getNetID(net);
					addLayer(poly, nodeTrans, netID, canPlacePseudo);
				}
			}
			return true;
		}

		private void recursivelyExtract(NodeInst ni, Network net, Map<NodeInst,Network> nodeExtraction)
		{
			Rectangle2D bounds = ni.getBounds();
			Rectangle2D bound = new Rectangle2D.Double(bounds.getMinX()-1, bounds.getMinY()-1, bounds.getWidth()+2, bounds.getHeight()+2);
			Cell cell = ni.getParent();
			for(Iterator<RTBounds> it = cell.searchIterator(bound); it.hasNext(); )
			{
				// must be a pure-layer node on the same layer
				Geometric geom = (Geometric)it.next();
				if (!(geom instanceof NodeInst)) continue;
				NodeInst oNi = (NodeInst)geom;
				if (oNi.getProto() != ni.getProto()) continue;

				// ignore if the node doesn't touch this one
				Rectangle2D oBounds = oNi.getBounds();
				if (!DBMath.rectsIntersect(bounds, oBounds)) continue;
//boolean debug = (oNi.getBounds().getMinX()-10 <= infoAreaXLow) && (oNi.getBounds().getMaxX()+10 >= infoAreaXLow) &&
//	(oNi.getBounds().getMinY()-10 <= infoAreaYLow) && (oNi.getBounds().getMaxY()+10 >= infoAreaYLow);

				Network oNet = nodeExtraction.get(oNi);
				if (oNet != null)
				{
					// merged into a different region: unify their network numbers
					if (net != oNet)
					{
//if (debug && upDeb) System.out.println("  MERGING NET "+net+" AND "+oNet+" ON "+oNi.getProto().describe(false)+" AT "+oNi.getBounds().getMinX()+"<=X<="+
//	oNi.getBounds().getMaxX()+" AND "+oNi.getBounds().getMinY()+"<=Y<="+oNi.getBounds().getMaxY());
						List<NodeInst> switchEm = new ArrayList<NodeInst>();
						for(NodeInst testNI : nodeExtraction.keySet())
							if (nodeExtraction.get(testNI) == net)
								switchEm.add(testNI);
						for(NodeInst testNI : switchEm)
							nodeExtraction.put(testNI, oNet);
					}
				} else
				{
//if (debug && upDeb) System.out.println("  INCLUDING NET "+net+" ON "+oNi.getProto().describe(false)+" AT "+oNi.getBounds().getMinX()+"<=X<="+
//	oNi.getBounds().getMaxX()+" AND "+oNi.getBounds().getMinY()+"<=Y<="+oNi.getBounds().getMaxY());
					// found new piece for this region
					nodeExtraction.put(oNi, net);
					recursivelyExtract(oNi, net, nodeExtraction);
				}
			}
		}
	}

	/**
	 * Method to create a NodeInst and update the R-Trees.
	 * @param np the prototype of the new NodeInst.
	 * @param loc the location of the new NodeInst.
	 * @param wid the width of the new NodeInst.
	 * @param hei the height of the new NodeInst.
	 * @param orient the orientation of the new NodeInst.
	 * @param cell the Cell in which to place the new NodeInst.
	 * @param netID the network ID of geometry in this NodeInst.
	 * @return the NodeInst that was created (null on error).
	 */
	private NodeInst makeNodeInst(NodeProto np, EPoint loc, double wid, double hei, Orientation orient,
		Cell cell, int netID)
	{
		NodeInst ni = NodeInst.makeInstance(np, loc, wid, hei, cell, orient, null);
		if (ni != null)
		{
			AffineTransform trans = ni.rotateOut();
			Poly[] nodeInstPolyList = tech.getShapeOfNode(ni, true, false, null);
			for (int i = 0; i < nodeInstPolyList.length; i++)
			{
				PolyBase poly = nodeInstPolyList[i];
				if (poly.getPort() == null) continue;
				poly.transform(trans);
				addLayer(poly, GenMath.MATID, netID, false);
			}
		}
		return ni;
	}

	/**
	 * Method to create an ArcInst and update the R-Trees.
	 * @param type the prototype of the new ArcInst.
	 * @param wid the width of the new ArcInst.
	 * @param from the head PortInst of the new ArcInst.
	 * @param to the tail PortInst of the new ArcInst.
	 * @param netID the network ID of geometry in this ArcInst.
	 * @return the ArcInst that was created (null on error).
	 */
	private ArcInst makeArcInst(ArcProto type, double wid, PortInst from, PortInst to, int netID)
	{
		ArcInst ai = ArcInst.makeInstanceBase(type, wid, from, to);
		if (ai != null)
		{
			PolyBase[] polys = tech.getShapeOfArc(ai);
			for (int i = 0; i < polys.length; i++)
				addLayer(polys[i], GenMath.MATID, netID, false);

			// accumulate the total length of wires placed
			PolyBase fromPoly = from.getPoly();
			PolyBase toPoly = to.getPoly();
			double length = fromPoly.getCenter().distance(toPoly.getCenter());
			totalWireLength += length;
		}
		return ai;
	}

	/**
	 * Method to add geometry to the R-Tree.
	 * @param poly the polygon to add (only rectangles are added, so the bounds is used).
	 * @param trans a transformation matrix to apply to the polygon.
	 * @param netID the global network ID of the geometry.
	 * @param canPlacePseudo true if pseudo-layers should be considered
	 * (converted to non-pseudo and stored). False to ignore pseudo-layers.
	 */
	private void addLayer(PolyBase poly, AffineTransform trans, int netID, boolean canPlacePseudo)
	{
		if (!canPlacePseudo && poly.isPseudoLayer()) return;
		Layer layer = poly.getLayer();
		if (canPlacePseudo) layer = layer.getNonPseudoLayer(); else
		{
			if (layer.isPseudoLayer()) return;
		}
		Layer.Function fun = layer.getFunction();
		if (fun.isMetal())
		{
			poly.transform(trans);
			Rectangle2D bounds = poly.getBox();
			if (bounds == null) addPolygon(poly, layer, netID); else
				addRectangle(bounds, layer, netID);
		} else if (fun.isContact())
		{
			Rectangle2D bounds = poly.getBounds2D();
			DBMath.transformRect(bounds, trans);
			addVia(new EPoint(bounds.getCenterX(), bounds.getCenterY()), layer, netID);
		}
	}

	/**
	 * Method to add a rectangle to the metal R-Tree.
	 * @param bounds the rectangle to add.
	 * @param layer the metal layer on which to add the rectangle.
	 * @param netID the global network ID of the geometry.
	 */
	private void addRectangle(Rectangle2D bounds, Layer layer, int netID)
	{
		RTNode root = metalTrees.get(layer);
		if (root == null)
		{
			root = RTNode.makeTopLevel();
			metalTrees.put(layer, root);
		}
		RTNode newRoot = RTNode.linkGeom(null, root, new SOGBound(ERectangle.fromLambda(bounds), netID));
		if (newRoot != root) metalTrees.put(layer, newRoot);
	}

	/**
	 * Method to add a polygon to the metal R-Tree.
	 * @param poly the polygon to add.
	 * @param layer the metal layer on which to add the rectangle.
	 * @param netID the global network ID of the geometry.
	 */
	private void addPolygon(PolyBase poly, Layer layer, int netID)
	{
		RTNode root = metalTrees.get(layer);
		if (root == null)
		{
			root = RTNode.makeTopLevel();
			metalTrees.put(layer, root);
		}
		Rectangle2D bounds = poly.getBounds2D();
		RTNode newRoot = RTNode.linkGeom(null, root, new SOGPoly(ERectangle.fromLambda(bounds), netID, poly));
		if (newRoot != root) metalTrees.put(layer, newRoot);
	}

	/**
	 * Method to add a point to the via R-Tree.
	 * @param loc the point to add.
	 * @param layer the via layer on which to add the point.
	 * @param netID the global network ID of the geometry.
	 */
	private void addVia(EPoint loc, Layer layer, int netID)
	{
		RTNode root = viaTrees.get(layer);
		if (root == null)
		{
			root = RTNode.makeTopLevel();
			viaTrees.put(layer, root);
		}
		RTNode newRoot = RTNode.linkGeom(null, root, new SOGVia(loc, netID));
		if (newRoot != root) viaTrees.put(layer, newRoot);
	}

	/**
	 * Class to define a list of possible nodes that can connect two layers.
	 * This includes orientation.
	 */
	private static class MetalVia
	{
		PrimitiveNode via;
		int orientation;

		MetalVia(PrimitiveNode v, int o)
		{
			via = v;
			orientation = o;
		}
	}

	/**
	 * Class to define a list of possible nodes that can connect two layers.
	 * This includes orientation.
	 */
	private static class MetalVias
	{
		List<MetalVia> vias = new ArrayList<MetalVia>();

		void addVia(PrimitiveNode pn, int o)
		{
			vias.add(new MetalVia(pn, o));
			Collections.sort(vias, new PrimsBySize());
		}

		List<MetalVia> getVias() { return vias; }
	}

	/**
	 * Comparator class for sorting primitives by their size.
	 */
	private static class PrimsBySize implements Comparator<MetalVia>
	{
		/**
		 * Method to sort primitives by their size.
		 */
		public int compare(MetalVia mv1, MetalVia mv2)
		{
			PrimitiveNode pn1 = mv1.via;
			PrimitiveNode pn2 = mv2.via;
			double sz1 = pn1.getDefWidth() * pn1.getDefHeight();
			double sz2 = pn2.getDefWidth() * pn2.getDefHeight();
			if (sz1 < sz2) return -1;
			if (sz1 > sz2) return 1;
			return 0;
		}
	}

	/************************************** DEBUGGING **************************************/

//	/**
//	 * Debugging method to dump an R-Tree plane.
//	 * @param z the metal layer to dump.
//	 */
//	private void dumpPlane(int z)
//	{
//		System.out.println("---------------------- METAL " + (z + 1) + " ----------------------");
//		Map<Integer, Map<Integer, SearchVertex>> plane = searchVertexPlanes[z];
//		if (plane == null) return;
//		List<Integer> yValues = new ArrayList<Integer>();
//		for (Iterator<Integer> it = plane.keySet().iterator(); it.hasNext();)
//			yValues.add(it.next());
//		Collections.sort(yValues);
//		int lowY = yValues.get(0).intValue();
//		int highY = yValues.get(yValues.size() - 1).intValue();
//		int lowX = 1, highX = 0;
//		for (Integer y : yValues)
//		{
//			Map<Integer, SearchVertex> row = plane.get(y);
//			for (Iterator<Integer> it = row.keySet().iterator(); it.hasNext();)
//			{
//				Integer x = it.next();
//				int xv = x.intValue();
//				if (lowX > highX) lowX = highX = xv; else
//				{
//					if (xv < lowX) lowX = xv;
//					if (xv > highX) highX = xv;
//				}
//			}
//		}
//		String lowXStr = " " + lowX;
//		String highXStr = " " + highX;
//		int numXDigits = Math.max(lowXStr.length(), highXStr.length());
//		String lowYStr = " " + lowY;
//		String highYStr = " " + highY;
//		int numYDigits = Math.max(lowYStr.length(), highYStr.length());
//
//		String space = "   ";
//		while (space.length() < numYDigits + 3) space += " ";
//		System.out.print(space);
//		for (int x = lowX; x <= highX; x++)
//		{
//			String xCoord = " " + x;
//			while (xCoord.length() < numXDigits) xCoord = " " + xCoord;
//			System.out.print(xCoord);
//		}
//		System.out.println();
//
//		for (int y = highY; y >= lowY; y--)
//		{
//			String yCoord = " " + y;
//			while (yCoord.length() < numYDigits) yCoord = " " + yCoord;
//			System.out.print("Row" + yCoord);
//			Map<Integer, SearchVertex> row = plane.get(y);
//			if (row != null)
//			{
//				for (int x = lowX; x <= highX; x++)
//				{
//					SearchVertex sv = row.get(x);
//					String xCoord;
//					if (sv == null) xCoord = " X"; else xCoord = " " + sv.cost;
//					while (xCoord.length() < numXDigits) xCoord = " " + xCoord;
//					System.out.print(xCoord);
//				}
//			}
//			System.out.println();
//		}
//	}

	/************************************** 3D DEBUGGING **************************************/

	private void showSearchVertices(Set<SearchVertex> active, List<SearchVertex> inactive, Cell cell,
		double fromX, double fromY, double toX, double toY, Map<Layer, RTNode> trees, Rectangle2D bounds)
	{
		SwingUtilities.invokeLater(new Show3DRoute(active, inactive, viaTrees, metalLayers, viaLayers, cell,
			fromX, fromY, toX, toY, trees, bounds));
	}

	/**
	 * Class to render a failed route in 3D.
	 */
	private static class Show3DRoute implements Runnable
	{
		private Set<SearchVertex> active;
		private List<SearchVertex> inactive;
		private Map<Layer,RTNode> vias;
		private Layer[] metalLayers;
		private Layer[] viaLayers;
		private Cell cell;
		private double fromX, fromY, toX, toY;
		private Map<Layer, RTNode> trees;
		private Rectangle2D bounds;

		/**
		 * Constructor to show a failed route in 3D.
		 * @param metalLayers the actual metal Layers.
		 * @param cell the Cell being rendered.
		 * @param fromX start X coordinate of the route.
		 * @param fromY end X coordinate of the route.
		 * @param toX start Y coordinate of the route.
		 * @param toY end Y coordinate of the route.
		 */
		Show3DRoute(Set<SearchVertex> active, List<SearchVertex> inactive, Map<Layer,RTNode> vias,
			Layer[] metalLayers, Layer[] viaLayers, Cell cell, double fromX, double fromY, double toX, double toY,
			Map<Layer, RTNode> trees, Rectangle2D bounds)
		{
			this.active = active;
			this.inactive = inactive;
			this.vias = vias;
			this.metalLayers = metalLayers;
			this.viaLayers = viaLayers;
			this.cell = cell;
			this.fromX = fromX;
			this.fromY = fromY;
			this.toX = toX;
			this.toY = toY;
			this.trees = trees;
			this.bounds = bounds;
		}

		public void run()
		{
			// see if 3D is available
			Method showPolysMethod = null;
			try
			{
				Class<?> threeDClass = Class.forName("com.sun.electric.plugins.j3d.View3DWindow");
				showPolysMethod = threeDClass.getMethod("show3DPolygons", new Class[] {ArrayList.class});
			} catch (Exception e)
			{
				System.out.println("Problem with 3D view: "+e.getMessage());
			}
//showPolysMethod = null;
			if (showPolysMethod == null)
			{
				EditWindow_ wnd = Job.getUserInterface().getCurrentEditWindow_();
				Cell drawCell = wnd.getCell();
				wnd.clearHighlighting();
				for(SearchVertex sv : active)
					showSV(sv, wnd, drawCell);
				for(SearchVertex sv : inactive)
					showSV(sv, wnd, drawCell);
				wnd.finishedHighlighting();
				return;
			}

			List<PolyBase> polys = new ArrayList<PolyBase>();

			// show the search vertices
			for(SearchVertex sv : active)
				displaySearchVertex(sv, polys);
			for(SearchVertex sv : inactive)
				displaySearchVertex(sv, polys);

//			// aggregate the search points in areas
//			PolyMerge pm = new PolyMerge();
//			for (int i = 0; i < planes.length; i++)
//			{
//				int numDots = 0;
//				Map<Double, Set<Double>> plane = planes[i];
//				if (plane == null) continue;
//				for (Double y : plane.keySet())
//				{
//					double yv = y.doubleValue();
//					Set<Double> row = plane.get(y);
//					for (Double x : row)
//					{
//						double xv = x.doubleValue();
//						Point2D[] showPT = new Point2D[4];
//						showPT[0] = new Point2D.Double(xv - 1, yv - 1);
//						showPT[1] = new Point2D.Double(xv - 1, yv + 1);
//						showPT[2] = new Point2D.Double(xv + 1, yv + 1);
//						showPT[3] = new Point2D.Double(xv + 1, yv - 1);
//						Poly poly = new Poly(showPT);
//						poly.setLayer(layers[i]);
//						poly.setStyle(Type.FILLED);
//						pm.addPolygon(poly.getLayer(), poly);
//						numDots++;
//						if (i > 0 && numDots > 100) break;
//					}
//					if (i > 0 && numDots > 100) break;
//				}
//			}
//			for (Layer layer : pm.getKeySet())
//			{
//				double thickness = layer.getThickness();
//				double lowZ = layer.getDistance();
//				double highZ = lowZ + thickness;
//				double inset = thickness/3;
//				lowZ += inset;   highZ -= inset;
//				List<PolyBase> merges = pm.getMergedPoints(layer, true);
//				for(PolyBase merged : merges)
//				{
//					Poly3D p3d = new Poly3D(merged.getPoints(), lowZ, highZ);
//					p3d.setLayer(layer);
//					p3d.setStyle(Type.FILLED);
//					polys.add(p3d);
//				}
//			}

			// show all blockages in the area
			int highestUsedLayer = 0;
			for(int lay = 0; lay < metalLayers.length; lay++)
			{
				Layer layer = metalLayers[lay];
				RTNode rtree = trees.get(layer);
				if (rtree != null)
				{
					for (RTNode.Search sea = new RTNode.Search(bounds, rtree, true); sea.hasNext(); )
					{
						SOGBound sBound = (SOGBound) sea.next();
						double lX = sBound.getBounds().getMinX();
						double hX = sBound.getBounds().getMaxX();
						double lY = sBound.getBounds().getMinY();
						double hY = sBound.getBounds().getMaxY();
						if (hX <= bounds.getMinX() || lX >= bounds.getMaxX() ||
							hY <= bounds.getMinY() || lY >= bounds.getMaxY()) continue;
						if (lX < bounds.getMinX()) lX = bounds.getMinX();
						if (hX > bounds.getMaxX()) hX = bounds.getMaxX();
						if (lY < bounds.getMinY()) lY = bounds.getMinY();
						if (hY > bounds.getMaxY()) hY = bounds.getMaxY();
						double lowZ = layer.getDistance();
						Poly3D poly = new Poly3D((lX+hX)/2, (lY+hY)/2, hX-lX, hY-lY, lowZ, lowZ);
						poly.setStyle(Type.FILLED);
						EGraphics graphics = layer.getGraphics();
						poly.setColor(graphics.getColor());
						poly.setTransparency(0.1f);
						polys.add(poly);
						if (lay > highestUsedLayer) highestUsedLayer = lay;
					}
				}
			}

			// show vias
			for(int i=0; i<highestUsedLayer; i++)
			{
				Layer layer = viaLayers[i];
				Layer m1Layer = metalLayers[i];
				Layer m2Layer = metalLayers[i+1];
				RTNode rtree = vias.get(layer);
				if (rtree != null)
				{
					for (RTNode.Search sea = new RTNode.Search(bounds, rtree, true); sea.hasNext(); )
					{
						SOGVia sBound = (SOGVia)sea.next();
						double cX = sBound.getBounds().getCenterX();
						double cY = sBound.getBounds().getCenterY();
						if (cX <= bounds.getMinX() || cX >= bounds.getMaxX() ||
							cY <= bounds.getMinY() || cY >= bounds.getMaxY()) continue;
						double lowZ = m1Layer.getDistance();
						double highZ = m2Layer.getDistance();
						Poly3D poly = new Poly3D(cX, cY, 1, 1, lowZ, highZ);
						poly.setStyle(Type.FILLED);
						poly.setColor(Color.BLACK);
						poly.setTransparency(0.1f);
						polys.add(poly);
					}
				}
			}

			// draw the line from start to end of routing path
			showRoutingPath(polys, highestUsedLayer);

			// render it
			try
			{
				showPolysMethod.invoke(null, new Object[] {polys});
			} catch (Exception e)
			{
				System.out.println("3D rendering error: "+e.getMessage());
			}
		}

		private void showSV(SearchVertex sv, EditWindow_ wnd, Cell cellToUse)
		{
			String message = "M" + (sv.getZ()+1) + "=" + sv.cost;
			Point2D loc = new Point2D.Double(sv.getX(), sv.getY());
			wnd.addHighlightMessage(cellToUse, message, loc);
		}

		private void showRoutingPath(List<PolyBase> polys, int highestUsedLayer)
		{
			double overallDepth = 20;
			Technology tech = cell.getTechnology();

			// include a line indicating the desired route
			Point2D fromPT = new Point2D.Double(fromX, fromY);
			Point2D toPT = new Point2D.Double(toX, toY);
			Point2D[] twoPT = new Point2D[2];
			twoPT[0] = fromPT;   twoPT[1] = toPT;
			double lowZ = metalLayers[0].getDistance() - overallDepth;
			double highZ = metalLayers[highestUsedLayer].getDistance() + metalLayers[highestUsedLayer].getThickness() + overallDepth;
			Poly3D routePoly = new Poly3D(twoPT, lowZ, lowZ);
			routePoly.setLayer(tech.findLayer("Metal-1"));
			routePoly.setStyle(Type.OPENED);
			routePoly.setColor(Color.WHITE);
			polys.add(routePoly);

			// draw an arrowhead showing the routing direction
			double cX = (fromX + toX) / 2, cY = (fromY + toY) / 2;
			Point2D ctr = new Point2D.Double(cX, cY);
			int angle = DBMath.figureAngle(toPT, fromPT);
			int arrowAngle = (angle + 300) % 3600;
			double arrowHeadLength = GenMath.distBetweenPoints(fromPT, toPT) / 10;
			double endX = DBMath.cos(arrowAngle) * arrowHeadLength + cX;
			double endY = DBMath.sin(arrowAngle) * arrowHeadLength + cY;
			twoPT = new Point2D[2];
			twoPT[0] = ctr;   twoPT[1] = new Point2D.Double(endX, endY);
			routePoly = new Poly3D(twoPT, lowZ, lowZ);
			routePoly.setLayer(tech.findLayer("Metal-1"));
			routePoly.setStyle(Type.OPENED);
			routePoly.setColor(Color.WHITE);
			polys.add(routePoly);
			arrowAngle = (angle + 3300) % 3600;
			endX = DBMath.cos(arrowAngle) * arrowHeadLength + cX;
			endY = DBMath.sin(arrowAngle) * arrowHeadLength + cY;
			twoPT = new Point2D[2];
			twoPT[0] = ctr;   twoPT[1] = new Point2D.Double(endX, endY);
			routePoly = new Poly3D(twoPT, lowZ, lowZ);
			routePoly.setLayer(tech.findLayer("Metal-1"));
			routePoly.setStyle(Type.OPENED);
			routePoly.setColor(Color.WHITE);
			polys.add(routePoly);

			// draw vertical posts at the ends
			Point2D[] onePT = new Point2D[1];
			onePT[0] = fromPT;
			routePoly = new Poly3D(onePT, lowZ, highZ);
			routePoly.setLayer(tech.findLayer("Metal-1"));
			routePoly.setStyle(Type.OPENED);
			routePoly.setColor(Color.WHITE);
			polys.add(routePoly);
			onePT = new Point2D[1];
			onePT[0] = toPT;
			routePoly = new Poly3D(onePT, lowZ, highZ);
			routePoly.setLayer(tech.findLayer("Metal-1"));
			routePoly.setStyle(Type.OPENED);
			routePoly.setColor(Color.WHITE);
			polys.add(routePoly);
		}

		private void displaySearchVertex(SearchVertex sv, List<PolyBase> polys)
		{
			if (sv.last == null) return;
			Technology tech = cell.getTechnology();
			double pathSize = 0.75;
			if (sv.getZ() == sv.last.getZ())
			{
				double z = metalLayers[sv.getZ()].getDistance() - 5;
				Point2D[] pts = new Point2D[4];
				double lX = Math.min(sv.xv, sv.last.xv);
				double hX = Math.max(sv.xv, sv.last.xv);
				double lY = Math.min(sv.yv, sv.last.yv);
				double hY = Math.max(sv.yv, sv.last.yv);
				pts[0] = new Point2D.Double(lX-pathSize, lY-pathSize);
				pts[1] = new Point2D.Double(lX-pathSize, hY+pathSize);
				pts[2] = new Point2D.Double(hX+pathSize, hY+pathSize);
				pts[3] = new Point2D.Double(hX+pathSize, lY-pathSize);
				Poly3D routePoly = new Poly3D(pts, z, z);
				routePoly.setLayer(tech.findLayer("Metal-1"));
				routePoly.setColor(Color.RED);
				routePoly.setStyle(Type.CLOSED);
				polys.add(routePoly);
				return;
			} else
			{
				double fromZ = metalLayers[sv.getZ()].getDistance() - 5;
				double toZ = metalLayers[sv.last.getZ()].getDistance() - 5;
				Point2D[] pts = new Point2D[4];
				pts[0] = new Point2D.Double(sv.xv-pathSize, sv.yv-pathSize);
				pts[1] = new Point2D.Double(sv.xv-pathSize, sv.yv+pathSize);
				pts[2] = new Point2D.Double(sv.xv+pathSize, sv.yv+pathSize);
				pts[3] = new Point2D.Double(sv.xv+pathSize, sv.yv-pathSize);
				Poly3D routePoly = new Poly3D(pts, fromZ, toZ);
				routePoly.setLayer(tech.findLayer("Metal-1"));
				routePoly.setColor(Color.GREEN);
				routePoly.setStyle(Type.CLOSED);
				polys.add(routePoly);

				Point2D[] pts1 = new Point2D[1];
				pts1[0] = new Point2D.Double(sv.xv, sv.yv);
				routePoly = new Poly3D(pts1, fromZ, fromZ);
				routePoly.setLayer(tech.findLayer("Metal-1"));
				routePoly.setColor(Color.BLACK);
				routePoly.setStyle(Type.TEXTCENT);
				routePoly.setText(sv.cost+"");
				polys.add(routePoly);
				return;
			}
		}
	}

}
