/*
 * Electric(tm) VLSI Design System
 *
 * File: usrdiacom.c
 * Special command dialogs
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "usrdiacom.h"
#include "usrtrack.h"
#include "efunction.h"
#include "tecart.h"
#include "tecschem.h"
#include "tecgen.h"
#include "usredtec.h"
#include "eio.h"
#include "edialogs.h"
#include "conlay.h"
#include "tecmocmossub.h"
#include <math.h>

struct butlist
{
	INTBIG value;  INTBIG button;
};

/* File Progress */
DIALOGITEM us_progressdialogitems[] =
{
 /*  1 */ {0, {32,8,49,230}, PROGRESS, ""},
 /*  2 */ {0, {8,8,24,230}, MESSAGE, "Reading file..."}
};
DIALOG us_progressdialog = {{50,75,112,314}, 0, 2, us_progressdialogitems};

/* icons for text and port dialogs */
unsigned char us_icon200[] =
{
	0377, 0377, 0377, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 040, 0, 0200, 0200, 060, 0, 0200, 0203, 0370, 0, 0200,
	0200, 060, 0, 0200, 0200, 040, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0377, 0377, 0377, 0200, 0, 0, 0, 0,
	0377, 0377, 0377, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 010, 0, 0200, 0200, 010, 0, 0200,
	0200, 010, 0, 0200, 0200, 010, 0, 0200, 0200, 076, 0, 0200, 0200, 034, 0, 0200,
	0200, 010, 0, 0200, 0200, 0, 0, 0200, 0377, 0377, 0377, 0200, 0, 0, 0, 0
};
unsigned char us_icon201[] =
{
	0377, 0377, 0377, 0200, 0200, 0, 0, 0200, 0200, 010, 0, 0200, 0200, 034, 0, 0200,
	0200, 076, 0, 0200, 0200, 010, 0, 0200, 0200, 010, 0, 0200, 0200, 010, 0, 0200,
	0200, 010, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0377, 0377, 0377, 0200, 0, 0, 0, 0,
	0377, 0377, 0377, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 010, 0200, 0200, 0, 014, 0200, 0200, 0, 0376, 0200,
	0200, 0, 014, 0200, 0200, 0, 010, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0377, 0377, 0377, 0200, 0, 0, 0, 0
};
unsigned char us_icon202[] =
{
	0377, 0377, 0377, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0210, 0, 0, 0200, 0230, 0, 0, 0200, 0277, 0200, 0, 0200,
	0230, 0, 0, 0200, 0210, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0377, 0377, 0377, 0200, 0, 0, 0, 0,
	0377, 0377, 0377, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0100, 0200,
	0200, 0, 040, 0200, 0200, 0, 022, 0200, 0200, 0, 016, 0200, 0200, 0, 016, 0200,
	0200, 0, 036, 0200, 0200, 0, 0, 0200, 0377, 0377, 0377, 0200, 0, 0, 0, 0
};
unsigned char us_icon203[] =
{
	0377, 0377, 0377, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0201, 0, 0, 0200,
	0202, 0, 0, 0200, 0244, 0, 0, 0200, 0270, 0, 0, 0200, 0270, 0, 0, 0200,
	0274, 0, 0, 0200, 0200, 0, 0, 0200, 0377, 0377, 0377, 0200, 0, 0, 0, 0,
	0377, 0377, 0377, 0200, 0200, 0, 0, 0200, 0200, 0, 036, 0200, 0200, 0, 016, 0200,
	0200, 0, 016, 0200, 0200, 0, 022, 0200, 0200, 0, 040, 0200, 0200, 0, 0100, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0377, 0377, 0377, 0200, 0, 0, 0, 0
};
unsigned char us_icon204[] =
{
	0377, 0377, 0377, 0200, 0200, 0, 0, 0200, 0274, 0, 0, 0200, 0270, 0, 0, 0200,
	0270, 0, 0, 0200, 0244, 0, 0, 0200, 0202, 0, 0, 0200, 0201, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200, 0200, 0, 0, 0200,
	0200, 0, 0, 0200, 0200, 0, 0, 0200, 0377, 0377, 0377, 0200, 0, 0, 0, 0,
	0377, 0377, 0377, 0200, 0200, 0, 0, 0200, 0200, 010, 0, 0200, 0200, 034, 0, 0200,
	0200, 076, 0, 0200, 0210, 010, 010, 0200, 0230, 010, 014, 0200, 0277, 0200, 0376, 0200,
	0230, 010, 014, 0200, 0210, 010, 010, 0200, 0200, 076, 0, 0200, 0200, 034, 0, 0200,
	0200, 010, 0, 0200, 0200, 0, 0, 0200, 0377, 0377, 0377, 0200, 0, 0, 0, 0
};

static char *us_lastplacetextmessage = 0;

/*
 * Routine to free all memory associated with this module.
 */
void us_freediacommemory(void)
{
	if (us_lastplacetextmessage != 0) efree((char *)us_lastplacetextmessage);
}

/****************************** 3D DEPTH DIALOG ******************************/

/* 3D Depth */
DIALOGITEM us_3ddepthdialogitems[] =
{
 /*  1 */ {0, {508,52,532,132}, BUTTON, "OK"},
 /*  2 */ {0, {472,52,496,132}, BUTTON, "Cancel"},
 /*  3 */ {0, {32,8,404,168}, SCROLL, ""},
 /*  4 */ {0, {32,176,544,380}, USERDRAWN, ""},
 /*  5 */ {0, {416,8,432,88}, MESSAGE, "Thickness:"},
 /*  6 */ {0, {416,96,432,168}, EDITTEXT, "0"},
 /*  7 */ {0, {8,8,24,324}, MESSAGE, ""},
 /*  8 */ {0, {448,8,464,160}, CHECK, "Use Perspective"}
};
DIALOG us_3ddepthdialog = {{75,75,628,464}, "3D Options", 8, us_3ddepthdialogitems};

/*
 * special case for the "3D depth" dialog
 * List of layers        = 3 (scroll)
 * Layer heights diagram = 4 (user item)
 * Thickness             = 6 (edit text)
 * Technology name       = 7 (message)
 * Use perspective       = 8 (check)
 */
RECTAREA  us_3dheightrect;
INTBIG    us_3dcurlayer;
INTBIG   *us_3dheight;
INTBIG   *us_3dthickness;
INTBIG   *us_3dlayerindex;
INTBIG    us_3dlayercount;
INTBIG    us_3dlowheight, us_3dhighheight;

void us_redraw3ddepth(RECTAREA *bigr);
INTSML us_3ddepthstroke(INTBIG ox, INTBIG oy);

INTSML us_3ddepthdlog(char *paramstart[])
{
	INTBIG itemHit;
	INTSML x, y;
	char line[20];
	REGISTER WINDOWPART *w;
	REGISTER INTBIG i, j, funct, functp, thickness, height;
	REGISTER VARIABLE *heightvar, *thicknessvar;

	/* create default height information if none exists */
	heightvar = getvalkey((INTBIG)el_curtech, VTECHNOLOGY, VINTEGER|VISARRAY,
		us_techlayer3dheightkey);
	thicknessvar = getvalkey((INTBIG)el_curtech, VTECHNOLOGY, VINTEGER|VISARRAY,
		us_techlayer3dthicknesskey);
	if (heightvar == NOVARIABLE || thicknessvar == NOVARIABLE)
		us_3ddefaultlayerheight(el_curtech);
	heightvar = getvalkey((INTBIG)el_curtech, VTECHNOLOGY, VINTEGER|VISARRAY,
		us_techlayer3dheightkey);
	thicknessvar = getvalkey((INTBIG)el_curtech, VTECHNOLOGY, VINTEGER|VISARRAY,
		us_techlayer3dthicknesskey);
	if (heightvar == NOVARIABLE || thicknessvar == NOVARIABLE) return(0);

	/* cache the heights and thicknesses */
	us_3dheight = (INTBIG *)emalloc(el_curtech->layercount * SIZEOFINTBIG, el_tempcluster);
	us_3dthickness = (INTBIG *)emalloc(el_curtech->layercount * SIZEOFINTBIG, el_tempcluster);
	us_3dlayerindex = (INTBIG *)emalloc(el_curtech->layercount * SIZEOFINTBIG, el_tempcluster);
	for(i=0; i<el_curtech->layercount; i++)
	{
		us_3dheight[i] = ((INTBIG *)heightvar->addr)[i];
		us_3dthickness[i] = ((INTBIG *)thicknessvar->addr)[i];
	}

	/* determine which layers are useful */
	us_3dlayercount = 0;
	for(i=0; i<el_curtech->layercount; i++)
	{
		funct = layerfunction(el_curtech, (INTSML)i);
		if ((funct&LFPSEUDO) != 0) continue;
		us_3dlayerindex[us_3dlayercount++] = i;
	}

	/* display the 3D options dialog box */
	if (DiaInitDialog(&us_3ddepthdialog) != 0) return(0);
	(void)initinfstr();
	(void)addstringtoinfstr("Layer heights for technology ");
	(void)addstringtoinfstr(el_curtech->techname);
	DiaSetText(7, returninfstr());
	if ((us_useroptions&NO3DPERSPECTIVE) == 0) DiaSetControl(8, 1);

	/* setup list of layer names */
	DiaInitTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);
	for(i=0; i<us_3dlayercount; i++)
		DiaStuffLine(3, layername(el_curtech, (INTSML)us_3dlayerindex[i]));
	DiaSelectLine(3, 0);
	us_3dcurlayer = us_3dlayerindex[0];
	sprintf(line, "%ld", us_3dthickness[us_3dcurlayer]);
	DiaSetText(6, line);

	/* setup layer height area */
	DiaItemRect(4, &us_3dheightrect);
	DiaRedispRoutine(4, us_redraw3ddepth);
	for(i=0; i<el_curtech->layercount; i++)
	{
		height = us_3dheight[i] * 2;
		thickness = us_3dthickness[i] * 2;
		if (i == 0)
		{
			us_3dlowheight = height - thickness/2;
			us_3dhighheight = height + thickness/2;
		} else
		{
			if (height - thickness/2 < us_3dlowheight) us_3dlowheight = height - thickness/2;
			if (height + thickness/2 > us_3dhighheight) us_3dhighheight = height + thickness/2;
		}
	}
	us_3dlowheight -= 4;
	us_3dhighheight += 4;
	us_redraw3ddepth(&us_3dheightrect);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK) break;
		if (itemHit == 3)
		{
			j = DiaGetCurLine(3);
			us_3dcurlayer = us_3dlayerindex[j];
			sprintf(line, "%ld", us_3dthickness[us_3dcurlayer]);
			DiaSetText(6, line);
			us_redraw3ddepth(&us_3dheightrect);
			continue;
		}
		if (itemHit == 6)
		{
			j = DiaGetCurLine(3);
			us_3dcurlayer = us_3dlayerindex[j];
			us_3dthickness[us_3dcurlayer] = atoi(DiaGetText(6));
			us_redraw3ddepth(&us_3dheightrect);
			continue;
		}
		if (itemHit == 4)
		{
			DiaGetMouse(&x, &y);
			height = ((us_3dheightrect.bottom - y) * (us_3dhighheight - us_3dlowheight) +
				(us_3dheightrect.bottom - us_3dheightrect.top)/2) /
				(us_3dheightrect.bottom - us_3dheightrect.top) + us_3dlowheight;
			if (us_3dheight[us_3dcurlayer] * 2 != height)
			{
				for(i=0; i<us_3dlayercount; i++)
					if (us_3dheight[us_3dlayerindex[i]] == height / 2) break;
				if (i >= us_3dlayercount) continue;
				us_3dcurlayer = us_3dlayerindex[i];
				DiaSelectLine(3, i);
				sprintf(line, "%ld", us_3dthickness[us_3dcurlayer]);
				DiaSetText(6, line);
			}
			us_redraw3ddepth(&us_3dheightrect);
			trackcursor(0, us_nullup, us_nullvoid, us_3ddepthstroke,
				us_nullchar, us_nullvoid, TRACKNORMAL);
			continue;
		}
		if (itemHit == 8)
		{
			DiaSetControl(itemHit, 1 - DiaGetControl(itemHit));
			continue;
		}
	}

	if (itemHit != CANCEL)
	{
		/* copy regular layers to pseudo-layers */
		for(i=0; i<el_curtech->layercount; i++)
		{
			functp = layerfunction(el_curtech, (INTSML)i);
			if ((functp&LFPSEUDO) == 0) continue;

			/* pseudo layer found: look for real one */
			for(j=0; j<el_curtech->layercount; j++)
			{
				funct = layerfunction(el_curtech, (INTSML)j);
				if ((funct&LFPSEUDO) != 0) continue;
				if ((functp & ~LFPSEUDO) == funct)
				{
					us_3dheight[i] = us_3dheight[j];
					us_3dthickness[i] = us_3dthickness[j];
					break;
				}
			}
		}
		setvalkey((INTBIG)el_curtech, VTECHNOLOGY, us_techlayer3dheightkey, (INTBIG)us_3dheight,
			VINTEGER|VISARRAY|(el_curtech->layercount<<VLENGTHSH));
		setvalkey((INTBIG)el_curtech, VTECHNOLOGY, us_techlayer3dthicknesskey, (INTBIG)us_3dthickness,
			VINTEGER|VISARRAY|(el_curtech->layercount<<VLENGTHSH));
		j = us_useroptions;
		if (DiaGetControl(8) == 0) j |= NO3DPERSPECTIVE; else j &= ~NO3DPERSPECTIVE;
		(void)setvalkey((INTBIG)us_aid, VAID, us_optionflags, j, VINTEGER);
	}
	DiaDoneDialog();
	efree((char *)us_3dheight);
	efree((char *)us_3dthickness);
	if (itemHit != CANCEL)
	{
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if ((w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
			if (w->redisphandler != 0) (*w->redisphandler)(w);
		}
	}
	paramstart[0] = "";
	return(1);
}

void us_redraw3ddepth(RECTAREA *bigr)
{
	REGISTER INTBIG i, height, thickness, ypos1, ypos2, ypos, layer;
	INTBIG wid, hei;
	char *pt;

	DiaFrameRect(4, bigr);
	for(i=0; i<us_3dlayercount; i++)
	{
		layer = us_3dlayerindex[i];
		height = us_3dheight[layer] * 2;
		thickness = us_3dthickness[layer] * 2;
		ypos = bigr->bottom - (height - us_3dlowheight) *
			(bigr->bottom - bigr->top) / (us_3dhighheight - us_3dlowheight);
		if (layer == us_3dcurlayer)
		{
			DiaDrawLine(4, bigr->left, ypos, bigr->left+8, ypos, DLMODEON);
			DiaDrawLine(4, bigr->left+4, ypos-4, bigr->left+4, ypos+4, DLMODEON);
			DiaDrawLine(4, bigr->left, ypos-4, bigr->left+8, ypos+4, DLMODEON);
			DiaDrawLine(4, bigr->left+8, ypos-4, bigr->left, ypos+4, DLMODEON);
		}
		if (thickness == 0)
		{
			DiaDrawLine(4, bigr->left+10, ypos, bigr->left+70, ypos, DLMODEON);
		} else
		{
			ypos1 = bigr->bottom - (height - thickness/2 - us_3dlowheight) *
				(bigr->bottom - bigr->top) / (us_3dhighheight - us_3dlowheight) - 2;
			ypos2 = bigr->bottom - (height + thickness/2 - us_3dlowheight) *
				(bigr->bottom - bigr->top) / (us_3dhighheight - us_3dlowheight) + 2;
			DiaDrawLine(4, bigr->left+10, ypos1, bigr->left+40, ypos1, DLMODEON);
			DiaDrawLine(4, bigr->left+40, ypos1, bigr->left+50, ypos, DLMODEON);
			DiaDrawLine(4, bigr->left+40, ypos2, bigr->left+50, ypos, DLMODEON);
			DiaDrawLine(4, bigr->left+10, ypos2, bigr->left+40, ypos2, DLMODEON);
			DiaDrawLine(4, bigr->left+50, ypos, bigr->left+70, ypos, DLMODEON);
		}
		pt = layername(el_curtech, (INTSML)layer);
		DiaGetTextInfo(pt, &wid, &hei);
		DiaPutText(4, pt, bigr->left+70, ypos - hei/2);
	}
}

INTSML us_3ddepthstroke(INTBIG ox, INTBIG oy)
{
	INTSML x, y;
	REGISTER INTBIG height;

	DiaGetMouse(&x, &y);
	height = ((us_3dheightrect.bottom - y) * (us_3dhighheight - us_3dlowheight) +
		(us_3dheightrect.bottom - us_3dheightrect.top)/2) /
		(us_3dheightrect.bottom - us_3dheightrect.top) + us_3dlowheight;
	if (us_3dheight[us_3dcurlayer] == height / 2) return(0);
	us_3dheight[us_3dcurlayer] = height / 2;
	us_redraw3ddepth(&us_3dheightrect);
	return(0);
}

/****************************** ABOUT ELECTRIC DIALOG ******************************/

/*
 * the list of contributors to Electric (not including Steven M. Rubin)
 */
char *us_castofthousands[] =
{
	"Philip Attfield",
	"Ron Bolton",
	"Mark Brinsmead",
	"Stefano Concina",
	"Jonathan Gainsley",
	"Peter Gallant",
	"R. Brian Gardiner",
	"T. J. Goodman",
	"D. Guptill",
	"Robert Hon",
	"Sundaravarathan Iyengar",
	"Allan Jost",
	"Wallace Kroeker",
	"Andrew Kostiuk",
	"Oliver Laumann",
	"Glen Lawson",
	"David Lewis",
	"Erwin Liu",
	"John Mohammed",
	"Mark Moraes",
	"Sid Penstone",
	"J. P. Polonovski",
	"Kevin Ryan",
	"Nora Ryan",
	"Brent Serbin",
	"Boaz Shmueli",
	"Ken Stevens",
	"Lyndon Swab",
	"Brian W. Thomson",
	"Burnie West",
	"Telle Whitney",
	"Rob Winstanley",
	"Russell Wright",
	"David J. Yurach",
	0
};

char *us_gnucopying[] =
{
	"TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION",
	"",
	"0. This License applies to any program or other work which contains a notice placed by",
	"the copyright holder saying it may be distributed under the terms of this General",
	"Public License. The 'Program', below, refers to any such program or work, and a",
	"'work based on the Program' means either the Program or any derivative work under",
	"copyright law: that is to say, a work containing the Program or a portion of it,",
	"either verbatim or with modifications and/or translated into another language.",
	"(Hereinafter, translation is included without limitation in the term 'modification'.)",
	"Each licensee is addressed as 'you'.",
	"",
	"Activities other than copying, distribution and modification are not covered by this",
	"License; they are outside its scope. The act of running the Program is not restricted,",
	"and the output from the Program is covered only if its contents constitute a work based",
	"on the Program (independent of having been made by running the Program). Whether that",
	"is true depends on what the Program does.",
	"",
	"1. You may copy and distribute verbatim copies of the Program's source code as you",
	"receive it, in any medium, provided that you conspicuously and appropriately publish",
	"on each copy an appropriate copyright notice and disclaimer of warranty; keep intact",
	"all the notices that refer to this License and to the absence of any warranty; and",
	"give any other recipients of the Program a copy of this License along with the Program.",
	"",
	"You may charge a fee for the physical act of transferring a copy, and you may at your",
	"option offer warranty protection in exchange for a fee.",
	"",
	"2. You may modify your copy or copies of the Program or any portion of it, thus forming",
	"a work based on the Program, and copy and distribute such modifications or work under",
	"the terms of Section 1 above, provided that you also meet all of these conditions:",
	"",
	"*	a) You must cause the modified files to carry prominent notices stating that you",
	"	changed the files and the date of any change.",
	"",
	"*	b) You must cause any work that you distribute or publish, that in whole or",
	"	in part contains or is derived from the Program or any part thereof, to be licensed",
	"	as a whole at no charge to all third parties under the terms of this License.",
	"",
	"*	c) If the modified program normally reads commands interactively when run, you",
	"	must cause it, when started running for such interactive use in the most ordinary",
	"	way, to print or display an announcement including an appropriate copyright notice",
	"	and a notice that there is no warranty (or else, saying that you provide a warranty)",
	"	and that users may redistribute the program under these conditions, and telling the",
	"	user how to view a copy of this License. (Exception: if the Program itself is",
	"	interactive but does not normally print such an announcement, your work based on the",
	"	Program is not required to print an announcement.)",
	"",
	"These requirements apply to the modified work as a whole. If identifiable sections",
	"of that work are not derived from the Program, and can be reasonably considered independent",
	"and separate works in themselves, then this License, and its terms, do not apply to those",
	"sections when you distribute them as separate works. But when you distribute the same",
	"sections as part of a whole which is a work based on the Program, the distribution of",
	"the whole must be on the terms of this License, whose permissions for other licensees",
	"extend to the entire whole, and thus to each and every part regardless of who wrote it.",
	"",
	"Thus, it is not the intent of this section to claim rights or contest your rights to",
	"work written entirely by you; rather, the intent is to exercise the right to control",
	"the distribution of derivative or collective works based on the Program.",
	"",
	"In addition, mere aggregation of another work not based on the Program with the Program",
	"(or with a work based on the Program) on a volume of a storage or distribution medium",
	"does not bring the other work under the scope of this License.",
	"",
	"3. You may copy and distribute the Program (or a work based on it, under Section 2)",
	"in object code or executable form under the terms of Sections 1 and 2 above provided",
	"that you also do one of the following:",
	"",
	"*	a) Accompany it with the complete corresponding machine-readable source code,",
	"which must be distributed under the terms of Sections 1 and 2 above on a medium",
	"customarily used for software interchange; or,",
	"",
	"*	b) Accompany it with a written offer, valid for at least three years, to give",
	"any third party, for a charge no more than your cost of physically performing source",
	"distribution, a complete machine-readable copy of the corresponding source code,",
	"to be distributed under the terms of Sections 1 and 2 above on a medium customarily",
	"used for software interchange; or,",
	"",
	"*	c) Accompany it with the information you received as to the offer to distribute",
	"corresponding source code. (This alternative is allowed only for noncommercial",
	"distribution and only if you received the program in object code or executable",
	"form with such an offer, in accord with Subsection b above.)",
	"",
	"The source code for a work means the preferred form of the work for making",
	"modifications to it. For an executable work, complete source code means all",
	"the source code for all modules it contains, plus any associated interface",
	"definition files, plus the scripts used to control compilation and installation",
	"of the executable. However, as a special exception, the source code distributed",
	"need not include anything that is normally distributed (in either source or binary",
	"form) with the major components (compiler, kernel, and so on) of the operating",
	"system on which the executable runs, unless that component itself accompanies the executable.",
	"",
	"If distribution of executable or object code is made by offering access to copy",
	"from a designated place, then offering equivalent access to copy the source code",
	"from the same place counts as distribution of the source code, even though third",
	"parties are not compelled to copy the source along with the object code.",
	"",
	"4. You may not copy, modify, sublicense, or distribute the Program except as",
	"expressly provided under this License. Any attempt otherwise to copy, modify,",
	"sublicense or distribute the Program is void, and will automatically terminate your",
	"rights under this License. However, parties who have received copies, or rights,",
	"from you under this License will not have their licenses terminated so long as",
	"such parties remain in full compliance.",
	"",
	"5. You are not required to accept this License, since you have not signed it.",
	"However, nothing else grants you permission to modify or distribute the Program or",
	"its derivative works. These actions are prohibited by law if you do not accept this",
	"License. Therefore, by modifying or distributing the Program (or any work based on",
	"the Program), you indicate your acceptance of this License to do so, and all its",
	"terms and conditions for copying, distributing or modifying the Program or works based on it.",
	"",
	"6. Each time you redistribute the Program (or any work based on the Program),",
	"the recipient automatically receives a license from the original licensor to copy,",
	"distribute or modify the Program subject to these terms and conditions. You may not",
	"impose any further restrictions on the recipients' exercise of the rights granted",
	"herein. You are not responsible for enforcing compliance by third parties to this License.",
	"",
	"7. If, as a consequence of a court judgment or allegation of patent infringement",
	"or for any other reason (not limited to patent issues), conditions are imposed",
	"on you (whether by court order, agreement or otherwise) that contradict the conditions",
	"of this License, they do not excuse you from the conditions of this License. If you",
	"cannot distribute so as to satisfy simultaneously your obligations under this",
	"License and any other pertinent obligations, then as a consequence you may not",
	"distribute the Program at all. For example, if a patent license would not permit",
	"royalty-free redistribution of the Program by all those who receive copies directly",
	"or indirectly through you, then the only way you could satisfy both it and this",
	"License would be to refrain entirely from distribution of the Program.",
	"",
	"If any portion of this section is held invalid or unenforceable under any",
	"particular circumstance, the balance of the section is intended to apply and",
	"the section as a whole is intended to apply in other circumstances.",
	"",
	"It is not the purpose of this section to induce you to infringe any patents",
	"or other property right claims or to contest validity of any such claims; this",
	"section has the sole purpose of protecting the integrity of the free software",
	"distribution system, which is implemented by public license practices. Many",
	"people have made generous contributions to the wide range of software distributed",
	"through that system in reliance on consistent application of that system; it is",
	"up to the author/donor to decide if he or she is willing to distribute software",
	"through any other system and a licensee cannot impose that choice.",
	"",
	"This section is intended to make thoroughly clear what is believed to be a",
	"consequence of the rest of this License.",
	"",
	"8. If the distribution and/or use of the Program is restricted in certain",
	"countries either by patents or by copyrighted interfaces, the original copyright",
	"holder who places the Program under this License may add an explicit geographical",
	"distribution limitation excluding those countries, so that distribution is permitted",
	"only in or among countries not thus excluded. In such case, this License incorporates",
	"the limitation as if written in the body of this License.",
	"",
	"9. The Free Software Foundation may publish revised and/or new versions of the",
	"General Public License from time to time. Such new versions will be similar in",
	"spirit to the present version, but may differ in detail to address new problems",
	"or concerns.",
	"",
	"Each version is given a distinguishing version number. If the Program specifies",
	"a version number of this License which applies to it and 'any later version',",
	"you have the option of following the terms and conditions either of that version",
	"or of any later version published by the Free Software Foundation. If the Program",
	"does not specify a version number of this License, you may choose any version ever",
	"published by the Free Software Foundation.",
	"",
	"10. If you wish to incorporate parts of the Program into other free programs",
	"whose distribution conditions are different, write to the author to ask for",
	"permission. For software which is copyrighted by the Free Software Foundation,",
	"write to the Free Software Foundation; we sometimes make exceptions for this.",
	"Our decision will be guided by the two goals of preserving the free status of",
	"all derivatives of our free software and of promoting the sharing and reuse of",
	"software generally.",
	0
};

char *us_gnuwarranty[] =
{
	"NO WARRANTY",
	"",
	"11. Because the program is licensed free of charge, there is no warranty for the",
	"program, to the extent permitted by applicable law. Except when otherwise stated",
	"in writing the copyright holders and/or other parties provide the program 'as is'",
	"without warranty of any kind, either expressed or implied, including, but not",
	"limited to, the implied warranties of merchantability and fitness for a particular",
	"purpose. The entire risk as to the quality and performance of the program is with you.",
	"Should the program prove defective, you assume the cost of all necessary servicing,",
	"repair or correction. ",
	"",
	"12. In no event unless required by applicable law or agreed to in writing will any",
	"copyright holder, or any other party who may modify and/or redistribute the program",
	"as permitted above, be liable to you for damages, including any general, special,",
	"incidental or consequential damages arising out of the use or inability to use the",
	"program (including but not limited to loss of data or data being rendered inaccurate",
	"or losses sustained by you or third parties or a failure of the program to operate",
	"with any other programs), even if such holder or other party has been advised of",
	"the possibility of such damages. ",
	0
};

/* icons for the "About Electric" dialog */
unsigned char us_icon130[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 03, 0377, 0377, 0, 07, 0377, 0377, 0, 014, 0, 0, 0, 030, 0, 0,
	0, 060, 0, 0, 0, 0140, 0, 0, 0, 0300, 0, 0, 01, 0200, 0, 0,
	03, 0, 0, 0, 03, 0, 0, 0, 06, 0, 0, 0, 06, 0, 0, 0,
	014, 03, 0374, 0, 014, 03, 0374, 0, 030, 03, 0374, 0, 030, 03, 0374, 0,
	060, 03, 0374, 0, 060, 03, 0374, 0, 0140, 03, 0374, 0, 0140, 03, 0374, 0,
	0300, 03, 0374, 0, 0300, 03, 0374, 0, 0300, 03, 0374, 0, 0300, 03, 0374, 0,
	0300, 03, 0374, 0, 0300, 03, 0374, 0, 0300, 03, 0374, 0, 0300, 03, 0374, 0
};
unsigned char us_icon129[] =
{
	0300, 03, 0374, 0, 0300, 03, 0374, 0, 0300, 03, 0374, 0, 0300, 03, 0374, 0,
	0300, 03, 0374, 0, 0300, 03, 0374, 0, 0300, 0, 0, 0, 0300, 0, 0, 0,
	0140, 0, 0, 03, 0140, 0, 0, 07, 060, 0, 0, 017, 060, 0, 0, 037,
	030, 0, 0, 077, 030, 0, 0, 077, 014, 0, 0, 077, 014, 0, 0, 077,
	06, 0, 0, 077, 06, 0, 0, 077, 03, 0, 0, 077, 03, 0, 0, 077,
	01, 0200, 0, 0, 0, 0300, 0, 0, 0, 0140, 0, 0, 0, 060, 0, 0,
	0, 030, 0, 0, 0, 014, 0, 0, 0, 07, 0377, 0377, 0, 03, 0377, 0377,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon131[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0377, 0377, 0300, 0, 0377, 0377, 0340, 0, 0, 0, 060, 0, 0, 0, 030, 0,
	0, 0, 014, 0, 0, 0, 06, 0, 0, 0, 03, 0, 0, 0, 01, 0200,
	0, 0, 0, 0300, 0, 0, 0, 0300, 0, 0, 0, 0140, 0, 0, 0, 0140,
	0, 077, 0300, 060, 0, 077, 0300, 060, 0, 077, 0300, 030, 0, 077, 0300, 030,
	0, 077, 0300, 014, 0, 077, 0300, 014, 0, 077, 0300, 06, 0, 077, 0300, 06,
	0, 077, 0300, 03, 0, 077, 0300, 03, 0, 077, 0300, 03, 0, 077, 0300, 03,
	0, 077, 0300, 03, 0, 077, 0300, 03, 0, 077, 0300, 03, 0, 077, 0300, 03
};
unsigned char us_icon132[] =
{
	0, 077, 0300, 03, 0, 077, 0300, 03, 0, 077, 0300, 03, 0, 077, 0300, 03,
	0, 0, 0, 03, 0, 0, 0, 03, 0, 0, 0, 03, 0, 0, 0, 03,
	0300, 0, 0, 06, 0340, 0, 0, 06, 0360, 0, 0, 014, 0370, 0, 0, 014,
	0374, 0, 0, 030, 0374, 0, 0, 030, 0374, 0, 0, 060, 0374, 0, 0, 060,
	0374, 0, 0, 0140, 0374, 0, 0, 0140, 0374, 0, 0, 0300, 0374, 0, 0, 0300,
	0, 0, 01, 0200, 0, 0, 03, 0, 0, 0, 06, 0, 0, 0, 014, 0,
	0, 0, 030, 0, 0, 0, 060, 0, 0377, 0377, 0340, 0, 0377, 0377, 0300, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

/* About Electric */
DIALOGITEM us_aboutdialogitems[] =
{
 /*  1 */ {0, {216,24,240,104}, BUTTON, "Continue"},
 /*  2 */ {0, {88,8,104,448}, MESSAGE, "Protected by license, redistribution prohibited"},
 /*  3 */ {0, {64,8,80,373}, MESSAGE, "Copyright (c) 2000 Static Free Software"},
 /*  4 */ {0, {120,8,136,209}, MESSAGE, "Written by Steven M. Rubin"},
 /*  5 */ {0, {8,8,24,295}, MESSAGE, "The Electric(tm) Design System"},
 /*  6 */ {0, {32,8,48,246}, MESSAGE, "Version XXXX"},
 /*  7 */ {0, {16,376,48,408}, ICON|INACTIVE, (char *)us_icon130},
 /*  8 */ {0, {48,376,80,408}, ICON|INACTIVE, (char *)us_icon129},
 /*  9 */ {0, {16,408,48,440}, ICON|INACTIVE, (char *)us_icon131},
 /* 10 */ {0, {48,408,80,440}, ICON|INACTIVE, (char *)us_icon132},
 /* 11 */ {0, {144,136,264,445}, SCROLL, ""},
 /* 12 */ {0, {120,216,138,404}, BUTTON, "And a Cast of Thousands"}
};
DIALOG us_aboutdialog = {{50,75,323,532}, 0, 12, us_aboutdialogitems};

/* GNU About Electric */
DIALOGITEM us_aboutgnudialogitems[] =
{
 /*  1 */ {0, {28,320,52,400}, BUTTON, "Continue"},
 /*  2 */ {0, {300,12,316,356}, MESSAGE, "Electric comes with ABSOLUTELY NO WARRANTY"},
 /*  3 */ {0, {64,8,80,373}, MESSAGE, "Copyright (c) 2000 Static Free Software"},
 /*  4 */ {0, {96,8,112,221}, MESSAGE, "Written by Steven M. Rubin"},
 /*  5 */ {0, {8,8,24,295}, MESSAGE, "The Electric(tm) Design System"},
 /*  6 */ {0, {32,8,48,246}, MESSAGE, "Version XXXX"},
 /*  7 */ {0, {16,420,48,452}, ICON|INACTIVE, (char *)us_icon130},
 /*  8 */ {0, {48,420,80,452}, ICON|INACTIVE, (char *)us_icon129},
 /*  9 */ {0, {16,452,48,484}, ICON|INACTIVE, (char *)us_icon131},
 /* 10 */ {0, {48,452,80,484}, ICON|INACTIVE, (char *)us_icon132},
 /* 11 */ {0, {120,8,293,487}, SCROLL, ""},
 /* 12 */ {0, {96,232,114,420}, BUTTON, "And a Cast of Thousands"},
 /* 13 */ {0, {324,12,340,330}, MESSAGE, "This is free software, and you are welcome to"},
 /* 14 */ {0, {300,358,318,487}, BUTTON, "Warranty details"},
 /* 15 */ {0, {336,358,354,487}, BUTTON, "Copying details"},
 /* 16 */ {0, {344,12,360,309}, MESSAGE, "redistribute it under certain conditions"}
};
DIALOG us_aboutgnudialog = {{50,75,419,573}, 0, 16, us_aboutgnudialogitems};

/*
 * special case for the "About Electric" dialog
 * version number        = 6 (message)
 * authors               = 12 (button)
 * restrictions (nonGNU) = 13 (button)
 * warranty (GNU)        = 14 (button)
 * copying (GNU)         = 15 (button)
 */
void us_aboutdlog(char *licline)
{
	char line[256], date[30];
	INTBIG itemHit, i;

#if EGNU
	/* show the GNU "about" dialog */
	if (DiaInitDialog(&us_aboutgnudialog) != 0) return;
	DiaInitTextDialog(11, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSMALLFONT|SCHORIZBAR);
#else
	/* show the commercial "about" dialog */
	if (DiaInitDialog(&us_aboutdialog) != 0) return;
	DiaInitTextDialog(11, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1, 0);
#endif

	/* show the version and copyright information */
	(void)strcpy(line, "Version ");
	(void)strcat(line, el_version);
	(void)strcat(line, ", ");
	(void)strcat(line, __DATE__);
	DiaSetText(6, line);
	strcpy(date, timetostring(getcurrenttime()));
	date[24] = 0;
	(void)sprintf(line, "Copyright (c) %s Static Free Software", &date[20]);
	DiaSetText(3, line);

	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK) break;
		if (itemHit == 12)
		{
			DiaLoadTextDialog(11, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
			for(i=0; us_castofthousands[i] != 0; i++) DiaStuffLine(11, us_castofthousands[i]);
			DiaSelectLine(11, -1);
			continue;
		}
		if (itemHit == 14)
		{
			DiaLoadTextDialog(11, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
			for(i=0; us_gnuwarranty[i] != 0; i++) DiaStuffLine(11, us_gnuwarranty[i]);
			DiaSelectLine(11, -1);
			continue;
		}
		if (itemHit == 15)
		{
			DiaLoadTextDialog(11, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
			for(i=0; us_gnucopying[i] != 0; i++) DiaStuffLine(11, us_gnucopying[i]);
			DiaSelectLine(11, -1);
			continue;
		}
	}
	DiaDoneDialog();
}

/****************************** ALIGNMENT OPTIONS DIALOG ******************************/

/* Alignment Options */
DIALOGITEM us_alignmentdialogitems[] =
{
 /*  1 */ {0, {68,340,92,404}, BUTTON, "OK"},
 /*  2 */ {0, {68,32,92,96}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,8,24,205}, MESSAGE, "Alignment of cursor to grid:"},
 /*  4 */ {0, {40,8,56,205}, MESSAGE, "Alignment of edges to grid:"},
 /*  5 */ {0, {8,208,24,280}, EDITTEXT, ""},
 /*  6 */ {0, {40,208,56,280}, EDITTEXT, ""},
 /*  7 */ {0, {16,284,32,426}, MESSAGE, "Values of zero will"},
 /*  8 */ {0, {32,284,48,428}, MESSAGE, "cause no alignment."}
};
DIALOG us_alignmentdialog = {{50,75,154,512}, "Alignment Options", 8, us_alignmentdialogitems};

/*
 * special case for the "alignment options" dialog
 * Grid alignment     = 5 (edit text)
 * Edge alignment     = 6 (edit text)
 */
INTSML us_alignmentdlog(char *paramstart[])
{
	INTBIG itemHit;
	INTBIG retval1;

	if (us_needwindow()) return(0);

	/* display the alignment settings dialog box */
	if (DiaInitDialog(&us_alignmentdialog) != 0) return(0);
	DiaSetText(5, latoa(us_alignment));
	DiaSetText(6, latoa(us_edgealignment));

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
	}

	if (itemHit != CANCEL)
	{
		/* see if alignment changed */
		retval1 = atola(DiaGetText(5));
		if (retval1 != us_alignment)
			(void)setvalkey((INTBIG)us_aid, VAID, us_alignment_obj, retval1, VINTEGER);
		retval1 = atola(DiaGetText(6));
		if (retval1 != us_edgealignment)
			(void)setvalkey((INTBIG)us_aid, VAID, us_alignment_edge, retval1, VINTEGER);
	}
	DiaDoneDialog();
	paramstart[0] = "x";
	return(1);
}

/****************************** ARC CREATION OPTIONS DIALOG ******************************/

/* New arc options */
DIALOGITEM us_defarcdialogitems[] =
{
 /*  1 */ {0, {192,312,216,384}, BUTTON, "OK"},
 /*  2 */ {0, {192,72,216,144}, BUTTON, "Cancel"},
 /*  3 */ {0, {15,150,31,430}, POPUP, ""},
 /*  4 */ {0, {15,15,31,145}, RADIO, "Defaults for arc:"},
 /*  5 */ {0, {40,15,56,175}, RADIO, "Defaults for all arcs"},
 /*  6 */ {0, {160,368,176,423}, BUTTON, "Set pin"},
 /*  7 */ {0, {80,16,96,73}, CHECK, "Rigid"},
 /*  8 */ {0, {80,112,96,214}, CHECK, "Fixed-angle"},
 /*  9 */ {0, {80,224,96,299}, CHECK, "Slidable"},
 /* 10 */ {0, {104,16,120,95}, CHECK, "Negated"},
 /* 11 */ {0, {104,112,120,205}, CHECK, "Directional"},
 /* 12 */ {0, {104,224,120,344}, CHECK, "Ends extended"},
 /* 13 */ {0, {40,260,60,428}, BUTTON, "Reset to initial defaults"},
 /* 14 */ {0, {136,72,152,160}, EDITTEXT, ""},
 /* 15 */ {0, {136,16,152,64}, MESSAGE, "Width:"},
 /* 16 */ {0, {136,184,152,232}, MESSAGE, "Angle:"},
 /* 17 */ {0, {136,240,152,304}, EDITTEXT, ""},
 /* 18 */ {0, {160,16,176,47}, MESSAGE, "Pin:"},
 /* 19 */ {0, {160,48,176,358}, MESSAGE, ""}
};
DIALOG us_defarcdialog = {{50,75,278,514}, "New Arc Options", 19, us_defarcdialogitems};

static ARCPROTO *us_thisap;
static NODEPROTO *us_posprims;

void us_defarcload(ARCPROTO*, NODEPROTO**, INTBIG);
INTSML us_topofpins(char**);
char *us_nextpins(void);

/*
 * special case for the "defarc" dialog
 * defaults for specific type = 4 (radio button)
 * defaults for all types     = 5 (radio button)
 * specific arc               = 3 (message)
 * reset to initial state     = 13 (button)
 * rigid                      = 7 (check)
 * fixed-angle                = 8 (check)
 * slidable                   = 9 (check)
 * negated                    = 10 (check)
 * directional                = 11 (check)
 * ends extended              = 12 (check)
 * width                      = 14 (edit text)
 * width title                = 15 (message)
 * angle title                = 16 (message)
 * angle                      = 17 (edit text)
 * default pin title          = 18 (message)
 * default pin                = 19 (message)
 * set pin                    = 6 (button)
 */
INTSML us_defarcdlog(char *paramstart[])
{
	INTBIG itemHit;
	REGISTER INTBIG allstyle, i, j, origallstyle, bits;
	REGISTER VARIABLE *var;
	char **arcnames;
	REGISTER ARCPROTO *ap, **arcs;
	REGISTER NODEPROTO **pins, *np;
	extern DIALOG us_listdialog;

	/* display the default arc dialog box */
	if (DiaInitDialog(&us_defarcdialog) != 0) return(0);

	/* remember all state */
	var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_arcstyle);
	if (var != NOVARIABLE) allstyle = var->addr; else
		allstyle = WANTFIXANG;
	origallstyle = allstyle;
	for(i=0, ap = el_curtech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto) i++;
	pins = (NODEPROTO **)emalloc(i * (sizeof (NODEPROTO *)), el_tempcluster);
	arcs = (ARCPROTO **)emalloc(i * (sizeof (ARCPROTO *)), el_tempcluster);
	arcnames = (char **)emalloc(i * (sizeof (char *)), el_tempcluster);
	if (pins == 0 || arcs == 0 || arcnames == 0) return(0);
	for(i=0, ap = el_curtech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto, i++)
	{
		var = getvalkey((INTBIG)ap, VARCPROTO, VINTEGER, us_arcstyle);
		if (var != NOVARIABLE) ap->temp1 = var->addr; else
			ap->temp1 = ap->userbits;
		ap->temp2 = defaultarcwidth(ap);
		pins[i] = getpinproto(ap);
		arcs[i] = ap;
		arcnames[i] = ap->protoname;
	}

	/* initially load for first arc in technology */
	us_thisap = us_curarcproto;
	us_defarcload(us_thisap, pins, allstyle);
	DiaSetControl(4, 1);
	DiaSetPopup(3, i, arcnames);
	efree((char *)arcnames);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == 7 || itemHit == 8 || itemHit == 9 ||
			itemHit == 10 || itemHit == 11 || itemHit == 12)
		{
			j = DiaGetControl(itemHit);
			DiaSetControl(itemHit, 1-j);
			if (DiaGetControl(4) == 0) i = allstyle; else
				i = us_thisap->temp1;
			switch (itemHit)
			{
				case 7:
					if (j == 0) i |= WANTFIX; else i &= ~WANTFIX;
					break;
				case 8:
					if (j == 0) i |= WANTFIXANG; else i &= ~WANTFIXANG;
					break;
				case 9:
					if (j != 0) i |= WANTCANTSLIDE; else i &= ~WANTCANTSLIDE;
					break;
				case 10:
					if (j == 0) i |= WANTNEGATED; else i &= ~WANTNEGATED;
					break;
				case 11:
					if (j == 0) i |= WANTDIRECTIONAL; else i &= ~WANTDIRECTIONAL;
					break;
				case 12:
					if (j != 0) i |= WANTNOEXTEND; else i &= ~WANTNOEXTEND;
					break;
			}
			if (DiaGetControl(4) == 0) allstyle = i; else
				us_thisap->temp1 = i;
			continue;
		}
		if (itemHit == 4)
		{
			us_defarcload(us_thisap, pins, allstyle);
			DiaSetControl(4, 1);
			DiaSetControl(5, 0);
			continue;
		}
		if (itemHit == 5)
		{
			us_defarcload(NOARCPROTO, pins, allstyle);
			DiaSetControl(4, 0);
			DiaSetControl(5, 1);
			continue;
		}
		if (itemHit == 13)
		{
			allstyle = WANTFIXANG;
			DiaSetControl(7, 0);
			DiaSetControl(8, 1);
			DiaSetControl(9, 1);
			DiaSetControl(10, 0);
			DiaSetControl(11, 0);
			DiaSetControl(12, 1);
			continue;
		}
		if (itemHit == 3)
		{
			i = DiaGetPopupEntry(3);
			us_thisap = arcs[i];
			us_defarcload(us_thisap, pins, allstyle);
			continue;
		}
		if (itemHit == 14)
		{
			if (DiaGetControl(4) == 0) continue;
			us_thisap->temp2 = atola(DiaGetText(14));
			continue;
		}
		if (itemHit == 17)
		{
			if (DiaGetControl(4) == 0) continue;
			us_thisap->temp1 = (us_thisap->temp1 & ~AANGLEINC) | ((atoi(DiaGetText(17))%360) << AANGLEINCSH);
			continue;
		}
		if (itemHit == 6)
		{
			if (DiaGetControl(4) == 0) continue;

			if (DiaInitDialog(&us_listdialog) != 0) continue;
			DiaInitTextDialog(3, us_topofpins, us_nextpins, DiaNullDlogDone,
				0, SCSELMOUSE|SCSELKEY|SCDOUBLEQUIT);
			DiaSetText(4, "Select a node to use as a pin");

			for(;;)
			{
				itemHit = DiaNextHit();
				if (itemHit == OK || itemHit == CANCEL) break;
			}
			np = getnodeproto(DiaGetScrollLine(3, DiaGetCurLine(3)));
			DiaDoneDialog();
			if (itemHit == CANCEL) continue;

			for(i=0, ap = el_curtech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto, i++)
				if (us_thisap == ap) break;
			if (ap != NOARCPROTO) pins[i] = np;
			DiaSetText(19, describenodeproto(np));
			continue;
		}
	}

	if (itemHit != CANCEL)
	{
		for(i=0, ap = el_curtech->firstarcproto; ap != NOARCPROTO;
			ap = ap->nextarcproto, i++)
		{
			var = getvalkey((INTBIG)ap, VARCPROTO, VINTEGER, us_arcstyle);
			if (var != NOVARIABLE) bits = var->addr; else
				bits = ap->userbits;
			if (ap->temp1 != bits)
			{
				if ((ap->temp1 & AANGLEINC) != (bits & AANGLEINC))
					(void)setval((INTBIG)ap, VARCPROTO, "userbits", ap->temp1, VINTEGER);
				if ((ap->temp1 & ~AANGLEINC) != (bits & ~AANGLEINC))
					(void)setvalkey((INTBIG)ap, VARCPROTO, us_arcstyle, ap->temp1, VINTEGER);
			}
			if (ap->temp2 != defaultarcwidth(ap))
				(void)setvalkey((INTBIG)ap, VARCPROTO, el_arc_default_width,
					arcprotowidthoffset(ap) + ap->temp2, VINTEGER);
			np = getpinproto(ap);
			if (np != pins[i])
				(void)setval((INTBIG)ap, VARCPROTO, "ARC_Default_Pin", (INTBIG)pins[i], VNODEPROTO);
		}
		if (allstyle != origallstyle)
			(void)setvalkey((INTBIG)us_aid, VAID, us_arcstyle, allstyle, VINTEGER);
	}
	DiaDoneDialog();
	efree((char *)pins);
	efree((char *)arcs);
	paramstart[0] = "x";
	return(1);
}

INTSML us_topofpins(char **c)
{
	us_posprims = el_curtech->firstnodeproto;
	return(1);
}

char *us_nextpins(void)
{
	REGISTER char *nextname;
	REGISTER INTBIG i;
	REGISTER PORTPROTO *pp;

	for( ; us_posprims != NONODEPROTO; us_posprims = us_posprims->nextnodeproto)
	{
		/* test this pin for validity */
		pp = us_posprims->firstportproto;
		for(i=0; pp->connects[i] != NOARCPROTO; i++)
			if (pp->connects[i] == us_thisap) break;
		if (pp->connects[i] == NOARCPROTO) continue;
		nextname = us_posprims->primname;
		us_posprims = us_posprims->nextnodeproto;
		return(nextname);
	}
	return(0);
}

/*
 * Helper routine for arc options
 */
void us_defarcload(ARCPROTO *ap, NODEPROTO **pins, INTBIG allstyle)
{
	REGISTER NODEPROTO *np;
	REGISTER ARCPROTO *oap;
	REGISTER INTBIG i;
	REGISTER INTBIG style;
	char line[20];

	if (ap == NOARCPROTO)
	{
		style = allstyle;
		DiaDimItem(3);
		DiaUnDimItem(13);
		DiaDimItem(15);
		DiaSetText(14, "");
		DiaNoEditControl(14);
		DiaDimItem(16);
		DiaSetText(17, "");
		DiaNoEditControl(17);
		DiaDimItem(18);
		DiaSetText(19, "");
		DiaDimItem(6);
	} else
	{
		style = ap->temp1;
		DiaUnDimItem(3);
		DiaDimItem(13);
		DiaUnDimItem(15);
		DiaSetText(-14, latoa(defaultarcwidth(ap) - arcprotowidthoffset(ap)));
		DiaEditControl(14);
		DiaUnDimItem(16);
		(void)sprintf(line, "%ld", (ap->userbits&AANGLEINC) >> AANGLEINCSH);
		DiaSetText(17, line);
		DiaEditControl(17);
		DiaUnDimItem(18);
		np = NONODEPROTO;
		for(i=0, oap = el_curtech->firstarcproto; oap != NOARCPROTO; oap = oap->nextarcproto, i++)
			if (ap == oap) np = pins[i];
		DiaSetText(19, describenodeproto(np));
		DiaUnDimItem(6);
	}
	DiaSetControl(7, (style&WANTFIX) != 0 ? 1 : 0);
	DiaSetControl(8, (style&WANTFIXANG) != 0 ? 1 : 0);
	DiaSetControl(9, (style&WANTCANTSLIDE) == 0 ? 1 : 0);
	DiaSetControl(10, (style&WANTNEGATED) != 0 ? 1 : 0);
	DiaSetControl(11, (style&WANTDIRECTIONAL) != 0 ? 1 : 0);
	DiaSetControl(12, (style&WANTNOEXTEND) == 0 ? 1 : 0);
}

/****************************** ARC SIZE DIALOG ******************************/

/* Arc Size */
DIALOGITEM us_arcsizedialogitems[] =
{
 /*  1 */ {0, {36,96,60,176}, BUTTON, "OK"},
 /*  2 */ {0, {36,4,60,84}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,4,24,84}, MESSAGE|INACTIVE, "Width"},
 /*  4 */ {0, {8,92,24,172}, EDITTEXT, ""}
};
DIALOG us_arcsizedialog = {{75,75,144,260}, "Set Arc Size", 4, us_arcsizedialogitems};

/*
 * special case for the "arc size" dialog
 * width = 4 (edit text)
 */
INTSML us_arcsizedlog(char *paramstart[])
{
	INTBIG itemHit;
	INTSML ret;
	static char w[20];

	/* display the node size dialog box */
	if (DiaInitDialog(&us_arcsizedialog) != 0) return(0);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
	}

	ret = 0;
	if (itemHit != CANCEL)
	{
		strcpy(w, DiaGetText(4));
		paramstart[0] = w;
		ret = 1;
	}
	DiaDoneDialog();
	return(ret);
}

/****************************** ARRAY DIALOG ******************************/

/* Array */
DIALOGITEM us_arraydialogitems[] =
{
 /*  1 */ {0, {180,360,204,424}, BUTTON, "OK"},
 /*  2 */ {0, {180,264,204,328}, BUTTON, "Cancel"},
 /*  3 */ {0, {12,160,28,205}, EDITTEXT, "1"},
 /*  4 */ {0, {52,160,68,205}, EDITTEXT, "1"},
 /*  5 */ {0, {104,160,120,235}, EDITTEXT, "0"},
 /*  6 */ {0, {132,160,148,235}, EDITTEXT, "0"},
 /*  7 */ {0, {192,4,208,182}, CHECK, "Generate array indices"},
 /*  8 */ {0, {4,216,20,393}, CHECK, "Flip alternate columns"},
 /*  9 */ {0, {44,216,60,376}, CHECK, "Flip alternate rows"},
 /* 10 */ {0, {12,36,28,151}, MESSAGE, "X repeat factor:"},
 /* 11 */ {0, {52,36,68,151}, MESSAGE, "Y repeat factor:"},
 /* 12 */ {0, {104,4,120,154}, MESSAGE, "X edge overlap:"},
 /* 13 */ {0, {132,4,148,154}, MESSAGE, "Y centerline distance:"},
 /* 14 */ {0, {88,244,104,425}, RADIO, "Space by edge overlap"},
 /* 15 */ {0, {108,244,124,455}, RADIO, "Space by centerline distance"},
 /* 16 */ {0, {20,216,36,425}, CHECK, "Stagger alternate columns"},
 /* 17 */ {0, {60,216,76,400}, CHECK, "Stagger alternate rows"},
 /* 18 */ {0, {128,244,144,425}, RADIO, "Space by characteristic spacing"},
 /* 19 */ {0, {148,244,164,455}, RADIO, "Space by last measured distance"},
 /* 20 */ {0, {168,4,184,182}, CHECK, "Linear diagonal array"}
};
DIALOG us_arraydialog = {{50,75,267,539}, "Array Current Objects", 20, us_arraydialogitems};

/*
 * special case for the "array" dialog
 * X repeat factor        = 3 (edit text)
 * Y repeat factor        = 4 (edit text)
 * X spacing              = 5 (edit text)
 * Y spacing              = 6 (edit text)
 * Flip in X              = 8 (check)
 * Flip in Y              = 9 (check)
 * Add names              = 7 (check)
 * X spacing label        = 12 (message)
 * Y spacing label        = 13 (message)
 * Space by edge          = 14 (radio button)
 * Space by center        = 15 (radio button)
 * Stagger in X           = 16 (check)
 * Stagger in Y           = 17 (check)
 * Space by char. spacing = 18 (radio button)
 * Space by measured dist = 19 (radio button)
 * Linear diagonal array  = 20 (check)
 */
INTSML us_arraydlog(char *paramstart[])
{
	REGISTER INTBIG itemHit, first, xcentdist, ycentdist, xsize, ysize, lx, hx, ly, hy, i,
		chardistx, chardisty, curspacing, measdistx, measdisty, x, y, havechar;
	INTBIG xoverlap, yoverlap;
	static INTBIG lastXrepeat = 1, lastYrepeat = 1;
	static INTBIG lastXdist = 0, lastYdist = 0;
	static INTBIG lastspacing = 14;
	REGISTER VARIABLE *var;
	char line[40];
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER GEOM **list, *geom;
	REGISTER NODEPROTO *np;

	/* get the objects to be arrayed */
	list = us_gethighlighted(OBJNODEINST|OBJARCINST);
	if (list[0] == NOGEOM)
	{
		us_abortcommand("Must select circuitry before arraying it");
		return(0);
	}
	np = geomparent(list[0]);

	/* display the array dialog box */
	if (DiaInitDialog(&us_arraydialog) != 0) return(0);
	sprintf(line, "%ld", lastXrepeat);   DiaSetText(3, line);
	sprintf(line, "%ld", lastYrepeat);   DiaSetText(4, line);

	/* see if a facet was selected which has a characteristic distance */
	havechar = 0;
	for(i=0; list[i] != NOGEOM; i++)
	{
		if (list[i]->entrytype != OBJNODEINST) continue;
		ni = list[i]->entryaddr.ni;
		if (ni->proto->primindex != 0) continue;
		var = getval((INTBIG)ni->proto, VNODEPROTO, VINTEGER|VISARRAY, "FACET_characteristic_spacing");
		if (var == NOVARIABLE) continue;

		if (havechar != 0)
		{
			if (chardistx != ((INTBIG *)var->addr)[0] ||
				chardisty != ((INTBIG *)var->addr)[1])
			{
				havechar = 0;
				break;
			}
		}
		chardistx = ((INTBIG *)var->addr)[0];
		chardisty = ((INTBIG *)var->addr)[1];
		havechar = 1;
	}
	if (havechar == 0)
	{
		DiaDimItem(18);
		if (lastspacing == 18)
		{
			lastspacing = 14;
			lastXdist = lastYdist = 0;
		}
	}

	/* see if there was a measured distance */
	if (us_validmesaure != 0)
	{
		measdistx = abs(us_lastmeasurex);
		measdisty = abs(us_lastmeasurey);
	} else
	{
		DiaDimItem(19);
		if (lastspacing == 19)
		{
			lastspacing = 14;
			lastXdist = lastYdist = 0;
		}
	}

	sprintf(line, "%ld", lastXdist);     DiaSetText(5, line);
	sprintf(line, "%ld", lastYdist);     DiaSetText(6, line);
	curspacing = lastspacing;
	DiaSetControl(curspacing, 1);
	if (curspacing == 14)
	{
		DiaSetText(12, "X edge overlap:");
		DiaSetText(13, "Y edge overlap:");
	} else
	{
		DiaSetText(12, "X centerline distance:");
		DiaSetText(13, "Y centerline distance:");
	}

	/* mark the list of nodes and arcs in the facet that will be arrayed */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		ni->temp1 = 0;
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		ai->temp1 = 0;
	for(i=0; list[i] != NOGEOM; i++)
	{
		geom = list[i];
		if (geom->entrytype == OBJNODEINST)
		{
			ni = geom->entryaddr.ni;
			ni->temp1 = 1;
		} else
		{
			ai = geom->entryaddr.ai;
			ai->temp1 = 1;
			ai->end[0].nodeinst->temp1 = ai->end[1].nodeinst->temp1 = 1;
		}
	}

	/* determine spacing between arrayed objects */
	first = 1;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->temp1 == 0) continue;
		if (first != 0)
		{
			lx = ni->geom->lowx;   hx = ni->geom->highx;
			ly = ni->geom->lowy;   hy = ni->geom->highy;
			first = 0;
		} else
		{
			if (ni->geom->lowx < lx) lx = ni->geom->lowx;
			if (ni->geom->highx > hx) hx = ni->geom->highx;
			if (ni->geom->lowy < ly) ly = ni->geom->lowy;
			if (ni->geom->highy > hy) hy = ni->geom->highy;
		}
	}
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		if (ai->temp1 == 0) continue;
		if (first != 0)
		{
			lx = ai->geom->lowx;   hx = ai->geom->highx;
			ly = ai->geom->lowy;   hy = ai->geom->highy;
			first = 0;
		} else
		{
			if (ai->geom->lowx < lx) lx = ai->geom->lowx;
			if (ai->geom->highx > hx) hx = ai->geom->highx;
			if (ai->geom->lowy < ly) ly = ai->geom->lowy;
			if (ai->geom->highy > hy) hy = ai->geom->highy;
		}
	}
	xsize = xcentdist = hx - lx;
	ysize = ycentdist = hy - ly;
	xoverlap = yoverlap = 0;

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK && DiaValidEntry(3) && DiaValidEntry(4) && DiaValidEntry(5) &&
			DiaValidEntry(6)) break;
		if (itemHit == 7 || itemHit == 8 || itemHit == 9 || itemHit == 16 ||
			itemHit == 17 || itemHit == 20)
				DiaSetControl(itemHit, 1-DiaGetControl(itemHit));
		if (itemHit == 14 || itemHit == 15 || itemHit == 18 || itemHit == 19)
		{
			DiaSetControl(14, 0);
			DiaSetControl(15, 0);
			DiaSetControl(18, 0);
			DiaSetControl(19, 0);
			DiaSetControl(itemHit, 1);
			x = atola(DiaGetText(5));   y = atola(DiaGetText(6));
			switch (curspacing)
			{
				case 14: xoverlap = x;    yoverlap = y;    break;
				case 15: xcentdist = x;   ycentdist = y;   break;
				case 18: chardistx = x;   chardisty = y;   break;
				case 19: measdistx = x;   measdisty = y;   break;
			}
			curspacing = itemHit;
			if (curspacing == 14)
			{
				DiaSetText(12, "X edge overlap:");
				DiaSetText(13, "Y edge overlap:");
			} else
			{
				DiaSetText(12, "X centerline distance:");
				DiaSetText(13, "Y centerline distance:");
			}
			switch (curspacing)
			{
				case 14: x = xoverlap;    y = yoverlap;    break;
				case 15: x = xcentdist;   y = ycentdist;   break;
				case 18: x = chardistx;   y = chardisty;   break;
				case 19: x = measdistx;   y = measdisty;   break;
			}
			DiaSetText(5, latoa(x));
			DiaSetText(6, latoa(y));
			continue;
		}
	}
	lastXrepeat = atoi(DiaGetText(3));
	lastYrepeat = atoi(DiaGetText(4));
	lastXdist = atoi(DiaGetText(5));
	lastYdist = atoi(DiaGetText(6));
	lastspacing = curspacing;

	paramstart[0] = "";
	if (itemHit != CANCEL)
	{
		(void)initinfstr();
		(void)addstringtoinfstr(DiaGetText(3));
		if (DiaGetControl(8) != 0) (void)addtoinfstr('f');
		if (DiaGetControl(16) != 0) (void)addtoinfstr('s');
		(void)addtoinfstr(' ');
		(void)addstringtoinfstr(DiaGetText(4));
		if (DiaGetControl(9) != 0) (void)addtoinfstr('f');
		if (DiaGetControl(17) != 0) (void)addtoinfstr('s');
		(void)addtoinfstr(' ');
		xoverlap = atola(DiaGetText(5));
		yoverlap = atola(DiaGetText(6));
		if (DiaGetControl(14) == 0)
		{
			xoverlap = xsize - xoverlap;
			yoverlap = ysize - yoverlap;
		}
		(void)addstringtoinfstr(latoa(xoverlap));
		(void)addtoinfstr(' ');
		(void)addstringtoinfstr(latoa(yoverlap));
		if (DiaGetControl(7) == 0) (void)addstringtoinfstr(" no-names");
		if (DiaGetControl(20) != 0) (void)addstringtoinfstr(" diagonal");
		paramstart[0] = returninfstr();
	}
	DiaDoneDialog();
	return(1);
}

/****************************** ARTWORK AND COLOR DIALOGS ******************************/

/* Layer Patterns */
unsigned char us_icon300[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0210, 0210, 0210, 0210, 0104, 0104, 021, 021, 042, 042, 042, 042, 021, 021, 0104, 0104,
	0210, 0210, 0210, 0210, 0104, 0104, 021, 021, 042, 042, 042, 042, 021, 021, 0104, 0104,
	0210, 0210, 0210, 0210, 0104, 0104, 021, 021, 042, 042, 042, 042, 021, 021, 0104, 0104,
	0210, 0210, 0210, 0210, 0104, 0104, 021, 021, 042, 042, 042, 042, 021, 021, 0104, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon301[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0314, 0314, 0377, 0377, 0314, 0314, 0, 0, 063, 063, 0377, 0377, 063, 063, 0, 0,
	0314, 0314, 0377, 0377, 0314, 0314, 0, 0, 063, 063, 0377, 0377, 063, 063, 0, 0,
	0314, 0314, 0377, 0377, 0314, 0314, 0, 0, 063, 063, 0377, 0377, 063, 063, 0, 0,
	0314, 0314, 0377, 0377, 0314, 0314, 0, 0, 063, 063, 0377, 0377, 063, 063, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon302[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0125, 0125, 0140, 0140, 0125, 0125, 0220, 0220, 0125, 0125, 0220, 0220, 0125, 0125, 0140, 0140,
	0125, 0125, 06, 06, 0125, 0125, 011, 011, 0125, 0125, 011, 011, 0125, 0125, 06, 06,
	0125, 0125, 0140, 0140, 0125, 0125, 0220, 0220, 0125, 0125, 0220, 0220, 0125, 0125, 0140, 0140,
	0125, 0125, 06, 06, 0125, 0125, 011, 011, 0125, 0125, 011, 011, 0, 0, 06, 06,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon303[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	042, 042, 0104, 0104, 0, 0, 021, 021, 0210, 0210, 0104, 0104, 0, 0, 021, 021,
	042, 042, 0104, 0104, 0, 0, 021, 021, 0210, 0210, 0104, 0104, 0, 0, 021, 021,
	042, 042, 0104, 0104, 0, 0, 021, 021, 0210, 0210, 0104, 0104, 0, 0, 021, 021,
	042, 042, 0104, 0104, 0, 0, 021, 021, 0210, 0210, 0104, 0104, 0, 0, 021, 021,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon304[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	020, 020, 04, 04, 040, 040, 02, 02, 0100, 0100, 01, 01, 0200, 0200, 0200, 0200,
	01, 01, 0100, 0100, 02, 02, 040, 040, 04, 04, 020, 020, 010, 010, 010, 010,
	020, 020, 04, 04, 040, 040, 02, 02, 0100, 0100, 01, 01, 0200, 0200, 0200, 0200,
	01, 01, 0100, 0100, 02, 02, 040, 040, 04, 04, 020, 020, 010, 010, 010, 010,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon305[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0100, 0100, 020, 0, 0200, 0200, 0, 040, 01, 01, 0100, 0, 02, 02, 0, 0200,
	01, 01, 0, 01, 0200, 0200, 02, 0, 0100, 0100, 0, 04, 040, 040, 010, 0,
	0100, 0100, 0, 020, 0200, 0200, 040, 0, 01, 01, 0, 0100, 02, 02, 0200, 0,
	01, 01, 01, 0, 0200, 0200, 0, 02, 0100, 0100, 04, 0, 040, 040, 0, 010,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon306[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	010, 0, 0, 0, 0, 04, 03, 03, 02, 0, 0204, 0204, 0, 01, 03, 03,
	0, 0200, 0, 0, 0100, 0, 060, 060, 0, 040, 0110, 0110, 020, 0, 060, 060,
	0, 010, 0, 0, 04, 0, 03, 03, 0, 02, 0204, 0204, 01, 0, 03, 03,
	0200, 0, 0, 0, 0, 0100, 060, 060, 040, 0, 0110, 0110, 0, 020, 060, 060,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon307[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	034, 034, 0, 0, 076, 076, 0314, 0314, 066, 066, 0, 0, 076, 076, 0314, 0314,
	034, 034, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	034, 034, 0, 0, 076, 076, 0314, 0314, 066, 066, 0, 0, 076, 076, 0314, 0314,
	034, 034, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon308[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 042, 042, 021, 021, 0210, 0210, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 042, 042, 021, 021, 0210, 0210, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 042, 042, 021, 021, 0210, 0210, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 042, 042, 021, 021, 0210, 0210, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon309[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 042, 042, 042, 042, 0104, 0104, 0125, 0125, 0210, 0210, 042, 042,
	0, 0, 0, 0, 042, 042, 042, 042, 0104, 0104, 0125, 0125, 0210, 0210, 042, 042,
	0, 0, 0, 0, 042, 042, 042, 042, 0104, 0104, 0125, 0125, 0210, 0210, 042, 042,
	0, 0, 0, 0, 042, 042, 042, 042, 0104, 0104, 0125, 0125, 0210, 0210, 042, 042,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
unsigned char us_icon310[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0377, 0377, 0, 0, 0377, 0377, 0, 0, 0377, 0377, 0, 0, 0377, 0377,
	0, 0, 0377, 0377, 0, 0, 0377, 0377, 0, 0, 0377, 0377, 0, 0, 0377, 0377,
	0, 0, 0377, 0377, 0, 0, 0377, 0377, 0, 0, 0377, 0377, 0, 0, 0377, 0377,
	0, 0, 0377, 0377, 0, 0, 0377, 0377, 0, 0, 0377, 0377, 0, 0, 0377, 0377,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

UINTSML us_predefpats[] =
{
	0x8888,  /* X   X   X   X    */
	0x4444,  /*  X   X   X   X   */
	0x2222,  /*   X   X   X   X  */
	0x1111,  /*    X   X   X   X */
	0x8888,  /* X   X   X   X    */
	0x4444,  /*  X   X   X   X   */
	0x2222,  /*   X   X   X   X  */
	0x1111,  /*    X   X   X   X */

	0x8888,  /* X   X   X   X    */
	0x1111,  /*    X   X   X   X */
	0x2222,  /*   X   X   X   X  */
	0x4444,  /*  X   X   X   X   */
	0x8888,  /* X   X   X   X    */
	0x1111,  /*    X   X   X   X */
	0x2222,  /*   X   X   X   X  */
	0x4444,  /*  X   X   X   X   */

	0xCCCC,  /* XX  XX  XX  XX   */
	0xCCCC,  /* XX  XX  XX  XX   */
	0x3333,  /*   XX  XX  XX  XX */
	0x3333,  /*   XX  XX  XX  XX */
	0xCCCC,  /* XX  XX  XX  XX   */
	0xCCCC,  /* XX  XX  XX  XX   */
	0x3333,  /*   XX  XX  XX  XX */
	0x3333,  /*   XX  XX  XX  XX */

	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0x0000,  /*                  */
	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0x0000,  /*                  */
	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0x0000,  /*                  */
	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0x0000,  /*                  */

	0xAAAA,  /* X X X X X X X X  */
	0xAAAA,  /* X X X X X X X X  */
	0xAAAA,  /* X X X X X X X X  */
	0xAAAA,  /* X X X X X X X X  */
	0xAAAA,  /* X X X X X X X X  */
	0xAAAA,  /* X X X X X X X X  */
	0xAAAA,  /* X X X X X X X X  */
	0xAAAA,  /* X X X X X X X X  */

	0x6060,  /*  XX      XX      */
	0x9090,  /* X  X    X  X     */
	0x9090,  /* X  X    X  X     */
	0x6060,  /*  XX      XX      */
	0x0606,  /*      XX      XX  */
	0x0909,  /*     X  X    X  X */
	0x0909,  /*     X  X    X  X */
	0x0606,  /*      XX      XX  */

	0x2222,  /*   X   X   X   X  */
	0x0000,  /*                  */
	0x8888,  /* X   X   X   X    */
	0x0000,  /*                  */
	0x2222,  /*   X   X   X   X  */
	0x0000,  /*                  */
	0x8888,  /* X   X   X   X    */
	0x0000,  /*                  */

	0x4444,  /*  X   X   X   X   */
	0x1111,  /*    X   X   X   X */
	0x4444,  /*  X   X   X   X   */
	0x1111,  /*    X   X   X   X */
	0x4444,  /*  X   X   X   X   */
	0x1111,  /*    X   X   X   X */
	0x4444,  /*  X   X   X   X   */
	0x1111,  /*    X   X   X   X */

	0x1010,  /*    X       X     */
	0x2020,  /*   X       X      */
	0x4040,  /*  X       X       */
	0x8080,  /* X       X        */
	0x0101,  /*        X       X */
	0x0202,  /*       X       X  */
	0x0404,  /*      X       X   */
	0x0808,  /*     X       X    */

	0x0808,  /*     X       X    */
	0x0404,  /*      X       X   */
	0x0202,  /*       X       X  */
	0x0101,  /*        X       X */
	0x8080,  /* X       X        */
	0x4040,  /*  X       X       */
	0x2020,  /*   X       X      */
	0x1010,  /*    X       X     */

	0x4040,  /*  X       X       */
	0x8080,  /* X       X        */
	0x0101,  /*        X       X */
	0x0202,  /*       X       X  */
	0x0101,  /*        X       X */
	0x8080,  /* X       X        */
	0x4040,  /*  X       X       */
	0x2020,  /*   X       X      */

	0x1000,  /*    X             */
	0x0020,  /*           X      */
	0x4000,  /*  X               */
	0x0080,  /*         X        */
	0x0001,  /*                X */
	0x0200,  /*       X          */
	0x0004,  /*              X   */
	0x0800,  /*     X            */

	0x0800,  /*     X            */
	0x0004,  /*              X   */
	0x0200,  /*       X          */
	0x0001,  /*                X */
	0x0080,  /*         X        */
	0x4000,  /*  X               */
	0x0020,  /*           X      */
	0x1000,  /*    X             */

	0x0000,  /*                  */
	0x0303,  /*       XX      XX */
	0x4848,  /*  X  X    X  X    */
	0x0303,  /*       XX      XX */
	0x0000,  /*                  */
	0x3030,  /*   XX      XX     */
	0x8484,  /* X    X  X    X   */
	0x3030,  /*   XX      XX     */

	0x1C1C,  /*    XXX     XXX   */
	0x3E3E,  /*   XXXXX   XXXXX  */
	0x3636,  /*   XX XX   XX XX  */
	0x3E3E,  /*   XXXXX   XXXXX  */
	0x1C1C,  /*    XXX     XXX   */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x0000,  /*                  */

	0x0000,  /*                  */
	0xCCCC,  /* XX  XX  XX  XX   */
	0x0000,  /*                  */
	0xCCCC,  /* XX  XX  XX  XX   */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x0000,  /*                  */

	0x0000,  /*                  */
	0x0000,  /*                  */
	0x2222,  /*   X   X   X   X  */
	0x8888,  /* X   X   X   X    */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x2222,  /*   X   X   X   X  */
	0x8888,  /* X   X   X   X    */

	0x0000,  /*                  */
	0x0000,  /*                  */
	0x1111,  /*    X   X   X   X */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x1111,  /*    X   X   X   X */
	0x0000,  /*                  */

	0x0000,  /*                  */
	0x2222,  /*   X   X   X   X  */
	0x4444,  /*  X   X   X   X   */
	0x8888,  /* X   X   X   X    */
	0x0000,  /*                  */
	0x2222,  /*   X   X   X   X  */
	0x4444,  /*  X   X   X   X   */
	0x8888,  /* X   X   X   X    */

	0x0000,  /*                  */
	0x2222,  /*   X   X   X   X  */
	0x5555,  /*  X X X X X X X X */
	0x2222,  /*   X   X   X   X  */
	0x0000,  /*                  */
	0x2222,  /*   X   X   X   X  */
	0x5555,  /*  X X X X X X X X */
	0x2222,  /*   X   X   X   X  */

	0x0000,  /*                  */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x0000,  /*                  */
	0x0000,  /*                  */

	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0xFFFF,  /* XXXXXXXXXXXXXXXX */
	0xFFFF   /* XXXXXXXXXXXXXXXX */
};

/* Artwork Color/Layer */
DIALOGITEM us_artworkdialogitems[] =
{
 /*  1 */ {0, {168,340,192,404}, BUTTON, "OK"},
 /*  2 */ {0, {168,244,192,308}, BUTTON, "Cancel"},
 /*  3 */ {0, {120,292,136,419}, RADIO, "Outlined Pattern"},
 /*  4 */ {0, {32,16,160,272}, USERDRAWN, ""},
 /*  5 */ {0, {56,292,72,388}, RADIO, "Solid color"},
 /*  6 */ {0, {168,24,184,77}, MESSAGE, "Color:"},
 /*  7 */ {0, {88,292,104,388}, RADIO, "Use Pattern"},
 /*  8 */ {0, {0,16,32,48}, ICON, (char *)us_icon300},
 /*  9 */ {0, {0,48,32,80}, ICON, (char *)us_icon301},
 /* 10 */ {0, {0,80,32,112}, ICON, (char *)us_icon302},
 /* 11 */ {0, {0,112,32,144}, ICON, (char *)us_icon303},
 /* 12 */ {0, {0,144,32,176}, ICON, (char *)us_icon304},
 /* 13 */ {0, {0,176,32,208}, ICON, (char *)us_icon305},
 /* 14 */ {0, {0,208,32,240}, ICON, (char *)us_icon306},
 /* 15 */ {0, {0,240,32,272}, ICON, (char *)us_icon307},
 /* 16 */ {0, {0,272,32,304}, ICON, (char *)us_icon308},
 /* 17 */ {0, {0,304,32,336}, ICON, (char *)us_icon309},
 /* 18 */ {0, {0,336,32,368}, ICON, (char *)us_icon310},
 /* 19 */ {0, {168,80,184,214}, POPUP, "Overlappable 1"}
};
DIALOG us_artworkdialog = {{50,75,253,503}, "Set Look of Highlighted", 19, us_artworkdialogitems};

DIALOGITEM us_patterndialogitems[] =
{
 /*  1 */ {0, {176,296,200,360}, BUTTON, "OK"},
 /*  2 */ {0, {136,296,160,360}, BUTTON, "Cancel"},
 /*  3 */ {0, {4,8,20,184}, MESSAGE, "Stipple Pattern for Layer:"},
 /*  4 */ {0, {84,16,212,272}, USERDRAWN, ""},
 /*  5 */ {0, {4,184,20,368}, POPUP, ""},
 /*  6 */ {0, {64,184,80,336}, CHECK, "Outline Pattern"},
 /*  7 */ {0, {64,8,80,160}, CHECK, "Use Stipple Pattern"},
 /*  8 */ {0, {28,8,60,40}, ICON, (char *)us_icon300},
 /*  9 */ {0, {28,40,60,72}, ICON, (char *)us_icon301},
 /* 10 */ {0, {28,72,60,104}, ICON, (char *)us_icon302},
 /* 11 */ {0, {28,104,60,136}, ICON, (char *)us_icon303},
 /* 12 */ {0, {28,136,60,168}, ICON, (char *)us_icon304},
 /* 13 */ {0, {28,168,60,200}, ICON, (char *)us_icon305},
 /* 14 */ {0, {28,200,60,232}, ICON, (char *)us_icon306},
 /* 15 */ {0, {28,232,60,264}, ICON, (char *)us_icon307},
 /* 16 */ {0, {28,264,60,296}, ICON, (char *)us_icon308},
 /* 17 */ {0, {28,296,60,328}, ICON, (char *)us_icon309},
 /* 18 */ {0, {28,328,60,360}, ICON, (char *)us_icon310}
};
DIALOG us_patterndialog = {{50,75,271,453}, "Layer Pattern Options", 18, us_patterndialogitems};

/*
 * special case for the "artwork primitive color&pattern" dialog
 * Pattern squares = 4 (user item)
 * Color           = 19 (popup)
 * Solid           = 5 (radio button)
 * Use Pattern     = 7 (radio button)
 * Outline pattern = 3 (radio button)
 * Predefined pats = 8-18
 */
static RECTAREA us_strokerect;
static INTBIG us_origpattern[8];
static INTBIG us_strokeup;

void us_redrawpattern(RECTAREA*);
INTSML us_patternstroke(INTBIG, INTBIG);

void us_redrawpattern(RECTAREA *bigr)
{
	INTBIG j, k, bits;
	RECTAREA r;

	for(j=0; j<8; j++)
	{
		bits = us_origpattern[j] & 0xFFFF;
		for(k=0; k<16; k++)
		{
			r.left = us_strokerect.left + k*16;   r.right = r.left + 17;
			r.top = us_strokerect.top + j*16;     r.bottom = r.top + 17;
			if ((bits & (1<<(15-k))) != 0) DiaDrawRect(4, &r, 0, 0, 0); else DiaFrameRect(4, &r);
		}
	}
}

INTSML us_patternstroke(INTBIG ox, INTBIG oy)
{
	INTSML x, y;
	REGISTER INTBIG j, k, bits;
	RECTAREA r;

	DiaGetMouse(&x, &y);
	j = (y - us_strokerect.top) / 16;  if (j < 0) j = 0;  if (j > 7) j = 7;
	k = (x - us_strokerect.left) / 16;  if (k < 0) k = 0;  if (k > 15) k = 15;
	r.left = us_strokerect.left + k*16;   r.right = r.left + 17;
	r.top = us_strokerect.top + j*16;     r.bottom = r.top + 17;
	bits = us_origpattern[j] & 0xFFFF;
	if (us_strokeup == 0)
	{
		DiaFrameRect(4, &r);
		bits &= ~(1<<(15-k));
	} else
	{
		DiaDrawRect(4, &r, 0, 0, 0);
		bits |= 1<<(15-k);
	}
	us_origpattern[j] = bits;
	return(0);
}

INTSML us_artlookdlog(char *paramstart[])
{
	RECTAREA r;
	INTBIG addr, type, itemHit;
	UINTSML spattern[8];
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER VARIABLE *colorvar, *patternvar;
	INTBIG i, j, k, bits, err;
	INTSML x, y;
	HIGHLIGHT *high;
	static char *colornames[25] = {"Overlappable 1", "Overlappable 2", "Overlappable 3",
		"Overlappable 4", "Overlappable 5", "White", "Black", "Gray", "Light Gray",
		"Dark Gray", "Red", "Light Red", "Dark Red", "Green", "Light Green", "Dark Green",
		"Blue", "Light Blue", "Dark Blue", "Cyan", "Magenta", "Yellow", "Orange",
		"Purple", "Brown"};
	static INTSML valuelist[25] =
	{
		COLORT1,		/* overlappable 1 */
		COLORT2,		/* overlappable 2 */
		COLORT3,		/* overlappable 3 */
		COLORT4,		/* overlappable 4 */
		COLORT5,		/* overlappable 5 */
		WHITE,			/* white */
		BLACK,			/* black */
		GRAY,			/* gray */
		LGRAY,			/* light gray */
		DGRAY,			/* dark gray */
		RED,			/* red */
		LRED,			/* light red */
		DRED,			/* dark red */
		GREEN,			/* green */
		LGREEN,			/* light green */
		DGREEN,			/* dark green */
		BLUE,			/* blue */
		LBLUE,			/* light blue */
		DBLUE,			/* dark blue */
		CYAN,			/* cyan */
		MAGENTA,		/* magenta */
		YELLOW,			/* yellow */
		ORANGE,			/* orange */
		PURPLE,			/* purple */
		BROWN			/* brown */
	};

	/* make sure there is an artwork primitive highlighted */
	high = us_getonehighlight();
	if (high == NOHIGHLIGHT) return(0);
	err = 0;
	if ((high->status&HIGHTYPE) != HIGHFROM) err = 1; else
	{
		if (high->fromgeom->entrytype == OBJNODEINST)
		{
			ni = high->fromgeom->entryaddr.ni;
			if (ni->proto->primindex == 0 || ni->proto->tech != art_tech) err = 1;
			addr = (INTBIG)ni;
			type = VNODEINST;
		} else
		{
			ai = high->fromgeom->entryaddr.ai;
			if (ai->proto->tech != art_tech) err = 1;
			addr = (INTBIG)ai;
			type = VARCINST;
		}
	}
	if (err != 0)
	{
		us_abortcommand("First find a single node or arc in the Artwork technology");
		return(0);
	}

	/* display the artwork dialog box */
	if (DiaInitDialog(&us_artworkdialog) != 0) return(0);
	DiaSetPopup(19, 25, colornames);
	DiaItemRect(4, &us_strokerect);
	DiaRedispRoutine(4, us_redrawpattern);
	DiaFrameRect(4, &us_strokerect);

	/* set existing conditions */
	colorvar = getvalkey(addr, type, VINTEGER, art_colorkey);
	if (colorvar != NOVARIABLE)
	{
		for(i=0; i<25; i++) if (valuelist[i] == colorvar->addr)
		{
			DiaSetPopupEntry(19, i);
			break;
		}
	}
	for(i=0; i<8; i++) us_origpattern[i] = 0;
	patternvar = getvalkey(addr, type, -1, art_patternkey);
	if (patternvar != NOVARIABLE && getlength(patternvar) == 8)
	{
		if ((patternvar->type&VTYPE) == VINTEGER)
		{
			for(i=0; i<8; i++) us_origpattern[i] = ((UINTBIG *)patternvar->addr)[i];
			DiaSetControl(7, 1);
		} else if ((patternvar->type&VTYPE) == VSHORT)
		{
			for(i=0; i<8; i++) us_origpattern[i] = ((UINTSML *)patternvar->addr)[i];
			DiaSetControl(3, 1);
		}
	} else DiaSetControl(5, 1);
	us_redrawpattern(&us_strokerect);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK) break;
		if (itemHit == 4)
		{
			DiaGetMouse(&x, &y);
			j = (y - us_strokerect.top) / 16;  if (j < 0) j = 0;  if (j > 7) j = 7;
			k = (x - us_strokerect.left) / 16;  if (k < 0) k = 0;  if (k > 15) k = 15;
			bits = us_origpattern[j] & 0xFFFF;
			if ((bits & (1<<(15-k))) != 0) us_strokeup = 0; else us_strokeup = 1;
			trackcursor(0, us_nullup, us_nullvoid, us_patternstroke, us_nullchar,
				us_nullvoid, TRACKNORMAL);
			continue;
		}
		if (itemHit == 5 || itemHit == 7 || itemHit == 3)
		{
			DiaSetControl(5, 0);
			DiaSetControl(7, 0);
			DiaSetControl(3, 0);
			DiaSetControl(itemHit, 1);
			continue;
		}
		if (itemHit >= 8 && itemHit <= 18)
		{
			DiaGetMouse(&x, &y);
			DiaItemRect(itemHit, &r);
			j = (r.top+r.bottom)/2;
			if (y < j-8 || y > j+8) continue;
			i = (itemHit-8) * 2;
			if (x > (r.left+r.right)/2) i++;
			i *= 8;
			for(j=0; j<8; j++) us_origpattern[j] = us_predefpats[i++] & 0xFFFF;
			us_redrawpattern(&us_strokerect);
			continue;
		}
	}

	if (itemHit != CANCEL)
	{
		/* prepare for change */
		us_pushhighlight();
		us_clearhighlightcount();
		startobjectchange(addr, type);

		/* make the change */
		if (DiaGetControl(5) != 0)
		{
			if (patternvar != NOVARIABLE) (void)delvalkey(addr, type, art_patternkey);
		} else if (DiaGetControl(7) != 0)
		{
			(void)setvalkey(addr, type, art_patternkey, (INTBIG)us_origpattern,
				VINTEGER|VISARRAY|(8<<VLENGTHSH));
		} else
		{
			for(i=0; i<8; i++) spattern[i] = (UINTSML)us_origpattern[i];
			(void)setvalkey(addr, type, art_patternkey, (INTBIG)spattern, VSHORT|VISARRAY|(8<<VLENGTHSH));
		}
		i = DiaGetPopupEntry(19);
		if (valuelist[i] == 014)
		{
			/* black is the default */
			if (colorvar != NOVARIABLE) (void)delvalkey(addr, type, art_colorkey);
		} else (void)setvalkey(addr, type, art_colorkey, valuelist[i], VINTEGER);

		/* complete change */
		endobjectchange(addr, type);
		(void)us_pophighlight(0);
	}
	DiaDoneDialog();
	paramstart[0] = "";
	return(1);
}

/*
 * special case for the "layer pattern" dialog
 * Pattern squares = 4 (user item) (***MUST NOT CHANGE***)
 * Name of layer   = 5 (popup)
 * Outline pattern = 6 (check)
 * Use pattern     = 7 (check)
 * Predefined pats = 8-18 (icon)
 */
INTSML us_patterndlog(char *paramstart[])
{
	INTBIG itemHit, curlayer, i, j, k, newpat[9];
	INTSML *origbits, *orignature, *origchanged, x, y, bits;
	RECTAREA r;
	REGISTER VARIABLE *var;
	REGISTER char **layernames;

	/* get memory to save state */
	origbits = (INTSML *)emalloc(el_curtech->layercount * 8 * SIZEOFINTSML, us_aid->cluster);
	if (origbits == 0) return(0);
	orignature = (INTSML *)emalloc(el_curtech->layercount * SIZEOFINTSML, us_aid->cluster);
	if (orignature == 0) return(0);
	origchanged = (INTSML *)emalloc(el_curtech->layercount * SIZEOFINTSML, us_aid->cluster);
	if (origchanged == 0) return(0);
	for(i=0; i<el_curtech->layercount; i++)
	{
		orignature[i] = (el_curtech->layers[i])->colstyle;
		origchanged[i] = 0;
		for(j=0; j<8; j++) origbits[i*8+j] = (el_curtech->layers[i])->raster[j];

		/* see if there is an override */
		(void)initinfstr();
		(void)addstringtoinfstr("TECH_layer_pattern_");
		(void)addstringtoinfstr(layername(el_curtech, (INTSML)i));
		var = getval((INTBIG)el_curtech, VTECHNOLOGY, VINTEGER|VISARRAY, returninfstr());
		if (var != NOVARIABLE)
		{
			orignature[i] = (INTSML)(((INTBIG *)var->addr)[8]);
			for(j=0; j<8; j++) origbits[i*8+j] = (INTSML)(((INTBIG *)var->addr)[j]);
		}
	}
	curlayer = 0;

	/* display the pattern dialog box */
	if (DiaInitDialog(&us_patterndialog) != 0)
	{
		efree((char *)origbits);
		efree((char *)orignature);
		efree((char *)origchanged);
		return(0);
	}
	DiaItemRect(4, &us_strokerect);
	DiaRedispRoutine(4, us_redrawpattern);
	DiaFrameRect(4, &us_strokerect);

	/* setup list of layer names */
	layernames = (char **)emalloc(el_curtech->layercount * (sizeof (char *)), el_tempcluster);
	for(i=0; i<el_curtech->layercount; i++)
		allocstring(&layernames[i], layername(el_curtech, (INTSML)i), el_tempcluster);
	DiaSetPopup(5, el_curtech->layercount, layernames);
	for(i=0; i<el_curtech->layercount; i++)
		efree(layernames[i]);
	efree((char *)layernames);

	/* setup initial layer: number 0 */
	curlayer = 0;
	DiaSetPopupEntry(5, curlayer);
	if ((orignature[curlayer]&NATURE) == PATTERNED)
	{
		DiaUnDimItem(6);
		DiaSetControl(7, 1);
		if ((orignature[curlayer]&OUTLINEPAT) != 0) DiaSetControl(6, 1);
	} else DiaDimItem(6);
	for(j=0; j<8; j++) us_origpattern[j] = origbits[curlayer*8+j] & 0xFFFF;
	us_redrawpattern(&us_strokerect);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK) break;
		if (itemHit == 7)
		{
			/* the "Use Stipple" box */
			if (DiaGetControl(7) == 0)
			{
				DiaSetControl(7, 1);
				DiaUnDimItem(6);
				orignature[curlayer] = (orignature[curlayer] & ~NATURE) | PATTERNED;
			} else
			{
				DiaSetControl(7, 0);
				DiaSetControl(6, 0);
				DiaDimItem(6);
				orignature[curlayer] = (orignature[curlayer] & ~NATURE) | SOLIDC;
			}
			origchanged[curlayer] = 1;
			continue;
		}
		if (itemHit == 6)
		{
			/* the "Outline Pattern" box */
			if (DiaGetControl(6) == 0)
			{
				DiaSetControl(6, 1);
				orignature[curlayer] |= OUTLINEPAT;
			} else
			{
				DiaSetControl(6, 0);
				orignature[curlayer] &= ~OUTLINEPAT;
			}
			origchanged[curlayer] = 1;
			continue;
		}
		if (itemHit == 5)
		{
			/* the layer popup */
			curlayer = DiaGetPopupEntry(5);
			DiaSetControl(7, 0);
			DiaUnDimItem(6);
			DiaSetControl(6, 0);
			if ((orignature[curlayer]&NATURE) == PATTERNED)
			{
				DiaSetControl(7, 1);
				if ((orignature[curlayer]&OUTLINEPAT) != 0) DiaSetControl(6, 1);
			} else DiaDimItem(6);
			for(j=0; j<8; j++) us_origpattern[j] = origbits[curlayer*8+j] & 0xFFFF;
			us_redrawpattern(&us_strokerect);
			continue;
		}
		if (itemHit == 4)
		{
			DiaGetMouse(&x, &y);
			j = (y - us_strokerect.top) / 16;  if (j < 0) j = 0;  if (j > 7) j = 7;
			k = (x - us_strokerect.left) / 16;  if (k < 0) k = 0;  if (k > 15) k = 15;
			bits = origbits[curlayer*8+j] & 0xFFFF;
			if ((bits & (1<<(15-k))) != 0) us_strokeup = 0; else us_strokeup = 1;
			trackcursor(0, us_nullup, us_nullvoid, us_patternstroke,
				us_nullchar, us_nullvoid, TRACKNORMAL);
			for(j=0; j<8; j++) origbits[curlayer*8+j] = (UINTSML)us_origpattern[j];
			origchanged[curlayer] = 1;
			continue;
		}
		if (itemHit >= 8 && itemHit <= 18)
		{
			DiaGetMouse(&x, &y);
			DiaItemRect(itemHit, &r);
			j = (r.top+r.bottom)/2;
			if (y < j-8 || y > j+8) continue;
			i = (itemHit-8) * 2;
			if (x > (r.left+r.right)/2) i++;
			i *= 8;
			for(j=0; j<8; j++)
			{
				bits = us_predefpats[i++] & 0xFFFF;
				origbits[curlayer*8+j] = bits;
				us_origpattern[j] = bits;
			}
			us_redrawpattern(&us_strokerect);
			origchanged[curlayer] = 1;
			continue;
		}
	}

	if (itemHit != CANCEL)
	{
		for(i=0; i<el_curtech->layercount; i++)
		{
			if (origchanged[i] == 0) continue;
			(el_curtech->layers[i])->colstyle = orignature[i];
			for(j=0; j<8; j++) (el_curtech->layers[i])->raster[j] = origbits[i*8+j];

			/* save a shadow variable for this layer pattern */
			for(j=0; j<8; j++) newpat[j] = origbits[i*8+j];
			newpat[8] = orignature[i];
			(void)initinfstr();
			(void)addstringtoinfstr("TECH_layer_pattern_");
			(void)addstringtoinfstr(layername(el_curtech, (INTSML)i));
			(void)setval((INTBIG)el_curtech, VTECHNOLOGY, returninfstr(),
				(INTBIG)newpat, VINTEGER|VISARRAY|(9<<VLENGTHSH));
		}
	}
	efree((char *)origbits);
	efree((char *)orignature);
	efree((char *)origchanged);
	DiaDoneDialog();
	paramstart[0] = "";
	return(1);
}

/****************************** CHANGE DIALOG ******************************/

/* Change */
DIALOGITEM us_changedialogitems[] =
{
 /*  1 */ {0, {144,344,168,416}, BUTTON, "OK"},
 /*  2 */ {0, {144,248,168,320}, BUTTON, "Cancel"},
 /*  3 */ {0, {16,240,32,430}, RADIO, "Change this one only"},
 /*  4 */ {0, {72,240,88,430}, RADIO, "Change all in this facet"},
 /*  5 */ {0, {100,240,116,430}, RADIO, "Change all in this library"},
 /*  7 */ {0, {44,240,60,430}, RADIO, "Change all connected to this"},
 /*  6 */ {0, {8,16,168,231}, SCROLL, ""}
};
DIALOG us_changedialog = {{50,75,228,514}, "Change Type of Highlight", 7, us_changedialogitems};

INTSML us_onproto(PORTPROTO*, ARCPROTO*);

/*
 * special case for the "change" dialog
 * Alternate list = 7 (user item)
 * This only      = 3 (radio button)
 * In facet       = 4 (radio button)
 * Universally    = 5 (radio button)
 * Connected      = 6 (radio button)
 */
INTSML us_replacedlog(char *paramstart[])
{
	INTBIG i, itemHit;
	HIGHLIGHT high;
	REGISTER VARIABLE *var;
	REGISTER ARCPROTO *ap;
	REGISTER NODEPROTO *np;
	REGISTER ARCINST *ai;
	REGISTER PORTPROTO *pp1, *pp2;

	/* display the change dialog box */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE) return(0);
	if (getlength(var) > 1) return(0);
	if (us_makehighlight(((char **)var->addr)[0], &high) != 0) return(0);
	if ((high.status&HIGHTYPE) != HIGHFROM) return(0);
	if (DiaInitDialog(&us_changedialog) != 0) return(0);
	DiaInitTextDialog(7, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, 0,
		SCSELMOUSE|SCSELKEY|SCDOUBLEQUIT);
	if (high.fromgeom->entrytype == OBJARCINST)
	{
		/* load arcs in current technology, arc's technology, and generic technology */
		ai = high.fromgeom->entryaddr.ai;
		pp1 = ai->end[0].portarcinst->proto;
		pp2 = ai->end[1].portarcinst->proto;
		for(ap = el_curtech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		{
			if (us_onproto(pp1, ap) == 0) continue;
			if (us_onproto(pp2, ap) == 0) continue;
			DiaStuffLine(7, ap->protoname);
		}
		if (el_curtech != gen_tech)
			for(ap = gen_tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		{
			if (us_onproto(pp1, ap) == 0) continue;
			if (us_onproto(pp2, ap) == 0) continue;
			DiaStuffLine(7, describearcproto(ap));
		}
		if (ai->proto->tech != el_curtech && ai->proto->tech != gen_tech)
			for(ap = ai->proto->tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		{
			if (us_onproto(pp1, ap) == 0) continue;
			if (us_onproto(pp2, ap) == 0) continue;
			DiaStuffLine(7, describearcproto(ap));
		}
	} else
	{
		if (high.fromgeom->entryaddr.ni->proto->primindex == 0)
		{
			/* facet: only list other facets as replacements */
			DiaLoadTextDialog(7, us_topoffacets, us_nextparse, DiaNullDlogDone, 0);
		} else
		{
			/* primitive: list primitives in this and the generic technology */
			for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				DiaStuffLine(7, np->primname);
			if (el_curtech != gen_tech)
			{
				DiaStuffLine(7, "Generic:Universal-Pin");
				DiaStuffLine(7, "Generic:Invisible-Pin");
				DiaStuffLine(7, "Generic:Unrouted-Pin");
			}
		}
	}
	DiaSelectLine(7, 0);
	DiaSetControl(3, 1);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit >= 3 && itemHit <= 6)
		{
			for(i=3; i<=6; i++) DiaSetControl(i, 0);
			DiaSetControl(itemHit, 1);
		}
	}

	paramstart[0] = "";
	if (itemHit != CANCEL)
	{
		(void)initinfstr();
		(void)addstringtoinfstr(DiaGetScrollLine(7, DiaGetCurLine(7)));
		if (DiaGetControl(4) != 0) (void)addstringtoinfstr(" this-facet");
		if (DiaGetControl(5) != 0) (void)addstringtoinfstr(" universally");
		if (DiaGetControl(6) != 0) (void)addstringtoinfstr(" connected");
		paramstart[0] = returninfstr();
	}
	DiaDoneDialog();
	return(1);
}

/*
 * helper routine to determine whether arcproto "ap" can connect to portproto "pp".
 * returns nonzero if so
 */
INTSML us_onproto(PORTPROTO *pp, ARCPROTO *ap)
{
	REGISTER INTBIG i;

	for(i=0; pp->connects[i] != NOARCPROTO; i++)
		if (pp->connects[i] == ap) return(1);
	return(0);
}

/****************************** CHOICE DIALOGS ******************************/

/* No/Yes */
DIALOGITEM us_noyesdialogitems[] =
{
 /*  1 */ {0, {80,132,104,204}, BUTTON, "No"},
 /*  2 */ {0, {80,60,104,124}, BUTTON, "Yes"},
 /*  3 */ {0, {8,8,72,256}, MESSAGE, ""}
};
DIALOG us_noyesdialog = {{50,75,163,341}, "Choice", 3, us_noyesdialogitems};

/* Yes/No */
DIALOGITEM us_yesnodialogitems[] =
{
 /*  1 */ {0, {64,156,88,228}, BUTTON, "Yes"},
 /*  2 */ {0, {64,68,88,140}, BUTTON, "No"},
 /*  3 */ {0, {6,15,54,279}, MESSAGE, ""}
};
DIALOG us_yesnodialog = {{50,75,150,369}, "Choice", 3, us_yesnodialogitems};

/*
 * special case for "yes/no" options ("no" is default)
 * Message = 3 (message)
 * No = 1      (button)
 * Yes = 2     (button)
 */
INTSML us_noyesdlog(char *prompt, char *paramstart[])
{
	INTBIG itemHit;
	INTSML oldplease;

	/* display the no/yes dialog box */
	if (DiaInitDialog(&us_noyesdialog) != 0) return(0);

	/* load the message */
	DiaSetText(3, prompt);

	/* loop until done */
	oldplease = el_pleasestop;
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == 1 || itemHit == 2) break;
	}
	el_pleasestop = oldplease;

	if (itemHit == 1) paramstart[0] = "no"; else
		paramstart[0] = "yes";
	DiaDoneDialog();
	return(1);
}

/*
 * special case for "yes/no" options ("yes" is default)
 * Message = 3 (message)
 * Yes = 1     (button)
 * No = 2      (button)
 */
INTSML us_yesnodlog(char *prompt, char *paramstart[])
{
	INTBIG itemHit;
	INTSML oldplease;

	/* display the yes/no dialog box */
	if (DiaInitDialog(&us_yesnodialog) != 0) return(0);

	/* load the message */
	DiaSetText(3, prompt);

	/* loop until done */
	oldplease = el_pleasestop;
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == 1 || itemHit == 2) break;
	}
	el_pleasestop = oldplease;

	if (itemHit == 1) paramstart[0] = "yes"; else paramstart[0] = "no";
	DiaDoneDialog();
	return(1);
}

/****************************** COLOR OPTIONS DIALOG ******************************/

#define	DEGTORAD    (EPI/180.0)
#define INTENSITYDIVISIONS 20
#define WHEELRADIUS        96
#define WHEELEDGE          (WHEELRADIUS + WHEELRADIUS/16)

/* Color Mixing */
DIALOGITEM us_colormixdialogitems[] =
{
 /*  1 */ {0, {212,400,236,468}, BUTTON, "OK"},
 /*  2 */ {0, {212,320,236,388}, BUTTON, "Cancel"},
 /*  3 */ {0, {252,8,268,24}, USERDRAWN, ""},
 /*  4 */ {0, {252,36,268,472}, RADIO, "Entry 1"},
 /*  5 */ {0, {272,8,288,24}, USERDRAWN, ""},
 /*  6 */ {0, {272,36,288,472}, RADIO, "Entry 2"},
 /*  7 */ {0, {292,8,308,24}, USERDRAWN, ""},
 /*  8 */ {0, {292,36,308,472}, RADIO, "Entry 3"},
 /*  9 */ {0, {312,8,328,24}, USERDRAWN, ""},
 /* 10 */ {0, {312,36,328,472}, RADIO, "Entry 4"},
 /* 11 */ {0, {332,8,348,24}, USERDRAWN, ""},
 /* 12 */ {0, {332,36,348,472}, RADIO, "Entry 5"},
 /* 13 */ {0, {352,8,368,24}, USERDRAWN, ""},
 /* 14 */ {0, {352,36,368,472}, RADIO, "Entry 6"},
 /* 15 */ {0, {372,8,388,24}, USERDRAWN, ""},
 /* 16 */ {0, {372,36,388,472}, RADIO, "Entry 7"},
 /* 17 */ {0, {392,8,408,24}, USERDRAWN, ""},
 /* 18 */ {0, {392,36,408,472}, RADIO, "Entry 8"},
 /* 19 */ {0, {412,8,428,24}, USERDRAWN, ""},
 /* 20 */ {0, {412,36,428,472}, RADIO, "Entry 9"},
 /* 21 */ {0, {432,8,448,24}, USERDRAWN, ""},
 /* 22 */ {0, {432,36,448,472}, RADIO, "Entry 10"},
 /* 23 */ {0, {452,8,468,24}, USERDRAWN, ""},
 /* 24 */ {0, {452,36,468,472}, RADIO, "Entry 11"},
 /* 25 */ {0, {472,8,488,24}, USERDRAWN, ""},
 /* 26 */ {0, {472,36,488,472}, RADIO, "Entry 12"},
 /* 27 */ {0, {492,8,508,24}, USERDRAWN, ""},
 /* 28 */ {0, {492,36,508,472}, RADIO, "Entry 13"},
 /* 29 */ {0, {512,8,528,24}, USERDRAWN, ""},
 /* 30 */ {0, {512,36,528,472}, RADIO, "Entry 14"},
 /* 31 */ {0, {532,8,548,24}, USERDRAWN, ""},
 /* 32 */ {0, {532,36,548,472}, RADIO, "Entry 15"},
 /* 33 */ {0, {552,8,568,24}, USERDRAWN, ""},
 /* 34 */ {0, {552,36,568,472}, RADIO, "Entry 16"},
 /* 35 */ {0, {32,8,244,220}, USERDRAWN, ""},
 /* 36 */ {0, {32,228,244,308}, USERDRAWN, ""},
 /* 37 */ {0, {8,16,24,144}, MESSAGE, "Hue/Saturation:"},
 /* 38 */ {0, {8,228,24,308}, MESSAGE, "Intensity:"},
 /* 39 */ {0, {28,356,44,472}, RADIO, "Primaries"},
 /* 40 */ {0, {48,356,64,472}, RADIO, "Specials"},
 /* 41 */ {0, {68,356,84,472}, RADIO, "Layer 1"},
 /* 42 */ {0, {88,356,104,472}, RADIO, "Layer 2"},
 /* 43 */ {0, {108,356,124,472}, RADIO, "Layer 3"},
 /* 44 */ {0, {128,356,144,472}, RADIO, "Layer 4"},
 /* 45 */ {0, {148,356,164,472}, RADIO, "Layer 5"},
 /* 46 */ {0, {4,356,20,436}, MESSAGE, "Palettes:"},
 /* 47 */ {0, {176,316,200,472}, BUTTON, "Compute from Primaries"}
};
DIALOG us_colormixdialog = {{75,75,652,556}, "Color Mixing", 47, us_colormixdialogitems};

INTBIG us_colormixredmap[256], us_colormixgreenmap[256], us_colormixbluemap[256];
INTBIG us_colormixindex[16];
INTBIG us_colormixcurindex;
float us_colormixtheta, us_colormixr, us_colormixinten;
char *us_colormixnames[5];

void us_colormixreload(void);
char *us_colormixentryname(char *overlayernames[], INTBIG ind);
void us_colormixbuildindex(INTBIG layer1, INTBIG layer2, INTBIG layer3, INTBIG layer4, INTBIG layer5);
void us_colormixdrawpalette(RECTAREA *ra);
void us_colormixtogglemarker(void);
void us_colormixdrawsquare(INTBIG sindex);
void us_colormixmergecolor(INTBIG r1, INTBIG g1, INTBIG b1, INTBIG r2, INTBIG g2, INTBIG b2,
	INTBIG *ro, INTBIG *go, INTBIG *bo);
void us_colormixsmoothcolors(void);

/*
 * special case for the "Color Mixing" dialog
 * Color patches for palettes  = 3,5,7,9,11,13,15,17,19,21,23,25,27,29,31,33 (user)
 * Buttons for color palette   = 4,6,8,10,12,14,16,18,20,22,24,26,28,30,32,34 (radio)
 * Hue/Saturation wheel        = 35 (user)
 * Intensity slider            = 36 (user)
 * Primaries palette selection = 39 (radio)
 * Specials palette selection  = 40 (radio)
 * Layer 1 palette selection   = 41 (radio)
 * Layer 2 palette selection   = 42 (radio)
 * Layer 3 palette selection   = 43 (radio)
 * Layer 4 palette selection   = 44 (radio)
 * Layer 5 palette selection   = 45 (radio)
 * Compute from primaries      = 47 (button)
 */
INTSML us_colormixdlog(char *overlayernames[])
{
	INTBIG itemHit, i;
	INTSML x, y;
	INTBIG r, g, b;
	REGISTER INTBIG cx, cy;
	REGISTER VARIABLE *varred, *vargreen, *varblue;
	RECTAREA ra;

	varred = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_green);
	varblue = getvalkey((INTBIG)us_aid, VAID, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred == NOVARIABLE || vargreen == NOVARIABLE || varblue == NOVARIABLE) return(0);
	for(i=0; i<256; i++)
	{
		us_colormixredmap[i] = ((INTBIG *)varred->addr)[i];
		us_colormixgreenmap[i] = ((INTBIG *)vargreen->addr)[i];
		us_colormixbluemap[i] = ((INTBIG *)varblue->addr)[i];
	}

	/* display the color mixing dialog box */
	us_colormixdialogitems[40].msg = overlayernames[0];
	us_colormixdialogitems[41].msg = overlayernames[1];
	us_colormixdialogitems[42].msg = overlayernames[2];
	us_colormixdialogitems[43].msg = overlayernames[3];
	us_colormixdialogitems[44].msg = overlayernames[4];
	if (DiaInitDialog(&us_colormixdialog) != 0) return(0);
	for(i=0; i<5; i++) us_colormixnames[i] = overlayernames[i];
	DiaSetControl(39, 1);
	DiaSetControl(4, 1);
	us_colormixdrawpalette(&ra);
	DiaRedispRoutine(35, us_colormixdrawpalette);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit >= 3 && itemHit <= 34)
		{
			/* hit on one of the 16 colors */
			us_colormixtogglemarker();
			for(i=4; i<=34; i += 2) DiaSetControl(i, 0);
			i = (itemHit + 1) / 2 * 2;
			DiaSetControl(i, 1);
			us_colormixtogglemarker();
			continue;
		}
		if (itemHit >= 39 && itemHit <= 45)
		{
			/* hit on one of the 7 palette sets */
			us_colormixtogglemarker();
			for(i=39; i<=45; i++) DiaSetControl(i, 0);
			DiaSetControl(itemHit, 1);
			us_colormixreload();
			us_colormixtogglemarker();
			continue;
		}
		if (itemHit == 35)
		{
			/* hit in the hue/saturation wheel */
			DiaGetMouse(&x, &y);
			DiaItemRect(35, &ra);
			cx = (ra.left + ra.right) / 2;
			cy = (ra.top + ra.bottom) / 2;
			us_colormixtogglemarker();
			us_colormixtheta = (float)((figureangle(cx, cy, x, y) + 5) / 10);
			if (us_colormixtheta < 0.0) us_colormixtheta += 360.0;
			us_colormixr = (float)sqrt((float)((y-cy)*(y-cy) + (x-cx)*(x-cx)));
			if (us_colormixr > WHEELEDGE) us_colormixr = WHEELEDGE;
			us_hsvtorgb(us_colormixtheta / 360.0f, us_colormixr / WHEELEDGE, us_colormixinten, &r,
				&g, &b);
			us_colormixredmap[us_colormixindex[us_colormixcurindex]] = r;
			us_colormixgreenmap[us_colormixindex[us_colormixcurindex]] = g;
			us_colormixbluemap[us_colormixindex[us_colormixcurindex]] = b;
			us_colormixtogglemarker();
			us_colormixdrawsquare(us_colormixcurindex);
			continue;
		}
		if (itemHit == 36)
		{
			/* hit in the intensity slider */
			DiaGetMouse(&x, &y);
			DiaItemRect(36, &ra);
			cx = (ra.left + ra.right) / 2;
			cy = (ra.top + ra.bottom) / 2;
			us_colormixtogglemarker();
			us_colormixinten = (float)(ra.bottom - y) / (float)(ra.bottom - ra.top);
			us_hsvtorgb(us_colormixtheta / 360.0f, us_colormixr / WHEELEDGE, us_colormixinten, &r,
				&g, &b);
			us_colormixredmap[us_colormixindex[us_colormixcurindex]] = r;
			us_colormixgreenmap[us_colormixindex[us_colormixcurindex]] = g;
			us_colormixbluemap[us_colormixindex[us_colormixcurindex]] = b;
			us_colormixtogglemarker();
			us_colormixdrawsquare(us_colormixcurindex);
			continue;
		}
		if (itemHit == 47)
		{
			/* recompute colors based on layer depth of primaries */
			us_colormixsmoothcolors();
			us_colormixreload();
			us_colormixtogglemarker();
			continue;
		}
	}

	if (itemHit != CANCEL)
	{
		/* load the color map with the new values */
		startobjectchange((INTBIG)us_aid, VAID);
		(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_red, (INTBIG)us_colormixredmap,
			VINTEGER|VISARRAY|(256<<VLENGTHSH));
		(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_green, (INTBIG)us_colormixgreenmap,
			VINTEGER|VISARRAY|(256<<VLENGTHSH));
		(void)setvalkey((INTBIG)us_aid, VAID, us_colormap_blue, (INTBIG)us_colormixbluemap,
			VINTEGER|VISARRAY|(256<<VLENGTHSH));
		endobjectchange((INTBIG)us_aid, VAID);
	}
	DiaDoneDialog();
	return(0);
}

void us_colormixsmoothcolors(void)
{
	INTBIG funct[5], height[5], bit[5], red[5], green[5], blue[5], fun,
		r, g, b, gotcol, i, j;
	char *lname[5], *name;

	/* get information about the current transparent layers */
	for(i=0; i<5; i++) funct[i] = -1;
	for(i=0; i<el_curtech->layercount; i++)
	{
		fun = layerfunction(el_curtech, (INTSML)i);
		name = layername(el_curtech, (INTSML)i);
		if ((fun&LFTRANS1) != 0 && funct[0] < 0) { funct[0] = fun;  lname[0] = name; }
		if ((fun&LFTRANS2) != 0 && funct[1] < 0) { funct[1] = fun;  lname[1] = name; }
		if ((fun&LFTRANS3) != 0 && funct[2] < 0) { funct[2] = fun;  lname[2] = name; }
		if ((fun&LFTRANS4) != 0 && funct[3] < 0) { funct[3] = fun;  lname[3] = name; }
		if ((fun&LFTRANS5) != 0 && funct[4] < 0) { funct[4] = fun;  lname[4] = name; }
	}
	bit[0] = LAYERT1;
	bit[1] = LAYERT2;
	bit[2] = LAYERT3;
	bit[3] = LAYERT4;
	bit[4] = LAYERT5;
	for(i=0; i<5; i++)
		height[i] = layerfunctionheight(funct[i]);

	/* sort the layers by height */
	j = 0;
	while (j == 0)
	{
		j = 1;
		for(i=1; i<5; i++)
		{
			if (height[i] <= height[i-1]) continue;
			fun = height[i];  height[i] = height[i-1];  height[i-1] = fun;
			fun = funct[i];   funct[i] = funct[i-1];    funct[i-1] = fun;
			fun = bit[i];     bit[i] = bit[i-1];        bit[i-1] = fun;
			j = 0;
		}
	}
	for(i=0; i<5; i++)
	{
		red[i] = us_colormixredmap[bit[i]];
		green[i] = us_colormixgreenmap[bit[i]];
		blue[i] = us_colormixbluemap[bit[i]];
	}

	/* now reconstruct the colors */
	for(i=0; i<256; i++)
	{
		if ((i&(LAYERH|LAYEROE|LAYERG)) != 0) continue;
		gotcol = 0;
		for(j=0; j<5; j++)
		{
			if ((i&bit[j]) == 0) continue;
			if (gotcol == 0)
			{
				r = red[j];   g = green[j];   b = blue[j];
				gotcol = 1;
			} else
			{
				us_colormixmergecolor(r, g, b, red[j], green[j], blue[j], &r, &g, &b);
			}
		}
		if (gotcol != 0)
		{
			us_colormixredmap[i] = r;
			us_colormixgreenmap[i] = g;
			us_colormixbluemap[i] = b;
		}
	}

	/*
	 * uncomment this code to generate the file "bettercolors.txt" with the colormap
	 * in a form that can be inserted into the technology source code
	 */
#if 0
	{
		FILE *io;
		INTBIG k;

		io = fopen("bettercolors.txt", "w");
		if (io == 0) return;
		fprintf(io, "static TECH_COLORMAP %s_colmap[32] =\n", el_curtech->techname);
		fprintf(io, "{                  /*     %s %s %s %s %s */\n", lname[4],
			lname[3], lname[2], lname[1], lname[0]);
		j = 0;
		for(i=0; i<256; i++)
		{
			if ((i&(LAYERH|LAYEROE|LAYERG)) != 0) continue;
			fprintf(io, "\t{%3d,%3d,%3d}", us_colormixredmap[i],
				us_colormixgreenmap[i], us_colormixbluemap[i]);
			if (j != 31) fprintf(io, ","); else fprintf(io, " ");
			fprintf(io, " /* %2d: ", j);
			if ((i&LAYERT5) != 0) fprintf(io, "%s ", lname[4]); else
				for(k=0; k<=(INTBIG)strlen(lname[4]); k++) fprintf(io, " ");
			if ((i&LAYERT4) != 0) fprintf(io, "%s ", lname[3]); else
				for(k=0; k<=(INTBIG)strlen(lname[3]); k++) fprintf(io, " ");
			if ((i&LAYERT3) != 0) fprintf(io, "%s ", lname[2]); else
				for(k=0; k<=(INTBIG)strlen(lname[2]); k++) fprintf(io, " ");
			if ((i&LAYERT2) != 0) fprintf(io, "%s ", lname[1]); else
				for(k=0; k<=(INTBIG)strlen(lname[1]); k++) fprintf(io, " ");
			if ((i&LAYERT1) != 0) fprintf(io, "%s ", lname[0]); else
				for(k=0; k<=(INTBIG)strlen(lname[0]); k++) fprintf(io, " ");
			fprintf(io, "*/\n");
			j++;
		}
		fprintf(io, "};\n");
		fclose(io);
	}
#endif
}

void us_colormixmergecolor(INTBIG r1, INTBIG g1, INTBIG b1, INTBIG r2, INTBIG g2, INTBIG b2,
	INTBIG *ro, INTBIG *go, INTBIG *bo)
{
	float f1[3], f2[3], f3[3];

	f1[0] = r1 / 255.0f;
	f1[1] = g1 / 255.0f;
	f1[2] = b1 / 255.0f;
	vectornormalize3d(f1);
	vectormultiply3d(f1, 1.0f, f1);

	f2[0] = r2 / 255.0f;
	f2[1] = g2 / 255.0f;
	f2[2] = b2 / 255.0f;
	vectornormalize3d(f2);
	vectormultiply3d(f2, 0.5f, f2);

	vectoradd3d(f1, f2, f3);
	vectornormalize3d(f3);

	*ro = (INTBIG)(f3[0] * 255.0);
	*go = (INTBIG)(f3[1] * 255.0);
	*bo = (INTBIG)(f3[2] * 255.0);
}

/* Routine to draw the hue/saturation wheel and the intensity slider */
void us_colormixdrawpalette(RECTAREA *ra)
{
	REGISTER INTBIG theta, inc, rr, cx, cy, i, col, spacing;
	INTBIG r, g, b, x[4], y[4];
	RECTAREA rect, stripe;
	float intermed, intertwo;

	/* draw hue/saturation wheel */
	DiaItemRect(35, &rect);
	cx = (rect.left + rect.right) / 2;
	cy = (rect.top + rect.bottom) / 2;
	for(theta=0; theta<360; theta+=60)
	{
		for(inc=0; inc<60; inc+=20)
			for(rr = 0; rr < WHEELRADIUS/2; rr += WHEELRADIUS/8)
		{
			intermed = (float) theta + inc;
			intertwo = (float) rr + WHEELRADIUS/8;
			x[0] = (INTBIG)(((float)rr) * cos(intermed*DEGTORAD)) + cx;
			y[0] = (INTBIG)(((float)rr) * sin(intermed*DEGTORAD)) + cy;
			x[1] = (INTBIG)(intertwo * cos(intermed*DEGTORAD)) + cx;
			y[1] = (INTBIG)(intertwo * sin(intermed*DEGTORAD)) + cy;
			intermed += 21.0;
			x[2] = (INTBIG)(intertwo * cos(intermed*DEGTORAD)) + cx;
			y[2] = (INTBIG)(intertwo * sin(intermed*DEGTORAD)) + cy;
			x[3] = (INTBIG)(((float)rr) * cos(intermed*DEGTORAD)) + cx;
			y[3] = (INTBIG)(((float)rr) * sin(intermed*DEGTORAD)) + cy;
			us_hsvtorgb(((float)(theta+inc+10)) / 360.0f,
				((float)rr+WHEELRADIUS/16) / WHEELRADIUS, 1.0f, &r, &g, &b);
			DiaFillPoly(35, x, y, 4, r, g, b);
		}
		for(inc=0; inc<60; inc+=10)
			for(rr = WHEELRADIUS/8; rr < WHEELRADIUS; rr += WHEELRADIUS/8)
		{
			intermed = (float) theta + inc;
			intertwo = (float) rr + WHEELRADIUS/8;
			x[0] = (INTBIG)(((float)rr)* cos(intermed*DEGTORAD)) + cx;
			y[0] = (INTBIG)(((float)rr)* sin(intermed*DEGTORAD)) + cy;
			x[1] = (INTBIG)(intertwo * cos(intermed*DEGTORAD)) + cx;
			y[1] = (INTBIG)(intertwo * sin(intermed*DEGTORAD)) + cy;
			intermed += 12.0;
			x[2] = (INTBIG)(intertwo * cos(intermed*DEGTORAD)) + cx;
			y[2] = (INTBIG)(intertwo * sin(intermed*DEGTORAD)) + cy;
			x[3] = (INTBIG)(((float)rr) * cos(intermed*DEGTORAD)) + cx;
			y[3] = (INTBIG)(((float)rr) * sin(intermed*DEGTORAD)) + cy;
			us_hsvtorgb(((float)(theta+inc+5)) / 360.0f,
				((float)rr+WHEELRADIUS/16) / WHEELRADIUS, 1.0f, &r, &g, &b);
			DiaFillPoly(35, x, y, 4, r, g, b);
		}
	}

	/* draw the intensity slider */
	DiaItemRect(36, &rect);
	rect.left--;   rect.right++;
	rect.top--;    rect.bottom++;
	DiaFrameRect(36, &rect);
	rect.left++;   rect.right--;
	rect.top++;    rect.bottom--;
	spacing = (rect.bottom - rect.top) / INTENSITYDIVISIONS;
	for(i=0; i<INTENSITYDIVISIONS; i++)
	{
		stripe.left = rect.left;
		stripe.right = rect.right;
		stripe.top = (INTSML)(rect.bottom - (i+1)*spacing);
		stripe.bottom = (INTSML)(rect.bottom - i*spacing);
		col = i * 255 / INTENSITYDIVISIONS;
		DiaDrawRect(36, &stripe, col, col, col);
	}

	/* draw the current color marker */
	us_colormixtogglemarker();
	us_colormixreload();
}

/*
 * Routine to draw an "x" on the hue/saturation wheel and the intensity slider
 * to indicate the current color.
 */
void us_colormixtogglemarker(void)
{
	REGISTER INTBIG r, g, b, cx, cy, x, y;
	RECTAREA rect;

	/* get the current color */
	for(us_colormixcurindex=0; us_colormixcurindex<16; us_colormixcurindex++)
		if (DiaGetControl(us_colormixcurindex*2+4) != 0) break;
	r = us_colormixredmap[us_colormixindex[us_colormixcurindex]];
	g = us_colormixgreenmap[us_colormixindex[us_colormixcurindex]];
	b = us_colormixbluemap[us_colormixindex[us_colormixcurindex]];
	us_rgbtohsv((INTSML)r, (INTSML)g, (INTSML)b, &us_colormixtheta, &us_colormixr, &us_colormixinten);
	us_colormixtheta *= 360.0;
	us_colormixr *= WHEELEDGE;

	/* show the position in the hue/saturation area */
	DiaItemRect(35, &rect);
	cx = (rect.left + rect.right) / 2;
	cy = (rect.top + rect.bottom) / 2;
	x = (INTBIG)(us_colormixr * cos(us_colormixtheta * DEGTORAD)) + cx;
	y = (INTBIG)(us_colormixr * sin(us_colormixtheta * DEGTORAD)) + cy;
	DiaDrawLine(35, x-5, y-5, x+5, y+5, DLMODEINVERT);
	DiaDrawLine(35, x-5, y+5, x+5, y-5, DLMODEINVERT);

	/* show the position in the intensity area */
	DiaItemRect(36, &rect);
	cx = (rect.left + rect.right) / 2;
	y = rect.bottom - (INTBIG)(us_colormixinten * (float)(rect.bottom - rect.top));
	DiaDrawLine(36, cx-5, y-5, cx+5, y+5, DLMODEINVERT);
	DiaDrawLine(36, cx-5, y+5, cx+5, y-5, DLMODEINVERT);
}

/*
 * Routine to reload the 16 color entries according to the selected palette set.
 */
void us_colormixreload(void)
{
	INTBIG i;

	if (DiaGetControl(39) != 0)
	{
		us_colormixindex[0]  = ALLOFF;
		us_colormixindex[1]  = LAYERT1;
		us_colormixindex[2]  = LAYERT2;
		us_colormixindex[3]  = LAYERT3;
		us_colormixindex[4]  = LAYERT4;
		us_colormixindex[5]  = LAYERT5;
		us_colormixindex[6]  = LAYERT1 | LAYERT2;
		us_colormixindex[7]  = LAYERT1 | LAYERT3;
		us_colormixindex[8]  = LAYERT1 | LAYERT4;
		us_colormixindex[9]  = LAYERT1 | LAYERT5;
		us_colormixindex[10] = LAYERT2 | LAYERT3;
		us_colormixindex[11] = LAYERT2 | LAYERT4;
		us_colormixindex[12] = LAYERT2 | LAYERT5;
		us_colormixindex[13] = LAYERT3 | LAYERT4;
		us_colormixindex[14] = LAYERT3 | LAYERT5;
		us_colormixindex[15] = LAYERT4 | LAYERT5;
	}
	if (DiaGetControl(40) != 0)
	{
		us_colormixindex[0]  = ALLOFF;
		us_colormixindex[1]  = GRID;
		us_colormixindex[2]  = HIGHLIT;
		us_colormixindex[3]  = el_colfacettxt;
		us_colormixindex[4]  = el_colfacet;
		us_colormixindex[5]  = el_colwinbor;
		us_colormixindex[6]  = el_colhwinbor;
		us_colormixindex[7]  = el_colmenbor;
		us_colormixindex[8]  = el_colhmenbor;
		us_colormixindex[9]  = el_colmentxt;
		us_colormixindex[10] = el_colmengly;
		us_colormixindex[11] = el_colcursor;
		us_colormixindex[12] = 0354;
		us_colormixindex[13] = 0364;
		us_colormixindex[14] = 0374;
		us_colormixindex[15] = ALLOFF;
	}
	if (DiaGetControl(41) != 0)
		us_colormixbuildindex(LAYERT1, LAYERT2, LAYERT3, LAYERT4, LAYERT5);
	if (DiaGetControl(42) != 0)
		us_colormixbuildindex(LAYERT2, LAYERT1, LAYERT3, LAYERT4, LAYERT5);
	if (DiaGetControl(43) != 0)
		us_colormixbuildindex(LAYERT3, LAYERT1, LAYERT2, LAYERT4, LAYERT5);
	if (DiaGetControl(44) != 0)
		us_colormixbuildindex(LAYERT4, LAYERT1, LAYERT2, LAYERT3, LAYERT5);
	if (DiaGetControl(45) != 0)
		us_colormixbuildindex(LAYERT5, LAYERT1, LAYERT2, LAYERT3, LAYERT4);
	for(i=0; i<16; i++)
	{
		DiaSetText(i*2+4, us_colormixentryname(us_colormixnames, us_colormixindex[i]));
		us_colormixdrawsquare(i);
	}
}

/*
 * Routine to build the entries desired given the five layers "layer1-5".
 */
void us_colormixbuildindex(INTBIG layer1, INTBIG layer2, INTBIG layer3, INTBIG layer4, INTBIG layer5)
{
	us_colormixindex[0]  = layer1;
	us_colormixindex[1]  = layer1 | layer2;
	us_colormixindex[2]  = layer1 |          layer3;
	us_colormixindex[3]  = layer1 | layer2 | layer3;
	us_colormixindex[4]  = layer1 |                   layer4;
	us_colormixindex[5]  = layer1 | layer2 |          layer4;
	us_colormixindex[6]  = layer1 |          layer3 | layer4;
	us_colormixindex[7]  = layer1 | layer2 | layer3 | layer4;
	us_colormixindex[8]  = layer1 |                            layer5;
	us_colormixindex[9]  = layer1 | layer2 |                   layer5;
	us_colormixindex[10] = layer1 |          layer3 |          layer5;
	us_colormixindex[11] = layer1 | layer2 | layer3 |          layer5;
	us_colormixindex[12] = layer1 |                   layer4 | layer5;
	us_colormixindex[13] = layer1 | layer2 |          layer4 | layer5;
	us_colormixindex[14] = layer1 |          layer3 | layer4 | layer5;
	us_colormixindex[15] = layer1 | layer2 | layer3 | layer4 | layer5;
}

/*
 * Routine to redraw color square "sindex" from the current settings.
 */
void us_colormixdrawsquare(INTBIG sindex)
{
	RECTAREA r;

	DiaItemRect(sindex*2+3, &r);
	r.left--;   r.right++;
	r.top--;    r.bottom++;
	DiaFrameRect(sindex*2+3, &r);
	r.left++;   r.right--;
	r.top++;    r.bottom--;
	DiaDrawRect(sindex*2+3, &r, us_colormixredmap[us_colormixindex[sindex]],
		us_colormixgreenmap[us_colormixindex[sindex]], us_colormixbluemap[us_colormixindex[sindex]]);
}

/*
 * Routine to construct the name of color map entry "ind", given the transparent layer names in "overlayernames".
 */
char *us_colormixentryname(char *overlayernames[], INTBIG ind)
{
	INTBIG gotname;

	if (ind == ALLOFF) return("Background");
	if (ind == GRID) return("Grid");
	if (ind == HIGHLIT) return("Highlight");
	if (ind == 0354) return("Extra 1");
	if (ind == 0364) return("Extra 2");
	if (ind == 0374) return("Extra 3");
	if (ind == el_colfacettxt) return("Facet Name");
	if (ind == el_colfacet) return("Facet Outline");
	if (ind == el_colwinbor) return("Window Border");
	if (ind == el_colhwinbor) return("Current Window Border");
	if (ind == el_colmenbor) return("Component Menu Border");
	if (ind == el_colhmenbor) return("Highlighted Component Menu Border");
	if (ind == el_colmentxt) return("Text in Component Menu");
	if (ind == el_colmengly) return("Glyphs in Component Menu");
	if (ind == el_colcursor) return("Cursor");
	(void)initinfstr();
	gotname = 0;
	if ((ind&LAYERT1) != 0)
	{
		if (gotname != 0) (void)addstringtoinfstr(", ");
		(void)addstringtoinfstr(overlayernames[0]);
		gotname = 1;
	}
	if ((ind&LAYERT2) != 0)
	{
		if (gotname != 0) (void)addstringtoinfstr(", ");
		(void)addstringtoinfstr(overlayernames[1]);
		gotname = 1;
	}
	if ((ind&LAYERT3) != 0)
	{
		if (gotname != 0) (void)addstringtoinfstr(", ");
		(void)addstringtoinfstr(overlayernames[2]);
		gotname = 1;
	}
	if ((ind&LAYERT4) != 0)
	{
		if (gotname != 0) (void)addstringtoinfstr(", ");
		(void)addstringtoinfstr(overlayernames[3]);
		gotname = 1;
	}
	if ((ind&LAYERT5) != 0)
	{
		if (gotname != 0) (void)addstringtoinfstr(", ");
		(void)addstringtoinfstr(overlayernames[4]);
	}
	return(returninfstr());
}

/****************************** COMPONENT MENU DIALOG ******************************/

/* Component Menu */
DIALOGITEM us_menuposdialogitems[] =
{
 /*  1 */ {0, {128,168,152,232}, BUTTON, "OK"},
 /*  2 */ {0, {88,168,112,232}, BUTTON, "Cancel"},
 /*  3 */ {0, {32,200,48,248}, EDITTEXT, ""},
 /*  4 */ {0, {64,8,80,136}, RADIO, "Menu at Top"},
 /*  5 */ {0, {88,8,104,136}, RADIO, "Menu at Bottom"},
 /*  6 */ {0, {112,8,128,136}, RADIO, "Menu on Left"},
 /*  7 */ {0, {136,8,152,136}, RADIO, "Menu on Right"},
 /*  8 */ {0, {8,8,24,197}, MESSAGE, "Number of Entries Across:"},
 /*  9 */ {0, {32,8,48,197}, MESSAGE, "Number of Entries Down:"},
 /* 10 */ {0, {8,200,24,248}, EDITTEXT, ""},
 /* 11 */ {0, {160,8,176,100}, RADIO, "No Menu"}
};
DIALOG us_menuposdialog = {{50,75,235,334}, "Component Menu Configuration", 11, us_menuposdialogitems};

/*
 * special case for the "menu" dialog
 * Menus at top    = 4 (radio button)
 * Menus at bottom = 5 (radio button)
 * Menus at left   = 6 (radio button)
 * Menus at right  = 7 (radio button)
 * No menu         = 11 (radio button)
 * Across label    = 8 (message)
 * Down label      = 9 (message)
 * Across          = 10 (edit text)
 * Down            = 3 (edit text)
 */
INTSML us_menudlog(char *paramstart[])
{
	INTBIG itemHit, large, smallf;
	char amt[10];

	/* display the array dialog box */
	if (DiaInitDialog(&us_menuposdialog) != 0) return(0);
	if ((us_aid->aidstate&MENUON) == 0)
	{
		DiaSetControl(11, 1);
		DiaNoEditControl(3);
		DiaNoEditControl(10);
		DiaDimItem(8);
		DiaDimItem(9);
	} else
	{
		switch (us_menupos)
		{
			case 0: DiaSetControl(4, 1);   break;
			case 1: DiaSetControl(5, 1);   break;
			case 2: DiaSetControl(6, 1);   break;
			case 3: DiaSetControl(7, 1);   break;
		}
		DiaEditControl(3);
		DiaEditControl(10);
		DiaUnDimItem(8);
		DiaUnDimItem(9);
	}
	if (us_menux < us_menuy) { large = us_menuy;   smallf = us_menux; } else
		{ large = us_menux;   smallf = us_menuy; }
	if (us_menupos <= 1)
	{
		(void)sprintf(amt, "%d", large);
		DiaSetText(10, amt);
		(void)sprintf(amt, "%d", smallf);
		DiaSetText(3, amt);
	} else
	{
		(void)sprintf(amt, "%d", smallf);
		DiaSetText(10, amt);
		(void)sprintf(amt, "%d", large);
		DiaSetText(3, amt);
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK && DiaValidEntry(10) && DiaValidEntry(3)) break;
		if (itemHit == 4 || itemHit == 5 || itemHit == 6 || itemHit == 7 || itemHit == 11)
		{
			DiaSetControl(4, 0);
			DiaSetControl(5, 0);
			DiaSetControl(6, 0);
			DiaSetControl(7, 0);
			DiaSetControl(11, 0);
			DiaSetControl(itemHit, 1);
			if (itemHit == 11)
			{
				DiaNoEditControl(3);
				DiaNoEditControl(10);
				DiaDimItem(8);
				DiaDimItem(9);
			} else
			{
				DiaEditControl(3);
				DiaEditControl(10);
				DiaUnDimItem(8);
				DiaUnDimItem(9);
			}
		}
	}

	paramstart[0] = "";
	if (itemHit != CANCEL)
	{
		(void)initinfstr();
		if (DiaGetControl(11) != 0) (void)addstringtoinfstr("off"); else
		{
			if (DiaGetControl(4) != 0) (void)addstringtoinfstr("top");
			if (DiaGetControl(5) != 0) (void)addstringtoinfstr("bottom");
			if (DiaGetControl(6) != 0) (void)addstringtoinfstr("left");
			if (DiaGetControl(7) != 0) (void)addstringtoinfstr("right");
			(void)addstringtoinfstr(" size ");
			(void)addstringtoinfstr(DiaGetText(10));
			(void)addtoinfstr(' ');
			(void)addstringtoinfstr(DiaGetText(3));
		}
		paramstart[0] = returninfstr();
	}
	DiaDoneDialog();
	return(1);
}

/****************************** CREATE TO DIALOG ******************************/

/* Create To */
DIALOGITEM us_createtodialogitems[] =
{
 /*  1 */ {0, {64,96,88,160}, BUTTON, "OK"},
 /*  2 */ {0, {64,8,88,72}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,48,24,142}, EDITTEXT, ""},
 /*  4 */ {0, {8,20,24,42}, MESSAGE, "X:"},
 /*  5 */ {0, {32,48,48,142}, EDITTEXT, ""},
 /*  6 */ {0, {32,20,48,42}, MESSAGE, "Y:"}
};
DIALOG us_createtodialog = {{50,75,147,244}, "Create At Location", 6, us_createtodialogitems};

/*
 * special case for the "Create To" dialog
 * X coordinate     = 3 (edit text)
 * Y coordinate     = 5 (edit text)
 */
INTSML us_createtodlog(char *paramstart[])
{
	INTBIG itemHit;
	INTBIG x, y;

	/* determine default location (use snap point if there is one) */
	if (us_getonesnappoint(&x, &y) == 0) x = y = 0;

	/* get coordinates from dialog */
	if (DiaInitDialog(&us_createtodialog) != 0) return(0);
	DiaSetText(3, latoa(x));
	DiaSetText(5, latoa(y));
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK) break;
	}
	paramstart[0] = us_putintoinfstr(DiaGetText(3));
	paramstart[1] = us_putintoinfstr(DiaGetText(5));
	DiaDoneDialog();
	if (itemHit == CANCEL) return(0);
	return(2);
}

/****************************** CROSS-LIBRARY COPY DIALOG ******************************/

/* Copy between libraries */
DIALOGITEM us_copyfacetdialogitems[] =
{
 /*  1 */ {0, {276,168,300,240}, BUTTON, "Done"},
 /*  2 */ {0, {8,8,24,207}, POPUP, ""},
 /*  3 */ {0, {276,40,300,112}, BUTTON, "<< Copy"},
 /*  4 */ {0, {276,296,300,368}, BUTTON, "Copy >>"},
 /*  5 */ {0, {32,8,268,408}, SCROLL, ""},
 /*  6 */ {0, {8,209,24,408}, POPUP, ""}
};
DIALOG us_copyfacetdialog = {{50,75,359,492}, "Cross-Library Copy", 6, us_copyfacetdialogitems};

INTBIG us_loadlibraryfacets(LIBRARY *lib, LIBRARY *otherlib);

/*
 * special case for the "copyfacet" dialog
 * First library = 2 (popup)
 * << Copy       = 3 (button)
 * Copy >>       = 4 (button)
 * Facet list    = 5 (scroll list)
 * Other library = 6 (popup)
 */
INTSML us_copyfacetdlog(char *prompt, char *paramstart[])
{
	INTBIG itemHit, libcount, i, len, libindex, otherlibindex;
	char *lastfacetname, *param0, *param1;
	REGISTER INTBIG charpos;
	REGISTER LIBRARY *lib, *otherlib, *olib;
	REGISTER char **liblist, *pt;

	/* see how many libraries should be in the popups */
	libcount = 0;
	lib = el_curlib;   otherlib = NOLIBRARY;
	for(olib = el_curlib; olib != NOLIBRARY; olib = olib->nextlibrary)
	{
		if ((olib->userbits&HIDDENLIBRARY) != 0) continue;
		libcount++;
		if (olib != lib && otherlib == NOLIBRARY) otherlib = olib;
	}
	if (libcount < 2)
	{
		ttyputerr("There must be two libraries read in before copying between them");
		return(0);
	}

	/* see if the initial library is specified */
	if (*prompt == '!')
	{
		olib = getlibrary(&prompt[1]);
		if (olib != NOLIBRARY && olib != lib)
			otherlib = olib;
	}

	/* load up list of libraries */
	liblist = (char **)emalloc(libcount * (sizeof (char *)), el_tempcluster);
	if (liblist == 0) return(0);
	for(olib = el_curlib, i=0; olib != NOLIBRARY; olib = olib->nextlibrary)
	{
		if ((olib->userbits&HIDDENLIBRARY) != 0) continue;
		liblist[i] = olib->libname;
		if (olib == lib) libindex = i;
		if (olib == otherlib) otherlibindex = i;
		i++;
	}
	libcount = i;

	/* display the copyfacet dialog box */
	if (DiaInitDialog(&us_copyfacetdialog) != 0)
	{
		efree((char *)liblist);
		return(0);
	}
	DiaSetPopup(2, libcount, liblist);
	DiaSetPopupEntry(2, libindex);
	DiaSetPopup(6, libcount, liblist);
	DiaSetPopupEntry(6, otherlibindex);

	DiaInitTextDialog(5, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, 0,
		SCREPORT|SCSELMOUSE|SCDOUBLEQUIT|SCSMALLFONT|SCFIXEDWIDTH|SCHORIZBAR);
	charpos = us_loadlibraryfacets(lib, otherlib);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK) break;
		if (itemHit == 3)
		{
			/* copy into the current library ("<< Copy") */
			i = DiaGetCurLine(5);
			if (i < 0) continue;
			pt = DiaGetScrollLine(5, i);
			if (pt[charpos] == 0) continue;
			allocstring(&lastfacetname, &pt[charpos], el_tempcluster);

			(void)initinfstr();
			(void)addstringtoinfstr(otherlib->libname);
			(void)addtoinfstr(':');
			(void)addstringtoinfstr(lastfacetname);
			allocstring(&param0, returninfstr(), el_tempcluster);
			paramstart[0] = param0;

			(void)initinfstr();
			(void)addstringtoinfstr(lib->libname);
			(void)addtoinfstr(':');
			for(i=0; lastfacetname[i] != 0; i++)
			{
				if (lastfacetname[i] == '{' || lastfacetname[i] == ';') break;
				(void)addtoinfstr(lastfacetname[i]);
			}
			allocstring(&param1, returninfstr(), el_tempcluster);
			paramstart[1] = param1;

			/* do the copy */
			us_copyfacet(2, paramstart);
			efree(param0);
			efree(param1);

			/* reload the dialog */
			charpos = us_loadlibraryfacets(lib, otherlib);

			/* reselect the last selected line */
			for(i=0; ; i++)
			{
				pt = DiaGetScrollLine(5, i);
				if (pt == 0) break;
				if (strcmp(lastfacetname, &pt[charpos]) != 0) continue;
				DiaSelectLine(5, i);
				break;
			}
			efree(lastfacetname);
			continue;
		}
		if (itemHit == 4)
		{
			/* copy out of the current library ("Copy >>") */
			i = DiaGetCurLine(5);
			if (i < 0) continue;
			pt = DiaGetScrollLine(5, i);
			if (pt[0] == 0) continue;
			(void)initinfstr();
			for(i=0; i<charpos-9; i++)
			{
				if (pt[i] == ' ') break;
				(void)addtoinfstr(pt[i]);
			}
			allocstring(&lastfacetname, returninfstr(), el_tempcluster);

			(void)initinfstr();
			(void)addstringtoinfstr(lib->libname);
			(void)addtoinfstr(':');
			(void)addstringtoinfstr(lastfacetname);
			allocstring(&param0, returninfstr(), el_tempcluster);
			paramstart[0] = param0;

			(void)initinfstr();
			(void)addstringtoinfstr(otherlib->libname);
			(void)addtoinfstr(':');
			for(i=0; lastfacetname[i] != 0; i++)
			{
				if (lastfacetname[i] == '{' || lastfacetname[i] == ';') break;
				(void)addtoinfstr(lastfacetname[i]);
			}
			allocstring(&param1, returninfstr(), el_tempcluster);
			paramstart[1] = param1;

			/* do the copy */
			us_copyfacet(2, paramstart);
			efree(param0);
			efree(param1);

			/* reload the dialog */
			charpos = us_loadlibraryfacets(lib, otherlib);

			/* reselect the last selected line */
			len = strlen(lastfacetname);
			for(i=0; ; i++)
			{
				pt = DiaGetScrollLine(5, i);
				if (pt == 0) break;
				if (strncmp(lastfacetname, pt, len) != 0 || pt[len] != ' ') continue;
				DiaSelectLine(5, i);
				break;
			}
			efree(lastfacetname);
			continue;
		}
		if (itemHit == 2)
		{
			i = DiaGetPopupEntry(2);
			olib = getlibrary(liblist[i]);
			if (olib == NOLIBRARY) continue;
			lib = olib;
			charpos = us_loadlibraryfacets(lib, otherlib);
			continue;
		}
		if (itemHit == 6)
		{
			i = DiaGetPopupEntry(6);
			olib = getlibrary(liblist[i]);
			if (olib == NOLIBRARY) continue;
			otherlib = olib;
			charpos = us_loadlibraryfacets(lib, otherlib);
			continue;
		}
	}

	DiaDoneDialog();
	efree((char *)liblist);
	return(0);
}

INTBIG us_loadlibraryfacets(LIBRARY *lib, LIBRARY *otherlib)
{
	REGISTER NODEPROTO *np, **curnp, **othernp, *curf, *otherf;
	REGISTER INTBIG i, j, sorted, len, widest, curcount, othercount, curpos, otherpos, op;
	REGISTER char *pt, **curnames, **othernames;
	REGISTER LIBRARY *oldlib;

	DiaLoadTextDialog(5, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, 0);

	/* gather a list of facet names in "lib" */
	curcount = 0;
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		curcount++;
	if (curcount > 0)
	{
		curnames = (char **)emalloc(curcount * (sizeof (char *)), el_tempcluster);
		curnp = (NODEPROTO **)emalloc(curcount * (sizeof (NODEPROTO *)), el_tempcluster);
		oldlib = el_curlib;   el_curlib = lib;
		for(i=0, np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto, i++)
		{
			curnp[i] = np;
			(void)allocstring(&curnames[i], describenodeproto(np), el_tempcluster);
		}
		el_curlib = oldlib;
	}

	/* gather a list of facet names in "otherlib" */
	othercount = 0;
	for(np = otherlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		othercount++;
	if (othercount > 0)
	{
		othernames = (char **)emalloc(othercount * (sizeof (char *)), el_tempcluster);
		othernp = (NODEPROTO **)emalloc(othercount * (sizeof (NODEPROTO *)), el_tempcluster);
		oldlib = el_curlib;   el_curlib = otherlib;
		for(i=0, np = otherlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto, i++)
		{
			othernp[i] = np;
			(void)allocstring(&othernames[i], describenodeproto(np), el_tempcluster);
		}
		el_curlib = oldlib;
	}

	/* sort the list of names in the current library */
	sorted = 0;
	while (sorted == 0)
	{
		sorted = 1;
		for(i=1; i<curcount; i++)
		{
			j = namesame(curnames[i-1], curnames[i]);
			if (j < 0) continue;
			pt = curnames[i-1];   curnames[i-1] = curnames[i];   curnames[i] = pt;
			np = curnp[i-1];      curnp[i-1] = curnp[i];         curnp[i] = np;
			sorted = 0;
		}
	}

	/* sort the list of names in the other library */
	sorted = 0;
	while (sorted == 0)
	{
		sorted = 1;
		for(i=1; i<othercount; i++)
		{
			j = namesame(othernames[i-1], othernames[i]);
			if (j < 0) continue;
			pt = othernames[i-1];   othernames[i-1] = othernames[i];   othernames[i] = pt;
			np = othernp[i-1];      othernp[i-1] = othernp[i];         othernp[i] = np;
			sorted = 0;
		}
	}

	/* determine the widest facet name in the current library */
	widest = 0;
	for(i=0; i<curcount; i++)
	{
		len = strlen(curnames[i]);
		if (len > widest) widest = len;
	}
	widest += 9;

	/* put out the parallel list of facets in the two libraries */
	curpos = otherpos = 0;
	for(;;)
	{
		if (curpos >= curcount && otherpos >= othercount) break;
		if (curpos >= curcount) op = 2; else
			if (otherpos >= othercount) op = 1; else
		{
			curf = curnp[curpos];
			otherf = othernp[otherpos];
			j = namesame(curnames[curpos], othernames[otherpos]);
			if (j < 0) op = 1; else
				if (j > 0) op = 2; else
					op = 3;
		}

		(void)initinfstr();
		if (op == 1 || op == 3)
		{
			pt = curnames[curpos++];
			(void)addstringtoinfstr(pt);
		} else pt = "";

		if (op == 3)
		{
			for(i = strlen(pt); i<widest-9; i++) (void)addtoinfstr(' ');
			if (curf->revisiondate < otherf->revisiondate)
			{
				(void)addstringtoinfstr(" <-OLD   ");
			} else if (curf->revisiondate > otherf->revisiondate)
			{
				(void)addstringtoinfstr("   OLD-> ");
			} else
			{
				(void)addstringtoinfstr(" -EQUAL- ");
			}
		} else
		{
			for(i = strlen(pt); i<widest; i++) (void)addtoinfstr(' ');
		}

		if (op == 2 || op == 3)
		{
			pt = othernames[otherpos++];
			(void)addstringtoinfstr(pt);
		}
		DiaStuffLine(5, returninfstr());
	}
	DiaSelectLine(5, 0);

	/* clean up */
	for(i=0; i<curcount; i++) efree(curnames[i]);
	if (curcount > 0)
	{
		efree((char *)curnames);
		efree((char *)curnp);
	}
	for(i=0; i<othercount; i++) efree(othernames[i]);
	if (othercount > 0)
	{
		efree((char *)othernames);
		efree((char *)othernp);
	}
	return(widest);
}

/****************************** FACET EDIT/CREATE DIALOGS ******************************/

/* Edit facet */
DIALOGITEM us_editfacetdialoglist[] =
{
 /*  1 */ {0, {292,208,316,272}, BUTTON, "OK"},
 /*  2 */ {0, {292,16,316,80}, BUTTON, "Cancel"},
 /*  3 */ {0, {32,8,208,281}, SCROLL, ""},
 /*  4 */ {0, {216,8,232,153}, CHECK, "Show Old Versions"},
 /*  5 */ {0, {264,8,280,231}, CHECK, "Make New Window for Facet"},
 /*  6 */ {0, {292,104,316,187}, BUTTON, "New Facet"},
 /*  7 */ {0, {8,8,24,67}, MESSAGE, "Library:"},
 /*  8 */ {0, {8,72,24,280}, POPUP, "Library"},
 /*  9 */ {0, {240,8,256,231}, CHECK, "Show Facets from Cell-Library"}
};
DIALOG us_editfacetdialog = {{50,75,375,365}, "Edit Facet", 9, us_editfacetdialoglist};

/* New facet */
DIALOGITEM us_newfacetdialogitems[] =
{
 /*  1 */ {0, {56,304,80,368}, BUTTON, "OK"},
 /*  2 */ {0, {56,12,80,76}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,160,24,367}, EDITTEXT, ""},
 /*  4 */ {0, {8,8,24,157}, MESSAGE, "Name of new facet:"},
 /*  5 */ {0, {32,160,48,367}, POPUP, "Unknown"},
 /*  6 */ {0, {32,56,48,149}, MESSAGE, "Facet view:"},
 /*  7 */ {0, {60,84,78,297}, CHECK, "Make new window for facet"}
};
DIALOG us_newfacetdialog = {{350,75,445,455}, "New Facet Creation", 7, us_newfacetdialogitems};

/*
 * special case for the "edit facet" command
 * Facet list                = 3 (message)
 * Show old versions         = 4 (check)
 * Make new window for facet = 5 (check)
 * New facet                 = 6 (button)
 * Library name              = 8 (popup)
 * Show from cell-library    = 9 (check)
 *
 * special case for the "new facet" button
 * Facet name                = 3 (edit text)
 * View list                 = 5 (popup)
 * Make new window for facet = 7 (check)
 */
static NODEPROTO *us_oldfacetprotos;
static INTBIG us_showoldversions;
static INTBIG us_showcellibraryfacets;
static LIBRARY *us_curlib;

INTSML us_oldfacettopoffacets(char**);
char *us_oldfacetnextfacets(void);

INTSML us_oldfacettopoffacets(char **c)
{
	us_oldfacetprotos = us_curlib->firstnodeproto;
	return(1);
}

char *us_oldfacetnextfacets(void)
{
	REGISTER NODEPROTO *thisnp;
	REGISTER LIBRARY *savelibrary;
	REGISTER char *ret;

	while (us_oldfacetprotos != NONODEPROTO)
	{
		thisnp = us_oldfacetprotos;
		us_oldfacetprotos = us_oldfacetprotos->nextnodeproto;
		if (us_showoldversions == 0 && thisnp->newestversion != thisnp) continue;
		if (us_showcellibraryfacets == 0 && (thisnp->userbits&INCELLLIBRARY) != 0)
			continue;
		savelibrary = el_curlib;
		el_curlib = us_curlib;
		ret = describenodeproto(thisnp);
		el_curlib = savelibrary;
		return(ret);
	}
	return(0);
}

INTSML us_editfacetdlog(char *prompt, char *paramstart[])
{
	INTBIG itemHit;
	REGISTER INTBIG viewcount, librarycount, i, libindex, defaultindex;
	REGISTER NODEPROTO *np, *curfacet;
	char **viewlist, **librarylist, *pt, *ptr;
	REGISTER VIEW *v;
	REGISTER GEOM **list;
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *var;
	REGISTER LIBRARY *lib, *savelib;
	static INTBIG makenewwindow = -1;
	static VIEW *lastview = NOVIEW;
	VIEW *thisview;

	/* see what the current view is (if any) */
	thisview = NOVIEW;
	if (el_curwindowpart != NOWINDOWPART)
	{
		if (el_curwindowpart->curnodeproto != NONODEPROTO)
			thisview = el_curwindowpart->curnodeproto->cellview;
	}
	if (thisview == NOVIEW) thisview = lastview;

	/* make a list of view names */
	for(viewcount=0, v=el_views; v!=NOVIEW; v=v->nextview) viewcount++;
	viewlist = (char **)emalloc(viewcount * (sizeof (char *)), el_tempcluster);
	if (viewlist == 0) return(0);
	viewlist[0] = "unknown";   i = 1;
	defaultindex = 0;
	for(v = el_views; v != NOVIEW; v = v->nextview)
	{
		if (namesame(v->viewname, "unknown") != 0)
		{
			if (v == thisview) defaultindex = i;
			viewlist[i++] = v->viewname;
		}
	}

	/* the general case: display the dialog box */
	us_curlib = el_curlib;
	for(pt = prompt; *pt != 0; pt++) if (*pt == '/') break;
	if (*pt == '/')
	{
		*pt++ = 0;
		us_curlib = getlibrary(prompt);
		if (us_curlib == NOLIBRARY) us_curlib = el_curlib;
		prompt = pt;
	}

	us_editfacetdialog.movable = prompt;
	if (us_curlib->firstnodeproto == NONODEPROTO)
	{
		us_editfacetdialoglist[0].type = BUTTON;
		us_editfacetdialoglist[5].type = DEFBUTTON;
	} else
	{
		us_editfacetdialoglist[0].type = DEFBUTTON;
		us_editfacetdialoglist[5].type = BUTTON;
	}
	if (DiaInitDialog(&us_editfacetdialog) != 0) return(0);

	/* make a list of library names */
	for(librarycount=0, lib=el_curlib; lib!=NOLIBRARY; lib=lib->nextlibrary) librarycount++;
	librarylist = (char **)emalloc(librarycount * (sizeof (char *)), el_tempcluster);
	if (librarylist == 0) return(0);
	i = 0;
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
	{
		if ((lib->userbits&HIDDENLIBRARY) != 0) continue;
		if (lib == us_curlib) libindex = i;
		librarylist[i++] = lib->libname;
	}
	DiaSetPopup(8, i, librarylist);
	DiaSetPopupEntry(8, libindex);

	/* show the facets */
	us_showoldversions = 1;
	us_showcellibraryfacets = 1;
	DiaInitTextDialog(3, us_oldfacettopoffacets, us_oldfacetnextfacets, DiaNullDlogDone,
		0, SCSELMOUSE|SCSELKEY|SCDOUBLEQUIT);

	/* find the current node and make it the default */
	np = getcurfacet();
	if (np != NONODEPROTO)
	{
		savelib = el_curlib;
		el_curlib = us_curlib;
		pt = describenodeproto(np);
		el_curlib = savelib;
		if (namesame(np->cell->cellname, "facetstructure") == 0)
		{
			list = us_gethighlighted(OBJNODEINST);
			if (list[0] != NOGEOM && list[1] == NOGEOM)
			{
				if (list[0]->entrytype == OBJNODEINST)
				{
					ni = list[0]->entryaddr.ni;
					var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
					if (var != NOVARIABLE)
					{
						pt = (char *)var->addr;
					}
				}
			}
		}
		for(i=0; ; i++)
		{
			ptr = DiaGetScrollLine(3, i);
			if (*ptr == 0) break;
			if (strcmp(ptr, pt) != 0) continue;
			DiaSelectLine(3, i);
			break;
		}
	}

	/* see if there are any old versions */
	for(np = us_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (np->newestversion != np) break;
	if (np == NONODEPROTO) DiaDimItem(4); else
	{
		DiaUnDimItem(4);
		DiaSetControl(4, us_showoldversions);
	}

	/* see if there are any cell-library facets */
	for(np = us_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if ((np->userbits&INCELLLIBRARY) != 0) break;
	if (np == NONODEPROTO) DiaDimItem(9); else
	{
		DiaUnDimItem(9);
		DiaSetControl(9, us_showcellibraryfacets);
	}

	/* if the current window has a facet in it and multiple windows are supported, offer new one */
	curfacet = getcurfacet();
	if (makenewwindow < 0)
	{
		makenewwindow = 0;
		if (curfacet != NONODEPROTO && us_graphicshas(CANUSEFRAMES)) makenewwindow = 1;
	}
	if (curfacet == NONODEPROTO) makenewwindow = 0;
	DiaSetControl(5, makenewwindow);

	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL)
		{
			(void)initinfstr();
			if (us_curlib != el_curlib)
			{
				(void)addstringtoinfstr(us_curlib->libname);
				(void)addtoinfstr(':');
			}
			(void)addstringtoinfstr(DiaGetScrollLine(3, DiaGetCurLine(3)));
			paramstart[0] = returninfstr();
			break;
		}
		if (itemHit == 5)
		{
			DiaSetControl(5, 1 - DiaGetControl(5));
			continue;
		}
		if (itemHit == 8)
		{
			i = DiaGetPopupEntry(8);
			us_curlib = getlibrary(librarylist[i]);
			DiaLoadTextDialog(3, us_oldfacettopoffacets, us_oldfacetnextfacets, DiaNullDlogDone, 0);

			/* see if there are any old versions */
			for(np = us_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				if (np->newestversion != np) break;
			if (np == NONODEPROTO) DiaDimItem(4); else
			{
				DiaUnDimItem(4);
				DiaSetControl(4, us_showoldversions);
			}

			/* see if there are any cell-library facets */
			for(np = us_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				if ((np->userbits&INCELLLIBRARY) != 0) break;
			if (np == NONODEPROTO) DiaDimItem(9); else
			{
				DiaUnDimItem(9);
				DiaSetControl(9, us_showcellibraryfacets);
			}
			continue;
		}
		if (itemHit == 4)
		{
			us_showoldversions = 1 - DiaGetControl(4);
			DiaSetControl(4, us_showoldversions);
			i = DiaGetCurLine(3);
			if (i >= 0) pt = DiaGetScrollLine(3, i); else pt = "";
			DiaLoadTextDialog(3, us_oldfacettopoffacets, us_oldfacetnextfacets, DiaNullDlogDone, 0);
			if (*pt != 0)
			{
				for(i=0; ; i++)
				{
					ptr = DiaGetScrollLine(3, i);
					if (*ptr == 0) break;
					if (strcmp(ptr, pt) != 0) continue;
					DiaSelectLine(3, i);
					break;
				}
			}
			continue;
		}
		if (itemHit == 9)
		{
			us_showcellibraryfacets = 1 - DiaGetControl(9);
			DiaSetControl(9, us_showcellibraryfacets);
			i = DiaGetCurLine(3);
			if (i >= 0) pt = DiaGetScrollLine(3, i); else pt = "";
			DiaLoadTextDialog(3, us_oldfacettopoffacets, us_oldfacetnextfacets, DiaNullDlogDone, 0);
			if (*pt != 0)
			{
				for(i=0; ; i++)
				{
					ptr = DiaGetScrollLine(3, i);
					if (*ptr == 0) break;
					if (strcmp(ptr, pt) != 0) continue;
					DiaSelectLine(3, i);
					break;
				}
			}
			continue;
		}
		if (itemHit == 6)
		{
			/* display the new facet dialog box */
			makenewwindow = DiaGetControl(5);
			if (DiaInitDialog(&us_newfacetdialog) != 0) continue;
			DiaSetPopup(5, viewcount, viewlist);
			DiaSetControl(7, makenewwindow);
			DiaSetPopupEntry(5, defaultindex);

			/* loop until done */
			for(;;)
			{
				itemHit = DiaNextHit();
				if (itemHit == CANCEL) break;
				if (itemHit == 7)
				{
					DiaSetControl(7, 1 - DiaGetControl(7));
					continue;
				}
				if (itemHit == OK && DiaValidEntry(3)) break;
			}

			paramstart[0] = "";
			if (itemHit != CANCEL)
			{
				i = DiaGetPopupEntry(5);
				(void)initinfstr();
				if (us_curlib != el_curlib)
				{
					(void)addstringtoinfstr(us_curlib->libname);
					(void)addtoinfstr(':');
				}
				(void)addstringtoinfstr(DiaGetText(3));
				for(v = el_views; v != NOVIEW; v = v->nextview)
					if (namesame(viewlist[i], v->viewname) == 0)
				{
					if (*v->sviewname == 0) break;
					(void)addtoinfstr('{');
					(void)addstringtoinfstr(v->sviewname);
					(void)addtoinfstr('}');
					lastview = v;
					break;
				}
				paramstart[0] = returninfstr();
			}
			makenewwindow = DiaGetControl(7);
			DiaDoneDialog();
			if (itemHit == CANCEL) continue;
			DiaSetControl(5, makenewwindow);
			break;
		}
	}
	makenewwindow = DiaGetControl(5);
	DiaDoneDialog();
	efree((char *)librarylist);
	efree((char *)viewlist);
	if (itemHit != CANCEL)
	{
		if (makenewwindow != 0)
		{
			paramstart[1] = "new-window";
			us_editfacet(2, paramstart);
		} else
		{
			us_editfacet(1, paramstart);
		}
	}
	return(0);
}

/****************************** FACET LIST DIALOGS ******************************/

/* Facet List */
DIALOGITEM us_facetlistdialoglist[] =
{
 /*  1 */ {0, {232,288,256,352}, BUTTON, "OK"},
 /*  2 */ {0, {232,212,256,276}, BUTTON, "Cancel"},
 /*  3 */ {0, {4,8,219,356}, SCROLL, ""},
 /*  4 */ {0, {224,8,242,153}, CHECK, "Show Old Versions"},
 /*  5 */ {0, {248,8,266,205}, CHECK, "Show Facets from Cell-Library"}
};
DIALOG us_facetlistdialog = {{50,75,325,440}, "Facet List", 5, us_facetlistdialoglist};

/*
 * special case for the "Delete Facet", "New Facet Instance", "List Facet Usage", and
 * "plot simulation in facet window" command
 * Facet list               = 3 (message)
 * Show old versions        = 4 (check)
 * Show cell-library facets = 5 (check)
 */
INTSML us_facetlist(char *prompt, char *paramstart[], INTSML curinstance)
{
	INTBIG itemHit;
	REGISTER char *curnode, *line;
	REGISTER INTBIG i;
	REGISTER GEOM **list;
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *var;

	/* display the dialog box */
	us_facetlistdialog.movable = prompt;
	if (DiaInitDialog(&us_facetlistdialog) != 0) return(0);
	us_showoldversions = 1;
	us_showcellibraryfacets = 1;
	us_curlib = el_curlib;
	DiaInitTextDialog(3, us_oldfacettopoffacets, us_oldfacetnextfacets, DiaNullDlogDone,
		0, SCSELMOUSE|SCSELKEY|SCDOUBLEQUIT);

	/* find the current node and make it the default */
	np = getcurfacet();
	if (np != NONODEPROTO)
	{
		curnode = describenodeproto(np);
		if (curinstance != 0)
		{
			list = us_gethighlighted(OBJNODEINST);
			if (list[0] != NOGEOM && list[1] == NOGEOM)
			{
				if (list[0]->entrytype == OBJNODEINST)
				{
					ni = list[0]->entryaddr.ni;
					if (ni->proto->primindex == 0)
						curnode = describenodeproto(ni->proto);
				}
			}
		}
		if (namesame(np->cell->cellname, "facetstructure") == 0)
		{
			list = us_gethighlighted(OBJNODEINST);
			if (list[0] != NOGEOM && list[1] == NOGEOM)
			{
				if (list[0]->entrytype == OBJNODEINST)
				{
					ni = list[0]->entryaddr.ni;
					var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, art_messagekey);
					if (var != NOVARIABLE)
					{
						curnode = (char *)var->addr;
					}
				}
			}
		}
		for(i=0; ; i++)
		{
			line = DiaGetScrollLine(3, i);
			if (*line == 0) break;
			if (strcmp(line, curnode) == 0)
			{
				DiaSelectLine(3, i);
				break;
			}
		}
	}

	/* see if there are any old versions */
	for(np = us_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (np->newestversion != np) break;
	if (np == NONODEPROTO) DiaDimItem(4); else
	{
		DiaUnDimItem(4);
		DiaSetControl(4, us_showoldversions);
	}

	/* see if there are any cell-library facets */
	for(np = us_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if ((np->userbits&INCELLLIBRARY) != 0) break;
	if (np == NONODEPROTO) DiaDimItem(5); else
	{
		DiaUnDimItem(5);
		DiaSetControl(5, us_showcellibraryfacets);
	}

	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == 4)
		{
			us_showoldversions = 1 - DiaGetControl(4);
			DiaSetControl(4, us_showoldversions);
			DiaLoadTextDialog(3, us_oldfacettopoffacets, us_oldfacetnextfacets, DiaNullDlogDone, 0);
		}
		if (itemHit == 5)
		{
			us_showcellibraryfacets = 1 - DiaGetControl(5);
			DiaSetControl(5, us_showcellibraryfacets);
			DiaLoadTextDialog(3, us_oldfacettopoffacets, us_oldfacetnextfacets, DiaNullDlogDone, 0);
		}
	}
	paramstart[0] = us_putintoinfstr(DiaGetScrollLine(3, DiaGetCurLine(3)));
	DiaDoneDialog();
	if (itemHit == CANCEL) return(0);
	return(1);
}

/****************************** FACET OPTIONS DIALOG ******************************/

/* Facet options */
DIALOGITEM us_facetdialogitems[] =
{
 /*  1 */ {0, {224,352,248,416}, BUTTON, "OK"},
 /*  2 */ {0, {184,352,208,416}, BUTTON, "Cancel"},
 /*  3 */ {0, {260,24,276,324}, CHECK, "Tiny facet instances hashed out"},
 /*  4 */ {0, {8,4,24,126}, MESSAGE, "For facet:"},
 /*  5 */ {0, {8,128,24,338}, POPUP, ""},
 /*  6 */ {0, {212,24,228,324}, CHECK, "Disallow modification of facet instances"},
 /*  7 */ {0, {176,4,177,416}, DIVIDELINE, ""},
 /*  8 */ {0, {188,4,204,152}, MESSAGE, "For all facets:"},
 /*  9 */ {0, {32,32,48,376}, CHECK, "Disallow modification of anything in this facet"},
 /* 10 */ {0, {236,24,252,324}, CHECK, "Place facet-centers when making icons"},
 /* 11 */ {0, {56,32,72,376}, CHECK, "Disallow modification of instances in this facet"},
 /* 12 */ {0, {128,32,144,192}, MESSAGE|INACTIVE, "Characteristic X Spacing:"},
 /* 13 */ {0, {152,32,168,192}, MESSAGE|INACTIVE, "Characteristic Y Spacing:"},
 /* 14 */ {0, {128,196,144,272}, EDITTEXT, ""},
 /* 15 */ {0, {152,196,168,272}, EDITTEXT, ""},
 /* 16 */ {0, {80,32,96,376}, CHECK, "Part of a cell-library"},
 /* 17 */ {0, {332,24,348,168}, BUTTON, "All part of cell-library"},
 /* 18 */ {0, {332,180,348,344}, BUTTON, "All not part of cell-library"},
 /* 19 */ {0, {308,24,324,244}, MESSAGE|INACTIVE, "Hash facets when scale is more than:"},
 /* 20 */ {0, {308,248,324,308}, EDITTEXT, ""},
 /* 21 */ {0, {308,312,324,416}, MESSAGE|INACTIVE, "lambda per pixel"},
 /* 22 */ {0, {104,32,120,220}, RADIO, "Expand new instances"},
 /* 23 */ {0, {104,228,120,416}, RADIO, "Unexpand new instances"},
 /* 24 */ {0, {284,24,300,324}, CHECK, "Check facet dates during creation"}
};
DIALOG us_facetdialog = {{50,75,407,500}, "Facet Options", 24, us_facetdialogitems};

/*
 * special case for the "facet info" dialog
 * Tiny facets                    =  3 (check)
 * Facet name                     =  5 (popup)
 * Allow all inst mod             =  6 (check)
 * Allow all mod in facet         =  9 (check)
 * Allow inst mod in facet        = 11 (check)
 * X Characteristic spacing       = 14 (edit)
 * Y Characteristic spacing       = 15 (edit)
 * Part of cell library           = 16 (check)
 * Make all from cell library     = 17 (button)
 * Make all not from cell library = 18 (button)
 * Tiny lambda per pixel          = 20 (edit)
 * Expand new instances           = 22 (radio)
 * Unexpand new instances         = 23 (radio)
 * Check dates                    = 24 (check)
 */
INTSML us_facetdlog(char *paramstart[])
{
	INTBIG itemHit, i, numfacets, curindex;
	INTBIG lx;
	REGISTER NODEPROTO *thisfacet, *np;
	REGISTER VARIABLE *var;
	REGISTER char **facetnames, *pt;
	typedef struct
	{
		INTBIG formerbits;
		INTBIG validcharacteristicspacing;
		INTBIG characteristicspacing[2];
	} FACETINFO;
	FACETINFO *fi;

	/* display the facet dialog box */
	if (DiaInitDialog(&us_facetdialog) != 0) return(0);

	thisfacet = NONODEPROTO;
	if (el_curwindowpart != NOWINDOWPART)
	{
		thisfacet = el_curwindowpart->curnodeproto;
		if (thisfacet != NONODEPROTO && thisfacet->cell->lib != el_curlib) thisfacet = NONODEPROTO;
	}
	if (thisfacet == NONODEPROTO)
		thisfacet = el_curlib->firstnodeproto;

	/* construct lists of facets */
	numfacets = 0;
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		numfacets++;
	if (numfacets != 0)
		facetnames = (char **)emalloc(numfacets * (sizeof (char *)), el_tempcluster);
	numfacets = 0;
	curindex = 0;
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (np == thisfacet) curindex = numfacets;
		fi = (FACETINFO *)emalloc(sizeof (FACETINFO), el_tempcluster);
		if (fi == 0) return(0);
		fi->formerbits = np->userbits;
		var = getval((INTBIG)np, VNODEPROTO, VINTEGER|VISARRAY, "FACET_characteristic_spacing");
		if (var == NOVARIABLE) fi->validcharacteristicspacing = 0; else
		{
			fi->characteristicspacing[0] = ((INTBIG *)var->addr)[0];
			fi->characteristicspacing[1] = ((INTBIG *)var->addr)[1];
			fi->validcharacteristicspacing = 1;
		}
		np->temp1 = (INTBIG)fi;
		np->temp2 = numfacets;
		(void)allocstring(&facetnames[numfacets++], describenodeproto(np),
			el_tempcluster);
	}
	DiaSetPopup(5, numfacets, facetnames);
	DiaSetPopupEntry(5, curindex);
	for(i=0; i<numfacets; i++) efree(facetnames[i]);
	if (numfacets != 0) efree((char *)facetnames);

	/* load defaults for primitives and facets */
	if (thisfacet != NONODEPROTO)
	{
		DiaSetPopupEntry(5, thisfacet->temp2);
		if ((thisfacet->userbits&NLOCKED) != 0) DiaSetControl(9, 1); else
			DiaSetControl(9, 0);
		if ((thisfacet->userbits&NILOCKED) != 0) DiaSetControl(11, 1); else
			DiaSetControl(11, 0);
		if ((thisfacet->userbits&INCELLLIBRARY) != 0) DiaSetControl(16, 1); else
			DiaSetControl(16, 0);
		if ((thisfacet->userbits&WANTNEXPAND) != 0) DiaSetControl(22, 1); else
			DiaSetControl(23, 1);
		fi = (FACETINFO *)thisfacet->temp1;
		if (fi->validcharacteristicspacing != 0)
		{
			DiaSetText(14, latoa(fi->characteristicspacing[0]));
			DiaSetText(15, latoa(fi->characteristicspacing[1]));
		}
	} else
	{
		DiaDimItem(5);
		DiaDimItem(9);
		DiaDimItem(11);
		DiaDimItem(14);
		DiaDimItem(15);
		DiaDimItem(16);
		DiaDimItem(17);
		DiaDimItem(18);
		DiaDimItem(22);
		DiaDimItem(23);
	}

	/* load defaults for all nodes */
	DiaSetControl(6, (us_useroptions&NOFACETCHANGES) != 0 ? 1 : 0);
	DiaSetControl(3, (us_useroptions&DRAWTINYFACETS) == 0 ? 1 : 0);
	DiaSetControl(10, (us_useroptions&PUTFACETCENTERINICON) != 0 ? 1 : 0);
	DiaSetControl(24, (us_useroptions&CHECKDATE) != 0 ? 1 : 0);
	DiaSetText(20, frtoa(us_tinyratio));

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == 5)
		{
			i = DiaGetPopupEntry(5);
			for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				if (np->temp2 == i) break;
			thisfacet = np;
			if (thisfacet == NONODEPROTO) continue;
			fi = (FACETINFO *)thisfacet->temp1;
			if ((fi->formerbits&NLOCKED) != 0) DiaSetControl(9, 1); else
				DiaSetControl(9, 0);
			if ((fi->formerbits&NILOCKED) != 0) DiaSetControl(11, 1); else
				DiaSetControl(11, 0);
			if ((fi->formerbits&INCELLLIBRARY) != 0) DiaSetControl(16, 1); else
				DiaSetControl(16, 0);
			if ((fi->formerbits&WANTNEXPAND) != 0)
			{
				DiaSetControl(22, 1);
				DiaSetControl(23, 0);
			} else
			{
				DiaSetControl(22, 0);
				DiaSetControl(23, 1);
			}
			if (fi->validcharacteristicspacing != 0)
			{
				DiaSetText(14, latoa(fi->characteristicspacing[0]));
				DiaSetText(15, latoa(fi->characteristicspacing[1]));
			} else
			{
				DiaSetText(14, "");
				DiaSetText(15, "");
			}
			continue;
		}
		if (itemHit == 9)
		{
			if (DiaGetControl(9) != 0)
			{
				if (thisfacet != NONODEPROTO) fi->formerbits &= ~NLOCKED;
				DiaSetControl(itemHit, 0);
			} else
			{
				if (thisfacet != NONODEPROTO) fi->formerbits |= NLOCKED;
				DiaSetControl(itemHit, 1);
			}
			continue;
		}
		if (itemHit == 11)
		{
			if (DiaGetControl(11) != 0)
			{
				if (thisfacet != NONODEPROTO) fi->formerbits &= ~NILOCKED;
				DiaSetControl(itemHit, 0);
			} else
			{
				if (thisfacet != NONODEPROTO) fi->formerbits |= NILOCKED;
				DiaSetControl(itemHit, 1);
			}
			continue;
		}
		if (itemHit == 16)
		{
			if (DiaGetControl(16) != 0)
			{
				if (thisfacet != NONODEPROTO) fi->formerbits &= ~INCELLLIBRARY;
				DiaSetControl(itemHit, 0);
			} else
			{
				if (thisfacet != NONODEPROTO) fi->formerbits |= INCELLLIBRARY;
				DiaSetControl(itemHit, 1);
			}
			continue;
		}
		if (itemHit == 22 || itemHit == 23)
		{
			DiaSetControl(22, 0);
			DiaSetControl(23, 0);
			DiaSetControl(itemHit, 1);
			if (itemHit == 22)
			{
				if (thisfacet != NONODEPROTO) fi->formerbits |= WANTNEXPAND;
			} else
			{
				if (thisfacet != NONODEPROTO) fi->formerbits &= ~WANTNEXPAND;
			}
			continue;
		}
		if (itemHit == 17)
		{
			for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			{
				fi = (FACETINFO *)np->temp1;
				fi->formerbits |= INCELLLIBRARY;
			}
			DiaSetControl(16, 1);
			continue;
		}
		if (itemHit == 18)
		{
			for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			{
				fi = (FACETINFO *)np->temp1;
				fi->formerbits &= ~INCELLLIBRARY;
			}
			DiaSetControl(16, 0);
			continue;
		}
		if (itemHit == 14)
		{
			if (thisfacet == NONODEPROTO) continue;
			pt = DiaGetText(14);
			if (fi->validcharacteristicspacing == 0 && *pt == 0) continue;
			if (fi->validcharacteristicspacing == 0)
				fi->characteristicspacing[1] = 0;
			fi->validcharacteristicspacing = 1;
			fi->characteristicspacing[0] = atola(pt);
			continue;
		}
		if (itemHit == 15)
		{
			if (thisfacet == NONODEPROTO) continue;
			pt = DiaGetText(15);
			if (fi->validcharacteristicspacing == 0 && *pt == 0) continue;
			if (fi->validcharacteristicspacing == 0)
				fi->characteristicspacing[0] = 0;
			fi->validcharacteristicspacing = 1;
			fi->characteristicspacing[1] = atola(pt);
			continue;
		}
		if (itemHit == 6 || itemHit == 3 || itemHit == 10 || itemHit == 24)
		{
			DiaSetControl(itemHit, 1-DiaGetControl(itemHit));
			continue;
		}
	}

	paramstart[0] = "x";
	if (itemHit != CANCEL)
	{
		/* handle facet changes */
		for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			fi = (FACETINFO *)np->temp1;
			if ((fi->formerbits&(NLOCKED|NILOCKED|INCELLLIBRARY|WANTNEXPAND)) !=
				(INTBIG)(np->userbits&(NLOCKED|NILOCKED|INCELLLIBRARY|WANTNEXPAND)))
					(void)setval((INTBIG)np, VNODEPROTO, "userbits", fi->formerbits, VINTEGER);
			if (fi->validcharacteristicspacing != 0)
				setval((INTBIG)np, VNODEPROTO, "FACET_characteristic_spacing",
					(INTBIG)fi->characteristicspacing, VINTEGER|VISARRAY|(2<<VLENGTHSH));
		}

		/* handle changes to all nodes */
		lx = us_useroptions;
		i = DiaGetControl(6);
		if (i != 0) lx |= NOFACETCHANGES; else lx &= ~NOFACETCHANGES;
		i = DiaGetControl(3);
		if (i == 0) lx |= DRAWTINYFACETS; else lx &= ~DRAWTINYFACETS;
		i = DiaGetControl(10);
		if (i != 0) lx |= PUTFACETCENTERINICON; else lx &= ~PUTFACETCENTERINICON;
		i = DiaGetControl(24);
		if (i != 0) lx |= CHECKDATE; else lx &= ~CHECKDATE;
		(void)setvalkey((INTBIG)us_aid, VAID, us_optionflags, lx, VINTEGER);
		i = atofr(DiaGetText(20));
		(void)setvalkey((INTBIG)us_aid, VAID, us_tinylambdaperpixel, i, VFRACT);
	}
	DiaDoneDialog();
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		efree((char *)np->temp1);
	return(1);
}

/****************************** FRAME OPTIONS DIALOG ******************************/

/* Drawing Options */
DIALOGITEM us_drawingoptdialogitems[] =
{
 /*  1 */ {0, {64,244,88,308}, BUTTON, "OK"},
 /*  2 */ {0, {64,32,88,96}, BUTTON, "Cancel"},
 /*  3 */ {0, {12,8,28,125}, MESSAGE, "Company Name:"},
 /*  4 */ {0, {12,128,28,328}, EDITTEXT, ""},
 /*  5 */ {0, {36,8,52,125}, MESSAGE, "Designer Name:"},
 /*  6 */ {0, {36,128,52,328}, EDITTEXT, ""}
};
DIALOG us_drawingoptdialog = {{50,75,149,417}, "Frame Options", 6, us_drawingoptdialogitems};

/*
 * special case for the "Drawing Options" dialog
 * Company Name      = 4 (edit text)
 * Designer Name     = 6 (edit text)
 */
INTSML us_frameoptionsdlog(char *paramstart[])
{
	INTBIG itemHit;
	REGISTER VARIABLE *var;
	REGISTER char *pt;

	/* show the drawing options dialog */
	if (DiaInitDialog(&us_drawingoptdialog) != 0) return(0);
	var = getval((INTBIG)us_aid, VAID, VSTRING, "USER_drawing_company_name");
	if (var != NOVARIABLE) DiaSetText(4, (char *)var->addr);
	var = getval((INTBIG)us_aid, VAID, VSTRING, "USER_drawing_designer_name");
	if (var != NOVARIABLE) DiaSetText(6, (char *)var->addr);
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK) break;
	}
	if (itemHit == OK)
	{
		pt = DiaGetText(4);
		while (*pt == ' ' || *pt == '\t') pt++;
		if (*pt == 0)
		{
			/* remove company information */
			if (var != NOVARIABLE) delval((INTBIG)us_aid, VAID, "USER_drawing_company_name");
		} else
		{
			(void)setval((INTBIG)us_aid, VAID, "USER_drawing_company_name", (INTBIG)pt, VSTRING);
		}

		pt = DiaGetText(6);
		while (*pt == ' ' || *pt == '\t') pt++;
		if (*pt == 0)
		{
			/* remove designer information */
			if (var != NOVARIABLE) delval((INTBIG)us_aid, VAID, "USER_drawing_designer_name");
		} else
		{
			(void)setval((INTBIG)us_aid, VAID, "USER_drawing_designer_name", (INTBIG)pt, VSTRING);
		}
	}
	DiaDoneDialog();
	return(0);
}

/****************************** GET INFO DIALOGS ******************************/

/* Multiple Object Info */
DIALOGITEM us_multigetinfodialogitems[] =
{
 /*  1 */ {0, {248,220,272,300}, BUTTON, "Info"},
 /*  2 */ {0, {248,8,272,88}, BUTTON, "Cancel"},
 /*  3 */ {0, {20,4,236,304}, SCROLL, ""},
 /*  4 */ {0, {0,4,16,304}, MESSAGE, "0 Objects:"}
};
DIALOG us_multigetinfodialog = {{75,75,356,388}, "Multiple Object Information", 4, us_multigetinfodialogitems};

/* Node info */
DIALOGITEM us_shownodedialogitems[] =
{
 /*  1 */ {0, {372,288,396,360}, BUTTON, "OK"},
 /*  2 */ {0, {372,32,396,104}, BUTTON, "Cancel"},
 /*  3 */ {0, {236,8,364,392}, SCROLL, ""},
 /*  4 */ {0, {196,88,212,184}, RADIO, "Expanded"},
 /*  5 */ {0, {196,192,212,296}, RADIO, "Unexpanded"},
 /*  6 */ {0, {8,64,24,392}, MESSAGE, ""},
 /*  7 */ {0, {32,64,48,392}, EDITTEXT, ""},
 /*  8 */ {0, {76,80,92,174}, EDITTEXT, ""},
 /*  9 */ {0, {100,80,116,174}, EDITTEXT, ""},
 /* 10 */ {0, {124,80,140,128}, EDITTEXT, ""},
 /* 11 */ {0, {124,136,140,246}, CHECK, "Transposed"},
 /* 12 */ {0, {76,296,92,392}, EDITTEXT, ""},
 /* 13 */ {0, {100,296,116,392}, EDITTEXT, ""},
 /* 14 */ {0, {8,8,24,56}, MESSAGE, "Type:"},
 /* 15 */ {0, {32,8,48,56}, MESSAGE, "Name:"},
 /* 16 */ {0, {76,8,92,72}, MESSAGE, "X size:"},
 /* 17 */ {0, {100,8,116,72}, MESSAGE, "Y size:"},
 /* 18 */ {0, {124,8,140,72}, MESSAGE, "Rotation:"},
 /* 19 */ {0, {76,208,92,288}, MESSAGE, "X position:"},
 /* 20 */ {0, {100,208,116,288}, MESSAGE, "Y position:"},
 /* 21 */ {0, {220,8,236,76}, MESSAGE, "Ports:"},
 /* 22 */ {0, {172,160,188,392}, MESSAGE, ""},
 /* 23 */ {0, {148,8,164,160}, MESSAGE, ""},
 /* 24 */ {0, {148,160,164,392}, MESSAGE, ""},
 /* 25 */ {0, {172,8,188,160}, MESSAGE, ""},
 /* 26 */ {0, {372,156,396,228}, BUTTON, "See Arc"},
 /* 27 */ {0, {56,76,72,193}, MESSAGE, ""},
 /* 28 */ {0, {56,292,72,392}, MESSAGE, "Lower-left:"},
 /* 29 */ {0, {124,264,140,392}, CHECK, "Easy to Select"}
};
DIALOG us_shownodedialog = {{50,75,455,484}, "Node Information", 29, us_shownodedialogitems};

/* Arc info */
DIALOGITEM us_showarcdialogitems[] =
{
 /*  1 */ {0, {164,336,188,408}, BUTTON, "OK"},
 /*  2 */ {0, {120,336,144,408}, BUTTON, "Cancel"},
 /*  3 */ {0, {224,312,240,392}, CHECK, "Rigid"},
 /*  4 */ {0, {256,312,272,408}, CHECK, "Fixed-angle"},
 /*  5 */ {0, {8,80,24,385}, MESSAGE, ""},
 /*  6 */ {0, {32,88,48,393}, MESSAGE, ""},
 /*  7 */ {0, {56,88,72,393}, EDITTEXT, ""},
 /*  8 */ {0, {104,88,120,173}, MESSAGE, ""},
 /*  9 */ {0, {128,88,144,320}, MESSAGE, ""},
 /* 10 */ {0, {80,280,96,365}, MESSAGE, ""},
 /* 11 */ {0, {172,88,188,320}, MESSAGE, ""},
 /* 12 */ {0, {8,16,24,80}, MESSAGE, "Type:"},
 /* 13 */ {0, {32,16,48,80}, MESSAGE, "Network:"},
 /* 14 */ {0, {80,16,96,80}, MESSAGE, "Width:"},
 /* 15 */ {0, {104,16,120,80}, MESSAGE, "Angle:"},
 /* 16 */ {0, {128,16,144,80}, MESSAGE, "Head:"},
 /* 17 */ {0, {80,216,96,280}, MESSAGE, "Bus size:"},
 /* 18 */ {0, {172,16,188,80}, MESSAGE, "Tail:"},
 /* 19 */ {0, {272,312,288,408}, CHECK, "Slidable"},
 /* 20 */ {0, {240,16,256,112}, CHECK, "Negated"},
 /* 21 */ {0, {256,16,272,112}, CHECK, "Directional"},
 /* 22 */ {0, {272,16,288,120}, CHECK, "Ends extend"},
 /* 23 */ {0, {240,136,256,240}, CHECK, "Ignore head"},
 /* 24 */ {0, {256,136,272,232}, CHECK, "Ignore tail"},
 /* 25 */ {0, {272,136,288,304}, CHECK, "Reverse head and tail"},
 /* 26 */ {0, {240,328,256,424}, CHECK, "Temporary"},
 /* 27 */ {0, {80,88,96,173}, EDITTEXT, ""},
 /* 28 */ {0, {56,16,72,72}, MESSAGE, "Name:"},
 /* 29 */ {0, {216,16,232,240}, MESSAGE, ""},
 /* 30 */ {0, {148,40,164,80}, MESSAGE, "At:"},
 /* 31 */ {0, {148,88,164,223}, MESSAGE, ""},
 /* 32 */ {0, {148,232,164,272}, BUTTON, "See"},
 /* 33 */ {0, {192,40,208,80}, MESSAGE, "At:"},
 /* 34 */ {0, {192,88,208,223}, MESSAGE, ""},
 /* 35 */ {0, {192,232,208,272}, BUTTON, "See"},
 /* 36 */ {0, {148,280,164,320}, BUTTON, "Info"},
 /* 37 */ {0, {192,280,208,320}, BUTTON, "Info"},
 /* 38 */ {0, {104,200,120,320}, CHECK, "Easy to Select"}
};
DIALOG us_showarcdialog = {{50,75,347,508}, "Arc Information", 38, us_showarcdialogitems};

/* Text info */
DIALOGITEM us_showtextdialogitems[] =
{
 /*  1 */ {0, {200,248,224,320}, BUTTON, "OK"},
 /*  2 */ {0, {200,160,224,232}, BUTTON, "Cancel"},
 /*  3 */ {0, {144,8,160,64}, RADIO, "Left"},
 /*  4 */ {0, {40,16,56,332}, EDITTEXT, ""},
 /*  5 */ {0, {72,232,88,340}, POPUP, "Small"},
 /*  6 */ {0, {80,8,96,80}, RADIO, "Center"},
 /*  7 */ {0, {96,8,112,80}, RADIO, "Bottom"},
 /*  8 */ {0, {112,8,128,64}, RADIO, "Top"},
 /*  9 */ {0, {128,8,144,72}, RADIO, "Right"},
 /* 10 */ {0, {160,8,176,104}, RADIO, "Lower right"},
 /* 11 */ {0, {176,8,192,104}, RADIO, "Lower left"},
 /* 12 */ {0, {192,8,208,104}, RADIO, "Upper right"},
 /* 13 */ {0, {208,8,224,96}, RADIO, "Upper left"},
 /* 14 */ {0, {224,8,240,80}, RADIO, "Boxed"},
 /* 15 */ {0, {72,152,88,231}, MESSAGE, "Text size:"},
 /* 16 */ {0, {64,16,80,110}, MESSAGE, "Text corner:"},
 /* 17 */ {0, {104,224,120,304}, EDITTEXT, ""},
 /* 18 */ {0, {128,224,144,304}, EDITTEXT, ""},
 /* 19 */ {0, {104,152,120,217}, MESSAGE, "X offset:"},
 /* 20 */ {0, {128,152,144,217}, MESSAGE, "Y offset:"},
 /* 21 */ {0, {80,112,112,144}, ICON, (char *)us_icon200},
 /* 22 */ {0, {112,112,144,144}, ICON, (char *)us_icon201},
 /* 23 */ {0, {144,112,176,144}, ICON, (char *)us_icon202},
 /* 24 */ {0, {176,112,208,144}, ICON, (char *)us_icon203},
 /* 25 */ {0, {208,112,240,144}, ICON, (char *)us_icon204},
 /* 26 */ {0, {8,16,24,340}, MESSAGE, ""},
 /* 27 */ {0, {160,190,180,290}, BUTTON, "Edit Text"}
};
DIALOG us_showtextdialog = {{50,75,304,424}, "Information on Highlighted Text", 27, us_showtextdialogitems};

/* Port info */
DIALOGITEM us_portinfodialogitems[] =
{
 /*  1 */ {0, {184,328,208,400}, BUTTON, "OK"},
 /*  2 */ {0, {152,328,176,400}, BUTTON, "Cancel"},
 /*  3 */ {0, {120,8,136,56}, RADIO, "Left"},
 /*  4 */ {0, {8,112,24,408}, EDITTEXT, ""},
 /*  5 */ {0, {72,232,88,354}, POPUP, "Small"},
 /*  6 */ {0, {56,8,72,88}, RADIO, "Center"},
 /*  7 */ {0, {72,8,88,80}, RADIO, "Bottom"},
 /*  8 */ {0, {88,8,104,56}, RADIO, "Top"},
 /*  9 */ {0, {104,8,120,64}, RADIO, "Right"},
 /* 10 */ {0, {136,8,152,104}, RADIO, "Lower right"},
 /* 11 */ {0, {152,8,168,96}, RADIO, "Lower left"},
 /* 12 */ {0, {168,8,184,104}, RADIO, "Upper right"},
 /* 13 */ {0, {184,8,200,96}, RADIO, "Upper left"},
 /* 14 */ {0, {200,8,216,72}, RADIO, "Boxed"},
 /* 15 */ {0, {72,160,88,231}, MESSAGE, "Text size:"},
 /* 16 */ {0, {40,16,56,107}, MESSAGE, "Text corner:"},
 /* 17 */ {0, {104,232,120,312}, EDITTEXT, ""},
 /* 18 */ {0, {128,232,144,312}, EDITTEXT, ""},
 /* 19 */ {0, {104,160,120,225}, MESSAGE, "X offset:"},
 /* 20 */ {0, {128,160,144,225}, MESSAGE, "Y offset:"},
 /* 21 */ {0, {56,112,88,144}, ICON, (char *)us_icon200},
 /* 22 */ {0, {88,112,120,144}, ICON, (char *)us_icon201},
 /* 23 */ {0, {120,112,152,144}, ICON, (char *)us_icon202},
 /* 24 */ {0, {152,112,184,144}, ICON, (char *)us_icon203},
 /* 25 */ {0, {184,112,216,144}, ICON, (char *)us_icon204},
 /* 26 */ {0, {8,8,24,104}, MESSAGE, "Port name:"},
 /* 27 */ {0, {40,272,56,416}, POPUP, "Input"},
 /* 28 */ {0, {40,160,56,271}, MESSAGE, "Characteristics:"},
 /* 29 */ {0, {184,160,200,272}, CHECK, "Always drawn"},
 /* 30 */ {0, {160,160,176,252}, CHECK, "Body only"}
};
DIALOG us_portinfodialog = {{50,75,275,500}, "Port Information", 30, us_portinfodialogitems};

void us_getinfonode(NODEINST *ni, PORTPROTO *fromport);
void us_getinfoarc(ARCINST *ai);
void us_getinfotext(HIGHLIGHT *high, INTBIG canspecialize);
INTBIG us_dlogchatportproto(NODEINST*, PORTPROTO*, PORTPROTO*, INTBIG);
void us_updatenodeinfo(NODEINST*, VARIABLE*, char*, char*, char*);

INTSML us_showdlog(char *paramstart[], INTSML canspecialize)
{
	HIGHLIGHT high, *manyhigh;
	char buf[200];
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER GEOM *geom;
	REGISTER INTBIG len, i, cx1, cy1, cx2, cy2, got1, got2, dx, dy;
	INTBIG cx, cy;
	XARRAY trans;
	REGISTER INTBIG itemHit, which, highlightchanged;
	REGISTER VARIABLE *var;

	/* see if anything is highlighted */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE)
	{
		us_abortcommand("Nothing is highlighted");
		return(0);
	}

	/* special dialog when only 1 highlighted object */
	len = getlength(var);
	if (len == 1)
	{
		/* get the one highlighted object */
		if (us_makehighlight(((char **)var->addr)[0], &high) != 0)
		{
			us_abortcommand("Highlight unintelligible");
			return(0);
		}

		if ((high.status&HIGHTYPE) == HIGHTEXT)
		{
			us_getinfotext(&high, canspecialize);
			return(0);
		}
		if ((high.status&HIGHTYPE) == HIGHFROM)
		{
			if (high.fromgeom->entrytype == OBJARCINST)
			{
				/* arc getinfo */
				ai = (ARCINST *)high.fromgeom->entryaddr.ai;
				us_getinfoarc(ai);
				return(0);
			}
			ni = (NODEINST *)high.fromgeom->entryaddr.ni;
			us_getinfonode(ni, high.fromport);
			return(0);
		}
	}

	/* multiple objects highlighted: remember them all */
	manyhigh = (HIGHLIGHT *)emalloc(len * (sizeof (HIGHLIGHT)), el_tempcluster);
	if (manyhigh == 0) return(0);
	for(i=0; i<len; i++)
	{
		if (us_makehighlight(((char **)var->addr)[i], &manyhigh[i]) != 0)
		{
			us_abortcommand("Highlight unintelligible");
			break;
		}
	}

	if (DiaInitDialog(&us_multigetinfodialog) != 0)
	{
		efree((char *)manyhigh);
		return(0);
	}
	DiaInitTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCDOUBLEQUIT|SCREPORT);

	/* multiple objects selected: list them */
	sprintf(buf, "%ld %s:", len, makeplural("selection", len));
	DiaSetText(4, buf);
	got1 = got2 = 0;
	for(i=0; i<len; i++)
	{
		(void)initinfstr();
		switch (manyhigh[i].status&HIGHTYPE)
		{
			case HIGHFROM:
				geom = manyhigh[i].fromgeom;
				cx = (geom->lowx + geom->highx) / 2;
				cy = (geom->lowy + geom->highy) / 2;
				if (geom->entrytype == OBJNODEINST)
				{
					ni = (NODEINST *)geom->entryaddr.ni;
					(void)addstringtoinfstr("Node ");
					(void)addstringtoinfstr(describenodeinst(ni));
					var = getvalkey((INTBIG)ni->proto, VNODEPROTO, VINTEGER|VISARRAY, el_prototype_center);
					if (var != NOVARIABLE)
					{
						dx = ((INTBIG *)var->addr)[0] + (ni->lowx+ni->highx)/2 -
							(ni->proto->lowx+ni->proto->highx)/2;
						dy = ((INTBIG *)var->addr)[1] + (ni->lowy+ni->highy)/2 -
							(ni->proto->lowy+ni->proto->highy)/2;
						makerot(ni, trans);
						xform(dx, dy, &cx, &cy, trans);
					}
				} else
				{
					ai = (ARCINST *)geom->entryaddr.ai;
					(void)addstringtoinfstr("Arc ");
					(void)addstringtoinfstr(describearcinst(ai));
				}
				if (i == 0)
				{
					cx1 = cx;
					cy1 = cy;
					got1 = 1;
				} else if (i == 1)
				{
					cx2 = cx;
					cy2 = cy;
					got2 = 1;
				}
				break;
			case HIGHBBOX:
				(void)addstringtoinfstr("Bounds: X from ");
				(void)addstringtoinfstr(latoa(manyhigh[i].stalx));
				(void)addstringtoinfstr(" to ");
				(void)addstringtoinfstr(latoa(manyhigh[i].stahx));
				(void)addstringtoinfstr(", Y from ");
				(void)addstringtoinfstr(latoa(manyhigh[i].staly));
				(void)addstringtoinfstr(" to ");
				(void)addstringtoinfstr(latoa(manyhigh[i].stahy));
				break;
			case HIGHLINE:
				(void)addstringtoinfstr("Line (");
				(void)addstringtoinfstr(latoa(manyhigh[i].stalx));
				(void)addstringtoinfstr(", ");
				(void)addstringtoinfstr(latoa(manyhigh[i].staly));
				(void)addstringtoinfstr(") to (");
				(void)addstringtoinfstr(latoa(manyhigh[i].stahx));
				(void)addstringtoinfstr(", ");
				(void)addstringtoinfstr(latoa(manyhigh[i].stahy));
				(void)addstringtoinfstr(")");
				break;
			case HIGHTEXT:
				if (manyhigh[i].fromport != NOPORTPROTO)
				{
					(void)addstringtoinfstr("Port ");
					(void)addstringtoinfstr(manyhigh[i].fromport->protoname);
					break;
				}
				if (manyhigh[i].fromvar != NOVARIABLE)
				{
					if (manyhigh[i].fromgeom->entrytype == OBJARCINST)
					{
						ai = (ARCINST *)manyhigh[i].fromgeom->entryaddr.ai;
						(void)addstringtoinfstr("Text on arc ");
						(void)addstringtoinfstr(describearcinst(ai));
						break;
					}
					ni = (NODEINST *)manyhigh[i].fromgeom->entryaddr.ni;
					(void)addstringtoinfstr("Text on node ");
					(void)addstringtoinfstr(describenodeinst(ni));
					break;
				}
				(void)addstringtoinfstr("Text object");
				break;
		}
		DiaStuffLine(3, returninfstr());
	}

	/* if there are exactly two node/arc objects, show the distance between them */
	if (len == 2 && got1 != 0 && got2 != 0)
	{
		DiaStuffLine(3, "----------------------------");
		DiaStuffLine(3, "Distance between centers is:");
		(void)initinfstr();
		(void)addstringtoinfstr("  X: ");
		(void)addstringtoinfstr(latoa(abs(cx1-cx2)));
		(void)addstringtoinfstr(", Y: ");
		(void)addstringtoinfstr(latoa(abs(cy1-cy2)));
		DiaStuffLine(3, returninfstr());
	}
	DiaSelectLine(3, 0);

	/* loop dialog */
	highlightchanged = 0;
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == 3)
		{
			which = DiaGetCurLine(3);
			if (which >= len)
			{
				DiaDimItem(1);
				continue;
			} else
			{
				DiaUnDimItem(1);
			}
			high = manyhigh[which];
			us_clearhighlightcount();
			(void)us_addhighlight(&high);
			us_showallhighlight();
			us_endchanges(NOWINDOWPART);
			highlightchanged = 1;
		}
	}
	if (itemHit == OK)
	{
		which = DiaGetCurLine(3);
		high = manyhigh[which];
	} else
	{
		if (highlightchanged != 0)
		{
			us_clearhighlightcount();
			for(i=0; i<len; i++)
				(void)us_addhighlight(&manyhigh[i]);
		}
	}
	DiaDoneDialog();
	efree((char *)manyhigh);
	if (itemHit == OK)
	{
		if ((high.status&HIGHTYPE) == HIGHTEXT)
		{
			us_getinfotext(&high, canspecialize);
			return(0);
		}
		if ((high.status&HIGHTYPE) == HIGHFROM)
		{
			if (high.fromgeom->entrytype == OBJARCINST)
			{
				/* arc getinfo */
				ai = (ARCINST *)high.fromgeom->entryaddr.ai;
				us_getinfoarc(ai);
				return(0);
			}
			ni = (NODEINST *)high.fromgeom->entryaddr.ni;
			us_getinfonode(ni, high.fromport);
			return(0);
		}
	}
	return(0);
}

/*
 * special case for the "Text Get Info" dialog
 * text name             = 4 (edit text)
 * text size             = 5 (popup)
 * X offset              = 17 (edit text)
 * Y offset              = 18 (edit text)
 * center                = 6 (radio button)
 * bottom                = 7 (radio button)
 * top                   = 8 (radio button)
 * right                 = 9 (radio button)
 * left                  = 3 (radio button)
 * lower right           = 10 (radio button)
 * lower left            = 11 (radio button)
 * upper right           = 12 (radio button)
 * upper left            = 13 (radio button)
 * boxed                 = 14 (radio button)
 * title                 = 26 (message)
 *
 * special case for the "Port Get Info" dialog
 * text name             = 4 (edit text)
 * text size             = 5 (popup)
 * X offset              = 17 (edit text)
 * Y offset              = 18 (edit text)
 * center                = 6 (radio button)
 * bottom                = 7 (radio button)
 * top                   = 8 (radio button)
 * right                 = 9 (radio button)
 * left                  = 3 (radio button)
 * lower right           = 10 (radio button)
 * lower left            = 11 (radio button)
 * upper right           = 12 (radio button)
 * upper left            = 13 (radio button)
 * boxed                 = 14 (radio button)
 * characteristics       = 27 (popup)
 * always drawn          = 29 (check)
 * body only             = 30 (check)
 */
void us_getinfotext(HIGHLIGHT *high, INTBIG canspecialize)
{
	INTBIG itemHit, j, haslw, trandessize, skipchrs, changed,
		posnotoffset, cantbox;
	INTSML x, y;
	char *str, *newstr, *pt, *paramstart[10];
	RECTAREA itemRect;
	INTBIG descript, i, newval, newdescript, objtype, lambda,
		xcur, ycur, newbit, lindex, xc, yc, nlx, nhx, nly, nhy;
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *var;
	REGISTER DIALOG *whichdialog;
	static char *sizenames[12] = {"4 Points", "6 Points", "8 Points", "10 Points", "12 Points",
		"14 Points", "16 Points", "18 Points", "20 Points", "Small", "Medium", "Large"};
	static char *charnames[15] = {"Input", "Output", "Bidirectional", "Power", "Ground",
		"Reference Output", "Reference Input", "Clock", "Clock phase 1", "Clock phase 2",
		"Clock phase 3", "Clock phase 4", "Clock phase 5", "Clock phase 6", "Unknown"};
	static UINTBIG charlist[15] = {INPORT, OUTPORT, BIDIRPORT, PWRPORT, GNDPORT,
		REFOUTPORT, REFINPORT, CLKPORT, C1PORT, C2PORT, C3PORT, C4PORT, C5PORT, C6PORT};
	static INTBIG sizelist[12] = {TXT4P, TXT6P, TXT8P, TXT10P, TXT12P, TXT14P, TXT16P,
		TXT18P, TXT20P, TXTSMALL, TXTMEDIUM, TXTLARGE};
	static struct butlist poslist[10] =
	{
		{VTPOSCENT,       6},
		{VTPOSUP,         7},
		{VTPOSDOWN,       8},
		{VTPOSLEFT,       9},
		{VTPOSRIGHT,      3},
		{VTPOSUPLEFT,    10},
		{VTPOSUPRIGHT,   11},
		{VTPOSDOWNLEFT,  12},
		{VTPOSDOWNRIGHT, 13},
		{VTPOSBOXED,     14}
	};

	/* display text */
	skipchrs = 0;
	posnotoffset = 0;
	cantbox = 0;
	if (high->fromvar != NOVARIABLE)
	{
		if (high->fromgeom->entrytype == OBJNODEINST) objtype = VNODEINST; else
			objtype = VARCINST;

		/* special case if known variables are selected */
		if (canspecialize != 0)
		{
			if (high->fromvar->key == sch_resistancekey)
			{
				i = us_resistancedlog("Resistor", paramstart, (char *)high->fromvar->addr);
				if (i == 0) return;
				us_pushhighlight();
				us_clearhighlightcount();
				startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
				(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype, sch_resistancekey,
					(INTBIG)paramstart[0], high->fromvar->type);
				endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
				(void)us_pophighlight(0);
				return;
			}
			if (high->fromvar->key == sch_capacitancekey)
			{
				str = (char *)high->fromvar->addr;
				while (*str == ' ') str++;
				if (*str == 'e' || *str == 'E') str++;
				i = us_capacitancedlog(paramstart, str);
				if (i == 0) return;
				us_pushhighlight();
				us_clearhighlightcount();
				startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
				(void)initinfstr();
				for(pt = (char *)high->fromvar->addr; pt != str; pt++)
					(void)addtoinfstr(*pt);
				(void)addstringtoinfstr(paramstart[0]);
				(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype, sch_capacitancekey,
					(INTBIG)returninfstr(), high->fromvar->type);
				endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
				(void)us_pophighlight(0);
				return;
			}
			if (high->fromvar->key == sch_inductancekey)
			{
				i = us_inductancedlog(paramstart, (char *)high->fromvar->addr);
				if (i == 0) return;
				us_pushhighlight();
				us_clearhighlightcount();
				startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
				(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype, sch_inductancekey,
					(INTBIG)paramstart[0], high->fromvar->type);
				endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
				(void)us_pophighlight(0);
				return;
			}
			if (high->fromvar->key == sch_sourcekey)
			{
				i = us_sourcedlog(paramstart, (char *)high->fromvar->addr);
				if (i == 0) return;
				us_pushhighlight();
				us_clearhighlightcount();
				startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
				(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype, sch_sourcekey,
					(INTBIG)paramstart[0], high->fromvar->type);
				endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
				(void)us_pophighlight(0);
				return;
			}
			if (high->fromvar->key == sch_diodekey)
			{
				str = (char *)high->fromvar->addr;
				while (*str == ' ') str++;
				if (*str == 'z' || *str == 'Z') str++;
				i = us_areadlog("Diode", paramstart, str);
				if (i == 0) return;
				us_pushhighlight();
				us_clearhighlightcount();
				startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
				(void)initinfstr();
				for(pt = (char *)high->fromvar->addr; pt != str; pt++)
					(void)addtoinfstr(*pt);
				(void)addstringtoinfstr(paramstart[0]);
				(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype, sch_diodekey,
					(INTBIG)returninfstr(), high->fromvar->type);
				endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
				(void)us_pophighlight(0);
				return;
			}
			if (high->fromvar->key == sch_twoportkey)
			{
				str = (char *)high->fromvar->addr;
				while (*str == ' ') str++;
				if (*str == 'h' || *str == 'H')
				{
					str++;
					i = us_resistancedlog("CCVS", paramstart, str);
					if (i == 0) return;
					us_pushhighlight();
					us_clearhighlightcount();
					startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)initinfstr();
					for(pt = (char *)high->fromvar->addr; pt != str; pt++)
						(void)addtoinfstr(*pt);
					(void)addstringtoinfstr(paramstart[0]);
					(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype, sch_twoportkey,
						(INTBIG)returninfstr(), high->fromvar->type);
					endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)us_pophighlight(0);
					return;
				} else if (*str == 'g' || *str == 'G')
				{
					str++;
					i = us_vccsdlog(paramstart, str);
					if (i == 0) return;
					us_pushhighlight();
					us_clearhighlightcount();
					startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)initinfstr();
					for(pt = (char *)high->fromvar->addr; pt != str; pt++)
						(void)addtoinfstr(*pt);
					(void)addstringtoinfstr(paramstart[0]);
					(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype, sch_twoportkey,
						(INTBIG)returninfstr(), high->fromvar->type);
					endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)us_pophighlight(0);
					return;
				} else if (*str == 'u' || *str == 'U')
				{
					str++;
					i = us_gaindlog("VCVS", paramstart, str);
					if (i == 0) return;
					us_pushhighlight();
					us_clearhighlightcount();
					startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)initinfstr();
					for(pt = (char *)high->fromvar->addr; pt != str; pt++)
						(void)addtoinfstr(*pt);
					(void)addstringtoinfstr(paramstart[0]);
					(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype, sch_twoportkey,
						(INTBIG)returninfstr(), high->fromvar->type);
					endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)us_pophighlight(0);
					return;
				} else if (*str == 'f' || *str == 'F')
				{
					str++;
					i = us_gaindlog("CCCS", paramstart, str);
					if (i == 0) return;
					us_pushhighlight();
					us_clearhighlightcount();
					startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)initinfstr();
					for(pt = (char *)high->fromvar->addr; pt != str; pt++)
						(void)addtoinfstr(*pt);
					(void)addstringtoinfstr(paramstart[0]);
					(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype, sch_twoportkey,
						(INTBIG)returninfstr(), high->fromvar->type);
					endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)us_pophighlight(0);
					return;
				}
			}
			if (high->fromvar->key == sch_transistortypekey)
			{
				str = (char *)high->fromvar->addr;
				haslw = trandessize = 0;
				if (namesamen(str, "nmos", 4) == 0)  { trandessize = 4;  haslw = 1;   } else
				if (namesamen(str, "dmos", 4) == 0)  { trandessize = 4;  haslw = 1;   } else
				if (namesamen(str, "pmos", 4) == 0)  { trandessize = 4;  haslw = 1;   } else
				if (namesamen(str, "npn", 3) == 0)   { trandessize = 3;  } else
				if (namesamen(str, "pnp", 3) == 0)   { trandessize = 3;  } else
				if (namesamen(str, "njfet", 5) == 0) { trandessize = 5;  } else
				if (namesamen(str, "pjfet", 5) == 0) { trandessize = 5;  } else
				if (namesamen(str, "dmes", 4) == 0)  { trandessize = 4;  } else
				if (namesamen(str, "emes", 4) == 0)  { trandessize = 4;  }
				if (haslw != 0)
				{
					i = us_widlendlog(paramstart, &str[trandessize]);
					if (i == 0) return;
					us_pushhighlight();
					us_clearhighlightcount();
					startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)initinfstr();
					for(i=0; i<trandessize; i++) (void)addtoinfstr(str[i]);
					(void)addstringtoinfstr(paramstart[0]);
					(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype,
						sch_transistortypekey, (INTBIG)returninfstr(), high->fromvar->type);
					endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)us_pophighlight(0);
					return;
				} else if (trandessize != 0)
				{
					i = us_areadlog("Transistor", paramstart, &str[trandessize]);
					if (i == 0) return;
					us_pushhighlight();
					us_clearhighlightcount();
					startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)initinfstr();
					for(i=0; i<trandessize; i++) (void)addtoinfstr(str[i]);
					(void)addstringtoinfstr(paramstart[0]);
					(void)setvalkey((INTBIG)high->fromgeom->entryaddr.blind, objtype,
						sch_transistortypekey, (INTBIG)returninfstr(), high->fromvar->type);
					endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);
					(void)us_pophighlight(0);
					return;
				}
			}

			/* if this is a displayable text variable, edit it in place */
			if ((high->fromvar->type&VDISPLAY) != 0 &&
				(high->fromvar->type&VTYPE) == VSTRING)
			{
				/* save and clear highlighting */
				us_pushhighlight();
				us_clearhighlightcount();

				/* edit the variable */
				us_editvariabletext(high->fromvar, objtype, (INTBIG)high->fromgeom->entryaddr.blind,
					makename(high->fromvar->key));

				/* restore highlighting */
				(void)us_pophighlight(0);
				return;
			}
		}

		/* display the text dialog box */
		whichdialog = &us_showtextdialog;
		if (DiaInitDialog(&us_showtextdialog) != 0) return;
		DiaSetPopup(5, 12, sizenames);
		DiaDimItem(27);

		(void)initinfstr();
		if (high->fromgeom->entrytype == OBJNODEINST)
		{
			if (high->fromgeom->entryaddr.ni->proto == gen_invispinprim)
			{
				if (namesame(makename(high->fromvar->key), "VERILOG_code") == 0)
					(void)addstringtoinfstr("Verilog code"); else
				if (namesame(makename(high->fromvar->key), "VERILOG_declaration") == 0)
					(void)addstringtoinfstr("Verilog declaration"); else
						(void)addstringtoinfstr("Text on Message Node");
				DiaSetText(19, "X position:");
				DiaSetText(20, "Y position:");
				posnotoffset = 1;
			} else
			{
				(void)addstringtoinfstr("Text on node ");
				(void)addstringtoinfstr(describenodeinst(high->fromgeom->entryaddr.ni));
			}
			boundobj(high->fromgeom, &nlx, &nhx, &nly, &nhy);
			if (nhx == nlx || nhy == nly)
			{
				cantbox = 1;
				DiaDimItem(14);
			}
		} else
		{
			(void)addstringtoinfstr("Text on arc ");
			(void)addstringtoinfstr(describearcinst(high->fromgeom->entryaddr.ai));
		}
		DiaSetText(26, returninfstr());
		lindex = -1;
		DiaEditControl(4);
		if ((high->fromvar->type&VISARRAY) != 0)
		{
			if (getlength(high->fromvar) == 1) lindex = 0; else
			{
				DiaNoEditControl(4);
				DiaUnDimItem(27);
			}
		} else skipchrs = (high->fromvar->type&VLENGTH) >> VLENGTHSH;
		str = describevariable(high->fromvar, (INTSML)lindex, -1);
		descript = high->fromvar->textdescript;
		DiaSetText(-4, &str[skipchrs]);
	} else if (high->fromport != NOPORTPROTO)
	{
		/* display the port dialog box */
		whichdialog = &us_portinfodialog;
		if (DiaInitDialog(&us_portinfodialog) != 0) return;
		DiaSetPopup(5, 12, sizenames);
		DiaSetPopup(27, 15, charnames);

		str = high->fromport->protoname;
		descript = high->fromport->textdescript;
		objtype = VNODEINST;

		/* set the port characteristics button */
		for(i=0; i<15; i++)
			if (charlist[i] == (high->fromport->userbits&STATEBITS))
		{
			DiaSetPopupEntry(27, i);
			break;
		}
		if ((high->fromport->userbits&PORTDRAWN) != 0) DiaSetControl(29, 1);
		if ((high->fromport->userbits&BODYONLY) != 0) DiaSetControl(30, 1);
		DiaSetText(-4, str);
		DiaEditControl(4);
	} else if (high->fromgeom->entrytype == OBJNODEINST)
	{
		/* display the text dialog box */
		whichdialog = &us_showtextdialog;
		if (DiaInitDialog(&us_showtextdialog) != 0) return;
		DiaSetPopup(5, 12, sizenames);

		ni = high->fromgeom->entryaddr.ni;
		str = describenodeproto(ni->proto);
		descript = ni->textdescript;
		objtype = VNODEINST;
		DiaNoEditControl(4);
		DiaSetText(4, str);
		DiaSetText(26, "Text information for facet name:");
	} else return;

	/* if this is on a node and the node is rotated, modify grab-point */
	if (high->fromgeom->entrytype == OBJNODEINST)
	{
		ni = high->fromgeom->entryaddr.ni;
		if (ni->rotation != 0 || ni->transpose != 0)
		{
			descript = us_rotatedescript(high->fromgeom, descript);
		}
	}

	/* set the orientation button */
	for(i=0; i<10; i++)
		if (poslist[i].value == (descript&VTPOSITION))
	{
		DiaSetControl(poslist[i].button, 1);
		break;
	}

	/* set the size button */
	for(i=0; i<12; i++)
		if (sizelist[i] == ((descript&VTSIZE) >> VTSIZESH))
	{
		DiaSetPopupEntry(5, i);
		break;
	}

	/* set offsets */
	if (posnotoffset != 0)
	{
		ni = high->fromgeom->entryaddr.ni;
		DiaSetText(17, latoa((ni->lowx + ni->highx) / 2));
		DiaSetText(18, latoa((ni->lowy + ni->highy) / 2));
	} else
	{
		lambda = figurelambda(high->fromgeom);
		i = (descript&VTXOFF) >> VTXOFFSH;
		if ((descript&VTXOFFNEG) != 0) i = -i;
		DiaSetText(17, latoa(i * lambda / 4));
		j = (descript&VTYOFF) >> VTYOFFSH;
		if ((descript&VTYOFFNEG) != 0) j = -j;
		DiaSetText(18, latoa(j * lambda / 4));
	}

	/* loop until done */
	changed = 0;
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL ||
			(itemHit == 27 && whichdialog == &us_showtextdialog)) break;
		if (itemHit >= 21 && itemHit <= 25)
		{
			DiaItemRect(itemHit, &itemRect);
			DiaGetMouse(&x, &y);
			itemHit = (itemHit-21) * 2;
			if (y > (itemRect.top + itemRect.bottom) / 2) itemHit++;
			itemHit = poslist[itemHit].button;
			if (itemHit == 14 && cantbox != 0) continue;
		}
		for(i=0; i<10; i++) if (itemHit == poslist[i].button)
		{
			for(j=0; j<10; j++) DiaSetControl(poslist[j].button, 0);
			DiaSetControl(itemHit, 1);
			changed++;
			continue;
		}
		if (itemHit == 29 || itemHit == 30)
		{
			DiaSetControl(itemHit, 1-DiaGetControl(itemHit));
			changed++;
		}
		if (itemHit == 5 || itemHit == 17 || itemHit == 18 ||
			(itemHit == 27 && whichdialog == &us_portinfodialog)) changed++;
	}

	if (itemHit != CANCEL)
	{
		/* get new port characteristics if applicable */
		if (whichdialog == &us_portinfodialog && changed != 0)
		{
			newbit = high->fromport->userbits;
			i = DiaGetPopupEntry(27);
			newbit = (newbit & ~STATEBITS) | charlist[i];
			if (DiaGetControl(29) == 0) newbit &= ~PORTDRAWN; else
				newbit |= PORTDRAWN;
			if (DiaGetControl(30) == 0) newbit &= ~BODYONLY; else
				newbit |= BODYONLY;
			if (newbit != (INTBIG)high->fromport->userbits)
			{
				high->fromport->userbits = newbit;
				changeallports(high->fromport);
			}
		}

		/* get the new descriptor */
		newdescript = descript;
		if (changed != 0)
		{
			xcur = atola(DiaGetText(17));
			ycur = atola(DiaGetText(18));
			if (posnotoffset != 0)
			{
				ni = high->fromgeom->entryaddr.ni;
				us_pushhighlight();
				us_clearhighlightcount();
				startobjectchange((INTBIG)ni, VNODEINST);
				xc = (ni->lowx + ni->highx) / 2;
				yc = (ni->lowy + ni->highy) / 2;
				modifynodeinst(ni, xcur-xc, ycur-yc, xcur-xc, ycur-yc, 0, 0);
				endobjectchange((INTBIG)ni, VNODEINST);
				(void)us_pophighlight(1);
			} else
			{
				newdescript = us_setdescriptoffset(newdescript, xcur*4/lambda, ycur*4/lambda);
			}
			j = DiaGetPopupEntry(5);
			newdescript = (newdescript & ~VTSIZE) | (sizelist[j] << VTSIZESH);
			for(j=0; j<10; j++)
				if (DiaGetControl(poslist[j].button) != 0) break;
			newdescript = (newdescript & ~VTPOSITION) | poslist[j].value;

			/* if this is on a node and the node is rotated, modify grab-point */
			if (high->fromgeom->entrytype == OBJNODEINST)
			{
				ni = high->fromgeom->entryaddr.ni;
				if (ni->rotation != 0 || ni->transpose != 0)
				{
					newdescript = us_rotatedescriptI(high->fromgeom, newdescript);
				}
			}
		}

		/* get the new name */
		newstr = DiaGetText(4);

		/* see if changes were made */
		if (newdescript != descript || strcmp(&str[skipchrs], newstr) != 0)
		{
			/* save highlighting */
			us_pushhighlight();
			us_clearhighlightcount();

			/* set the new descriptor */
			startobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);

			/* handle changes */
			if (newdescript != descript) us_modifytextdescript(high, newdescript);
			if (strcmp(&str[skipchrs], newstr) != 0)
			{
				/* message field changed */
				var = high->fromvar;
				if (var != NOVARIABLE)
				{
					switch (var->type&VTYPE)
					{
						case VINTEGER:
						case VSHORT:
							newval = myatoi(newstr);
							break;
						case VSTRING:
							if (skipchrs == 0) newval = (INTBIG)newstr; else
							{
								(void)initinfstr();
								for(j=0; j<skipchrs; j++) (void)addtoinfstr(str[j]);
								(void)addstringtoinfstr(newstr);
								newval = (INTBIG)returninfstr();
							}
							break;
						case VFLOAT:
						case VDOUBLE:
							newval = castint((float)atof(newstr));
							break;
					}
					if ((var->type&VISARRAY) == 0)
					{
						(void)setval((INTBIG)high->fromgeom->entryaddr.blind, objtype,
							makename(var->key), newval, var->type);
					} else
					{
#if 0
						(void)setind((INTBIG)high->fromgeom->entryaddr.blind, objtype,
							makename(var->key), 0, newval);
#endif
					}
				} else if (high->fromport != NOPORTPROTO)
				{
					us_renameport(high->fromport, newstr);
				}
			}

			/* redisplay the text */
			endobjectchange((INTBIG)high->fromgeom->entryaddr.blind, objtype);

			/* restore highlighting */
			(void)us_pophighlight(1);
		}
	}
	DiaDoneDialog();
	if (high->fromvar != NOVARIABLE && itemHit == 27)
	{
		/* save and clear highlighting */
		us_pushhighlight();
		us_clearhighlightcount();

		/* edit the variable */
		us_editvariabletext(high->fromvar, objtype,
			(INTBIG)high->fromgeom->entryaddr.blind, makename(high->fromvar->key));

		/* restore highlighting */
		(void)us_pophighlight(0);
	}
}

/*
 * special case for the "Arc Get Info" dialog
 * rigid                 = 3 (check)
 * fixed-angle           = 4 (check)
 * type                  = 5 (message)
 * network               = 6 (message)
 * name                  = 7 (edit text)
 * angle                 = 8 (message)
 * head node             = 9 (message)
 * bus size              = 10 (message)
 * tail node             = 11 (message)
 * slidable              = 19 (check)
 * negated               = 20 (check)
 * directional           = 21 (check)
 * ends extend           = 22 (check)
 * ignore head           = 23 (check)
 * ignore tail           = 24 (check)
 * reverse               = 25 (check)
 * temporarily           = 26 (check)
 * width                 = 27 (edit text)
 * constraints           = 29 (message)
 * head coordinate       = 31 (message)
 * see head              = 32 (button)
 * tail coordinate       = 34 (message)
 * see tail              = 35 (button)
 * head info             = 36 (button)
 * tail info             = 37 (button)
 * easy to select        = 38 (check)
 */
void us_getinfoarc(ARCINST *ai)
{
	INTBIG itemHit, rigstate, oldrigstate, changed;
	char ent[50], *str, *newstr;
	INTBIG i, wid, oldwid, end;
	REGISTER VARIABLE *var;
	HIGHLIGHT high;
	extern INTBIG cla_changeclock;	/* from layout constraints: "conlay.c" */

	if (DiaInitDialog(&us_showarcdialog) != 0) return;

	/* set prototype name */
	DiaSetText(5, ai->proto->protoname);

	/* load the network and arc names if any */
	var = getvalkey((INTBIG)ai, VARCINST, VSTRING, el_arc_name);
	if (var != NOVARIABLE)
	{
		str = (char *)var->addr;
		DiaSetText(-7, str);
	} else str = "";
	if (ai->network != NONETWORK && ai->network->namecount != 0)
		DiaSetText(6, describenetwork(ai->network)); else
			DiaSetText(6, "*** NONE ***");

	/* load the width */
	oldwid = ai->width - arcwidthoffset(ai);
	DiaSetText(27, latoa(oldwid));

	/* load the position */
	(void)sprintf(ent, "(%s,%s)", latoa(ai->end[1].xpos), latoa(ai->end[1].ypos));
	DiaSetText(31, ent);
	DiaSetText(9, describenodeinst(ai->end[1].nodeinst));
	(void)sprintf(ent, "(%s,%s)", latoa(ai->end[0].xpos), latoa(ai->end[0].ypos));
	DiaSetText(34, ent);
	DiaSetText(11, describenodeinst(ai->end[0].nodeinst));

	/* load angle */
	(void)sprintf(ent, " %ld", (ai->userbits&AANGLE) >> AANGLESH);
	DiaSetText(8, ent);

	/* load the selectability factor */
	if ((ai->userbits&HARDSELECTA) == 0) DiaSetControl(38, 1);

	/* load bus width if any */
	if (ai->network != NONETWORK && ai->network->signals > 1)
	{
		(void)sprintf(ent, " %d", ai->network->signals);
		DiaSetText(10, ent);
	} else DiaSetText(10, "N/A");

	/* set the constraint buttons */
	if (el_curconstraint == cla_constraint)
	{
		oldrigstate = 1;
		if (((ai->userbits&FIXED) == 0 || ai->changed == cla_changeclock+3) &&
			ai->changed != cla_changeclock+2)
		{
			if (ai->changed == cla_changeclock+3) oldrigstate = 3;
		} else
		{
			if (ai->changed == cla_changeclock+2) oldrigstate = 2; else
				oldrigstate = 0;
		}
		switch (oldrigstate)
		{
			case 0: DiaSetControl(3, 1);    break;
			case 1:                         break;
			case 2: DiaSetControl(3, 1);
			case 3: DiaSetControl(26, 1);   break;
		}
		if ((ai->userbits&FIXANG) != 0) DiaSetControl(4, 1);
		if ((ai->userbits&CANTSLIDE) == 0) DiaSetControl(19, 1);
	} else
	{
		DiaDimItem(3);
		DiaDimItem(4);
		DiaDimItem(19);
		DiaDimItem(26);
		newstr = (char *)(*(el_curconstraint->request))("describearc", (INTBIG)ai);
		if (*newstr != 0)
		{
			(void)initinfstr();
			(void)addstringtoinfstr("Constraint: ");
			(void)addstringtoinfstr(newstr);
			DiaSetText(29, returninfstr());
		}
	}
	if ((ai->userbits&ISNEGATED) != 0) DiaSetControl(20, 1);
	if ((ai->userbits&ISDIRECTIONAL) != 0) DiaSetControl(21, 1);
	if ((ai->userbits&NOEXTEND) == 0) DiaSetControl(22, 1);
	if ((ai->userbits&NOTEND1) != 0) DiaSetControl(23, 1);
	if ((ai->userbits&NOTEND0) != 0) DiaSetControl(24, 1);
	if ((ai->userbits&REVERSEEND) != 0) DiaSetControl(25, 1);

	/* if the arc can't be edited, disable all changes */
	if (us_canedit(ai->parent, NONODEPROTO, 0) != 0)
	{
		DiaDimItem(3);
		DiaDimItem(4);
		DiaDimItem(7);
		DiaDimItem(8);
		DiaDimItem(19);
		DiaDimItem(20);
		DiaDimItem(21);
		DiaDimItem(22);
		DiaDimItem(23);
		DiaDimItem(24);
		DiaDimItem(25);
		DiaDimItem(26);
		DiaDimItem(27);
		DiaDimItem(38);
	}

	/* loop until done */
	changed = 0;
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL || itemHit == 36 || itemHit == 37) break;
		if (itemHit == 3 || itemHit == 4 || itemHit == 19 || itemHit == 20 || itemHit == 21 ||
			itemHit == 22 || itemHit == 23 || itemHit == 24 || itemHit == 25 ||
			itemHit == 26 || itemHit == 38)
		{
			DiaSetControl(itemHit, 1-DiaGetControl(itemHit));
			changed++;
			continue;
		}
		if (itemHit == 7 || itemHit == 27)
		{
			changed++;
			continue;
		}
		if (itemHit == 32 || itemHit == 35)
		{
			/* highlight an end */
			if (itemHit == 32) end = 1; else end = 0;
			us_clearhighlightcount();
			high.status = HIGHFROM;
			high.facet = ai->parent;
			high.fromgeom = ai->end[end].nodeinst->geom;
			high.fromport = ai->end[end].portarcinst->proto;
			high.frompoint = 0;
			high.fromvar = NOVARIABLE;
			(void)us_addhighlight(&high);
			us_showallhighlight();
			us_endchanges(NOWINDOWPART);
			continue;
		}
	}

	if (itemHit != CANCEL && changed != 0)
	{
		us_pushhighlight();
		us_clearhighlightcount();
		startobjectchange((INTBIG)ai, VARCINST);
		wid = atola(DiaGetText(27));
		if (wid != oldwid && wid >= 0)
		{
			wid = arcwidthoffset(ai) + wid;
			if (wid != ai->width)
			{
				startobjectchange((INTBIG)ai, VARCINST);
				(void)modifyarcinst(ai, wid - ai->width, 0, 0, 0, 0);
				endobjectchange((INTBIG)ai, VARCINST);
			}
		}
		if (DiaGetControl(3) == 0)
		{
			if (DiaGetControl(26) == 0) rigstate = 1; else
				rigstate = 3;
		} else
		{
			if (DiaGetControl(26) == 0) rigstate = 0; else
				rigstate = 2;
		}
		if (rigstate != oldrigstate) switch (rigstate)
		{
			case 0:
				(void)(*el_curconstraint->setobject)((INTBIG)ai,VARCINST,CHANGETYPEREMOVETEMP,0);
				(void)(*el_curconstraint->setobject)((INTBIG)ai,VARCINST,CHANGETYPERIGID,0);
				break;
			case 1:
				(void)(*el_curconstraint->setobject)((INTBIG)ai,VARCINST,CHANGETYPEREMOVETEMP,0);
				(void)(*el_curconstraint->setobject)((INTBIG)ai,VARCINST,CHANGETYPEUNRIGID,0);
				break;
			case 2:
				(void)(*el_curconstraint->setobject)((INTBIG)ai,VARCINST,CHANGETYPETEMPRIGID,0);
				break;
			case 3:
				(void)(*el_curconstraint->setobject)((INTBIG)ai,VARCINST,CHANGETYPETEMPUNRIGID,0);
				break;
		}
		i = DiaGetControl(4);
		if (i != 0 && (ai->userbits&FIXANG) == 0)
			(void)(*el_curconstraint->setobject)((INTBIG)ai,VARCINST,CHANGETYPEFIXEDANGLE,0);
		if (i == 0 && (ai->userbits&FIXANG) != 0)
			(void)(*el_curconstraint->setobject)((INTBIG)ai,VARCINST,CHANGETYPENOTFIXEDANGLE,0);
		i = DiaGetControl(19);
		if (i != 0 && (ai->userbits&CANTSLIDE) != 0)
			(void)(*el_curconstraint->setobject)((INTBIG)ai,VARCINST,CHANGETYPESLIDABLE,0);
		if (i == 0 && (ai->userbits&CANTSLIDE) == 0)
			(void)(*el_curconstraint->setobject)((INTBIG)ai,VARCINST,CHANGETYPENOTSLIDABLE,0);
		i = DiaGetControl(20);
		if (i != 0 && (ai->userbits&ISNEGATED) == 0)
			(void)setval((INTBIG)ai, VARCINST,"userbits", ai->userbits | ISNEGATED, VINTEGER);
		if (i == 0 && (ai->userbits&ISNEGATED) != 0)
			(void)setval((INTBIG)ai, VARCINST,"userbits", ai->userbits & ~ISNEGATED, VINTEGER);
		i = DiaGetControl(21);
		if (i != 0 && (ai->userbits&ISDIRECTIONAL) == 0)
			(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits | ISDIRECTIONAL, VINTEGER);
		if (i == 0 && (ai->userbits&ISDIRECTIONAL) != 0)
			(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits & ~ISDIRECTIONAL, VINTEGER);
		i = DiaGetControl(22);
		if (i != 0 && (ai->userbits&NOEXTEND) != 0)
			(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits & ~NOEXTEND, VINTEGER);
		if (i == 0 && (ai->userbits&NOEXTEND) == 0)
			(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits | NOEXTEND, VINTEGER);
		i = DiaGetControl(23);
		if (i != 0 && (ai->userbits&NOTEND1) == 0)
			(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits | NOTEND1, VINTEGER);
		if (i == 0 && (ai->userbits&NOTEND1) != 0)
			(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits & ~NOTEND1, VINTEGER);
		i = DiaGetControl(24);
		if (i != 0 && (ai->userbits&NOTEND0) == 0)
			(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits | NOTEND0, VINTEGER);
		if (i == 0 && (ai->userbits&NOTEND0) != 0)
			(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits & ~NOTEND0, VINTEGER);
		i = DiaGetControl(25);
		if (i != 0 && (ai->userbits&REVERSEEND) == 0)
			(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits | REVERSEEND, VINTEGER);
		if (i == 0 && (ai->userbits&REVERSEEND) != 0)
			(void)setval((INTBIG)ai, VARCINST, "userbits", ai->userbits & ~REVERSEEND, VINTEGER);
		if (DiaGetControl(38) != 0) ai->userbits &= ~HARDSELECTA; else
			ai->userbits |= HARDSELECTA;
		newstr = DiaGetText(7);
		while (*newstr == ' ') newstr++;
		if (strcmp(str, newstr) != 0)
		{
			if (*newstr == 0) us_setarcname(ai, (char *)0); else
				us_setarcname(ai, newstr);
		}
		endobjectchange((INTBIG)ai, VARCINST);
		(void)us_pophighlight(0);
	}
	DiaDoneDialog();
	if (itemHit == 36 || itemHit == 37)
	{
		if (itemHit == 36) end = 1; else end = 0;
		us_clearhighlightcount();
		high.status = HIGHFROM;
		high.facet = ai->parent;
		high.fromgeom = ai->end[end].nodeinst->geom;
		high.fromport = ai->end[end].portarcinst->proto;
		high.frompoint = 0;
		high.fromvar = NOVARIABLE;
		(void)us_addhighlight(&high);
		us_showallhighlight();
		us_endchanges(NOWINDOWPART);
		us_getinfonode(ai->end[end].nodeinst, ai->end[end].portarcinst->proto);
	}
}

/*
 * special case for the "Node Get Info" dialog
 * expanded              = 4 (radio button)
 * unexpanded            = 5 (radio button)
 * type                  = 6 (message)
 * name                  = 7 (edit text)
 * connection list       = 3 (user item)
 * X size                = 8 (edit text)
 * Y size                = 9 (edit text)
 * rotation              = 10 (edit text)
 * transposed            = 11 (check)
 * X position            = 12 (edit text)
 * X position            = 13 (edit text)
 * Special value value   = 22 (stat/edit text)
 * Special feature title = 23 (message)
 * Special feature value = 24 (message)
 * Special value title   = 25 (message)
 * See arc               = 26 (button)
 * Size information      = 27 (message)
 * Position information  = 28 (message)
 * Easy to Select        = 29 (check)
 */
void us_getinfonode(NODEINST *ni, PORTPROTO *fromport)
{
	INTBIG itemHit, r, t, haslw, wantedit, wantpopup, holdstrace,
		fun, changed, positionchanged, xyrev;
	char ent[50], *str, *newstr, *namestr, *extra;
	double startoffset, endangle, newstart, newend;
	INTBIG descript, i, plx, ply, phx, phy, wid, len, xs, ys, xc, yc, dx, dy,
		nlx, nhx, nly, nhy, cox, coy, serpwidth, lineno, highlineno, techedrel;
	HIGHLIGHT high;
	REGISTER NODEPROTO *np, *onp;
	REGISTER PORTPROTO *pp;
	REGISTER PORTARCINST *pi;
	REGISTER ARCINST *ai, *selectedai;
	REGISTER VARIABLE *var;
	XARRAY trans;
	static char *diodetype[2] = {"Regular", "Zener"};
	static char *capacitortype[2] = {"Regular", "Electrolytic"};
	static char *sourcetype[11] = {"Unknown", "Voltage", "Current", "Metered Current",
		"Transient", "DC Analysis", "AC Analysis", "Nodeset", "Extension", "Bulk", "Special"};
	static char *twoporttype[5] = {"VCCS", "CCVS", "VCVS", "CCCS", "Transmission"};
	static char *transistortype[9] = {"N-type MOS", "Depletion MOS", "P-type MOS",
		"NPN Bipolar", "PNP Bipolar", "N-type JFET", "P-type JFET", "Depletion MESFET",
		"Enhancement MESFET"};
	static char *colortype[25] = {"Overlappable-1", "Overlappable-2", "Overlappable-3",
		"Overlappable-4", "Overlappable-5", "White", "Black", "Red", "Blue", "Green",
		"Cyan", "Magenta", "Yellow", "Gray", "Orange", "Purple", "Brown", "Light Gray",
		"Dark Gray", "Light Red", "Dark Red", "Light Green", "Dark Green", "Light Blue",
		"Dark Blue"};

	/* node getinfo: determine what will be in the dialog box */
	np = ni->proto;
	wantedit = wantpopup = holdstrace = 0;
	serpwidth = -1;
	var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER|VISARRAY, el_trace);
	if (var != NOVARIABLE) holdstrace = 1;
	fun = (np->userbits&NFUNCTION) >> NFUNCTIONSH;
	if (fun == NPTRANMOS || fun == NPTRADMOS || fun == NPTRAPMOS)
	{
		if (np->primindex != 0 && (np->userbits&HOLDSTRACE) != 0 && holdstrace != 0)
		{
			serpwidth = 0;
			wantedit = 1;
		}
	}
	techedrel = us_tecedgetoption(ni);
	if (np == sch_resistorprim || np == sch_inductorprim ||
		np == sch_bboxprim || np == art_circleprim)
			wantedit = 1;
	if (np == sch_diodeprim || np == sch_capacitorprim || np == sch_sourceprim ||
		np == sch_twoportprim || np == sch_transistorprim ||
			(np->primindex != 0 && np->tech == art_tech && techedrel < 0))
				wantedit = wantpopup = 1;
	if (wantedit == 0) us_shownodedialogitems[22-1].type = MESSAGE; else
		us_shownodedialogitems[22-1].type = EDITTEXT;
	if (wantpopup == 0) us_shownodedialogitems[24-1].type = MESSAGE; else
		us_shownodedialogitems[24-1].type = POPUP;

	/* display the dialog box */
	if (DiaInitDialog(&us_shownodedialog) != 0) return;
	DiaInitTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT|SCHORIZBAR);

	/* load the prototype name */
	DiaSetText(6, describenodeproto(np));

	/* load the node name if any */
	var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
	if (var != NOVARIABLE)
	{
		namestr = (char *)var->addr;
		DiaSetText(-7, namestr);
	} else namestr = "";

	/* load the size */
	xyrev = 0;
	if ((fun == NPTRANMOS || fun == NPTRADMOS || fun == NPTRAPMOS) && serpwidth < 0)
	{
		DiaSetText(16, "Width:");
		DiaSetText(17, "Length:");
	} else
	{
		if (ni->transpose == 0)
		{
			if (ni->rotation == 900 || ni->rotation == 2700) xyrev = 1;
		} else
		{
			if (ni->rotation == 0 || ni->rotation == 1800) xyrev = 1;
		}
		if (xyrev != 0) DiaSetText(27, "Transformed:"); else
		{
			if ((ni->rotation%900) != 0)
				DiaSetText(27, "Untransformed:");
		}
	}
	nodesizeoffset(ni, &plx, &ply, &phx, &phy);
	if (xyrev == 0)
	{
		DiaSetText(8, latoa(ni->highx-ni->lowx-plx-phx));
		DiaSetText(9, latoa(ni->highy-ni->lowy-ply-phy));
	} else
	{
		DiaSetText(9, latoa(ni->highx-ni->lowx-plx-phx));
		DiaSetText(8, latoa(ni->highy-ni->lowy-ply-phy));
	}

	/* set the expansion button */
	if (np->primindex == 0)
	{
		if ((ni->userbits&NEXPAND) != 0) i = 4; else i = 5;
		DiaSetControl(i, 1);
		DiaNoEditControl(8);
		DiaNoEditControl(9);
	} else
	{
		DiaDimItem(4);
		DiaDimItem(5);
		DiaEditControl(8);
		DiaEditControl(9);
	}

	/* no size controls if this is polygonally defined */
	if (holdstrace != 0)
	{
		DiaDimItem(8);
		DiaDimItem(9);
	}

	/* load the position */
	if ((us_useroptions&CENTEREDPRIMITIVES) == 0)
	{
		corneroffset(ni, np, ni->rotation, ni->transpose, &cox, &coy, 0);
		DiaSetText(12, latoa(ni->lowx+cox));
		DiaSetText(13, latoa(ni->lowy+coy));
		DiaSetText(28, "Lower-left:");
	} else
	{
		var = getvalkey((INTBIG)ni->proto, VNODEPROTO, VINTEGER|VISARRAY, el_prototype_center);
		if (var != NOVARIABLE)
		{
			dx = ((INTBIG *)var->addr)[0] + (ni->lowx+ni->highx)/2 -
				(ni->proto->lowx+ni->proto->highx)/2;
			dy = ((INTBIG *)var->addr)[1] + (ni->lowy+ni->highy)/2 -
				(ni->proto->lowy+ni->proto->highy)/2;
			makerot(ni, trans);
			xform(dx, dy, &cox, &coy, trans);
			DiaSetText(12, latoa(cox));
			DiaSetText(13, latoa(coy));
			DiaSetText(28, "Facet center:");
		} else
		{
			DiaSetText(12, latoa((ni->lowx+ni->highx)/2));
			DiaSetText(13, latoa((ni->lowy+ni->highy)/2));
			DiaSetText(28, "Center:");
		}
	}

	/* load rotation */
	DiaSetText(10, frtoa(ni->rotation*WHOLE/10));
	if (ni->transpose != 0) DiaSetControl(11, 1);

	/* load easy of selection */
	if ((ni->userbits&HARDSELECTN) == 0) DiaSetControl(29, 1);
	if (ni->proto->primindex == 0 && (us_useroptions&NOINSTANCESELECT) != 0)
		DiaDimItem(29);

	/* load special features, if any */
	if (serpwidth == 0)
	{
		transistorsize(ni, &len, &wid);
		if (len != -1 && wid != -1)
		{
			DiaSetText(23, "Transistor Width:");
			DiaSetText(24, latoa(wid));
			DiaSetText(25, "Transistor Length:");
			DiaSetText(22, latoa(len));
			serpwidth = len;
		}
	}
	if (np == sch_diodeprim)
	{
		DiaSetText(25, "Diode size:");
		DiaSetText(23, "Diode type:");
		DiaSetPopup(24, 2, diodetype);
		var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, sch_diodekey);
		if (var != NOVARIABLE)
		{
			str = (char *)var->addr;
			while (*str == ' ') str++;
			if (*str == 'z' || *str == 'Z')
			{
				str++;
				DiaSetPopupEntry(24, 1);
			}
			DiaSetText(22, str);
		} else DiaSetText(22, "0");
	}
	if (np == sch_capacitorprim)
	{
		DiaSetText(25, "Capacitance:");
		DiaSetText(23, "Capacitor type:");
		DiaSetPopup(24, 2, capacitortype);
		var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, sch_capacitancekey);
		if (var != NOVARIABLE)
		{
			str = (char *)var->addr;
			while (*str == ' ') str++;
			if (*str == 'e' || *str == 'E')
			{
				str++;
				DiaSetPopupEntry(24, 1);
			}
			DiaSetText(22, str);
		} else DiaSetText(22, "0");
	}
	if (np == sch_resistorprim)
	{
		DiaSetText(25, "Resistance:");
		var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, sch_resistancekey);
		if (var != NOVARIABLE) DiaSetText(22, (char *)var->addr); else
			DiaSetText(22, "0");
	}
	if (np == sch_inductorprim)
	{
		DiaSetText(25, "Inductance:");
		var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, sch_inductancekey);
		if (var != NOVARIABLE) DiaSetText(22, (char *)var->addr); else
			DiaSetText(22, "0");
	}
	if (np == sch_bboxprim)
	{
		DiaSetText(25, "Function:");
		var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, sch_functionkey);
		if (var != NOVARIABLE) DiaSetText(22, (char *)var->addr); else
			DiaSetText(22, "");
	}
	if (np == sch_sourceprim)
	{
		DiaSetText(25, "SPICE card:");
		DiaSetText(23, "Source type:");
		DiaSetPopup(24, 11, sourcetype);
		switch (nodefunction(ni, &extra))
		{
			case NPSOURCE:   DiaSetPopupEntry(24, 0);   break;
			case NPSOURCEV:  DiaSetPopupEntry(24, 1);   break;
			case NPSOURCEC:  DiaSetPopupEntry(24, 2);   break;
			case NPSOURCECM: DiaSetPopupEntry(24, 3);   break;
			case NPSOURCET:  DiaSetPopupEntry(24, 4);   break;
			case NPSOURCEDC: DiaSetPopupEntry(24, 5);   break;
			case NPSOURCEAC: DiaSetPopupEntry(24, 6);   break;
			case NPSOURCEN:  DiaSetPopupEntry(24, 7);   break;
			case NPSOURCEX:  DiaSetPopupEntry(24, 8);   break;
			case NPSOURCEB:  DiaSetPopupEntry(24, 9);   break;
			case NPSOURCES:  DiaSetPopupEntry(24, 10);  break;
		}
		if (extra != 0) DiaSetText(22, extra);
	}
	if (np == sch_twoportprim)
	{
		DiaSetText(23, "Two-port type:");
		DiaSetPopup(24, 5, twoporttype);
		var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, sch_twoportkey);
		if (var != NOVARIABLE)
		{
			str = (char *)var->addr;
			while (*str == ' ') str++;
			switch (*str)
			{
				case 'g': DiaSetPopupEntry(24, 0);   DiaSetText(25, "Transconductance:");   break;
				case 'h': DiaSetPopupEntry(24, 1);   DiaSetText(25, "Transresistance:");    break;
				case 'u': DiaSetPopupEntry(24, 2);   DiaSetText(25, "Gain:");               break;
				case 'f': DiaSetPopupEntry(24, 3);   DiaSetText(25, "Gain:");               break;
				case 'l': DiaSetPopupEntry(24, 4);   DiaSetText(25, "Value:");              break;
			}
			DiaSetText(22, &str[1]);
		} else
		{
			DiaSetPopupEntry(24, 5);
			DiaSetText(25, "Value:");
		}
	}
	if (np == sch_transistorprim)
	{
		DiaSetText(23, "Transistor type:");
		DiaSetPopup(24, 9, transistortype);
		var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, sch_transistortypekey);
		if (var != NOVARIABLE)
		{
			str = (char *)var->addr;
			haslw = 0;
			if (namesamen(str, "nmos", 4) == 0)  { str += 4;  haslw = 1;   DiaSetPopupEntry(24, 0); } else
			if (namesamen(str, "dmos", 4) == 0)  { str += 4;  haslw = 1;   DiaSetPopupEntry(24, 1); } else
			if (namesamen(str, "pmos", 4) == 0)  { str += 4;  haslw = 1;   DiaSetPopupEntry(24, 2); } else
			if (namesamen(str, "npn", 3) == 0)   { str += 3;  DiaSetPopupEntry(24, 3); } else
			if (namesamen(str, "pnp", 3) == 0)   { str += 3;  DiaSetPopupEntry(24, 4); } else
			if (namesamen(str, "njfet", 5) == 0) { str += 5;  DiaSetPopupEntry(24, 5); } else
			if (namesamen(str, "pjfet", 5) == 0) { str += 5;  DiaSetPopupEntry(24, 6); } else
			if (namesamen(str, "dmes", 4) == 0)  { str += 4;  DiaSetPopupEntry(24, 7); } else
			if (namesamen(str, "emes", 4) == 0)  { str += 4;  DiaSetPopupEntry(24, 8); }
			if (haslw != 0) DiaSetText(25, "Width/Length:"); else
				DiaSetText(25, "Area:");
			DiaSetText(22, str);
		} else DiaSetText(25, "Size:");
	}
	if (techedrel >= 0)
	{
		DiaSetText(23, "Tech. edit relevance:");
		switch (techedrel)
		{
			case ARCFIXANG:
				DiaSetText(24, "Arc fixed-angle factor");
				break;
			case ARCFUNCTION:
				DiaSetText(24, "Arc function");
				break;
			case ARCINC:
				DiaSetText(24, "Arc angle increment");
				break;
			case ARCNOEXTEND:
				DiaSetText(24, "Arc extension");
				break;
			case ARCWIPESPINS:
				DiaSetText(24, "Arc coverage");
				break;
			case CENTEROBJ:
				DiaSetText(24, "Grab point");
				break;
			case LAYER3DHEIGHT:
				DiaSetText(24, "3D height");
				break;
			case LAYER3DTHICK:
				DiaSetText(24, "3D thickness");
				break;
			case LAYERCIF:
				DiaSetText(24, "CIF name");
				break;
			case LAYERCOLOR:
				DiaSetText(24, "Color definition");
				break;
			case LAYERDXF:
				DiaSetText(24, "DXF name(s)");
				break;
			case LAYERDRCMINWID:
				DiaSetText(24, "DRC minimum width");
				break;
			case LAYERFUNCTION:
				DiaSetText(24, "Layer function");
				break;
			case LAYERGDS:
				DiaSetText(24, "GDS-II number");
				break;
			case LAYERLETTERS:
				DiaSetText(24, "Layer letters");
				break;
			case LAYERPATTERN:
				DiaSetText(24, "Layer stipple pattern");
				break;
			case LAYERSTYLE:
				DiaSetText(24, "Style definition");
				break;
			case LAYERPATCH:
			case HIGHLIGHTOBJ:
				var = getval((INTBIG)ni, VNODEINST, VNODEPROTO, "EDTEC_layer");
				if (var == NOVARIABLE) DiaSetText(24, "UNKNOWN LAYER"); else
				{
					onp = (NODEPROTO *)var->addr;
					if (onp == NONODEPROTO) DiaSetText(24, "Highlight box"); else
					{
						(void)strcpy(ent, "Layer ");
						(void)strcat(ent, &onp->cell->cellname[6]);
						DiaSetText(24, ent);
					}
				}
				break;
			case NODEFUNCTION:
				DiaSetText(24, "Node function");
				break;
			case NODELOCKABLE:
				DiaSetText(24, "Node lockability");
				break;
			case NODESERPENTINE:
				DiaSetText(24, "Serpentine transistor");
				break;
			case NODESQUARE:
				DiaSetText(24, "Square node");
				break;
			case NODEWIPES:
				DiaSetText(24, "Disappearing pin");
				break;
			case PORTOBJ:
				var = getval((INTBIG)ni, VNODEINST, VSTRING, "EDTEC_portname");
				if (var == NOVARIABLE) DiaSetText(24, "Unnamed port"); else
				{
					(void)strcpy(ent, "Port ");
					(void)strcat(ent, (char *)var->addr);
					DiaSetText(24, ent);
				}
				break;
			case TECHDESCRIPT:
				DiaSetText(24, "Technology description");
				break;
			case TECHLAMBDA:
				DiaSetText(24, "Lambda value");
				break;
		}
	} else if (np->primindex != 0 && np->tech == art_tech)
	{
		DiaSetText(23, "Color:");
		DiaSetPopup(24, 25, colortype);
		var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, art_colorkey);
		if (var == NOVARIABLE) i = 6; else
			switch (var->addr)
		{
			case LAYERT1: i = 0;   break;
			case LAYERT2: i = 1;   break;
			case LAYERT3: i = 2;   break;
			case LAYERT4: i = 3;   break;
			case LAYERT5: i = 4;   break;
			case WHITE:   i = 5;   break;
			case BLACK:   i = 6;   break;
			case RED:     i = 7;   break;
			case BLUE:    i = 8;   break;
			case GREEN:   i = 9;   break;
			case CYAN:    i = 10;  break;
			case MAGENTA: i = 11;  break;
			case YELLOW:  i = 12;  break;
			case GRAY:    i = 13;  break;
			case ORANGE:  i = 14;  break;
			case PURPLE:  i = 15;  break;
			case BROWN:   i = 16;  break;
			case LGRAY:   i = 17;  break;
			case DGRAY:   i = 18;  break;
			case LRED:    i = 19;  break;
			case DRED:    i = 20;  break;
			case LGREEN:  i = 21;  break;
			case DGREEN:  i = 22;  break;
			case LBLUE:   i = 23;  break;
			case DBLUE:   i = 24;  break;
		}
		DiaSetPopupEntry(24, i);
	}

	/* load the special-value entries if appropriate */
	if (np == art_circleprim)
	{
		getarcdegrees(ni, &startoffset, &endangle);
		if (startoffset != 0.0)
		{
			DiaSetText(25, "Offset angle / Degrees of circle:");
			sprintf(ent, "%g / %g", startoffset*180.0/EPI, endangle*180.0/EPI);
			DiaSetText(22, ent);
		} else
		{
			DiaSetText(25, "Degrees of circle:");
			if (endangle == 0.0) DiaSetText(22, "360"); else
			{
				sprintf(ent, "%g", endangle*180.0/EPI);
				DiaSetText(22, ent);
			}
		}
	}

	/* describe all ports */
	lineno = 0;
	highlineno = -1;
	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (pp == fromport) highlineno = lineno;
		lineno = us_dlogchatportproto(ni, pp, fromport, lineno);
	}
	DiaSelectLine(3, highlineno);

	/* if the node can't be edited, disable all changes */
	if (us_canedit(ni->parent, ni->proto, 0) != 0)
	{
		DiaDimItem(4);
		DiaDimItem(5);
		DiaDimItem(7);
		DiaDimItem(8);
		DiaDimItem(9);
		DiaDimItem(10);
		DiaDimItem(11);
		DiaDimItem(12);
		DiaDimItem(13);
		DiaDimItem(22);
		DiaDimItem(29);
	}

	/* loop until done */
	changed = positionchanged = 0;
	selectedai = NOARCINST;
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL || itemHit == 26) break;
		if (itemHit == 11 || itemHit == 29)
		{
			DiaSetControl(itemHit, 1-DiaGetControl(itemHit));
			changed++;
			continue;
		}
		if (itemHit == 4 || itemHit == 5)
		{
			DiaSetControl(4, 0);   DiaSetControl(5, 0);
			DiaSetControl(itemHit, 1);
			changed++;
			continue;
		}
		if (itemHit == 7 || itemHit == 8 || itemHit == 9 || itemHit == 10 ||
			itemHit == 22 || itemHit == 24)
				changed++;
		if (itemHit == 12 || itemHit == 13)
		{
			changed++;
			positionchanged++;
		}
		if (itemHit == 3)
		{
			lineno = DiaGetCurLine(3);
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
			{
				ai = pi->conarcinst;
				if (ai->temp2 == lineno) break;
			}
			if (pi == NOPORTARCINST) continue;
			selectedai = ai;
			us_clearhighlightcount();
			high.status = HIGHFROM;
			high.facet = selectedai->parent;
			high.fromgeom = selectedai->geom;
			high.fromport = NOPORTPROTO;
			high.frompoint = 0;
			high.fromvar = NOVARIABLE;
			(void)us_addhighlight(&high);
			us_showallhighlight();
			us_endchanges(NOWINDOWPART);
			continue;
		}
	}

	if (itemHit != CANCEL && changed != 0)
	{
		/* see if size, rotation, or position changed */
		if (xyrev == 0)
		{
			xs = atola(DiaGetText(8));
			ys = atola(DiaGetText(9));
		} else
		{
			xs = atola(DiaGetText(9));
			ys = atola(DiaGetText(8));
		}
		xc = atola(DiaGetText(12));
		yc = atola(DiaGetText(13));
		r = atofr(DiaGetText(10))*10/WHOLE;
		t = (DiaGetControl(11) != 0 ? 1 : 0);
		if (positionchanged != 0 || r != ni->rotation || t != ni->transpose ||
			xs != ni->highx-ni->lowx-plx-phx || ys != ni->highy-ni->lowy-ply-phy)
		{
			us_pushhighlight();
			us_clearhighlightcount();
			startobjectchange((INTBIG)ni, VNODEINST);
			if (positionchanged == 0)
			{
				/* only size changed: adjust position appropriately */
				dx = (xs + plx + phx) - (ni->highx - ni->lowx);
				dy = (ys + ply + phy) - (ni->highy - ni->lowy);
				nlx = ni->lowx - dx/2;
				nhx = nlx + xs + plx + phx;
				nly = ni->lowy - dy/2;
				nhy = nly + ys + ply + phy;
				modifynodeinst(ni, nlx-ni->lowx, nly-ni->lowy, nhx-ni->highx,
					nhy-ni->highy, (INTSML)(r - ni->rotation), (INTSML)(t - ni->transpose));
			} else
			{
				/* position (and possibly size) changed: update from dialog */
				if ((us_useroptions&CENTEREDPRIMITIVES) == 0)
				{
					dx = xc - ni->lowx - cox;   dy = yc - ni->lowy - coy;
					nlx = ni->lowx + dx;
					nhx = nlx + xs + plx + phx;
					nly = ni->lowy + dy;
					nhy = nly + ys + ply + phy;
				} else
				{
					var = getvalkey((INTBIG)ni->proto, VNODEPROTO, VINTEGER|VISARRAY, el_prototype_center);
					if (var != NOVARIABLE)
					{
						dx = ((INTBIG *)var->addr)[0] + (ni->lowx+ni->highx)/2 -
							(ni->proto->lowx+ni->proto->highx)/2;
						dy = ((INTBIG *)var->addr)[1] + (ni->lowy+ni->highy)/2 -
							(ni->proto->lowy+ni->proto->highy)/2;
						makerot(ni, trans);
						xform(dx, dy, &cox, &coy, trans);
						xc -= cox - (ni->lowx+ni->highx)/2;
						yc -= coy - (ni->lowy+ni->highy)/2;
					}
					nlx = xc - xs/2 - plx;
					nhx = nlx + xs + plx + phx;
					nly = yc - ys/2 - ply;
					nhy = nly + ys + ply + phy;
				}
				modifynodeinst(ni, nlx-ni->lowx, nly-ni->lowy, nhx-ni->highx,
					nhy-ni->highy, (INTSML)(r - ni->rotation), (INTSML)(t - ni->transpose));
			}
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_pophighlight(1);
		}

		/* update transistor width if it changed */
		if (serpwidth >= 0)
		{
			newstr = DiaGetText(22);
			while (*newstr == ' ') newstr++;
			if (*newstr != 0 && atola(newstr) != serpwidth)
			{
				us_pushhighlight();
				us_clearhighlightcount();
				startobjectchange((INTBIG)ni, VNODEINST);
				(void)setvalkey((INTBIG)ni, VNODEINST, el_transistor_width, atofr(newstr), VFRACT);
				endobjectchange((INTBIG)ni, VNODEINST);
				(void)us_pophighlight(0);
			}
		}

		/* update diode area if it changed */
		if (np == sch_diodeprim)
		{
			(void)initinfstr();
			if (DiaGetPopupEntry(24) == 1) addtoinfstr('z');
			addstringtoinfstr(DiaGetText(22));
			us_pushhighlight();
			us_clearhighlightcount();
			startobjectchange((INTBIG)ni, VNODEINST);
			(void)setvalkey((INTBIG)ni, VNODEINST, sch_diodekey, (INTBIG)returninfstr(),
				VSTRING|VDISPLAY);
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_pophighlight(1);
		}

		/* update capacitance if it changed */
		if (np == sch_capacitorprim)
		{
			(void)initinfstr();
			if (DiaGetPopupEntry(24) == 1) addtoinfstr('e');
			addstringtoinfstr(DiaGetText(22));
			us_pushhighlight();
			us_clearhighlightcount();
			startobjectchange((INTBIG)ni, VNODEINST);
			(void)setvalkey((INTBIG)ni, VNODEINST, sch_capacitancekey, (INTBIG)returninfstr(),
				VSTRING|VDISPLAY);
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_pophighlight(1);
		}

		/* update resistance if it changed */
		if (np == sch_resistorprim)
		{
			us_pushhighlight();
			us_clearhighlightcount();
			startobjectchange((INTBIG)ni, VNODEINST);
			(void)setvalkey((INTBIG)ni, VNODEINST, sch_resistancekey, (INTBIG)DiaGetText(22),
				VSTRING|VDISPLAY);
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_pophighlight(1);
		}

		/* update inductance if it changed */
		if (np == sch_inductorprim)
		{
			us_pushhighlight();
			us_clearhighlightcount();
			startobjectchange((INTBIG)ni, VNODEINST);
			(void)setvalkey((INTBIG)ni, VNODEINST, sch_inductancekey, (INTBIG)DiaGetText(22),
				VSTRING|VDISPLAY);
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_pophighlight(1);
		}

		/* update black-box function if it changed */
		if (np == sch_bboxprim)
		{
			us_pushhighlight();
			us_clearhighlightcount();
			startobjectchange((INTBIG)ni, VNODEINST);
			(void)setvalkey((INTBIG)ni, VNODEINST, sch_functionkey, (INTBIG)DiaGetText(22),
				VSTRING|VDISPLAY);
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_pophighlight(1);
		}

		/* update source information if it changed */
		if (np == sch_sourceprim)
		{
			(void)initinfstr();
			switch (DiaGetPopupEntry(24))
			{
				case 0:
				case 1:  addstringtoinfstr("v/");   break;
				case 2:  addstringtoinfstr("c/");   break;
				case 3:  addstringtoinfstr("cm/");  break;
				case 4:  addstringtoinfstr("t/");   break;
				case 5:  addstringtoinfstr("vd/");  break;
				case 6:  addstringtoinfstr("a/");   break;
				case 7:  addstringtoinfstr("n/");   break;
				case 8:  addstringtoinfstr("x/");   break;
				case 9:  addstringtoinfstr("b/");   break;
				case 10: addstringtoinfstr("s/");   break;
			}
			addstringtoinfstr(DiaGetText(22));
			us_pushhighlight();
			us_clearhighlightcount();
			startobjectchange((INTBIG)ni, VNODEINST);
			(void)setvalkey((INTBIG)ni, VNODEINST, sch_sourcekey, (INTBIG)returninfstr(),
				VSTRING|VDISPLAY);
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_pophighlight(1);
		}

		/* update two-port information if it changed */
		if (np == sch_twoportprim)
		{
			(void)initinfstr();
			switch (DiaGetPopupEntry(24))
			{
				case 0: addtoinfstr('g');    break;
				case 1: addtoinfstr('h');    break;
				case 2: addtoinfstr('u');    break;
				case 3: addtoinfstr('f');    break;
				case 4: addtoinfstr('l');    break;
			}
			addstringtoinfstr(DiaGetText(22));
			us_pushhighlight();
			us_clearhighlightcount();
			startobjectchange((INTBIG)ni, VNODEINST);
			(void)setvalkey((INTBIG)ni, VNODEINST, sch_twoportkey, (INTBIG)returninfstr(),
				VSTRING|VDISPLAY);
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_pophighlight(1);
		}

		/* update transistor size if it changed */
		if (np == sch_transistorprim)
		{
			(void)initinfstr();
			switch (DiaGetPopupEntry(24))
			{
				case 0: addstringtoinfstr("nmos");   break;
				case 1: addstringtoinfstr("dmos");   break;
				case 2: addstringtoinfstr("pmos");   break;
				case 3: addstringtoinfstr("npn");    break;
				case 4: addstringtoinfstr("pnp");    break;
				case 5: addstringtoinfstr("njfet");  break;
				case 6: addstringtoinfstr("pjfet");  break;
				case 7: addstringtoinfstr("dmes");   break;
				case 8: addstringtoinfstr("emes");   break;
			}
			addstringtoinfstr(DiaGetText(22));
			us_pushhighlight();
			us_clearhighlightcount();
			startobjectchange((INTBIG)ni, VNODEINST);
			(void)setvalkey((INTBIG)ni, VNODEINST, sch_transistortypekey, (INTBIG)returninfstr(),
				VSTRING|VDISPLAY);
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_pophighlight(1);
		}

		/* update color if it changed */
		if (np->primindex != 0 && np->tech == art_tech && techedrel < 0)
		{
			switch (DiaGetPopupEntry(24))
			{
				case 0:  i = LAYERT1;   break;
				case 1:  i = LAYERT2;   break;
				case 2:  i = LAYERT3;   break;
				case 3:  i = LAYERT4;   break;
				case 4:  i = LAYERT5;   break;
				case 5:  i = WHITE;     break;
				case 6:  i = BLACK;     break;
				case 7:  i = RED;       break;
				case 8:  i = BLUE;      break;
				case 9:  i = GREEN;     break;
				case 10: i = CYAN;      break;
				case 11: i = MAGENTA;   break;
				case 12: i = YELLOW;    break;
				case 13: i = GRAY;      break;
				case 14: i = ORANGE;    break;
				case 15: i = PURPLE;    break;
				case 16: i = BROWN;     break;
				case 17: i = LGRAY;     break;
				case 18: i = DGRAY;     break;
				case 19: i = LRED;      break;
				case 20: i = DRED;      break;
				case 21: i = LGREEN;    break;
				case 22: i = DGREEN;    break;
				case 23: i = LBLUE;     break;
				case 24: i = DBLUE;     break;
			}
			us_pushhighlight();
			us_clearhighlightcount();
			startobjectchange((INTBIG)ni, VNODEINST);
			if (i == BLACK)
			{
				var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, art_colorkey);
				if (var != NOVARIABLE)
					delvalkey((INTBIG)ni, VNODEINST, art_colorkey);
			} else
			{
				setvalkey((INTBIG)ni, VNODEINST, art_colorkey, i, VINTEGER);
			}
			endobjectchange((INTBIG)ni, VNODEINST);
			(void)us_pophighlight(0);
		}

		/* update ease of selection if changed */
		if (DiaGetControl(29) != 0) ni->userbits &= ~HARDSELECTN; else
			ni->userbits |= HARDSELECTN;

		/* update expansion bit if changed */
		if (np->primindex == 0)
		{
			i = ni->userbits;
			if (DiaGetControl(4) != 0) i |= NEXPAND; else
				i &= ~NEXPAND;
			if (i != (INTBIG)ni->userbits)
			{
				us_pushhighlight();
				us_clearhighlightcount();
				startobjectchange((INTBIG)ni, VNODEINST);
				ni->userbits = i;
				endobjectchange((INTBIG)ni, VNODEINST);
				(void)us_pophighlight(0);
			}
		}

		/* update name if it changed */
		newstr = DiaGetText(7);
		while (*newstr == ' ') newstr++;
		if (strcmp(newstr, namestr) != 0)
		{
			us_pushhighlight();
			us_clearhighlightcount();

			/* change the name of the nodeinst */
			startobjectchange((INTBIG)ni, VNODEINST);
			if (*newstr == 0)
				(void)delvalkey((INTBIG)ni, VNODEINST, el_node_name); else
			{
				var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
				if (var != NOVARIABLE) descript = var->textdescript; else
					descript = defaulttextdescript(NOGEOM);
				var = setvalkey((INTBIG)ni, VNODEINST, el_node_name, (INTBIG)newstr, VSTRING|VDISPLAY);
				if (var != NOVARIABLE) var->textdescript = descript;

				/* shift text down if on a facet instance */
				if (var != NOVARIABLE && np->primindex == 0)
				{
					var->textdescript = us_setdescriptoffset(var->textdescript,
						0, (ni->highy-ni->lowy) / el_curtech->deflambda);
				}
			}
			endobjectchange((INTBIG)ni, VNODEINST);

			/* restore highlighting */
			(void)us_pophighlight(0);
		}

		/* update the circle degrees if it changed */
		if (np == art_circleprim)
		{
			str = DiaGetText(25);
			if (strcmp(str, "Degrees of circle:") == 0)
			{
				newend = atof(DiaGetText(22));
				if (newend == 360.0) newend = 0.0;
				newend = newend * EPI / 180.0;
				if (newend != endangle)
				{
					us_pushhighlight();
					us_clearhighlightcount();
					startobjectchange((INTBIG)ni, VNODEINST);
					setarcdegrees(ni, startoffset, newend);
					endobjectchange((INTBIG)ni, VNODEINST);
					(void)us_pophighlight(1);
				}
			} else
			{
				str = DiaGetText(22);
				while (*str == ' ' || *str == '\t') str++;
				newstart = atof(str);
				while (*str != 0 && *str != '/') str++;
				if (*str == 0)
				{
					newend = newstart;
					newstart = 0.0;
				} else
				{
					str++;
					while (*str == ' ' || *str == '\t') str++;
					newend = atof(str);
				}
				
				if (newend == 360.0) newend = newstart = 0.0;
				newend = newend * EPI / 180.0;
				newstart = newstart * EPI / 180.0;
				if (newend != endangle || newstart != startoffset)
				{
					us_pushhighlight();
					us_clearhighlightcount();
					startobjectchange((INTBIG)ni, VNODEINST);
					setarcdegrees(ni, newstart, newend);
					endobjectchange((INTBIG)ni, VNODEINST);
					(void)us_pophighlight(1);
				}
			}
		}
	}
	DiaDoneDialog();
	if (itemHit == 26 && selectedai != NOARCINST)
		us_getinfoarc(selectedai);
}

void us_updatenodeinfo(NODEINST *ni, VARIABLE *var, char *varname, char *wholemessage,
	char *variablepart)
{
	REGISTER char *newstr, *pt;

	newstr = DiaGetText(22);
	if (strcmp(newstr, variablepart) == 0) return;

	us_pushhighlight();
	us_clearhighlightcount();
	startobjectchange((INTBIG)ni, VNODEINST);
	(void)initinfstr();
	for(pt = wholemessage; pt < variablepart; pt++) (void)addtoinfstr(*pt);
	(void)addstringtoinfstr(newstr);
	(void)setval((INTBIG)ni, VNODEINST, varname, (INTBIG)returninfstr(), var->type);
	endobjectchange((INTBIG)ni, VNODEINST);
	(void)us_pophighlight(1);
}

/*
 * routine to display information about port prototype "pp" on nodeinst "ni".
 * If the port prototype's "temp1" is nonzero, this port has already been
 * mentioned and should not be done again.
 */
INTBIG us_dlogchatportproto(NODEINST *ni, PORTPROTO *pp, PORTPROTO *selected, INTBIG lineno)
{
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER ARCINST *ai;
	REGISTER ARCPROTO *ap;
	REGISTER INTBIG i, count, moreinfo;
	REGISTER char *activity;
	char line[256];

	/* talk about the port prototype */
	(void)initinfstr();
	activity = us_describeportbits(pp);
	if (*activity != 0)
	{
		(void)addstringtoinfstr(activity);
		(void)addstringtoinfstr(" port ");
	} else (void)addstringtoinfstr("Port ");
	(void)addstringtoinfstr(pp->protoname);
	(void)addstringtoinfstr(" connects to");
	count = 0;
	for(i=0; pp->connects[i] != NOARCPROTO; i++)
	{
		ap = pp->connects[i];
		if ((ni->proto->primindex == 0 || ni->proto->tech != gen_tech) &&
			ap->tech == gen_tech) continue;
		if (count > 0) (void)addstringtoinfstr(",");
		(void)addstringtoinfstr(" ");
		(void)addstringtoinfstr(ap->protoname);
		count++;
	}
	moreinfo = 0;
	if (pp == selected) moreinfo = 1;
	for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		if (pi->proto == pp) moreinfo = 1;
	for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
		if (pe->proto == pp) moreinfo = 1;
	if (moreinfo != 0) (void)addstringtoinfstr(":");
	DiaStuffLine(3, returninfstr());
	lineno++;

	/* mention if it is highlighted */
	if (pp == selected)
	{
		DiaStuffLine(3, "  Highlighted port");
		lineno++;
	}

	/* talk about any arcs on this prototype */
	for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
	{
		if (pi->proto != pp) continue;
		ai = pi->conarcinst;
		if (ai->end[0].portarcinst == pi) i = 0; else i = 1;
		(void)sprintf(line, "  Connected at (%s,%s) to %s arc", latoa(ai->end[i].xpos),
			latoa(ai->end[i].ypos), describearcinst(ai));
		ai->temp2 = lineno;
		DiaStuffLine(3, line);
		lineno++;
	}

	/* talk about any exported ports of this prototype */
	for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
		if (pe->proto == pp)
	{
		(void)initinfstr();
		(void)addstringtoinfstr("  Exported as ");
		(void)addstringtoinfstr(us_describeportbits(pe->exportproto));
		(void)addstringtoinfstr(" port '");
		(void)addstringtoinfstr(pe->exportproto->protoname);
		(void)addstringtoinfstr("'");
		DiaStuffLine(3, returninfstr());
		lineno++;
	}
	return(lineno);
}

/****************************** GRID OPTIONS DIALOG ******************************/

/* Grid options */
DIALOGITEM us_griddialogitems[] =
{
 /*  1 */ {0, {116,320,140,384}, BUTTON, "OK"},
 /*  2 */ {0, {116,12,140,76}, BUTTON, "Cancel"},
 /*  3 */ {0, {32,8,48,185}, MESSAGE, "Grid dot spacing:"},
 /*  4 */ {0, {32,192,48,272}, EDITTEXT, ""},
 /*  5 */ {0, {8,192,24,286}, MESSAGE, "Horizontal:"},
 /*  6 */ {0, {32,292,48,372}, EDITTEXT, ""},
 /*  7 */ {0, {60,8,76,185}, MESSAGE, "Default grid spacing:"},
 /*  8 */ {0, {60,192,76,272}, EDITTEXT, ""},
 /*  9 */ {0, {8,292,24,386}, MESSAGE, "Vertical:"},
 /* 10 */ {0, {60,292,76,372}, EDITTEXT, ""},
 /* 11 */ {0, {88,8,104,185}, MESSAGE, "Distance between bold dots:"},
 /* 12 */ {0, {88,192,104,272}, EDITTEXT, ""},
 /* 13 */ {0, {120,100,136,297}, CHECK, "Align grid with circuitry"},
 /* 14 */ {0, {88,292,104,372}, EDITTEXT, ""}
};
DIALOG us_griddialog = {{50,75,199,482}, "Grid Options", 14, us_griddialogitems};

/*
 * special case for the "grid settings" dialog
 * Horizontal spacing           = 4 (edit text)
 * Vertical spacing             = 6 (edit text)
 * Default horizontal spacing   = 8 (edit text)
 * Default vertical spacing     = 10 (edit text)
 * Bold dot horizontal distance = 12 (edit text)
 * Bold dot vertical distance   = 14 (edit text)
 * Align with circuitry         = 13 (check)
 */
INTSML us_griddlog(char *paramstart[])
{
	INTBIG itemHit;
	INTBIG defgrid[2], val, xspacing, yspacing;
	REGISTER VARIABLE *var;
	char buf[20];

	if (us_needwindow()) return(0);

	/* display the grid settings dialog box */
	if (DiaInitDialog(&us_griddialog) != 0) return(0);
	DiaSetText(-4, latoa(el_curwindowpart->gridx));
	DiaSetText(6, latoa(el_curwindowpart->gridy));
	var = getval((INTBIG)us_aid, VAID, VFRACT|VISARRAY, "USER_default_grid");
	if (var == NOVARIABLE)
	{
		DiaSetText(8, "1.0");
		DiaSetText(10, "1.0");
	} else
	{
		DiaSetText(8, frtoa(((INTBIG *)var->addr)[0]));
		DiaSetText(10, frtoa(((INTBIG *)var->addr)[1]));
	}

	var = getvalkey((INTBIG)us_aid, VAID, -1, us_gridboldspacing);
	if (var == NOVARIABLE) xspacing = yspacing = 10; else
	{
		if ((var->type&VISARRAY) == 0)
			xspacing = yspacing = var->addr; else
		{
			xspacing = ((INTBIG *)var->addr)[0];
			yspacing = ((INTBIG *)var->addr)[1];
		}
	}
	sprintf(buf, "%ld", xspacing);   DiaSetText(12, buf);
	sprintf(buf, "%ld", yspacing);   DiaSetText(14, buf);

	var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_gridfloats);
	if (var == NOVARIABLE) val = 0; else
		val = var->addr;
	if (val != 0) DiaSetControl(13, 1);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == 13)
		{
			DiaSetControl(itemHit, 1 - DiaGetControl(itemHit));
			continue;
		}
	}

	if (itemHit != CANCEL)
	{
		us_pushhighlight();
		us_clearhighlightcount();
		startobjectchange((INTBIG)el_curwindowpart, VWINDOWPART);

		/* turn grid off if on */
		if ((el_curwindowpart->state&GRIDON) != 0)
			(void)setval((INTBIG)el_curwindowpart, VWINDOWPART, "state", el_curwindowpart->state & ~GRIDON, VINTEGER);

		/* change grid options */
		defgrid[0] = atofr(DiaGetText(8));
		defgrid[1] = atofr(DiaGetText(10));
		(void)setval((INTBIG)us_aid, VAID, "USER_default_grid",
			(INTBIG)defgrid, VFRACT|VISARRAY|(2<<VLENGTHSH));
		defgrid[0] = atoi(DiaGetText(12));
		defgrid[1] = atoi(DiaGetText(14));
		(void)setvalkey((INTBIG)us_aid, VAID, us_gridboldspacing,
			(INTBIG)defgrid, VINTEGER|VISARRAY|(2<<VLENGTHSH));
		val = DiaGetControl(13);
		(void)setvalkey((INTBIG)us_aid, VAID, us_gridfloats,
			(INTBIG)val, VINTEGER);

		/* adjust grid */
		(void)setval((INTBIG)el_curwindowpart, VWINDOWPART, "gridx", atola(DiaGetText(4)), VINTEGER);
		(void)setval((INTBIG)el_curwindowpart, VWINDOWPART, "gridy", atola(DiaGetText(6)), VINTEGER);

		/* show new grid */
		us_gridset(el_curwindowpart, GRIDON);

		/* restore highlighting */
		endobjectchange((INTBIG)el_curwindowpart, VWINDOWPART);
		(void)us_pophighlight(0);

	}
	DiaDoneDialog();
	paramstart[0] = "x";
	return(1);
}

/****************************** HELP DIALOG ******************************/

/* Help */
DIALOGITEM us_helpdialogitems[] =
{
 /*  1 */ {0, {288,376,312,440}, BUTTON, "OK"},
 /*  2 */ {0, {8,32,24,91}, MESSAGE, "Topics:"},
 /*  3 */ {0, {8,192,280,636}, SCROLL, ""},
 /*  4 */ {0, {24,8,309,177}, SCROLL, ""}
};
DIALOG us_helpdialog = {{50,75,378,720}, "Help", 4, us_helpdialogitems};

/*
 * special case for the "help" dialog
 * topics       = 4 (user item)
 * help         = 3 (user item)
 */
INTSML us_helpdlog(char *prompt, char *paramstart[])
{
	char *filename, *line, buf[256], *platform, *pt;
	INTBIG itemHit, i;
	FILE *in;
	REGISTER VARIABLE *var;

	/* get the help file */
	(void)initinfstr();
	(void)addstringtoinfstr(el_libdir);
	(void)addstringtoinfstr(prompt);
	line = returninfstr();
	in = xopen(line, FILETYPEHELP, "", &filename);
	if (in == NULL)
	{
		us_abortcommand("Cannot find help file: %s", line);
		return(0);
	}

	/* show the "help" dialog */
	if (DiaInitDialog(&us_helpdialog) != 0)
	{
		xclose(in);
		return(0);
	}
	DiaInitTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCHORIZBAR|SCSMALLFONT|SCFIXEDWIDTH);
	DiaInitTextDialog(4, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCSELKEY|SCREPORT|SCSMALLFONT);

	/* determine the platform that is running */
	var = getval((INTBIG)us_aid, VAID, VSTRING, "USER_machine");
	if (var == NOVARIABLE) platform = ""; else
		platform = (char *)var->addr;

	/* load the topics list */
	for(;;)
	{
		if (xfgets(buf, 256, in) != 0) break;
		if (buf[0] != '!') continue;
		for(pt = buf; *pt != 0; pt++)
			if (*pt == '[') break;
		if (*pt == '[')
		{
			*pt++ = 0;
			if (namesamen(pt, platform, strlen(platform)) != 0) continue;
		}
		DiaStuffLine(4, &buf[1]);
	}
	DiaSelectLine(4, -1);

	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK) break;
		if (itemHit == 4)
		{
			line = DiaGetScrollLine(4, DiaGetCurLine(4));
			xseek(in, 0, 0);
			DiaLoadTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
			for(;;)
			{
				if (xfgets(buf, 256, in) != 0) break;
				if (buf[0] != '!') continue;
				for(pt = buf; *pt != 0; pt++)
					if (*pt == '[') break;
				if (*pt == '[')
				{
					*pt++ = 0;
					if (namesamen(pt, platform, strlen(platform)) != 0) continue;
				}
				if (strcmp(&buf[1], line) == 0) break;
			}
			for(i=0; ; i++)
			{
				if (xfgets(buf, 256, in) != 0) break;
				if (buf[0] == '!') break;
				if (buf[0] == 0 && i == 0) continue;
				DiaStuffLine(3, buf);
			}
			DiaSelectLine(3, -1);
			continue;
		}
	}
	DiaDoneDialog();
	xclose(in);
	paramstart[0] = "";
	return(1);
}

/****************************** LAYER HIGHLIGHT DIALOG ******************************/

/* Layer Highlighting */
DIALOGITEM us_highlightlayerdialogitems[] =
{
 /*  1 */ {0, {108,184,132,248}, BUTTON, "Done"},
 /*  2 */ {0, {12,184,36,248}, BUTTON, "None"},
 /*  3 */ {0, {8,8,136,170}, SCROLL, ""}
};
DIALOG us_highlightlayerdialog = {{50,75,195,334}, "Layer to Highlight", 3, us_highlightlayerdialogitems};

/*
 * special case for the "highlight layer" command
 * No layer           = 2 (button)
 * Layer list         = 3 (scroll)
 */
INTSML us_highlayerlog(char *paramstart[])
{
	REGISTER INTBIG itemHit, i;
	REGISTER INTBIG funct;
	REGISTER char *la;
	char buf[2];

	if (us_needwindow()) return(0);

	/* display the grid settings dialog box */
	if (DiaInitDialog(&us_highlightlayerdialog) != 0) return(0);
	DiaInitTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, 0,
		SCREPORT|SCSELMOUSE);
	for(i=0; i<el_curtech->layercount; i++)
	{
		if (el_curtech->layers[i]->bits == LAYERO ||
			el_curtech->layers[i]->bits == LAYERN) continue;
		funct = layerfunction(el_curtech, (INTSML)i);
		if ((funct&LFPSEUDO) != 0) continue;
		DiaStuffLine(3, layername(el_curtech, (INTSML)i));
	}
	DiaSelectLine(3, -1);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK) break;
		if (itemHit == 2)
		{
			DiaSelectLine(3, -1);
			paramstart[0] = "default";
			us_color(1, paramstart);
			continue;
		}
		if (itemHit == 3)
		{
			i = DiaGetCurLine(3);
			if (i < 0) continue;
			la = DiaGetScrollLine(3, i);
			for(i=0; i<el_curtech->layercount; i++)
				if (strcmp(la, layername(el_curtech, (INTSML)i)) == 0) break;
			if (i >= el_curtech->layercount) continue;
			paramstart[0] = "default";
			us_color(1, paramstart);
			la = us_layerletters(el_curtech, (INTSML)i);
			buf[0] = *la;
			buf[1] = 0;
			paramstart[0] = "highlight";
			paramstart[1] = buf;
			us_color(2, paramstart);
			continue;
		}
	}
	DiaDoneDialog();
	return(0);
}

/****************************** LIBRARY PATH DIALOG ******************************/

/* Library paths */
DIALOGITEM us_librarypathdialogitems[] =
{
 /*  1 */ {0, {76,312,100,376}, BUTTON, "OK"},
 /*  2 */ {0, {76,32,100,96}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,16,24,220}, MESSAGE, "Location of library files:"},
 /*  4 */ {0, {32,8,64,400}, EDITTEXT, ""}
};
DIALOG us_librarypathdialog = {{50,75,159,485}, "Current Library Path", 4, us_librarypathdialogitems};

/*
 * special case for the "library paths" dialog
 * library file location = 4 (message)
 */
INTSML us_librarypathdlog(char *paramstart[])
{
	INTBIG itemHit;
	char *pt;

	/* display the library paths dialog box */
	if (DiaInitDialog(&us_librarypathdialog) != 0) return(0);
	DiaSetText(4, el_libdir);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK && DiaValidEntry(4)) break;
	}

	paramstart[0] = "";
	if (itemHit != CANCEL)
	{
		pt = DiaGetText(4);
		if (strcmp(pt, el_libdir) != 0) (void)reallocstring(&el_libdir, pt, db_cluster);
	}
	DiaDoneDialog();
	return(1);
}

/****************************** LIBRARY SELECTION DIALOG ******************************/

/* Change Library */
DIALOGITEM us_chglibrarydialogitems[] =
{
 /*  1 */ {0, {164,208,188,288}, BUTTON, "OK"},
 /*  2 */ {0, {116,208,140,288}, BUTTON, "Cancel"},
 /*  3 */ {0, {4,4,20,100}, MESSAGE, "Current Library:"},
 /*  4 */ {0, {4,104,20,288}, MESSAGE, ""},
 /*  5 */ {0, {52,4,196,196}, SCROLL, ""},
 /*  6 */ {0, {32,16,48,168}, MESSAGE|INACTIVE, "Switch to Library:"}
};
DIALOG us_chglibrarydialog = {{75,75,280,372}, "Set Current Library", 6, us_chglibrarydialogitems};

/*
 * special case for the "change current library" dialog
 * current library = 4 (message)
 * library list    = 5 (scroll)
 */
INTSML us_oldlibrarydlog(char *paramstart[])
{
	REGISTER INTBIG itemHit;
	REGISTER LIBRARY *lib;
	REGISTER char *pt;

	/* display the node size dialog box */
	if (DiaInitDialog(&us_chglibrarydialog) != 0) return(0);
	DiaSetText(4, el_curlib->libname);
	DiaInitTextDialog(5, topoflibs, nextlibs, DiaNullDlogDone, -1,
		SCSELMOUSE|SCDOUBLEQUIT);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
	}

	lib = el_curlib;
	if (itemHit != CANCEL)
	{
		pt = DiaGetScrollLine(5, DiaGetCurLine(5));
		lib = getlibrary(pt);
	}
	DiaDoneDialog();
	if (lib != NOLIBRARY && lib != el_curlib)
		us_switchtolibrary(lib);
	return(0);
}

/****************************** NODE: CREATE ANNULAR RING DIALOG ******************************/

/* Annular Ring */
DIALOGITEM us_annringdialogitems[] =
{
 /*  1 */ {0, {268,176,292,240}, BUTTON, "OK"},
 /*  2 */ {0, {268,20,292,84}, BUTTON, "Cancel"},
 /*  3 */ {0, {164,12,180,160}, MESSAGE, "Inner Radius:"},
 /*  4 */ {0, {164,164,180,244}, EDITTEXT, ""},
 /*  5 */ {0, {188,12,204,156}, MESSAGE, "Outer Radius:"},
 /*  6 */ {0, {188,164,204,244}, EDITTEXT, ""},
 /*  7 */ {0, {212,12,228,160}, MESSAGE, "Number of segments:"},
 /*  8 */ {0, {212,164,228,244}, EDITTEXT, "32"},
 /*  9 */ {0, {236,12,252,160}, MESSAGE, "Number of degrees:"},
 /* 10 */ {0, {236,164,252,244}, EDITTEXT, "360"},
 /* 11 */ {0, {8,8,24,172}, MESSAGE, "Layer to use for ring:"},
 /* 12 */ {0, {28,8,156,244}, SCROLL, ""}
};
DIALOG us_annringdialog = {{75,75,376,330}, "Annulus Construction", 12, us_annringdialogitems};

/*
 * special case for the "Annular ring" dialog
 * inner radius       = 4 (edit text)
 * outer radius       = 6 (edit text)
 * number of segments = 8 (edit text)
 * number of degrees  = 10 (edit text)
 * layer to use       = 12 (scroll)
 */
INTSML us_annularringdlog(char *paramstart[])
{
	INTBIG itemHit, i, selectedlayer;
	REGISTER INTBIG lx, hx, ly, hy, cx, cy;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *layer, *parent;
	char inner[20], outer[20], segs[20], degrees[20], *pars[6];
	static POLYGON *poly = NOPOLYGON;
	HIGHLIGHT newhigh;

	parent = us_needfacet();
	if (parent == NONODEPROTO) return(0);

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);

	/* display the window view dialog box */
	if (DiaInitDialog(&us_annringdialog) != 0) return(0);
	DiaInitTextDialog(12, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, 0,
		SCSELMOUSE);
	for(i=0; i<el_curtech->layercount; i++)
		DiaStuffLine(12, layername(el_curtech, (INTSML)i));
	DiaSelectLine(12, 0);
	DiaSetText(8, "32");
	DiaSetText(10, "360");

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
	}

	if (itemHit == OK)
	{
		/* figure out which pure-layer node to create */
		selectedlayer = DiaGetCurLine(12);
		layer = getpurelayernode(el_curtech, selectedlayer, 0);
		if (layer == NONODEPROTO) { DiaDoneDialog();   return(0); }

		/* create the pure-layer node */
		cx = (el_curwindowpart->screenlx + el_curwindowpart->screenhx) / 2;
		cy = (el_curwindowpart->screenly + el_curwindowpart->screenhy) / 2;
		lx = cx - (layer->highx - layer->lowx) / 2;
		hx = lx + layer->highx - layer->lowx;
		ly = cy - (layer->highy - layer->lowy) / 2;
		hy = ly + layer->highy - layer->lowy;
		ni = newnodeinst(layer, lx, hx, ly, hy, 0, 0, parent);
		if (ni == NONODEINST) { DiaDoneDialog();   return(0); }
		endobjectchange((INTBIG)ni, VNODEINST);

		/* highlight the pure-layer node */
		us_clearhighlightcount();
		newhigh.status = HIGHFROM;
		newhigh.facet = parent;
		newhigh.fromgeom = ni->geom;
		newhigh.fromport = NOPORTPROTO;
		newhigh.frompoint = 0;
		(void)us_addhighlight(&newhigh);

		/* turn it into an annular ring */
		strcpy(inner, DiaGetText(4));
		strcpy(outer, DiaGetText(6));
		strcpy(segs, DiaGetText(8));
		strcpy(degrees, DiaGetText(10));
		pars[0] = "trace";
		pars[1] = "construct-annulus";
		pars[2] = inner;
		pars[3] = outer;
		pars[4] = segs;
		pars[5] = degrees;
		us_node(6, pars);
	}
	DiaDoneDialog();
	return(0);
}

/****************************** NODE: CREATE LAYOUT TEXT DIALOG ******************************/

/* Node: Make Text Layout */
DIALOGITEM us_spelldialogitems[] =
{
 /*  1 */ {0, {124,172,148,252}, BUTTON, "OK"},
 /*  2 */ {0, {124,12,148,92}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,76,24,180}, POPUP, ""},
 /*  4 */ {0, {8,8,24,72}, MESSAGE, "Size:"},
 /*  5 */ {0, {92,8,108,72}, MESSAGE, "Message:"},
 /*  6 */ {0, {92,76,108,252}, EDITTEXT, ""},
 /*  7 */ {0, {64,76,80,180}, POPUP, ""},
 /*  8 */ {0, {64,8,80,72}, MESSAGE, "Layer:"},
 /*  9 */ {0, {36,76,52,124}, EDITTEXT, ""},
 /* 10 */ {0, {36,8,52,72}, MESSAGE, "Scale:"}
};
DIALOG us_spelldialog = {{75,75,232,336}, "Create Text Layout", 10, us_spelldialogitems};

/*
 * special case for the "Layout Text" dialog
 * Text size       = 3 (popup)
 * Layer           = 7 (popup)
 * Text to place   = 6 (edit text)
 * Scale           = 9 (edit text)
 */
INTSML us_placetextdlog(char *paramstart[])
{
	INTBIG itemHit, layers, fun;
	char **layernames, line[20];
	static INTBIG lastprim = 0, lastfont = 4, lastscale = 1;
	REGISTER NODEPROTO *prim;
	static char *fontnames[12] = {"4 Points", "6 Points", "8 Points", "10 Points",
		"12 Points", "14 Points", "16 Points", "18 Points", "20 Points"};

	/* count all pure-layer nodes in the current technology */
	layers = 0;
	for(prim = el_curtech->firstnodeproto; prim != NONODEPROTO; prim = prim->nextnodeproto)
	{
		fun = (INTSML)((prim->userbits&NFUNCTION) >> NFUNCTIONSH);
		if (fun == NPNODE) layers++;
	}
	if (layers <= 0)
	{
		us_abortcommand("This technology has no pure-layer nodes");
		return(0);
	}

	/* display the text-layout dialog box */
	if (DiaInitDialog(&us_spelldialog) != 0) return(0);
	DiaSetPopup(3, 9, fontnames);
	DiaSetPopupEntry(3, lastfont);
	layernames = (char **)emalloc(layers * (sizeof (char *)), el_tempcluster);
	if (layernames == 0) return(0);
	layers = 0;
	for(prim = el_curtech->firstnodeproto; prim != NONODEPROTO; prim = prim->nextnodeproto)
	{
		fun = (INTSML)((prim->userbits&NFUNCTION) >> NFUNCTIONSH);
		if (fun != NPNODE) continue;
		layernames[layers++] = prim->primname;
	}
	DiaSetPopup(7, layers, layernames);
	if (lastprim < layers) DiaSetPopupEntry(7, lastprim);
	sprintf(line, "%ld", lastscale);
	DiaSetText(9, line);
	if (us_lastplacetextmessage != 0)
		DiaSetText(-6, us_lastplacetextmessage);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
	}
	lastfont = DiaGetPopupEntry(3);
	lastprim = DiaGetPopupEntry(7);
	lastscale = atoi(DiaGetText(9));
	if (us_lastplacetextmessage != 0) efree((char *)us_lastplacetextmessage);
	(void)allocstring(&us_lastplacetextmessage, DiaGetText(6), us_aid->cluster);

	if (itemHit == OK)
	{
		us_layouttext(layernames[lastprim], fontnames[lastfont],
			lastscale, us_lastplacetextmessage);
	}
	DiaDoneDialog();
	efree((char *)layernames);
	return(0);
}

/****************************** NODE CREATION OPTIONS DIALOG ******************************/

/* New node options */
DIALOGITEM us_defnodedialogitems[] =
{
 /*  1 */ {0, {184,388,208,452}, BUTTON, "OK"},
 /*  2 */ {0, {144,388,168,452}, BUTTON, "Cancel"},
 /*  3 */ {0, {32,208,48,318}, EDITTEXT, ""},
 /*  4 */ {0, {32,24,48,205}, MESSAGE, "X size of new primitives:"},
 /*  5 */ {0, {52,24,68,205}, MESSAGE, "Y size of new primitives:"},
 /*  6 */ {0, {52,208,68,318}, EDITTEXT, ""},
 /*  7 */ {0, {8,4,24,142}, MESSAGE, "For primitive:"},
 /*  8 */ {0, {8,144,24,354}, POPUP, ""},
 /*  9 */ {0, {116,24,132,230}, MESSAGE, "Rotation of new nodes:"},
 /* 10 */ {0, {116,240,132,293}, EDITTEXT, ""},
 /* 11 */ {0, {116,316,132,421}, CHECK, "Transposed"},
 /* 12 */ {0, {80,4,81,452}, DIVIDELINE, ""},
 /* 13 */ {0, {92,4,108,152}, MESSAGE, "For all nodes:"},
 /* 14 */ {0, {140,24,156,338}, CHECK, "Disallow modification of locked primitives"},
 /* 15 */ {0, {164,24,180,338}, CHECK, "Move after Duplicate"},
 /* 16 */ {0, {188,24,204,338}, CHECK, "Duplicate/Array/Extract copies ports"}
};
DIALOG us_defnodedialog = {{50,75,267,536}, "New Node Options", 16, us_defnodedialogitems};

/*
 * special case for the "defnode" dialog
 * X size           = 3 (edit text)
 * Y size           = 6 (edit text)
 * Primitive name   = 8 (popup)
 * Rotation         = 10 (edit text)
 * Transposed       = 11 (check)
 * Allow prim mod   = 14 (check)
 * Move after dup   = 15 (check)
 * Dup copies ports = 16 (check)
 */
INTSML us_defnodedlog(char *paramstart[])
{
	INTBIG itemHit, i, j, pangle, numprims;
	INTBIG plx, ply, phx, phy, lx, pxs, pys, nodesize[2];
	REGISTER NODEPROTO *thisprim, *np;
	REGISTER VARIABLE *var;
	REGISTER char **primnames;

	/* display the defnode dialog box */
	if (DiaInitDialog(&us_defnodedialog) != 0) return(0);

	/* construct lists of primitives and facets */
	numprims = 0;
	for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		numprims++;
	primnames = (char **)emalloc(numprims * (sizeof (char *)), el_tempcluster);
	numprims = 0;
	for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		primnames[numprims++] = np->primname;
	DiaSetPopup(8, numprims, primnames);
	efree((char *)primnames);

	/* save existing state */
	for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		defaultnodesize(np, &pxs, &pys);
		nodeprotosizeoffset(np, &plx, &ply, &phx, &phy);
		np->temp1 = pxs - plx - phx;
		np->temp2 = pys - ply - phy;
	}

	/* load defaults for primitives */
	thisprim = el_curtech->firstnodeproto;
	DiaSetPopupEntry(8, thisprim->primindex-1);
	defaultnodesize(thisprim, &pxs, &pys);
	nodeprotosizeoffset(thisprim, &plx, &ply, &phx, &phy);
	DiaSetText(3, latoa(pxs-plx-phx));
	DiaSetText(6, latoa(pys-ply-phy));

	/* load defaults for all nodes */
	var = getvalkey((INTBIG)us_aid, VAID, VINTEGER, us_placement_angle);
	if (var == NOVARIABLE) pangle = 0; else pangle = var->addr;
	DiaSetText(-10, frtoa(pangle%3600*WHOLE/10));
	DiaSetControl(11, (pangle >= 3600 ? 1 : 0));
	DiaSetControl(14, (us_useroptions&NOPRIMCHANGES) != 0 ? 1 : 0);
	DiaSetControl(15, (us_useroptions&NOMOVEAFTERDUP) == 0 ? 1 : 0);
	DiaSetControl(16, (us_useroptions&DUPCOPIESPORTS) != 0 ? 1 : 0);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == 8)
		{
			i = DiaGetPopupEntry(8);
			for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				if (np->primindex == i+1) break;
			thisprim = np;
			defaultnodesize(thisprim, &pxs, &pys);
			nodeprotosizeoffset(thisprim, &plx, &ply, &phx, &phy);
			DiaSetText(3, latoa(pxs-plx-phx));
			DiaSetText(6, latoa(pys-ply-phy));
			continue;
		}
		if (itemHit == 3 || itemHit == 6)
		{
			thisprim->temp1 = atola(DiaGetText(3));
			thisprim->temp2 = atola(DiaGetText(6));
			continue;
		}
		if (itemHit == 11 || itemHit == 14 || itemHit == 15 || itemHit == 16)
		{
			DiaSetControl(itemHit, 1-DiaGetControl(itemHit));
			continue;
		}
	}

	paramstart[0] = "x";
	if (itemHit != CANCEL)
	{
		/* handle primitive size changes */
		for(np = el_curtech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			defaultnodesize(np, &pxs, &pys);
			nodeprotosizeoffset(np, &plx, &ply, &phx, &phy);
			if (np->temp1 != pxs - plx - phx ||
				np->temp2 != pys - ply - phy)
			{
				nodesize[0] = np->temp1+plx+phx;
				nodesize[1] = np->temp2+ply+phy;
				(void)setvalkey((INTBIG)np, VNODEPROTO, el_node_default_size,
					(INTBIG)nodesize, VINTEGER|VISARRAY|(2<<VLENGTHSH));
			}
		}

		/* handle changes to all nodes */
		lx = us_useroptions;
		i = DiaGetControl(14);
		if (i != 0) lx |= NOPRIMCHANGES; else lx &= ~NOPRIMCHANGES;
		i = DiaGetControl(15);
		if (i == 0) lx |= NOMOVEAFTERDUP; else lx &= ~NOMOVEAFTERDUP;
		i = DiaGetControl(16);
		if (i != 0) lx |= DUPCOPIESPORTS; else lx &= ~DUPCOPIESPORTS;
		(void)setvalkey((INTBIG)us_aid, VAID, us_optionflags, lx, VINTEGER);
		j = (atofr(DiaGetText(10))*10/WHOLE) % 3600;
		if (DiaGetControl(11) != 0) j += 3600;
		if (j != pangle)
			(void)setvalkey((INTBIG)us_aid, VAID, us_placement_angle, j, VINTEGER);
	}
	DiaDoneDialog();
	return(1);
}

/****************************** NODE INFORMATION DIALOGS ******************************/

/* Resistance */
DIALOGITEM us_resistancedialogitems[] =
{
 /*  1 */ {0, {40,192,64,256}, BUTTON, "OK"},
 /*  2 */ {0, {40,16,64,80}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,24,24,118}, EDITTEXT, ""},
 /*  4 */ {0, {8,128,24,239}, POPUP, "Ohms"},
 /*  5 */ {0, {40,104,64,168}, BUTTON, "More..."}
};
DIALOG us_resistancedialog = {{50,75,124,345}, "Resistance", 5, us_resistancedialogitems};

/* Capacitance */
DIALOGITEM us_capacitancedialogitems[] =
{
 /*  1 */ {0, {40,176,64,240}, BUTTON, "OK"},
 /*  2 */ {0, {40,16,64,80}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,16,24,110}, EDITTEXT, ""},
 /*  4 */ {0, {8,120,24,246}, POPUP, "Milli-farads"},
 /*  5 */ {0, {40,96,64,160}, BUTTON, "More..."}
};
DIALOG us_capacitancedialog = {{50,75,123,330}, "Capacitance", 5, us_capacitancedialogitems};

/* Inductance */
DIALOGITEM us_inductancedialogitems[] =
{
 /*  1 */ {0, {40,168,64,232}, BUTTON, "OK"},
 /*  2 */ {0, {40,8,64,72}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,16,24,110}, EDITTEXT, ""},
 /*  4 */ {0, {8,120,24,237}, POPUP, "Henrys"},
 /*  5 */ {0, {40,88,64,152}, BUTTON, "More..."}
};
DIALOG us_inductancedialog = {{50,75,126,321}, "Inductance", 5, us_inductancedialogitems};

/* VCCS Transconductance */
DIALOGITEM us_vccsdialogitems[] =
{
 /*  1 */ {0, {40,176,64,240}, BUTTON, "OK"},
 /*  2 */ {0, {40,16,64,80}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,16,24,110}, EDITTEXT, ""},
 /*  4 */ {0, {8,120,24,247}, POPUP, "Siemens"},
 /*  5 */ {0, {40,96,64,160}, BUTTON, "More..."}
};
DIALOG us_vccsdialog = {{50,75,125,331}, "VCCS Transconductance", 5, us_vccsdialogitems};

/* Area */
DIALOGITEM us_areadialogitems[] =
{
 /*  1 */ {0, {40,184,64,248}, BUTTON, "OK"},
 /*  2 */ {0, {40,8,64,72}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,80,24,174}, EDITTEXT, ""},
 /*  4 */ {0, {40,96,64,160}, BUTTON, "More..."}
};
DIALOG us_areadialog = {{50,75,124,333}, "Area", 4, us_areadialogitems};

/* Width/Length */
DIALOGITEM us_widlendialogitems[] =
{
 /*  1 */ {0, {88,184,112,248}, BUTTON, "OK"},
 /*  2 */ {0, {88,8,112,72}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,108,24,202}, EDITTEXT, ""},
 /*  4 */ {0, {8,8,24,59}, MESSAGE, "Width:"},
 /*  5 */ {0, {32,108,48,202}, EDITTEXT, ""},
 /*  6 */ {0, {32,8,48,60}, MESSAGE, "Length:"},
 /*  7 */ {0, {56,108,72,202}, EDITTEXT, ""},
 /*  8 */ {0, {56,8,72,96}, MESSAGE, "SPICE name:"},
 /*  9 */ {0, {88,96,112,160}, BUTTON, "More"}
};
DIALOG us_widlendialog = {{50,75,171,332}, "Transistor Information", 9, us_widlendialogitems};

/* Gain */
DIALOGITEM us_gaindialogitems[] =
{
 /*  1 */ {0, {40,176,64,240}, BUTTON, "OK"},
 /*  2 */ {0, {40,16,64,80}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,88,24,213}, EDITTEXT, ""},
 /*  4 */ {0, {8,40,24,83}, MESSAGE, "Gain:"},
 /*  5 */ {0, {40,96,64,160}, BUTTON, "More..."}
};
DIALOG us_gaindialog = {{50,75,123,334}, "Gain", 5, us_gaindialogitems};

/* Source Information */
DIALOGITEM us_sourcedialogitems[] =
{
 /*  1 */ {0, {72,268,96,336}, BUTTON, "OK"},
 /*  2 */ {0, {72,20,96,88}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,112,24,264}, POPUP, "Source Type"},
 /*  4 */ {0, {36,8,52,100}, MESSAGE, "SPICE Card:"},
 /*  5 */ {0, {8,8,24,100}, MESSAGE, "Source type:"},
 /*  6 */ {0, {36,112,52,348}, EDITTEXT, ""},
 /*  7 */ {0, {72,144,96,212}, BUTTON, "More..."}
};
DIALOG us_sourcedialog = {{75,75,186,440}, "Source Information", 7, us_sourcedialogitems};

void us_movedialogtonode(DIALOG*, GEOM*);

/*
 * routine to move dialog "dia" to a place close to object "geom" in the current window
 */
void us_movedialogtonode(DIALOG *dia, GEOM *geom)
{
	REGISTER INTBIG x, y, wid, hei;
	INTSML swid, shei;

	if (geom == NOGEOM) return;
	if (el_curwindowpart == NOWINDOWPART) return;
	wid = dia->windowRect.right - dia->windowRect.left;
	hei = dia->windowRect.bottom - dia->windowRect.top;

	/* see which side the dialog goes on */
	x = applyxscale(el_curwindowpart, geom->highx-el_curwindowpart->screenlx) + el_curwindowpart->uselx;
	us_getwindowsize(el_curwindowpart->frame, &swid, &shei);
	if (x + wid > swid)
	{
		/* attach on left side */
		x = applyxscale(el_curwindowpart, geom->lowx-el_curwindowpart->screenlx) + el_curwindowpart->uselx;
		dia->windowRect.left = (INTSML)(x - wid);   dia->windowRect.right = (INTSML)x;
	} else
	{
		/* attach on right side */
		dia->windowRect.left = (INTSML)x;   dia->windowRect.right = (INTSML)(x + wid);
	}

	y = applyyscale(el_curwindowpart, geom->highy-el_curwindowpart->screenly) + el_curwindowpart->usely;
	if (shei-y-hei <= 39)
	{
		/* attach on bottom side */
		y = applyyscale(el_curwindowpart, geom->lowy-el_curwindowpart->screenly) + el_curwindowpart->usely;
		dia->windowRect.bottom = (INTSML)(shei - y + hei);
		dia->windowRect.top = (INTSML)(shei - y);
	} else
	{
		/* attach on top side */
		dia->windowRect.bottom = (INTSML)(shei - y);
		dia->windowRect.top = (INTSML)(shei - y - hei);
	}
}

/*
 * special case for the "resistance" dialog
 * resistance value   = 3 (edit)
 * units              = 4 (popup)
 * More...            = 5 (button)
 */
INTSML us_resistancedlog(char *prompt, char *paramstart[], char *initial)
{
	INTBIG itemHit, i;
	static char *resnames[] = {"Ohms", "Kilo-ohms", "Mega-ohms", "Giga-Ohms"};
	REGISTER HIGHLIGHT *high;
	char line[70];

	/* if an initial value is given, search for highlighted node and move dialog to it */
	if (initial != 0)
	{
		high = us_getonehighlight();
		if (high != NOHIGHLIGHT) us_movedialogtonode(&us_resistancedialog, high->fromgeom);
	}

	/* display the resistance dialog box */
	if (namesame(prompt, "ccvs") == 0) (void)strcpy(line, "CCVS transresistance"); else
		(void)sprintf(line, "%s resistance", prompt);
	us_resistancedialog.movable = line;
	if (initial != 0) us_resistancedialog.items = 5; else us_resistancedialog.items = 4;
	if (DiaInitDialog(&us_resistancedialog) != 0) return(0);
	DiaSetPopup(4, 4, resnames);
	if (initial == 0) DiaSetText(3, "100"); else
	{
		i = strlen(initial);
		if (i > 0 && namesame(&initial[i-1], "K") == 0)
		{
			DiaSetPopupEntry(4, 1);
			initial[i-1] = 0;
			DiaSetText(3, initial);
			initial[i-1] = 'K';
		} else if (i > 2 && namesame(&initial[i-3], "MEG") == 0)
		{
			DiaSetPopupEntry(4, 2);
			initial[i-3] = 0;
			DiaSetText(3, initial);
			initial[i-3] = 'M';
		} else if (i > 0 && namesame(&initial[i-1], "G") == 0)
		{
			DiaSetPopupEntry(4, 3);
			initial[i-1] = 0;
			DiaSetText(3, initial);
			initial[i-1] = 'G';
		} else DiaSetText(3, initial);
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK || itemHit == 5) break;
	}

	(void)initinfstr();
	(void)addstringtoinfstr(DiaGetText(3));
	i = DiaGetPopupEntry(4);
	switch (i)
	{
		case 1: (void)addstringtoinfstr("K");     break;
		case 2: (void)addstringtoinfstr("MEG");   break;
		case 3: (void)addstringtoinfstr("G");     break;
	}
	paramstart[0] = returninfstr();
	DiaDoneDialog();
	if (itemHit == OK) return(1);
	if (itemHit == 5) (void)us_showdlog(paramstart, 0);
	return(0);
}

/*
 * special case for the "capacitance" dialog
 * capacitance value   = 3 (edit)
 * units               = 4 (popup)
 * More...             = 5 (button)
 */
INTSML us_capacitancedlog(char *paramstart[], char *initial)
{
	INTBIG itemHit, i;
	static char *capnames[] = {"Femto-farads", "Pico-farads", "Micro-farads", "Milli-farads",
		"Farads"};
	REGISTER HIGHLIGHT *high;

	/* if an initial value is given, search for highlighted node and move dialog to it */
	if (initial != 0)
	{
		high = us_getonehighlight();
		if (high != NOHIGHLIGHT) us_movedialogtonode(&us_capacitancedialog, high->fromgeom);
	}

	/* display the capacitance dialog box */
	if (initial != 0) us_capacitancedialog.items = 5; else us_capacitancedialog.items = 4;
	if (DiaInitDialog(&us_capacitancedialog) != 0) return(0);
	DiaSetPopup(4, 5, capnames);
	if (initial == 0)
	{
		DiaSetText(3, "100");
		DiaSetPopupEntry(4, 3);
	} else
	{
		i = strlen(initial);
		if (i > 0 && namesame(&initial[i-1], "F") == 0)
		{
			initial[i-1] = 0;
			DiaSetText(3, initial);
			initial[i-1] = 'F';
		} else if (i > 0 && namesame(&initial[i-1], "P") == 0)
		{
			DiaSetPopupEntry(4, 1);
			initial[i-3] = 0;
			DiaSetText(3, initial);
			initial[i-3] = 'P';
		} else if (i > 0 && namesame(&initial[i-1], "U") == 0)
		{
			DiaSetPopupEntry(4, 2);
			initial[i-1] = 0;
			DiaSetText(3, initial);
			initial[i-1] = 'U';
		} else if (i > 0 && namesame(&initial[i-1], "M") == 0)
		{
			DiaSetPopupEntry(4, 3);
			initial[i-1] = 0;
			DiaSetText(3, initial);
			initial[i-1] = 'M';
		} else
		{
			DiaSetPopupEntry(4, 4);
			DiaSetText(3, initial);
		}
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK || itemHit == 5) break;
	}

	(void)initinfstr();
	(void)addstringtoinfstr(DiaGetText(3));
	i = DiaGetPopupEntry(4);
	switch (i)
	{
		case 0: (void)addstringtoinfstr("F");   break;
		case 1: (void)addstringtoinfstr("P");   break;
		case 2: (void)addstringtoinfstr("U");   break;
		case 3: (void)addstringtoinfstr("M");   break;
	}
	paramstart[0] = returninfstr();
	DiaDoneDialog();
	if (itemHit == OK) return(1);
	if (itemHit == 5) (void)us_showdlog(paramstart, 0);
	return(0);
}

/*
 * special case for the "inductance" dialog
 * inductance value   = 3 (edit)
 * units              = 4 (popup)
 * More...            = 5 (button)
 */
INTSML us_inductancedlog(char *paramstart[], char *initial)
{
	INTBIG itemHit, i;
	static char *indnames[] = {"Micro-henrys", "Milli-henrys", "Henrys"};
	REGISTER HIGHLIGHT *high;

	/* if an initial value is given, search for highlighted node and move dialog to it */
	if (initial != 0)
	{
		high = us_getonehighlight();
		if (high != NOHIGHLIGHT) us_movedialogtonode(&us_inductancedialog, high->fromgeom);
	}

	/* display the inductance dialog box */
	if (initial != 0) us_inductancedialog.items = 5; else us_inductancedialog.items = 4;
	if (DiaInitDialog(&us_inductancedialog) != 0) return(0);
	DiaSetPopup(4, 3, indnames);
	if (initial == 0)
	{
		DiaSetText(3, "100");
		DiaSetPopupEntry(4, 2);
	} else
	{
		i = strlen(initial);
		if (i > 0 && namesame(&initial[i-1], "U") == 0)
		{
			initial[i-1] = 0;
			DiaSetText(3, initial);
			initial[i-1] = 'U';
		} else if (i > 0 && namesame(&initial[i-1], "M") == 0)
		{
			DiaSetPopupEntry(4, 1);
			initial[i-1] = 0;
			DiaSetText(3, initial);
			initial[i-1] = 'M';
		} else
		{
			DiaSetPopupEntry(4, 2);
			DiaSetText(3, initial);
		}
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK || itemHit == 5) break;
	}

	(void)initinfstr();
	(void)addstringtoinfstr(DiaGetText(3));
	i = DiaGetPopupEntry(4);
	switch (i)
	{
		case 0: (void)addstringtoinfstr("U");   break;
		case 1: (void)addstringtoinfstr("M");   break;
	}
	paramstart[0] = returninfstr();
	DiaDoneDialog();
	if (itemHit == OK) return(1);
	if (itemHit == 5) (void)us_showdlog(paramstart, 0);
	return(0);
}

/*
 * special case for the "VCCS Transconductance" dialog
 * siemens value       = 3 (edit)
 * units               = 4 (popup)
 * More...             = 5 (button)
 */
INTSML us_vccsdlog(char *paramstart[], char *initial)
{
	INTBIG itemHit, i;
	static char *sienames[] = {"Micro-siemens", "Milli-siemens", "Siemens"};
	REGISTER HIGHLIGHT *high;

	/* if an initial value is given, search for highlighted node and move dialog to it */
	if (initial != 0)
	{
		high = us_getonehighlight();
		if (high != NOHIGHLIGHT) us_movedialogtonode(&us_vccsdialog, high->fromgeom);
	}

	/* display the siemens dialog box */
	if (initial != 0) us_vccsdialog.items = 5; else us_vccsdialog.items = 4;
	if (DiaInitDialog(&us_vccsdialog) != 0) return(0);
	DiaSetPopup(4, 3, sienames);
	if (initial == 0)
	{
		DiaSetText(3, "100");
		DiaSetPopupEntry(4, 2);
	} else
	{
		i = strlen(initial);
		if (i > 0 && namesame(&initial[i-1], "U") == 0)
		{
			initial[i-1] = 0;
			DiaSetText(3, initial);
			initial[i-1] = 'U';
		} else if (i > 0 && namesame(&initial[i-1], "M") == 0)
		{
			DiaSetPopupEntry(4, 1);
			initial[i-1] = 0;
			DiaSetText(3, initial);
			initial[i-1] = 'M';
		} else
		{
			DiaSetPopupEntry(4, 2);
			DiaSetText(3, initial);
		}
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK || itemHit == 5) break;
	}

	(void)initinfstr();
	(void)addstringtoinfstr(DiaGetText(3));
	i = DiaGetPopupEntry(4);
	switch (i)
	{
		case 0: (void)addstringtoinfstr("U");   break;
		case 1: (void)addstringtoinfstr("M");   break;
	}
	paramstart[0] = returninfstr();
	DiaDoneDialog();
	if (itemHit == OK) return(1);
	if (itemHit == 5) (void)us_showdlog(paramstart, 0);
	return(0);
}

/*
 * special case for the "area" dialog
 * area value           = 3 (edit)
 * More...              = 4 (button)
 */
INTSML us_areadlog(char *prompt, char *paramstart[], char *initial)
{
	INTBIG itemHit;
	static char line[80];
	REGISTER HIGHLIGHT *high;

	/* if an initial value is given, search for highlighted node and move dialog to it */
	if (initial != 0)
	{
		high = us_getonehighlight();
		if (high != NOHIGHLIGHT) us_movedialogtonode(&us_areadialog, high->fromgeom);
	}

	/* display the area dialog box */
	(void)sprintf(line, "%s area", prompt);
	us_areadialog.movable = line;
	if (initial != 0) us_areadialog.items = 4; else us_areadialog.items = 3;
	if (DiaInitDialog(&us_areadialog) != 0) return(0);
	if (initial == 0) DiaSetText(3, "10"); else
		DiaSetText(3, initial);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK || itemHit == 4) break;
	}

	strcpy(line, DiaGetText(3));
	paramstart[0] = line;
	DiaDoneDialog();
	if (itemHit == OK) return(1);
	if (itemHit == 4) (void)us_showdlog(paramstart, 0);
	return(0);
}

/*
 * special case for the "width/length" dialog
 * width value      = 3 (edit)
 * length value     = 5 (edit)
 * More...          = 9 (button)
 * SPICE name       = 7 (edit)
 */
INTSML us_widlendlog(char *paramstart[], char *initial)
{
	INTBIG itemHit;
	REGISTER char *pt, *ptex, *start;
	static char line[90];
	REGISTER HIGHLIGHT *high;

	/* if an initial value is given, search for highlighted node and move dialog to it */
	if (initial != 0)
	{
		high = us_getonehighlight();
		if (high != NOHIGHLIGHT) us_movedialogtonode(&us_widlendialog, high->fromgeom);
	}

	/* display the width/length dialog box */
	if (initial != 0) us_widlendialog.items = 9; else us_widlendialog.items = 8;
	if (DiaInitDialog(&us_widlendialog) != 0) return(0);
	if (initial == 0)
	{
		DiaSetText(3, "2");
		DiaSetText(5, "2");
		DiaSetText(7, "");
	} else
	{
		for(pt = initial; *pt != 0 && *pt != '/'; pt++) ;
		if (*pt == '/')
		{
			*pt = 0;
			DiaSetText(3, initial);
			*pt++ = '/';
			start = pt;
			for( ; *pt != 0 && *pt != '/'; pt++) ;
			if (*pt == '/')
			{
				*pt = 0;
				DiaSetText(5, start);
				*pt++ = '/';
				DiaSetText(7, pt);
			} else
			{
				DiaSetText(5, start);
				DiaSetText(7, "");
			}
		} else
		{
			DiaSetText(3, initial);
			DiaSetText(5, initial);
			DiaSetText(7, "");
		}
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK || itemHit == 9) break;
	}

	strcpy(line, DiaGetText(3));
	pt = DiaGetText(5);
	while (*pt == ' ' || *pt == '\t') pt++;
	ptex = DiaGetText(7);
	while (*ptex == ' ' || *ptex == '\t') ptex++;

	if (*pt != 0 || *ptex != 0)
	{
		strcat(line, "/");
		strcat(line, pt);
	}
	if (*ptex != 0)
	{
		strcat(line, "/");
		strcat(line, ptex);
	}
	paramstart[0] = line;
	DiaDoneDialog();
	if (itemHit == OK) return(1);
	if (itemHit == 9) (void)us_showdlog(paramstart, 0);
	return(0);
}

/*
 * special case for the "gain" dialog
 * area value           = 3 (edit)
 * units                = 4 (popup)
 * More...              = 5 (button)
 */
INTSML us_gaindlog(char *prompt, char *paramstart[], char *initial)
{
	INTBIG itemHit;
	static char line[50];
	REGISTER HIGHLIGHT *high;

	/* if an initial value is given, search for highlighted node and move dialog to it */
	if (initial != 0)
	{
		high = us_getonehighlight();
		if (high != NOHIGHLIGHT) us_movedialogtonode(&us_gaindialog, high->fromgeom);
	}

	/* display the gain dialog box */
	(void)sprintf(line, "%s gain", prompt);
	us_gaindialog.movable = line;
	if (initial != 0) us_gaindialog.items = 5; else us_gaindialog.items = 4;
	if (DiaInitDialog(&us_gaindialog) != 0) return(0);
	if (initial == 0) DiaSetText(3, "10"); else DiaSetText(3, initial);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK || itemHit == 5) break;
	}

	paramstart[0] = us_putintoinfstr(DiaGetText(3));
	DiaDoneDialog();
	if (itemHit == OK) return(1);
	if (itemHit == 5) (void)us_showdlog(paramstart, 0);
	return(0);
}

/*
 * special case for the "source" dialog
 * source type          = 3 (popup)
 * SPICE card           = 6 (edit text)
 * More...              = 7 (button)
 */
INTSML us_sourcedlog(char *paramstart[], char *initial)
{
	INTBIG itemHit, sourcetype;
	char *extra;
	REGISTER NODEINST *ni;
	enum {kSourceUnknown=0, kSourceVoltage, kSourceCurrent, kSourceCurrentMeter,
		kSourceTransient, kSourceDC, kSourceAC, kSourceNodeSet, kSourceExtension,
		kSourceBulk, kNodeSpecial};
	static char *sourcetypes[] = {"Unknown", "Voltage Source", "Current Source",
		"Current Meter", "Transient Analysis", "DC Analysis", "AC Analysis",
		"Node Set", "Extension", "Bulk", "Special"};

	/* if an initial value is given, search for highlighted node and move dialog to it */
	ni = (NODEINST *)us_getobject(OBJNODEINST, 1);
	if (ni == NONODEINST)
	{
		us_abortcommand("Select a source node first");
		return(0);
	}
	if (initial != 0)
		us_movedialogtonode(&us_sourcedialog, ni->geom);

	/* display the gain dialog box */
	if (DiaInitDialog(&us_sourcedialog) != 0) return(0);

	switch (nodefunction(ni, &extra))
	{
		case NPSOURCE:   sourcetype = kSourceUnknown;        break;
		case NPSOURCEV:  sourcetype = kSourceVoltage;        break;
		case NPSOURCEC:  sourcetype = kSourceCurrent;        break;
		case NPSOURCECM: sourcetype = kSourceCurrentMeter;   break;
		case NPSOURCET:  sourcetype = kSourceTransient;      break;
		case NPSOURCEDC: sourcetype = kSourceDC;             break;
		case NPSOURCEAC: sourcetype = kSourceAC;             break;
		case NPSOURCEN:  sourcetype = kSourceNodeSet;        break;
		case NPSOURCEX:  sourcetype = kSourceExtension;      break;
		case NPSOURCEB:  sourcetype = kSourceBulk;           break;
		case NPSOURCES:  sourcetype = kNodeSpecial;          break;
	}

	if (initial != 0) DiaSetText(-6, extra);
	DiaSetPopup(3, 10, sourcetypes);
	DiaSetPopupEntry(3, sourcetype);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK || itemHit == 7) break;
	}

	(void)initinfstr();
	switch (DiaGetPopupEntry(3))
	{
		case kSourceUnknown:
		case kSourceVoltage:      (void)addstringtoinfstr("v/");   break;
		case kSourceCurrent:      (void)addstringtoinfstr("c/");   break;
		case kSourceCurrentMeter: (void)addstringtoinfstr("cm/");  break;
		case kSourceTransient:    (void)addstringtoinfstr("t/");   break;
		case kSourceDC:           (void)addstringtoinfstr("vd/");  break;
		case kSourceAC:           (void)addstringtoinfstr("a/");   break;
		case kSourceNodeSet:      (void)addstringtoinfstr("n/");   break;
		case kSourceExtension:    (void)addstringtoinfstr("x/");   break;
		case kSourceBulk:         (void)addstringtoinfstr("b/");   break;
		case kNodeSpecial:        (void)addstringtoinfstr("s/");   break;
	}
	(void)addstringtoinfstr(DiaGetText(6));
	paramstart[0] = returninfstr();
	DiaDoneDialog();
	if (itemHit == OK) return(1);
	if (itemHit == 7) (void)us_showdlog(paramstart, 0);
	return(0);
}

/****************************** NODE SIZE DIALOG ******************************/

/* Node Size */
DIALOGITEM us_nodesizedialogitems[] =
{
 /*  1 */ {0, {104,132,128,212}, BUTTON, "OK"},
 /*  2 */ {0, {104,4,128,84}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,12,24,92}, MESSAGE|INACTIVE, "X Size:"},
 /*  4 */ {0, {36,12,52,92}, MESSAGE|INACTIVE, "Y Size:"},
 /*  5 */ {0, {8,100,24,200}, EDITTEXT, ""},
 /*  6 */ {0, {36,100,52,200}, EDITTEXT, ""},
 /*  7 */ {0, {64,4,96,212}, MESSAGE|INACTIVE, ""}
};
DIALOG us_nodesizedialog = {{75,75,212,297}, "Set Node Size", 7, us_nodesizedialogitems};

/*
 * special case for the "node size" dialog
 * X size        = 5 (edit text)
 * Y size        = 6 (edit text)
 * Extra message = 7 (message)
 */
INTSML us_nodesizedlog(char *paramstart[])
{
	INTBIG itemHit, allmanhattan;
	INTSML ret;
	REGISTER INTBIG i;
	static char x[20], y[20];
	REGISTER GEOM **list;
	REGISTER NODEINST *ni;

	/* display the node size dialog box */
	if (DiaInitDialog(&us_nodesizedialog) != 0) return(0);

	/* see if there are nonmanhattan nodes selected */
	allmanhattan = 1;
	list = us_gethighlighted(OBJNODEINST);
	for(i=0; list[i] != NOGEOM; i++)
	{
		ni = list[i]->entryaddr.ni;
		if ((ni->rotation % 900) != 0) allmanhattan = 0;
	}
	if (allmanhattan == 0)
	{
		DiaSetText(7, "Nonmanhattan nodes selected: sizing in unrotated directions");
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
	}

	ret = 0;
	if (itemHit != CANCEL)
	{
		strcpy(x, DiaGetText(5));
		strcpy(y, DiaGetText(6));
		paramstart[ret++] = x;
		paramstart[ret++] = y;
		if (allmanhattan != 0) paramstart[ret++] = "use-transformation";
	}
	DiaDoneDialog();
	return(ret);
}

/****************************** OPTION SAVING DIALOG ******************************/

/* Option Saving */
DIALOGITEM us_optionsavingdialogitems[] =
{
 /*  1 */ {0, {232,216,256,288}, BUTTON, "OK"},
 /*  2 */ {0, {232,12,256,84}, BUTTON, "Cancel"},
 /*  3 */ {0, {4,8,20,291}, MESSAGE, "Marked options are saved with library."},
 /*  4 */ {0, {44,8,220,298}, SCROLL, ""},
 /*  5 */ {0, {24,8,40,294}, MESSAGE, "Click an option to change its mark."}
};
DIALOG us_optionsavingdialog = {{50,75,318,382}, "Option Saving", 5, us_optionsavingdialogitems};

/*
 * special case for the "Option Saving" dialog
 * List of options = 4 (scroll)
 */
INTSML us_optionsavingdlog(char *paramstart[])
{
	INTBIG itemHit, i, err;
	INTBIG bits, savebits;
	REGISTER VARIABLE *var;
	char *name, *msg;

	var = getval((INTBIG)el_curlib, VLIBRARY, VINTEGER, "LIB_save_options");
	if (var == NOVARIABLE) savebits = 0; else savebits = var->addr;

	/* display the visibility dialog box */
	if (DiaInitDialog(&us_optionsavingdialog) != 0) return(0);
	DiaInitTextDialog(4, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);
	for(i=0; ; i++)
	{
		err = describeoptions(i, &name, &bits);
		if (err != 0) break;
		(void)initinfstr();
		if ((savebits&bits) != 0) (void)addtoinfstr('>'); else
			(void)addtoinfstr(' ');
		(void)addtoinfstr(' ');
		(void)addstringtoinfstr(name);
		DiaStuffLine(4, returninfstr());
	}
	DiaSelectLine(4, -1);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == 4)
		{
			/* toggle layer visibility for this layer */
			i = DiaGetCurLine(4);
			msg = DiaGetScrollLine(4, i);
			if (*msg == ' ') *msg = '>'; else *msg = ' ';
			DiaSetScrollLine(4, i, msg);
			DiaSelectLine(4, -1);
			continue;
		}
	}

	if (itemHit == OK)
	{
		savebits = 0;
		for(i=0; ; i++)
		{
			err = describeoptions(i, &name, &bits);
			if (err != 0) break;
			msg = DiaGetScrollLine(4, i);
			if (*msg == '>') savebits |= bits;
		}
		(void)setval((INTBIG)el_curlib, VLIBRARY, "LIB_save_options",
			savebits, VINTEGER);
	}
	DiaDoneDialog();
	return(0);
}

/****************************** PORT CREATION DIALOG ******************************/

/* New port */
DIALOGITEM us_portdialogitems[] =
{
 /*  1 */ {0, {88,232,112,304}, BUTTON, "OK"},
 /*  2 */ {0, {88,32,112,104}, BUTTON, "Cancel"},
 /*  3 */ {0, {32,160,48,324}, POPUP, "Input"},
 /*  4 */ {0, {8,104,24,320}, EDITTEXT, ""},
 /*  5 */ {0, {8,16,24,96}, MESSAGE, "Port name:"},
 /*  6 */ {0, {56,16,72,136}, CHECK, "Always drawn"},
 /*  7 */ {0, {32,16,48,159}, MESSAGE, "Port characteristics:"},
 /*  8 */ {0, {56,168,72,288}, CHECK, "Body only"}
};
DIALOG us_portdialog = {{50,75,174,410}, "Create Port on Highlighted", 8, us_portdialogitems};

/*
 * special case for the "port" dialog
 * Name             = 4 (edit text)
 * Characteristics  = 3 (popup)
 * Always drawn     = 6 (check)
 * Body only        = 8 (check)
 */
INTSML us_portdlog(char *paramstart[])
{
	INTBIG itemHit;
	static INTBIG lastchar = 0;
	struct portbut { INTBIG button;  char *name; };
	static char *charnames[15] = {"Unknown", "Input", "Output", "Bidirectional", "Power",
		"Ground", "Reference Output", "Reference Input", "Clock", "Clock phase 1",
		"Clock phase 2", "Clock phase 3", "Clock phase 4", "Clock phase 5", "Clock phase 6"};
	static char *intnames[15] = {"", " input", " output", " bidirectional", " power", " ground",
		" refout", " refin", " clock", " clock1", " clock2", " clock3", " clock4", " clock5",
		" clock6"};

	/* display the port dialog box */
	if (DiaInitDialog(&us_portdialog) != 0) return(0);
	DiaSetText(-4, "");
	DiaSetPopup(3, 15, charnames);
	DiaSetPopupEntry(3, lastchar);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK && DiaValidEntry(4)) break;
		if (itemHit == 6 || itemHit == 8)
			DiaSetControl(itemHit, 1-DiaGetControl(itemHit));
	}

	paramstart[0] = "";
	if (itemHit != CANCEL)
	{
		(void)initinfstr();
		(void)addstringtoinfstr(DiaGetText(4));
		lastchar = DiaGetPopupEntry(3);
		(void)addstringtoinfstr(intnames[lastchar]);
		if (DiaGetControl(6) != 0) (void)addstringtoinfstr(" always-drawn");
		if (DiaGetControl(8) != 0) (void)addstringtoinfstr(" body-only");
		paramstart[0] = returninfstr();
	}
	DiaDoneDialog();
	return(1);
}

/****************************** PORT DISPLAY DIALOG ******************************/

/* Port Display Options */
DIALOGITEM us_portdisplaydialogitems[] =
{
 /*  1 */ {0, {100,92,124,156}, BUTTON, "OK"},
 /*  2 */ {0, {100,8,124,72}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,8,24,155}, RADIO, "Full Port Names"},
 /*  4 */ {0, {28,8,44,155}, RADIO, "Short Port Names"},
 /*  5 */ {0, {48,8,64,155}, RADIO, "Ports as Crosses"},
 /*  6 */ {0, {68,8,84,155}, RADIO, "No Ports Shown"}
};
DIALOG us_portdisplaydialog = {{133,131,270,304}, "Port Display Options", 6, us_portdisplaydialogitems};

/*
 * special case for the "Port Display" dialog
 * Full Port Names     = 3 (button)
 * Short Port Names    = 4 (button)
 * Ports as Crosses    = 5 (button)
 * No Ports Shown      = 6 (button)
 */
INTSML us_portdisplaydlog(char *paramstart[])
{
	INTBIG itemHit;
	INTBIG labels;
	WINDOWPART *w;

	/* display the port labels dialog */
	if (us_needwindow()) return(0);
	if (DiaInitDialog(&us_portdisplaydialog) != 0) return(0);
	switch (us_useroptions&PORTLABELS)
	{
		case PORTSFULL:  DiaSetControl(3, 1);   break;
		case PORTSSHORT: DiaSetControl(4, 1);   break;
		case PORTSCROSS: DiaSetControl(5, 1);   break;
		case PORTSOFF:   DiaSetControl(6, 1);   break;
	}
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK) break;
		if (itemHit == 3 || itemHit == 4 || itemHit == 5 || itemHit == 6)
		{
			DiaSetControl(3, 0);
			DiaSetControl(4, 0);
			DiaSetControl(5, 0);
			DiaSetControl(6, 0);
			DiaSetControl(itemHit, 1);
			continue;
		}
	}
	if (DiaGetControl(3) != 0) labels = PORTSFULL; else
		if (DiaGetControl(4) != 0) labels = PORTSSHORT; else
			if (DiaGetControl(5) != 0) labels = PORTSCROSS; else
				labels = PORTSOFF;
	DiaDoneDialog();
	if (itemHit == OK)
	{
		(void)setvalkey((INTBIG)us_aid, VAID, us_optionflags,
			(us_useroptions & ~PORTLABELS) | labels, VINTEGER);
		us_pushhighlight();
		us_clearhighlightcount();
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			startobjectchange((INTBIG)w, VWINDOWPART);
			if (w->redisphandler != 0) (*w->redisphandler)(w);
			endobjectchange((INTBIG)w, VWINDOWPART);
		}
		(void)us_pophighlight(0);
	}
	return(0);
}

/****************************** PRINT OPTIONS DIALOG ******************************/

/* Printing Options */
DIALOGITEM us_printingoptdialogitems[] =
{
 /*  1 */ {0, {8,356,32,414}, BUTTON, "OK"},
 /*  2 */ {0, {40,356,64,414}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,148,24,344}, RADIO, "Plot only Highlighted Area"},
 /*  4 */ {0, {8,8,24,142}, RADIO, "Plot Entire Facet"},
 /*  5 */ {0, {72,104,88,220}, CHECK, "Encapsulated"},
 /*  6 */ {0, {36,8,52,163}, CHECK, "Plot Date In Corner"},
 /*  7 */ {0, {240,108,256,166}, RADIO, "HPGL"},
 /*  8 */ {0, {240,180,256,254}, RADIO, "HPGL/2"},
 /*  9 */ {0, {260,20,276,187}, RADIO, "HPGL/2 plot fills page"},
 /* 10 */ {0, {280,20,296,177}, RADIO, "HPGL/2 plot fixed at:"},
 /* 11 */ {0, {280,180,296,240}, EDITTEXT, ""},
 /* 12 */ {0, {280,244,296,406}, MESSAGE, "internal units per pixel"},
 /* 13 */ {0, {196,20,212,172}, CHECK, "Synchronize to file:"},
 /* 14 */ {0, {196,172,228,418}, MESSAGE, ""},
 /* 15 */ {0, {176,20,192,108}, MESSAGE, "EPS Scale:"},
 /* 16 */ {0, {176,124,192,164}, EDITTEXT, "2"},
 /* 17 */ {0, {156,8,172,80}, MESSAGE, "For facet:"},
 /* 18 */ {0, {156,80,172,388}, MESSAGE, ""},
 /* 19 */ {0, {236,8,237,418}, DIVIDELINE, ""},
 /* 20 */ {0, {72,8,88,96}, MESSAGE, "PostScript:"},
 /* 21 */ {0, {72,240,88,360}, CHECK, "Color"},
 /* 22 */ {0, {68,8,69,418}, DIVIDELINE, ""},
 /* 23 */ {0, {240,8,256,98}, MESSAGE, "HPGL Level:"},
 /* 24 */ {0, {96,20,112,78}, RADIO, "Printer"},
 /* 25 */ {0, {116,20,132,78}, RADIO, "Plotter"},
 /* 26 */ {0, {96,88,112,168}, MESSAGE, "Width (in):"},
 /* 27 */ {0, {96,172,112,212}, EDITTEXT, "2"},
 /* 28 */ {0, {116,88,132,168}, MESSAGE, "Width (in):"},
 /* 29 */ {0, {116,172,132,212}, EDITTEXT, "2"},
 /* 30 */ {0, {96,224,112,304}, MESSAGE, "Height (in):"},
 /* 31 */ {0, {96,308,112,348}, EDITTEXT, "2"},
 /* 32 */ {0, {136,20,152,240}, CHECK, "Rotate plot 90 degrees"},
 /* 33 */ {0, {136,244,152,324}, MESSAGE, "Margin (in):"},
 /* 34 */ {0, {136,328,152,368}, EDITTEXT, "2"}
};
DIALOG us_printingoptdialog = {{50,75,361,502}, "Printing Options", 34, us_printingoptdialogitems};

/*
 * special case for the "print options" dialog
 * plot entire facet       = 4 (radio)
 * focus on highlighted    = 3 (radio)
 * encapsulated postscript = 5 (check)
 * date in corner          = 6 (check)
 * HPGL                    = 7 (radio)
 * HPGL/2                  = 8 (radio)
 * HPGL/2 plot fills page  = 9 (radio)
 * HPGL/2 plot fixed at    = 10 (radio)
 * HPGL/2 plot scale       = 11 (edit text)
 * synchronize to file     = 13 (check)
 * EPS scale label         = 15 (message)
 * EPS scale               = 16 (edit text)
 * Facet                   = 18 (message)
 * Divider                 = 19 (message)
 * Color postscript        = 21 (check)
 * Divider                 = 22 (message)
 * Printer                 = 24 (radio)
 * Plotter                 = 25 (radio)
 * Printer width           = 27 (edit text)
 * Plotter width           = 29 (edit text)
 * Printer height          = 31 (edit text)
 * Rotate output           = 32 (check)
 * Print margin            = 34 (edit text)
 */
INTSML us_plotoptionsdlog(char *paramstart[])
{
	INTSML oldplease;
	REGISTER INTBIG itemHit, scale, i, state, wid, hei;
	REGISTER VARIABLE *var;
	REGISTER NODEPROTO *np;
	REGISTER char *pt;
	char buf[50], *subparams[3];
	extern COMCOMP us_colorwritep;

	/* display the print options dialog box */
	if (DiaInitDialog(&us_printingoptdialog) != 0) return(0);
	var = getvalkey((INTBIG)io_aid, VAID, VINTEGER, io_state);
	if (var != NOVARIABLE) state = var->addr; else state = 0;
	if ((state&PLOTDATES) != 0) DiaSetControl(6, 1);
	if ((state&PLOTFOCUS) != 0) DiaSetControl(3, 1); else
		DiaSetControl(4, 1);
	if ((state&EPSPSCRIPT) != 0) DiaSetControl(5, 1);
	if ((state&PSCOLOR) != 0) DiaSetControl(21, 1);
	if ((state&PSROTATE) != 0) DiaSetControl(32, 1);
	var = getval((INTBIG)io_aid, VAID, VFRACT, "IO_postscript_width");
	if (var == NOVARIABLE) wid = muldiv(DEFAULTPSWIDTH, WHOLE, 75); else
		wid = var->addr;
	DiaSetText(27, frtoa(wid));
	DiaSetText(29, frtoa(wid));
	var = getval((INTBIG)io_aid, VAID, VFRACT, "IO_postscript_height");
	if (var == NOVARIABLE) hei = muldiv(DEFAULTPSHEIGHT, WHOLE, 75); else
		hei = var->addr;
	DiaSetText(31, frtoa(hei));
	var = getval((INTBIG)io_aid, VAID, VFRACT, "IO_postscript_margin");
	if (var == NOVARIABLE) i = muldiv(DEFAULTPSMARGIN, WHOLE, 75); else
		i = var->addr;
	DiaSetText(34, frtoa(i));
	if ((state&PSPLOTTER) != 0)
	{
		DiaSetControl(25, 1);
		DiaDimItem(26);
		DiaDimItem(27);
		DiaDimItem(30);
		DiaDimItem(31);
	} else
	{
		DiaSetControl(24, 1);
		DiaDimItem(28);
		DiaDimItem(29);
	}
	np = el_curlib->curnodeproto;
	DiaDimItem(15);
	DiaDimItem(16);
	if (np == NONODEPROTO)
	{
		DiaDimItem(13);
		DiaDimItem(17);
	} else
	{
		DiaSetText(18, describenodeproto(np));
		DiaUnDimItem(17);
		if ((state&EPSPSCRIPT) != 0)
		{
			DiaUnDimItem(15);
			DiaUnDimItem(16);
			var = getvalkey((INTBIG)np, VNODEPROTO, VFRACT, io_postscriptepsscale);
			if (var != NOVARIABLE)
			{
				DiaSetText(16, frtoa(var->addr));
			}
		}
		DiaUnDimItem(13);
		var = getvalkey((INTBIG)np, VNODEPROTO, VSTRING, io_postscriptfilename);
		if (var != NOVARIABLE)
		{
			DiaSetControl(13, 1);
			DiaSetText(14, (char *)var->addr);
		}
	}
	if ((state&HPGL2) != 0)
	{
		DiaSetControl(8, 1);
		DiaUnDimItem(9);   DiaUnDimItem(10);   DiaUnDimItem(12);
		DiaEditControl(11);
		var = getval((INTBIG)io_aid, VAID, VINTEGER, "IO_hpgl2_scale");
		if (var == NOVARIABLE)
		{
			DiaSetControl(9, 1);
			DiaSetText(11, "20");
		} else
		{
			DiaSetControl(10, 1);
			(void)sprintf(buf, "%ld", var->addr);
			DiaSetText(11, buf);
		}
	} else
	{
		DiaSetControl(7, 1);
		DiaSetControl(9, 1);
		DiaSetText(11, "20");
		DiaDimItem(9);   DiaDimItem(10);   DiaDimItem(12);
		DiaNoEditControl(11);
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK) break;
		if (itemHit == 5 || itemHit == 6 || itemHit == 13 || itemHit == 21 || itemHit == 32)
		{
			i = 1 - DiaGetControl(itemHit);
			DiaSetControl(itemHit, i);
			if (itemHit == 5)
			{
				if (i == 0)
				{
					DiaDimItem(15);
					DiaDimItem(16);
				} else
				{
					DiaUnDimItem(15);
					DiaUnDimItem(16);
				}
			}
			if (itemHit == 13)
			{
				if (i != 0)
				{
					/* select a file name to synchronize with this facet */
					oldplease = el_pleasestop;
					i = ttygetparam("ps/PostScript file", &us_colorwritep, 1, subparams);
					el_pleasestop = oldplease;
					if (i != 0 && subparams[0][0] != 0)
					{
						DiaUnDimItem(14);
						DiaSetText(14, subparams[0]);
					} else
					{
						DiaSetControl(13, 0);
					}
				} else
				{
					DiaDimItem(14);
				}
			}
			continue;
		}
		if (itemHit == 3 || itemHit == 4)
		{
			DiaSetControl(3, 0);
			DiaSetControl(4, 0);
			DiaSetControl(itemHit, 1);
			continue;
		}
		if (itemHit == 7)
		{
			DiaSetControl(7, 1);   DiaSetControl(8, 0);
			DiaDimItem(9);   DiaDimItem(10);   DiaDimItem(12);
			DiaNoEditControl(11);
			continue;
		}
		if (itemHit == 8)
		{
			DiaSetControl(7, 0);   DiaSetControl(8, 1);
			DiaUnDimItem(9);   DiaUnDimItem(10);   DiaUnDimItem(12);
			DiaEditControl(11);
			continue;
		}
		if (itemHit == 9 || itemHit == 10)
		{
			DiaSetControl(9, 0);
			DiaSetControl(10, 0);
			DiaSetControl(itemHit, 1);
			continue;
		}
		if (itemHit == 24 || itemHit == 25)
		{
			DiaSetControl(24, 0);
			DiaSetControl(25, 0);
			DiaSetControl(itemHit, 1);
			if (itemHit == 25)
			{
				DiaUnDimItem(28);
				DiaUnDimItem(29);
				DiaDimItem(26);
				DiaDimItem(27);
				DiaDimItem(30);
				DiaDimItem(31);
			} else
			{
				DiaUnDimItem(26);
				DiaUnDimItem(27);
				DiaUnDimItem(30);
				DiaUnDimItem(31);
				DiaDimItem(28);
				DiaDimItem(29);
			}
			continue;
		}
	}

	paramstart[0] = "";
	if (itemHit != CANCEL)
	{
		if (DiaGetControl(3) != 0) state |= PLOTFOCUS; else state &= ~PLOTFOCUS;
		if (DiaGetControl(5) != 0) state |= EPSPSCRIPT; else state &= ~EPSPSCRIPT;
		if (DiaGetControl(21) != 0) state |= PSCOLOR; else state &= ~PSCOLOR;
		if (DiaGetControl(32) != 0) state |= PSROTATE; else state &= ~PSROTATE;
		if (DiaGetControl(6) != 0) state |= PLOTDATES; else state &= ~PLOTDATES;
		if (DiaGetControl(8) != 0) state |= HPGL2; else state &= ~HPGL2;
		if (DiaGetControl(24) != 0)
		{
			/* printed */
			state &= ~PSPLOTTER;
			pt = DiaGetText(27);
			(void)setval((INTBIG)io_aid, VAID, "IO_postscript_width", atofr(pt), VFRACT);
			pt = DiaGetText(31);
			(void)setval((INTBIG)io_aid, VAID, "IO_postscript_height", atofr(pt), VFRACT);
		} else
		{
			state |= PSPLOTTER;
			pt = DiaGetText(29);
			(void)setval((INTBIG)io_aid, VAID, "IO_postscript_width", atofr(pt), VFRACT);
		}
		(void)setvalkey((INTBIG)io_aid, VAID, io_state, state, VINTEGER);
		pt = DiaGetText(34);
		(void)setval((INTBIG)io_aid, VAID, "IO_postscript_margin", atofr(pt), VFRACT);
		if (np != NONODEPROTO)
		{
			if (DiaGetControl(13) != 0)
			{
				/* add a synchronization file */
				pt = DiaGetText(14);
				(void)setvalkey((INTBIG)np, VNODEPROTO, io_postscriptfilename,
					(INTBIG)pt, VSTRING);
			} else
			{
				/* remove a synchronization file */
				if (getvalkey((INTBIG)np, VNODEPROTO, VSTRING, io_postscriptfilename) != NOVARIABLE)
					delvalkey((INTBIG)np, VNODEPROTO, io_postscriptfilename);
			}
			if (DiaGetControl(5) != 0)
			{
				pt = DiaGetText(16);
				(void)setvalkey((INTBIG)np, VNODEPROTO, io_postscriptepsscale,
					atofr(pt), VFRACT);
			}
		}
		if ((state&HPGL2) != 0)
		{
			if (DiaGetControl(9) != 0)
			{
				if (getval((INTBIG)io_aid, VAID, VINTEGER, "IO_hpgl2_scale") != NOVARIABLE)
					(void)delval((INTBIG)io_aid, VAID, "IO_hpgl2_scale");
			} else
			{
				scale = myatoi(DiaGetText(11));
				if (scale <= 0) scale = 1;
				(void)setval((INTBIG)io_aid, VAID, "IO_hpgl2_scale", scale, VINTEGER);
			}
		}
	}
	DiaDoneDialog();
	return(1);
}

/****************************** QUICK KEY OPTIONS DIALOG ******************************/

/* Quick Keys */
DIALOGITEM us_quickkeydialogitems[] =
{
 /*  1 */ {0, {520,320,544,384}, BUTTON, "OK"},
 /*  2 */ {0, {520,12,544,76}, BUTTON, "Cancel"},
 /*  3 */ {0, {24,4,168,168}, SCROLL, ""},
 /*  4 */ {0, {192,20,336,184}, SCROLL, ""},
 /*  5 */ {0, {4,4,20,68}, MESSAGE, "Menu:"},
 /*  6 */ {0, {172,20,188,136}, MESSAGE, "SubMenu/Item:"},
 /*  7 */ {0, {360,36,504,200}, SCROLL, ""},
 /*  8 */ {0, {340,36,356,140}, MESSAGE, "SubItem:"},
 /*  9 */ {0, {24,228,504,392}, SCROLL, ""},
 /* 10 */ {0, {4,228,20,328}, MESSAGE, "Quick Key:"},
 /* 11 */ {0, {256,192,280,220}, BUTTON, ">>"},
 /* 12 */ {0, {520,236,544,296}, BUTTON, "Remove"},
 /* 13 */ {0, {520,96,544,216}, BUTTON, "Factory Settings"}
};
DIALOG us_quickkeydialog = {{75,75,634,478}, "Quick Key Options", 13, us_quickkeydialogitems};

void us_setlastquickkeys(void);
void us_setmiddlequickkeys(void);
void us_loadquickkeys(POPUPMENU *pm);
char *us_makequickkey(INTBIG i);

/*
 * special case for the "Quick Keys" dialog
 * menu list          = 3 (scroll)
 * submenu/item list  = 4 (scroll)
 * subitem list       = 7 (scroll)
 * quick key list     = 9 (scroll)
 * ">>"               = 11 (button)
 * "<<"               = 12 (button)
 * "Factory settings" = 13 (button)
 */
#define MAXQUICKKEYS 50
char           us_quickkeyskey[MAXQUICKKEYS];
POPUPMENU     *us_quickkeysmenu[MAXQUICKKEYS];
INTBIG         us_quickkeysindex[MAXQUICKKEYS];
INTBIG         us_quickkeyscount;

INTSML us_quickkeydlog(char *paramstart[])
{
	INTBIG itemHit, i, j, k, which, whichmiddle, whichbottom;
	char **quickkeylist;
	INTBIG quickkeycount;
	REGISTER char *pt, *menuname, *menucommand;
	REGISTER POPUPMENU *pm;
	REGISTER POPUPMENUITEM *mi;
	REGISTER USERCOM *uc;

	/* display the window view dialog box */
	if (DiaInitDialog(&us_quickkeydialog) != 0) return(0);
	DiaInitTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);
	DiaInitTextDialog(4, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);
	DiaInitTextDialog(7, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);
	DiaInitTextDialog(9, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);

	/* make a list of quick keys */
	us_buildquickkeylist();
	for(i=0; i<us_quickkeyscount; i++)
		DiaStuffLine(9, us_makequickkey(i));
	DiaSelectLine(9, 0);

	/* load the list of menus */
	for(i=0; i<us_pulldownmenucount; i++)
		DiaStuffLine(3, us_removeampersand(us_pulldowns[i]->header));
	DiaSelectLine(3, 0);
	us_setmiddlequickkeys();

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == 3)
		{
			/* click in the top list (the pulldown menus) */
			us_setmiddlequickkeys();
			continue;
		}
		if (itemHit == 4)
		{
			/* click in the middle list (the submenus/items) */
			us_setlastquickkeys();
			continue;
		}
		if (itemHit == 7)
		{
			/* click in the lower list (the subitems) */
			which = DiaGetCurLine(3);
			whichmiddle = DiaGetCurLine(4);
			whichbottom = DiaGetCurLine(7);
			pm = us_pulldowns[which];
			mi = &pm->list[whichmiddle];
			uc = mi->response;
			pm = uc->menu;
			if (pm == NOPOPUPMENU) continue;
			for(j=0; j<us_quickkeyscount; j++)
				if (pm == us_quickkeysmenu[j] && whichbottom == us_quickkeysindex[j]) break;
			if (j < us_quickkeyscount) DiaSelectLine(9, j);
			continue;
		}
		if (itemHit == 11)
		{
			/* click in the ">>" button (add command to quick keys) */
			which = DiaGetCurLine(3);
			whichmiddle = DiaGetCurLine(4);
			whichbottom = DiaGetCurLine(7);
			pm = us_pulldowns[which];
			mi = &pm->list[whichmiddle];
			uc = mi->response;
			if (uc->menu != NOPOPUPMENU)
			{
				pm = uc->menu;
				whichmiddle = whichbottom;
			}
			which = DiaGetCurLine(9);
			us_quickkeysmenu[which] = pm;
			us_quickkeysindex[which] = whichmiddle;
			DiaSetScrollLine(9, which, us_makequickkey(which));
			continue;
		}
		if (itemHit == 12)
		{
			/* click in the "<<" button (remove command from quick key) */
			which = DiaGetCurLine(9);
			us_quickkeysmenu[which] = NOPOPUPMENU;
			DiaSetScrollLine(9, which, us_makequickkey(which));
			continue;
		}
		if (itemHit == 13)
		{
			/* click in the "Factory Settings" button */
			for(i=0; i<us_quickkeyscount; i++)
				us_quickkeysmenu[i] = NOPOPUPMENU;
			for(i=0; i<us_quickkeycount; i++)
			{
				pt = us_quickkeylist[i];
				for(j=0; j<us_quickkeyscount; j++)
					if (us_quickkeyskey[j] == *pt) break;
				if (j >= us_quickkeyscount) continue;
				menuname = &pt[2];
				for(pt = menuname; *pt != 0 && *pt != '/'; pt++) ;
				if (*pt == 0) continue;
				*pt = 0;
				pm = us_getpopupmenu(menuname);
				*pt = '/';
				menucommand = pt + 1;
				for(k=0; k<pm->total; k++)
				{
					mi = &pm->list[k];
					if (namesame(us_removeampersand(mi->attribute), menucommand) == 0) break;
				}
				if (k >= pm->total) continue;
				us_quickkeysmenu[j] = pm;
				us_quickkeysindex[j] = k;
			}
			DiaLoadTextDialog(9, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
			for(i=0; i<us_quickkeyscount; i++)
				DiaStuffLine(9, us_makequickkey(i));
			DiaSelectLine(9, 0);
			continue;
		}
	}

	DiaDoneDialog();
	if (itemHit == OK)
	{
		us_getquickkeylist(&quickkeycount, &quickkeylist);
		(void)setvalkey((INTBIG)us_aid, VAID, us_quickkeys, (INTBIG)quickkeylist,
			VSTRING|VISARRAY|(quickkeycount<<VLENGTHSH));
		for(i=0; i<quickkeycount; i++)
			efree((char *)quickkeylist[i]);
		efree((char *)quickkeylist);
	}
	return(0);
}

/*
 * Helper routine for "us_quickkeydlog" to construct the string describing key "i".
 */
char *us_makequickkey(INTBIG i)
{
	REGISTER POPUPMENUITEM *mi;

	(void)initinfstr();
	(void)addtoinfstr(us_quickkeyskey[i]);
	if (us_quickkeysmenu[i] != NOPOPUPMENU)
	{
		mi = &us_quickkeysmenu[i]->list[us_quickkeysindex[i]];
		(void)addstringtoinfstr("   ");
		(void)addstringtoinfstr(us_removeampersand(mi->attribute));
	}
	return(returninfstr());
}

/*
 * Routine to scan all pulldown menus and build an internal list of quick keys.
 */
void us_buildquickkeylist(void)
{
	REGISTER INTBIG i;

	us_quickkeyscount = 0;
	for(i=0; i<26; i++)
		us_quickkeyskey[us_quickkeyscount++] = (char)('A' + i);
	for(i=0; i<10; i++)
		us_quickkeyskey[us_quickkeyscount++] = (char)('0' + i);
	for(i=0; i<us_quickkeyscount; i++) us_quickkeysmenu[i] = NOPOPUPMENU;
	for(i=0; i<us_pulldownmenucount; i++)
		us_loadquickkeys(us_pulldowns[i]);
}

/*
 * Routine to convert the internal list of quick keys to an array of bindings
 * in "quickkeylist" (that is "quickkeycount" long).
 */
void us_getquickkeylist(INTBIG *quickkeycount, char ***quickkeylist)
{
	REGISTER INTBIG count, i;
	REGISTER char **keylist;
	REGISTER POPUPMENUITEM *mi;

	count = 0;
	for(i=0; i<us_quickkeyscount; i++)
		if (us_quickkeysmenu[i] != NOPOPUPMENU) count++;
	keylist = (char **)emalloc(count * (sizeof (char *)), us_aid->cluster);
	if (keylist == 0)
	{
		*quickkeycount = 0;
		return;
	}
	count = 0;
	for(i=0; i<us_quickkeyscount; i++)
	{
		if (us_quickkeysmenu[i] == NOPOPUPMENU) continue;
		(void)initinfstr();
		(void)addtoinfstr(us_quickkeyskey[i]);
		(void)addtoinfstr('/');
		(void)addstringtoinfstr(us_quickkeysmenu[i]->name);
		(void)addtoinfstr('/');
		mi = &us_quickkeysmenu[i]->list[us_quickkeysindex[i]];
		(void)addstringtoinfstr(us_removeampersand(mi->attribute));
		(void)allocstring(&keylist[count], returninfstr(), us_aid->cluster);
		count++;
	}
	*quickkeycount = count;
	*quickkeylist = keylist;
}

/*
 * Helper routine for "us_quickkeydlog" to recursively examine menu "pm" and
 * load the quick keys tables.
 */
void us_loadquickkeys(POPUPMENU *pm)
{
	REGISTER INTBIG j, i;
	REGISTER POPUPMENUITEM *mi;
	REGISTER USERCOM *uc;
	REGISTER char *pt;

	for(i=0; i<pm->total; i++)
	{
		mi = &pm->list[i];
		uc = mi->response;
		if (uc->menu != NOPOPUPMENU)
		{
			us_loadquickkeys(uc->menu);
			continue;
		}
		if (uc->active < 0 && *mi->attribute == 0) continue;
		
		for(pt = mi->attribute; *pt != 0; pt++)
			if (*pt == '/') break;
		if (*pt == 0) continue;
		for(j=0; j<us_quickkeyscount; j++)
		{
			if (us_quickkeyskey[j] != pt[1]) continue;
			us_quickkeysmenu[j] = pm;
			us_quickkeysindex[j] = i;
		}
	}
}

/*
 * Helper routine for "us_quickkeydlog" to load the middle table (the
 * submenu/items) when the selected top table has changed.
 */
void us_setmiddlequickkeys(void)
{
	REGISTER INTBIG which, i;
	REGISTER POPUPMENU *pm;
	REGISTER POPUPMENUITEM *mi;
	REGISTER USERCOM *uc;

	DiaLoadTextDialog(4, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
	which = DiaGetCurLine(3);
	pm = us_pulldowns[which];
	for(i=0; i<pm->total; i++)
	{
		mi = &pm->list[i];
		uc = mi->response;
		if (uc->active < 0 && *mi->attribute == 0)
		{
			DiaStuffLine(4, "---");
			continue;
		}
		DiaStuffLine(4, us_removeampersand(mi->attribute));
	}
	DiaSelectLine(4, 0);
	us_setlastquickkeys();
}

/*
 * Helper routine for "us_quickkeydlog" to load the bottom table (the
 * subitems) when the selected middle table has changed.
 */
void us_setlastquickkeys(void)
{
	REGISTER INTBIG which, whichmiddle, i, j;
	REGISTER POPUPMENU *pm;
	REGISTER POPUPMENUITEM *mi;
	REGISTER USERCOM *uc;

	DiaLoadTextDialog(7, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
	which = DiaGetCurLine(3);
	whichmiddle = DiaGetCurLine(4);
	pm = us_pulldowns[which];
	mi = &pm->list[whichmiddle];
	uc = mi->response;
	if (uc->menu != NOPOPUPMENU)
	{
		pm = uc->menu;
		for(i=0; i<pm->total; i++)
		{
			mi = &pm->list[i];
			uc = mi->response;
			if (uc->active < 0 && *mi->attribute == 0)
			{
				DiaStuffLine(7, "---");
				continue;
			}
			DiaStuffLine(7, us_removeampersand(mi->attribute));
		}
		DiaSelectLine(7, 0);
		mi = &pm->list[0];
		whichmiddle = 0;
	}

	for(j=0; j<us_quickkeyscount; j++)
		if (pm == us_quickkeysmenu[j] && whichmiddle == us_quickkeysindex[j]) break;
	if (j < us_quickkeyscount) DiaSelectLine(9, j);
}

/****************************** QUIT DIALOG ******************************/

/* Quit */
DIALOGITEM us_quitdialogitems[] =
{
 /*  1 */ {0, {101,16,125,80}, BUTTON, "Yes"},
 /*  2 */ {0, {136,136,160,192}, BUTTON, "Cancel"},
 /*  3 */ {0, {136,16,160,80}, BUTTON, "No"},
 /*  4 */ {0, {8,16,90,208}, MESSAGE, ""}
};
DIALOG us_quitdialog = {{50,75,224,315}, 0, 4, us_quitdialogitems};

/*
 * special case for the "quit" command
 * Message = 4 (message)
 * Yes     = 1 (button)
 * Cancel  = 2 (button)
 * No      = 3 (button)
 */
INTSML us_quitdlog(char *prompt, char *paramstart[])
{
	INTBIG itemHit, i, len;
	INTSML oldplease;

	/* display the quit dialog box */
	if (DiaInitDialog(&us_quitdialog) != 0) return(0);

	/* load the message */
	len = strlen(prompt);
	for(i=0; i<len; i++) if (strncmp(&prompt[i], " has changed.  ", 15) == 0) break;
	if (i >= len) DiaSetText(4, prompt); else
	{
		(void)initinfstr();
		prompt[i+15] = 0;
		(void)addstringtoinfstr(prompt);
		(void)addstringtoinfstr("Save?");
		DiaSetText(4, returninfstr());
	}

	/* loop until done */
	oldplease = el_pleasestop;
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == 1 || itemHit == 2 || itemHit == 3) break;
	}
	el_pleasestop = oldplease;

	switch (itemHit)
	{
		case 2:  paramstart[0] = "no";     break;
		case 3:  paramstart[0] = "yes";    break;
		default: paramstart[0] = "save";   break;
	}
	DiaDoneDialog();
	return(1);
}

/****************************** RENAME DIALOG ******************************/

/* Rename */
DIALOGITEM us_rendialogitems[] =
{
 /*  1 */ {0, {216,208,240,288}, BUTTON, "OK"},
 /*  2 */ {0, {216,44,240,124}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,132,168,336}, SCROLL, ""},
 /*  4 */ {0, {184,8,200,88}, MESSAGE, "New name:"},
 /*  5 */ {0, {184,96,200,336}, EDITTEXT, ""},
 /*  6 */ {0, {80,8,96,124}, RADIO, "Library"},
 /*  7 */ {0, {152,8,168,124}, RADIO, "Technology"},
 /*  8 */ {0, {104,8,120,124}, RADIO, "Primitive Node"},
 /*  9 */ {0, {128,8,144,124}, RADIO, "Arc"},
 /* 10 */ {0, {32,8,48,124}, RADIO, "Port"},
 /* 11 */ {0, {8,8,24,124}, RADIO, "Cell"},
 /* 12 */ {0, {56,8,72,124}, RADIO, "Network"}
};
DIALOG us_rendialog = {{75,75,324,420}, "Rename Object", 12, us_rendialogitems};

static LIBRARY *us_renamelib;
static TECHNOLOGY *us_renametech;
static NODEPROTO *us_renameprim;
static ARCPROTO *us_renamearc;
static PORTPROTO *us_renamediaport;
static CELL *us_renamecell;
static NETWORK *us_renamenet;

INTSML us_initrenamelist(char **c);
char *us_renamelistitem(void);

/*
 * special case for the "rename" dialog
 * list of old names = 3 (scroll)
 * new name          = 5 (edit text)
 * library           = 6 (radio)
 * technology        = 7 (radio)
 * prim node         = 8 (radio)
 * arc               = 9 (radio)
 * port              = 10 (radio)
 * cell              = 11 (radio)
 * network           = 12 (radio)
 */
INTSML us_renamedlog(char *paramstart[])
{
	REGISTER INTBIG itemHit, i;
	REGISTER NODEPROTO *np;
	char *par[3];

	/* display the node size dialog box */
	if (DiaInitDialog(&us_rendialog) != 0) return(0);
	np = getcurfacet();
	if (np == NONODEPROTO)
	{
		DiaDimItem(10);
		DiaDimItem(12);
	}
	if (el_curlib->firstcell != NOCELL) DiaSetControl(11, 1); else
		DiaSetControl(6, 1);
	DiaInitTextDialog(3, us_initrenamelist, us_renamelistitem, DiaNullDlogDone, 0,
		SCSELMOUSE);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK)
		{
			i = DiaGetCurLine(3);
			if (i < 0) continue;
			if (DiaValidEntry(5) == 0) continue;
			break;
		}
		if (itemHit >= 6 && itemHit <= 12)
		{
			for(i=6; i<=12; i++) DiaSetControl(i, 0);
			DiaSetControl(itemHit, 1);
			DiaLoadTextDialog(3, us_initrenamelist, us_renamelistitem, DiaNullDlogDone, 0);
			continue;
		}
	}

	if (itemHit != CANCEL)
	{
		if (DiaGetControl(6) != 0) par[2] = "l"; else
		if (DiaGetControl(7) != 0) par[2] = "t"; else
		if (DiaGetControl(8) != 0) par[2] = "p"; else
		if (DiaGetControl(9) != 0) par[2] = "a"; else
		if (DiaGetControl(10) != 0) par[2] = "r"; else
		if (DiaGetControl(11) != 0) par[2] = "c"; else
		if (DiaGetControl(12) != 0) par[2] = "n";
		i = DiaGetCurLine(3);
		par[0] = DiaGetScrollLine(3, i);
		par[1] = DiaGetText(5);
		us_rename(2, par);
	}
	DiaDoneDialog();
	return(0);
}

INTSML us_initrenamelist(char **c)
{
	REGISTER NODEPROTO *np;

	us_renamelib = el_curlib;
	us_renametech = el_technologies;
	us_renameprim = el_curtech->firstnodeproto;
	us_renamearc = el_curtech->firstarcproto;
	us_renamecell = el_curlib->firstcell;
	np = getcurfacet();
	if (np != NONODEPROTO)
	{
		us_renamediaport = np->firstportproto;
		us_renamenet = np->firstnetwork;
	}
	return(0);
}

char *us_renamelistitem(void)
{
	REGISTER LIBRARY *lib;
	REGISTER TECHNOLOGY *tech;
	REGISTER ARCPROTO *ap;
	REGISTER NODEPROTO *np;
	REGISTER CELL *cell;
	REGISTER NETWORK *net;
	REGISTER PORTPROTO *pp;

	if (DiaGetControl(6) != 0)
	{
		for(;;)
		{
			if (us_renamelib == NOLIBRARY) return(0);
			lib = us_renamelib;
			us_renamelib = us_renamelib->nextlibrary;
			if ((lib->userbits&HIDDENLIBRARY) != 0) continue;
			return(lib->libname);
		}
	}
	if (DiaGetControl(7) != 0)
	{
		if (us_renametech == NOTECHNOLOGY) return(0);
		tech = us_renametech;
		us_renametech = us_renametech->nexttechnology;
		return(tech->techname);
	}
	if (DiaGetControl(8) != 0)
	{
		if (us_renameprim == NONODEPROTO) return(0);
		np = us_renameprim;
		us_renameprim = us_renameprim->nextnodeproto;
		return(np->primname);
	}
	if (DiaGetControl(9) != 0)
	{
		if (us_renamearc == NOARCPROTO) return(0);
		ap = us_renamearc;
		us_renamearc = us_renamearc->nextarcproto;
		return(ap->protoname);
	}
	if (DiaGetControl(10) != 0)
	{
		if (us_renamediaport == NOPORTPROTO) return(0);
		pp = us_renamediaport;
		us_renamediaport = us_renamediaport->nextportproto;
		return(pp->protoname);
	}
	if (DiaGetControl(11) != 0)
	{
		if (us_renamecell == NOCELL) return(0);
		cell = us_renamecell;
		us_renamecell = us_renamecell->nextcell;
		return(cell->cellname);
	}
	if (DiaGetControl(12) != 0)
	{
		for(;;)
		{
			if (us_renamenet == NONETWORK) return(0);
			net = us_renamenet;
			us_renamenet = us_renamenet->nextnetwork;
			if (net->namecount <= 0) continue;
			return(net->netname);
		}
	}
	return(0);
}

/****************************** SELECTION: OPTIONS DIALOG ******************************/

/* Selection Options */
DIALOGITEM us_seloptdialogitems[] =
{
 /*  1 */ {0, {108,172,132,252}, BUTTON, "OK"},
 /*  2 */ {0, {108,4,132,84}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,4,24,252}, CHECK, "Easy selection of facet instances"},
 /*  4 */ {0, {32,4,48,252}, CHECK, "Easy selection of annotation text"},
 /*  5 */ {0, {56,4,72,252}, CHECK, "Constrained dragging by default"},
 /*  6 */ {0, {80,4,96,252}, CHECK, "Center-based primitives"}
};
DIALOG us_seloptdialog = {{75,75,216,339}, "Selection Options", 6, us_seloptdialogitems};

/*
 * special case for the "selection options" dialog
 * easy selection of instances     = 3 (check)
 * easy selection of annotation    = 4 (check)
 * Constrained dragging by default = 5 (check)
 * Center-based primitives         = 6 (check)
 */
INTSML us_selectoptdlog(char *paramstart[])
{
	REGISTER INTBIG itemHit;
	REGISTER INTBIG options;

	/* display the dialog box */
	if (DiaInitDialog(&us_seloptdialog) != 0) return(0);
	if ((us_useroptions&NOINSTANCESELECT) == 0) DiaSetControl(3, 1);
	if ((us_useroptions&NOTEXTSELECT) == 0) DiaSetControl(4, 1);
	if ((us_useroptions&FLIPINTERACTIVECON) != 0) DiaSetControl(5, 1);
	if ((us_useroptions&CENTEREDPRIMITIVES) != 0) DiaSetControl(6, 1);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit >= 3 && itemHit <= 6)
		{
			DiaSetControl(itemHit, 1 - DiaGetControl(itemHit));
			continue;
		}
	}

	if (itemHit != CANCEL)
	{
		options = us_useroptions;
		if (DiaGetControl(3) != 0) options &= ~NOINSTANCESELECT; else
			options |= NOINSTANCESELECT;
		if (DiaGetControl(4) != 0) options &= ~NOTEXTSELECT; else
			options |= NOTEXTSELECT;
		if (DiaGetControl(5) == 0) options &= ~FLIPINTERACTIVECON; else
			options |= FLIPINTERACTIVECON;
		if (DiaGetControl(6) == 0) options &= ~CENTEREDPRIMITIVES; else
			options |= CENTEREDPRIMITIVES;
		(void)setvalkey((INTBIG)us_aid, VAID, us_optionflags, options, VINTEGER);
	}
	DiaDoneDialog();
	return(0);
}

/****************************** SELECTION: PORT/NET DIALOG ******************************/

/* Selection: Port/Net */
DIALOGITEM us_selnamedialogitems[] =
{
 /*  1 */ {0, {192,112,216,192}, BUTTON, "Select"},
 /*  2 */ {0, {192,8,216,88}, BUTTON, "Done"},
 /*  3 */ {0, {8,8,180,192}, SCROLL, ""}
};
DIALOG us_selnamedialog = {{75,75,300,276}, "Select Port", 3, us_selnamedialogitems};

INTSML     us_selportnet;
PORTPROTO *us_selport;
NETWORK   *us_selnet;

INTSML us_topofportnet(char **c);
char *us_nextportnet(void);

INTSML us_topofportnet(char **c)
{
	REGISTER NODEPROTO *np;

	np = us_needfacet();
	if (us_selportnet != 0) us_selport = np->firstportproto; else
		us_selnet = np->firstnetwork;
	return(1);
}

char *us_nextportnet(void)
{
	REGISTER char *retname;
	REGISTER NETWORK *retnet;

	if (us_selportnet != 0)
	{
		if (us_selport == NOPORTPROTO) return(0);
		retname = us_selport->protoname;
		us_selport = us_selport->nextportproto;
		return(retname);
	}
	for(;;)
	{
		if (us_selnet == NONETWORK) return(0);
		retnet = us_selnet;
		us_selnet = us_selnet->nextnetwork;
		if (retnet->namecount != 0) break;
	}
	return(retnet->netname);
}

/*
 * special case for the "selection port/net" dialog
 * list of ports/nets     = 3 (scroll)
 */
INTSML us_selectportnetdlog(char *paramstart[], INTSML selport)
{
	REGISTER INTBIG itemHit, which;
	REGISTER NODEPROTO *np;
	REGISTER char *pt;
	char *par[3];
	extern AIDENTRY *net_aid;

	np = us_needfacet();
	if (np == NONODEPROTO) return(0);

	/* display the dialog box */
	us_selportnet = selport;
	if (selport != 0) us_selnamedialog.movable = "Select Port"; else
		us_selnamedialog.movable = "Select Network";
	if (DiaInitDialog(&us_selnamedialog) != 0) return(0);
	DiaInitTextDialog(3, us_topofportnet, us_nextportnet, DiaNullDlogDone, 0,
		SCSELMOUSE|SCDOUBLEQUIT);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK)
		{
			which = DiaGetCurLine(3);
			if (which < 0) continue;
			pt = DiaGetScrollLine(3, which);
			if (selport != 0)
			{
				/* find port "pt" */
				par[0] = "export";
				par[1] = pt;
				us_find(2, par);
			} else
			{
				par[0] = "highlight";
				par[1] = pt;
				(void)tellaid(net_aid, 2, par);
			}
			us_showallhighlight();
			us_endchanges(NOWINDOWPART);
		}
	}
	DiaDoneDialog();
	return(0);
}

/****************************** SPREAD DIALOG ******************************/

/* Spread */
DIALOGITEM us_spreaddialogitems[] =
{
 /*  1 */ {0, {96,128,120,200}, BUTTON, "OK"},
 /*  2 */ {0, {96,16,120,88}, BUTTON, "Cancel"},
 /*  3 */ {0, {55,15,71,205}, EDITTEXT, ""},
 /*  4 */ {0, {20,230,36,380}, RADIO, "Spread up"},
 /*  5 */ {0, {45,230,61,380}, RADIO, "Spread down"},
 /*  6 */ {0, {70,230,86,380}, RADIO, "Spread left"},
 /*  7 */ {0, {95,230,111,380}, RADIO, "Spread right"},
 /*  8 */ {0, {25,15,41,180}, MESSAGE, "Distance to spread"}
};
DIALOG us_spreaddialog = {{50,75,188,464}, "Spread About Highlighted", 8, us_spreaddialogitems};

/*
 * special case for the "spread" dialog
 * Distance = 3 (edit text)
 * Up       = 4 (radio button)
 * Down     = 5 (radio button)
 * Left     = 6 (radio button)
 * Right    = 7 (radio button)
 */
INTSML us_spreaddlog(char *paramstart[])
{
	INTBIG itemHit, defdir;

	/* display the array dialog box */
	if (DiaInitDialog(&us_spreaddialog) != 0) return(0);

	/* "up" is the default direction, distance is 1 */
	DiaSetText(-3, "1");
	defdir = 4;
	DiaSetControl(defdir, 1);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK && DiaValidEntry(3)) break;
		if (itemHit >= 4 && itemHit <= 7)
		{
			DiaSetControl(defdir, 0);
			defdir = itemHit;
			DiaSetControl(defdir, 1);
		}
	}

	paramstart[0] = "";
	if (itemHit != CANCEL)
	{
		(void)initinfstr();
		switch (defdir)
		{
			case 4: (void)addstringtoinfstr("up ");      break;
			case 5: (void)addstringtoinfstr("down ");    break;
			case 6: (void)addstringtoinfstr("left ");    break;
			case 7: (void)addstringtoinfstr("right ");   break;
		}
		(void)addstringtoinfstr(DiaGetText(3));
		paramstart[0] = returninfstr();
	}
	DiaDoneDialog();
	return(1);
}

/****************************** TECHNOLOGY EDIT: DEPENDENT LIBRARIES DIALOG ******************************/

/* Dependent Libraries */
DIALOGITEM us_dependentlibdialogitems[] =
{
 /*  1 */ {0, {208,368,232,432}, BUTTON, "OK"},
 /*  2 */ {0, {208,256,232,320}, BUTTON, "Cancel"},
 /*  3 */ {0, {32,8,177,174}, SCROLL, ""},
 /*  4 */ {0, {8,8,24,153}, MESSAGE, "Dependent Libraries:"},
 /*  5 */ {0, {208,8,224,165}, MESSAGE, "Libraries are examined"},
 /*  6 */ {0, {40,192,64,256}, BUTTON, "Remove"},
 /*  7 */ {0, {88,192,112,256}, BUTTON, "<< Add"},
 /*  8 */ {0, {128,280,144,427}, MESSAGE, "Library (if not in list):"},
 /*  9 */ {0, {152,280,168,432}, EDITTEXT, ""},
 /* 10 */ {0, {8,272,24,361}, MESSAGE, "All Libraries:"},
 /* 11 */ {0, {224,8,240,123}, MESSAGE, "from bottom up"},
 /* 12 */ {0, {32,272,118,438}, SCROLL, ""},
 /* 13 */ {0, {184,8,200,67}, MESSAGE, "Current:"},
 /* 14 */ {0, {184,72,200,254}, MESSAGE, ""}
};
DIALOG us_dependentlibdialog = {{50,75,299,524}, "Dependent Library Selection", 14, us_dependentlibdialogitems};

void us_showliblist(char**, INTBIG);

/*
 * special case for the "dependent libraries" dialog
 * Dependent list = 3 (user item)
 * Library list   = 12 (user item)
 * Add lib        = 7 (button)
 * Remove lib     = 6 (button)
 * New name       = 9 (edit text)
 * Current lib    = 14 (message)
 */
INTSML us_dependentlibdlog(char *paramstart[])
{
	INTBIG itemHit, i, j, liblistlen;
	REGISTER VARIABLE *var;
	char **liblist, **newliblist, *pt;

	/* display the dependent library dialog box */
	if (DiaInitDialog(&us_dependentlibdialog) != 0) return(0);
	DiaSetText(14, el_curlib->libname);
	DiaInitTextDialog(12, topoflibs, nextlibs, DiaNullDlogDone, 0, SCSELMOUSE|SCSELKEY);
	DiaInitTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, 0,
		SCSELMOUSE|SCSELKEY);
	var = getval((INTBIG)el_curlib, VLIBRARY, VSTRING|VISARRAY, "EDTEC_dependent_libraries");
	if (var == NOVARIABLE) liblistlen = 0; else
	{
		liblistlen = getlength(var);
		liblist = (char **)emalloc(liblistlen * (sizeof (char *)), el_tempcluster);
		if (liblist == 0) return(0);
		for(i=0; i<liblistlen; i++)
			(void)allocstring(&liblist[i], ((char **)var->addr)[i], el_tempcluster);
	}
	us_showliblist(liblist, liblistlen);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK) break;
		if (itemHit == 6)
		{
			/* remove */
			i = DiaGetCurLine(3);
			if (i < 0 || i >= liblistlen) continue;
			efree(liblist[i]);
			for(j=i; j<liblistlen-1; j++) liblist[j] = liblist[j+1];
			liblistlen--;
			if (liblistlen == 0) efree((char *)liblist);
			us_showliblist(liblist, liblistlen);
			continue;
		}
		if (itemHit == 7)
		{
			/* add */
			pt = DiaGetText(9);
			while (*pt == ' ') pt++;
			if (*pt == 0) pt = DiaGetScrollLine(12, DiaGetCurLine(12));
			i = DiaGetCurLine(3);
			if (i < 0) i = 0;

			/* create a new list */
			newliblist = (char **)emalloc((liblistlen+1) * (sizeof (char *)), el_tempcluster);
			if (newliblist == 0) return(0);
			for(j=0; j<liblistlen; j++) newliblist[j] = liblist[j];
			if (liblistlen != 0) efree((char *)liblist);
			liblist = newliblist;

			for(j=liblistlen; j>i; j--) liblist[j] = liblist[j-1];
			liblistlen++;
			(void)allocstring(&liblist[i], pt, el_tempcluster);
			us_showliblist(liblist, liblistlen);
			DiaSetText(9, "");
			continue;
		}
	}

	if (itemHit != CANCEL)
	{
		if (liblistlen == 0)
		{
			if (var != NOVARIABLE)
				delval((INTBIG)el_curlib, VLIBRARY, "EDTEC_dependent_libraries");
		} else
		{
			(void)setval((INTBIG)el_curlib, VLIBRARY, "EDTEC_dependent_libraries",
				(INTBIG)liblist, VSTRING|VISARRAY|(liblistlen<<VLENGTHSH));
		}
	}
	for(i=0; i<liblistlen; i++) efree(liblist[i]);
	if (liblistlen != 0) efree((char *)liblist);
	DiaDoneDialog();
	return(0);
}

void us_showliblist(char **liblist, INTBIG liblistlen)
{
	REGISTER INTBIG i;

	DiaLoadTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
	if (liblistlen == 0)
	{
		DiaSelectLine(3, -1);
		return;
	}
	for(i=0; i<liblistlen; i++) DiaStuffLine(3, liblist[i]);
	DiaSelectLine(3, 0);
}

/****************************** TECHNOLOGY EDIT: VARIABLES DIALOG ******************************/

/* Technology Variables */
DIALOGITEM us_techvarsdialogitems[] =
{
 /*  1 */ {0, {208,472,232,536}, BUTTON, "OK"},
 /*  2 */ {0, {208,376,232,440}, BUTTON, "Cancel"},
 /*  3 */ {0, {24,8,143,264}, SCROLL, ""},
 /*  4 */ {0, {176,16,192,55}, MESSAGE, "Type:"},
 /*  5 */ {0, {176,56,192,142}, MESSAGE, ""},
 /*  6 */ {0, {152,104,168,536}, MESSAGE, ""},
 /*  7 */ {0, {24,280,143,536}, SCROLL, ""},
 /*  8 */ {0, {8,16,24,240}, MESSAGE, "Current Variables on Technology:"},
 /*  9 */ {0, {8,288,24,419}, MESSAGE, "Possible Variables:"},
 /* 10 */ {0, {208,280,232,344}, BUTTON, "<< Copy"},
 /* 11 */ {0, {208,24,232,88}, BUTTON, "Remove"},
 /* 12 */ {0, {176,216,192,533}, EDITTEXT, ""},
 /* 13 */ {0, {208,136,232,237}, BUTTON, "Edit Strings"},
 /* 14 */ {0, {176,168,192,212}, MESSAGE, "Value:"},
 /* 15 */ {0, {152,16,168,98}, MESSAGE, "Description:"}
};
DIALOG us_techvarsdialog = {{50,75,293,622}, "Technology Variables", 15, us_techvarsdialogitems};

void us_setcurrenttechvar(TECHVAR*);

/*
 * special case for the "technology variables" dialog
 * Type          =  5 (message)
 * Description   =  6 (message)
 * Current vars  =  3 (user)
 * Known vars    =  7 (user)
 * Remove        = 11 (button)
 * Copy          = 10 (button)
 * Edit Strings  = 13 (button)
 * the Value     = 12 (edit text)
 */
INTSML us_techvarsdlog(char *paramstart[])
{
	char **varnames, *name, *cmd[5];
	INTBIG itemHit, i, j;
	REGISTER VARIABLE *var, *ovar;
	TECHVAR *newvars, *tvar, *ltvar, *t;

	/* show the "technology variables" dialog */
	if (DiaInitDialog(&us_techvarsdialog) != 0) return(0);
	DiaInitTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);
	DiaInitTextDialog(7, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);

	/* load the known variables list */
	for(i=0; us_knownvars[i].varname != 0; i++) DiaStuffLine(7, us_knownvars[i].varname);
	DiaSelectLine(7, -1);

	/* see what variables are already in the list */
	var = getval((INTBIG)el_curlib, VLIBRARY, VSTRING|VISARRAY, "EDTEC_variable_list");
	newvars = NOTECHVAR;
	if (var != NOVARIABLE)
	{
		j = getlength(var);
		varnames = (char **)var->addr;
		for(i=0; i<j; i++)
		{
			ovar = getval((INTBIG)el_curlib, VLIBRARY, -1, varnames[i]);
			if (ovar == NOVARIABLE) continue;
			DiaStuffLine(3, varnames[i]);
			tvar = (TECHVAR *)emalloc(sizeof (TECHVAR), el_tempcluster);
			if (tvar == 0) break;
			(void)allocstring(&tvar->varname, varnames[i], el_tempcluster);
			tvar->nexttechvar = newvars;
			tvar->changed = 0;
			switch (ovar->type&(VTYPE|VISARRAY))
			{
				case VFLOAT:   tvar->fval = castfloat(ovar->addr);   break;
				case VINTEGER: tvar->ival = ovar->addr;              break;
				case VSTRING:
					(void)allocstring(&tvar->sval, (char *)ovar->addr, el_tempcluster);
					break;
			}
			tvar->vartype = ovar->type;
			newvars = tvar;
		}
	}
	DiaSelectLine(3, -1);

	/* set dialog allowances state */
	us_setcurrenttechvar(newvars);
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL || itemHit == 13) break;

		/* hit in one scroll area turns off highlight in the other */
		if (itemHit == 3)
		{
			DiaSelectLine(7, -1);
			us_setcurrenttechvar(newvars);
		}
		if (itemHit == 7)
		{
			DiaSelectLine(3, -1);
			us_setcurrenttechvar(newvars);
		}

		/* change to the value */
		if (itemHit == 12)
		{
			i = DiaGetCurLine(3);
			if (i < 0) continue;
			name = DiaGetScrollLine(3, i);
			for(t = newvars; t != NOTECHVAR; t = t->nexttechvar)
				if (namesame(t->varname, name) == 0) break;
			if (t == NOTECHVAR) continue;
			switch (t->vartype&(VTYPE|VISARRAY))
			{
				case VINTEGER:
					t->ival = myatoi(DiaGetText(12));
					t->changed++;
					break;
				case VFLOAT:
					t->fval = (float)atof(DiaGetText(12));
					t->changed++;
					break;
				case VSTRING:
					(void)reallocstring(&t->sval, DiaGetText(12), el_tempcluster);
					t->changed++;
					break;
			}
			continue;
		}

		/* the "<< Copy" button */
		if (itemHit == 10)
		{
			i = DiaGetCurLine(7);
			if (i < 0) continue;
			name = DiaGetScrollLine(7, i);
			for(t = newvars; t != NOTECHVAR; t = t->nexttechvar)
				if (namesame(t->varname, name) == 0) break;
			if (t != NOTECHVAR) continue;

			tvar = (TECHVAR *)emalloc(sizeof (TECHVAR), el_tempcluster);
			if (tvar == 0) break;
			(void)allocstring(&tvar->varname, name, el_tempcluster);
			tvar->vartype = us_knownvars[i].vartype;
			tvar->ival = 0;
			tvar->fval = 0.0;
			if ((tvar->vartype&(VTYPE|VISARRAY)) == VSTRING)
				(void)allocstring(&tvar->sval, "", el_tempcluster);
			tvar->changed = 1;
			tvar->nexttechvar = newvars;
			newvars = tvar;
			DiaLoadTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
			for(t = newvars; t != NOTECHVAR; t = t->nexttechvar) DiaStuffLine(3, t->varname);
			DiaSelectLine(3, -1);
			us_setcurrenttechvar(newvars);
			continue;
		}

		/* the "Remove" button */
		if (itemHit == 11)
		{
			i = DiaGetCurLine(3);
			if (i < 0) continue;
			name = DiaGetScrollLine(3, i);

			ltvar = NOTECHVAR;
			for(t = newvars; t != NOTECHVAR; t = t->nexttechvar)
			{
				if (namesame(t->varname, name) == 0) break;
				ltvar = t;
			}
			if (t == NOTECHVAR) continue;
			if (ltvar == NOTECHVAR) newvars = t->nexttechvar; else
				ltvar->nexttechvar = t->nexttechvar;
			if ((t->vartype&(VTYPE|VISARRAY)) == VSTRING) efree(t->sval);
			efree(t->varname);
			efree((char *)t);
			DiaLoadTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
			for(t = newvars; t != NOTECHVAR; t = t->nexttechvar) DiaStuffLine(3, t->varname);
			DiaSelectLine(3, -1);
			us_setcurrenttechvar(newvars);
			continue;
		}
	}
	if (itemHit == 13)
	{
		i = DiaGetCurLine(3);
		if (i < 0) itemHit = OK; else
			name = DiaGetScrollLine(3, i);
	}
	DiaDoneDialog();
	if (itemHit == OK || itemHit == 13)
	{
		j = 0;
		for(t = newvars; t != NOTECHVAR; t = t->nexttechvar) j++;
		if (j > 0)
		{
			varnames = (char **)emalloc(j * (sizeof (char *)), el_tempcluster);
			if (varnames == 0) return(0);
			j = 0;
			for(t = newvars; t != NOTECHVAR; t = t->nexttechvar)
			{
				(void)allocstring(&varnames[j], t->varname, el_tempcluster);
				j++;
				if (t->changed == 0) continue;
				switch (t->vartype&(VTYPE|VISARRAY))
				{
					case VINTEGER:
						(void)setval((INTBIG)el_curlib, VLIBRARY, t->varname, t->ival, VINTEGER);
						break;
					case VFLOAT:
						(void)setval((INTBIG)el_curlib, VLIBRARY, t->varname, castint(t->fval), VFLOAT);
						break;
					case VSTRING:
						(void)setval((INTBIG)el_curlib, VLIBRARY, t->varname, (INTBIG)t->sval, VSTRING);
						break;
					case VSTRING|VISARRAY:
						cmd[0] = "EMPTY";
						(void)setval((INTBIG)el_curlib, VLIBRARY, t->varname,
							(INTBIG)cmd, VSTRING|VISARRAY|(1<<VLENGTHSH));
						break;
				}
			}
			(void)setval((INTBIG)el_curlib, VLIBRARY, "EDTEC_variable_list",
				(INTBIG)varnames, VSTRING|VISARRAY|(j<<VLENGTHSH));
			for(i=0; i<j; i++) efree(varnames[i]);
			efree((char *)varnames);
		} else
		{
			if (getval((INTBIG)el_curlib, VLIBRARY, VSTRING|VISARRAY, "EDTEC_variable_list") != NOVARIABLE)
				(void)delval((INTBIG)el_curlib, VLIBRARY, "EDTEC_variable_list");
		}
	}
	if (itemHit == 13)
	{
		cmd[0] = "textedit";
		(void)initinfstr();
		(void)addstringtoinfstr("lib:~.");
		(void)addstringtoinfstr(name);
		(void)allocstring(&cmd[1], returninfstr(), el_tempcluster);
		cmd[2] = "header";
		(void)initinfstr();
		(void)addstringtoinfstr("Editing technology variable: ");
		(void)addstringtoinfstr(name);
		(void)allocstring(&cmd[3], returninfstr(), el_tempcluster);
		us_var(4, cmd);
		efree(cmd[1]);
		efree(cmd[3]);
	}
	paramstart[0] = "";
	return(1);
}

void us_setcurrenttechvar(TECHVAR *newvars)
{
	TECHVAR *t;
	INTBIG i;
	char line[20], *name;

	DiaDimItem(13);      DiaSetText(12, "");   DiaNoEditControl(12);
	DiaDimItem(11);      DiaDimItem(10);
	DiaSetText(5, "");   DiaSetText(6, "");
	i = DiaGetCurLine(3);
	if (i >= 0)
	{
		DiaUnDimItem(11);
		name = DiaGetScrollLine(3, i);
		for(i=0; us_knownvars[i].varname != 0; i++)
			if (namesame(us_knownvars[i].varname, name) == 0) break;
		if (us_knownvars[i].varname != 0) DiaSetText(6, us_knownvars[i].description);
		for(t = newvars; t != NOTECHVAR; t = t->nexttechvar)
			if (namesame(t->varname, name) == 0) break;
		if (t != NOTECHVAR) switch (t->vartype&(VTYPE|VISARRAY))
		{
			case VINTEGER:
				DiaSetText(5, "Integer");
				DiaEditControl(12);
				(void)sprintf(line, "%ld", t->ival);
				DiaSetText(-12, line);
				break;
			case VFLOAT:
				DiaSetText(5, "Real");
				DiaEditControl(12);
				(void)sprintf(line, "%g", t->fval);
				DiaSetText(-12, line);
				break;
			case VSTRING:
				DiaSetText(5, "String");
				DiaEditControl(12);
				DiaSetText(-12, t->sval);
				break;
			case VSTRING|VISARRAY:
				DiaSetText(5, "Strings");
				DiaUnDimItem(13);
				break;
		}
	}

	i = DiaGetCurLine(7);
	if (i >= 0)
	{
		name = DiaGetScrollLine(7, i);
		for(i=0; us_knownvars[i].varname != 0; i++)
			if (namesame(us_knownvars[i].varname, name) == 0) break;
		if (us_knownvars[i].varname != 0)
		{
			DiaSetText(6, us_knownvars[i].description);
			switch (us_knownvars[i].vartype&(VTYPE|VISARRAY))
			{
				case VINTEGER:         DiaSetText(5, "Integer");   break;
				case VFLOAT:           DiaSetText(5, "Real");      break;
				case VSTRING:          DiaSetText(5, "String");    break;
				case VSTRING|VISARRAY: DiaSetText(5, "Strings");   break;
			}
		}
		DiaUnDimItem(10);
	}
}

/****************************** TECHNOLOGY SELECTION DIALOG ******************************/

/* Technologies */
DIALOGITEM us_techselectdialogitems[] =
{
 /*  1 */ {0, {96,216,120,280}, BUTTON, "OK"},
 /*  2 */ {0, {24,216,48,280}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,8,153,193}, SCROLL, ""},
 /*  4 */ {0, {160,8,192,292}, MESSAGE, ""}
};
DIALOG us_techselectdialog = {{50,75,251,376}, "Change Current Technology", 4, us_techselectdialogitems};

/*
 * the meaning of "us_techlist":
 *  0  list all technologies without modification
 *  1  list all technologies, splitting "schematics" into "digital" and "analog"
 *  2  list all technologies that can be edited
 *  3  list all technologies that can be deleted
 */
int us_techlist;
static TECHNOLOGY *us_postechcomcomp;

void us_stufftechdescript(void);
INTSML us_topoftechs(char**);
char *us_nexttechs(void);

INTSML us_topoftechs(char **c) { us_postechcomcomp = el_technologies; return(1); }

char *us_nexttechs(void)
{
	REGISTER char *retname;
	REGISTER TECHNOLOGY *tech;

	for(;;)
	{
		if (us_postechcomcomp == NOTECHNOLOGY)
		{
			us_postechcomcomp = 0;
			if (us_techlist == 1) return("schematic, analog");
		}
		if (us_postechcomcomp == 0) return(0);

		/* get the next technology in the list */
		tech = us_postechcomcomp;
		us_postechcomcomp = us_postechcomcomp->nexttechnology;

		/* adjust the name if requested */
		retname = tech->techname;
		if (tech == sch_tech && us_techlist == 1)
			retname = "schematic, digital";

		/* ignore if requested */
		if (us_techlist == 2 && (tech->userbits&NONSTANDARD) != 0) continue;
		if (us_techlist == 3 && tech == gen_tech) continue;

		/* accept */
		break;
	}
	return(retname);
}

void us_stufftechdescript(void)
{
	REGISTER TECHNOLOGY *t;
	REGISTER char *tech;

	tech = DiaGetScrollLine(3, DiaGetCurLine(3));
	if (namesamen(tech, "schematic", 9) == 0)
	{
		t = sch_tech;
		DiaSetText(4, t->techdescript);
		return;
	}

	for(t = el_technologies; t != NOTECHNOLOGY; t = t->nexttechnology)
		if (strcmp(t->techname, tech) == 0) break;
	if (t == NOTECHNOLOGY) return;
	DiaSetText(4, t->techdescript);
}

/*
 * special case for the "select technology" dialog
 * Technology description = 5 (message)
 * Technology list        = 3 (user item)
 */
INTSML us_technologydlog(char *prompt, char *paramstart[])
{
	REGISTER INTBIG itemHit, i;
	REGISTER NODEPROTO *np;
	REGISTER char *defaulttech, *pt;

	/* display the new facet dialog box */
	us_techselectdialog.movable = prompt;

	/* the list of technologies depends on the nature of the operation */
	us_techlist = 0;
	defaulttech = "";
	if (namesame(prompt, "Change current technology") == 0)
	{
		us_techlist = 1;
		np = getcurfacet();
		if (np != NONODEPROTO)
			defaulttech = us_techname(np);
	} else if (namesame(prompt, "Edit technology") == 0) 
	{
		us_techlist = 2;
		defaulttech = el_curtech->techname;
	} else if (namesame(prompt, "Document technology") == 0)
	{
		us_techlist = 0;
		defaulttech = el_curtech->techname;
	} else if (namesame(prompt, "Convert to new technology") == 0) 
	{
		us_techlist = 0;
	} else if (namesame(prompt, "Delete technology") == 0)
	{
		us_techlist = 3;
	}

	if (DiaInitDialog(&us_techselectdialog) != 0) return(0);
	DiaInitTextDialog(3, us_topoftechs, us_nexttechs, DiaNullDlogDone, 0,
		SCSELMOUSE|SCSELKEY|SCDOUBLEQUIT|SCREPORT);
	us_stufftechdescript();
	for(i=0; ; i++)
	{
		pt = DiaGetScrollLine(3, i);
		if (*pt == 0) break;
		if (strcmp(pt, defaulttech) == 0)
		{
			DiaSelectLine(3, i);
			break;
		}
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK) break;
		if (itemHit == 3)
		{
			us_stufftechdescript();
			continue;
		}
	}

	paramstart[0] = "";
	if (itemHit != CANCEL)
	{
		paramstart[0] = us_putintoinfstr(DiaGetScrollLine(3, DiaGetCurLine(3)));
	}
	DiaDoneDialog();
	return(1);
}

/****************************** TECHNOLOGY OPTIONS DIALOG ******************************/

/* Technology Options */
DIALOGITEM us_techsetdialogitems[] =
{
 /*  1 */ {0, {204,220,228,284}, BUTTON, "OK"},
 /*  2 */ {0, {204,32,228,96}, BUTTON, "Cancel"},
 /*  3 */ {0, {68,20,84,156}, MESSAGE, "Metal layers:"},
 /*  4 */ {0, {68,164,84,300}, POPUP, ""},
 /*  5 */ {0, {48,20,64,156}, RADIO, "Full Geometry"},
 /*  6 */ {0, {48,164,64,300}, RADIO, "Stick Figures"},
 /*  7 */ {0, {100,8,116,112}, MESSAGE, "Artwork:"},
 /*  8 */ {0, {120,20,136,156}, CHECK, "Arrows filled"},
 /*  9 */ {0, {152,8,168,112}, MESSAGE, "Schematics:"},
 /* 10 */ {0, {172,20,188,156}, MESSAGE, "Negating Bubble Size"},
 /* 11 */ {0, {172,160,188,220}, EDITTEXT, ""},
 /* 12 */ {0, {8,8,24,192}, MESSAGE, "MOSIS CMOS Submicron:"},
 /* 13 */ {0, {28,20,44,200}, CHECK, "Switch N and P Well"},
 /* 14 */ {0, {92,8,93,300}, DIVIDELINE, ""},
 /* 15 */ {0, {144,8,145,300}, DIVIDELINE, ""}
};
DIALOG us_techsetdialog = {{75,75,312,384}, "Technology Options", 15, us_techsetdialogitems};

/*
 * special case for the "Technology Options" dialog
 * MOCMOSSUB metal layers   = 4 (popup)
 * MOCMOSSUB full-geometry  = 5 (radio)
 * MOCMOSSUB stick-figure   = 6 (radio)
 * ARTWORK filled arrow     = 8 (check)
 * SCHEMATICS invert bubble = 11 (edit text)
 * Switch N and P Well      = 13 (check)
 */
INTSML us_techoptdlog(char *paramstart[])
{
	INTBIG itemHit, which;
	INTBIG mocmosbits, artbits, origmocmosbits, origartbits, schbubblesize,
		origschbubblesize;
	WINDOWPART *w;
	static char *metalcount[] = {"2 Layers", "3 Layers", "4 Layers", "5 Layers", "6 Layers"};
	char *par[2];

	/* display the options dialog box */
	if (DiaInitDialog(&us_techsetdialog) != 0) return(0);
	DiaSetPopup(4, 5, metalcount);
	origmocmosbits = mocmosbits = asktech(mocmossub_tech, "get-state");
	origartbits = artbits = asktech(art_tech, "get-state");
	origschbubblesize = schbubblesize = asktech(sch_tech, "get-bubble-size");
	switch (mocmosbits&MOCMOSSUBMETALS)
	{
		case MOCMOSSUB2METAL: DiaSetPopupEntry(4, 0);   break;
		case MOCMOSSUB3METAL: DiaSetPopupEntry(4, 1);   break;
		case MOCMOSSUB4METAL: DiaSetPopupEntry(4, 2);   break;
		case MOCMOSSUB5METAL: DiaSetPopupEntry(4, 3);   break;
		case MOCMOSSUB6METAL: DiaSetPopupEntry(4, 4);   break;
	}
	if ((mocmosbits&MOCMOSSUBSTICKFIGURE) != 0) DiaSetControl(6, 1); else
		DiaSetControl(5, 1);
	if ((artbits&ARTWORKFILLARROWHEADS) != 0) DiaSetControl(8, 1);
	DiaSetText(11, frtoa(schbubblesize));

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == 5 || itemHit == 6)
		{
			DiaSetControl(5, 0);
			DiaSetControl(6, 0);
			DiaSetControl(itemHit, 1);
			continue;
		}
		if (itemHit == 13 || itemHit == 8)
		{
			DiaSetControl(itemHit, 1-DiaGetControl(itemHit));
			continue;
		}
	}

	if (itemHit == OK)
	{
		if (DiaGetControl(13) != 0)
		{
			(void)asktech(mocmossub_tech, "switch-n-and-p");
		}
		which = DiaGetPopupEntry(4);
		mocmosbits &= ~MOCMOSSUBMETALS;
		switch (which)
		{
			case 0: mocmosbits |= MOCMOSSUB2METAL;   break;
			case 1: mocmosbits |= MOCMOSSUB3METAL;   break;
			case 2: mocmosbits |= MOCMOSSUB4METAL;   break;
			case 3: mocmosbits |= MOCMOSSUB5METAL;   break;
			case 4: mocmosbits |= MOCMOSSUB6METAL;   break;
		}
		if (DiaGetControl(6) != 0) mocmosbits |= MOCMOSSUBSTICKFIGURE; else
			mocmosbits &= ~MOCMOSSUBSTICKFIGURE;
		if (origmocmosbits != mocmosbits)
		{
			setval((INTBIG)mocmossub_tech, VTECHNOLOGY, "TECH_state", mocmosbits, VINTEGER);
			if ((origmocmosbits&MOCMOSSUBSTICKFIGURE) != (mocmosbits&MOCMOSSUBSTICKFIGURE))
				us_figuretechopaque(mocmossub_tech);
			if (el_curtech == mocmossub_tech)
			{
				par[0] = "size";
				par[1] = "auto";
				us_menu(2, par);
				us_setmenunodearcs();
			}
		}

		if (DiaGetControl(8) != 0) artbits |= ARTWORKFILLARROWHEADS; else
			artbits &= ~ARTWORKFILLARROWHEADS;
		if (artbits != origartbits)
		{
			setval((INTBIG)art_tech, VTECHNOLOGY, "TECH_state", artbits, VINTEGER);
			(void)asktech(art_tech, "set-state", artbits);
			if (el_curtech == art_tech)
				us_setmenunodearcs();
		}

		schbubblesize = atofr(DiaGetText(11));
		(void)asktech(sch_tech, "set-bubble-size", schbubblesize);

		if ((origmocmosbits&MOCMOSSUBSTICKFIGURE) != (mocmosbits&MOCMOSSUBSTICKFIGURE) ||
			origartbits != artbits || origschbubblesize != schbubblesize)
		{
			us_pushhighlight();
			us_clearhighlightcount();
			for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
				if (w->redisphandler != 0) (*w->redisphandler)(w);
			(void)us_pophighlight(0);
		}
	}
	DiaDoneDialog();
	return(0);
}

/****************************** TEXT OPTIONS DIALOG ******************************/

/* Text options */
DIALOGITEM us_deftextdialogitems[] =
{
 /*  1 */ {0, {220,292,244,364}, BUTTON, "OK"},
 /*  2 */ {0, {220,16,244,88}, BUTTON, "Cancel"},
 /*  3 */ {0, {112,32,128,88}, RADIO, "Left"},
 /*  4 */ {0, {8,80,24,196}, POPUP, "Small"},
 /*  5 */ {0, {48,32,64,104}, RADIO, "Center"},
 /*  6 */ {0, {64,32,80,104}, RADIO, "Bottom"},
 /*  7 */ {0, {80,32,96,88}, RADIO, "Top"},
 /*  8 */ {0, {96,32,112,96}, RADIO, "Right"},
 /*  9 */ {0, {128,32,144,128}, RADIO, "Lower right"},
 /* 10 */ {0, {144,32,160,128}, RADIO, "Lower left"},
 /* 11 */ {0, {160,32,176,128}, RADIO, "Upper right"},
 /* 12 */ {0, {176,32,192,120}, RADIO, "Upper left"},
 /* 13 */ {0, {192,32,208,104}, RADIO, "Boxed"},
 /* 14 */ {0, {8,8,24,79}, MESSAGE, "Text size:"},
 /* 15 */ {0, {32,8,48,103}, MESSAGE, "Text corner:"},
 /* 16 */ {0, {48,136,80,168}, ICON, (char *)us_icon200},
 /* 17 */ {0, {80,136,112,168}, ICON, (char *)us_icon201},
 /* 18 */ {0, {112,136,144,168}, ICON, (char *)us_icon202},
 /* 19 */ {0, {144,136,176,168}, ICON, (char *)us_icon203},
 /* 20 */ {0, {176,136,208,168}, ICON, (char *)us_icon204},
 /* 21 */ {0, {176,208,192,280}, RADIO, "Outside"},
 /* 22 */ {0, {64,208,80,280}, RADIO, "Off"},
 /* 23 */ {0, {48,180,64,357}, MESSAGE, "Smart Vertical Placement:"},
 /* 24 */ {0, {80,208,96,280}, RADIO, "Inside"},
 /* 25 */ {0, {96,208,112,280}, RADIO, "Outside"},
 /* 26 */ {0, {144,208,160,280}, RADIO, "Off"},
 /* 27 */ {0, {128,180,144,376}, MESSAGE, "Smart Horizontal Placement:"},
 /* 28 */ {0, {160,208,176,280}, RADIO, "Inside"}
};
DIALOG us_deftextdialog = {{50,75,303,460}, "Text Options", 28, us_deftextdialogitems};

/*
 * special case for "text options" dialog
 * text size                = 4 (popup)
 * center                   = 5 (radio button)
 * bottom                   = 6 (radio button)
 * top                      = 7 (radio button)
 * right                    = 8 (radio button)
 * left                     = 3 (radio button)
 * lower right              = 9 (radio button)
 * lower left               = 10 (radio button)
 * upper right              = 11 (radio button)
 * upper left               = 12 (radio button)
 * boxed                    = 13 (radio button)
 * smart vertical off       = 22 (radio button)
 * smart vertical inside    = 24 (radio button)
 * smart vertical outside   = 25 (radio button)
 * smart horizontal off     = 26 (radio button)
 * smart horizontal inside  = 28 (radio button)
 * smart horizontal outside = 21 (radio button)
 */
INTSML us_deftextdlog(char *prompt, char *paramstart[])
{
	INTBIG itemHit, i;
	INTSML x, y;
	REGISTER VARIABLE *var;
	REGISTER INTBIG grabpoint, font, smarthstyle, smartvstyle;
	RECTAREA itemRect;
	static char *sizenames[12] = {"4 Points", "6 Points", "8 Points", "10 Points", "12 Points",
		"14 Points", "16 Points", "18 Points", "20 Points", "Small", "Medium", "Large"};
	static INTBIG sizelist[12] = {TXT4P, TXT6P, TXT8P, TXT10P, TXT12P, TXT14P, TXT16P,
		TXT18P, TXT20P, TXTSMALL, TXTMEDIUM, TXTLARGE};
	static struct butlist poslist[10] =
	{
		{VTPOSCENT,       5},
		{VTPOSUP,         6},
		{VTPOSDOWN,       7},
		{VTPOSLEFT,       8},
		{VTPOSRIGHT,      3},
		{VTPOSUPLEFT,     9},
		{VTPOSUPRIGHT,   10},
		{VTPOSDOWNLEFT,  11},
		{VTPOSDOWNRIGHT, 12},
		{VTPOSBOXED,     13}
	};

	/* display the default text dialog box */
	if (DiaInitDialog(&us_deftextdialog) != 0) return(0);
	DiaSetPopup(4, 12, sizenames);

	/* set current defaults */
	var = getval((INTBIG)us_aid, VAID, VINTEGER, "USER_default_text_style");
	if (var == NOVARIABLE) grabpoint = VTPOSCENT; else grabpoint = var->addr;
	for(i=0; i<10; i++) if (grabpoint == poslist[i].value)
	{
		DiaSetControl(poslist[i].button, 1);
		break;
	}

	var = getval((INTBIG)us_aid, VAID, VINTEGER, "USER_default_text_smart_style");
	if (var == NOVARIABLE)
	{
		smarthstyle = smartvstyle = 0;
	} else
	{
		smarthstyle = var->addr & 03;
		smartvstyle = (var->addr >> 2) & 03;
	}
	switch (smarthstyle)
	{
		case 1:  DiaSetControl(28, 1);  break;
		case 2:  DiaSetControl(21, 1);  break;
		default: DiaSetControl(26, 1);  break;
	}
	switch (smartvstyle)
	{
		case 1:  DiaSetControl(24, 1);  break;
		case 2:  DiaSetControl(25, 1);  break;
		default: DiaSetControl(22, 1);  break;
	}

	var = getval((INTBIG)us_aid, VAID, VINTEGER, "USER_default_text_size");
	if (var == NOVARIABLE) font = TXTSMALL; else font = var->addr;
	for(i=0; i<12; i++) if (font == sizelist[i])
	{
		DiaSetPopupEntry(4, i);
		break;
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;

		if (itemHit >= 16 && itemHit <= 20)
		{
			DiaItemRect(itemHit, &itemRect);
			DiaGetMouse(&x, &y);
			itemHit = (itemHit-16) * 2;
			if (y > (itemRect.top + itemRect.bottom) / 2) itemHit++;
			itemHit = poslist[itemHit].button;
		}

		/* hits on the orientation buttons */
		for(i=0; i<10; i++) if (itemHit == poslist[i].button)
		{
			DiaSetControl(poslist[grabpoint].button, 0);
			grabpoint = i;
			DiaSetControl(poslist[i].button, 1);
			break;
		}

		if (itemHit == 22 || itemHit == 24 || itemHit == 25)
		{
			DiaSetControl(22, 0);
			DiaSetControl(24, 0);
			DiaSetControl(25, 0);
			DiaSetControl(itemHit, 1);
			continue;
		}
		if (itemHit == 26 || itemHit == 28 || itemHit == 21)
		{
			DiaSetControl(26, 0);
			DiaSetControl(28, 0);
			DiaSetControl(21, 0);
			DiaSetControl(itemHit, 1);
			continue;
		}
	}

	if (itemHit == OK)
	{
		font = sizelist[DiaGetPopupEntry(4)];
		setval((INTBIG)us_aid, VAID, "USER_default_text_size", font, VINTEGER);
		setval((INTBIG)us_aid, VAID, "USER_default_text_style", grabpoint, VINTEGER);
		if (DiaGetControl(26) != 0) smarthstyle = 0; else
			if (DiaGetControl(28) != 0) smarthstyle = 1; else
				if (DiaGetControl(21) != 0) smarthstyle = 2;
		if (DiaGetControl(22) != 0) smartvstyle = 0; else
			if (DiaGetControl(24) != 0) smartvstyle = 1; else
				if (DiaGetControl(25) != 0) smartvstyle = 2;
		setval((INTBIG)us_aid, VAID, "USER_default_text_smart_style",
			smarthstyle | (smartvstyle << 2), VINTEGER);
	}
	DiaDoneDialog();
	paramstart[0] = "x";
	return(1);
}

/****************************** TRACE INFO DIALOG ******************************/

/* Trace Info */
DIALOGITEM us_tracedialogitems[] =
{
 /*  1 */ {0, {96,208,120,272}, BUTTON, "OK"},
 /*  2 */ {0, {40,208,64,272}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,8,168,192}, SCROLL, ""},
 /*  4 */ {0, {184,8,200,28}, MESSAGE, "X:"},
 /*  5 */ {0, {184,32,200,104}, EDITTEXT, ""},
 /*  6 */ {0, {216,8,232,28}, MESSAGE, "Y:"},
 /*  7 */ {0, {216,32,232,104}, EDITTEXT, ""},
 /*  8 */ {0, {208,160,232,272}, BUTTON, "Duplicate Point"},
 /*  9 */ {0, {176,160,200,272}, BUTTON, "Delete Point"}
};
DIALOG us_tracedialog = {{50,75,291,356}, "Outline Information", 9, us_tracedialogitems};

/* special case for the "trace" dialog
 * Points    = 3 (user item)
 * X         = 5 (edit text)
 * Y         = 7 (edit text)
 * duplicate = 8 (button)
 * delete    = 9 (button)
 */
INTSML us_tracedlog(char *paramstart[])
{
	INTBIG itemHit, len, i, j, space, changed;
	INTBIG *pts, *newpts, x, y;
	char lne[256];
	HIGHLIGHT *high;
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *var;

	/* make sure there is a highlighted node with trace information */
	high = us_getonehighlight();
	if (high == NOHIGHLIGHT) return(0);
	if ((high->status&HIGHTYPE) != HIGHFROM) return(0);
	if (high->fromgeom->entrytype != OBJNODEINST) return(0);
	ni = (NODEINST *)high->fromgeom->entryaddr.ni;
	if ((ni->proto->userbits&HOLDSTRACE) == 0) return(0);
	var = gettrace(ni);

	/* display the trace dialog box */
	if (DiaInitDialog(&us_tracedialog) != 0) return(0);
	DiaInitTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);

	/* copy trace data and display it */
	if (var == NOVARIABLE) len = 0; else len = getlength(var) / 2;
	x = (ni->highx + ni->lowx) / 2;
	y = (ni->highy + ni->lowy) / 2;
	space = len+1;
	pts = (INTBIG *)emalloc(space * 2 * SIZEOFINTBIG, el_tempcluster);
	for(i=0; i<len; i++)
	{
		pts[i*2] = ((INTBIG *)var->addr)[i*2] + x;
		pts[i*2+1] = ((INTBIG *)var->addr)[i*2+1] + y;
		(void)sprintf(lne, "%d: (%s, %s)", i, latoa(pts[i*2]), latoa(pts[i*2+1]));
		DiaStuffLine(3, lne);
	}
	if (len == 0) DiaSelectLine(3, -1); else
	{
		DiaSelectLine(3, 0);
		DiaSetText(-5, latoa(pts[0]));
		DiaSetText(7, latoa(pts[1]));
	}

	/* loop until done */
	changed = 0;
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL || itemHit == OK) break;
		if (itemHit == 3)
		{
			/* load this point */
			i = DiaGetCurLine(3);
			if (i < 0 || i >= len) continue;
			DiaSetText(5, latoa(pts[i*2]));
			DiaSetText(7, latoa(pts[i*2+1]));
			continue;
		}
		if (itemHit == 8)
		{
			/* duplicate point */
			changed++;
			if (len == 0)
			{
				pts[0] = pts[1] = 0;
				(void)sprintf(lne, "%d: (%s, %s)", 0, latoa(pts[0]), latoa(pts[1]));
				DiaSetScrollLine(3, 0, lne);
				len++;
				continue;
			}

			if (len >= space)
			{
				newpts = (INTBIG *)emalloc((len+1) * 2 * SIZEOFINTBIG, el_tempcluster);
				if (newpts == 0) continue;
				for(i=0; i<len; i++)
				{
					newpts[i*2] = pts[i*2];
					newpts[i*2+1] = pts[i*2+1];
				}
				efree((char *)pts);
				pts = newpts;
				space = len + 1;
			}
			i = DiaGetCurLine(3);
			for(j=len; j>i; j--)
			{
				pts[j*2] = pts[(j-1)*2];
				pts[j*2+1] = pts[(j-1)*2+1];
				(void)sprintf(lne, "%d: (%s, %s)", j, latoa(pts[j*2]), latoa(pts[j*2+1]));
				DiaSetScrollLine(3, j, lne);
			}
			len++;
			continue;
		}
		if (itemHit == 9)
		{
			/* delete point */
			changed++;
			if (len <= 1) continue;
			i = DiaGetCurLine(3);
			for(j=i; j<len-1; j++)
			{
				pts[j*2] = pts[(j+1)*2];
				pts[j*2+1] = pts[(j+1)*2+1];
				(void)sprintf(lne, "%d: (%s, %s)", j, latoa(pts[j*2]), latoa(pts[j*2+1]));
				DiaSetScrollLine(3, j, lne);
			}
			DiaSetScrollLine(3, len-1, "");
			len--;
			if (i == len) DiaSelectLine(3, i-1);
			continue;
		}
		if (itemHit == 5 || itemHit == 7)
		{
			i = DiaGetCurLine(3);
			if (i < 0) continue;
			x = atola(DiaGetText(5));
			y = atola(DiaGetText(7));
			if (pts[i*2] == x && pts[i*2+1] == y) continue;
			changed++;
			pts[i*2] = x;   pts[i*2+1] = y;
			(void)sprintf(lne, "%d: (%s, %s)", i, latoa(pts[i*2]), latoa(pts[i*2+1]));
			DiaSetScrollLine(3, i, lne);
			continue;
		}
	}

	paramstart[0] = "";
	if (itemHit != CANCEL && changed != 0)
	{
		us_pushhighlight();
		us_clearhighlightcount();
		us_settrace(ni, pts, (INTSML)len);
		(void)us_pophighlight(1);
	}
	DiaDoneDialog();
	efree((char *)pts);
	return(1);
}

/****************************** UNITS DIALOG ******************************/

/* Units */
DIALOGITEM us_unitsdialogitems[] =
{
 /*  1 */ {0, {84,236,108,308}, BUTTON, "OK"},
 /*  2 */ {0, {88,24,112,96}, BUTTON, "Cancel"},
 /*  3 */ {0, {32,248,48,323}, EDITTEXT, ""},
 /*  4 */ {0, {32,8,48,244}, MESSAGE, "Internal units per Lambda unit:"},
 /*  5 */ {0, {8,128,24,288}, POPUP, "Lambda units"},
 /*  6 */ {0, {8,8,24,127}, MESSAGE, "Display Units:"},
 /*  7 */ {0, {56,128,72,288}, POPUP, "Half-millimicrons"},
 /*  8 */ {0, {56,8,72,127}, MESSAGE, "Internal Units:"}
};
DIALOG us_unitsdialog = {{50,75,171,407}, "Change Units", 8, us_unitsdialogitems};

/*
 * special case for the "units" dialog
 * Display units    = 5 (popup)
 * Lambda           = 3 (edit text)
 * Internal units   = 7 (popup)
 */
INTSML us_lambdadlog(char *paramstart[])
{
	INTBIG itemHit, i;
	INTSML intunit;
	char ent[20];
	static char *dispunitnames[8] = {"Lambda units", "Inches", "Centimeters",
		"Millimeters", "Mils", "Microns", "Centimicrons", "Millimicrons"};
	static char *intunitnames[2] = {"Half-Millimicrons", "Half-Decimicrons"};

	/* display the units dialog box */
	if (DiaInitDialog(&us_unitsdialog) != 0) return(0);
	(void)sprintf(ent, "%ld", el_curtech->deflambda);
	DiaSetText(-3, ent);
	if ((el_units&DISPLAYUNITS) == DISPUNITLAMBDA)
	{
		DiaUnDimItem(4);
		DiaUnDimItem(3);
		DiaEditControl(3);
	} else
	{
		DiaDimItem(4);
		DiaDimItem(3);
		DiaNoEditControl(3);
	}
	DiaSetPopup(5, 8, dispunitnames);
	switch (el_units&DISPLAYUNITS)
	{
		case DISPUNITLAMBDA: DiaSetPopupEntry(5, 0);   break;
		case DISPUNITINCH:   DiaSetPopupEntry(5, 1);   break;
		case DISPUNITCM:     DiaSetPopupEntry(5, 2);   break;
		case DISPUNITMM:     DiaSetPopupEntry(5, 3);   break;
		case DISPUNITMIL:    DiaSetPopupEntry(5, 4);   break;
		case DISPUNITMIC:    DiaSetPopupEntry(5, 5);   break;
		case DISPUNITCMIC:   DiaSetPopupEntry(5, 6);   break;
		case DISPUNITMMIC:   DiaSetPopupEntry(5, 7);   break;
	}
	DiaSetPopup(7, 2, intunitnames);
	switch (el_units&INTERNALUNITS)
	{
		case INTUNITHMMIC:   DiaSetPopupEntry(7, 0);   break;
		case INTUNITHDMIC:   DiaSetPopupEntry(7, 1);   break;
	}

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK && DiaValidEntry(3)) break;
		if (itemHit == 5)
		{
			if (DiaGetPopupEntry(5) == 0)
			{
				DiaUnDimItem(4);
				DiaUnDimItem(3);
				DiaEditControl(3);
			} else
			{
				DiaDimItem(4);
				DiaDimItem(3);
				DiaNoEditControl(3);
			}
		}
	}

	paramstart[0] = "";
	if (itemHit != CANCEL)
	{
		/* set display unit */
		i = DiaGetPopupEntry(5);
		switch (i)
		{
			case 0: el_units = (el_units & ~DISPLAYUNITS) | DISPUNITLAMBDA;   break;
			case 1: el_units = (el_units & ~DISPLAYUNITS) | DISPUNITINCH;     break;
			case 2: el_units = (el_units & ~DISPLAYUNITS) | DISPUNITCM;       break;
			case 3: el_units = (el_units & ~DISPLAYUNITS) | DISPUNITMM;       break;
			case 4: el_units = (el_units & ~DISPLAYUNITS) | DISPUNITMIL;      break;
			case 5: el_units = (el_units & ~DISPLAYUNITS) | DISPUNITMIC;      break;
			case 6: el_units = (el_units & ~DISPLAYUNITS) | DISPUNITCMIC;     break;
			case 7: el_units = (el_units & ~DISPLAYUNITS) | DISPUNITMMIC;     break;
		}

		/* see if internal unit was changed */
		i = DiaGetPopupEntry(7);
		switch (i)
		{
			case 0: intunit = INTUNITHMMIC;   break;
			case 1: intunit = INTUNITHDMIC;   break;
		}
		if (intunit != (el_units&INTERNALUNITS))
		{
			/* only change internal unit, not lambda */
			changeinternalunits(NOLIBRARY, el_units, intunit);
		} else
		{
			/* accept change to lambda value */
			paramstart[0] = us_putintoinfstr(DiaGetText(3));
		}
	}
	DiaDoneDialog();
	return(1);
}

/****************************** VARIABLES DIALOG ******************************/

/* Variables */
DIALOGITEM us_variabledialogitems[] =
{
 /*  1 */ {0, {408,344,432,400}, BUTTON, "OK"},
 /*  2 */ {0, {352,8,368,56}, MESSAGE, "Value:"},
 /*  3 */ {0, {336,8,337,408}, DIVIDELINE, ""},
 /*  4 */ {0, {24,8,40,64}, MESSAGE, "Object:"},
 /*  5 */ {0, {8,80,24,240}, RADIO, "Currently Highlighted"},
 /*  6 */ {0, {24,80,40,240}, RADIO, "Current Facet"},
 /*  7 */ {0, {40,80,56,240}, RADIO, "Current Cell"},
 /*  8 */ {0, {56,80,72,240}, RADIO, "Current Library"},
 /*  9 */ {0, {8,256,24,408}, RADIO, "Current Technology"},
 /* 10 */ {0, {24,256,40,408}, RADIO, "Current Aid"},
 /* 11 */ {0, {144,24,160,96}, MESSAGE, "Attribute:"},
 /* 12 */ {0, {160,8,304,184}, SCROLL, ""},
 /* 13 */ {0, {312,32,328,152}, CHECK, "New Attribute:"},
 /* 14 */ {0, {312,160,328,400}, EDITTEXT, ""},
 /* 15 */ {0, {216,192,232,251}, CHECK, "Array"},
 /* 16 */ {0, {240,200,256,248}, MESSAGE, "Index:"},
 /* 17 */ {0, {240,250,256,312}, EDITTEXT, ""},
 /* 18 */ {0, {408,192,432,296}, BUTTON, "Set Attribute"},
 /* 19 */ {0, {344,80,376,400}, EDITTEXT, ""},
 /* 20 */ {0, {408,24,432,144}, BUTTON, "Delete Attribute"},
 /* 21 */ {0, {168,192,184,288}, CHECK, "Displayable"},
 /* 22 */ {0, {192,192,208,288}, CHECK, "Temporary"},
 /* 23 */ {0, {276,224,300,361}, BUTTON, "Examine Attribute"},
 /* 24 */ {0, {112,40,128,80}, MESSAGE, "Type:"},
 /* 25 */ {0, {112,80,128,216}, MESSAGE, ""},
 /* 26 */ {0, {144,184,160,224}, MESSAGE, "Type:"},
 /* 27 */ {0, {144,224,160,383}, MESSAGE, ""},
 /* 28 */ {0, {136,8,137,408}, DIVIDELINE, ""},
 /* 29 */ {0, {80,80,112,408}, MESSAGE, ""},
 /* 30 */ {0, {80,32,96,80}, MESSAGE, "Name:"},
 /* 31 */ {0, {168,304,184,410}, POPUP, "Not Code"},
 /* 32 */ {0, {384,80,400,160}, MESSAGE, "Evaluation:"},
 /* 33 */ {0, {384,160,400,400}, MESSAGE, ""},
 /* 34 */ {0, {232,320,248,366}, BUTTON, "Next"},
 /* 35 */ {0, {248,320,264,366}, BUTTON, "Prev"},
 /* 36 */ {0, {40,256,56,408}, RADIO, "Current Window"},
 /* 37 */ {0, {56,256,72,408}, RADIO, "Current Constraint"}
};
DIALOG us_variabledialog = {{50,75,492,494}, "Variable Control", 37, us_variabledialogitems};

/* special case for the "variables" dialog
 * Currently highlighted object = 5 (radio button)
 * Current facet                = 6 (radio button)
 * Current cell                 = 7 (radio button)
 * Current library              = 8 (radio button)
 * Current technology           = 9 (radio button)
 * Current aid                  = 10 (radio button)
 * Object type                  = 25 (message)
 * Object name                  = 29 (message)
 * Object/Attribute Divider bar = 28 (message)
 * Attribute name list          = 12 (scroll)
 * Attribute type               = 27 (message)
 * Examine                      = 23 (button)
 * Displayable attribute        = 21 (check)
 * Temporary attribute          = 22 (check)
 * Array attribute              = 15 (check)
 * Array index                  = 17 (edit text)
 * New Attribute                = 13 (check)
 * New Attribute name           = 14 (edit text)
 * Attribute/Value Divider bar  = 3 (message)
 * Attribute value              = 19 (edit text)
 * Attribute language           = 31 (popup)
 * Set Attribute                = 18 (button)
 * OK                           = 1 (button)
 * Delete Attribute             = 20 (button)
 * CANCEL (not a button)        = 2 (returned by  us_modaldialog()
 */
INTBIG us_possearch, us_varaddr, us_curvaraddr;
INTBIG us_vartype, us_curvartype;
VARIABLE *us_variablesvar;

INTSML us_topofdlgvars(char**);
char *us_nextdlgvars(void);
void us_varestablish(INTBIG, INTBIG, char*);
void us_varidentify(void);

INTSML us_topofdlgvars(char **c)
{
	us_possearch = initobjlist(us_varaddr, us_vartype, 0);
	return(1);
}
char *us_nextdlgvars(void)
{
	VARIABLE *var;

	return(nextobjectlist(&var, us_possearch));
}

INTSML us_variablesdlog(char *paramstart[])
{
	INTBIG itemHit, i, j, which, len;
	REGISTER INTBIG search, newval, newtype, *newarray, oldlen, descript;
	char *varname, *name, *newstr, *pp, line[100];
	HIGHLIGHT high;
	VARIABLE *var;
	NODEPROTO *curfacet;
	static char nullstr[] = "";
	static char *languages[] = {"Not Code", "TCL", "LISP", "JAVA"};

	/* display the variables dialog box */
	if (DiaInitDialog(&us_variabledialog) != 0) return(0);
#if LANGTCL == 0
	languages[1] = "TCL (not available)";
#endif
#if LANGLISP == 0
	languages[2] = "LISP (not available)";
#endif
#if LANGJAVA == 0
	languages[3] = "JAVA (not available)";
#endif
	DiaSetPopup(31, 4, languages);
	DiaNoEditControl(14);
	DiaNoEditControl(17);

	/* presume the current facet or library */
	curfacet = getcurfacet();
	us_varaddr = (INTBIG)curfacet;
	us_vartype = VNODEPROTO;
	if (curfacet == NONODEPROTO)
	{
		/* no current facet: cannot select facet/cell */
		DiaDimItem(6);
		DiaDimItem(7);
		us_varaddr = (INTBIG)el_curlib;
		us_vartype = VLIBRARY;
	}

	/* see if a single object is highlighted */
	us_curvaraddr = 0;
	us_curvartype = VUNKNOWN;
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var != NOVARIABLE)
	{
		if (getlength(var) == 1)
		{
			if (us_makehighlight(((char **)var->addr)[0], &high) == 0)
			{
				if ((high.status&HIGHTYPE) == HIGHFROM)
				{
					if (high.fromgeom->entrytype == OBJARCINST) us_vartype = VARCINST; else
						us_vartype = VNODEINST;
					us_curvaraddr = us_varaddr = (INTBIG)high.fromgeom->entryaddr.blind;
					us_curvartype = us_vartype;
				}
			}
		}
	}
	if (us_vartype == VNODEPROTO || us_vartype == VLIBRARY) DiaDimItem(5);

	/* initialize the attribute list */
	DiaInitTextDialog(12, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);
	us_varestablish(us_varaddr, us_vartype, "");

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == 1 || itemHit == 18 || itemHit == 20 || itemHit == 2) break;
		if (itemHit == 12)
		{
			/* select a new attribute name */
			us_varidentify();
			continue;
		}
		if (itemHit == 13)
		{
			/* want a new variable */
			DiaSetText(14, "");
			i = DiaGetControl(13);
			DiaSetControl(13, 1-i);
			if (i != 0) DiaNoEditControl(14); else DiaEditControl(14);
			DiaSetText(19, "");
			DiaSetText(-14, "");
			DiaUnDimItem(15);
			DiaUnDimItem(21);   DiaUnDimItem(22);
			DiaUnDimItem(31);
			DiaDimItem(23);
			DiaDimItem(20);
			DiaUnDimItem(18);
			continue;
		}
		if (itemHit == 23)
		{
			/* examine the variable */
			which = DiaGetCurLine(12);
			if (which < 0) continue;
			varname = DiaGetScrollLine(12, which);
			search = initobjlist(us_varaddr, us_vartype, 0);
			for(;;)
			{
				name = nextobjectlist(&var, search);
				if (name == 0) break;
				if (strcmp(name, varname) == 0)
				{
					if ((var->type&VISARRAY) != 0)
					{
						i = myatoi(DiaGetText(17));
						if (i < 0) i = 0;
						if (i > getlength(var)) i = getlength(var)-1;
						(void)sprintf(line, "%s[%d]", varname, i);
						us_varestablish(((INTBIG *)var->addr)[i], var->type&VTYPE, line);
					} else us_varestablish(var->addr, var->type&VTYPE, varname);
					break;
				}
			}
			continue;
		}
		if (itemHit == 17)
		{
			/* changing index */
			if (DiaGetControl(13) != 0) continue;
			i = myatoi(DiaGetText(17));
			if (i < 0) i = 0;
			len = getlength(us_variablesvar);
			if (i >= len) i = len-1;
			DiaSetText(19, describevariable(us_variablesvar, (INTSML)i, -1));
			continue;
		}
		if (itemHit == 34)
		{
			/* increment index */
			i = myatoi(DiaGetText(17)) + 1;
			if (DiaGetControl(13) == 0)
			{
				len = getlength(us_variablesvar);
				if (i >= len) i = len-1;
			}
			(void)sprintf(line, "%d", i);
			DiaSetText(17, line);
			if (DiaGetControl(13) == 0)
				DiaSetText(19, describevariable(us_variablesvar, (INTSML)i, -1));
			continue;
		}
		if (itemHit == 35)
		{
			/* decrement index */
			i = myatoi(DiaGetText(17)) - 1;
			if (i < 0) continue;
			(void)sprintf(line, "%d", i);
			DiaSetText(17, line);
			if (DiaGetControl(13) == 0)
				DiaSetText(19, describevariable(us_variablesvar, (INTSML)i, -1));
			continue;
		}
		if (itemHit == 5)
		{
			/* want current object */
			us_varestablish(us_curvaraddr, us_curvartype, "");
			continue;
		}
		if (itemHit == 6)
		{
			/* want current facet */
			us_varestablish((INTBIG)curfacet, VNODEPROTO, "");
			continue;
		}
		if (itemHit == 7)
		{
			/* want current cell */
			us_varestablish((INTBIG)curfacet->cell, VCELL, "");
			continue;
		}
		if (itemHit == 8)
		{
			/* want current library */
			us_varestablish((INTBIG)el_curlib, VLIBRARY, "");
			continue;
		}
		if (itemHit == 9)
		{
			/* want current technology */
			us_varestablish((INTBIG)el_curtech, VTECHNOLOGY, "");
			continue;
		}
		if (itemHit == 10)
		{
			/* want current aid */
			us_varestablish((INTBIG)us_aid, VAID, "");
			continue;
		}
		if (itemHit == 36)
		{
			/* want current window */
			us_varestablish((INTBIG)el_curwindowpart, VWINDOWPART, "");
			continue;
		}
		if (itemHit == 37)
		{
			/* want current constraint */
			us_varestablish((INTBIG)el_curconstraint, VCONSTRAINT, "");
			continue;
		}
		if (itemHit == 21)
		{
			/* displayable attribute */
			DiaSetControl(21, 1-DiaGetControl(21));
			continue;
		}
		if (itemHit == 22)
		{
			/* temporary attribute */
			DiaSetControl(22, 1-DiaGetControl(22));
			continue;
		}
		if (itemHit == 15)
		{
			/* array attribute */
			i = DiaGetControl(15);
			DiaSetControl(15, 1-i);
			if (i == 0)
			{
				DiaUnDimItem(16);   DiaUnDimItem(17);
				DiaUnDimItem(34);   DiaUnDimItem(35);
				DiaEditControl(17);
				DiaSetText(17, "0");
			} else
			{
				DiaDimItem(16);   DiaDimItem(17);
				DiaDimItem(34);   DiaDimItem(35);
				DiaNoEditControl(17);
				DiaSetText(17, "");
			}
			continue;
		}
	}

	/* handle attribute deletion */
	if (itemHit == 20)
	{
		startobjectchange(us_varaddr, us_vartype);
		delvalkey(us_varaddr, us_vartype, us_variablesvar->key);
		endobjectchange(us_varaddr, us_vartype);
	}

	/* handle attribute setting */
	if (itemHit == 18)
	{
		/* initialize object to be changed */
		startobjectchange(us_varaddr, us_vartype);
		if (DiaGetControl(5) != 0)
		{
			us_pushhighlight();
			us_clearhighlightcount();
		}

		/* get new attribute string */
		newstr = DiaGetText(19);

		/* determine type of attribute */
		if (DiaGetControl(13) != 0)
		{
			/* setting new attribute */
			varname = DiaGetText(14);
			if (isanumber(newstr) == 0) newtype = VSTRING; else
			{
				newtype = VINTEGER;
				for(pp = newstr; *pp != 0; pp++) if (*pp == '.')
				{
					newtype = VFLOAT;
					break;
				}
			}
			if (DiaGetControl(15) != 0) newtype |= VISARRAY;
		} else
		{
			/* modifying existing attribute */
			which = DiaGetCurLine(12);
			if (which >= 0) varname = DiaGetScrollLine(12, which); else
				varname = "";
			newtype = us_variablesvar->type;
		}
		if (DiaGetControl(21) != 0) newtype |= VDISPLAY; else newtype &= ~VDISPLAY;
		if (DiaGetControl(22) != 0) newtype |= VCANTSET; else newtype &= ~VCANTSET;
		newtype &= ~(VCODE1 | VCODE2);
		switch (DiaGetPopupEntry(31))
		{
			case 1: newtype |= VTCL;    break;
			case 2: newtype |= VLISP;   break;
			case 3: newtype |= VJAVA;   break;
		}

		/* get proper attribute value */
		newval = (INTBIG)newstr;
		if ((newtype&(VCODE1|VCODE2)) == 0)
		{
			switch (newtype&VTYPE)
			{
				case VINTEGER:
				case VADDRESS:
				case VSHORT:
					newval = myatoi(newstr);
					break;
				case VFRACT:
					newval = atofr(newstr);
					break;
				case VFLOAT:
				case VDOUBLE:
					newval = castint((float)atof(newstr));
					break;
			}
		}

		/* set the attribute if valid */
		if (*varname != 0)
		{
			/* see if it is an array attribute */
			if (DiaGetControl(15) != 0)
			{
				/* get the array index, examine former attribute */
				which = myatoi(DiaGetText(17));
				if (which < 0) which = 0;
				var = getval(us_varaddr, us_vartype, -1, varname);
				if (var == NOVARIABLE)
				{
					/* attribute did not exist: create the array */
					newarray = emalloc(((which+1) * SIZEOFINTBIG), el_tempcluster);
					if (newarray == 0) return(0);
					for(j=0; j<=which; j++) newarray[j] = newval;
					newtype |= VISARRAY | ((which+1)<<VLENGTHSH);
					var = setval(us_varaddr, us_vartype, varname, (INTBIG)newarray, newtype);
					if (var != NOVARIABLE) var->textdescript = defaulttextdescript(NOGEOM);
					efree((char *)newarray);
				} else if (getlength(var) <= which)
				{
					/* extend existing array attribute */
					oldlen = getlength(var);
					newarray = (INTBIG *)emalloc(((which+1) * SIZEOFINTBIG), el_tempcluster);
					if (newarray == 0) return(0);
					if ((newtype&VTYPE) == VSTRING)
					{
						for(j=0; j<oldlen; j++)
							(void)allocstring((char **)&newarray[j],
								(char *)((INTBIG *)var->addr)[j], el_tempcluster);
						for(j=oldlen; j<which; j++) newarray[j] = (INTBIG)nullstr;
					} else
					{
						for(j=0; j<oldlen; j++) newarray[j] = ((INTBIG *)var->addr)[j];
						for(j=oldlen; j<which; j++) newarray[j] = 0;
					}
					newarray[which] = newval;
					newtype = (newtype & ~VLENGTH) | ((which+1)<<VLENGTHSH);
					(void)setval(us_varaddr, us_vartype, varname, (INTBIG)newarray, newtype);
					if ((newtype&VTYPE) == VSTRING)
						for(j=0; j<oldlen; j++) efree((char *)newarray[j]);
					efree((char *)newarray);
				} else
				{
					/* set a single attribute entry */
					(void)setind(us_varaddr, us_vartype, varname, which, newval);
				}
			} else
			{
				/* setting non-array or code attribute */
				var = getval(us_varaddr, us_vartype, newtype, varname);
				if (var != NOVARIABLE) descript = var->textdescript; else
					descript = defaulttextdescript(NOGEOM);
				var = setval(us_varaddr, us_vartype, varname, newval, newtype);
				if (var != NOVARIABLE) var->textdescript = descript;
			}
		}

		/* finish the change */
		if (DiaGetControl(5) != 0) (void)us_pophighlight(1);
		endobjectchange(us_varaddr, us_vartype);
	}
	DiaDoneDialog();
	paramstart[0] = "x";
	return(1);
}

void us_varestablish(INTBIG addr, INTBIG type, char *thisname)
{
	INTBIG i;

	/* determine which radio button to set */
	for(i=5; i<=10; i++) DiaSetControl(i, 0);
	DiaSetControl(36, 0);
	DiaSetControl(37, 0);
	if (us_curvaraddr != 0 && addr == us_curvaraddr && type == us_curvartype)
	{
		DiaSetControl(5, 1);
		DiaSetText(29, "~");
		thisname = "";
	} else if (getcurfacet() != NONODEPROTO && addr == (INTBIG)getcurfacet() && type == VNODEPROTO)
	{
		DiaSetControl(6, 1);
		DiaSetText(29, "facet:~");
		thisname = "";
	} else if (getcurfacet() != NONODEPROTO && addr == (INTBIG)getcurfacet()->cell && type == VCELL)
	{
		DiaSetControl(7, 1);
		DiaSetText(29, "cell:~");
		thisname = "";
	} else if (addr == (INTBIG)el_curlib && type == VLIBRARY)
	{
		DiaSetControl(8, 1);
		DiaSetText(29, "lib:~");
		thisname = "";
	} else if (addr == (INTBIG)el_curtech && type == VTECHNOLOGY)
	{
		DiaSetControl(9, 1);
		DiaSetText(29, "tech:~");
		thisname = "";
	} else if (addr == (INTBIG)us_aid && type == VAID)
	{
		DiaSetControl(10, 1);
		DiaSetText(29, "aid:~");
		thisname = "";
	} else if (addr == (INTBIG)el_curwindowpart && type == VWINDOWPART)
	{
		DiaSetControl(36, 1);
		DiaSetText(29, "window:~");
		thisname = "";
	} else if (addr == (INTBIG)el_curconstraint && type == VCONSTRAINT)
	{
		DiaSetControl(37, 1);
		DiaSetText(29, "constraint:~");
		thisname = "";
	}

	if (*thisname != 0)
	{
		(void)initinfstr();
		(void)addstringtoinfstr(DiaGetText(29));
		(void)addtoinfstr('.');
		(void)addstringtoinfstr(thisname);
		DiaSetText(29, returninfstr());
	}
	us_varaddr = addr;
	us_vartype = type;
	DiaSetText(25, us_variabletypename(type));
	DiaLoadTextDialog(12, us_topofdlgvars, us_nextdlgvars, DiaNullDlogDone, 0);
	us_varidentify();
}

void us_varidentify(void)
{
	INTBIG which, aindex, type;
	REGISTER INTBIG search, language;
	char *varname, *name;
	VARIABLE retvar;

	DiaSetControl(13, 0);
	DiaSetText(14, "");
	DiaNoEditControl(14);
	DiaDimItem(15);
	DiaDimItem(31);
	DiaDimItem(18);
	DiaDimItem(20);
	DiaDimItem(21);   DiaDimItem(22);
	DiaDimItem(23);
	DiaDimItem(16);   DiaDimItem(34);   DiaDimItem(35);
	which = DiaGetCurLine(12);
	if (which < 0) return;
	varname = DiaGetScrollLine(12, which);
	search = initobjlist(us_varaddr, us_vartype, 0);
	for(;;)
	{
		name = nextobjectlist(&us_variablesvar, search);
		if (name == 0) break;
		if (strcmp(name, varname) == 0) break;
	}
	if (name == 0) return;

	type = us_variablesvar->type&VTYPE;
	if ((us_variablesvar->type&VCREF) == 0)
	{
		DiaUnDimItem(20);   DiaUnDimItem(21);   DiaUnDimItem(22);
		DiaUnDimItem(31);
	}
	if ((us_variablesvar->type&VCANTSET) == 0) DiaUnDimItem(18);
	DiaSetText(27, us_variabletypename(type));
	if ((us_variablesvar->type&VDISPLAY) != 0) DiaSetControl(21, 1); else
		DiaSetControl(21, 0);
	if ((us_variablesvar->type&VDONTSAVE) != 0) DiaSetControl(22, 1); else
		DiaSetControl(22, 0);
	language = us_variablesvar->type & (VCODE1|VCODE2);
	switch (language)
	{
		case 0:     DiaSetPopupEntry(31, 0);   break;
		case VTCL:  DiaSetPopupEntry(31, 1);   break;
		case VLISP: DiaSetPopupEntry(31, 2);   break;
		case VJAVA: DiaSetPopupEntry(31, 3);   break;
	}
	if ((us_variablesvar->type&VISARRAY) != 0)
	{
		DiaSetControl(15, 1);
		DiaUnDimItem(16);   DiaUnDimItem(34);   DiaUnDimItem(35);
		DiaSetText(17, "0");
		DiaEditControl(17);
		aindex = 0;
	} else
	{
		DiaSetControl(15, 0);
		DiaSetText(17, "");
		DiaNoEditControl(17);
		aindex = -1;
	}
	DiaSetText(19, describevariable(us_variablesvar, (INTSML)aindex, -1));
	if (language != 0)
	{
		retvar.key = us_variablesvar->key;
		retvar.type = us_variablesvar->type & ~(VCODE1|VCODE2);
		retvar.textdescript = us_variablesvar->textdescript;
		if (doquerry((char *)us_variablesvar->addr, language, retvar.type, &retvar.addr) != 0)
			retvar.addr = 0;
		DiaSetText(33, describevariable(&retvar, (INTSML)aindex, -1));
		DiaSetText(32, "Evaluation:");
	} else
	{
		DiaSetText(33, "");
		DiaSetText(32, "");
	}
	if (type != VUNKNOWN && type != VINTEGER && type != VADDRESS && type != VCHAR &&
		type != VSTRING && type != VFLOAT && type != VDOUBLE && type != VFRACT && type != VSHORT)
			DiaUnDimItem(23);
}

/****************************** VIEW CREATION DIALOG ******************************/

/* New view */
DIALOGITEM us_newviewdialogitems[] =
{
 /*  1 */ {0, {64,232,88,304}, BUTTON, "OK"},
 /*  2 */ {0, {64,16,88,88}, BUTTON, "Cancel"},
 /*  3 */ {0, {8,8,24,145}, MESSAGE, "New view name:"},
 /*  4 */ {0, {32,8,48,145}, MESSAGE, "View abbreviation:"},
 /*  5 */ {0, {8,148,24,304}, EDITTEXT, ""},
 /*  6 */ {0, {32,148,48,304}, EDITTEXT, ""},
 /*  7 */ {0, {68,104,84,213}, CHECK, "Textual View"}
};
DIALOG us_newviewdialog = {{50,75,154,391}, "New View", 7, us_newviewdialogitems};

/*
 * special case for the "new view" dialog
 * New view name     = 5 (edit text)
 * View abbreviation = 6 (edit text)
 * Textual view      = 7 (check)
 */
INTSML us_newviewdlog(char *paramstart[])
{
	INTBIG itemHit;

	/* display the port dialog box */
	if (DiaInitDialog(&us_newviewdialog) != 0) return(0);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == OK && DiaValidEntry(5) && DiaValidEntry(6)) break;
		if (itemHit == 7)
		{
			DiaSetControl(itemHit, 1-DiaGetControl(itemHit));
			continue;
		}
	}

	paramstart[0] = "";
	if (itemHit != CANCEL)
	{
		(void)initinfstr();
		(void)addstringtoinfstr(DiaGetText(5));
		(void)addtoinfstr(' ');
		(void)addstringtoinfstr(DiaGetText(6));
		if (DiaGetControl(7) != 0) (void)addstringtoinfstr(" text");
		paramstart[0] = returninfstr();
	}
	DiaDoneDialog();
	return(1);
}

/****************************** VISIBLE LAYERS DIALOG ******************************/

/* Layer visibility */
DIALOGITEM us_visiblelaydialogitems[] =
{
 /*  1 */ {0, {168,328,192,400}, BUTTON, "OK"},
 /*  2 */ {0, {168,232,192,304}, BUTTON, "Cancel"},
 /*  3 */ {0, {48,224,64,412}, MESSAGE, "Marked layers are visible."},
 /*  4 */ {0, {32,8,208,222}, SCROLL, ""},
 /*  5 */ {0, {32,224,48,412}, MESSAGE, "Click to change visibility."},
 /*  6 */ {0, {88,232,112,404}, BUTTON, "Make all layers visible"},
 /*  7 */ {0, {120,232,144,404}, BUTTON, "Make all layers invisible"},
 /*  8 */ {0, {8,8,24,82}, MESSAGE, "Layer set:"},
 /*  9 */ {0, {8,88,24,222}, POPUP, "Electric Layers"}
};
DIALOG us_visiblelaydialog = {{50,75,267,496}, "Layer Visibility", 9, us_visiblelaydialogitems};

/* special case for the "visiblelayers" dialog
 * Layers             = 4 (user item)
 * Make all visible   = 6 (button)
 * Make all invisible = 7 (button)
 */
#define LAYERSELEC 0		/* Electric layers */
#define LAYERSDXF  1		/* DXF layers */
#define LAYERSGDS  2		/* GDS layers */

void us_vislayerloadset(void);

void us_vislayerloadset(void)
{
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var, *cvar;
	REGISTER INTBIG i, layerset;
	REGISTER char *lname, *entry;
	char lbuff[100], lnum[20];

	DiaLoadTextDialog(4, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, 0);
	layerset = DiaGetPopupEntry(9);
	switch (layerset)
	{
		case LAYERSELEC:
			for(i=0; i<el_curtech->layercount; i++)
			{
				(void)initinfstr();
				if ((el_curtech->layers[i]->colstyle&INVISIBLE) != 0) (void)addtoinfstr(' '); else
					(void)addtoinfstr('>');
				(void)addtoinfstr(' ');
				(void)addstringtoinfstr(layername(el_curtech, (INTSML)i));
				DiaStuffLine(4, returninfstr());
			}
			break;
		case LAYERSDXF:
		case LAYERSGDS:
			np = getcurfacet();
			if (np == NONODEPROTO) break;
			for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			{
				if (layerset == LAYERSDXF)
				{
					var = getval((INTBIG)ni, VNODEINST, VSTRING, "IO_dxf_layer");
					if (var == NOVARIABLE) continue;
					lname = (char *)var->addr;
				} else
				{
					var = getval((INTBIG)ni, VNODEINST, VINTEGER, "IO_gds_layer");
					if (var == NOVARIABLE) continue;
					sprintf(lnum, "%ld", var->addr);
					lname = lnum;
				}
				for(i=0; ; i++)
				{
					entry = DiaGetScrollLine(4, i);
					if (*entry == 0) break;
					if (namesame(&entry[2], lname) == 0) break;
				}
				if (*entry == 0)
				{
					cvar = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, art_colorkey);
					if (cvar != NOVARIABLE && cvar->addr == 0)
						sprintf(lbuff, "  %s", lname); else
							sprintf(lbuff, "> %s", lname);
					DiaStuffLine(4, lbuff);
				}
			}
			break;
	}
	DiaSelectLine(4, -1);
}

INTSML us_visiblelayersdlog(char *prompt, char *paramstart[])
{
	INTBIG itemHit, i, changes, val, layerset;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var;
	WINDOWPART *w;
	char *msg, *entry, *layersets[3], lnum[20], *lname;

	if (us_needwindow()) return(0);

	/* display the visibility dialog box */
	if (DiaInitDialog(&us_visiblelaydialog) != 0) return(0);
	layersets[LAYERSELEC] = "Electric layers";
	layersets[LAYERSDXF] = "DXF layers";
	layersets[LAYERSGDS] = "GDS layers";
	DiaSetPopup(9, 3, layersets);
	if (namesame(prompt, "DXF") == 0) DiaSetPopupEntry(9, LAYERSDXF); else
		if (namesame(prompt, "GDS") == 0) DiaSetPopupEntry(9, LAYERSGDS);
	DiaInitTextDialog(4, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1,
		SCSELMOUSE|SCREPORT);
	us_vislayerloadset();

	/* loop until done */
	changes = 0;
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == OK || itemHit == CANCEL) break;
		if (itemHit == 4)
		{
			/* toggle layer visibility for this layer */
			i = DiaGetCurLine(4);
			msg = DiaGetScrollLine(4, i);
			if (*msg == ' ') *msg = '>'; else *msg = ' ';
			DiaSetScrollLine(4, i, msg);
			DiaSelectLine(4, -1);
			changes++;
			continue;
		}
		if (itemHit == 6 || itemHit == 7)
		{
			/* set visibility for all layers */
			for(i=0; ; i++)
			{
				msg = DiaGetScrollLine(4, i);
				if (*msg == 0) break;
				if (itemHit == 6) *msg = '>'; else *msg = ' ';
				DiaSetScrollLine(4, i, msg);
			}
			changes++;
			continue;
		}
		if (itemHit == 9)
		{
			/* construct new layer set */
			us_vislayerloadset();
			changes = 0;
			continue;
		}
	}

	if (itemHit != CANCEL && changes != 0)
	{
		layerset = DiaGetPopupEntry(9);
		switch (layerset)
		{
			case LAYERSELEC:
				/* save highlighting */
				us_pushhighlight();

				/* start of change to all display windows */
				for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
					if ((w->state&WINDOWTYPE) == DISPWINDOW ||
						(w->state&WINDOWTYPE) == DISP3DWINDOW)
							startobjectchange((INTBIG)w, VWINDOWPART);

				/* change the layer visibility */
				for(i=0; i<el_curtech->layercount; i++)
				{
					msg = DiaGetScrollLine(4, i);
					val = el_curtech->layers[i]->colstyle;
					if ((val&INVISIBLE) == 0 && *msg == ' ')
					{
						(void)setval((INTBIG)el_curtech->layers[i], VGRAPHICS,
							"colstyle", val | INVISIBLE, VSHORT);
					} else if ((val&INVISIBLE) != 0 && *msg != ' ')
					{
						(void)setval((INTBIG)el_curtech->layers[i], VGRAPHICS,
							"colstyle", val & ~INVISIBLE, VSHORT);
					}
				}

				/* end of change to all display windows (redraws) */
				for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
					if ((w->state&WINDOWTYPE) == DISPWINDOW ||
						(w->state&WINDOWTYPE) == DISP3DWINDOW)
							endobjectchange((INTBIG)w, VWINDOWPART);

				/* restore highlighting */
				(void)us_pophighlight(0);
				break;
			case LAYERSDXF:
			case LAYERSGDS:
				np = getcurfacet();
				if (np == NONODEPROTO) break;
				for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
				{
					if (layerset == LAYERSDXF)
					{
						var = getval((INTBIG)ni, VNODEINST, VSTRING, "IO_dxf_layer");
						if (var == NOVARIABLE) continue;
						lname = (char *)var->addr;
					} else
					{
						var = getval((INTBIG)ni, VNODEINST, VINTEGER, "IO_gds_layer");
						if (var == NOVARIABLE) continue;
						sprintf(lnum, "%ld", var->addr);
						lname = lnum;
					}
					for(i=0; ; i++)
					{
						entry = DiaGetScrollLine(4, i);
						if (*entry == 0) break;
						if (namesame(&entry[2], lname) == 0) break;
					}
					if (*entry != 0)
					{
						startobjectchange((INTBIG)ni, VNODEINST);
						if (entry[0] == '>')
						{
							/* make node visible */
							var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER, art_colorkey);
							if (var != NOVARIABLE && var->addr == 0)
								delvalkey((INTBIG)ni, VNODEINST, art_colorkey);
						} else
						{
							/* make node invisible */
							setvalkey((INTBIG)ni, VNODEINST, art_colorkey, 0, VINTEGER);
						}
						endobjectchange((INTBIG)ni, VNODEINST);
					}
				}
				break;
		}
	}
	DiaDoneDialog();
	paramstart[0] = "x";
	return(1);
}

/****************************** WINDOW VIEW DIALOG ******************************/

/* Window Views */
DIALOGITEM us_windowviewdialogitems[] =
{
 /*  1 */ {0, {256,56,280,166}, BUTTON, "Restore View"},
 /*  2 */ {0, {216,8,240,72}, BUTTON, "Done"},
 /*  3 */ {0, {32,8,208,234}, SCROLL, ""},
 /*  4 */ {0, {8,96,24,229}, EDITTEXT, ""},
 /*  5 */ {0, {216,120,240,230}, BUTTON, "Save This View"},
 /*  6 */ {0, {8,8,24,90}, MESSAGE, "View name:"}
};
DIALOG us_windowviewdialog = {{50,75,342,318}, "Window Views", 6, us_windowviewdialogitems};

/*
 * special case for the "Window View" dialog
 * restore view     = 1 (button)
 * save view        = 5 (button)
 * view list        = 3 (scroll)
 * new view name    = 4 (edit text)
 */
INTSML us_windowviewdlog(char *paramstart[])
{
	INTBIG itemHit, i;
	REGISTER VARIABLE *var;
	char *par[3], *pt;

	/* display the window view dialog box */
	if (DiaInitDialog(&us_windowviewdialog) != 0) return(0);
	DiaInitTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, 0,
		SCSELMOUSE | SCSELKEY | SCDOUBLEQUIT);
	for(i=0; i<us_aid->numvar; i++)
	{
		var = &us_aid->firstvar[i];
		pt = makename(var->key);
		if (namesamen(pt, "USER_windowview_", 16) == 0) DiaStuffLine(3, &pt[16]);
	}
	DiaSelectLine(3, -1);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == CANCEL) break;
		if (itemHit == 1)
		{
			/* restore selected view */
			par[0] = "name";
			i = DiaGetCurLine(3);
			if (i < 0) continue;
			pt = DiaGetScrollLine(3, i);
			(void)allocstring(&par[1], pt, el_tempcluster);
			break;
		}

		if (itemHit == 5)
		{
			/* save selected view */
			if (DiaValidEntry(4) == 0) continue;
			par[0] = "save";
			par[1] = DiaGetText(4);
			us_window(2, par);
			DiaLoadTextDialog(3, DiaNullDlogList, DiaNullDlogItem, DiaNullDlogDone, -1);
			for(i=0; i<us_aid->numvar; i++)
			{
				var = &us_aid->firstvar[i];
				pt = makename(var->key);
				if (namesamen(pt, "USER_windowview_", 16) == 0) DiaStuffLine(3, &pt[16]);
			}
			DiaSelectLine(3, -1);
			continue;
		}
	}

	DiaDoneDialog();
	if (itemHit == 1)
	{
		us_window(2, par);
		efree(par[1]);
	}
	return(0);
}
