/**
*** Program jdbcMysqlStmt.java
***    in product twz1jdbcForMysql, 
***    Copyright 1997, 1998 by Terrence W. Zellers.
***   
***  All rights explicitly reserved.
***
***  See file "LICENSE" in this package for conditions of use.
**/

package twz1.jdbc.mysql;
import twz1.jdbc.mysql.jdbcMysqlMutex;
import java.sql.*;
import java.util.StringTokenizer;
import java.util.Vector;

public class jdbcMysqlStmt implements Statement
{

/** Point back to connection. */
jdbcMysqlConnex cx;

/** My object */
int myOID;

/** executed string. */
byte[] lastExecuted;

/** Command word */
String commandWord;

/** Command value */
int commandValue;

/** Maximum number of rows in result */
int maxRows;

/** Maximum field size */
int maxFieldSize;

/** Input bag from connex */
jdbcMysqlBag inBag;

/** Output bag from connex */
jdbcMysqlBag outBag;

/** Statement is available */
boolean open; 

/** Statement status */
int status;

/** Practice safe threading. */
jdbcMysqlMutex guard;

/** timeout on mutex */
int lTimeout;

/** last insert id on an insert */
long lastInsertID;

/** Query result */
jdbcMysqlResult qResult;

/** should rsmd queries throw type exceptions ? */
boolean rsmdXcept;

/** Do we lock for ResultSet retrievals ? */
boolean rsLock;

/** Do we attempt reconnection. */
boolean autoReX;

static final int QTYPE_EU  = 1;
static final int QTYPE_EQ  = 2;
static final int QTYPE_EX  = 3;

static final int C_CLOSE               = 1;
static final int C_GETMAXFIELDSIZE     = 2;
static final int C_SETMAXFIELDSIZE     = 3;
static final int C_GETMAXROWS          = 4;
static final int C_SETMAXROWS          = 5;
static final int C_SETESCAPEPROCESSING = 6;
static final int C_GETQUERYTIMEOUT     = 7;
static final int C_SETQUERYTIMEOUT     = 8;
static final int C_CANCEL              = 9;
static final int C_GETWARNINGS         = 10;
static final int C_CLEARWARNINGS       = 11;
static final int C_SETCURSORNAME       = 12;
static final int C_GETUPDATECOUNT      = 13;
static final int C_GETMORERESULTS      = 14;
static final int C_SETNULL             = 15; 
static final int C_SETBOOLEAN          = 16;
static final int C_SETBYTE             = 17;
static final int C_SETSHORT            = 18;
static final int C_SETINT              = 19;
static final int C_SETLONG             = 20;
static final int C_SETFLOAT            = 21;
static final int C_SETDOUBLE           = 22;
static final int C_SETSTRING           = 23;
static final int C_SETBYTES            = 24;
static final int C_SETDATE             = 25;
static final int C_SETTIME             = 26;
static final int C_SETTIMESTAMP        = 27;
static final int C_SETASCIISTREAM      = 28;
static final int C_SETBINARYSTREAM     = 29;
static final int C_SETUNICODESTREAM    = 30;
static final int C_SETBIGDECIMAL       = 31;
static final int C_SETOBJECT           = 32;
static final int C_CLEARPARAMETERS     = 33;

static final String[] apis = {
                             "",
                             "close()",
                             "getMaxFieldSize()",
                             "setMaxFieldSize()",
                             "getMaxRows()",
                             "setMaxRows()",
                             "setEscapeProcessing()",
                             "getQueryTimeout()",
                             "setQueryTimeout()",
                             "cancel()",
                             "getWarnings()",
                             "clearWarnings()",
                             "setCursorName()",
                             "getUpdateCount()",
                             "getMoreResults()",
                             "setNull()",
                             "setBoolean()",
                             "setByte()",  
                             "setShort()", 
                             "setInt()",
                             "setLong()",
			     "setFloat()",
                             "setDouble()",
                             "setString()",
                             "setBytes()",
                             "setDate()",
                             "setTime()",
                             "setTimestamp()",
                             "setAsciiStream()",
                             "setBinaryStream()",
                             "setUnicodeStream()",
                             "setBigDecimal()",
                             "setObject()",
                             "clearParameters()",
                             };

static final String[] errs = {
    "E0600 No argument to execute.",
    "E0601 Invalid command to executeUpdate() -",
    "E0602 Error in executeUpdate() --",
    "E0603 executeUpdate() error --",
    "E0604 Bizarre status code from executeUpdate() - ",
    "E0605 Invalid command to executeQuery() -",
    "E0606 executeQuery() error --",
    "E0607 Thread guard timed out.", 
    "E0608 Unexpected result from mutex.",
    "E0609 Error in executeUpdate()",
    "E0610 Error in executeQuery()",
    "E0611 Error from db for executeQuery() -",
    "E0612 Unexpected status in executeQuery() - ",
    "E0613 Invalid command to execute() -",
    "E0614 execute() error --",
    "E0615 Error handling API method --",
    "E0616 Error: statement is closed.",
    "E0617 Invalid value specified.",
    "E0618 Not supported.",
    "P0619 Progerr sAction called with bad option.",
    "E0620 Invalid property value",
    "E0621 Prepared statement may not begin with a \"?\".",
    "E0622 Prepared statement requires a statement to prepare.",
    "E0623 Error in Prepared Statement API method --",   
    "E0624 Parameter value is out of range.", 
    "P0625 Prepared Statement command value error. -- ",
    "E0626 Error setting Parameters for execute."
                             };

jdbcMysqlStmt(jdbcMysqlConnex connection) throws SQLException
    {
    myOID = jdbcMysqlBase.getOID();
    this.cx = connection;
    this.lastExecuted = null;
    this.maxRows = Integer.MAX_VALUE;
    this.maxFieldSize = 65535;
    this.inBag = cx.getInBag();
    this.outBag = cx.getOutBag();
    this.open = true;
    this.guard = new jdbcMysqlMutex();
    this.lTimeout = cx.connexTimeout;
    this.qResult = new jdbcMysqlResult(this, this.cx);
    this.lastInsertID = -1;
 
    String test;
    int testnum;

    test = cx.getProperty("maxRows");
    testnum = jdbcMysqlBase.intValue(test, -1, -2);
    if(testnum == 0) maxRows = Integer.MAX_VALUE;
    if(testnum == -2) errHandlerM(20, "maxRows");
    if(testnum > 0) this.maxRows = testnum;

    test = cx.getProperty("maxField");
    testnum = jdbcMysqlBase.intValue(test, -1, -2);
    if(testnum == -2) errHandlerM(20, "maxField");
    if(testnum > 0) this.maxFieldSize = testnum;

    test = cx.getProperty("rsmdXcept");
    if(jdbcMysqlBase.boolValue(test, 0, 0) == 1) rsmdXcept = true;

    test = cx.getProperty("RSLock");
    rsLock = jdbcMysqlBase.boolValue(test, 0, 0) == 1 ? true : false ;

    test = cx.getProperty("autoReX");
    autoReX = jdbcMysqlBase.boolValue(test, 1, 0) == 1 ? true : false;
   
    }

/*===================================================================+
||                        executeUpdate()                           ||
+===================================================================*/

public int executeUpdate(String sql) throws SQLException
    {
    int reqID = jdbcMysqlBase.getOID();
    boolean lockset = false;
    int rc = -1;
    try {
        lockset = lock(true, reqID, lTimeout); 
        qResult.clear(); 
        getCommandValue(sql);
        if(commandValue < 100 || commandValue > 199)
                errHandlerM(1, commandWord);
        mExecUpdate(sql.getBytes());
        rc= qResult.getRows();
        lockset = lock(false, reqID, lTimeout);
        }
    catch(Exception e) 
        {
        if(lockset) lock(false, reqID, lTimeout);
        errHandlerE(9, e);
        }
    return rc;
    }

void mExecUpdate(byte[] sql) throws SQLException
    {
    int status = -1;
    int rc  = -1;
    String message = null;
    if(commandValue == jdbcMysqlBase.COM_COMMENT)
        {
        qResult.getResult(QTYPE_EU, commandValue);
        return;
        }
    boolean lockset = false;
    try {
        lockset = cx.lock(true, myOID, lTimeout);    
        mxEU(sql);
        lockset = cx.lock(false, myOID,  1);
        }
    catch(Exception e)
        {
        boolean redone =  false;
        if(cx.broken && autoReX)
	    {
	    try { 
                cx.xOpen(); 
                mxEU(sql);
                redone = true;
                }
            catch(Exception e1){}
            }
        if(lockset){ cx.lock(false, myOID, 2); }
        if(!redone){ errHandlerE(2, e); }
        }
 
    status  = qResult.getStatus();

    if(status == 255) 
        {
        message = qResult.getMessage();
        errHandlerM(3, String.valueOf(rc) + " " + message);
        }
    if(status != 0) errHandlerM(4, String.valueOf(status));
    return;
    }


private void mxEU(byte[] sql) throws SQLException
    {
    outBag.newWrite(0);
    outBag.iToB(jdbcMysqlBase.COM_QUERY, 1);
    outBag.putZT(sql);
    outBag.write();
    qResult.getResult(QTYPE_EU, commandValue);
    }

/*===================================================================+
||                        executeQuery()                            ||
+===================================================================*/

public ResultSet executeQuery(String sql) throws SQLException
    { 
    int reqID = jdbcMysqlBase.getOID();
    boolean lockset = false;
    ResultSet rs = null;
    jdbcMysqlRSet jrs = null;
    try {
        lockset = lock(true, reqID, lTimeout); 
        getCommandValue(sql);
        if(commandValue < 200 || commandValue > 399)
                errHandlerM(5, commandWord);
        qResult.clear(); 
        rs = mExecQuery(sql.getBytes()); 
        if(rs != null){ jrs = (jdbcMysqlRSet) rs; jrs.retrieved(true);}
        lockset = lock(false, reqID, lTimeout);
        }
    catch(Exception e) 
        {
        if(lockset) lock(false, reqID, lTimeout);
        errHandlerE(10, e);
        }
    return rs;
    }

ResultSet mExecQuery(byte[] sql) throws SQLException
    {
    int status = -1;
    jdbcMysqlRSet rs = null;
    int rc  = -1;
    String message = null;
    boolean lockset = false;
    try {
        lockset = cx.lock(true, myOID, lTimeout);    
        rs = mxEQ(sql); 
        if(qResult.cacheMode != qResult.CM_NONE)
	    { lockset = cx.lock(false, myOID, 3); }
        }
    catch(Exception e)
        {
        if(cx.broken)
	    {
	    try {
                cx.xOpen();
                rs = mxEQ(sql);
                if(qResult.cacheMode != qResult.CM_NONE)
	            { lockset = cx.lock(false, myOID, 3);  }
	        }
            catch(Exception qe){}
            }
        if(rs == null)
	    { 
            if(lockset) cx.lock(false, myOID, 4);
            errHandlerE(6, e);
            }
        }
    return rs;
    }


private jdbcMysqlRSet mxEQ(byte[] sql) throws SQLException
    {
    jdbcMysqlRSet rs = null;
    outBag.newWrite(0);
    outBag.iToB(jdbcMysqlBase.COM_QUERY, 1);
    outBag.putZT(sql);
    outBag.write();
    qResult.getResult(QTYPE_EQ, commandValue);
    rs = new jdbcMysqlRSet(qResult);
    return rs;
    }

/*===================================================================+
||                       execute()                                  ||
+===================================================================*/

public boolean execute(String sql) throws SQLException 
    {
    int reqID = jdbcMysqlBase.getOID();
    boolean lockset = false;
    ResultSet rs = null;
    try {
        lockset = lock(true, reqID, lTimeout); 
        qResult.clear(); 
        getCommandValue(sql);
        if(commandValue > 200 && commandValue < 400)
                rs = mExecQuery(sql.getBytes()); 
        else if(commandValue > 100 && commandValue < 200)
                mExecUpdate(sql.getBytes()); 
        else errHandlerM(13, "");
        lockset = lock(false, reqID, lTimeout); 
        }
    catch(Exception e)
        {
        if(lockset)lock(false, reqID, lTimeout);
        errHandlerE(14, e);
        }
    return rs != null;
    }

/*-------------------------------------------------------------------+
|                       Create a pseudo result                       |
+-------------------------------------------------------------------*/

ResultSet pseudoResult(Vector data, RSMd rsmd) 
        throws SQLException
    {
    ResultSet rs;
    qResult.pseudoResult(data, rsmd);
    rs = new jdbcMysqlRSet(qResult);
    return rs;
    }

/*===================================================================+
||                     Other API methods                            ||
+===================================================================*/

/** The API is a little vague about this.  As implemented in 
*** invalidates results from any previous inquiries and renders
*** the statement unusable for future inquiries.  From a technical
*** standpoint there is no reason why the statement should not
*** be reusable, but I'll let it stand as is unless shown to be
*** wrong.
**/
public void close() throws SQLException
    { sAction(C_CLOSE, 0, null); }

/** See the API.  While showing maximum field size returnable
*** to the caller, there are dependencies on the communication
*** protocol and its packages for an upper bound which may not 
*** be reflected here.
**/
public int getMaxFieldSize() throws SQLException 
    { return sAction(C_GETMAXFIELDSIZE, 0, null); }

/** See the API.  While setting the maximum field size returnable
*** to the caller, there are dependencies on the communication
*** protocol and its packages for an upper bound which may not 
*** be reflected here.
**/
public void setMaxFieldSize(int s) throws SQLException 
    { sAction(C_SETMAXFIELDSIZE, s, null); }

/** See API. */
public int getMaxRows() throws SQLException 
    { return sAction(C_GETMAXROWS, 0, null); }

/** See API */
public void setMaxRows(int s) throws SQLException 
    { sAction(C_SETMAXROWS, s, null); }

/** See API.  Throws an exception as MySQL's escape processing 
*** is implicit and invariant.  At some future time (if there
*** is demand) this driver may be modified to do some escpae
*** processing, but as of now it does not.
**/
public void setEscapeProcessing(boolean t) throws SQLException 
    { sAction(C_SETESCAPEPROCESSING, 0, null);}

/** Varies somewhat from the API as the timeout value in this
*** implementation refers to the time the mutex lock on the
*** referent statement times out, rather than total duration of
*** execution.  IE the time before a "waiting" request times out:
*** the reqirement that a query to the connection be completed 
*** mandates this variation.
**/
public int getQueryTimeout() throws SQLException 
    { return sAction(C_GETQUERYTIMEOUT, 0, null); }

/** Varies somewhat from the API as the timeout value in this
*** implementation refers to the time the mutex lock on the
*** referent statement times out, rather than total duration of
*** execution.  IE the time before a "waiting" request times out:
*** the reqirement that a query to the connection be completed 
*** mandates this variation.
**/
public void setQueryTimeout(int t) throws SQLException 
    { sAction(C_SETQUERYTIMEOUT, t, null); }

/** See API.  In actuality this varies from the API because 
*** MySQL does not support multiple queries on a single connection.
*** If another thread is executing this statement, cancel() will be
*** blocked until that request clears, at which time the statement
*** will be effectively closed.
**/
public void cancel() throws SQLException 
    { sAction(C_CANCEL, 0, null);}

/** Returns null.  In the near future it will return the top of 
*** the error stack. 
**/
public SQLWarning getWarnings() throws SQLException 
    { sAction(C_GETWARNINGS, 0, null); return null; }


/** Does nothing now, in the future it will clear the warning
*** chain.           
**/
public void clearWarnings() throws SQLException 
    { sAction(C_CLEARWARNINGS, 0, null); }

/** See API: this throws an exception as cursors are not
*** supported in MySQL.
**/
public void setCursorName(String s) throws SQLException
    { sAction(C_SETCURSORNAME,0, null); }

/** See the API.  There may be exceptions to the API use of
*** this method following the execution of private SQL methods.
**/

public int getUpdateCount() throws SQLException 
    { return sAction(C_GETUPDATECOUNT, 0, null); }


/** See the API.  This will return true only if a query has
*** been executed by an execute() and the RS not yet retrieved.
**/
public boolean getMoreResults() throws SQLException 
    { return sAction(C_GETMORERESULTS, 0, null) == 1; }

/** See API.   For MySQL there should be only one RS generated 
*** per query.
**/
public ResultSet getResultSet() throws SQLException
    {
    int reqID = jdbcMysqlBase.getOID();
    boolean lockset = false;
    ResultSet rs = null;
    try {
        lockset = lock(true, reqID, lTimeout);
        if(!open) errHandlerM(16, "getResultSet()");
        rs = qResult.getUnretrievedRS();       
        lockset = lock(false, reqID, lTimeout); 
        }
    catch(Exception e)
        {
        if(lockset)lock(false, reqID, lTimeout);
        errHandlerE2(15, "getResultSet()",  e);
        }
    return rs;
    }

/*-------------------------------------------------------------------+
|                   do the work for most of the API's                |
+-------------------------------------------------------------------*/

private int sAction(int option, int vi, String vs) throws SQLException
    {
    int reqID = jdbcMysqlBase.getOID();
    boolean lockset = false;
    int res = -1;
    try {
        lockset = lock(true, reqID, lTimeout);
        if(!open) errHandlerM(16, apis[option]);
        switch(option)
            { 
            case C_CLOSE:
                qResult.clear(); 
                open = false;
                break;
            case C_GETMAXFIELDSIZE:
                res = maxFieldSize;
                break;                 
            case C_SETMAXFIELDSIZE:
                if(vi < 10) errHandlerM(17, apis[option]);
                maxFieldSize = vi;
                break;
            case C_GETMAXROWS:
                res = maxRows;
                break;
            case C_SETMAXROWS:
                if(vi < 0)errHandlerM(17, apis[option]);
                maxRows = vi;
                if(vi == 0) maxRows = Integer.MAX_VALUE;
                break;
            case C_GETWARNINGS:
            case C_CLEARWARNINGS:
                break;
            case C_SETCURSORNAME:
            case C_SETESCAPEPROCESSING:
                errHandlerM(18, apis[option]);
                break;
            case C_GETQUERYTIMEOUT:
                res = lTimeout;
                break;
            case C_SETQUERYTIMEOUT:
                lTimeout = vi;
                break;
            case C_GETMORERESULTS:
                if(qResult.areUnretrieved())res = 1;
                break;
            case C_GETUPDATECOUNT:
                res = qResult.getRows();
                break;
            case C_CANCEL:
                qResult.clear(); 
                break;              
            default: errHandlerM(19, String.valueOf(option));
            }
        lockset = lock(false, reqID, lTimeout); 
        }
    catch(Exception e)
        {
        if(lockset)lock(false, reqID, lTimeout);
        errHandlerE2(15, apis[option],  e);
        }
    return res;
    }

/*-------------------------------------------------------------------+
|                   Retrieve the integer command value               |
+-------------------------------------------------------------------*/

void getCommandValue(String sql) throws SQLException
    {
    StringTokenizer st = new StringTokenizer(sql);
    if(!st.hasMoreTokens()) errHandlerM(0, "");
    commandWord = st.nextToken();
    commandValue = jdbcMysqlBase.getCommand(commandWord);
    }

/*-------------------------------------------------------------------+
|                   Safe threading.                                  |
+-------------------------------------------------------------------*/

/** Invoke a mutex to protect access to the statement. 
*** @param lockOnOff set or clear the mutex.
*** @param oid Object setting the lock.
*** @param timeout time (seconds) to wait before tossing cookies.
**/
boolean lock(boolean lockOnOff, int oid, int timeout)
         throws SQLException
     {
     int rc;
     rc = guard.synch(oid, lockOnOff, timeout);
     if(lockOnOff && rc == guard.R_LOCKED) { return true; }
     if(!lockOnOff && rc == guard.R_UNLOCKED) { return false;}
     if(rc == guard.R_TIMEOUT) { errHandlerM(7, ""); }
     errHandlerM(8, String.valueOf(rc) );  
     return false;        
     }

/*===================================================================+
||                        Error handlers                            ||
+===================================================================*/

void errHandlerM(int n, String m) throws SQLException
    {
    String o = "\n" + errs[n] + m;
    jdbcMysqlBase.errMessage(o);
    }
    
void errHandlerE(int n, Exception e) throws SQLException
    {
    String o = "\n" +errs[n] + jdbcMysqlBase.eMessage(e);
    jdbcMysqlBase.errMessage(o);
    }

void errHandlerE2(int n, String s, Exception e) throws SQLException
    {
    String o = "\n" + errs[n] + " " + s + jdbcMysqlBase.eMessage(e);
    jdbcMysqlBase.errMessage(o);
    }



/*-------------------------------------------------------------------+
|               Internal equivalent of public methods                |
+-------------------------------------------------------------------*/

int iGetMaxFieldSize() { return maxFieldSize; }

}
