# copyright (C) 1997-2002 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: mystatus.tcl,v 2.29 2002/01/19 11:53:05 jfontain Exp $}


package provide mystatus [lindex {$Revision: 2.29 $} 1]
if {[lsearch -exact $auto_path /usr/lib]<0} {                         ;# in case Tcl/Tk is somewhere else than in the /usr hierarchy
    lappend auto_path /usr/lib
}
package require miscellaneous 1


namespace eval mystatus {}
source mystatus.dat

namespace eval mystatus {

    array set data {
        updates 0
        0,label display 0,type ascii 0,message {per second values are calculated for the last poll period using the absolute values from the server, and only for variables that make sense (otherwise a ? is displayed in the column)}
        0,0 absolute 1,0 {per second}
        pollTimes {10 5 20 30 60 120 300}
        switches {--dsn 1 --host 1 --password 1 --port 1 --user 1}
    }
    set file [open mystatus.htm]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

    proc initialize {optionsName} {
        upvar $optionsName options
        variable odbc
        variable connection
        variable data
        variable help
        variable type
        variable nameColumn

        catch {set user $::env(USER)}                                                                                  ;# by default
        catch {set user $::env(LOGNAME)}                                                          ;# more common in UNIX and even NT
        catch {set user $options(--user)}
        if {[info exists options(--dsn)]} {                                               ;# ODBC mode id data source name specified
            set odbc 1
            package require tclodbc 2                               ;# so that it works with both UNIX 2.2.1 et Windows 2.3 versions
            if {[info exists options(--host)]||[info exists options(--port)]} {
                error {--host and --port options incompatible with ODBC mode}
            }
            set arguments [list $options(--dsn)]
            catch {lappend arguments $user}
            catch {lappend arguments $options(--password)}
            set connection [eval database odbc $arguments]                                           ;# use a unique connection name
            set data(identifier) mystatus($options(--dsn))
            scan [lindex [lindex [$connection {show variables like 'version'}] 0] 1] %u.%u.%u major minor subMinor
        } else {
            set odbc 0
            package require mysqltcl 2
            set arguments {}
            catch {lappend arguments -host $options(--host)}
            catch {lappend arguments -user $user}
            catch {lappend arguments -password $options(--password)}
            catch {lappend arguments -port $options(--port)}
            set connection [eval mysqlconnect $arguments]
            set host [mysqlinfo $connection host]
            set data(identifier) mystatus($host)
            scan [lindex [mysqlsel $connection {show variables like 'version'} -flatlist] 1] %u.%u.%u major minor subMinor
        }
        switch $major {
            3 {
                if {$minor!=23} {error "cannot monitor a server version 3.$minor"}
                if {$subMinor<47} {error {cannot monitor a server below version 3.23.47 in the 3.23 series}}
            }
            4 {
                if {$minor!=0} {error "cannot monitor a server version 4.$minor"}
                if {$subMinor<1} {error {cannot monitor a server below version 4.0.1 in the 4.0 series}}
            }
            default {
                error "cannot monitor a server version $major"
            }
        }
        if {$odbc} {
            $connection statement $connection.query {show status}
            $connection.query execute
        } else {
            mysqlsel $connection {show status}
        }
        set columns(main) 0
        set columns(queries) 0
        set column 1
        while {1} {                                                                     ;# gather existing variables from the server
            if {$odbc} {
                set list [$connection.query fetch]
            } else {
                set list [mysqlnext $connection]
            }
            if {[llength $list]==0} break
            set variable [lindex $list 0]
            set nameColumn($variable) $column                                                            ;# remember variable column
            set data($column,label) $variable
            set value real                                                                                             ;# by default
            catch {set value $type($variable)}
            if {![string equal $value real]} {
                set data(1,$column) ?                                                                    ;# no per second value ever
            }
            set data($column,type) $value
            if {![info exists help($variable)]} {
                puts stderr "\"$variable\" variable is not recognized by mystatus: please contact author."
                set data($column,message) {not recognized: please contact author}
            } elseif {[string length $help($variable)]==0} {
                set data($column,message) {undocumented in the MySQL documentation}
            } else {
                set data($column,message) $help($variable)
                if {[string match *Bytes* $variable]} {                             ;# anything in bytes is transformed to kilobytes
                    append data($column,message) { (in kilobytes)}
                }
            }
            set data($column,anchor) left
            if {[string match Com_* $variable]} {
                lappend columns(queries) $column                                         ;# put queries counters in a separate table
            } else {
                lappend columns(main) $column
            }
            incr column
        }
        if {$odbc} {
            $connection.query drop
        }
        unset help type                                                                                          ;# save some memory
        set data(views) [list [list indices $columns(main) swap 1] [list indices $columns(queries) swap 1]]
    }

    proc update {} {
        variable odbc
        variable connection
        variable last
        variable data
        variable nameColumn

        set error 0
        if {$odbc} {
            set error [catch {
                $connection statement $connection.query {show status}
                $connection.query execute
            } message]
        } else {
            set error [catch {mysqlsel $connection {show status}} message]
        }
        if {$error} {                                                                                     ;# problem reaching server
            flashMessage "mystatus error: $message"
            catch {unset last}
            foreach {name column} [array get nameColumn] {
                if {$column==0} continue                                                                           ;# headers column
                set data(0,$column) ?
                if {[string equal $data($column,type) real]} {                                     ;# required for per second values
                    set data(1,$column) ?
                }
            }
        } else {
            set clock [expr {[clock clicks -milliseconds]/1000.0}]                                         ;# store clock in seconds
            catch {set period [expr {$clock-$last(clock)}]}
            set last(clock) $clock
            while {1} {
                if {$odbc} {
                    set list [$connection.query fetch]
                } else {
                    set list [mysqlnext $connection]
                }
                if {[llength $list]==0} break
                foreach {variable value} $list {}
                set column $nameColumn($variable)
                if {[string equal $variable Uptime]} {
                    set data(0,$column) [formattedTime $value]
                    set data(1,$column) {}
                } else {
                    if {[string equal $data($column,type) real]} {                                 ;# required for per second values
                        if {[string match *Bytes* $variable]} {                     ;# anything in bytes is transformed to kilobytes
                            append value .0                                                                      ;# convert to float
                            set value [expr {$value/1000.0}]
                            set data(0,$column) [format %.2f $value]
                        } else {
                            set data(0,$column) $value                               ;# display as is as it really may be an integer
                        }
                        if {[info exists period]} {
                            set data(1,$column) [format %.2f [expr {($value-$last($column))/$period}]]
                        } else {
                            set data(1,$column) ?
                        }
                        set last($column) $value
                    } else {
                        set data(0,$column) $value
                    }
                }
            }
        }
        if {$odbc} {
            catch {$connection.query drop}            ;# ignore connection errors at this point (they will be reported at next poll)
        }
        incr data(updates)
    }

    proc terminate {} {
        variable odbc
        variable connection

        if {$odbc} {
            catch {$connection disconnect}
        } else {
            catch {mysqlclose $connection}
        }
    }

}
