// $Id: gweMysql.java,v 1.13 1997/11/25 21:21:14 xzhu Exp $
//////////////////////////////////////////////////////////////////
// mysql JDBC Driver
//
// Version : 0.9.2a
//
// JDBC : 1.22
//
// LastModified: 1997/09/04 by X.K. ZHU
// LastModified: 16.03.1997
//
// Copyright (c) 1997 GWE Technologies Ltd. All rights reserved
// See http://gwe.co.uk/mysql/jdbc/copyright.html for full details

package gwe.sql;

import java.net.*;
import java.io.*;
import java.sql.*;
import java.util.Hashtable;
import gwe.sql.gweMysqlColumn;
import zxk.util.*;


// Class to read/write packets to the mysql server

class gweMysqlPacket implements DBugConstant
{
	DataInputStream sin;
	DataOutputStream sout;
	byte packetSequence;
	int bpos, packetLength;
	public final int HEADERLENGTH=4;
	public final int MAXBUFFER=655350;
	byte buffer[]=new byte[MAXBUFFER];
	//byte extraBuffer[] = new byte[MAXBUFFER];

	// Set up given an input and output datastream
	public gweMysqlPacket(DataInputStream i, DataOutputStream o) {
		sin=i;
		sout=o;
	}

  void clearReceive() 
    throws SQLException
  {
    if (zxk_debug)
      DBug.dbug.dbEnter("clearReceive()", "gweMysqlPacket", 49);
    
    try
    {
      int len = sin.available();
      if (len > 0) // sometime there are big extra bytes which
      {            // larger than the socket buffer.
        if (zxk_debug) 
          DBug.dbug.dbPrint(57, "cr_1", "extra characters length: " + len);
        //sin.read(extraBuffer, 0, len);
	sin.skipBytes(len);
      }
    } catch (IOException e)
    { throw new SQLException("gweMysql.clearReceive: " + e); }

    if (zxk_debug)
      DBug.dbug.dbReturn(312);    
  }

  void clearAllReceive() 
    throws SQLException
  {
    if (zxk_debug)
      DBug.dbug.dbEnter("clearAllReceive()", "gweMysqlPacket", 49);
    
    try
    {
      int len = sin.available();
      if (len > 0)
      {
        next();
	if (buffer[0] == (byte)0xff)
	{
	  clearReceive();
	  return;
	}
        while (!isLastDataPacket()) // sometime there are big extra bytes which
        {            // larger than the socket buffer.
          next();
	  if (buffer[0] == (byte)0xff)
	    break;
        }
      }
      clearReceive();
    } catch (IOException e)
    { throw new SQLException("gweMysql.clearAllReceive: " + e); }

    if (zxk_debug)
      DBug.dbug.dbReturn(312);    
  }

  // Set to -1; packetSequence gets ++'ed before sending
  void resetPacketSequence() 
  {
	packetSequence=-1;
  }
	
  // Read in next packet; get upset if need be
  void next() throws SQLException 
  {		
    if (zxk_debug)
      DBug.dbug.dbEnter("next()", "gweMysqlPacket", 321);

    try {
      int i;
      //byte[] b3 = new byte[3];
      byte b1;
      byte b2;
      byte b0;
			
      // [3: packetLength] [packetSequence]
      //for (i=0; i<3; i++) b3[i]=sin.readByte();
      b0 = sin.readByte();
      b1 = sin.readByte();
      b2 = sin.readByte();
      if (zxk_debug) 
        DBug.dbug.dbPrint(12, "n_1", "Packet head = " + ub(b0) + "," + ub(b1) + "," + ub(b2));
      packetLength=(int) (ub(b0)+(256*ub(b1))+(256*256*ub(b2)));
      packetSequence=sin.readByte();

      if (zxk_debug) 
      {
        DBug.dbug.dbPrint(12, "n_1", "Packet length = "+packetLength);
        DBug.dbug.dbPrint(12, "n_1", "Packet sequence = "+packetSequence);
      }
			
      // Get the data (block if necessary - thread higher up the us)
      sin.readFully(buffer, 0, packetLength);
      buffer[packetLength]=0;
      bpos=0;
			
      if (zxk_debug) 
      {
	for (int j=0; j<packetLength; j++) 
	{
          if (buffer[j]>30) 
	    DBug.dbug.dbPrint(12, "n_1", ""+j+" : "+ub(buffer[j])+"  : "+(char)buffer[j]);
          else 
	    DBug.dbug.dbPrint(12, "n_1", ""+j+" : "+ub(buffer[j])+"  : ");
	}
      }
    } catch (Exception e) 
    {
      throw new SQLException("gweMysql.next: "+e);
    }

    if (zxk_debug)
      DBug.dbug.dbReturn(312);    
  }

  // Return unsigned byte as int
	int ub(byte b) {
		return b<0?(int)(256+b):b;
	}

	// Where are we?
	int currentPos() {
		return bpos;
	}

	// Set position withtin packet data
	void setPos(int p) {
		bpos=p;
	}

	void decPos()
	{
	  bpos--;
	}

	byte readByte() {
		return buffer[bpos++];
	}

	int read2Bytes() {
		return (int)(ub(buffer[bpos++])+(256*ub(buffer[bpos++])));
	}

	int read3Bytes() {
		return (int)(ub(buffer[bpos++])+(256*ub(buffer[bpos++]))+(256*256*ub(buffer[bpos++])));
	}

	long read4Bytes() {
		return (long)(ub(buffer[bpos++])+(256*ub(buffer[bpos++]))+(256*256*ub(buffer[bpos++]))+(256*256*256*ub(buffer[bpos++])));
	}


	// Read n bytes depending
	int readnBytes() {
		switch(ub(buffer[bpos++])) {
			case 1 : return ub(buffer[bpos++]);
			case 2 : return this.read2Bytes();
			case 3 : return this.read3Bytes();
			case 4 : return (int)this.read4Bytes();
			default : return 255;
		}
	}

	// read [:length]
	long readLength() {
		switch(ub(buffer[bpos])) {
			case 251 : {bpos++; return (long) 0; }
			case 252 : {bpos++; return (long) read2Bytes();}
			case 253 : {bpos++; return (long) read3Bytes();}
			case 254 : {bpos++; return (long) read4Bytes();}
			default  : return (long) ub(buffer[bpos++]);
		}
	}
	
	// Read null-terminated string
	String readString() {
		StringBuffer b = new StringBuffer();
		int i=bpos;
		while(buffer[i]!=0) b.append((char)(buffer[i++]& 0xff));
		bpos=i+1;
		return b.toString();
	}

	// Read given-length string
	String readLenString() {
		long len = this.readLength();
		if (len==0) return new String("");
		StringBuffer b = new StringBuffer((int)len);
		for (int i=0; i<len; i++) {
			if (buffer[bpos]==0) break;
			else b.append((char)(buffer[bpos++]& 0xff));
		}
		return b.toString();
	}

	// Is this the last packet?
	boolean isLastDataPacket() {
		return ((packetLength==1)&&(ub(buffer[0])==254));
	}

	// Reset position to start of packet data
	void clear() {
		bpos=HEADERLENGTH;
	}

  // Send packet
  void send() 
  {
    if (zxk_debug)
      DBug.dbug.dbEnter("executeConnect()", "jmysql", 321);

    try 
    {
      int l=bpos;
      bpos=0;
      packetSequence++;
      this.write3Bytes(l-HEADERLENGTH);
      this.writeByte(packetSequence);

      if (zxk_debug) 
      {
        DBug.dbug.dbPrint(12, "s_1", "cw" + (l - HEADERLENGTH) + "," + packetSequence);
        for (int j=0; j<l; j++) 
	{
          if (buffer[j]>30) 
            DBug.dbug.dbPrint(12, "s_1", j+" : "+ub(buffer[j])+"  : "+(char)buffer[j]);
          else 
            DBug.dbug.dbPrint(12, "s_1", j+" : "+ub(buffer[j])+"  : ");
        }
      }

      sout.write(buffer, 0, l);
      sout.flush();
    } catch (Exception e) 
    { System.out.println("send() Error : "+e);}

    if (zxk_debug)
      DBug.dbug.dbReturn(312);    
  }

	void writeByte(byte b) {
		buffer[bpos++]=b;
	}

	void write2Bytes(int i) {
		int r;
		r=i%256;
		buffer[bpos++]=(byte)r;
		i=i/256;
		buffer[bpos++]=(byte)i;
	}

	void write3Bytes(int i) {
		int r;
		r=i%256;
		buffer[bpos++]=(byte)r;
		i=i/256;
		r=i%256;
		buffer[bpos++]=(byte)r;
		i=i/256;
		buffer[bpos++]=(byte)i;
	}

	void write4Bytes(long i) {
		long r;
		r=i%256;
		buffer[bpos++]=(byte)r;
		i=i/256;
		r=i%256;
		buffer[bpos++]=(byte)r;
		i=i/256;
		r=i%256;
		buffer[bpos++]=(byte)r;
		i=i/256;
		buffer[bpos++]=(byte)i;
	}

	// Write null-terminated string
	void writeString(String s) {
		int l = s.length();
		for(int i=0; i<l; i++) buffer[bpos++]=(byte)s.charAt(i);
		buffer[bpos++]=0;
	}

	// Write string, with no termination
	void writeStringNoNull(String s) {
		int l = s.length();
		for(int i=0; i<l; i++) buffer[bpos++]=(byte)s.charAt(i);
	}

}


public class gweMysql implements DBugConstant
{
  // Constants defined from mysql
  public static final int SLEEP=0;
  public static final int QUIT=1;
  public static final int INIT_DB=2;
  public static final int QUERY=3;
  public static final int FIELD_LIST=4;
  public static final int CREATE_DB=5;
  public static final int DROP_DB=6;
  public static final int RELOAD=7;
  public static final int SHUTDOWN=8;
  public static final int STATISTICS=9;
  public static final int PROCESS_INFO=10;
  public static final int CONNECT=11;
  public static final int PROCESS_KILL=12;
  public static final int FIELD_TYPE_DECIMAL=0;
  public static final int FIELD_TYPE_CHAR=1;
  public static final int FIELD_TYPE_SHORT=2;
  public static final int FIELD_TYPE_LONG=3;
  public static final int FIELD_TYPE_FLOAT=4;
  public static final int FIELD_TYPE_DOUBLE=5;
  public static final int FIELD_TYPE_NULL=6;
  public static final int FIELD_TYPE_TIMESTAMP=7;
  public static final int FIELD_TYPE_LONGLONG=8;
  public static final int FIELD_TYPE_INT24=9;
  public static final int FIELD_TYPE_DATE=10;
  public static final int FIELD_TYPE_TIME=11;
  public static final int FIELD_TYPE_DATETIME=12;
  public static final int FIELD_TYPE_TINY_BLOB=249;
  public static final int FIELD_TYPE_MEDIUM_BLOB=250;
  public static final int FIELD_TYPE_LONG_BLOB=251;
  public static final int FIELD_TYPE_BLOB=252;
  public static final int FIELD_TYPE_VAR_STRING=253;
  public static final int FIELD_TYPE_STRING=254;

  String host;
  int port;
  String database;
  String user;
  String password;
  String command;
  String extraParam;
  Socket sock;
  gweMysqlPacket packet;
  byte protocolVersion;
  String serverVersion;
  long threadId;
  String cryptSeed;
  long columnCount;
  gweMysqlColumn[] columnResult;
  Hashtable columnName;
  Hashtable columnFullName;
  int[] dataStart;
  SQLWarning warning = null;

  private long newColumnCount=0;
  private int[] pseudoMap = null;
  private long pseudoColumnCount=0;
  private Hashtable pseudoColumnName=null;
  private Hashtable pseudoFullName=null;
  private String[] pseudoColumnValue=null;
  private gweMysqlColumn[] pseudoColumnResult;
  private int[] pseudoColumnMode=null;
  private int splitColumnCount=0;
  private int rowCount=0;
  private String[] splitColumn=null;
  private int filterColumnPosition=-1;
  private String filterString=null;
  private int clientParam;
  private int updateCount;
  private int updateID;

  public gweMysql() 
  {
    warning = null;
    updateCount = -1;
    updateID = -1;
  }

  // Connect to mysql host		
  public synchronized void Connect(String h, int p, String d, String u, 
    String pw, String extraParam) throws SQLException 
  {
    if (zxk_debug)
      DBug.dbug.dbEnter("Connect()", "gweMysql", 219);

    host = h;
    port = p;
    database = d;
    user = u;
    password = pw;
    this.extraParam = extraParam;

    if (user.equals("")) 
    { 
      user = "nobody"; 
    }
    DriverManager.println("gweMysql: host: " + host + 
        " :port: " + port + " :database: " + database +
        " :user: " + user + " :password: " + password +
	" :extraParam: " + extraParam);

    if (zxk_debug)
      DBug.dbug.dbPrint(12, "c_1", "gweMysql: host: " + host + 
        " :port: " + port + " :database: " + database +
        " :user: " + user + " :password: " + password +
	" :extraParam: " + extraParam);

    try 
    {
      // Open TCP/IP socket
      sock = new Socket(host, port);
      DataInputStream sin = new DataInputStream(sock.getInputStream());
      DataOutputStream sout = new DataOutputStream(sock.getOutputStream());

      // Create new packet handler
      packet = new gweMysqlPacket(sin,sout);

//      packet.TRACING=false;

      // Get the first packet from mysql
      packet.next();
      protocolVersion = packet.readByte();
      serverVersion = packet.readString();
      long threadId = packet.read4Bytes();
      cryptSeed = packet.readString();
      if (zxk_debug)
        DBug.dbug.dbPrint(12, "c_1", protocolVersion+" "+serverVersion+" "+cryptSeed);

      // Send who we are
      packet.clear();
      // send client information for access check
      if (extraParam == null || extraParam.length() == 0)
      {
        if (protocolVersion > 9)
          clientParam = 1; // long password
	else
          clientParam = 0;
      }
      else
        clientParam = new Integer(extraParam).intValue();

      packet.write2Bytes(clientParam);
      packet.write3Bytes(packet.MAXBUFFER);
      packet.writeString(user);
      if (protocolVersion > 9)
        packet.writeString(cryptPasswordNew(password,cryptSeed));
      else
        packet.writeString(cryptPassword(password,cryptSeed));
      packet.send();
      // Check if all is ok
      packet.next();
      byte statusCode = packet.readByte();
      if (statusCode==(byte)0xff)  // get error message
      {
        String errorString;
	int errorNo = 2000;
        if (protocolVersion > 9)
	{
	  errorNo = packet.read2Bytes();
          errorString = packet.readString();
	}
	else
          errorString = packet.readString();
	packet.clearReceive();
        throw new SQLException(errorString, "", errorNo);
      } 
      else if (statusCode==0x00) 
      {
        long affectedRows = packet.readLength();
        long uniqueId = packet.readLength();
      } 
      else 
        throw new SQLException("Unknown status code '"+statusCode+"'");

    } catch (Exception e) 
    { throw new SQLException("gweMysql.Connect: "+e); }

    if (zxk_debug)
      DBug.dbug.dbReturn(312);    
  }

  /** execute one url command.
   *  @param command one of mysql server command.
   *  @param database involved database, maybe is null or is a thread id 
                      number when for command 'kill'.
   */
  public void urlCommand (String command, String database)
    throws SQLException 
  {
    if (zxk_debug)
      DBug.dbug.dbEnter("urlCommand()", "gweMysql", 219);

    byte reloadParam = (byte)0;
    if (command.equals("reload"))
      reloadParam = (byte)1;
    else if (command.equals("refresh"))
    {
      command = new String("reload");
      reloadParam = (byte)0xfe;
    }

    this.command = command;
    DriverManager.println("gweMysql:  command: " + command + 
       " :extraParam: " + extraParam);
    if (zxk_debug)
      DBug.dbug.dbPrint(12, "u_1", "gweMysql:  command: " + command + 
       " :extraParam: " + extraParam);

    try 
    {
      int cmd;
      if ((cmd = getCommandCode(command)) == -1)
        throw new SQLException("Unknow command: " + command); 
      byte statusCode = sendCommand(cmd, database, reloadParam);
//System.out.println("return: " + statusCode);
      if (cmd == STATISTICS)
      {
        packet.decPos();
        String msg = packet.readString();
        SQLWarning nw = new SQLWarning("Command=" + command + ": " + msg);
        if (warning != null)
        nw.setNextException(warning);
        warning = nw;
      }
      else if (cmd == PROCESS_INFO)
      {
        packet.decPos();
        long colLength = packet.readLength();
        StringBuffer msg = new StringBuffer();
	for (int i=0; i<colLength; ++i)
	{
	  packet.next();
	  packet.readByte();
	  String s2 = packet.readLenString();
	  msg.append(s2);
	  msg.append(' ');
	}
	packet.next();
	msg.append('\n');

        packet.next();
	while (!packet.isLastDataPacket())
	{
	  for (int i=0; i<colLength; ++i)
	  {
	    String s2 = packet.readLenString();
	    msg.append(s2);
	    msg.append(' ');
	  }
	  msg.append('\n');
	  packet.next();
	}
        SQLWarning nw = new SQLWarning("Command=" + command + ": " + new String(msg));
        if (warning != null)
	  nw.setNextException(warning);
	warning = nw;
      }
    } catch (Exception e) 
    { throw new SQLException("gweMysql.Connect: "+e); }

    if (zxk_debug)
      DBug.dbug.dbReturn(312);    
  }

  private byte sendCommand(int cmd, String extra, byte reloadParam)
    throws SQLException
  {
    if (zxk_debug)
      DBug.dbug.dbEnter("sendCommand()", "gweMysql", 219);

    packet.clear();
    packet.resetPacketSequence();
    packet.writeByte((byte)cmd);
    if ( cmd == INIT_DB || cmd == CREATE_DB || 
         cmd == DROP_DB || cmd == QUERY)
      packet.writeStringNoNull(extra);
    else if (cmd == PROCESS_KILL)
    {
      long id = new Long(extra).longValue();
      packet.write4Bytes(id);
    }
    else if (cmd == RELOAD && protocolVersion > 9)
      packet.writeByte(reloadParam);
    packet.send();

    // check the return
    packet.next();
    byte statusCode = packet.readByte();
    if (statusCode==(byte)0xff) // get error message
    {
      String errorString;
      int errorNo = 2000;
      if (protocolVersion > 9)
      {
	errorNo = packet.read2Bytes();
        errorString = packet.readString();
      }
      else
        errorString = packet.readString();
      packet.clearReceive();
      if (zxk_debug)
        DBug.dbug.dbReturn(312);    
      throw new SQLException(errorString, "", errorNo);
    } 
    else if (statusCode==0x00) // ok
    {
      //long affectedRows = packet.readLength();
      //long uniqueId = packet.readLength();
      if (cmd == CREATE_DB || cmd == DROP_DB)
      {
        SQLWarning nw = new SQLWarning("Command=" + command + ": ");
        if (warning != null)
          nw.setNextException(warning);
        warning = nw;
      }
    } 
    else if (packet.isLastDataPacket())  // end of field
    					  // statusCode == (byte)0xfe
    {
      SQLWarning nw = new SQLWarning("Command=" + command + ": ");
      if (warning != null)
        nw.setNextException(warning);
      warning = nw;
    }
   // else 
    if (zxk_debug)
      DBug.dbug.dbReturn(312);    

    return statusCode;
//throw new SQLException("Unknown status code '"+statusCode+"'");
  }

  private int getCommandCode(String command)
  {
    if (command.startsWith("initdb"))
      return INIT_DB;
    else if (command.startsWith("query"))
      return QUERY;
    else if (command.startsWith("create"))
      return CREATE_DB;
    else if (command.startsWith("drop"))
      return DROP_DB;
    else if (command.startsWith("reload"))
      return RELOAD;
    else if (command.startsWith("shutdown"))
      return SHUTDOWN;
    else if (command.startsWith("status"))
      return STATISTICS;
    else if (command.startsWith("proclist"))
      return PROCESS_INFO;
    else if (command.startsWith("kill"))
      return PROCESS_KILL;
    else
      return -1;
  }

  // Return hash for given string
  long hashPassword(String p) 
  {
    long  nr=1345345333;
    long  nr2=7;
    long tmp;
    for (int i=0; i<p.length(); i++) 
    {
      if ((p.charAt(i)==' ') || (p.charAt(i)=='\t') ) 
        continue;
      tmp = (long)p.charAt(i);
      nr ^= (((nr & 63) + nr2) * tmp) + (nr << 8);
      nr2 += tmp;
    }
    return nr & (((long) 1L << 31) -1L); 
  }

  long[] hashPasswordNew(String p)
  {
    long nr=1345345333L; 
    long add=7;
    long nr2=0x12345671L;
    long tmp;
    for (int i=0; i<p.length(); ++i)
    {
      if (p.charAt(i) == ' ' || p.charAt(i) == '\t')
        continue;	// skipp space in password
      tmp = (long)(0xff & p.charAt(i));
      nr ^= (((nr & 63)+add)*tmp)+ (nr << 8);
      nr2 += (nr2 << 8) ^ nr;
      add += tmp;
/*      System.err.println("tmp "+p.charAt(i) + " "+Integer.toHexString((int)(0x7fffffff&tmp))+" "+
Integer.toHexString((int)(0x7fffffff&add))+" "+
Integer.toHexString((int)(0x7fffffff&nr))+" "+
Integer.toHexString((int)(0x7fffffff&nr2))); */
    }
    long[] result = new long[2];
    result[0] = nr & 0x7fffffffL; // Don't use sign bit
    result[1] = nr2 & 0x7fffffffL;
    return result;
  }

  // return encrypted response, given clear message and password seed 
  String cryptPasswordNew(String pw, String msg)
  {
    //handle case for empty/null password
    if (pw == null || pw.length() == 0 ) 
      return pw;
    // msg = new String("/mVIJv7J");
//System.err.println(pw + " " + msg);
    long[] hpw = hashPasswordNew(pw);
    long[] hmsg = hashPasswordNew(msg);
//System.err.println(""+hpw[0]+" "+hpw[1]+" "+hmsg[0]+" "+hmsg[1]);
    long max_value = 0x3fffffffL;
    long seed = (hpw[0] ^ hmsg[0]) % max_value;
    long seed2 = (hpw[1] ^ hmsg[1]) % max_value;
    char[] cr = new char[msg.length()];
    byte b;
    double d;
    for (int i=0; i<msg.length(); i++) 
    {
      seed = (seed*3 + seed2) % max_value;
      seed2 = (seed + seed2 + 33) % max_value;
      d = (double)seed / (double)max_value;
      b = (byte)java.lang.Math.floor((d * 31) + 64);
      cr[i] = (char)b;
    }
    seed = (seed*3 + seed2) % max_value;
    seed2 = (seed + seed2 + 33) % max_value;
    d = (double)seed / (double)max_value;
    b = (byte)java.lang.Math.floor(d * 31);
    for (int i=0; i<msg.length(); i++) 
      cr[i] ^= (char)b;
//    System.err.println("crtpt " + new String(cr));
    return new String(cr);
  }

  // Return encrypted response, given clear password and seed message
  String cryptPassword(String p, String m) 
  {
    long hp, hm, s1, s2;
    long max_value= 0x01FFFFFF;
    double d;
    byte b;

    //handle case for empty/null password
    if (p == null || p.length() == 0 ) 
      return p;

    hp = hashPassword(p);
    hm = hashPassword(m);
    long nr = hp ^ hm;
    nr %= max_value;
    s1 = nr;
    s2 = nr/2;
		
    char[] cr = new char[m.length()];
    for (int i=0; i<m.length(); i++) 
    {
      s1 = (s1 * 3 + s2) % max_value;
      s2 = (s1 + s2 + 33) % max_value;
      d = (double)s1 / max_value;
      b = (byte)java.lang.Math.floor((d * 31) + 64);
      cr[i]=(char)b;
    }
    return new String(cr);
  }

  public gweMysql sqlQuery(String q) throws SQLException 
  {
    if (zxk_debug)
      DBug.dbug.dbEnter("sqlQuery()", "gweMysql", 219);

    DriverManager.println("gweMysql.sqlQuery: " + q);
    if (zxk_debug)
      DBug.dbug.dbPrint(12, "s_3", "gweMysql.sqlQuery: " + q);

    updateCount = -1;
    updateID = -1;

    /* 
       this QUERY have a special net protocol 
       which return the column length in "long" and 
       others command include QUERY with update all return the status in "byte".
     */
    // Send query command and sql query string
    packet.clearAllReceive();
    sendCommand(QUERY, q, (byte)0);
    // back 1 byte since sendCommand() read 1 byte.
    packet.decPos(); 
    columnCount = packet.readLength();
    if (columnCount==0) 
    { 
      updateCount = (int)packet.readLength();
      updateID = (int)packet.readLength();
      columnResult = null;
      if (zxk_debug)
	DBug.dbug.dbReturn(462);
      return this;
    } 
    else 
    {
    columnResult = new gweMysqlColumn[(int)columnCount];
    columnName=new Hashtable((int)columnCount);
    columnFullName=new Hashtable((int)columnCount);
    dataStart = new int[(int)columnCount];
    pseudoMap = new int[(int)columnCount];
    for (int i=0; i<columnCount; ++i)
      pseudoMap[i] = i;
    pseudoColumnCount = 0;
    splitColumnCount = 0;
    newColumnCount = columnCount;

    // Read in the column information
    for (int i=0;i<columnCount; i++) 
    {
      packet.next();
      String tableName=packet.readLenString();
      String colName=packet.readLenString();
      int colLength = packet.readnBytes();
      int colType=packet.readnBytes();
      packet.readByte(); // We know it's currently 2
      short colFlag = (short) packet.ub(packet.readByte());
      int colDecimals = packet.ub(packet.readByte());
      columnResult[i] = new gweMysqlColumn(tableName, colName, colLength, 
        colType, colFlag, colDecimals);
      if (zxk_debug)
        DBug.dbug.dbPrint(12, "fld", tableName + "," + colName + "," + 
	colLength + "," + colType + "," + colFlag + "," + colDecimals);
      columnName.put(colName, new Integer(i));
      String fullName = tableName.concat(".").concat(colName);
      columnFullName.put(fullName, new Integer(i));
    }
    packet.next(); //Redundant - we know how many, this is end of section marker packet
    }
    if (zxk_debug)
      DBug.dbug.dbReturn(312);    
    return this;
  }

  public void sqlQuit() throws SQLException 
  {
	// Send quit command to mysql
	packet.clear();
	packet.resetPacketSequence();
	packet.writeByte((byte)gweMysql.QUIT);
	packet.send();
  }

  public boolean nextResult() throws SQLException 
  {
    if (zxk_debug)
      DBug.dbug.dbEnter("nextResult()", "gweMysql", 219);

    int i, p;
    boolean flag = false;

	// Get the next incoming packet
	packet.next();
	// check error. bug fixed, 1997/09/12, by zxk
	if (packet.readByte() == (byte)0xff) // get error message
	{
          String errorString;
	  int errorNo = 2000;
          if (protocolVersion > 9)
  	  {
	    errorNo = packet.read2Bytes();
            errorString = packet.readString();
	  }
	  else
            errorString = packet.readString();
	  packet.clearReceive();
          if (zxk_debug)
            DBug.dbug.dbReturn(312);    
	  throw new SQLException(errorString, "", errorNo);
	}
	else
	  packet.decPos();
	// It it's not the last one, note start positions of data
	if(!packet.isLastDataPacket()) {
		for (i=0; i<columnCount; i++) {
			p=packet.currentPos();
			dataStart[i]=p;
			// packet.setPos(1+p+(int)packet.readLength());
			packet.setPos((int)packet.readLength() + packet.currentPos());
		}
		// only works for getColumns() of DatabaseMetaData
		if (splitColumnCount == 0)
		{
		  flag = true;
		}
		else if (splitColumnCount == -1) // do row filter
		{
		  rowCount++;
//System.out.println("zxk1: " + filterColumnPosition + filterString);
		  packet.setPos(dataStart[filterColumnPosition]);
		  String col = packet.readLenString();
		  if (col.equals(filterString))
		    flag = true;
		  else
		  {
		    if (nextResult() == true)
		      flag = true;
		    else
		      flag = false;
		  }
		}
		else  // split column specially for getColumns() method
		{
		  rowCount++;
		  packet.setPos(dataStart[1]);
		  String col1 = packet.readLenString();
		  packet.setPos(dataStart[2]);
		  String col2 = packet.readLenString();
		  splitColumn1_2(col1, col2);
		  flag = true;
		}
	} 
        if (zxk_debug)
          DBug.dbug.dbReturn(312);    
	return flag;
  }

  public int getColumnCount() {
		return (int) (newColumnCount + pseudoColumnCount);
	}
	
	public gweMysqlColumn getColumn(int c) throws SQLException 
	{
	  c--;	// Columnd go from 1..columnCount to 0..columnCount-1
	  c = pseudoMap[c];
	  if ( c < columnCount) 
	  {
		if ((c<columnCount)&&(c>=0)) {
			return columnResult[c];
		} else {
			throw new SQLException("Invalid column number : "+c);
		}
	  }
	  else // pseudo column
	  {
	    c -= columnCount;
	    if ( (c >= 0) && (c < pseudoColumnCount) )
	      return pseudoColumnResult[c];
            else
	      throw new SQLException("Invalid pseudo column number : "+c);
	  }
	}

	public boolean isNull(int c) throws SQLException 
	{
	  c--;
	  c = pseudoMap[c];
	  if ( c < columnCount) 
	  {
		if ((c<columnCount)&&(c>=0)) {
			packet.setPos(dataStart[c]);
			return (packet.readLength()==0);
		} else throw new SQLException("Invalid column number : "+c);
	  }
	  else
	  {
	    c -= columnCount;
	    if ( (c >= 0) && (c < pseudoColumnCount) )
	      return pseudoColumnResult[c].getType() == 6;
            else
	      throw new SQLException("Invalid pseudo column number : "+c);
	  }
	}

	public int getColumnNum(String cn) throws SQLException {
		Integer nI = (Integer) columnName.get(cn);
		if (nI==null) nI = (Integer) columnFullName.get(cn);
		if (nI!=null) return nI.intValue()+1;
		//
		// try pseudo column
		//
		if (pseudoColumnCount == 0)
		  throw new SQLException("Invalid column string : "+cn);
		nI = (Integer) pseudoColumnName.get(cn);
		if (nI==null) nI = (Integer) pseudoFullName.get(cn);
		if (nI!=null) return nI.intValue()+1;
		throw new SQLException("Invalid column name : "+cn);
	}


	public String getString(int c) throws SQLException 
	{
		c--;
		c = pseudoMap[c];
	  if ( c < columnCount) 
	  {
		if ((c<columnCount)&&(c>=0)) 
		{
			packet.setPos(dataStart[c]);
			String s = packet.readLenString();
			if (s.length() == 0)
			  s = null;
			return s;
		} else throw new SQLException("Invalid column number : "+c);
	  }
	  else
	  {
	    c -= columnCount;
	    if ( (c >= 0) && (c < pseudoColumnCount) )
	    {
	      if (pseudoColumnMode[c] == -1) // pseudo fix column
	        return pseudoColumnValue[c];
	      else if (pseudoColumnMode[c] == -2) // pseudo auto-increase column
	        return new String(new StringBuffer().append(rowCount));
	      else
	        return splitColumn[pseudoColumnMode[c]];
	    }
            else
	      throw new SQLException("Invalid pseudo column number : "+c);
	  }
	}

	public String getString(String cn) throws SQLException {
		Integer nI = (Integer) columnName.get(cn);
		if (nI==null) nI = (Integer) columnFullName.get(cn);
		if (nI!=null) {
			int c = nI.intValue()+1;
			if (c<columnCount) {
				packet.setPos(dataStart[c]);
				return packet.readLenString();
			}
		}
		//
		// try pseudo column
		//
		if (pseudoColumnCount == 0)
		  throw new SQLException("Invalid column string : "+cn);
		nI = (Integer) pseudoColumnName.get(cn);
		if (nI==null) nI = (Integer) pseudoFullName.get(cn);
		if (nI!=null) {
			int c = nI.intValue();
			if (c<columnCount) 
			{
	      		  if (pseudoColumnMode[c] == -1) // pseudo fix column
	        	    return pseudoColumnValue[c];
	      		  else if (pseudoColumnMode[c] == -2) // pseudo auto-increase column
	        	    return new String(new StringBuffer().append(rowCount));
	      		  else
	        	    return splitColumn[pseudoColumnMode[c]];
			}
		}
		throw new SQLException("Invalid column string : "+cn);
	}

	public byte[] getBytes(int c) throws SQLException 
	{
		c--;
		c = pseudoMap[c];
	  if ( c < columnCount) 
	  {
		if ((c<columnCount)&&(c>=0)) {
			packet.setPos(dataStart[c]);
			int l = (int)packet.readLength();
		    byte[] r = new byte[l];
			for (int i=0; i<l; i++) {
				r[i]=packet.readByte();
			}
			return r;
		}
		else
		  throw new SQLException("Invalid column number : "+c);
	  }
	  else
	  {
	    c -= columnCount;
	    if ( (c >= 0) && (c < pseudoColumnCount) )
	    {
	      if (pseudoColumnValue[c] != null)
	        return pseudoColumnValue[c].getBytes();
              else
	        return null;
	    }
            else
	      throw new SQLException("Invalid pseudo column number : "+c);
	  }
	}


	public void reConnect() throws SQLException {
		try {
			sqlQuit();
			sock.close();
			this.Connect(host, port, database, user, password, extraParam);
			this.urlCommand(command, database);
		} catch (Exception e) {throw new SQLException("gweMysql.reConnect: "+e);}
	}
		

	public void Close() throws SQLException {
		try {
			sqlQuit(); // bug fixed, 1997/09/12, by zxk
			sock.close();
		} catch (Exception e) {throw new SQLException("gweMysql.Close: "+e);}
	}

	//
	// The following methods added by X.K. Zhu
	// to support DatabaseMetaData, 1997/09/03
	//

	public String getHost()
	{
	  return host;
	}

	public int getPort()
	{
	  return port;
	}

	public String getUser()
	{
	  return user;
	}

	public String getDbName()
	{
	  return database;
	}

	/** Modify the column head to get standard JDBC name.
	 *  @param colName  standard name.
	 */
	public void modifyColumnHead(String[] colName)
	{
	  int i;
	  for (i=0; i<colName.length; ++i)
	    if (colName[i] != null)  // bypass any null which means do not
	    {			     // modify
	      columnResult[i].setName(colName[i]);
	    }
	}

  /** Add logic view which include some pseudo column which JDBC asked.
   *  @param tabName the current table name (maybe is null).
   *  @param realColMap the index of original column map into the logic view.
   *  @param pseudoColMap the index of new pseudo column map into the logic view.
   *  @param colName new pseudo column name.
   *  @param colType new pseudo column type.
   *  @param colValue new pseudo column value.
   *  @param pseudoColMode new pseudo column mode.
   */
  public void addPseudoColumn(String tabName, int[] realColMap, 
      int[] pseudoColMap, String[] colName, int[] colType, 
      String[] colValue, int[] pseudoColMode)
  {
    splitColumn = null;
    rowCount = 0;
    filterColumnPosition = -1;
    filterString = null;

    pseudoColumnCount = pseudoColMap.length;
    pseudoMap = new int[(int)(columnCount + pseudoColumnCount)];
    int i;
    splitColumnCount = 0;
    newColumnCount = 0;
    for (i=0; i<columnCount; ++i)
    {
      if (realColMap[i] == -1)
      {
        continue;
      }
      pseudoMap[realColMap[i]] = i;
      newColumnCount++;
    }
    for (i=0; i<pseudoColumnCount; ++i)
    {
      pseudoMap[pseudoColMap[i]] = i + (int)columnCount;
      if (pseudoColMode[i] >= 0)
        splitColumnCount++; // found the pseudo created from other real column
    }

    pseudoColumnResult = new gweMysqlColumn[(int)pseudoColumnCount];
    pseudoColumnName=new Hashtable((int)pseudoColumnCount);
    pseudoFullName=new Hashtable((int)pseudoColumnCount);
    pseudoColumnValue = colValue;
    pseudoColumnMode = pseudoColMode;

    for (i=0; i<columnCount; ++i)
    {
      columnResult[i].setTable(tabName);
      String fullName = tabName.concat(".").concat(columnResult[i].getName());
      columnFullName.put(fullName, new Integer(i));
    }
    for (i=0; i<pseudoColumnCount; ++i)
    {
      if (colType[i] != 6) // NULL
        pseudoColumnResult[i] = new gweMysqlColumn(tabName, colName[i], 
                                                   32, colType[i], (short)1, 0);
      else
        pseudoColumnResult[i] = new gweMysqlColumn(tabName, colName[i], 
                                                   32, colType[i], (short)0, 0);
      pseudoColumnName.put(colName[i], new Integer(i));
      String fullName = tabName.concat(".").concat(colName[i]);
      pseudoFullName.put(fullName, new Integer(i));
    }

    if (splitColumnCount != 0)
      splitColumn = new String[splitColumnCount];
  }

  /** Set row filter and only return all rows which satify the condition
   *  @param filterColPos indicate which column need be check for condition.
   *  @param filterString the condition string.
   */
  public void setRowFilter(int filterColPos, String filterString)
  {
    filterColumnPosition = filterColPos;
    this.filterString = filterString;
    splitColumnCount = -1; // speciall mark for filter process
  }

  private void splitColumn1_2(String col1, String col2)
  {
    int i = col1.indexOf('(');
    if (i == -1) // no "()"
    {
      splitColumn[0] = col1;  // DATA_TYPE
      splitColumn[2] = null; //  COLUMN_SIZE
      splitColumn[3] = null; // DECIMAL_DIGITS
    }
    else
    {
      splitColumn[0] = col1.substring(0, i);  // DATA_TYPE
      String s1 = col1.substring(i+1, col1.length()-1); // skip '(' and cut ')'
      i = s1.indexOf(',');
      if (i == -1)
      {
        splitColumn[2] = s1;  // COLUMN_SIZE
        splitColumn[3] = null;  // DECIMAL_DIGITS
      }
      else
      {
        splitColumn[2] = s1.substring(i+1);  // COLUMN_SIZE
        splitColumn[3] = s1.substring(0, i);  // DECIMAL_DIGITS
      }
    }
//System.out.println(col1 + "<==zxk==>" + splitColumn[0] + "," + splitColumn[2] 
//  + "," + splitColumn[3]);
    splitColumn[4] = null;  // NUM_PREC_RADIX
    splitColumn[5] = null;  // ORDINAL_POSITION

    if (splitColumn[0].equals("bigint"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(8)));
      splitColumn[4] = new String(new StringBuffer().append(10));
    }
    else if(splitColumn[0].equals("blob"))
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(252)));
    else if(splitColumn[0].equals("char"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(254)));
      splitColumn[5] = new String(new StringBuffer().
      	append(new Integer(splitColumn[2]).intValue()*2));
    }
    else if(splitColumn[0].equals("date"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(10)));
      splitColumn[2] = new String(new StringBuffer().append(10));
      splitColumn[5] = new String(new StringBuffer().append(20));
    }
    else if(splitColumn[0].equals("datetime"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(12)));
      splitColumn[2] = new String(new StringBuffer().append(19));
      splitColumn[5] = new String(new StringBuffer().append(38));
    }
    else if(splitColumn[0].equals("decimal"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(0)));
      splitColumn[4] = new String(new StringBuffer().append(10));
    }
    else if(splitColumn[0].equals("double"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(5)));
      splitColumn[4] = new String(new StringBuffer().append(10));
    }
    else if(splitColumn[0].equals("float"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(4)));
      splitColumn[4] = new String(new StringBuffer().append(10));
    }
    else if(splitColumn[0].equals("int"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(3)));
      splitColumn[4] = new String(new StringBuffer().append(10));
    }
    else if(splitColumn[0].equals("longblob"))
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(251)));
    else if(splitColumn[0].equals("mediumblob"))
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(250)));
    else if(splitColumn[0].equals("mediumint"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(9)));
      splitColumn[4] = new String(new StringBuffer().append(10));
    }
    else if(splitColumn[0].equals("smallint"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(2)));
      splitColumn[4] = new String(new StringBuffer().append(10));
    }
    else if(splitColumn[0].equals("tinyblob"))
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(249)));
    else if(splitColumn[0].equals("tinyint"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(1)));
      splitColumn[4] = new String(new StringBuffer().append(10));
    }
    else if(splitColumn[0].equals("varchar"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(253)));
      splitColumn[5] = new String(new StringBuffer().
      	append(new Integer(splitColumn[2]).intValue()*2));
    }
    else if(splitColumn[0].equals("time"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(11)));
      splitColumn[2] = new String(new StringBuffer().append(8));
      splitColumn[5] = new String(new StringBuffer().append(16));
    }
    else if(splitColumn[0].equals("timestamp"))
    {
      splitColumn[1] = new String(new StringBuffer().append(mapColumnType(7)));
      splitColumn[5] = new String(new StringBuffer().
      	append(new Integer(splitColumn[2]).intValue()*2));
    }

    if (col2 == null)
      splitColumn[6] = new String(new StringBuffer().append(2));
    else if (col2.equals("YES"))
      splitColumn[6] = new String(new StringBuffer().append(1));
    else 
      splitColumn[6] = new String(new StringBuffer().append(0));
  }

  /**
   * Gives the column type using the types in java.sql.Types.
   * @see java.sqlTypes
   * @see java.sql.ResultSetMetaData#getColumnType
   * @exception SQLException thrown for any number of reasons
   * @param type the column type information is needed on
   * @return the type as listed in java.sql.Types
   */
  private int mapColumnType(int type) {
		switch (type) {
			case 0 : return Types.DECIMAL;
			case 1 : return Types.CHAR;
			case 2 : return Types.SMALLINT;
			case 3 : return Types.INTEGER;
			case 4 : return Types.FLOAT;
			case 5 : return Types.DOUBLE;
			case 6 : return Types.NULL;
			case 7 : return Types.TIMESTAMP;
			case 8 : return Types.BIGINT;
			case 9 : return Types.INTEGER;
			case 10 : return Types.DATE;
			case 11 : return Types.TIME;
			case 12 : return Types.TIMESTAMP;
			case 249 : return Types.VARBINARY;
			case 250 : return Types.LONGVARBINARY;
			case 251 : return Types.LONGVARBINARY;
			case 252 : return Types.LONGVARBINARY;
			case 253 : return Types.VARCHAR;
			case 254 : return Types.VARCHAR;
			default: return Types.VARCHAR;
		}
  }

  public SQLWarning getWarnings() throws SQLException
  {
    return warning;
  }

  public void clearWarnings()
  {
    warning = null;
  }

  public String getDbVersion()
  {
    return new String(serverVersion + " " + protocolVersion);
  }

  public void closeResultSet()
    throws SQLException
  {
    try {
      packet.clearAllReceive();
    } catch (SQLException e)
    { throw e; }
  }

  public int getUpdateCount()
  {
    return updateCount;
  }

  public int getUpdateID()
  {
    return updateID;
  }

  public boolean reallyResultSet()
  {
    return (columnResult != null);
  }
}

