/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.securityanalytics.threatIntel.service;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.OpenSearchException;
import org.opensearch.OpenSearchStatusException;
import org.opensearch.action.ActionRequest;
import org.opensearch.action.ActionType;
import org.opensearch.action.DocWriteRequest;
import org.opensearch.action.admin.indices.create.CreateIndexRequest;
import org.opensearch.action.admin.indices.create.CreateIndexResponse;
import org.opensearch.action.admin.indices.delete.DeleteIndexRequestBuilder;
import org.opensearch.action.bulk.BulkRequest;
import org.opensearch.action.bulk.BulkResponse;
import org.opensearch.action.index.IndexRequest;
import org.opensearch.action.search.SearchRequest;
import org.opensearch.action.support.GroupedActionListener;
import org.opensearch.action.support.IndicesOptions;
import org.opensearch.action.support.WriteRequest;
import org.opensearch.client.Client;
import org.opensearch.cluster.metadata.IndexNameExpressionResolver;
import org.opensearch.cluster.service.ClusterService;
import org.opensearch.common.settings.ClusterSettings;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.common.xcontent.XContentFactory;
import org.opensearch.core.action.ActionListener;
import org.opensearch.core.rest.RestStatus;
import org.opensearch.core.xcontent.NamedXContentRegistry;
import org.opensearch.core.xcontent.ToXContent;
import org.opensearch.core.xcontent.XContentBuilder;
import org.opensearch.securityanalytics.model.ThreatIntelFeedData;
import org.opensearch.securityanalytics.settings.SecurityAnalyticsSettings;
import org.opensearch.securityanalytics.threatIntel.action.PutTIFJobAction;
import org.opensearch.securityanalytics.threatIntel.action.PutTIFJobRequest;
import org.opensearch.securityanalytics.threatIntel.action.ThreatIntelIndicesResponse;
import org.opensearch.securityanalytics.threatIntel.common.StashedThreadContext;
import org.opensearch.securityanalytics.threatIntel.model.TIFMetadata;
import org.opensearch.securityanalytics.threatIntel.service.TIFJobParameterService;
import org.opensearch.securityanalytics.threatIntel.util.ThreatIntelFeedDataUtils;
import org.opensearch.securityanalytics.util.IndexUtils;
import org.opensearch.securityanalytics.util.SecurityAnalyticsException;

public class ThreatIntelFeedDataService {
    private static final Logger log = LogManager.getLogger(ThreatIntelFeedDataService.class);
    public static final String SETTING_INDEX_REFRESH_INTERVAL = "index.refresh_interval";
    private static final Map<String, Object> INDEX_SETTING_TO_CREATE = Map.of("index.number_of_shards", 1, "index.number_of_replicas", 0, "index.refresh_interval", -1, "index.hidden", true);
    private final ClusterService clusterService;
    private final ClusterSettings clusterSettings;
    private final NamedXContentRegistry xContentRegistry;
    private final Client client;
    private final IndexNameExpressionResolver indexNameExpressionResolver;

    public ThreatIntelFeedDataService(ClusterService clusterService, Client client, IndexNameExpressionResolver indexNameExpressionResolver, NamedXContentRegistry xContentRegistry) {
        this.client = client;
        this.indexNameExpressionResolver = indexNameExpressionResolver;
        this.xContentRegistry = xContentRegistry;
        this.clusterService = clusterService;
        this.clusterSettings = clusterService.getClusterSettings();
    }

    public void getThreatIntelFeedData(ActionListener<List<ThreatIntelFeedData>> listener) {
        try {
            String tifdIndex = this.getLatestIndexByCreationDate();
            if (tifdIndex == null) {
                this.createThreatIntelFeedData(listener);
            } else {
                this.fetchThreatIntelFeedDataFromIndex(tifdIndex, listener);
            }
        }
        catch (Exception e) {
            log.error("Failed to get threat intel feed data", (Throwable)e);
            listener.onFailure(e);
        }
    }

    private String getLatestIndexByCreationDate() {
        return IndexUtils.getNewIndexByCreationDate(this.clusterService.state(), this.indexNameExpressionResolver, ".opensearch-sap-threat-intel*");
    }

    public void createIndexIfNotExists(String indexName, ActionListener<CreateIndexResponse> listener) {
        if (this.clusterService.state().metadata().hasIndex(indexName)) {
            listener.onResponse((Object)new CreateIndexResponse(true, true, indexName));
            return;
        }
        CreateIndexRequest createIndexRequest = (CreateIndexRequest)new CreateIndexRequest(indexName).settings(INDEX_SETTING_TO_CREATE).mapping(this.getIndexMapping()).timeout((TimeValue)this.clusterSettings.get(SecurityAnalyticsSettings.THREAT_INTEL_TIMEOUT));
        StashedThreadContext.run(this.client, () -> this.client.admin().indices().create(createIndexRequest, ActionListener.wrap(response -> {
            if (response.isAcknowledged()) {
                listener.onResponse(response);
            } else {
                listener.onFailure((Exception)new OpenSearchStatusException("Threat intel feed index creation failed", RestStatus.INTERNAL_SERVER_ERROR, new Object[0]));
            }
        }, arg_0 -> ((ActionListener)listener).onFailure(arg_0))));
    }

    public void parseAndSaveThreatIntelFeedDataCSV(String indexName, Iterator<CSVRecord> iterator, Runnable renewLock, TIFMetadata tifMetadata, ActionListener<ThreatIntelIndicesResponse> listener) throws IOException {
        if (indexName == null || iterator == null || renewLock == null) {
            throw new IllegalArgumentException("Parameters cannot be null, failed to save threat intel feed data");
        }
        TimeValue timeout = (TimeValue)this.clusterSettings.get(SecurityAnalyticsSettings.THREAT_INTEL_TIMEOUT);
        Integer batchSize = (Integer)this.clusterSettings.get(SecurityAnalyticsSettings.BATCH_SIZE);
        ArrayList<BulkRequest> bulkRequestList = new ArrayList<BulkRequest>();
        BulkRequest bulkRequest = new BulkRequest();
        bulkRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        ArrayList<ThreatIntelFeedData> tifdList = new ArrayList<ThreatIntelFeedData>();
        while (iterator.hasNext()) {
            CSVRecord record = iterator.next();
            String iocType = tifMetadata.getIocType();
            Integer colNum = tifMetadata.getIocCol();
            String iocValue = record.values()[colNum].split(" ")[0];
            if (iocType.equals("ip") && !ThreatIntelFeedDataService.isValidIp(iocValue)) {
                log.info("Invalid IP address, skipping this ioc record.");
                continue;
            }
            String feedId = tifMetadata.getFeedId();
            Instant timestamp = Instant.now();
            ThreatIntelFeedData threatIntelFeedData = new ThreatIntelFeedData(iocType, iocValue, feedId, timestamp);
            tifdList.add(threatIntelFeedData);
        }
        for (ThreatIntelFeedData tifd : tifdList) {
            XContentBuilder tifData = tifd.toXContent(XContentFactory.jsonBuilder(), ToXContent.EMPTY_PARAMS);
            IndexRequest indexRequest = new IndexRequest(indexName);
            indexRequest.source(tifData);
            indexRequest.opType(DocWriteRequest.OpType.INDEX);
            bulkRequest.add(indexRequest);
            if (bulkRequest.requests().size() != batchSize.intValue()) continue;
            bulkRequestList.add(bulkRequest);
            bulkRequest = new BulkRequest();
            bulkRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        }
        bulkRequestList.add(bulkRequest);
        GroupedActionListener bulkResponseListener = new GroupedActionListener(ActionListener.wrap(bulkResponses -> {
            int idx = 0;
            for (BulkResponse response : bulkResponses) {
                BulkRequest request = (BulkRequest)bulkRequestList.get(idx);
                if (!response.hasFailures()) continue;
                throw new OpenSearchException("error occurred while ingesting threat intel feed data in {} with an error {}", new Object[]{StringUtils.join((Object[])new Set[]{request.getIndices()}), response.buildFailureMessage()});
            }
            listener.onResponse((Object)new ThreatIntelIndicesResponse(true, List.of(indexName)));
        }, arg_0 -> listener.onFailure(arg_0)), bulkRequestList.size());
        for (int i = 0; i < bulkRequestList.size(); ++i) {
            this.saveTifds((BulkRequest)bulkRequestList.get(i), timeout, (ActionListener<BulkResponse>)bulkResponseListener);
        }
        renewLock.run();
    }

    public static boolean isValidIp(String ip) {
        if (StringUtils.isBlank((CharSequence)ip)) {
            return false;
        }
        String ipPattern = "^\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}$";
        Pattern pattern = Pattern.compile(ipPattern);
        Matcher matcher = pattern.matcher(ip);
        return matcher.matches();
    }

    public void saveTifds(BulkRequest bulkRequest, TimeValue timeout, ActionListener<BulkResponse> listener) {
        try {
            StashedThreadContext.run(this.client, () -> this.client.bulk(bulkRequest, listener));
        }
        catch (OpenSearchException e) {
            log.error("failed to save threat intel feed data", (Throwable)e);
        }
    }

    public void deleteThreatIntelDataIndex(List<String> indices) {
        if (indices == null || indices.isEmpty()) {
            return;
        }
        Optional<String> invalidIndex = indices.stream().filter(index -> !index.startsWith(".opensearch-sap-threat-intel")).findAny();
        if (invalidIndex.isPresent()) {
            throw new OpenSearchException("the index[{}] is not threat intel data index which should start with {}", new Object[]{invalidIndex.get(), ".opensearch-sap-threat-intel"});
        }
        StashedThreadContext.run(this.client, () -> ((DeleteIndexRequestBuilder)this.client.admin().indices().prepareDelete(indices.toArray(new String[0])).setIndicesOptions(IndicesOptions.LENIENT_EXPAND_OPEN_CLOSED_HIDDEN).setTimeout((TimeValue)this.clusterSettings.get(SecurityAnalyticsSettings.THREAT_INTEL_TIMEOUT))).execute(ActionListener.wrap(response -> {
            if (!response.isAcknowledged()) {
                log.error((Object)new OpenSearchException("failed to delete threat intel feed index[{}]", new Object[]{String.join((CharSequence)",", indices)}));
            }
        }, e -> log.error("failed to delete threat intel feed index [{}]", (Throwable)e))));
    }

    private void createThreatIntelFeedData(ActionListener<List<ThreatIntelFeedData>> listener) {
        this.client.execute((ActionType)PutTIFJobAction.INSTANCE, (ActionRequest)new PutTIFJobRequest("feed_updater", (TimeValue)this.clusterSettings.get(SecurityAnalyticsSettings.TIF_UPDATE_INTERVAL)), ActionListener.wrap(r -> {
            if (!r.isAcknowledged()) {
                listener.onFailure(new Exception("Failed to acknowledge Put Tif job action"));
                return;
            }
            log.debug("Acknowledged threat intel feed updater job created");
            String tifdIndex = this.getLatestIndexByCreationDate();
            this.fetchThreatIntelFeedDataFromIndex(tifdIndex, listener);
        }, e -> {
            log.debug("Failed to create threat intel feed updater job", (Throwable)e);
            listener.onFailure(e);
        }));
    }

    private void fetchThreatIntelFeedDataFromIndex(String tifdIndex, ActionListener<List<ThreatIntelFeedData>> listener) {
        SearchRequest searchRequest = new SearchRequest(new String[]{tifdIndex});
        searchRequest.source().size(9999);
        String finalTifdIndex = tifdIndex;
        this.client.search(searchRequest, ActionListener.wrap(r -> listener.onResponse(ThreatIntelFeedDataUtils.getTifdList(r, this.xContentRegistry)), e -> {
            log.error(String.format("Failed to fetch threat intel feed data from system index %s", finalTifdIndex), (Throwable)e);
            listener.onFailure(e);
        }));
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private String getIndexMapping() {
        try (InputStream is = TIFJobParameterService.class.getResourceAsStream("/mappings/threat_intel_feed_mapping.json");){
            String string;
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8));){
                string = reader.lines().map(String::trim).collect(Collectors.joining());
            }
            return string;
        }
        catch (IOException e) {
            log.error("Runtime exception when getting the threat intel index mapping", (Throwable)e);
            throw new SecurityAnalyticsException("Runtime exception when getting the threat intel index mapping", RestStatus.INTERNAL_SERVER_ERROR, e);
        }
    }
}

