/*
 * GQview image viewer
 * (C)2000 John Ellis
 *
 * Author: John Ellis
 *
 */

#include "gqview.h"
#include "clist_edit.h"
#include <gdk/gdkkeysyms.h>

#define COLUMN_PADDING 3	/* stolen from gtkclist.c */
#define COLUMN_SPACING 1	/* same */

static void clist_edit_make_row_visible(ClistEditData *ced, gint row)
{
	if (gtk_clist_row_is_visible(ced->clist, row) == GTK_VISIBILITY_FULL) return;

	if (row == 0 || gtk_clist_row_is_visible(ced->clist, row - 1) == GTK_VISIBILITY_NONE)
		{
		gtk_clist_moveto(ced->clist, row, 1, 0.0, 0.0);
		}
	else
		{
		gtk_clist_moveto(ced->clist, row, 1, 1.0, 0.0);
		}
}

static void clist_edit_close(ClistEditData *ced)
{
	gtk_grab_remove(ced->window);
	gdk_pointer_ungrab(GDK_CURRENT_TIME);

	gtk_widget_destroy(ced->window);

	g_free(ced->old_name);
	g_free(ced->new_name);

	g_free(ced);
}

static void clist_edit_do(ClistEditData *ced)
{
	ced->new_name = g_strdup(gtk_entry_get_text(GTK_ENTRY(ced->entry)));

	if (strcmp(ced->new_name, ced->old_name) != 0)
		{
		if (ced->edit_func)
			{
			ced->edit_func(ced, ced->old_name, ced->new_name, ced->edit_data);
			}
		}
}

static void clist_edit_click_end_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	ClistEditData *ced = data;

	clist_edit_do(ced);
	clist_edit_close(ced);
}

static void clist_edit_click_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	ClistEditData *ced = data;

	gint x, y;
	gint w, h;

	gint xr, yr;

	xr = (gint)event->x_root;
	yr = (gint)event->y_root;

	gdk_window_get_origin(ced->window->window, &x, &y);
	gdk_window_get_size(ced->window->window, &w, &h);

	if (xr < x || yr < y || xr > x + w || yr > y + h)
		{
		/* gobble the release event, so it does not propgate to an underlying widget */
		gtk_signal_connect(GTK_OBJECT(ced->window), "button_release_event", clist_edit_click_end_cb, ced);
		}
}

static gint clist_edit_key_press_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	ClistEditData *ced = data;

	switch (event->keyval)
		{
		case GDK_Return:
		case GDK_Tab: 		/* ok, we are going to intercept the focus change
					   from keyboard and act like return was hit */
		case GDK_ISO_Left_Tab:
		case GDK_Up:
		case GDK_Down:
		case GDK_KP_Up:
		case GDK_KP_Down:
		case GDK_KP_Left:
		case GDK_KP_Right:
			clist_edit_do(ced);
			clist_edit_close(ced);
			break;
		case GDK_Escape:
			clist_edit_close(ced);
			break;
		default:
			break;
		}

	return FALSE;
}

gint clist_edit_by_row(GtkCList *clist, gint row, gint column,
		       void (*edit_func)(ClistEditData *, gchar *, gchar *, gpointer), gpointer edit_data)
{
	ClistEditData *ced;
	gint x, y, w, h;	/* geometry of cell within clist */
	gint wx, wy;		/* geometry of clist from root window */
	gchar *text;
	guint8 spacing = 0;
	gint pix_width = 0;
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask;	/* bah, ...get_pixtext() is broke, can't be a NULL*  */

	if (row < 0 || row >= clist->rows) return FALSE;
	if (column < 0 || column >= clist->columns) return FALSE;
	if (!edit_func) return FALSE;
	if (!GTK_WIDGET_VISIBLE(clist)) return FALSE;

	if (!gtk_clist_get_text(clist, row, column, &text) &&
	    !gtk_clist_get_pixtext(clist, row, column, &text, &spacing, &pixmap, &mask))
		{
		return FALSE;
		}

	if (!text) text = "";

	if (pixmap) gdk_window_get_size(pixmap, &pix_width, NULL);

	ced = g_new0(ClistEditData, 1);

	ced->old_name = g_strdup(text);
	ced->new_name = NULL;

	ced->edit_func = edit_func;
	ced->edit_data = edit_data;

	ced->clist = clist;
	ced->row = row;
	ced->column = column;

	clist_edit_make_row_visible(ced, row);

	/* figure the position of the rename window
	   (some borrowed from clist internals, may break in future ?) */

	x = GTK_CLIST(ced->clist)->column[(ced->column)].area.x + GTK_CLIST(ced->clist)->hoffset + pix_width +spacing - COLUMN_SPACING - COLUMN_PADDING;
	y = GTK_CLIST(ced->clist)->row_height * (row) + (((row) + 1) * 1) + GTK_CLIST(ced->clist)->voffset;

	w = GTK_CLIST(ced->clist)->column[(ced->column)].area.width - pix_width - spacing + (COLUMN_SPACING + COLUMN_PADDING) * 2;
	h = GTK_CLIST(ced->clist)->row_height;

	gdk_window_get_origin(ced->clist->clist_window, &wx, &wy);

	x += wx;
	y += wy;

	/* create the window */

	ced->window = gtk_window_new (GTK_WINDOW_POPUP);
	gtk_window_set_policy(GTK_WINDOW(ced->window), FALSE, FALSE, FALSE);
	gtk_signal_connect(GTK_OBJECT(ced->window), "button_press_event", clist_edit_click_cb, ced);
	gtk_signal_connect(GTK_OBJECT(ced->window), "key_press_event", GTK_SIGNAL_FUNC(clist_edit_key_press_cb), ced);

	ced->entry = gtk_entry_new_with_max_length(255);
	gtk_entry_set_text(GTK_ENTRY(ced->entry), ced->old_name);
	gtk_entry_select_region(GTK_ENTRY(ced->entry), 0, strlen(ced->old_name));
	gtk_container_add(GTK_CONTAINER(ced->window), ced->entry);
	gtk_widget_show(ced->entry);

	/* now show it */
	gtk_widget_set_uposition (ced->window, x, y);
	gtk_widget_set_usize (ced->window, w, h);
	gtk_widget_realize (ced->window);
	gdk_window_resize (ced->window->window, w, h);
	gtk_widget_show (ced->window);

	/* grab it */
	gtk_widget_grab_focus(ced->entry);
	gtk_grab_add(ced->window);
	gdk_pointer_grab(ced->window->window, TRUE,
			 GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_MOTION_MASK,
			 NULL, NULL, GDK_CURRENT_TIME);

	return TRUE;
}

