%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\chapter{Repository format}
\label{repository_format}

A repository consists of a working directory, which has within it a
directory called \verb!_darcs!. There must also be subdirectories within
\verb!_darcs! named \verb!current! and \verb!patches!. The \verb!current!
directory contains the version of the tree which has been recorded, while
\verb!patches! contains the actual patches which are in the repository.

Also within \verb!_darcs! is the \verb!inventory! file, which lists all the
patches that are in the repo. Moreover, it also gives the order of the
representation of the patches as they are stored. Given a source of patches,
i.e. any other set of repositories which have between them all the patches
contained in a given repo, that repo can be reproduced based on only the
information in the \verb!inventory! file. Under those circumstances, the
order of the patches specified in the \verb!inventory! file would be
unimportant, as this order is only needed to provide context for the
interperetation of the stored patches in this repository.

\begin{code}
module Repository ( PatchSequence, slurp_current, slurp_recorded,
                    read_pending, write_pending, format_inventory,
                    write_inventory, add_to_inventory, read_repo,
                    get_unrecorded, is_repo,
                    get_markedup_file, get_markedup_repo,
                    copy_repo_patches, am_in_repo,
                    PatchSet,
                    read_patch,
                    absolute_dir,
                  ) where

import IO
import Directory
import IOExts--PrelIOBase
import Monad ( liftM, when )
import List ( sortBy, sort )
import FastPackedString ( packString, readFilePS )

import SlurpDirectory
import Patch
import PatchInfo
import Diff
import External
import DarcsArguments ( DarcsFlag(IgnoreTimes, Verbose, AnyOrder) )
import RepoTypes ( PatchSet, PatchSequence )
import Depends ( optimize_patchset )
import RepoPrefs ( filetype_function, FileType(..) )
\end{code}

\begin{code}
--am_in_repo is designed to be a wonderfully fast version of is_repo, just
--to check if i am in a repo.
am_in_repo :: IO Bool
am_in_repo = doesFileExist "_darcs/inventory" `mand`
             doesDirectoryExist "_darcs/patches" `mand`
             doesDirectoryExist "_darcs/current"
a `mand` b = do isa <- a
                if isa then b else return False
is_repo :: FilePath -> IO Bool
is_repo d = do
  ps <- read_repo d
  if ps /= []
     then return True
     else do
       isdir <- doesDirectoryExist d
       if not isdir then return False
         else do
         has_darcs <- doesDirectoryExist $ d++ "/_darcs"
         if not has_darcs then return False
           else do
           has_inv <- doesFileExist $ d++ "/_darcs/inventory"
           has_patches <- doesDirectoryExist $ d++ "/_darcs/patches"
           has_cur <- doesDirectoryExist $ d++ "/_darcs/current"
           return $ has_cur && has_patches && has_inv
\end{code}

There is a very special patch which may be stored in \verb!patches! which
is called `pending'.  This patch describes any changes which have not yet
been recorded, and cannot be determined by a simple diff.  For example file
additions or renames are placed in pending until they are recorded.
Similarly, token replaces are stored in pending until they are recorded.

\begin{code}
read_pending :: IO (Maybe Patch)
write_pending :: Patch -> IO ()

read_pending = do
  pend <- readFilePS "_darcs/patches/pending" `catch`
          (\e -> return $ packString "")
  case readPatchPS pend of
    Nothing -> return Nothing
    Just (p,_) -> return $ Just p

write_pending p =
  writePatch "_darcs/patches/pending" p
\end{code}

\begin{code}
get_unrecorded :: [DarcsFlag] -> IO (Maybe Patch)
get_unrecorded opts = do
    ocur <- slurp "_darcs/current"
    owork <- co_slurp ocur "."
    when (Verbose `elem` opts) $ putStr "syncing dir...\n"
    sync "./_darcs/current" ocur owork
    when (Verbose `elem` opts) $ putStr "done syncing dir...\n"
    cur <- slurp_current "."
    work <- co_slurp cur "."
    pend <- read_pending
    when (Verbose `elem` opts) $ putStr "diffing dir...\n"
    ftf <- filetype_function
    case mydiff ftf cur work of
      Nothing -> return pend
      Just di->
        case pend of
        Nothing -> return $ Just di
        Just pp ->
            if AnyOrder `elem` opts
            then return $ Just $ join_patches $ flatten $ join_patches [pp,di]
            else return $ Just $ reorder $ join_patches $
                 flatten $ join_patches [pp,di]
    where mydiff ftf = if IgnoreTimes `elem` opts
                       then paranoid_diff ftf else diff ftf

slurp_recorded :: FilePath -> IO Slurpy
slurp_recorded d = slurp $ d++"/_darcs/current"
\end{code}

\begin{comment}
slurp_current, below is poorly named.  It doesn't give you the contents of
"current". but rather current with pending applied.  This is some sort of a
nomenclature bug.  FIXME
\end{comment}

\begin{code}
slurp_current :: FilePath -> IO Slurpy
slurp_current d = do
  cur <- slurp $ d ++ "/_darcs/current"
  mbpend <- read_pending
  case mbpend of
    Just pend ->
      case apply_to_slurpy pend cur of
        Just pendcur -> return pendcur
        Nothing -> do putStr "Yikes, pending has conflicts!\n"
                      return cur
    Nothing -> return cur
\end{code}

\begin{code}
--format_inventory will soon be deprecated for use outside of the
--Repository module itself.
format_inventory :: PatchSequence -> String
format_inventory [] = ""
format_inventory ((pi,p):ps) = show pi++"\n"++format_inventory ps

write_inventory :: FilePath -> PatchSet -> IO ()
-- Note that write_inventory optimizes the inventory it writes out by
-- checking on tag dependencies.
-- FIXME: There is also a problem that write_inventory always writes
-- out the entire inventory, including the parts that you haven't
-- changed...
write_inventory dir ps = do
    createDirectory (dir++"/_darcs/inventories") `catch` (\_->return ())
    simply_write_inventory "inventory" dir $ optimize_patchset ps
simply_write_inventory name dir [] =
    writeFile (dir++"/_darcs/"++name) ""
simply_write_inventory name dir [ps] = do
    writeFile (dir++"/_darcs/"++name) $ format_inventory $ reverse ps
simply_write_inventory name dir (ps:pss) = do
    tagname <- return $ make_filename $ fst $ head $ reverse ps
    writeFile (dir++"/_darcs/"++name) $ "Starting with tag:\n"
              ++ format_inventory (reverse ps)
    simply_write_inventory ("inventories/"++tagname) dir pss

add_to_inventory :: FilePath -> PatchInfo -> IO ()
add_to_inventory dir pi =
    appendFile (dir++"/_darcs/inventory") $ show pi++"\n"
\end{code}

\begin{code}
copy_repo_patches :: Bool -> FilePath -> FilePath -> IO ()
copy_repo_patches verb dir out = do
  realdir <- absolute_dir dir
  pns <- liftM (map (make_filename . fst) . concat) $ read_repo "."
  sequence_ $ map (\pn -> do when verb $ putStr $ "Copying "++pn++"\n"
                             copyFileOrUrl
                               (realdir++"/_darcs/patches/"++pn)
                               (out++"/_darcs/patches/"++pn)) pns

read_repo :: String -> IO PatchSet
read_repo d = do
  realdir <- absolute_dir d
  read_repo_private realdir "inventory"
read_repo_private d iname = do
    i <- fetchFile $ d++"/_darcs/"++iname
    (rest,str) <- case break (=='\n') i of
                  ("Starting with tag:",pistr) -> do
                      r <- unsafeInterleaveIO $ read_repo_private d $
                           "inventories/"++
                           make_filename (head $ read_patch_ids pistr)

                      return (r,pistr)
                  _ -> return ([],i)
    pis <- return $ reverse $ read_patch_ids str
    isdir <- doesDirectoryExist d
    these <- if isdir
             then read_patches_local d pis
             else read_patches_remote d pis
    return $ these : rest

read_patch :: String -> PatchInfo -> IO Patch
read_patch repo i = do
  s <- fetchFilePS pn
  case readPatchPS s of
    Just (p,_) -> return p
    Nothing -> ioError $ userError $ "couldn't read "++pn
  where pn = repo++"/_darcs/patches/"++make_filename i

-- Note that read_repo_patches will soon be deprecated and removed in
-- favor of read_repo.
read_repo_patches :: String -> IO PatchSequence
read_repo_patches d = do pset <- read_repo d
                         return $ reverse $ concat pset
read_patches_remote :: FilePath -> [PatchInfo] -> IO PatchSequence
read_patches_remote dir [] = return []
read_patches_remote dir (i:is) = do
  p <- unsafeInterleaveIO $ readFileQuick $
                                 dir++"/_darcs/patches/"++make_filename i
  rest <- read_patches_remote dir is
  return $ (i,read_one_patch p) : rest
read_patches_local :: FilePath -> [PatchInfo] -> IO PatchSequence
read_patches_local dir [] = return []
read_patches_local dir (i:is) = do
  mp <- unsafeInterleaveIO $
        do s <- readFilePS $ dir++"/_darcs/patches/"++make_filename i
           return $ fst `liftM` (readPatchPS s)
  rest <- read_patches_local dir is
  return $ (i,mp) : rest

readFileQuick :: FilePath -> IO String
readFileQuick f = fetchFile f
readHandleQuick :: Handle -> IO String
readHandleQuick h = do
  done <- hIsEOF h
  if done
     then return ""
     else do
          c <- hGetChar h
          rest <- readHandleQuick h
          return (c:rest)

read_patch_ids :: String -> [PatchInfo]
read_patch_ids [] = []
read_patch_ids inv =
    case reads inv of
    [(pi,r)] ->
        pi : read_patch_ids r
    [] -> []

absolute_dir :: FilePath -> IO FilePath
absolute_dir dir = do
  isdir <- doesDirectoryExist dir
  if not isdir
     then return dir -- hope it's an URL
     else do
          former_dir <- getCurrentDirectory
          setCurrentDirectory dir
          realdir <- getCurrentDirectory -- This one is absolute!
          setCurrentDirectory former_dir
          return realdir
\end{code}

\input{RepoPrefs.lhs}

\begin{comment}
\section{Getting interesting info on change history}

One can query the repository for the contents of the repository after a
given patch was applied.

\begin{code}
get_markedup_repo :: PatchInfo -> IO MarkedUpRepo
get_markedup_repo pi = do
  ps <- read_repo_patches "."
  return $ sortBy comp_fn $ do_markup_of pi ps []

comp_fn (f1,_,_,_) (f2,_,_,_) = compare f1 f2

do_markup_of pi ((pi',Just p):ps) mk
    | pi == pi' = markup_repo pi p $ clean_markedup_repo mk
    | otherwise = do_markup_of pi ps $ markup_repo pi' p mk
do_markup_of _ [] mk = mk
do_markup_of pi ((pi',Nothing):ps) mk
    | pi == pi' =
        (make_filename pi,pi,"Error!",MovedFile "Error reading patch") : clean_markedup_repo mk
    | otherwise = do_markup_of pi ps $ (make_filename pi',pi',"Error!",
                                        MovedFile "Error reading patch") : mk
\end{code}

One can query the repository for the entire markup history of a file.  This
provides a data structure which contains a history of \emph{all} the
revisions ever made on a given file.

\begin{code}
get_markedup_file :: PatchInfo -> FilePath -> IO MarkedUpFile
get_markedup_file pi f = do
  patches <- liftM (dropWhile (\ (pi',_)-> pi' /= pi)) $ read_repo_patches "."
  return $ snd $ do_mark_all patches (f, [])
do_mark_all ((n,Just p):pps) (f, mk) =
    do_mark_all pps $ markup_file n p (f, mk)
do_mark_all ((n,Nothing):pps) (f, mk) =
    (f, [(packString "Error reading a patch!",None)])
do_mark_all [] (f, mk) = (f, mk)
\end{code}

\end{comment}
