%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs record}
\begin{code}
module Record ( record, select_changes ) where
import DarcsCommands
import DarcsArguments
import Directory
import IO
import System
import Repository
import Patch
import PatchInfo
import SlurpDirectory
import Diff
import Time
import DarcsArguments
import Test ( test_slurpy )
import Monad
import PrintPatch ( printPatch )
\end{code}
\begin{code}
record_description =
 "Record changes as a named patch."
\end{code}

\options{record}

Record is used to name a set of changes and record the patch to the
repository.
\begin{code}
record_help =
 "Record is used to name a set of changes.\n"
\end{code}
\begin{code}
record = DarcsCommand {command_name = "record",
                       command_help = record_help,
                       command_description = record_description,
                       command_extra_args = 0,
                       command_command = record_cmd,
                       command_prereq = am_in_repo,
                       command_get_arg_possibilities = return [],
                       command_argdefaults = nodefaults,
                       command_darcsoptions = [patchname_option, author,
                                          verbose, notest, all_patches,
                                          askdeps, ignoretimes]}
\end{code}
\begin{code}
record_cmd opts args = do
    date <- get_date opts
    author <- get_author opts
    verb <- return $ Verbose `elem` opts
    when verb $ putStr "About to slurp once.\n"
    recorded <- slurp_recorded "."
    when verb $ putStr "About to slurp again.\n"
    recorded' <- slurp_recorded "."
    when verb $ putStr "About to get the unrecorded changes.\n"
    changes <- if All `elem` opts then get_unrecorded (AnyOrder:opts)
               else get_unrecorded opts
    when verb $ putStr "I've gotten unrecorded.\n"
    case changes of
      Nothing -> putStr "No changes!\n"
      Just ch ->
        do
          mychs <- select_changes "record" opts $ flatten ch
          when verb $ putStr "I've selected the changes.\n"
          case mychs of
            [] -> putStr "Ok, if you don't want to record anything, that's fine!\n"
            chs ->
              do
                deps <- if AskDeps `elem` opts
                        then ask_about_depends $ join_patches chs
                        else return []
                when verb $ putStr "I've asked about dependencies...\n"
                name <- get_patchname opts
                log <- get_log opts
                let mypatch = namepatch date name author log $ join_patches chs
                    myinfo = patchinfo date name author log
                    myfn = "_darcs/patches/"++make_filename myinfo
                 in
                 case apply_to_slurpy mypatch recorded of
                   Nothing -> do putStr "Unable to apply patch!\n"
                                 exitWith $ ExitFailure 1
                   Just working -> do
                     when verb $ putStr "I've applied to slurpy.\n"
                     want_test <- want_to_do_test opts
                     when want_test $ do testproblem <- test_slurpy working
                                         when (testproblem /= ExitSuccess) $
                                              exitWith $ ExitFailure 1
                     when verb $ putStr "Writing the patch file...\n"
                     writePatch myfn $ adddeps mypatch deps
                     former_dir <- getCurrentDirectory
                     setCurrentDirectory "_darcs/current"
                     when verb $ putStr "Applying to current...\n"
                     case apply_to_slurpy mypatch recorded' of
                       Just s' -> slurp_write_dirty s'
                       Nothing -> do
                                  putStr "Bizarre error in recording...\n"
                                  exitWith $ ExitFailure 1
                     setCurrentDirectory former_dir
                     add_to_inventory "." myinfo
                     writeFile "_darcs/patches/pending" ""
                     putStr $ "Finished recording patch '"++name++"'\n"
\end{code}
Each patch is given a name, which typically would consist of a brief
description of the changes.  This name is later used to describe the patch.
The name must fit on one line (i.e. cannot have any embedded newlines).  If
you have more to say, stick it in the log.  (FIXME: Currently recording a
patch deletes any pending changes that aren't included in the recorded
patch.  This is clearly a bug, but is a bit tedious to fix.)
\begin{code}
cmdline_patchname :: [DarcsFlag] -> Bool
cmdline_patchname (PatchName _:_) = True
cmdline_patchname (_:fs) = cmdline_patchname fs
cmdline_patchname [] = False
get_patchname :: [DarcsFlag] -> IO String
get_patchname (PatchName n:_) = return n
get_patchname (_:flags) = get_patchname flags
get_patchname [] = do
    putStr "What is the patch name? "
    hFlush stdout
    getLine
\end{code}
The patch is also flagged with the author of the change, taken by default
from the \verb!DARCS_EMAIL! environment variable, and if that doesn't
exist, from the \verb!EMAIL! environment variable.  The date on which the
patch was recorded is also included.  Currently there is no provision for
keeping track of when a patch enters a given repository.
\begin{code}
get_date :: [DarcsFlag] -> IO String -- No, get_date doesn't ask a girl out for you!
get_date _ = do
    clocktime <- getClockTime
    caltime <- toCalendarTime clocktime
    return $ calendarTimeToString caltime
\end{code}
Finally, each changeset should have a full log (which may be empty).  This
log is for detailed notes which are too lengthy to fit in the name.  If you
answer that you do want to create a comment file, darcs will open an editor
so that you can enter the comment in.  The choice of editor proceeds as
follows.  If one of the \verb!$DARCSEDITOR!, \verb!$VISUAL! or
\verb!$EDITOR! environment variables is defined, its value is used (with
precedence proceeding in the order listed).  If not, ``vi'', ``emacs'',
``emacs~-nw'' and ``nano'' are tried in that order.
\begin{code}
get_log :: [DarcsFlag] -> IO [String]
get_log opts =
  if cmdline_patchname opts then return []
  else do putStr "Do you want to add a long comment? "
          hFlush stdout
          yorn <- getLine
          case yorn of
            ('y':_) -> do writeFile ".darcs-temp-log" ""
                          edit_file ".darcs-temp-log"
                          liftM lines $ readFile ".darcs-temp-log"
            _ -> return []
\end{code}

\begin{code}
select_help verb = "y: "++verb++" this patch
n: don't "++verb++" it
s: skip all patches to this file
a: "++verb++" all remaining patches
h or ?: show this help\n"

select_changes :: String -> [DarcsFlag] -> [Patch] -> IO [Patch]
select_changes _ _ [] = return []
select_changes verb opts (p:ps) =
  if All `elem` opts then return (p:ps)
  else do
    printPatch p
    putStr $ "Shall I "++verb++" this patch? [ynsah] "
    hFlush stdout
    yorn <- getLine
    case yorn of
      ('a':_) -> select_changes verb [All] (p:ps)
      ('y':_) -> if is_addfile p && ps /= [] && is_similar p (head ps)
                 then do rest <- select_changes verb [] $ tail ps
                         return $ p : (head ps) : rest
                 else do rest <- select_changes verb [] ps
                         return $ p : rest
      ('s':_) -> do select_changes verb [] $ move_similar_after p ps
      ('h':_) -> do putStr $ select_help verb
                    select_changes verb [] (p:ps)
      ('?':_) -> do putStr $ select_help verb
                    select_changes verb [] (p:ps)
      otherwise -> select_changes verb [] $ move_patch_after p ps
\end{code}

\begin{code}
move_patch_after :: Patch -> [Patch] -> [Patch]
move_patch_after _ [] = []
move_patch_after p (pp: pps) =
    case commute (pp, p) of
    Nothing -> move_patch_after p $ move_patch_after pp pps
    Just (p', pp') -> pp' : move_patch_after p' pps

move_similar_after :: Patch -> [Patch] -> [Patch]
move_similar_after p [] = []
move_similar_after p (pp:pps) =
    if is_similar p pp
    then move_similar_after p $ move_patch_after pp pps
    else pp : move_similar_after p pps
\end{code}

Each patch may depend on any number of previous patches.  If you choose to
make your patch depend on a previous patch, that patch is required to be
applied before your patch can be applied to a repo.  This can be used, for
example, if a piece of code requires that a function be defined, which has
was defined in an earlier patch.

If you want to manually define any dependencies for your patch, you can use
the \verb!--ask-deps! flag, and darcs will ask you for the patch's
dependencies.

\begin{code}
depends_help = "y: depend on this patch
n: don't depend on this patch
s: don't depend on any patch
? or h: show this help
"

ask_about_depends :: Patch -> IO [PatchInfo]
ask_about_depends pa = do
  ps <- read_repo "."
  ask_deps (head ps) pa

ask_deps :: PatchSequence -> Patch -> IO [PatchInfo]
ask_deps ((pi,mp):pds) pa =
  case mp of
    Nothing -> return []
    Just p ->
      case commute (pa,p) of
      Nothing -> ask_deps (pass_back_patch p pds) pa
      Just (p',pa') -> do
        putStr $ show pi
        putStr "\nDoes your patch depend on this patch? [ynsh] "
        hFlush stdout
        yorn <- getLine
        case yorn of
          ('y':_) -> do rest <- ask_deps pds pa'
                        return $ pi : rest
          ('s':_) -> return []
          ('h':_) -> do putStr depends_help
                        ask_deps ((pi,mp):pds) pa
          ('?':_) -> do putStr depends_help
                        ask_deps ((pi,mp):pds) pa
          otherwise -> ask_deps pds pa'
ask_deps [] pa = return []

pass_back_patch :: Patch -> PatchSequence -> PatchSequence
pass_back_patch p1 ((pi,mp2):pds) =
  case mp2 of
  Nothing -> ((pi,Nothing):pds)
  Just p2 -> case commute (p1,p2) of
             Nothing -> ((pi,Nothing):pds)
             Just (p2',p1') -> ((pi,Just p2'):pass_back_patch p1' pds)
\end{code}

FIXME: Change this code to use the Readline module so arrow keys, etc, will
work properly for entering in the text.

If you configure darcs to run a test suite, darcs will run this test on the
recorded repo to make sure it is valid.  Darcs first creates a pristine
copy of the source tree (in \verb!/tmp!), then it runs the test, using its
return value to decide if the record is valid.  If it is not valid, the
record will be aborted.  This is a handy way to avoid making stupid
mistakes like forgetting to `darcs add' a new file.  It also can be
tediously slow, so there is an option (\verb!--no-test!) to skip the test.

\begin{code}
want_to_do_test :: [DarcsFlag] -> IO Bool
want_to_do_test (NoTest:_) = return False
want_to_do_test (_:flags) = want_to_do_test flags
want_to_do_test [] = return True
\end{code}
