%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs mv}
\begin{code}
module Mv ( mv ) where
import Monad ( when )

import DarcsCommands
import DarcsArguments
import Directory
import IO
import System
import Repository
import Patch
import SlurpDirectory
\end{code}

\begin{code}
mv_description =
 "Move a file or directory to a different location or name."
\end{code}

\options{mv}

\haskell{mv_help} This is why ``mv'' isn't called ``move'', since it is
really almost equivalent to the unix command ``mv''.  I could add an
equivalent command named ``move'' for those who like vowels.

\begin{code}
mv_help =
 "Darcs mv needs to be called whenever you want to rename or move a file or
directory.  Unlike remove, mv actually performs the move itself in your
working directory.
"
\end{code}

\begin{code}
mv = DarcsCommand {command_name = "mv",
                   command_help = mv_help,
                   command_description = mv_description,
                   command_extra_args = 2,
                   command_command = mv_cmd,
                   command_prereq = am_in_repo,
                   command_get_arg_possibilities = list_files,
                   command_argdefaults = nodefaults,
                   command_darcsoptions = [verbose]}
\end{code}

\begin{code}
mv_cmd _ [old,new] = do
    cur <- slurp_current "."
    work <- slurp "."
    pend <- read_pending
    ps <- return $ []
    when (slurp_has_anycase new work || slurp_has_anycase new cur) $ do
         putStr $ "Sorry, but a file or dir named "++new++" already exists.\n"
         exitWith $ ExitFailure 2
    if slurp_hasfile old cur &&
       slurp_hasfile old work
       then renameFile old new
       else if slurp_hasdir old cur &&
               slurp_hasdir old work
               then renameDirectory old new
               else do putStr $ "Can't move "++old++" to "++new++"\n"
                       exitWith $ ExitFailure 1
    case pend of
        Nothing -> write_pending $ move old new
        Just op -> write_pending $ join_patches $
                   flatten $ join_patches [op, move old new]
\end{code}
